package HTTP::PublicKeyPins;

use 5.006;
use strict;
use warnings;
use Crypt::OpenSSL::X509();
use Crypt::OpenSSL::RSA();
use Digest();
use MIME::Base64();
use Exporter();
use Carp();
*import = \&Exporter::import;
our @EXPORT_OK = qw(
  pin_sha256
);
our %EXPORT_TAGS = ( 'all' => \@EXPORT_OK, );

our $VERSION = '0.03';

sub pin_sha256 {
    my ($path) = @_;
    my $x509 = Crypt::OpenSSL::X509->new_from_file($path);
    my $string;
    if ( $x509->key_alg_name() eq 'rsaEncryption' ) {
        my $pubkey = Crypt::OpenSSL::RSA->new_public_key( $x509->pubkey() );
        $string = $pubkey->get_public_key_x509_string();
    }
    else {
        $string = $x509->pubkey();
    }
    $string =~ s/^[-]+BEGIN[^\n]+\n//smx;
    $string =~ s/^[-]+END[^\n]+\n//smx;
    my $der    = MIME::Base64::decode($string);
    my $digest = Digest->new('SHA-256');
    $digest->add($der);
    my $base64 = MIME::Base64::encode_base64( $digest->digest() );
    chomp $base64;
    return $base64;
}

1;    # End of HTTP::PublicKeyPins
__END__

=head1 NAME

HTTP::PublicKeyPins - Generate RFC 7469 HTTP Public Key Pin (HPKP) header values

=head1 VERSION

Version 0.03

=head1 SYNOPSIS

Make it more difficult for the bad guys to Man-In-The-Middle your users TLS sessions

    use HTTP::Headers();
    use HTTP::PublicKeyPins qw( pin_sha256 );

    ...
    my $h = HTTP::Headers->new();
    $h->header( 'Public-Key-Pins-Report-Only',
            'pin-sha256="'
          . pin_sha256('/etc/pki/tls/certs/example.pem')
          . '"; pin-sha256="'
          . pin_sha256('/etc/pki/tls/certs/backup.pem')
          . '"; report-uri="https://example.com/pkp-report.pl' );

    ...

OR 

    perl -MHTTP::PublicKeyPins=pin_sha256 -e 'print pin_sha256("/etc/pki/tls/certs/example.pem") . "\n";'

=head1 DESCRIPTION

This module allows the calculation of RFC 7469 HTTP Public Key Pin header values. This can be used to verify your TLS session to a remote server has not been hit by a Man-In-The-Middle attack OR to instruct your users to ignore any TLS sessions to your web service that does not use your Public Key

=head1 EXPORT

=head2 pin_sha256

This function accepts the path to a certificate.  It will load the public key from the certificate and prepare the appropriate value for the pin_sha256 parameter of the Public-Key-Pins value.

=head1 SUBROUTINES/METHODS

None.  This module only has the one exported function.

=head1 DIAGNOSTICS
 
None. Exceptions may be thrown by modules in the Dependencies below;
 
=head1 CONFIGURATION AND ENVIRONMENT
 
HTTP::PublicKeyPins requires no configuration files or environment variables.
 
=head1 DEPENDENCIES
 
HTTP::PublicKeyPins requires the following non-core modules
 
  Crypt::OpenSSL::X509;
  Crypt::OpenSSL::RSA;
  Digest
 
=head1 INCOMPATIBILITIES
 
None known.

=head1 SEE ALSO

L<RFC 7469 - Public Key Pinning Extension for HTTP|http://tools.ietf.org/html/rfc7469>

=head1 AUTHOR

David Dick, C<< <ddick at cpan.org> >>

=head1 BUGS AND LIMITATIONS
 
Please report any bugs or feature requests to C<bug-http-publickeypins at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=HTTP-PublicKeyPins>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc HTTP::PublicKeyPins


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=HTTP-PublicKeyPins>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/HTTP-PublicKeyPins>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/HTTP-PublicKeyPins>

=item * Search CPAN

L<http://search.cpan.org/dist/HTTP-PublicKeyPins/>

=back


=head1 LICENSE AND COPYRIGHT

Copyright 2015 David Dick.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

