# dir.pl -- directory browser (with filter)
#
# $Id: dir.pl,v 2.6 1993/07/16 18:50:18 sanders Exp $
#
# Tony Sanders <sanders@bsdi.com>, April 1993
#

sub retrieve {
    local($path, $filter) = @_;
    -e $path || &error('not_found', "document \`\`$path\'\' does not exist");
    return -d $path ? &index_dir($path, $filter) : &wrap_file($path);
}

sub wrap_file {
    local($path) = @_;
    local($FD) = "main'FD";
    local($content);

    alarm($http_timeout);
    open($FD, $path) || &error('not_found', "$path: $!");
    flock($FD, &LOCK_SH);
    &set_last_modified($path);

    # &add_tranlations creates a pipeline of processes that read
    # from $FD (the real file) and write to the returned $FD
    # (which is the last processes stdout).
    ($FD, $content) = &add_translations($FD, &deduce_content($path));

    &wrap_fd($FD, $content);
    close($FD);
    &end_translations();			# cleanup
}

sub wrap_fd {
    local($FD, $content) = @_;
    if ($version) {
	&MIME_header('ok', $content);		# wrap HTTP/1.0 headers
    } else {
	local($ext) = $path =~ m/\.(\w+)$/;
	print '<PLAINTEXT>'
	    unless(defined $ext{$ext} || defined $encoding{$ext});
    }
    &raw_fd($FD);
    close($FD);
}

sub raw_file {
    local($path) = @_;

    alarm($http_timeout);
    open(FD, $path) || &error('not_found', "$path: $!");
    flock(FD, &LOCK_SH);
    &raw_fd(FD);
    close(FD);
}

# handles partial writes
sub raw_fd {
    local($FROM) = @_;
    local($_, $len, $written, $offset);

    while (($len = sysread($FROM, $_, 8192)) != 0) {
	alarm($http_timeout);
        if (!defined $len) {
            next if $! =~ /^Interrupted/;
            &error('internal_error', "System read error: $!");
	}
	$offset = 0;
	while ($len) {
	    $written = syswrite(STDOUT, $_, $len, $offset);
	    &error('internal_error', "System write error: $!")
	        unless defined $written;
	    $len -= $written;
	    $offset += $written;
	}
    }
}

sub set_last_modified {
    local($path) = @_;
    &add_header(*out_headers,
	sprintf("Last-Modified: %s", &fmt_date((stat($path))[9])));
}

sub deduce_content {
    local($path) = @_;
    local(@ext, $ext, $encoding) = split(/\./, $path);

    while (($ext = pop(@ext)) && (defined $encoding{$ext})) {
	$encoding .= '; ' if defined $encoding;
	$encoding .= $encoding{$ext};
    }
    &add_header(*out_headers, "Content-encoding: $encoding")
        if defined $encoding;
    return $ext{$ext} if defined $ext{$ext};
    return (-B $path ? $content_binary : $content_text);
}

sub index_dir {
    local($dir, $filter) = @_;
    local($count, $_) = 1;

    local($ndx) = "$dir/$http_index";
    do { &wrap_file($ndx); return; } if -f $ndx;

    $http_indexdirs || &error('bad_request', "$dir is a directory");
    defined($hidden{$dir}) && &error('bad_request', "$dir is a directory");
    $filter = '.*' unless $filter;

    alarm($http_timeout);

    &MIME_header('ok', 'text/html');
    if (-f "$dir/$dir_header") {
	&raw_file("$dir/$dir_header")
    } else {
	print <<EOT;
<HEAD>
<ISINDEX><TITLE>Index of /$dir: filter=$filter</TITLE>
</HEAD>
<BODY>
<H1>Directory Index of /$dir</H1>
Use the Search Keyword to pass a <A HREF="$http_perlexp">
perl regular expression</A> to be used as a filter.
<EM>Note:</EM> it cannot end with a special file suffix
like gif, just add a ``\$'' on the end (e.g., ``gif\$'').
See Also <A HREF="$http_dirdoc">The Directory Browser</A>.
EOT
    }

    print "<PRE>\n";
    $_ = $dir; while (chop ne '/' && length) { ; }
    printf "%39s%s", "", "<A NAME=$count HREF=\"/$_\">Parent Directory</A>\n"; $count++;

    # Get directory listing
    alarm($http_timeout);
    opendir(DH, "$dir") || &error('not_found', "$dir: $!");
    @dirs = sort(readdir(DH));
    while ($_ = shift @dirs) {
	alarm($http_timeout);
	next if /^\.$/ || /^\.\.$/;
	next if defined($hidden{"$dir/$_"});	# hidden from listing
	next unless /$filter/;
	local($dev, $ino, $mode, $nlink, $uid, $gid, $rdev, $size,
	    $atime, $mtime, $ctime, $blksize, $blocks) = stat("$dir/$_");
	local($date) = &ctime($mtime); chop $date;
	printf "%s %9d ", $date, $size;
	print "<A NAME=$count HREF=\"/$dir/$_\">$_</A>"; $count++;
	&S_ISDIR($mode) && print "/";
	print "\n";
    }
    closedir(DH);

    print "</PRE>\n";

    if (-f "$dir/$dir_footer") {
        &raw_file("$dir/$dir_footer");
    } else {
	print "</BODY>\n";
    }
}

#
# Currently we only search one deep in the conversion tree
# XXX: needs to grok Accept: fields.
#
sub add_translations {
    local($FD, $content) = @_;
    local($map, @conversions);

    # XXX: should I call &end_translations???
    @fds = ();						# init global

    # These two are required by the RFC
    $accepted{'text/html'} = $accepted{'text/plain'} = 1;
    return ($FD, $content) if defined $accepted{$content};
    # map if you can
    if (defined ($map = $trans{$content})) {
	@conversions = split(/:/, $map);
        CONV:
        while ($#conversions != $[) {
	    $to = shift @conversions;
	    $how = shift @conversions;
	    do { $content = $to; $FD = &pipe($FD, "&$how"); last CONV; }
	        if defined $accepted{$to};
	}
	# after final conversion prepare to copyout
	$FD = pop(@fds); foreach(@fds) { close($_); }
    }
    # even if we can't find a conversion to something known
    return ($FD, $content);
}

sub end_translations {
    while(wait != -1) {; }
}

$rdrs = "main'rdr0000";
$wrts = "main'wrt0000";
@fds = ();

# $stdin is STDIN for child
# $stdout is STDOUT from child
# $wrt is childs side of STDOUT pipe
sub pipe {
    local($stdin, $cmd) = @_;
    local($stdout, $wrt) = ($rdrs++, $wrts++);

    # DEBUG: print STDERR "$stdin >> $cmd << $wrt\n";

    pipe($stdout, $wrt) || die "pipe: $!";
    if (($pid = fork) < 0) {
        die "fork: $!";
    } elsif ($pid == 0) {
	open(STDIN, "<& $stdin") || die "dup stdin: $!";
        open(STDOUT, ">& $wrt") || die "dup stdout: $!";
        foreach(@fds){close($_);}
        close($stdout); close($wrt);
        eval $cmd; die $@ if $@; exit 0;
    }
    close($wrt);
    push(@fds, $stdout);
    return $stdout;
}

1;
