;;; $Id: vcs-sccs.el,v 1.5 26-Oct-1992 21:55:10 EST don Exp $
;;; 
;;; SCCS support for vcs.el
;;; 
;;; Copyright (C) Donald Beaudry <don@vicorp.com> 1992
;;;
;;; This file is not part of GNU Emacs, but is made available under
;;; the same conditions.
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 1, or
;;; (at your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; $Log: vcs-sccs.el,v $
;;; Revision 1.5  26-Oct-1992 21:55:10 EST  don
;;; Ok, so it works...
;;;
;;; Revision 1.4  25-Oct-1992 21:36:47 EST  don
;;; changed for use with the new form stuff... still needs to be tested though.
;;;
;;; Revision 1.3  12-Sep-1992 19:25:01 EDT  don
;;; Merged put and finish put
;;;
;;; Revision 1.2  12-Sep-1992 18:39:27 EDT  don
;;; Separation from vcs.el
;;;
;;; 

(require 'form)

;;;
;;;
;;;
(defun vcs-sccs-get (history-file working-file lock)
  (if (vcs-sccs-has-history-p history-file)
      (progn
	(vcs-execute-command (file-name-directory working-file)
			     "rm -f" working-file 
			     "\nget" (if lock "-e") history-file)
	t)))

(vcs-add-hook 'vcs-get-hooks 'vcs-sccs-get)


;;;
;;;
;;;
(defun vcs-sccs-unlock (history-file working-file)
  (if (vcs-sccs-has-history-p history-file)
      (progn
	(vcs-execute-command (file-name-directory working-file)
			     "unget -s -n" history-file)
	t)))

(vcs-add-hook 'vcs-unlock-hooks 'vcs-sccs-unlock)


;;;
;;;
;;;
(defun vcs-sccs-insert-info (history-file working-file)
  (if (vcs-sccs-has-history-p history-file)
      (progn
	(insert (format "History file: %s\n" history-file))
	(insert (format "Working file: %s\n" working-file))
	(insert "\n")
	(insert "Locks: ")
	(if (vcs-sccs-locker-name history-file)
	    (call-process vcs-shell-path nil buf nil "-c"
			  (concat "sact " history-file
				  "| sed -e '2,$s/\\(.\\)/       \\1/'"))
	  (insert "<none>\n"))
	(insert "\n")
	(call-process vcs-shell-path nil buf nil "-c"
		      (concat "get -g -lp " history-file))
	(goto-char (point-min))
	t)))

(vcs-add-hook 'vcs-insert-info-hooks 'vcs-sccs-insert-info)


;;;
;;;
;;;
(defform vcs-sccs-put
  "Major mode checking files into SCCS"
  (when-finished vcs-sccs-finish-put)
  (buffer vcs-put-buffer)
  (mode-name "VCS-sccs-put")
  (text "                       ====== SCCS Put ======\n\n\n")
  (field history-file
	 (prompt "History File: ")
	 (default history-file)
  	 (verifier (lambda (b e)
		     (if (< b e)
			 t
		       (error "A history file must be specified")))))
  (text "\n")
  (field working-file
	 (prompt "Working File: ")
	 (default working-file)
	 (verifier (lambda (b e)
		     (if (file-exists-p (buffer-substring b e))
			 t
		       (error "Working file does not exist")))))
  (text "\n\n")
  (field comment
	 (prompt (if (not (file-exists-p history-file))
		     "--Descriptive text--\n"
		   "--Log message--\n"))
	 (verifier (lambda (b e)
		     (if (< b e)
			 t
		       (error "A log message must be specified")))))
  (text "\n\n--\n"))


;;;
;;;
;;;
(defun vcs-sccs-setup-for-put (history-file working-file)
  (if (vcs-sccs-working-file history-file)
      (progn
	(vcs-sccs-put-mode nil vcs-use-other-window 'comment)
	(setq default-directory (file-name-directory working-file))
	(run-hooks 'vcs-put-mode-hooks)
	t)))

(vcs-add-hook 'vcs-put-hooks 'vcs-sccs-setup-for-put)


;;;
;;;
;;;
(defun vcs-sccs-locker-name (history-file)
  "Return the name of the person who has FILE-NAME locked under sccs
Or nil if there are no lockers or the file is not under sccs."
  (if (vcs-sccs-has-history-p history-file)
      (let (lockers (buf (get-buffer-create vcs-temp-buffer)))
	(save-excursion
	  (set-buffer buf)
	  (goto-char (point-max))
	  (condition-case nil
	      (vcs-execute-command nil "sact" history-file)
	    (error nil))
	  (while (re-search-forward "^[0-9.]+ [0-9.]+ \\([^ ]*\\) "
				    (point-max) t)
	    (setq lockers
		  (if lockers
		      (concat lockers ", "
			      (buffer-substring (match-beginning 1)
						(match-end 1)))
		    (buffer-substring (match-beginning 1)
				      (match-end 1))))))
	lockers)))

(vcs-add-hook 'vcs-locker-name-hooks 'vcs-sccs-locker-name)


;;;
;;;
;;;
(defun vcs-sccs-working-file (file-name)
  "Return the working file name for a given sccs history FILE-NAME."
  (if file-name
      (let* ((name (file-name-nondirectory file-name))
	     (directory (file-name-directory file-name)))
	(if (not (string-match "^s\\." name))
	    nil
	  (setq name (substring name (match-end 0)))
	  (if (string-match "/SCCS/$" directory)
	      (setq directory (substring directory 0
					 (1+ (match-beginning 0))))) 
	  (concat directory name)))))
     
(vcs-add-hook 'vcs-working-file-hooks 'vcs-sccs-working-file)

;;;
;;;
;;;
(defun vcs-sccs-history-file (file-name)
  "Return the name of the sccs history file name for FILE-NAME"
  (let ((case-fold-search nil))
    (if (not file-name)
	nil
      (let ((directory (file-name-directory (expand-file-name file-name)))
	    (name (file-name-nondirectory file-name)))
	(concat (if (and (file-directory-p (concat directory "SCCS"))
			 (not (string-match "/SCCS/$" directory)))
		    (concat directory "SCCS/")
		  directory)
		(if (string-match "^s\\." name)
		    name
		  (concat "s." name)))))))
      
(vcs-add-hook 'vcs-history-file-hooks 'vcs-sccs-history-file)


;;;
;;; check command output for an SCCS error message
;;;
(defun vcs-sccs-scan-for-error ()
  "Scan the shell command output buffer for an sccs error message
and signal an error if one was found."
  (save-excursion
    (if (re-search-forward "^ERROR .*]:" nil t)
	(let ((start-pos (1+ (point))))
	  (search-forward "\n" nil t)
	  (buffer-substring start-pos (1- (point)))))))

(vcs-add-hook 'vcs-scan-for-error-hooks 'vcs-sccs-scan-for-error)


;;;
;;;
;;;
(defun vcs-sccs-has-history-p (file-name)
  "Returns a non NIL value if FILE-NAME is under sccs control."
  (setq file-name (vcs-sccs-working-file file-name))
  (if file-name
      (file-exists-p (vcs-sccs-history-file file-name))))


;;;
;;;
;;;
(defun vcs-sccs-finish-put (put-data)
  (if (equal (current-buffer) (get-buffer vcs-put-buffer))
      nil
    (error "Wrong buffer"))
  (let* ((history-file (form-field-string 'history-file put-data))
	 (working-file (form-field-string 'working-file put-data))
	 (comment-region (form-field-region 'comment put-data))
	 (comment-file (concat vcs-temp-dir "/" (make-temp-name "vcs")))
	 (buf (get-file-buffer working-file)))
    (if (and buf (buffer-modified-p buf))
	(save-excursion
	  (set-buffer buf)
	  (if (y-or-n-p (concat "Save " buffer-file-name "? "))
	      (save-buffer))))
    (message "Checking in %s..." history-file)
    (write-region (car comment-region) (cdr comment-region)
		  comment-file nil 'no-message)
    (if (file-exists-p history-file)
	(progn
	  (if (not (string= (file-name-nondirectory working-file)
			    (file-name-nondirectory
			     (vcs-working-file-name history-file))))
	      (error "History name does not match working name"))
	  (vcs-execute-command (file-name-directory working-file)
			       "delta"
			       (concat "\"-y`cat " comment-file "`\"")
			       history-file))
      (vcs-execute-command (file-name-directory working-file)
			   "ex -s -cwq" comment-file "\n"
			   "admin"
			   (concat "-t" comment-file)
			   (concat "-i" working-file)
			   history-file
			   "&& rm -f" working-file))
    (message "Checked in %s..." history-file)
    (vcs-cleanup-after-put history-file working-file)
    t))



