# DocViewer.tcl --
#
# A mock-up document viewer application that demonstrates some 
# of the Tix widgets: tixComboBox, tixControl, tixSelect, tixTree,
# tixNoteBook, tixPanedWindow, and tixScrolledText.
#
# This sample program is structured in such a way so that it can be
# executed from the Tix demo program "widget": it must have a
# procedure called "RunSample". It should also have the "if" statment
# at the end of this file so that it can be run as a standalone
# program using tixwish.
#
# $Id: DocViewer.tcl,v 1.1 2001/03/03 09:27:55 ioilam Exp $

namespace eval ::docviewer {
    variable section {}
}

# docviewer::RunSample --
#
#       Create the main window for this sample program. This procedure
#       is called either by main demo program "widget", or by
#       the block at the end of this file.
#
# Arguments:
#   w           Name of the top-level (already created) to use as
#               the main window of this sample program
# Results:
#   Returns nothing.
#
proc docviewer::RunSample {w} {
    global tcl_platform
    variable navList

    #
    # Create the menubar
    #
    menu $w.menuBar -tearoff 0
    $w.menuBar add cascade -menu $w.menuBar.file -label "File" -underline 0
    menu $w.menuBar.file -tearoff 0

    $w.menuBar.file add command -label "Exit" -command "destroy $w"
    $w configure -menu $w.menuBar

    #
    # Create the toolbar
    #
    set tb [frame $w.tb]
    pack $tb -side top -fill x

    if {"$tcl_platform(platform)" == "unix"} {
        catch {
            #
            # Try to make the toolbar look like the Unix menu bar
            #
        
            set sysMenuBar $w.#demo#menuBar
            $tb config -bd [$sysMenuBar cget -bd] \
                -relief [$sysMenuBar cget -relief]
        }
    } else {
        $tb config -bd 2 -relief groove
    }

    #
    # Use a tixComboBox for user to choose font name
    #
    set font [tixComboBox $tb.font -label {} \
        -options {
	    entry.width    11
	    listbox.height 4
	}]
    pack $font -side left -pady 2

    $font insert end "Courier"
    $font insert end "Helvetica"
    $font insert end "Lucida"
    $font insert end "Times Roman"
    $font config -value Helvetica

    #
    # Use a tixSelect widget for user to choose font style
    #
    set style [tixSelect $tb.style -allowzero true -radio false -label {}]
    pack $style -side left -pady 2
    $style add bold      -bitmap [tix getbitmap bold]
    $style add italic    -bitmap [tix getbitmap italic]
    $style add underline -bitmap [tix getbitmap underlin]
    $style add capital   -bitmap [tix getbitmap capital]

    #
    # Use a tixControl widget for choosing font size
    #
    set size [tixControl $tb.size -label {}]
    pack $size -side left -pady 2

    #
    # Create the PanedWindow which contains the tree and notebook
    #

    set pane [tixPanedWindow $w.pane -orient horizontal -paneborderwidth 0]
    pack $pane -side top -expand yes -fill both -padx 10 -pady 10

    set p1 [$pane add tree -size 180]
    set p2 [$pane add note -size 300]
    
    #
    # Create the Tree widget
    #
    set tree [createTree $p1]

    #
    # Create the NoteBook for displaying source and output
    #
    set notebook [createNoteBook $p2]

    #
    # Set the initial document section
    #
    $navList anchor set 1
    $navList selection set 1
    setSection 1
}

# docviewer::createTree --
#
#       Create a mocked-up content navigation tree that points
#       to the chapters and sections of the main document.
#
# Arguments:
#   p           Put the Tree in this frame (already created).
#
# Results:
#   The pathname of the created tixTree widget.
#
proc docviewer::createTree {p} {
    variable navList

    set tree [tixTree $p.tree]
    pack $tree -expand yes -fill both -padx 5

    set navList [$tree subwidget hlist]
    $navList config -browsecmd docviewer::setSection
    foreach link {
        {1      Introduction}
        {1.1    Overview}
        {1.2    Audience}
        {1.3    Acknowledgement}
        {2      {Programming Guide}}
        {2.1    {Using Basic Features}}
        {2.2    {Using Advanced Features}}
        {3      Appendix}
    } {
        set path [lindex $link 0]
        set title [lindex $link 1]

        $navList add $path -itemtype imagetext -image [tix getimage textfile] \
            -text $title
    }

    $tree autosetmode 
    return $tree
}

proc docviewer::setSection {path} {
    variable section
    variable sourceText
    variable outputText
    variable navList

    if {"$path" != "$section"} {
        set section $path

        if {[regexp {[.]} $path]} {
            set block section
            set Block Section
        } else {
            set block chapter
            set Block Chapter
        }

        set title [$navList entrycget $path -text]

        $sourceText delete 1.0 end
        $sourceText insert end "<$block> $title </$block>\n"

        $outputText delete 1.0 end
        $outputText insert end "$Block $path $title\n\n" title

        for {set i 0} {$i < 3} {incr i} {
            $sourceText insert end "<para>\n"
            set nchars 0
            for {set j 0} {$j < 30} {incr j} {
                set max [expr int(rand() * 6) + 1]
                for {set k 0} {$k < $max} {incr k} {
                    $outputText insert end "x" body
                    $sourceText insert end "x"
                    incr nchars
                }
                $outputText insert end " " body
                if {$nchars > 25} {
                    $sourceText insert end "\n"
                    set nchars 0
                } else {
                    $sourceText insert end " "
                }
            }
            $outputText insert end "\n\n"  body
            $sourceText insert end "\n</para>\n\n"
        }
    }
}

proc docviewer::createNoteBook {p} {
    variable sourceText
    variable outputText

    set notebook [tixNoteBook $p.notebook]
    pack $notebook -expand yes -fill both -padx 5

    #
    # Create the two pages, each containing a tixScrolledText widget
    #
    $notebook add output -label "Output"
    $notebook add source -label "Source"

    set o [$notebook subwidget output]
    tixScrolledText $o.st
    pack $o.st -expand yes -fill both -padx 4 -pady 4
    set outputText [$o.st subwidget text]

    # todo: don't hard code
    $outputText tag configure title -font {Helvetica 18 bold} -justify center
    $outputText tag configure body  -font {Helvetica 14}

    $outputText config -wrap word

    set s [$notebook subwidget source]
    tixScrolledText $s.st
    pack $s.st -expand yes -fill both -padx 4 -pady 4
    set sourceText [$s.st subwidget text]

    $sourceText config -wrap none

    return $notebook
}

# This "if" statement makes it possible to run this script file inside or
# outside of the main demo program "widget".
#
if {![info exists tix_demo_running]} {
    # This example uses other scripts auto-loaded from this
    # directory:
    lappend auto_path [file dirname [info script]]

    # This example uses bitmaps from ../bitmaps.
    set demo_dir [file join [file dirname [info script]] ..]
    tix addbitmapdir [file join $demo_dir bitmaps]

    wm withdraw .
    set w .demo
    toplevel $w; wm transient $w ""
    docviewer::RunSample $w
    bind $w <Destroy> {after 10 exit}
}

