/*****************************************************************************
 *
 *  File: escape.c
 *  Routines for adding backslash escapes to strings in order to supress
 *  C and Tcl substitutions.
 *
 *****************************************************************************
 *
 *  Copyright (c) 2000 David Cuthbert / Kanga International Group.
 *  All rights reserved.
 *
 *  Permission to use, copy, modify, distribute and sell this software
 *  and its documentation for any purpose is hereby granted without fee,
 *  provided that the above copyright notice appear in all copies and
 *  that both that copyright notice and this permission notice appear
 *  in supporting documentation.  The author(s) make no representations
 *  about the suitability of this software for any purpose.  It is 
 *  provided "as is" without express or implied warranty.
 *
 *****************************************************************************/

#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include "escape.h"

int GetEscapeCStringSize(char const *source, int n)
{
    int i, o;
    o = 0;

    for(i = 0; i < n; ++i) {
	switch(source[i]) {
	    case '\\': case '\a': case '\b': case '\f': case '\n':
	    case '\r': case '\t': case '\v': case '$':
	    o += 2;

	    default:
	    if(isprint(source[i])) o += 1;
	    else                   o += 4;
	}
    }

    return o;
}


int GetEscapeTclStringSize(char const *source, int n)
{
    int i;
    int o = 0;

    for(i = 0; i < n; ++i) {
	switch(source[i]) {
	    case '\\': case '\a': case '\b': case '\f': case '\n': 
	    case '\r': case '\t': case '\v': case '[':  case ']':  
	    case '$':  case '"':  
	    o += 2;
	    break;

	    default:
	    if(isprint(source[i])) o += 1;
	    else                   o += 4;
	}
    }

    return o;
}


int EscapeCString(char *dest, char const *source, int n)
{
    int i;
    int o = 0;

    for(i = 0; i < n; ++i) {
	switch(source[i]) {
	    case '\\': dest[o++] = '\\'; dest[o++] = '\\'; break;
	    case '\a': dest[o++] = '\\'; dest[o++] = 'a';  break;
	    case '\b': dest[o++] = '\\'; dest[o++] = 'b';  break;
	    case '\f': dest[o++] = '\\'; dest[o++] = 'f';  break;
	    case '\n': dest[o++] = '\\'; dest[o++] = 'n';  break;
	    case '\r': dest[o++] = '\\'; dest[o++] = 'r';  break;
	    case '\t': dest[o++] = '\\'; dest[o++] = 't';  break;
	    case '\v': dest[o++] = '\\'; dest[o++] = 'v';  break;
	    case '"':  dest[o++] = '\\'; dest[o++] = '\"'; break;

	    default:
	    if(isprint(source[i]))
		dest[o++] = source[i];
	    else {
		char buf[5];
		sprintf(buf, "\\%03o", (unsigned int)(unsigned char)source[i]);
		dest[o++] = buf[0];
		dest[o++] = buf[1];
		dest[o++] = buf[2];
		dest[o++] = buf[3];
	    }
	}
    }

    dest[o] = '\0';

    return o;
}



int EscapeTclString(char *dest, char const *source, int n)
{
    int i;
    int o = 0;

    for(i = 0; i < n; ++i) {
	switch(source[i]) {
	    case '\\': dest[o++] = '\\'; dest[o++] = '\\'; break;
	    case '\a': dest[o++] = '\\'; dest[o++] = 'a';  break;
	    case '\b': dest[o++] = '\\'; dest[o++] = 'b';  break;
	    case '\f': dest[o++] = '\\'; dest[o++] = 'f';  break;
	    case '\n': dest[o++] = '\\'; dest[o++] = 'n';  break;
	    case '\r': dest[o++] = '\\'; dest[o++] = 'r';  break;
	    case '\t': dest[o++] = '\\'; dest[o++] = 't';  break;
	    case '\v': dest[o++] = '\\'; dest[o++] = 'v';  break;
	    case '[':  dest[o++] = '\\'; dest[o++] = '[';  break;
	    case ']':  dest[o++] = '\\'; dest[o++] = ']';  break;
	    case '$':  dest[o++] = '\\'; dest[o++] = '$';  break;
	    case '"':  dest[o++] = '\\'; dest[o++] = '\"'; break;

	    default:
	    if(isprint(source[i]))
		dest[o++] = source[i];
	    else {
		char buf[5];
		sprintf(buf, "\\%03o", (unsigned int)(unsigned char)source[i]);
		dest[o++] = buf[0];
		dest[o++] = buf[1];
		dest[o++] = buf[2];
		dest[o++] = buf[3];
	    }
	}
    }

    dest[o] = '\0';

    return o;
}


void MangleName(char *dest, char const *proc_name)
{
    int   proc_name_length = strlen(proc_name);
    int   add_count = 0;
    char  buffer[32];
    int   i;

    dest[0] = '\0';

    for(i = 0; i < proc_name_length; ++i) {
	if(add_count >= 58) {
	    unsigned int hash = 0;

	    for(; i < proc_name_length; ++i) {
		hash = (hash << 13) | (hash >> 19) | 
		    ((unsigned int) proc_name[i]);
	    }

	    sprintf(buffer, "%08x", hash);
	    strcat(dest, buffer);
	    add_count += 8;
	    break;
	}

	if(proc_name[i] == '_' || isalpha(proc_name[i]))
	    dest[add_count++] = proc_name[i];
	else if(isdigit(proc_name[i])) {
	    if(add_count == 0)
		dest[add_count++] = '_';
	    dest[add_count++] = proc_name[i];
	}
	else {
	    dest[add_count++] = '_';
	}
    }

    dest[add_count] = '\0';

    return;
}
