/*
 * t4storage.cpp --
 *
 *	This file implements the command procedure invoked when the
 *	Tcl procedure "t4graph::storage" is called.
 *
 *	Authors: Jacob Levy and Jean-Claude Wippler.
 *		 jyl@best.com	jcw@equi4.com
 *
 * Copyright (c) 2000-2003, JYL Software Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE, EVEN IF
 * JYL SOFTWARE INC. IS MADE AWARE OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "t4graphrep.h"

/*
 * The various sub-commands for "t4graph::storage":
 */

static CONST84 char *subCommands[] = {
    (char *) "kind",
    (char *) "close",
    (char *) "commit",
    (char *) "configure",
    (char *) "copyto",
    (char *) "delete",
    (char *) "dogc",
    (char *) "root",
    (char *) "isvalid",
    (char *) "isstable",
    (char *) "markunstable",
    (char *) "needsgc",
    (char *) "node",
    (char *) "vertex",
    (char *) "name",
    (char *) "foreach",
    (char *) "statistic",
    (char *) "callback",
    (char *) "get",
    (char *) "share",
    (char *) NULL
};
typedef enum SSubCommands {
    SKind = 0, 
    SClose,
    SCommit,
    SConfigure,
    SCopyTo,
    SDelete,
    SDoGC,
    SRoot,
    SIsValid,
    SIsStable,
    SMarkUnstable,
    SNeedsGC,
    SNode,
    SVertex,
    SName,
    SForeach,
    SStatistic,
    SCallback,
    SGet,
    SShare
} SSubCommands;

/*
 * StorageCmdProc --
 *
 *	This procedure is invoked when the "tgraph::storage" command
 *	is called.
 *
 * Results:
 *	A standard Tcl result. The interpreter result may contain various
 *	results depending on the sub-command selected.
 *
 * Side effects:
 *	Whatever the selected sub-command does.
 */

int
T4Graph_StorageCmdProc(ClientData cd,
		       Tcl_Interp *interp,
		       int objc,
		       Tcl_Obj *CONST objv[])
{
    SSubCommands index;
    T4Storage *s;

    /*
     * There must be at least three arguments.
     */

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 0, NULL,
			 (char *) "$storage cmd ?arg ...?");
        return TCL_ERROR;
    }

    /*
     * The client data is the T4Storage for this instance.
     */

    s = (T4Storage *) cd;

    /*
     * Figure out what operation was requested. No default is provided,
     * so the caller must select an operation.
     */

    if (Tcl_GetIndexFromObj(interp, objv[1],
			    (CONST84 char **) subCommands,
			    (char *) "cmd",
			    0, (int *) &index)
	!= TCL_OK) {
	return TCL_ERROR;
    }

    objc -= 2;
    objv += 2;

    /*
     * Invoke the requested operation:
     */

    switch (index) {
    case SKind:
	Tcl_SetStringObj(Tcl_GetObjResult(interp), (char *) "storage", -1);
	return TCL_OK;
    case SClose:
	return s->Close(interp, objc, objv);
    case SCommit:
	return s->Commit(interp, objc, objv);
    case SConfigure:
	return s->Configure(interp, objc, objv);
    case SCopyTo:
	return s->CopyTo(interp, objc, objv);
    case SDelete:
	return s->Delete(interp, objc, objv);
    case SDoGC:
	return s->DoGC(interp, objc, objv);
    case SRoot:
	return s->Root(interp, objc, objv);
    case SIsValid:
	return s->IsValid(interp, objc, objv);
    case SIsStable:
	return s->IsStable(interp, objc, objv);
    case SMarkUnstable:
	return s->MarkUnstable(interp, objc, objv);
    case SName:
	return s->Name(interp, objc, objv);
    case SNeedsGC:
	return s->NeedsGC(interp, objc, objv);
    case SNode:
	return s->Node(interp, objc, objv);
    case SVertex:
	return s->Vertex(interp, objc, objv);
    case SForeach:
	return s->Foreach(interp, objc, objv);
    case SStatistic:
	return s->Statistic(interp, objc, objv);
    case SCallback:
	return s->Callback(interp, objc, objv);
    case SGet:
	return s->Get(interp, objc, objv);
    case SShare:
	return s->Share(interp, objc, objv);
    default:
	Tcl_AppendStringsToObj(Tcl_GetObjResult(interp),
			       "StorageCmdProc: unreachable code!",
			       (char *) NULL);
	return TCL_ERROR;
    }
}
