# resources.tcl --
#
#	XSLT extension providing access to resources.
#
# Copyright (c) 2001 Zveno Pty Ltd
# http://www.zveno.com/
#
# $Id: resources.tcl,v 1.1 2002/05/23 10:02:46 balls Exp $

package provide resources 1.0

namespace eval resources {
    namespace export list type exists
}

# resources::list --
#
#	List the resources available at a given location
#
# Arguments:
#	locn	Resource path to list
#	basedir	Base directory
#	args	not needed
#
# Results:
#	Returns list of resources

proc resources::list {locn {base {}} args} {
    # What kind of resource is this?  file, http, ftp, etc?

    if {[llength $args]} {
	return -code error "too many arguments"
    }

    if {[string match /* $base]} {
	regsub {^(/)} $locn {} locn
    }

    set result {}
    foreach entry [glob -nocomplain [file join $base $locn *]] {
	lappend result [file tail $entry]
    }

    return $result
}

# resources::type --
#
#	Gives the type of the resource
#
# Arguments:
#	locn	Resource path to type
#	args	not needed
#
# Results:
#	Returns string describing resource

proc resources::type {locn args} {

    if {[llength $args]} {
	return -code error "too many arguments"
    }

    if {[file isdir $locn]} {
	return directory
    } elseif {[file isfile $locn]} {
	return file
    } else {
	return other
    }
}

# resources::exists --
#
#	Check whether a resource exists
#
# Arguments:
#	locn	Resource path to type
#	args	not needed
#
# Results:
#	Returns boolean

proc resources::exists {locn args} {

    if {[llength $args]} {
	return -code error "too many arguments"
    }

    if {[file exists $locn]} {
	return 1
    } else {
	return 0
    }
}

# resources::mkdir --
#
#	Create a directory hierarchy.
#
# Arguments:
#	locn	Resource path for directory
#	args	not needed
#
# Results:
#	Returns directory created or empty string if unsuccessful

proc resources::mkdir {locn args} {

    if {[llength $args]} {
	return {}
    }

    set dir [file split $locn]
    set current [lindex $dir 0]
    set remaining [lrange $dir 1 end]
    while {[llength $remaining]} {
	set current [file join $current [lindex $remaining 0]]
	set remaining [lrange $remaining 1 end]
	if {[file exists $current]} {
	    if {![file isdir $current]} {
		return {}
	    }
	} else {
	    file mkdir $current
	}
    }

    return $locn
}

# resources::copy --
#
#	Copy a resource.
#
# Arguments:
#	src	Resource to copy
#	dest	Destination for resource
#	args	not needed
#
# Results:
#	Resource copied

proc resources::copy {src dest args} {
    if {[catch {file copy $src $dest}]} {
	return 0
    } else {
	return 1
    }
}
