/* 
 * mxFileMgr.c --
 *
 *	This file provides the top-level procedures for managing information
 *	about the file being edited.
 *
 * Copyright (C) 1986, 1987, 1988 Regents of the University of California
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * Copyright (c) 1992 Xerox Corporation.
 * Use and copying of this software and preparation of derivative works based
 * upon this software are permitted. Any distribution of this software or
 * derivative works must comply with all applicable United States export
 * control laws. This software is made available AS IS, and Xerox Corporation
 * makes no warranty about the software, its performance or its conformity to
 * any specification.
 */

#ifndef lint
static char rcsid[] = "$Header: /import/tcl7/src/mxedit/RCS/mxFileMgr.c,v 2.4 1994/02/01 18:16:45 welch Exp $ SPRITE (Berkeley)";
#endif not lint

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined(__hpux) | defined(SVR4)
#include <unistd.h>
#define getwd(arg)	getcwd(arg, (size_t)1023)
#endif

#include "mxWidget.h"

/*
 * Library imports:
 */

char *getenv();

/*
 * The context below maps from X window ids to MxWidget structures:
 */

static XContext mxContext;
static int init = 0;

/*
 * Head of list of all MxFileInfo structures:
 */

static MxFileInfo *fileListPtr = NULL;

/*
 * Total count of number of files open.  When this reaches zero, it
 * may be time to exit the program.
 */

int mx_FileCount = 0;

/*
 * Procedures defined in this file but referenced before defined:
 */

void		MxDetachFileInfo();
void		MxRecycleFileInfo();

/*
 * Additional flags to MxSetupFileInfo, used only within this module
 * (primarily when resetting the window):
 */

#define NO_NAME_MATCH		0x800
#define IGNORE_LOG_EXISTS	0x1000

/*
 *----------------------------------------------------------------------
 *
 * Mx_Cleanup --
 *
 *	This routine may be called to perform "last-ditch" cleanup
 *	before the process dies.  Typically, it's invoked in response
 *	to fatal signals.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Undo logs are deleted if they correspond to clean.
 *
 *----------------------------------------------------------------------
 */

void
Mx_Cleanup()
{
    register MxFileInfo *fileInfoPtr;

    for (fileInfoPtr = fileListPtr; fileInfoPtr != NULL;
	    fileInfoPtr = fileInfoPtr->nextPtr) {
	if ((fileInfoPtr->file != NULL)
		&& !Mx_FileModified(fileInfoPtr->file)
		&& (fileInfoPtr->log != NULL)) {
	    Undo_LogDelete(fileInfoPtr->log);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MxSwitchFile --
 *
 *	Change the file being edited in an Mx window.
 *
 * Results:
 *	TCL_OK is returned if everything went OK.  If something happened
 *	such that the user requested that the window contents not be
 *	switched after all, then TCL_ERROR is returned, a pointer to
 *	an error message is left in mxwPtr->interp->result, and the
 *	window is not changed.
 *
 * Side effects:
 *	The window is modified so that it displays the file named
 *	"name".  The caller should make sure that the old file was
 *	saved, is such is desired, since if this was the only window
 *	on the old file its modifications will be lost.
 *
 *----------------------------------------------------------------------
 */

int
MxSwitchFile(mxwPtr, name)
    register MxWidget *mxwPtr;	/* Window of interest. */
    char *name;			/* Name of new file to appear in window. */
{
    int undoable;
    Mx_Position eof;
    int result;
    char newName[110];
    char msg[100], *statusString;
    MxFileInfo *oldFileInfoPtr;

    if (mxwPtr->fileInfoPtr->log != NULL) {
	undoable = MX_UNDO;
    } else {
	undoable = 0;
    }

    /*
     * Save around the old file information, then attempt to set things
     * up for the new file.  If it fails, restore the old information.
     */

    oldFileInfoPtr = mxwPtr->fileInfoPtr;
    MxDetachFileInfo(mxwPtr);
    result = MxSetupFileInfo(mxwPtr, name, (Mx_File) NULL, undoable,
	    mxwPtr->interp);
    if (result == TCL_OK) {
	MxCleanupRedisplay(mxwPtr);
	MxRecycleFileInfo(oldFileInfoPtr);
    } else {
	mxwPtr->nextPtr = oldFileInfoPtr->mxwPtr;
	oldFileInfoPtr->mxwPtr = mxwPtr;
	mxwPtr->fileInfoPtr = oldFileInfoPtr;
	return result;
    }
    MxSetupRedisplay(mxwPtr, mxwPtr->widthChars, mxwPtr->heightLines);
    Tcl_SetVar(mxwPtr->interp, "mxFile", name, 1);
    /*
     * Change the name of the registered TCL interpreter.
     */
    sprintf(newName, "mxedit %s", name);
    Tk_ReregWindow(mxwPtr->interp, mxwPtr->tkwin, newName);
    Tcl_SetVar(mxwPtr->interp, "mxInterpName",
		Tk_Name(mxwPtr->tkwin), TCL_GLOBAL_ONLY);
    MxManagerRecordSwitch(mxwPtr->interp, Tk_Name(mxwPtr->tkwin));

    eof = Mx_EndOfFile(mxwPtr->fileInfoPtr->file);
    statusString = "";
    if (access(mxwPtr->fileInfoPtr->name, W_OK) != 0) {
	if (errno == EACCES) {
	    statusString = " (read-only)";
	} else if (errno == ENOENT) {
	    statusString = " (new file)";
	}
    }
    sprintf(msg, "Loaded \"%.50s\":%s %d lines", 
	    mxwPtr->fileInfoPtr->name, statusString, eof.lineIndex+1);
    Tcl_SetResult(mxwPtr->interp, msg, TCL_VOLATILE);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * MxResetFile --
 *
 *	Throw away the current contents of a file, reload it from disk,
 *	and reset all windows using the file to use the current disk
 *	version.
 *
 * Results:
 *	TCL_OK is returned if everything went OK;  in this case a message
 *	about the file is returned in mxwPtr->interp->result.  If something
 *	happened such that the window contents could not be reset after
 *	all, then TCL_ERROR is returned, a pointer to an error message is
 *	left in mxwPtr->interp->result, and no windows are changed.
 *
 * Side effects:
 *	All windows on mxwPtr->fileInfoPtr are reset.
 *
 *----------------------------------------------------------------------
 */

int
MxResetFile(mxwPtr)
    MxWidget *mxwPtr;		/* Window on file to reset. */
{
    int undoable, result;
    Mx_File newFile;
    register MxWidget *mxwPtr2;
    MxFileInfo *oldFileInfoPtr, *newFileInfoPtr;
    Mx_Position eof;
    char *statusString;

    oldFileInfoPtr = mxwPtr->fileInfoPtr;
    if (oldFileInfoPtr->log != NULL) {
	undoable = MX_UNDO;
    } else {
	undoable = 0;
    }

    /*
     * Read in the disk version of the file.
     */

    newFile = Mx_FileLoad(oldFileInfoPtr->name);
    if (newFile == NULL) {
	if (errno != ENOENT) {
	    int errNum;
	    char msg[2000];

	    errNum = errno;
	    sprintf(mxwPtr->interp->result,
		    "cannot read \"%.50s\": %.100s",
		    oldFileInfoPtr->name, strerror(errNum));
	    return TCL_ERROR;
	}
	newFile = Mx_FileLoad((char *) NULL);
    }

    /*
     * Attempt to set things up for the new file.  If an error occurs,
     * then restore all of the windows on the file to their previous
     * contents.
     */

    newFileInfoPtr = NULL;
    for (mxwPtr2 = oldFileInfoPtr->mxwPtr; mxwPtr2 != NULL; ) {
	MxWidget *nextPtr = mxwPtr2->nextPtr;

	MxDetachFileInfo(mxwPtr2);
	result = MxSetupFileInfo(mxwPtr2, oldFileInfoPtr->name, newFile,
		undoable|NO_NAME_MATCH|IGNORE_LOG_EXISTS, mxwPtr->interp);
	if (result != TCL_OK) {
	    mxwPtr2->nextPtr = oldFileInfoPtr->mxwPtr;
	    oldFileInfoPtr->mxwPtr = mxwPtr2;
	    mxwPtr2->fileInfoPtr = oldFileInfoPtr;
	    if (newFileInfoPtr != NULL) {
		for (mxwPtr2 = newFileInfoPtr->mxwPtr; mxwPtr2 != NULL; ) {
		    nextPtr = mxwPtr2->nextPtr;
		    MxDetachFileInfo(mxwPtr2);
		    mxwPtr2->nextPtr = oldFileInfoPtr->mxwPtr;
		    oldFileInfoPtr->mxwPtr = mxwPtr2;
		    mxwPtr2->fileInfoPtr = oldFileInfoPtr;
		    mxwPtr2 = nextPtr;
		}
		MxRecycleFileInfo(newFileInfoPtr);
	    }
	    return result;
	}
	newFileInfoPtr = mxwPtr2->fileInfoPtr;
	mxwPtr2 = nextPtr;
    }

    /*
     * Everything was OK, so make a second pass through the windows
     * to reset their display information.
     */

    for (mxwPtr2 = newFileInfoPtr->mxwPtr; mxwPtr2 != NULL;
	    mxwPtr2 = mxwPtr2->nextPtr) {
	MxCleanupRedisplay(mxwPtr2);
	MxSetupRedisplay(mxwPtr2, mxwPtr2->widthChars, mxwPtr2->heightLines);
    }
    MxRecycleFileInfo(oldFileInfoPtr);
    eof = Mx_EndOfFile(mxwPtr->fileInfoPtr->file);
    statusString = "";
    if (access(mxwPtr->fileInfoPtr->name, W_OK) != 0) {
	if (errno == EACCES) {
	    statusString = " (read-only)";
	} else if (errno == ENOENT) {
	    statusString = " (new file)";
	}
    }
    sprintf(mxwPtr->interp->result, "re-loaded \"%.50s\"%s: %d lines", 
	    mxwPtr->fileInfoPtr->name, statusString, eof.lineIndex+1);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * MxSetupFileInfo --
 *
 *	This utility procedure establishes the MxFileInfo structure
 *	for a window.
 *
 * Results:
 *	TCL_OK is returned if everything went OK.  If for some reason
 *	the user requested that the new window not be opened after
 *	all, then TCL_ERROR is returned, a pointer to an error message
 *	is left in interp->result, and the fileInfoPtr field of mxwPtr is
 *	left NULL.
 *
 * Side effects:
 *	A MxFileInfo structure gets allocated for mxwPtr, if one
 *	doesn't already exist, and mxwPtr gets linked into its list
 *	of windows.
 *
 *----------------------------------------------------------------------
 */

int
MxSetupFileInfo(mxwPtr, name, file, flags, interp)
    register MxWidget *mxwPtr;	/* Window for which file-related information
				 * is to be set up.   Its fontPtr and w
				 * fields must be valid. */
    char *name;			/* Name of file to be associated with mxwPtr.
				 * If caller has already opened the file,
				 * so we're not supposed to muck with it,
				 * then this is NULL. */
    Mx_File file;		/* If the caller has already opened the
				 * file, this points to it.  Otherwise,
				 * this is NULL. */
    int flags;			/* Some OR'ed combination of the flag bits
				 * MX_UNDO, MX_DELETE, NO_NAME_MATCH,
				 * and IGNORE_LOG_EXISTS. */
    Tcl_Interp *interp;		/* Interpreter to use for error reporting. */
{
    register MxFileInfo *fileInfoPtr;
    int option, result;
    Tcl_DString string;

    /*
     * Handle tilde's in name.
     */

    if (name != NULL) {
	name = Tcl_TildeSubst(interp, name, &string);
	if (name == NULL) {
	    mxwPtr->fileInfoPtr = NULL;
	    return TCL_ERROR;
	}
    } else {
	Tcl_DStringInit(&string);
    }

    /*
     * Create an MxFileInfo structure if there isn't one already.
     */

    for (fileInfoPtr = fileListPtr; fileInfoPtr != NULL;
	    fileInfoPtr = fileInfoPtr->nextPtr) {
	if ((fileInfoPtr->file == file)
		|| ((file == NULL) && (name != NULL)
		&& !(flags & NO_NAME_MATCH)
		&& (strcmp(name, fileInfoPtr->name) == 0))) {
	    file = fileInfoPtr->file;
	    break;
	}
    }
    if (fileInfoPtr == NULL) {
	fileInfoPtr = (MxFileInfo *) malloc(sizeof(MxFileInfo));
	if (file == NULL) {
	    fileInfoPtr->file = file = Mx_FileLoad(name);
	    if (fileInfoPtr->file == NULL) {
		if (errno != ENOENT) {
		    int errNum;
		    char msg[2000];

		    errNum = errno;
		    free((char *) fileInfoPtr);
		    mxwPtr->fileInfoPtr = NULL;
		    sprintf(interp->result,
			    "cannot read \"%.50s\": %.100s",
			    name, strerror(errNum));
		    Tcl_DStringFree(&string);
		    return TCL_ERROR;
		}
		fileInfoPtr->file = Mx_FileLoad((char *) NULL);
	    }
	} else {
	    fileInfoPtr->file = file;
	}
	if (name == NULL) {
	    name = "";
	}
	fileInfoPtr->name = (char *)
		malloc((unsigned) (strlen(name) + 1));
	strcpy(fileInfoPtr->name, name);
	fileInfoPtr->lastMod = MxLastMod(name);
	fileInfoPtr->log = NULL;
	fileInfoPtr->hlPtr = NULL;
	fileInfoPtr->mxwPtr = NULL;
	fileInfoPtr->nextPtr = fileListPtr;
	fileInfoPtr->flags = 0;
	if (flags & MX_DELETE) {
	    fileInfoPtr->flags |= DELETE;
	}
	fileInfoPtr->caretFirst = Mx_ZeroPosition;
	fileInfoPtr->caretLast = Mx_ZeroPosition;
	fileInfoPtr->caretWindow = NULL;
	(void) Mx_FloaterCreate(fileInfoPtr->file,
	    &fileInfoPtr->caretFirst, &fileInfoPtr->caretLast);
	(void) Mx_SpyCreate(fileInfoPtr->file, MX_BEFORE,
		MxCaretRedisplayText, (ClientData) fileInfoPtr);
	fileListPtr = fileInfoPtr;
	fileInfoPtr->openParenPtr = NULL;
	fileInfoPtr->closeParenPtr = NULL;
	mx_FileCount++;
    }
    mxwPtr->nextPtr = fileInfoPtr->mxwPtr;
    fileInfoPtr->mxwPtr = mxwPtr;
    mxwPtr->fileInfoPtr = fileInfoPtr;

    /*
     * Create an undo log if one is requested and there isn't
     * already one for the file.
     */

    if ((fileInfoPtr->log == NULL) && (flags & MX_UNDO)
	    && (name != NULL)) {
	char oldLogName[UNDO_NAME_LENGTH], logName[UNDO_NAME_LENGTH];
	char idLine[UNDO_ID_LENGTH];

	result = Undo_FindLog(name, logName, idLine, oldLogName, interp);
	if (result != TCL_OK) {
	    char *result;
	    char msg[200];
	    sprintf(msg, "[list %s %s %s", interp->result,
		    "  I can continue, but",
		    " without undoing or crash recovery.]");
	    if (Tcl_VarEval(mxwPtr->interp, "mxUndoErrorDialog ",
					msg, NULL) == TCL_ERROR) {
		fprintf(stderr, "mxUndoErrorDialog failed: %s\n",
			mxwPtr->interp->result);
		goto abort;
	    }
	    if (strcmp(mxwPtr->interp->result, "abort") == 0) {
		goto abort;
	    }
	    Tcl_Return(interp, (char *) NULL, TCL_STATIC);
	    oldLogName[0] = 0;
	} else {
	    fileInfoPtr->log = Undo_LogCreate(fileInfoPtr, logName,
		    idLine);
	    if (fileInfoPtr->log == NULL) {
		static char msg[100];
		unlink(logName);
		sprintf(msg, "[list Couldn't set up the undo log.  %s %s",
			"I can continue, but without undoing ",
			"or crash recovery.]");
		if (Tcl_VarEval(mxwPtr->interp, "mxUndoErrorDialog ",
					msg, NULL) == TCL_ERROR) {
		    fprintf(stderr, "mxUndoErrorDialog failed: %s\n",
			mxwPtr->interp->result);
		    goto abort;
		}
		if (strcmp(mxwPtr->interp->result, "abort") == 0) {
		    goto abort;
		}
	    }
	}
	if ((oldLogName[0] != 0) && !(flags & IGNORE_LOG_EXISTS)) {
	    char msg[300], unknown[20], *modTime, *userName;
	    struct stat atts;
	    struct passwd *pwd;
	    int i;

	    if (stat(oldLogName, &atts) == 0) {
		modTime = ctime(&atts.st_mtime);
		modTime[24] = 0;
		pwd = getpwuid((int) atts.st_uid);
		if (pwd == NULL) {
		    sprintf(unknown, "user %d", atts.st_uid);
		    userName = unknown;
		} else {
		    userName = pwd->pw_name;
		}
	    } else {
		modTime = userName = "??";
	    }
	    sprintf(msg, "[list There's a log file \"%.50s\" that describes \
edits made to \"%.50s\" by \"%s\" on %s.  This means that either \
someone's already editing the file \
(maybe you?), or the editor crashed during the middle of the last \
edit.]",
		    oldLogName, name, userName, modTime);
	    if (Tcl_VarEval(mxwPtr->interp, "mxRecoveryDialog ", msg, NULL)
			    == TCL_ERROR) {
		fprintf(stderr, "mxRecoveryDialog failed: %s\n",
			mxwPtr->interp->result);
		goto abort;
	    }

	    if (strcmp(mxwPtr->interp->result, "recover") == 0) {
		Tcl_Return(interp, (char *) NULL, TCL_STATIC);
		result = Undo_Recover(fileInfoPtr->file, oldLogName, interp);
		Undo_Mark(fileInfoPtr->log);
		if (result != TCL_OK) {
		    if (Tcl_VarEval(mxwPtr->interp, "mxRecoveryFailedDialog ",
				    mxwPtr->interp->result, NULL)
			    == TCL_ERROR) {
			fprintf(stderr, "mxRecoveryFailedDialog failed: %s\n",
				mxwPtr->interp->result);
			goto abort;
		    }
		    Tcl_Return(interp, (char *) NULL, TCL_STATIC);
		} else {
		    unlink(oldLogName);
		}
	    } else if (strcmp(mxwPtr->interp->result, "delete") == 0) {
		unlink(oldLogName);
	    } else if (strcmp(mxwPtr->interp->result, "abort") == 0) {
		goto abort;
	    }
	}
    }
    Tcl_DStringFree(&string);
    return TCL_OK;

    abort:
    Tcl_DStringFree(&string);
    MxDetachFileInfo(mxwPtr);
    MxRecycleFileInfo(fileInfoPtr);
    interp->result = "command aborted at your request";
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * MxDetachFileInfo --
 *
 *	Unlink a window from its MxFileInfo structure.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	MxwPtr will no longer be in the chain of windows attached to
 *	mxwPtr->fileInfoPtr.
 *
 *----------------------------------------------------------------------
 */

void
MxDetachFileInfo(mxwPtr)
    register MxWidget *mxwPtr;		/* Window to cut away from its file. */
{
    register MxFileInfo *fileInfoPtr;

    fileInfoPtr = mxwPtr->fileInfoPtr;
    if (fileInfoPtr == NULL) {
	return;
    }
    if (fileInfoPtr->mxwPtr == mxwPtr) {
	fileInfoPtr->mxwPtr = mxwPtr->nextPtr;
    } else {
	register MxWidget *mxwPtr2;
	for (mxwPtr2 = fileInfoPtr->mxwPtr; mxwPtr2 != NULL;
		mxwPtr2 = mxwPtr2->nextPtr) {
	    if (mxwPtr2->nextPtr == mxwPtr) {
		mxwPtr2->nextPtr = mxwPtr->nextPtr;
		break;
	    }
	}
    }
    mxwPtr->fileInfoPtr = NULL;
}


/*
 *----------------------------------------------------------------------
 *
 * MxRecycleFileInfo --
 *
 *	De-allocate everything in an MxFileInfo structure.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The stuff in fileInfoPtr gets released, if there are no
 *	windows referring to it anymore.
 *
 *----------------------------------------------------------------------
 */

void
MxRecycleFileInfo(fileInfoPtr)
    register MxFileInfo *fileInfoPtr;	/* File info to be cleaned up. */
{

    if (fileInfoPtr->mxwPtr != NULL) {
	return;
    }
    if (fileListPtr == fileInfoPtr) {
	fileListPtr = fileInfoPtr->nextPtr;
    } else {
	register MxFileInfo *fileInfoPtr2;
	for (fileInfoPtr2 = fileListPtr; fileInfoPtr2 != NULL;
		fileInfoPtr2 = fileInfoPtr2->nextPtr) {
	    if (fileInfoPtr2->nextPtr == fileInfoPtr) {
		fileInfoPtr2->nextPtr = fileInfoPtr->nextPtr;
		break;
	    }
	}
    }
    if (fileInfoPtr->name != NULL) {
	free((char *) fileInfoPtr->name);
    }
    if (fileInfoPtr->log != NULL) {
	Undo_LogDelete(fileInfoPtr->log);
    }
    while (fileInfoPtr->hlPtr != NULL) {
	MxHighlightDelete(fileInfoPtr->hlPtr);
    }
    if (fileInfoPtr->flags & DELETE) {
	Mx_FileClose(fileInfoPtr->file);
    }
    free((char *) fileInfoPtr);
    mx_FileCount--;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_CheckFileState --
 *
 *	This procedure loops through the files of the editor
 *	making sure their current state is properly displayed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	An TCL callback is made if the file's state has changed..
 *
 *----------------------------------------------------------------------
 */
void
Mx_CheckFileState()
{
    register MxFileInfo *infoPtr;
    register MxWidget *mxwPtr;		/* Window whose title is to be set. */
    int reTitle;

    for (infoPtr = fileListPtr; infoPtr != NULL; infoPtr = infoPtr->nextPtr) {
	/*
	 * Check to see if the file has just been modified for
	 * the first time, or if it has been undone in a way that
	 * totally reverses all modifications.  If so, update the
	 * title bar and icon title.
	 */

	reTitle = 0;
	if (Mx_FileModified(infoPtr->file)) {
	    if ((infoPtr->flags & MODIFIED) == 0) {
		reTitle = 1;
		infoPtr->flags |= MODIFIED;
	    }
	} else if (infoPtr->flags & MODIFIED) {
	    reTitle = 1;
	    infoPtr->flags &= ~MODIFIED;
	}

	for (mxwPtr = infoPtr->mxwPtr; mxwPtr != NULL;
		mxwPtr = mxwPtr->nextPtr) {
	    if (mxwPtr->flags & DESTROYED) {
		continue;
	    }
#ifdef notdef
	    /*
	     * TODO - figure out if this is the place that will get
	     * the caret displayed initially.  Cannot just use this
	     * code, however, as it screws up caret display.
	     */
	    if (mxwPtr->flags & NEEDS_UPDATE) {
		if (mxwPtr->flags & FOCUS_WINDOW) {
		    MxDisplayCaret(mxwPtr);
		}
	    }
#endif
	    if (reTitle) {
		MxWidgetStateChangeCallback(mxwPtr);
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_IsFileActive --
 *
 *	This procedure loops through the files of the editor
 *	to see if we are already editing this file.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
int
Mx_IsFileActive(filename)
    char *filename;
{
    register MxFileInfo *infoPtr;
    int doNameCompare = 0;
    struct stat stat1, stat2;

    if (stat(filename, &stat1) < 0) {
	doNameCompare = 1;
    }
    for (infoPtr = fileListPtr; infoPtr != NULL; infoPtr = infoPtr->nextPtr) {
	if (doNameCompare) {
	    if (strcmp(infoPtr->name, filename) == 0) {
		return 1;
	    }
	} else {
	    if (stat(infoPtr->name, &stat2) == 0) {
		if (stat1.st_dev == stat2.st_dev &&
		    stat1.st_ino == stat2.st_ino &&
		    stat1.st_size == stat2.st_size &&
		    stat1.st_mtime == stat2.st_mtime) {
		    return 1;
		}
	    }
	}
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_FileActiveCmd --
 *
 *	Tcl_Command layer on top of Mx_IsFileActive
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
int
Mx_FileActiveCmd(clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    char *filename;
    Tcl_DString string;
    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv[0], " \"filename\"",
                          (char *) NULL);
        return TCL_ERROR;
    }
    if (Mx_IsFileActive(Tcl_TildeSubst(interp,argv[1], &string))) {
	Tcl_SetResult (interp, "1", TCL_STATIC);
    } else {
	Tcl_SetResult (interp, "0", TCL_STATIC);
    }
    Tcl_DStringFree(&string);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_FileIdentifier --
 *
 *	Called to get a unique ID string for a file. This is used to
 *	determine if different editting sessions are on the same file.
 *
 * Results:
 *	Returns a string that corresponds uniquely to the
 *	file being editted.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
int
Mx_FileIdentifier(interp, name)
    Tcl_Interp *interp;
    char *name;
{
    char *pathname;
    Tcl_DString string;
    struct stat statBuf;
    char result[1024];
    if (name == NULL) {
	/* Scratch window */
	Tcl_SetResult(interp, "{}", TCL_STATIC);
	return TCL_OK;
    }
    pathname = Tcl_TildeSubst(interp, name, &string);
    if (stat(pathname, &statBuf) == 0) {
	sprintf(result, "%s [%d,%d,%x]", pathname,
		statBuf.st_dev, statBuf.st_ino, statBuf.st_mtime);
	Tcl_SetResult(interp, result, TCL_VOLATILE);
    } else {
	/*
	 * ADD IN CURRENT WORKING DIRECTORY PATHNAME
	 */
	if (pathname[0] != '/') {
	    char cwd[1024];
	    if (getwd(cwd) != 0) {
		strcpy(result, cwd);
		strcat(result, "/");
		strcat(result, pathname);
		pathname = result;
	    }
	}
	Tcl_SetResult(interp, pathname, TCL_VOLATILE);
    }
    Tcl_DStringFree(&string);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_FileIdentCmd --
 *
 *	This is the top-level procedure that implements the
 *	"mxFileIdent" command, which returns a string that uniquely
 *	identifies the file being editted.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_FileIdentCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if ((argc != 2)) {
	sprintf(interp->result,
		"wrong #/type of args: should be \"%.50s filename\"",
		argv[0]);
	return TCL_ERROR;
    }
    return Mx_FileIdentifier(interp, argv[1]);
}

