#
# mxedit.menus
#	Definitions for the menus used in the mxedit application.
#	This defines the following procedures that operate on menus
#	by their label name instead of by their TK widget name.
#	This makes it easier for users to understand customization.
#	mxMenu - define a menu by name
#	mxMenuAdd - add an entry to a menu
#	mxMenuBind - define a keyboard accelerator for a menu item.
#	mxMenuEntryConfigure - change a menu entry
#
# Copyright (c) 1992 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.

# Imported globals
# mxFile - the name of the current file

# Exported globals
# For each menu defined by mxMenu under label "Foo", there is a
# global variable called mxMenuFoo defined that holds the identity of
# the menu widget.  This is relied on by mxMenuAdd and mxMenuBind

# File globals
# mxMenubar - the frame that holds menubuttons

# mxMenuSetup --
# Called from mxsetup to create the mxMenuBar frame

proc mxMenuSetup { parent } {
    global mxMenuBar mxMenuList
    set mxMenuBar [buttonFrame $parent .buttons 2]
    set mxMenuList {}
    return $mxMenuBar
}

# mxCreateMenus --
#	This is invoked from mxsetup in order to fill out the mxMenuBar

proc mxCreateMenus { } {
    foreach menuProc { mxFileMenu mxEditMenu mxSearchMenu mxWindowMenu
		       mxFontMenu mxExtrasMenu mxHelpMenu  mxGeometryMenu} {
	if [catch $menuProc msg] {
	    puts stderr "$menuProc failed: $msg"
	}
    }
}

# mxMenu --
#	Add a new menu and associated menubutton to the mxedit mxMenuBar.
#	The name of the menu widget is remembered for later use
#	with mxMenuBind and mxMenuBind

proc mxMenu { label {where {left}} } {
    global mxMenuBar mxMenuList
    set name .mb${label}
    set menuName $mxMenuBar$name.menu
    packedMenuButton $mxMenuBar $name $label $menuName $where
    set menu [basicMenu $menuName]

    # Remember the widget name under a variable derived from the label.
    # This allows mxMenuBind to be passed the label instead of the widget.
    global mxMenu${label}
    set mxMenu${label} $menu

    lappend mxMenuList $mxMenuBar$name
    eval tk_menuBar $mxMenuBar $mxMenuList

    return $menu
}
# mxMenuBind --
# Bind a keystroke sequence to a menu entry.
# Use this procedure in order to keep the menus up-to-date
# with keyboard accelerators.
# This hardwires the binding to the mxedit widget.

proc mxMenuBind { sequence menuName label } {
    global mxedit
    global mxMenu${menuName}
    set menu [set mxMenu${menuName}]

    if [catch {
	set command [lindex [$menu entryconfigure $label -command] 4]
	bind $mxedit $sequence $command
	$menu entryconfigure [$menu index $label] -accelerator $sequence
    } msg] {
	mxFeedback "mxMenuBind $sequence $menuName $label: $msg"
    }
}

# mxMenuUnBind --
# Remove a binding from a menu entry
# Use this procedure in order to keep the menus up-to-date
# with keyboard accelerators.

proc mxMenuUnBind { menuName label } {
    global mxedit
    global mxMenu${menuName}
    set menu [set mxMenu${menuName}]

    if [catch {
	set sequence [lindex [$menu entryconfigure $label -accelerator] 4]
	bind $mxedit $sequence {}
	$menu entryconfigure [$menu index $label] -accelerator {}
    } msg] {
	mxFeedback "mxMenuUnBind $menuName $label: $msg"
    }
}

# mxMenuUnBindAll --
# Remove all bindings from a menu
# Use this when changing key binding sets

proc mxMenuUnBindAll { {menuName _ALL} } {
    global mxedit

    if {$menuName != "_ALL"} {
	global mxMenu${menuName}
	set menu [set mxMenu${menuName}]
	set menuList [list $menu]
    } else {
	global mxMenuList
	foreach item $mxMenuList {
	    lappend menuList ${item}.menu
	}
    }
    foreach menu $menuList {
	set numEntries [$menu index last]
	for {set i 0} {$i < $numEntries} {incr i} {
	    catch {
		set sequence [lindex [$menu entryconfigure $i -accelerator] 4]
		bind $mxedit $sequence {}
		$menu entryconfigure [$menu index $i] -accelerator {}
	    }
	}
    }
}

# mxMenuAdd --
#	Add an item to a menu.

proc mxMenuAdd { menuName label command } {
    global mxMenu${menuName}
    set menu [set mxMenu${menuName}]
    if [catch {$menu add command -label $label -command $command} msg] {
	mxFeedback "menu add $menu \"$label\" failed: $msg"
    }
}

# mxMenuAddRadio --
#	Add a radio-button item to a menu.

proc mxMenuAddRadio { menuName label var value { command  {info library} } } {
    global mxMenu${menuName}
    set menu [set mxMenu${menuName}]
    if [catch {$menu add radiobutton -label $label -command $command -value $value -variable $var} msg] {
	mxFeedback "menu add $menu \"$label\" failed: $msg"
    }
}

# mxMenuAddSeparator --
#	Add a separator to a menu.

proc mxMenuAddSeparator { menuName } {
    global mxMenu${menuName}
    set menu [set mxMenu${menuName}]
    if [catch {$menu add separator} msg] {
	mxFeedback "menu add separator $menu" failed: $msg"
    }
}

# mxMenuEntryConfigure --
#	Change an item in a menu.

proc mxMenuEntryConfigure { menuName label args } {
    global mxMenu${menuName}
    set menu [set mxMenu${menuName}]
    if [catch [concat $menu entryconfigure \"$label\" $args] msg] {
	mxFeedback "menu entryconfigure $menu \"$label\" failed: $msg"
    }
}

# mxMenuInsert --
#	Add an entry in a particular location (not just at the end)
#	This isn't supported directly by the menu implementation,
#	so we have to save the entries after the desired
#	location, delete them, add the new entry, and then
#	add the saved entries.

proc mxMenuInsert { menuName index label command } {
    global mxMenu${menuName}
    set menu [set mxMenu${menuName}]
    if [catch {$menu index 1000} numEntries] {
	mxFeedback "menu index 1000 failed: $msg"
	return
    }
    if {$index > $numEntries} {
	mxMenuAdd $menuName $label $command
	return
    }
    for {set i $index} {$i <= $numEntries} {incr i} {
	set conf($i) [$menu entryconfigure $i]
    }
    $menu delete $index $numEntries
    catch {mxMenuAdd $menuName $label $command}

    for {set i $index} {$i <= $numEntries} {incr i} {
	set cmd {}
	foreach thing $conf($i) {
	    set flag [lindex $thing 0]
	    set value [lindex $thing 4]
	    if {$value != {} } {
		lappend cmd $flag $value
	    }
	}
	if {$cmd == {}} {
	    # Implies separator
	    $menu add separator
	} else {
	    eval [concat [list $menu add command] $cmd]
	}
    }
}
# mxMenuDelete --
#	Delete an item from a menu.
#
proc mxMenuDelete { menuName label } {
    global mxMenu${menuName}
    set menu [set mxMenu${menuName}]
    if [catch {$menu delete -label $label} msg] {
	mxFeedback "menu delete $menu failed: $msg"
    }
}

# mxMenuDestroy --
#	Nuke a menu entirely
#
proc mxMenuDestroy {label} {
    global mxMenuBar mxMenuList
    set name ${mxMenuBar}.mb${label}
    if [catch {destroy $name} msg] {
	mxFeedback "destroy $menu \"$label\" failed: $msg"
    }
    # on supprime le menu de la liste :
    set tmp [lsearch $mxMenuList $name]
    while { $tmp != -1} {
	set mxMenuList [lreplace $mxMenuList $tmp $tmp]
	set tmp [lsearch $mxMenuList $name]
    }
}

# mxFileMenu --
#	Define the FILE menu

proc mxFileMenu { } {
    mxMenu File
    mxMenuAdd File "Directory Browser"	{DirShowWindow}
    mxMenuAddSeparator File
    mxMenuAdd File "Save and quit" 	{mxSave ; mxQuit}
    mxMenuAdd File "Save" 		{mxSave}
    mxMenuAdd File "Save in file SEL" 	{mxSaveSel}
    mxMenuAdd File "Save as ..." 	{mxSaveAs}
    mxMenuAddSeparator File
    mxMenuAdd File "New (scratch)" 	{mxOpen {}}
    mxMenuAdd File "Open new window" 	{mxOpen $mxFile}
    mxMenuAdd File "Open file SEL" 	{mxApplyToSelection mxOpen} 
    mxMenuAddSeparator File
    mxMenuAdd File "Switch to file SEL" {mxApplyToSelection mxSwitch} 
    mxMenuAdd File "Switch to previous file" 	{mxSwitchBack}
    mxMenuAdd File "Switch to tag SEL" 	{mxApplyToSelection mxTag} 
    mxMenuAddSeparator File
    mxMenuAdd File "Open on tag SEL" 	{mxApplyToSelection mxTagOpen}
    mxMenuAdd File "Go to line SEL" 	{mxHistory next mxHistory \
						{mxApplyToSelection mxLine}}
    mxMenuAddSeparator File
    mxMenuAdd File "Reset" 		{mxReset}
    mxMenuAdd File "Quit" 		{mxQuit}
}

# mxEditMenu --
#	Define the Edit menu

proc mxEditMenu { } {
    mxMenu Edit
    mxMenuAdd Edit "Undo" 		{mxUndo}
    mxMenuAdd Edit "Do Again" 		{mxHistory ignore mxRedo}
    mxMenuAddSeparator Edit
    mxMenuAdd Edit "Copy SEL"		{mxCopySave}
    mxMenuAdd Edit "Cut SEL"		{mxDeleteSave}
    mxMenuAdd Edit "Delete SEL"		{mxDeleteNoSave}
    mxMenuAdd Edit "Paste"		{mxPaste}
    mxMenuAdd Edit "Move SEL"		{mxMoveSel}
    mxMenuAddSeparator Edit
    mxMenuAdd Edit "Indent line" 	{mxIndentLine}
    mxMenuAdd Edit "Indent SEL"		{mxIndentSel}
    mxMenuAdd Edit "Outdent line" 	{mxOutdentLine}
    mxMenuAdd Edit "Outdent SEL" 	{mxOutdentSel}
    mxMenuAddSeparator Edit
    mxMenuAdd Edit "Cut eol"		{emacsKill line}
    mxMenuAdd Edit "Cut char"		{emacsKill char}
    mxMenuAdd Edit "Yank"		{emacsYank}
    mxMenuAdd Edit "Yank-pop"		{emacsYankPop}
    mxMenuAdd Edit "Reset Kill Bufs"	{emacsKillReset}
}

# mxSearchMenu --
proc mxSearchMenu { } {
    mxMenu Search
    mxMenuAdd Search "Forward" 		{mxFindInner forward}
    mxMenuAdd Search "Forward for SEL" 	{mxFindInner forwSel}
    mxMenuAdd Search "Backward" 	{mxFindInner backward} 
    mxMenuAdd Search "Backward for SEL"	{mxFindInner backSel}
    mxMenuAdd Search "Replace" 		{mxFindInner replace} 
    mxMenuAdd Search "Replace in SEL" 	{mxFindInner replaceSel}
    mxMenuAdd Search "Replace Everywhere" {mxFindInner replaceEverywhere}
}

# mxWindowMenu --
proc mxWindowMenu { } {
    mxMenu Window

    mxCommandMenuEntry Window

    mxMenuAdd Window "Search" 		{mxFind}
}

# mxHelpMenu --
proc mxHelpMenu { } {
    mxMenu Help
    mxMenuAdd Help "Quick Intro"		{mxHelp help}
    mxMenuAdd Help "Tutorial"			{mxHelp tutorial}
    mxMenuAddSeparator Help
    mxMenuAddRadio Help "Default key bindings" mxKeyBindingProc mxKeyBindings
    mxMenuAddRadio Help "Emacs key bindings" mxKeyBindingProc emacsKeyBindings
    mxMenuAddRadio Help "Personal key bindings" mxKeyBindingProc userKeyBindings
    mxMenuAddSeparator Help
    mxMenuAddRadio Help "Left-Right selection" mxSelectBindingProc mxSelectionBindings
    mxMenuAddRadio Help "Openwin selection" mxSelectBindingProc mxOpenwinSelectionBindings
    mxMenuAddRadio Help "Left-drag selection" mxSelectBindingProc mxMotifSelectionBindings
    mxMenuAddRadio Help "Personal selection" mxSelectBindingProc userSelectionBindings
    mxMenuAddSeparator Help
    mxMenuAddRadio Help "Middle scroll" mxScrollBindingProc mxScrollBindings
    mxMenuAddRadio Help "Openwin (right) scroll" mxScrollBindingProc	mxOpenwinScrollBindings
    mxMenuAddRadio Help "Shift-left scroll" mxScrollBindingProc	mxOldScrollBindings
    mxMenuAddRadio Help "Personal scroll" mxScrollBindingProc	userScrollBindings
}
# mxHelp --
#	Pop up a new window to display a few key bindings.

proc mxHelp { args } {
    global mxLibrary

    if {[llength $args] == 0} {
	set hfile mxedit.help
    } else {
	set hfile mxedit.[lindex $args 0]
    }
    if [file exists $mxLibrary/$hfile] {
	mxFeedback "Opening help window ($hfile)" ; update
	set newWindow [mxOpen {}]
	send $newWindow "mxRead $mxLibrary/$hfile"
	send $newWindow mxClean
	send $newWindow {mxSee 0.0}
	mxFeedback "Help window opened" ; update
    } else {
	mxFeedback "Cannot find help file for $hfile"
    }
}

# These two procedures change key bindings and swap the
# menu entry to reverse the process
proc mxEmacsKeyBindings {} {
    emacsBindings
    mxMenuEntryConfigure Help "Set emacs key bindings" \
	-command { mxDefaultKeyBindings } \
	-label "Set default key bindings"
}
proc mxDefaultKeyBindings {} {
    global mxedit
    mxKeyBindings $mxedit
    mxMenuEntryConfigure Help "Set default key bindings" \
	-command { mxEmacsKeyBindings } \
	-label "Set emacs key bindings"
}

# These two procedures change mouse bindings and swap the
# menu entry to reverse the process
proc mxSetMotifMouseBindings {} {
    global mxedit
    mxMotifSelectionBindings $mxedit
    mxMenuEntryConfigure Help "Set motif mouse bindings" \
	-command { mxDefaultMouseBindings } \
	-label "Set default mouse bindings"
}
proc mxDefaultMouseBindings {} {
    global mxedit
    mxSelectionBindings $mxedit
    mxMenuEntryConfigure Help "Set default mouse bindings" \
	-command { mxSetMotifMouseBindings } \
	-label "Set motif mouse bindings"
}

# mxFontMenu --
proc mxFontMenu { } {
    mxMenu Font
    mxMenuAdd Font "fixed"			{mxFont fixed}
    mxMenuAdd Font "6x10" 			{mxFont 6x10}
    mxMenuAdd Font "7x13" 			{mxFont 7x13}
    mxMenuAdd Font "9x15" 			{mxFont 9x15}
    mxMenuAdd Font "times8" 			{mxFont *times*-r-*-80*}
    mxMenuAdd Font "times12" 			{mxFont *times*-r-*-120*}
    mxMenuAdd Font "times14" 			{mxFont *times*-r-*-140*}
    mxMenuAdd Font "times18" 			{mxFont *times*-r-*-180*}
    mxMenuAdd Font "times24" 			{mxFont *times*-r-*-240*}
}

# mxGeometryMenu --
proc mxGeometryMenu { } {
    mxMenu Right {right}
    mxMenu Left {right}
    mxMenuAdd Right "Upper Right"	{mxUpperRight}
    mxMenuAdd Right "Lower Right"	{mxLowerRight}
    mxMenuAdd Right "Full Right"	{mxFullRight}

    mxMenuAdd Left "Upper Left"		{mxUpperLeft}
    mxMenuAdd Left "Lower Left"		{mxLowerLeft}
    mxMenuAdd Left "Full Left"		{mxFullLeft}
}
# mxUpperRight --
proc mxUpperRight { } {
    mxSetupTilingGeometry charsWide linesHigh right left top bottom
    mxGeometry ${charsWide}x${linesHigh}+${right}+${top}
}
# mxLowerRight --
proc mxLowerRight { } {
    mxSetupTilingGeometry charsWide linesHigh right left top bottom
    mxGeometry ${charsWide}x${linesHigh}+${right}+${bottom}
}
# mxUpperLeft --
proc mxUpperLeft { } {
    mxSetupTilingGeometry charsWide linesHigh right left top bottom
    mxGeometry ${charsWide}x${linesHigh}+${left}+${top}
}
# mxLowerLeft --
proc mxLowerLeft { } {
    mxSetupTilingGeometry charsWide linesHigh right left top bottom
    mxGeometry ${charsWide}x${linesHigh}+${left}+${bottom}
}
# mxFullRight --
proc mxFullRight { } {
    global mxRightMenu
    mxSetupFullGeometry Right \
		charsWide linesHigh right left top
    mxGeometry ${charsWide}x${linesHigh}+${right}+${top}
}
# mxFullLeft --
proc mxFullLeft { } {
    global mxLeftMenu
    mxSetupFullGeometry Left charsWide linesHigh right left top
    mxGeometry ${charsWide}x${linesHigh}+${left}+${top}
}

# mxSetupTilingGeometry
#	This uses the size of the screen and the size of the window to
#	figure out how to place the window in different tiled locations

proc mxSetupTilingGeometry { charsWide linesHigh right left top bottom } {
    upvar 1 $charsWide wide $linesHigh high $left L $right R $top T $bottom B

    scan [wm geometry .] "%dx%d" wide high
    scan [winfo geometry .] "%dx%d+%d+%d" mainWidth mainHeight xoff yoff

    set L 0
    set T 0
    if {[screenwidth] > (2 * $mainWidth)} {
	set R [expr {10 + $mainWidth}]
    } else {
	set R [expr {[screenwidth] - $mainWidth}]
    }

    if {[screenheight] > (2 * ($mainHeight+20))} {
	set B [expr {40 + $mainHeight}]
    } else {
	set B [expr {[screenheight] - $mainHeight}]
    }
}

# mxSetupFullGeometry
#	This uses the size of the screen and the size of the window to
#	figure out how to make a full sized window

proc mxSetupFullGeometry { menuName charsWide linesHigh right left top } {
    upvar 1 $charsWide wide $linesHigh high $left L $right R $top T

    # Happily, the wm grid command reports the original size of the window,
    # even if it has been resized.  winfo, happily, returns the current size
    # so that the following computation alternately computes a full sized
    # and original sized window
    scan [wm grid .] "%d %d %d %d" wide origHigh xinc yinc
    scan [winfo geometry .] "%dx%d+%d+%d" trueWidth trueHeight xoff yoff
    set extraHeight [expr {$trueHeight - ($yinc * $origHigh)}]
    set availHeight [expr {[screenheight] - $extraHeight}]
    set high [expr {($availHeight / $yinc) - 1}]
    set L 0
    set T 0
    if {[screenwidth] > (2 * $trueWidth)} {
	set R [expr {10 + $trueWidth}]
    } else {
	set R [expr {[screenwidth] - $trueWidth}]
    }
    # Fix up the menu so it reflects what will happen next
    # Remember that menuName is "Right" or "Left", and that
    # the value of menuLeft is the menu under the Left label...

    global mxMenu${menuName}
    set menu [set mxMenu${menuName}]

    if {$high > $origHigh} {
	mxMenuEntryConfigure Left "Full Left" -label "Small Left"
	mxMenuEntryConfigure Right "Full Right" -label "Small Right"
    } else {
	mxMenuEntryConfigure Left "Small Left" -label "Full Left"
	mxMenuEntryConfigure Right "Small Right" -label "Full Right"
    }
}

# mxExtrasMenu
#	A menu full of add-on features.  Many of these add their own menu

proc mxExtrasMenu {} {
    mxMenu Extras
    mxMenuAdd Extras "Tcl Eval"			{TclInit}
    mxMenuAdd Extras "MH Support"		{MhInit}
    mxMenuAdd Extras "Show session database"	{mxShowDbUser}
    mxMenuAdd Extras "Show key bindings" 	{mxShowBindings}
    mxMenuAdd Extras "Show variables" 		{mxShowVars}
    mxMenuAdd Extras "Show procedures" 		{mxShowProcs}
    mxMenuAdd Extras "Show cur history"		{mxShowHistory current}
    mxMenuAdd Extras "Show last history"	{mxShowHistory last}
    mxMenuAdd Extras "Show kill buffers"	{mxShowKillBuffers}
}
