/***********************************************************************
 *
 * Copyright (c) 1994 John F. Sefler.
 * All rights reserved.
 *
 ***********************************************************************
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written
 * agreement is hereby granted, provided the above copyright notice and
 * the following two paragraphs appear in all copies of this software.
 *
 * IN NO EVENT SHALL JOHN F. SEFLER BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF HE HAS
 * BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * JOHN F. SEFLER SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
 * "AS IS" BASIS, AND JOHN F. SEFLER HAS NO OBLIGATION TO PROVIDE
 * MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 **********************************************************************/

/***********************************************************************
 *
 * TITLE:	3D Maze Generator and Ghost Dispatcher
 * FILE:	maze.c
 * PURPOSE:	This file contains the C source code that generates
 *		3D mazes.  It also manages the position of ghosts
 *              throughout the maze.  Output from this module is in the
 *		form of new Tcl commands (See the functions that begin
 *		with TkMz_). Input to this module is through Tcl procs
 *		appended with _CProc (grep on _CProc).
 *
 * AUTHOR:	John F. Sefler
 * DATE:	FALL 1994
 *
 **********************************************************************/

/*********************************************************** Includes */
#include <stdio.h>
#include <stdlib.h>     /* included for rand() and srand() functions  */
#include <time.h>       /* included for rand() function               */
#include <ctype.h>
#include "commands.h"

/********************************* Structures, Enumerations, #Defines */
#undef FALSE
#undef TRUE

/* used to avoid infinite loops */
#define LOOP_CONVERGENCE 750

typedef enum 
{
  NO    = 0,
  YES   = 1,
  FALSE = 0,
  TRUE  = 1 
}BOOL;

typedef struct
{
  float x;
  float y;
  float z;
}PositionStruct, *Position;

typedef enum 
{
  MISSING,
  DOWN,
  UP
}WallState;

typedef enum 
{
  NORTH   = 0,
  SOUTH   = 1,
  WEST    = 2,
  EAST    = 3,
  FLOOR   = 4,
  CEILING = 5
}Direction;

typedef struct
{
  PositionStruct pos[4];
  WallState   state;
}WallStruct, *Wall;

typedef struct CellStruct
{
  int  lev;
  int  row;
  int  col;
  PositionStruct pos;
  BOOL visited;
  BOOL constructed;
  BOOL solution;
  BOOL elevator;
  BOOL ghost;
  BOOL start;
  BOOL finish;
  WallStruct wall[6];
  struct CellStruct *adj_cell[6];
}CellStruct, *Cell;
  
  
typedef struct
{
  int id;
  Cell cell;
  PositionStruct pos;
  int current_step;
  int num_steps_between_cells;
  float north_south_step_size;
  float west_east_step_size;
  float floor_ceiling_step_size;
  Direction dir;
}GhostStruct, *Ghost;

typedef struct
{
  int width;
  int height;
  int depth;
  int num_levs;
  int num_rows;
  int num_cols;
  int num_elevs;
  int num_ghosts;
  BOOL constructed;
  BOOL solved;
  GhostStruct  *ghost_array;
  CellStruct ***cell_matrix; /* A 3D matrix of Cells */
  Cell start_cell;
  Cell finish_cell;
}MazeStruct, *Maze;


/* This is it, the only maze. It contains everything! It's global. */
Maze gl_maze = NULL;



/********************************************************** Functions */

/* If even, return 1, if odd return 0 */
int IsIntEven(int num)
{
  if ((num % 2) == 0) return 1;
  else                return 0;
}

/* If odd, return 1, if even return 0 */
int IsIntOdd(int num)
{
  if (IsIntEven(num)) return 0;
  else                  return 1;
}

/* If even, return 1, if odd return 0 */
int IsFloatEven(float num)
{
  if (fabs(num - (int)num) > 0.00001) return 0;

  if (((int)num % 2) == 0) return 1;
  else                     return 0;
}

/* If odd, return 1, if even return 0 */
int IsFloatOdd(float num)
{
  if (IsFloatEven(num)) return 0;
  else                    return 1;
}

/* Returns a random number in range: lower <= ran_num <= upper. */
int GetRandomInt(int lower, int upper)
{       
   int ran_num;
   static BOOL seed_rand = YES;

   if(seed_rand == YES)  /* initializing the seed                     */
   {
      srand(time(0));    /* seeding rand() with the current time      */
      seed_rand = NO;    /* turns off the calling flag to seed rand() */
   }
                         /* lower <= ranm_num <= upper                */
   ran_num = (rand() % (upper - lower + 1)) + lower;

   return (ran_num);
}


/* Given a direction, return the opposite direction. (eg. NORTH -> SOUTH) */
Direction GetOppositeDirection(Direction dir)
{       
   /* this function assumes that opposite directions differ by 1 
      beginning with an even integer */
   if (IsIntEven((int)dir)) return ((Direction)(dir + 1));
   else                       return ((Direction)(dir - 1));
}


/* Reset the cell's visited flag to YES or NO in maze */
void ResetVisitedCellsInMaze(Maze maze, BOOL status)
{
  int lev, row, col;

  for (lev=0; lev<maze->num_levs; lev++)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      for (col=0; col<maze->num_cols; col++)
      {
        maze->cell_matrix[lev][row][col].visited = status;
      }
    }
  }
}

/* Reset the cell's solution flag to YES or NO in maze */
void ResetSolutionCellsInMaze(Maze maze, BOOL status)
{
  int lev, row, col;

  for (lev=0; lev<maze->num_levs; lev++)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      for (col=0; col<maze->num_cols; col++)
      {
        maze->cell_matrix[lev][row][col].solution = status;
      }
    }
  }
}

/* Set all solution cell's visited flag to YES in maze */
void SetSolutionCellsToVisitedCellsInMaze(Maze maze)
{
  int lev, row, col;

  for (lev=0; lev<maze->num_levs; lev++)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      for (col=0; col<maze->num_cols; col++)
      {
        if (maze->cell_matrix[lev][row][col].solution)
          maze->cell_matrix[lev][row][col].visited = YES;
      }
    }
  }
}

/* Return the number of adjacent cells marked as solution cells. */
int GetNumberOfAdjacentSolutionCells(Cell cell)
{
  int num_adjacent_solution_cells;
  int inx;

  num_adjacent_solution_cells = 0;
  for (inx = 0; inx < 6; inx++)
  {
    if (cell->wall[inx].state == DOWN)
    {
      if (cell->adj_cell[inx]->solution)
      {
        num_adjacent_solution_cells++;
      }
    }
  }

  return(num_adjacent_solution_cells);
}


/* Return an adjacent cell that has been marked as a solution cell. */
Cell GetAdjacentSolutionCell(Cell cell)
{
  int inx;

  for (inx = 0; inx < 6; inx++)
  {
    if (cell->wall[inx].state == DOWN)
    {
      if (cell->adj_cell[inx]->solution)
      {
        return(cell->adj_cell[inx]);
      }
    }
  }

  /* none of the adjacent cells are solution cells */
  return(NULL);
}

/* Return the number of accessable adjacent cells */
int GetNumberOfAdjacentAccessibleCells(Cell cell)
{
  int inx;
  int num_accessible = 0;

  for (inx = 0; inx < 6; inx++)
  {
    if (cell->wall[inx].state == DOWN)
    {
      num_accessible++;
    }
  }

  /* return the number of adjacent accessible cells */
  return(num_accessible);
}

/* Is this cell at a dead end? */
BOOL IsCellAtADeadEnd(Cell cell)
{
  /* at a dead end, there is only one accessible adjacent cell */
  if (GetNumberOfAdjacentAccessibleCells(cell) == 1)
  {
    return(TRUE);
  }
  else
  {
    return(FALSE);
  }
}

/* Is this cell at a cross path in the maze (ie. has a directional
   choice to advance) */
BOOL IsCellAtACrossPath(Cell cell)
{
  /* to be at a cross path, there must be more than 2 directions to go */
  if (GetNumberOfAdjacentAccessibleCells(cell) > 2)
  {
    return(TRUE);
  }
  else
  {
    return(FALSE);
  }
}

/* Ghost Initialization */
int InitializeGhostInMaze(int ghost_speed, Maze maze)
{
  Cell cell;
  Ghost  ghost;
  int    lev, row, col;
  int    ghost_id;
  float  cell_width, cell_height, cell_depth;

  /* allocate ghost array memory */
  if (!maze->ghost_array)
  {
    maze->ghost_array = (Ghost)malloc(sizeof(GhostStruct));
    maze->num_ghosts = 1;
  }
  else  /* expand ghost array memory */
  {
    maze->num_ghosts++;
    maze->ghost_array = (Ghost)realloc(maze->ghost_array,
                        maze->num_ghosts * sizeof(GhostStruct));
  }

  /* set a local variable for the ghost */
  ghost_id = maze->num_ghosts - 1;
  ghost = &(maze->ghost_array[ghost_id]);

  /* randomly choose the ghost's initial cell not equal to the start,
     finish, or another ghost */
  do
  {
    lev = GetRandomInt(0, maze->num_levs - 1);
    row = GetRandomInt(0, maze->num_rows - 1);
    col = GetRandomInt(0, maze->num_cols - 1);
    cell = &(maze->cell_matrix[lev][row][col]);
  } while (cell == maze->start_cell  ||
           cell == maze->finish_cell ||
           cell->ghost);

  ghost->id     = ghost_id;
  ghost->cell   = cell;
  ghost->pos.x  = cell->pos.x;
  ghost->pos.y  = cell->pos.y;
  ghost->pos.z  = cell->pos.z;
  cell->ghost   = YES;
  do {ghost->dir = GetRandomInt(0, 5);}
  while(ghost->cell->wall[ghost->dir].state == UP);

  /* a cell's width, height, depth */
  cell_width  = (float)(maze->width)/(float)(maze->num_cols);
  cell_height = (float)(maze->height)/(float)(maze->num_rows);
  cell_depth  = (float)(maze->depth)/(float)(maze->num_levs);

  /* determine how fast a ghost moves -- ghost_speed is the number 
     of steps that a ghost takes between two cells */
  ghost->current_step              = 0;
  ghost->north_south_step_size     = cell_height/(float)ghost_speed;
  ghost->west_east_step_size       = cell_width/(float)ghost_speed;
  ghost->floor_ceiling_step_size   = cell_depth/(float)ghost_speed;
  ghost->num_steps_between_cells = ghost_speed;

  /* return the id of the newly initialized ghost */
  return(ghost_id);
}


/* Ghost Advancement */
void AdvanceGhostInMaze(int ghost_id, Maze maze)
{
  Cell cell;
  Ghost  ghost;
  int    lev, row, col;
  float  cell_width, cell_height;
  int    num_possible_dir;
  Direction dir, from_dir;


  /* set a local variable for the ghost */
  ghost = &(maze->ghost_array[ghost_id]);

  /* choose a new direction of advancement */
  if (ghost->current_step == ghost->num_steps_between_cells)
  {
    /* update the ghost's current cell */
    ghost->cell = ghost->cell->adj_cell[ghost->dir];

    /* update the ghost's position to that of the cell */
    ghost->pos.x = ghost->cell->pos.x;
    ghost->pos.y = ghost->cell->pos.y;
    ghost->pos.z = ghost->cell->pos.z;
    
    /* is the ghost at a dead end? */
    num_possible_dir = 0;
    for (dir=0; dir<6; dir++)
    {
      if (ghost->cell->wall[dir].state == DOWN) num_possible_dir++;
    }
    
    /* only one possible direction, backwards */
    if (num_possible_dir == 1)
    {
      /* reverse the ghost direction */
      if (IsIntEven(ghost->dir)) ghost->dir++;
      else                       ghost->dir--;
    }
    /* randomly choose a new direction */
    else
    {
      if (IsIntEven(ghost->dir)) from_dir = ghost->dir + 1;
      else                       from_dir = ghost->dir - 1;

      do {dir = GetRandomInt(0, 5);}
      while(dir == from_dir || ghost->cell->wall[dir].state == UP);
      ghost->dir = dir;
    }

    ghost->current_step = 0;
  }

  /* advance the ghost another step */
  ghost->current_step++;

  switch (ghost->dir)
  {
    case (NORTH):
    {
      ghost->pos.y = ghost->cell->pos.y -
                     ghost->current_step * ghost->north_south_step_size;
      break;
    }
    case (SOUTH):
    {
      ghost->pos.y = ghost->cell->pos.y +
                     ghost->current_step * ghost->north_south_step_size;
      break;
    }
    case (WEST):
    {
      ghost->pos.x = ghost->cell->pos.x -
                     ghost->current_step * ghost->west_east_step_size;
      break;
    }
    case (EAST):
    {
      ghost->pos.x = ghost->cell->pos.x +
                     ghost->current_step * ghost->west_east_step_size;
      break;
    }
    case (FLOOR):
    {
      ghost->pos.z = ghost->cell->pos.z -
                     ghost->current_step * ghost->floor_ceiling_step_size;
      break;
    }
    case (CEILING):
    {
      ghost->pos.z = ghost->cell->pos.z +
                     ghost->current_step * ghost->floor_ceiling_step_size;
      break;
    }
  }
}
  


/* Initialize and get a Start Cell from the maze. */
Cell InitializeStartCellInMaze(Maze maze)
{
  int lev, row, col;

  /* randomly choose the start cell */
  lev = 0;
  row = maze->num_rows - 1;
  col = 0;

  maze->start_cell = &(maze->cell_matrix[lev][row][col]);

  /* put UP the floor and ceilings of the start cell */
  if (maze->start_cell->adj_cell[FLOOR] != NULL)
  {
    maze->start_cell->wall[FLOOR].state = UP;
    maze->start_cell->adj_cell[FLOOR]->wall[CEILING].state = UP;
  }
  if (maze->start_cell->adj_cell[CEILING] != NULL)
  {
    maze->start_cell->wall[CEILING].state = UP;
    maze->start_cell->adj_cell[CEILING]->wall[FLOOR].state = UP;
  }

  maze->start_cell->start = YES;

  return(maze->start_cell);
}



/* Initialize and get a Finish Cell from the maze. */
Cell InitializeFinishCellInMaze(Maze maze)
{
  int lev, row, col;

  /* randomly choose the finish cell */
  lev = maze->num_levs - 1;
  row = 0;
  col = maze->num_cols - 1;

  maze->finish_cell = &(maze->cell_matrix[lev][row][col]);

  /* put UP the floor and ceilings of the finish cell */
  if (maze->finish_cell->adj_cell[FLOOR] != NULL)
  {
    maze->finish_cell->wall[FLOOR].state = UP;
    maze->finish_cell->adj_cell[FLOOR]->wall[CEILING].state = UP;
  }
  if (maze->finish_cell->adj_cell[CEILING] != NULL)
  {
    maze->finish_cell->wall[CEILING].state = UP;
    maze->finish_cell->adj_cell[CEILING]->wall[FLOOR].state = UP;
  }

  maze->finish_cell->finish = YES;

  return(maze->finish_cell);
}



/* Cell Initialization */
void InitializeCellInMaze(int lev, int row, int col, Maze maze)
{
  Cell cell;
  int    inx;
  float  half_width, half_height, half_depth;

  cell = &(maze->cell_matrix[lev][row][col]);

  /* the cell has NOT been visited, constructed, solved, ... */
  cell->constructed = NO;
  cell->visited     = NO;
  cell->solution    = NO;
  cell->start       = NO;
  cell->finish      = NO;
  cell->elevator    = NO;
  cell->ghost       = NO;

  /* initialize the cell's maze location */
  cell->lev   = lev;
  cell->row   = row;
  cell->col   = col;

  /* initially, the walls of the maze are MISSING */
  for (inx=0; inx < 6; inx++) cell->wall[inx].state = MISSING;
  
  /* find the adjacent cells (cells outside of maze are NULL) */
  if (row > 0)
    cell->adj_cell[NORTH] = &(maze->cell_matrix[lev][row - 1][col]);
  else
    cell->adj_cell[NORTH] = NULL;

  if (row < maze->num_rows - 1)
    cell->adj_cell[SOUTH] = &(maze->cell_matrix[lev][row + 1][col]);
  else
    cell->adj_cell[SOUTH] = NULL;

  if (col > 0)
    cell->adj_cell[WEST] = &(maze->cell_matrix[lev][row][col - 1]);
  else
    cell->adj_cell[WEST] = NULL;

  if (col < maze->num_cols - 1)
    cell->adj_cell[EAST] = &(maze->cell_matrix[lev][row][col + 1]);
  else
    cell->adj_cell[EAST]  = NULL;
 
  if (lev > 0)
    cell->adj_cell[FLOOR] = &(maze->cell_matrix[lev - 1][row][col]);
  else
    cell->adj_cell[FLOOR] = NULL;

  if (lev < maze->num_levs - 1)
    cell->adj_cell[CEILING] = &(maze->cell_matrix[lev + 1][row][col]);
  else
    cell->adj_cell[CEILING]  = NULL;

 
  /* put UP walls adjacent to NULL cells (walls on maze border) */
  for (inx=0; inx < 6; inx++) 
  {
    if (cell->adj_cell[inx] == NULL) cell->wall[inx].state = UP;
  }

  /* a cell's half_width, half_height, half_depth */
  half_width  = (float)(maze->width)/(float)(2 * maze->num_cols);
  half_height = (float)(maze->height)/(float)(2 * maze->num_rows);
  half_depth  = (float)(maze->depth)/(float)(2 * maze->num_levs);

  /* initialize the cell's positional location */
  cell->pos.x = (float)(col + 0.5) * 2.0*half_width;
  cell->pos.y = (float)(row + 0.5) * 2.0*half_height;
  cell->pos.z = (float)(lev + 0.5) * 2.0*half_depth;

  /* initialize the positional location of the walls */
  /* north wall */
  cell->wall[NORTH].pos[0].x = cell->pos.x + half_width;
  cell->wall[NORTH].pos[1].x = cell->pos.x - half_width;
  cell->wall[NORTH].pos[2].x = cell->pos.x - half_width;
  cell->wall[NORTH].pos[3].x = cell->pos.x + half_width;
  cell->wall[NORTH].pos[0].y = cell->pos.y - half_height;
  cell->wall[NORTH].pos[1].y = cell->pos.y - half_height;
  cell->wall[NORTH].pos[2].y = cell->pos.y - half_height;
  cell->wall[NORTH].pos[3].y = cell->pos.y - half_height;
  cell->wall[NORTH].pos[0].z = cell->pos.z - half_depth;
  cell->wall[NORTH].pos[1].z = cell->pos.z - half_depth;
  cell->wall[NORTH].pos[2].z = cell->pos.z + half_depth;
  cell->wall[NORTH].pos[3].z = cell->pos.z + half_depth;

  /* south wall */
  cell->wall[SOUTH].pos[0].x = cell->pos.x - half_width;
  cell->wall[SOUTH].pos[1].x = cell->pos.x + half_width;
  cell->wall[SOUTH].pos[2].x = cell->pos.x + half_width;
  cell->wall[SOUTH].pos[3].x = cell->pos.x - half_width;
  cell->wall[SOUTH].pos[0].y = cell->pos.y + half_height;
  cell->wall[SOUTH].pos[1].y = cell->pos.y + half_height;
  cell->wall[SOUTH].pos[2].y = cell->pos.y + half_height;
  cell->wall[SOUTH].pos[3].y = cell->pos.y + half_height;
  cell->wall[SOUTH].pos[0].z = cell->pos.z - half_depth;
  cell->wall[SOUTH].pos[1].z = cell->pos.z - half_depth;
  cell->wall[SOUTH].pos[2].z = cell->pos.z + half_depth;
  cell->wall[SOUTH].pos[3].z = cell->pos.z + half_depth;

  /* west wall */
  cell->wall[WEST].pos[0].x = cell->pos.x - half_width;
  cell->wall[WEST].pos[1].x = cell->pos.x - half_width;
  cell->wall[WEST].pos[2].x = cell->pos.x - half_width;
  cell->wall[WEST].pos[3].x = cell->pos.x - half_width;
  cell->wall[WEST].pos[0].y = cell->pos.y - half_height;
  cell->wall[WEST].pos[1].y = cell->pos.y + half_height;
  cell->wall[WEST].pos[2].y = cell->pos.y + half_height;
  cell->wall[WEST].pos[3].y = cell->pos.y - half_height;
  cell->wall[WEST].pos[0].z = cell->pos.z - half_depth;
  cell->wall[WEST].pos[1].z = cell->pos.z - half_depth;
  cell->wall[WEST].pos[2].z = cell->pos.z + half_depth;
  cell->wall[WEST].pos[3].z = cell->pos.z + half_depth;

  /* east wall */
  cell->wall[EAST].pos[0].x = cell->pos.x + half_width;
  cell->wall[EAST].pos[1].x = cell->pos.x + half_width;
  cell->wall[EAST].pos[2].x = cell->pos.x + half_width;
  cell->wall[EAST].pos[3].x = cell->pos.x + half_width;
  cell->wall[EAST].pos[0].y = cell->pos.y + half_height;
  cell->wall[EAST].pos[1].y = cell->pos.y - half_height;
  cell->wall[EAST].pos[2].y = cell->pos.y - half_height;
  cell->wall[EAST].pos[3].y = cell->pos.y + half_height;
  cell->wall[EAST].pos[0].z = cell->pos.z - half_depth;
  cell->wall[EAST].pos[1].z = cell->pos.z - half_depth;
  cell->wall[EAST].pos[2].z = cell->pos.z + half_depth;
  cell->wall[EAST].pos[3].z = cell->pos.z + half_depth;

  /* floor wall */
  cell->wall[FLOOR].pos[0].x = cell->pos.x + half_width;
  cell->wall[FLOOR].pos[1].x = cell->pos.x - half_width;
  cell->wall[FLOOR].pos[2].x = cell->pos.x - half_width;
  cell->wall[FLOOR].pos[3].x = cell->pos.x + half_width;
  cell->wall[FLOOR].pos[0].y = cell->pos.y + half_height;
  cell->wall[FLOOR].pos[1].y = cell->pos.y + half_height;
  cell->wall[FLOOR].pos[2].y = cell->pos.y - half_height;
  cell->wall[FLOOR].pos[3].y = cell->pos.y - half_height;
  cell->wall[FLOOR].pos[0].z = cell->pos.z - half_depth;
  cell->wall[FLOOR].pos[1].z = cell->pos.z - half_depth;
  cell->wall[FLOOR].pos[2].z = cell->pos.z - half_depth;
  cell->wall[FLOOR].pos[3].z = cell->pos.z - half_depth;

  /* ceiling wall */
  cell->wall[CEILING].pos[0].x = cell->pos.x - half_width;
  cell->wall[CEILING].pos[1].x = cell->pos.x + half_width;
  cell->wall[CEILING].pos[2].x = cell->pos.x + half_width;
  cell->wall[CEILING].pos[3].x = cell->pos.x - half_width;
  cell->wall[CEILING].pos[0].y = cell->pos.y + half_height;
  cell->wall[CEILING].pos[1].y = cell->pos.y + half_height;
  cell->wall[CEILING].pos[2].y = cell->pos.y - half_height;
  cell->wall[CEILING].pos[3].y = cell->pos.y - half_height;
  cell->wall[CEILING].pos[0].z = cell->pos.z + half_depth;
  cell->wall[CEILING].pos[1].z = cell->pos.z + half_depth;
  cell->wall[CEILING].pos[2].z = cell->pos.z + half_depth;
  cell->wall[CEILING].pos[3].z = cell->pos.z + half_depth;

}



/* Maze Initialization */
Maze InitializeMaze(int width, int height, int depth,
		     int num_levs, int num_rows, int num_cols,
		     int num_elevs)
{
  Maze maze;
  int lev, row, col;
  int id;

  /* allocate maze memory */
  maze = (Maze)malloc(sizeof(MazeStruct));

  /* the maze has NOT been constructed, or solved */
  maze->constructed = NO;
  maze->solved      = NO;

  maze->start_cell  = NULL;
  maze->finish_cell = NULL;

  maze->width      = width;
  maze->height     = height;
  maze->depth      = depth;
  maze->num_levs   = num_levs;
  maze->num_rows   = num_rows;
  maze->num_cols   = num_cols;
  maze->num_elevs  = num_elevs;

  maze->num_ghosts  = 0;
  maze->ghost_array = NULL;

  /* allocate cell matrix memory */
  maze->cell_matrix =
       (Cell **)malloc(maze->num_levs * sizeof(Cell *));
  for (lev=0; lev<maze->num_levs; lev++)
  {
    maze->cell_matrix[lev] =
         (Cell *)malloc(maze->num_rows * sizeof(Cell));

    for (row=0; row<maze->num_rows; row++)
    {
      maze->cell_matrix[lev][row] =
           (Cell)malloc(maze->num_cols * sizeof(CellStruct));
    }
  }

  /* initialize the cells */
  for (lev=0; lev<maze->num_levs; lev++)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      for (col=0; col<maze->num_cols; col++)
      {
        InitializeCellInMaze(lev, row, col, maze);
      }
    }
  }

  return(maze);
}


/* Return a solution cell that is accessible through walls
   that are DOWN, has not already been visited, and
   if no such cell exists, NULL is returned. */
Cell GetAdjacentUnvisitedSolutionCell(Cell cell)
{
  int  inx;

  if (cell == NULL) return(NULL);

  /* find the adjacent unvisited solution cell */
  for (inx=0; inx<6; inx++)
  {
    if (cell->adj_cell[inx])
    {
      if (cell->wall[inx].state == DOWN)
      {
        if (cell->adj_cell[inx]->visited == NO)
        {
          if (cell->adj_cell[inx]->solution == YES)
          {
            return(cell->adj_cell[inx]);
          }
        }
      }
    }
  }

  /* sorry, no adjacent solution cells are
     accessible and unvisited */
  return(NULL);
}


/* Return a cell that is accessible through walls that are
   DOWN and has not been visited. If all adjacent cells have
   been visited, NULL is returned. */
Cell GetAdjacentUnvisitedCell(Cell cell)
{
  int  inx;

  /* find the adjacent unvisited cell */
  for (inx=0; inx<6; inx++)
  {
    if (cell->adj_cell[inx])
    {
      if (cell->wall[inx].state == DOWN)
      {
        if (cell->adj_cell[inx]->visited == NO)
        {
          return(cell->adj_cell[inx]);
        }
      }
    }
  }

  /* sorry, no adjacent cells are accessible and unvisited */
  return(NULL);
}


/* Construct elevators in the maze to get from level to level. */
void ConstructElevatorBetween(
     Cell floor_cell, Cell ceiling_cell, Maze maze)
{
  int lev, row, col;
  int num_elevs;
  int floor_level, ceiling_level;

  /* declare these cells to contain elevators */
  floor_cell->elevator   = YES;
  ceiling_cell->elevator = YES;

  /* should we worry about the number of elevators between levels? */
  if (maze->num_elevs <= 0) return;

  /* what are the levels that we are dealing with */
  floor_level   = floor_cell->lev;
  ceiling_level = ceiling_cell->lev;

  /* count the number of elevators between these two levels */
  num_elevs = 0;
  for (row=0; row<maze->num_rows; row++)
  {
    for (col=0; col<maze->num_cols; col++)
    {
      floor_cell   = &maze->cell_matrix[floor_level][row][col];
      ceiling_cell = &maze->cell_matrix[ceiling_level][row][col];

      if (floor_cell->elevator == YES &&
	  ceiling_cell->elevator == YES &&
	  floor_cell->wall[CEILING].state == DOWN) num_elevs++;
    }
  }

  /* limit the number of elevators between levels */
  if (num_elevs >= maze->num_elevs)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      for (col=0; col<maze->num_cols; col++)
      {
        floor_cell   = &maze->cell_matrix[floor_level][row][col];
        ceiling_cell = &maze->cell_matrix[ceiling_level][row][col];

	if (floor_cell->wall[CEILING].state == MISSING)
	    floor_cell->wall[CEILING].state =  UP;
	if (ceiling_cell->wall[FLOOR].state == MISSING)
	    ceiling_cell->wall[FLOOR].state =  UP;
      }
    }
  }

}


/* Return a cell that has not been constructed adjacent
   to the given cell.  If all adjacent cells have been
   constructed, NULL is returned. */
Cell GetAdjacentUnconstructedCell(Cell cell, Maze maze)
{
  Direction  random_dir, opp_dir;

  /* if no walls are MISSING, then adjacent cells have
     already been constructed, return NULL */
  if ((cell->wall[NORTH].state   != MISSING) &&
      (cell->wall[SOUTH].state   != MISSING) &&
      (cell->wall[WEST].state    != MISSING) &&
      (cell->wall[EAST].state    != MISSING) &&
      (cell->wall[FLOOR].state   != MISSING) &&
      (cell->wall[CEILING].state != MISSING)) return(NULL);

  /* randomly choose a/the wall that is MISSING */
  do { random_dir = GetRandomInt(0,5); }
  while(cell->wall[random_dir].state != MISSING);
  opp_dir = GetOppositeDirection(random_dir);

  /* put walls DOWN */
  cell->wall[random_dir].state = DOWN;
  cell->adj_cell[random_dir]->wall[opp_dir].state = DOWN;

  /* construct elevators between floors */
  if (random_dir == FLOOR || random_dir == CEILING)
  {
    ConstructElevatorBetween(cell->adj_cell[random_dir], cell,  maze);
  }

  /* return the adjacent cell */
  return(cell->adj_cell[random_dir]);
}


/* Solve a Cell -- The Visit step in depth-first search.
   Here the cells of the maze are visited and marked as a
   solution cell on the way to the finish cell.
   This is a recursive function. The return value indicates
   whether or not the cell visited is really a solution cell. */
BOOL SolveCellsInMaze(Cell cell, Maze maze)
{
  Cell next_cell;

  cell->visited = YES;

  /* recursively get cells adjacent to this cell */
  while((next_cell = GetAdjacentUnvisitedCell(cell)))
  {
    cell->solution = YES;

    if (next_cell == maze->finish_cell)
    {
	/* include the finish cell in the solution */
	next_cell->solution = YES;
        return(TRUE);
    }

    if (SolveCellsInMaze(next_cell, maze))
        return(TRUE);

    next_cell->solution = NO;
  }

  return(FALSE);
}


/* Add Loop Cells to Solution -- The Visit step in depth-first search.
   Here the cells of the maze are visited and marked as a solution
   cell on the way back to a cell currently on the solution path.
   This is a recursive function. The return value indicates
   whether or not the cell visited is in the solution loop. */
BOOL AddSolvedLoopCellsInMaze(Cell cell, Maze maze)
{
  Cell next_cell;

  if (cell->solution) return(TRUE);
  cell->visited = YES;

  /* recursively get cells adjacent to this cell */
  while((next_cell = GetAdjacentUnvisitedCell(cell)))
  {
    cell->solution = YES;

    if (next_cell->solution)
        return(TRUE);

    if (AddSolvedLoopCellsInMaze(next_cell, maze))
        return(TRUE);

    next_cell->solution = NO;
  }

  return(FALSE);
}

/* After removing a wall, this function should be called to add to
   the solution space.  The two cells used as arguments are the
   two cells on opposite sides of the wall removed. */
void SolveLoopCellsInMaze(Cell cell_a, Cell cell_b, Maze maze)
{
  Cell cell;

  /* don't try to add a solution loop when the maze hasn't been solved */
  if (!maze->solved) return;

  /* add a solution path from cell_a to the current solution */
  ResetVisitedCellsInMaze(maze, NO);
  cell_a->visited = YES;
  cell_b->visited = YES;
  AddSolvedLoopCellsInMaze(cell_a, maze);

  /* add a solution path from cell_b to the current solution */
  ResetVisitedCellsInMaze(maze, NO);
  cell_a->visited = YES;
  cell_b->visited = YES;
  AddSolvedLoopCellsInMaze(cell_b, maze);

  /* when the added solution path does not form a loop,
     undo the attempted solution loop */
  cell = cell_a;
  while (GetNumberOfAdjacentSolutionCells(cell) == 1)
  {
    if (cell == maze->start_cell || cell == maze->finish_cell)
    {
        break;
    }
    cell->solution = NO;
    cell = GetAdjacentSolutionCell(cell);
  }

  /* when the added solution path does not form a loop,
     undo the attempted solution loop */
  cell = cell_b;
  while (GetNumberOfAdjacentSolutionCells(cell) == 1)
  {
    if (cell == maze->start_cell || cell == maze->finish_cell)
    {
        break;
    }
    cell->solution = NO;
    cell = GetAdjacentSolutionCell(cell);
  }
}


/* Construct a Cell -- The Visit step in depth-first search.
   Here the walls of the cell are put UP depending on the constructed
   state of the adjacent cells.  This is a recursive function. */
void ConstructCellsInMaze(Cell cell, Maze maze)
{
  Direction dir, opp_dir;

  /* put UP MISSING walls between constructed cells */
  for (dir=0; dir<6; dir++)
  {
    opp_dir = GetOppositeDirection(dir);

    if (cell->adj_cell[dir] != NULL)
    {
      if (cell->adj_cell[dir]->constructed == YES)
      {
        if (cell->wall[dir].state == MISSING)
        {
          cell->wall[dir].state = UP;
	  cell->adj_cell[dir]->wall[opp_dir].state = UP;
        }
      }
    }
  } 

  /* this cell has now been constructed */
  cell->constructed = YES;

  /* recursively construct cells adjacent to this cell */
  while((cell = GetAdjacentUnconstructedCell(cell, maze)))
  {
    ConstructCellsInMaze(cell, maze);
  }
}


/* Solve the maze; in other words, find solution cells. */
void SolveMaze(Maze maze)
{
  if (maze->solved) return;

  /* make sure all cells have not been visited */
  ResetVisitedCellsInMaze(maze, NO);

  /* use the depth first search technique to recursively
     visit/solve cells within the maze begining with the
     start cell */
  SolveCellsInMaze(maze->start_cell, maze);

  /* don't inculde the start cell in solution */
  /*
  maze->start_cell->solution = NO;
  */

  /* this maze has now been solved */
  maze->solved = YES;
}



/* Solve the maze; in other words, find solution cells. */
void SolveMazeWrongWay(Maze maze)
{
  int lev, row, col;
  Cell cell;

  if (maze->solved) return;

  /* make sure all cells have not been visited,
     and assume that they are all solution cells */
  ResetVisitedCellsInMaze(maze, NO);
  ResetSolutionCellsInMaze(maze, YES);

  for (lev=0; lev<maze->num_levs; lev++)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      for (col=0; col<maze->num_cols; col++)
      {
        cell = &maze->cell_matrix[lev][row][col];

        /* only process cells at a dead end */
        if (IsCellAtADeadEnd(cell) &&
            cell != maze->start_cell    &&
            cell != maze->finish_cell)
        {
          do
          {
            /* mark the cell as visited and remove it from the solution */
            cell->visited = YES;
            cell->solution = NO;

            cell = GetAdjacentUnvisitedCell(cell);
          } while (!IsCellAtACrossPath(cell) &&
                   cell != maze->start_cell &&
                   cell != maze->finish_cell);
        }
      }
    }
  }

  /* this maze has now been solved */
  maze->solved = YES;
}

/* Put loops in the maze; in other words, put some walls DOWN.
   The number of loops refers to the number of loops per level. */
void LoopMaze(Maze maze, int num_loops_per_level)
{
  int lev, row, col;
  int num_loops;
  int num_iterations;
  Cell cell;
  Direction random_wall;
  BOOL pleasing_from_west,
       pleasing_from_east,
       pleasing_from_north,
       pleasing_from_south;

  /* create loops on every level */
  for (lev=0; lev<maze->num_levs; lev++)
  {
    num_loops = 0;
    num_iterations = 0;

    /* continue trying to make a loop until they have all been made */
    while (num_loops < num_loops_per_level)
    {
      /* infinite loop avoidance */
      num_iterations++;
      if (num_iterations > LOOP_CONVERGENCE) 
      {
        /*
        printf("Exceeded loop convergence of %d.\n", LOOP_CONVERGENCE);
        */
        break;
      }

      /* randomly choose a cell */
      row = GetRandomInt(0, maze->num_rows - 1);
      col = GetRandomInt(0, maze->num_cols - 1);
      cell = &(maze->cell_matrix[lev][row][col]);

      /* Randomly try to put DOWN a NORTH, SOUTH, WEST, or EAST wall */
      random_wall = GetRandomInt(0, 3);

      switch (random_wall)
      {
        case (NORTH):
        {
          if (cell->adj_cell[NORTH] == NULL) break;
          if (cell->wall[NORTH].state == DOWN) break;

          /* will it be esthetically pleasing from the WEST? */
          pleasing_from_west = FALSE;
          if (cell->wall[WEST].state == UP)
          {
              pleasing_from_west = TRUE;
          }
          if (cell->adj_cell[WEST] != NULL)
          {
            if (cell->adj_cell[WEST]->wall[NORTH].state == UP)
              pleasing_from_west = TRUE;
          }

          /* will it be esthetically pleasing from the EAST? */
          pleasing_from_east = FALSE;
          if (cell->wall[EAST].state == UP)
          {
              pleasing_from_east = TRUE;
          }
          if (cell->adj_cell[EAST] != NULL)
          {
            if (cell->adj_cell[EAST]->wall[NORTH].state == UP)
              pleasing_from_east = TRUE;
          }

          /* If it is pleasing, put the wall down */
          if (pleasing_from_west && pleasing_from_east)
          {
            cell->wall[NORTH].state                    = DOWN;
            cell->adj_cell[NORTH]->wall[SOUTH].state = DOWN;
            num_loops++;

            /* we may need to add to the solution cells */
            SolveLoopCellsInMaze(cell, cell->adj_cell[NORTH], maze);
          }
          break;
        }
        case (SOUTH):
        {
          if (cell->adj_cell[SOUTH] == NULL) break;
          if (cell->wall[SOUTH].state == DOWN) break;

          /* will it be esthetically pleasing from the WEST? */
          pleasing_from_west = FALSE;
          if (cell->wall[WEST].state == UP)
          {
              pleasing_from_west = TRUE;
          }
          if (cell->adj_cell[WEST] != NULL)
          {
            if (cell->adj_cell[WEST]->wall[SOUTH].state == UP)
              pleasing_from_west = TRUE;
          }

          /* will it be esthetically pleasing from the EAST? */
          pleasing_from_east = FALSE;
          if (cell->wall[EAST].state == UP)
          {
              pleasing_from_east = TRUE;
          }
          if (cell->adj_cell[EAST] != NULL)
          {
            if (cell->adj_cell[EAST]->wall[SOUTH].state == UP)
              pleasing_from_east = TRUE;
          }

          /* If it is pleasing, put the wall down */
          if (pleasing_from_west && pleasing_from_east)
          {
            cell->wall[SOUTH].state                    = DOWN;
            cell->adj_cell[SOUTH]->wall[NORTH].state = DOWN;
            num_loops++;

            /* we may need to add to the solution cells */
            SolveLoopCellsInMaze(cell, cell->adj_cell[SOUTH], maze);
          }
          break;
        }
        case (WEST):
        {
          if (cell->adj_cell[WEST] == NULL) break;
          if (cell->wall[WEST].state == DOWN) break;

          /* will it be esthetically pleasing from the NORTH? */
          pleasing_from_north = FALSE;
          if (cell->wall[NORTH].state == UP)
          {
              pleasing_from_north = TRUE;
          }
          if (cell->adj_cell[NORTH] != NULL)
          {
            if (cell->adj_cell[NORTH]->wall[WEST].state == UP)
              pleasing_from_north = TRUE;
          }

          /* will it be esthetically pleasing from the SOUTH? */
          pleasing_from_south = FALSE;
          if (cell->wall[SOUTH].state == UP)
          {
              pleasing_from_south = TRUE;
          }
          if (cell->adj_cell[SOUTH] != NULL)
          {
            if (cell->adj_cell[SOUTH]->wall[WEST].state == UP)
              pleasing_from_south = TRUE;
          }

          /* If it is pleasing, put the wall down */
          if (pleasing_from_north && pleasing_from_south)
          {
            cell->wall[WEST].state                   = DOWN;
            cell->adj_cell[WEST]->wall[EAST].state = DOWN;
            num_loops++;

            /* we may need to add to the solution cells */
            SolveLoopCellsInMaze(cell, cell->adj_cell[WEST], maze);
          }
          break;
        }
        case (EAST):
        {
          if (cell->adj_cell[EAST] == NULL) break;
          if (cell->wall[EAST].state == DOWN) break;

          /* will it be esthetically pleasing from the NORTH? */
          pleasing_from_north = FALSE;
          if (cell->wall[NORTH].state == UP)
          {
              pleasing_from_north = TRUE;
          }
          if (cell->adj_cell[NORTH] != NULL)
          {
            if (cell->adj_cell[NORTH]->wall[EAST].state == UP)
              pleasing_from_north = TRUE;
          }

          /* will it be esthetically pleasing from the SOUTH? */
          pleasing_from_south = FALSE;
          if (cell->wall[SOUTH].state == UP)
          {
              pleasing_from_south = TRUE;
          }
          if (cell->adj_cell[SOUTH] != NULL)
          {
            if (cell->adj_cell[SOUTH]->wall[EAST].state == UP)
              pleasing_from_south = TRUE;
          }

          /* If it is pleasing, put the wall down */
          if (pleasing_from_north && pleasing_from_south)
          {
            cell->wall[EAST].state                    = DOWN;
            cell->adj_cell[EAST]->wall[WEST].state = DOWN;
            num_loops++;

            /* we may need to add to the solution cells */
            SolveLoopCellsInMaze(cell, cell->adj_cell[EAST], maze);
          }
          break;
        }
      }
    }
  }
}


/* Construct the maze; in other words, put up the walls in an
   algorithmic, yet random order */
void ConstructMaze(Maze maze)
{
  Cell traversal_cell;
  int    lev, row, col;

  if (maze->constructed) return;

  /* initialize a start and finish cell for construction traversal */
  maze->start_cell  = InitializeStartCellInMaze(maze);
  maze->finish_cell = InitializeFinishCellInMaze(maze);

  /* choose the initial traversal cell in the maze */
  lev = GetRandomInt(0, maze->num_levs - 1);
  row = GetRandomInt(0, maze->num_rows - 1);
  col = GetRandomInt(0, maze->num_cols - 1);
  traversal_cell = &(maze->cell_matrix[lev][row][col]);

  /* use the depth first search technique to recursively
     visit/construct cells within the maze */
  ConstructCellsInMaze(traversal_cell, maze);

  /* this maze has now been constructed */
  maze->constructed = YES;
}


/* Destroy the maze by freeing all the allocated memory to it */
void DestroyMaze(Maze maze)
{
  int lev;
  int row;

  if (maze == NULL) return;

  /* free the cells */
  for (lev=0; lev<maze->num_levs; lev++)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      free(maze->cell_matrix[lev][row]);
    }
    free(maze->cell_matrix[lev]);
  }
  free(maze->cell_matrix);

  /* free the ghosts */
  if (maze->ghost_array) free(maze->ghost_array);

  /* finally, free the maze itself */
  free(maze);

  maze = NULL;
}



/* Evaluates Tcl commands. */
/* Draw the solution cells of a maze. */
void TkMz_DrawSolution(Tcl_Interp *interp, Maze maze)
{
  char tcl_string[200];
  int lev, row, col;
  float pos_x, pos_y;
  Cell cell;

  /* create rectangles for all solution cells */
  for (lev=0; lev<maze->num_levs; lev++)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      for (col=0; col<maze->num_cols; col++)
      {
	cell = &(maze->cell_matrix[lev][row][col]);
	pos_x = cell->pos.x;
	pos_y = cell->pos.y;

        if (cell->solution)
	{
	  /* create a solution cell */
          sprintf(tcl_string,
	      "CreateSolutionSquareAt %d %.2f %.2f",
	    lev + 1,
	    pos_x,
	    pos_y);
	  Tcl_Eval(interp, tcl_string);
	  continue;
        }
      }
    }
  }
}




/* Evaluates Tcl commands. */
/* Draw the start and finish of a maze. */
void TkMz_DrawStartFinish(Tcl_Interp *interp, Maze maze)
{
  char tcl_string[200];
  int lev, row, col;

  /* create a tcl command string to create a Start indicator */
  sprintf(tcl_string,
    "CreateTextAt %d %.2f %.2f Start start center",
    maze->start_cell->lev + 1,
    maze->start_cell->pos.x,
    maze->start_cell->pos.y);
  Tcl_Eval(interp, tcl_string);

  /* create a tcl command string to create a Finish indicator */
  sprintf(tcl_string,
    "CreateTextAt %d %.2f %.2f Finish finish center",
    maze->finish_cell->lev + 1,
    maze->finish_cell->pos.x,
    maze->finish_cell->pos.y);
  Tcl_Eval(interp, tcl_string);
}



/* Evaluates Tcl commands. */
/* Draw the walls of a maze. */
void TkMz_DrawWalls(Tcl_Interp *interp, Maze maze)
{
  char tcl_string[200];
  int lev, row, col;
  int inx;
  Cell cell;

  /* draw the UP walls */
  for (lev=0; lev<maze->num_levs; lev++)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      for (col=0; col<maze->num_cols; col++)
      {
        cell = &maze->cell_matrix[lev][row][col];

	for (inx=0; inx<4; inx++)
	{
	  /* avoid drawing a wall twice */
	  if (inx == NORTH && row != 0) continue;
	  if (inx == WEST  && col != 0) continue;

	  if (cell->wall[inx].state == UP)
	  {
	    /* create a tcl command string to create a line */
            sprintf(tcl_string,
	      "CreateWallAt %d %f %f %f %f ",
	      lev + 1,
	      cell->wall[inx].pos[0].x,
	      cell->wall[inx].pos[0].y,
	      cell->wall[inx].pos[1].x,
	      cell->wall[inx].pos[1].y);
	    Tcl_Eval(interp, tcl_string);
          }
        }
      }
    }
  }
}

/* Evaluates Tcl commands. */
/* Draw the elevators of a maze. */
void TkMz_DrawElevators(Tcl_Interp *interp, Maze maze)
{
  char tcl_string[200];
  int lev, row, col;
  float pos_x, pos_y;
  Cell cell;

  /* find the elevators */
  for (lev=0; lev<maze->num_levs; lev++)
  {
    for (row=0; row<maze->num_rows; row++)
    {
      for (col=0; col<maze->num_cols; col++)
      {
	cell = &(maze->cell_matrix[lev][row][col]);
	pos_x = cell->pos.x;
	pos_y = cell->pos.y;

	/* Up/Down elevator */
	if (cell->wall[FLOOR].state   == DOWN &&
	    cell->wall[CEILING].state == DOWN )
	{
	  /* create an UP/DOWN elevator */
          sprintf(tcl_string,
            "CreateUpDownElevatorAt %d %f %f %c%d_%c%d_%c%d",
	    lev + 1, pos_x, pos_y,
            'l', lev + 1, 'r', row + 1, 'c', col + 1);
	  Tcl_Eval(interp, tcl_string);
	  continue;
        }

	/* Up Elevator */
	if (cell->wall[FLOOR].state   != DOWN &&
	    cell->wall[CEILING].state == DOWN )
	{
	  /* create an UP elevator */
          sprintf(tcl_string,
            "CreateUpElevatorAt %d %f %f %c%d_%c%d_%c%d",
	    lev + 1, pos_x, pos_y,
            'l', lev + 1, 'r', row + 1, 'c', col + 1);
	  Tcl_Eval(interp, tcl_string);
	  continue;
        }

	/* Down Elevator */
	if (cell->wall[FLOOR].state   == DOWN &&
	    cell->wall[CEILING].state != DOWN )
	{
	  /* create a DOWN elevator */
          sprintf(tcl_string,
            "CreateDownElevatorAt %d %f %f %c%d_%c%d_%c%d",
	    lev + 1, pos_x, pos_y,
            'l', lev + 1, 'r', row + 1, 'c', col + 1);
	  Tcl_Eval(interp, tcl_string);
	  continue;
        }
      }
    }
  }
}


/* Evaluates Tcl commands. */
/* Places the Man at the start of the maze. */
void TkMz_StartMan(Tcl_Interp *interp, Maze maze)
{
  char tcl_string[200];
  int lev, row, col;
  float pos_x, pos_y;

  /* the start cell position */
  lev = maze->start_cell->lev;
  row = maze->start_cell->row;
  col = maze->start_cell->col;
  pos_x = maze->start_cell->pos.x;
  pos_y = maze->start_cell->pos.y;

  /* create a tcl command string to create the man */
  sprintf(tcl_string, "CreateManAt %d %f %f",
          lev + 1, pos_x, pos_y);
  Tcl_Eval(interp, tcl_string);
}


/* Evaluates Tcl commands. */
/* Move a ghost in the maze. */
void TkMz_MoveGhost(Tcl_Interp *interp, int ghost_id, Maze maze)
{
  char   tcl_string[200];
  int    lev, row, col;
  int    current_step, num_steps_between_cells;
  int    new_level;
  float  pos_x, pos_y;
  Ghost  ghost;
  Direction dir;

  /* set a local variable for the ghost */
  ghost = &(maze->ghost_array[ghost_id]);

  lev   = ghost->cell->lev;
  dir   = ghost->dir;
  pos_x = ghost->pos.x;
  pos_y = ghost->pos.y;
  current_step              = ghost->current_step;
  num_steps_between_cells = ghost->num_steps_between_cells;

  switch (dir)
  {
    case (NORTH):
    case (SOUTH):
    case (WEST):
    case (EAST):
    {
      /* create a tcl command string to move the ghost */
      sprintf(tcl_string, "MoveGhostTo %d %f %f %d",
              lev + 1, pos_x, pos_y, ghost_id);
      Tcl_Eval(interp, tcl_string);
      break;
    }
    case (FLOOR):
    case (CEILING):
    {

      /* create a tcl command string to move ghost to new level */
      if (current_step == (int)(num_steps_between_cells/2.0 + 0.5))
      {
        /* create a tcl command string to remove the ghost from level */
        sprintf(tcl_string, "DeleteGhost %d %d", lev + 1, ghost_id);
        Tcl_Eval(interp, tcl_string);

        /* create a tcl command string to move ghost onto a new level */
        if (dir == FLOOR) new_level = -1;
        else              new_level =  1;
        sprintf(tcl_string, "CreateGhostAt %d %f %f %d",
                lev + 1 + new_level, pos_x, pos_y, ghost_id);
        Tcl_Eval(interp, tcl_string);
      }

      /* create a tcl command string for the ghost to board elevator */
      else
      {
        sprintf(tcl_string, "GhostBoardElevator %d %d",
                lev + 1, ghost_id);
        Tcl_Eval(interp, tcl_string);
      }

      break;
    }
  }
}


/* Evaluates Tcl commands. */
/* Board ghost into an elevator in the maze. */
void TkMz_BoardGhost(Tcl_Interp *interp, int ghost_id, Maze maze)
{
  char   tcl_string[200];
  int    lev, row, col;
  Ghost  ghost;

  /* set a local variable for the ghost */
  ghost = &(maze->ghost_array[ghost_id]);

  lev   = ghost->cell->lev;

  /* create a tcl command string to create the ghost */
  sprintf(tcl_string, "GhostBoardElevator %d %d",
          lev + 1, ghost_id);
  Tcl_Eval(interp, tcl_string);
}


/* Evaluates Tcl commands. */
/* Draw a ghost in the maze. */
void TkMz_DrawGhost(Tcl_Interp *interp, int ghost_id, Maze maze)
{
  char   tcl_string[200];
  int    lev, row, col;
  float  pos_x, pos_y;
  Ghost  ghost;

  /* set a local variable for the ghost */
  ghost = &(maze->ghost_array[ghost_id]);

  lev   = ghost->cell->lev;
  pos_x = ghost->pos.x;
  pos_y = ghost->pos.y;

  /* create a tcl command string to create the ghost */
  sprintf(tcl_string, "CreateGhostAt %d %f %f %d",
          lev + 1, pos_x, pos_y, ghost_id);
  Tcl_Eval(interp, tcl_string);
}


/* A registered Tcl command. */
/* Advance all ghosts one step */
/* Arguments:	0	AdvanceGhostsOneStep_CProc
*/
int AdvanceGhostsOneStep(ClientData clientData, Tcl_Interp *interp,
		 int argc, char **argv)
{
  char tcl_string[200];

  int ghost_id;

  if (argc != 1)
  {
    sprintf(interp->result, "wrong # args to AdvanceGhostsOneStep(%d)", argc);
    return TCL_ERROR;
  }

  if (gl_maze == NULL)
  {
    sprintf(interp->result, 
      "The maze has not been initialized; cannot advance a ghost yet.");
    return TCL_ERROR;
  }

  /* advance all the ghosts */
  for (ghost_id = 0; ghost_id < gl_maze->num_ghosts; ghost_id++)
  {
    AdvanceGhostInMaze(ghost_id, gl_maze);

    /* move the ghost in the maze */
    TkMz_MoveGhost(interp, ghost_id, gl_maze);

  }

  return TCL_OK;
}


/* A registered Tcl command. */
/* Initialize a new ghost and store it in the global maze. */
/* Arguments:	0	AddGhost_CProc
		1	ghost_speed
*/
int AddGhost(ClientData clientData, Tcl_Interp *interp,
		 int argc, char **argv)
{
  char tcl_string[200];

  int ghost_speed;
  int ghost_id;

  if (argc != 2)
  {
    sprintf(interp->result, "wrong # args to AddGhost(%d)", argc);
    return TCL_ERROR;
  }

  if (gl_maze == NULL)
  {
    sprintf(interp->result, 
      "The maze has not been initialized; cannot add a ghost yet.");
    return TCL_ERROR;
  }

  /* extract the arguments from argv */
  ghost_speed = atoi(argv[1]);

  /* Initialize the ghost maze */
  ghost_id = InitializeGhostInMaze(ghost_speed, gl_maze);

  /* place the ghost in the maze */
  TkMz_DrawGhost(interp, ghost_id, gl_maze);

  return TCL_OK;
}


/* A registered Tcl command. */
/* Initialize a new maze and store it in the global maze. */
/* Arguments:	0	NewMaze_CProc
		1	width
		2	height
		3	num_levs
		4	num_rows
		5	num_cols
		6	num_elevs
		7	num_loops
*/
int NewMaze(ClientData clientData, Tcl_Interp *interp,
	    int argc, char **argv)
{
  char tcl_string[200];

  int width, height;
  int num_levs, num_rows, num_cols, num_elevs, num_loops;
  int num_ghosts, ghost_speed;

  if (argc != 8)
  {
    sprintf(interp->result, "wrong # args to NewMaze(%d)", argc);
    return TCL_ERROR;
  }

  /* extract the arguments from argv */
  width       = atoi(argv[1]);
  height      = atoi(argv[2]);
  num_levs    = atoi(argv[3]);
  num_rows    = atoi(argv[4]);
  num_cols    = atoi(argv[5]);
  num_elevs   = atoi(argv[6]);
  num_loops   = atoi(argv[7]);

  /* Destroy the current global maze */
  DestroyMaze(gl_maze);

  /* Initialize the global maze */
  gl_maze = InitializeMaze(width, height, height,
			    num_levs, num_rows, num_cols,
                            num_elevs);

  /* construct the maze */
  ConstructMaze(gl_maze);

  /* solve the maze */
  SolveMaze(gl_maze);
  
  /* insert loops in the maze (num_loops per level) */
  LoopMaze(gl_maze, num_loops);
  
  /* draw the solution cells to Tcl/Tk */
  TkMz_DrawSolution(interp, gl_maze);

  /* draw the maze to Tcl/Tk */
  TkMz_DrawWalls(interp, gl_maze);

  /* draw the start and finish Tcl/Tk */
  TkMz_DrawStartFinish(interp, gl_maze);

  /* place the man at the Start */
  TkMz_StartMan(interp, gl_maze);

  /* place the elevators */
  TkMz_DrawElevators(interp, gl_maze);

  return TCL_OK;
}

