# preferences.tcl
#
# User preferences.  This uses a table-driven scheme to set a bunch
# of variables in the rest of the application.  The results are
# written out to a Xresources-style file that is read by Preferences_Init
# at startup.
#
# Copyright (c) 1993 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.

# preferences becomes a list of lists, each sublist looks like
#	{ varname xresname defaultValue Comment HelpMsg }
# The varname can be a simple variable or an array element
# The xresname is an Xresource specification
# The defaultValue can be a list, which turns into a set of radio buttons
# or it can be "ON" or "OFF", which turns into a check box
# or if it is a single string, it turns into an entry containing that string


proc PrefVar { item } { lindex $item 0 }
proc PrefXres { item } { lindex $item 1 }
proc PrefDefault { item } { lindex $item 2 }
proc PrefComment { item } { lindex $item 3 }
proc PrefHelp { item } { lindex $item 4 }

proc Preferences_Init { appName userDefaults appDefaults } {
    global pref preferences

    set pref(uid) 0
    set pref(appName) $appName
    set pref(userDefaults) $userDefaults
    set pref(appDefaults) $appDefaults

    if [catch {option readfile $appDefaults startup} err] {
	Exmh_Status "Error in app-defaults $appDefaults: $err"
    }
    if [file exists $userDefaults] {
	if [catch {option readfile $userDefaults user} err] {
	    Exmh_Status "Error in user-defaults $userDefaults: $err"
	}
    }
    if [file exists ~/.exmh] {
	if ![info exists exmh(interp)] {
	    puts stderr "Warning, sourcing old .exmh file"
	    puts stderr "Please save from Preferences to create .exmh-defaults"
	}
	if [catch {uplevel #0 source ~/.exmh} err] {
	    puts stderr "Error in ~/.exmh: $err"
	}
    }
    set preferences {}
}
proc Preferences_Add { prefs {appName _default_} } {
    global preferences pref prefApp
    if {$appName == "_default_"} {
	set appName $pref(appName)
    }
    set preferences [concat $preferences $prefs]
    foreach item $prefs {
	set varName [PrefVar $item]
	set xresName [PrefXres $item]
	set value [PrefValue $varName $xresName]
	set prefApp($xresName) $appName
	Exmh_Debug Pref_Add $appName $varName $value
	if {$value == {}} {
	    # Set variables that are still not set
	    set default [PrefDefault $item]
	    if {[llength $default] > 1} {
		if {[lindex $default 0] == "CHOICE"} {
		    PrefValueSet $varName [lindex $default 1]
		} else {
		    PrefValueSet $varName $default
		}
	    } else {
		if {$default == "OFF" || $default == "ON"} {
		    # This is a boolean
		    if {$default == "OFF"} {
			PrefValueSet $varName 0
		    } else {
			PrefValueSet $varName 1
		    }
		} else {
		    # This is a string or numeric
		    PrefValueSet $varName $default
		}
	    }
	}
    }
}
# Return the value of the given variable,
# or the value from the xresource database,
# or {} if neither exist
proc PrefValue { _var _xres } {
    set _xresval [option get . $_xres {}]
    if [string match *(* $_var] {
	set _arrayName [lindex [split $_var (] 0]
	global $_arrayName
    } else {
	global $_var
    }
    if [catch {
	set $_var
    } _val ] {
	if {$_xresval != {}} {
	    set $_var $_xresval
	    return $_xresval
	} else {
	    return {}
	}
    } else {
	return $_val
    }
}
# set the value of the variable
proc PrefValueSet { _var _value } {
    if [catch {
	if [string match *(* $_var] {
	    set _arrayName [lindex [split $_var (] 0]
	    global $_arrayName
	} else {
	    global $_var
	}
    }] {
	return ""
    } else {
	return [set $_var $_value]
    }
}
proc PrefEntrySet { entry varName } {
    PrefValueSet $varName [$entry get]
}
proc PreferencesSaveEntries {} {
    global preferences PrefEntry
    foreach item $preferences {
	set varName [PrefVar $item]
	set xresName [PrefXres $item]
	set value [PrefValue $varName $xresName]
	set default [PrefDefault $item]
	if {[llength $default] > 1} {
	    if {[lindex $default 0] != "CHOICE"} {
		PrefEntrySet $PrefEntry($varName) $varName
	    }
	} else {
	    if {$default == "OFF" || $default == "ON"} {
		# This is a boolean
	    } else {
		# This is a string or numeric
		PrefEntrySet $PrefEntry($varName) $varName
	    }
	}
    }
}
proc PreferencesDismiss {{save 0}} {
    global exwin
    if $save {
	PreferencesSaveEntries
    }
    destroy .pref
    catch {destroy .prefhelp}
    catch {destroy .prefitemhelp}
    Exmh_Focus
}

proc PreferencesHelp {} {
    if {[catch {Widget_Toplevel .prefhelp "Preferences Help" Help }] == 0} {
	wm group .prefhelp .pref
	Widget_Frame .prefhelp row1 Help {top fill}
	Widget_AddBut .prefhelp.row1 quit "Dismiss" {destroy .prefhelp}
	Widget_Label .prefhelp.row1 label {left fill} \
	    -text "Help for Preferences"
	set helptext "
Information about each setting will appear if you click on
the short description of the preference item.  You can make
the info window go away by clicking on another label or
by clicking inside the info window.

There are three types of options you can set through the Preferences dialog.
Choices are represented by radio-style buttons where only one button
in the set can be enabled at once.  These take effect immediately.
Booleans are represented by check-sytle buttons.  If the checkbox is
dark, then the option is turned on.  This takes effect immediately.

Numeric and filename settings have entry widgets in which you can
type in a new value.  The new value takes effect when you type
<Return>, or when you click \"Save\".

The Save button will save your settings in a .exmh-defaults file
in your home directory.  This is an Xresources-style file.

The Cancel button will restore the settings from defaults and your
last saved preferences.
"
	set numLines [llength [split $helptext \n]]
	if {$numLines > 25} {set numLines 25}
	set t [Widget_Text .prefhelp $numLines -setgrid true]
	$t insert 1.0 $helptext
    } else {
	catch {destroy .prefhelp}
    }
}

proc Preferences_Dialog {} {
    global preferences env
    if {[catch {Widget_Toplevel .pref "Exmh Preferences" Pref}] == 0} {

	set buttons [Widget_Frame .pref but Menubar {top fill}]
	Widget_AddBut $buttons save Save {PreferencesSave}
	Widget_AddBut $buttons reset Cancel {PreferencesReset ; PreferencesDismiss}
	Widget_AddBut $buttons help Help {PreferencesHelp}
	Widget_Label $buttons label {left fill} \
	    -text "Click labels for info on each item"

	set body [Widget_Frame .pref b Rim]
	$body configure -borderwidth 2 -relief raised
	set body [Widget_Frame $body b Pad]
	$body configure -borderwidth 10
	set body [Widget_Frame $body body Body]

	set maxWidth 0
	foreach item $preferences {
	    set len [string length [PrefComment $item]]
	    if {$len > $maxWidth} {
		set maxWidth $len
	    }
	}
	foreach item $preferences {
	    PreferencesDialogItem $body $item $maxWidth
	}
    } else {
	wm deiconify .pref
    }
}

proc PreferencesDialogItem { frame item width } {
    global pref
    incr pref(uid)
    set f [Widget_Frame $frame p$pref(uid) Preference]
    $f configure -borderwidth 2
    Widget_Label $f label {left fill} \
	-text [PrefComment $item] -width $width -relief flat
    bind $f.label <1> [list PreferencesItemHelp  %X %Y [PrefHelp $item]]

    set default [PrefDefault $item]
    if {([llength $default] > 1) && ([lindex $default 0] == "CHOICE")} {
	# >1 This is a list of choices
	foreach choice [lreplace $default 0 0] {
	    incr pref(uid)
	    Widget_RadioBut $f c$pref(uid) $choice [PrefVar $item] {left}
	}
    } else {
	if {$default == "OFF" || $default == "ON"} {
	    # This is a boolean
	    set varName [PrefVar $item]
	    set xresName [PrefXres $item]
	    if {[PrefValue $varName $xresName] == {}} {
		if {$default == "OFF"} {
		    PrefValueSet $varName 0
		} else {
		    PrefValueSet $varName 1
		}
	    }
	    Widget_CheckBut $f check "On" $varName {left}
	} else {
	    # This is a string or numeric
	    global PrefEntry
	    Widget_Entry $f entry {left fill expand} -width 10 -background white
	    set PrefEntry([PrefVar $item]) $f.entry

	    set varName [PrefVar $item]
	    set xresName [PrefXres $item]
	    set curValue [PrefValue $varName $xresName]
	    if {$curValue != ""} {
		set default $curValue
	    }
	    $f.entry insert 0 $default
	    bind $f.entry <Return> [list PrefEntrySet %W $varName]
	}
    }
}
proc PreferencesItemHelp { x y text } {
    catch {destroy .prefitemhelp}
    if {$text == {}} {
	return
    }
    set self [Widget_Toplevel .prefitemhelp "Item help" Itemhelp [expr $x+10] [expr $y+10]]
    wm transient .prefitemhelp .pref
    Widget_Message $self msg -text $text -aspect 1500
    bind $self.msg <1> {PreferencesNukeItemHelp .prefitemhelp}
    .pref.but.label configure -text "Click on popup or another label"
}
proc PreferencesNukeItemHelp { t } {
    .pref.but.label configure -text "Click labels for info on each item"
    destroy $t
}

proc PreferencesSave {} {
    global preferences pref prefApp
    PreferencesSaveEntries
    if [catch {
	set old [open $pref(userDefaults) r]
	set oldValues [split [read $old] \n]
	close $old
    }] {
	set oldValues {}
    }
    if [catch {open $pref(userDefaults).new w} out] {
	.pref.but.label configure -text "Cannot save in $pref(userDefaults).new: $out"
	return
    }
    foreach line $oldValues {
	if {$line == "!!! Lines below here automatically added"} {
	    break
	} else {
	    puts $out $line
	}
    }
    puts $out "!!! Lines below here automatically added"
    puts $out "!!! [exec date]"
    puts $out "!!! Do not edit below here"
    foreach item $preferences {
	set varName [PrefVar $item]
	set xresName [PrefXres $item]
	set value [PrefValue $varName $xresName]
	set default [PrefDefault $item]
	if {[llength $default] == 1} {
	    if {$default != "OFF" && $default != "ON"} {
		global PrefEntry
		set entry $PrefEntry($varName)
		set value [$entry get]
	    }
	}
	puts $out [format "%s\t%s" *${xresName}: $value]
    }
    close $out
    set new [glob $pref(userDefaults).new]
    set old [file root $new]
    if [catch {exec mv $new $old} err] {
	Exmh_Status "Cannot install $new: $err"
	return
    }
    if [file exists ~/.exmh] {
	set orig [glob ~/.exmh]
	if ![info exists $orig.unused] {
	    exec mv $orig $orig.unused
	    puts stderr "Moved ~/.exmh to ~/.exmh.unused"
	    puts stderr "exmh user defaults now in ~/.exmh-defaults"
	}
    }
    PreferencesReset
    PreferencesDismiss
    Background_Preferences
}
proc PreferencesReset {} {
    global preferences pref
    # Re-read user defaults
    option clear
    catch {option readfile $pref(appDefaults) startup}
    catch {option readfile $pref(userDefaults) user}
    # Now set variables
    foreach item $preferences {
	set varName [PrefVar $item]
	set xresName [PrefXres $item]
	set xresval [option get . $xresName {}]
	if {$xresval != {}} {
	    set default $xresval
	} else {
	    set default [PrefDefault $item]
	}
	if {([llength $default] > 1) && ([lindex $default 0] == "CHOICE")} {
	    PrefValueSet $varName [lindex $default 1]
	} else {
	    if {$default == "OFF"} {
		PrefValueSet $varName 0
	    } else {
		if {$default == "ON"} {
		    PrefValueSet $varName 1
		} else {
		    global PrefEntry
		    if [info exists PrefEntry($varName)] {
			set entry $PrefEntry($varName)
			$entry delete 0 end
			$entry insert 0 $default
		    }
		    PrefValueSet $varName $default
		}
	    }
	}
    }
}
proc Preferences_Resource { _varName _rname _default } {
    set _rval [option get . $_rname {}]
    if {$_rval != {}} {
	PrefValueSet $_varName $_rval
    } else {
	PrefValueSet $_varName $_default
    }
}
