/* 
 * mxCmdSZ.c --
 *
 *	This file contains the top-level command procedures for
 *	all Mx commands whose names begin with the letters s-z.
 *	Some of these procedure are also used by Tx.
 *
 * Copyright (C) 1987, 1988 Regents of the University of California 
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * Copyright (c) 1992 Xerox Corporation.
 * Use and copying of this software and preparation of derivative works based
 * upon this software are permitted. Any distribution of this software or
 * derivative works must comply with all applicable United States export
 * control laws. This software is made available AS IS, and Xerox Corporation
 * makes no warranty about the software, its performance or its conformity to
 * any specification.
 */

#ifndef lint
static char rcsid[] = "$Header: /import/tcl7/src/mxedit/RCS/mxCmdSZ.c,v 2.5 1994/02/01 18:15:53 welch Exp $ SPRITE (Berkeley)";
#endif not lint

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <ctype.h>
#include <stdlib.h>
#include <strings.h>
#include <sys/file.h>
#include "mxWidget.h"
#include "regex.h"


/*
 *----------------------------------------------------------------------
 *
 * Mx_ScanCmd --
 *
 *	This is the top-level procedure that implements the
 *	"scan" command that provides scrolling support.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_ScanCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Mx_Position position;
    int result;

    if ((argc < 3) || (argc > 4)) {
	sprintf(interp->result,
		"wrong # args: should be \"%.50s [mark|dragto] y\"",
		argv[0]);
	return TCL_ERROR;
    }
    if (strcmp(argv[1], "mark") == 0) {
	mxwPtr->lastY = atoi(argv[2]);
    } else if (strcmp(argv[1], "dragto") == 0) {
	int delta, speed;
	int newY = atoi(argv[2]);
	if (argc == 4) {
	    speed = atoi(argv[3]);
	} else {
	    speed = 1;
	}
	delta = ((newY - mxwPtr->lastY)*speed)
		    /(mxwPtr->fontPtr->ascent + mxwPtr->fontPtr->descent);
	MxScroll(mxwPtr, speed, delta, &mxwPtr->lastY);
    } else {
	sprintf(interp->result,
		"wrong # args: should be \"%.50s [mark|dragto] y\"",
		argv[0]);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_SearchCmd --
 *
 *	This is the top-level procedure that implements the
 *	"search" command.  See the man page for details on what it
 *	does and what it returns.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_SearchCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    char *pattern;
    Mx_Position start, dummy, newFirst, newLast;
    int forward;
    int result;
    int regExp=1;		/* 1 if doing regular expression matching */
    char *errStr;		/* Holds returned error string */
    Mx_Position fileEnd;	/* Holds end of file position */
    char *value;

    if (argc > 3) {
	sprintf(interp->result,
		"too many args: should be \"%.50s forward|backward [pattern]\"",
		argv[0]);
	return TCL_ERROR;
    }

    value = Tcl_GetVar(mxwPtr->interp, "mxNoRegExps", 1);
    if ((value != NULL) && (*value == '1')) {
	regExp = 0;
    }

    if (argc == 1) {
	forward = 1;
    } else {
	int length;

	length = strlen(argv[1]);
	if (strncmp(argv[1], "forward", length) == 0) {
	    forward = 1;
	} else {
	    forward = 0;
	    if (strncmp(argv[1], "backward", length) != 0) {
		sprintf(interp->result, "bad \"%.50s\" direction \"%.50s\": must be forward or backward\"",
			argv[0], argv[1]);
		return TCL_ERROR;
	    }
	}
    }

    /*
     * Figure out what to search for.
     */

    if (argc == 3) {
	pattern = argv[2];
    } else {
	pattern = Tcl_GetVar(mxwPtr->interp, "mxSearchString", TCL_GLOBAL_ONLY);
	if ((pattern == NULL) || (pattern[0] == '\0')) {
	    sprintf(interp->result, "no search string");
	    return TCL_ERROR;
	}
    }

    /*
     * Do the search.
     */

    if (regExp) {
	errStr = Mx_CompileRegExp(pattern);
	if (errStr != NULL) {
	    sprintf(interp->result, "bad \"%.50s\" pattern: %.50s",
		    argv[0], errStr);
	    return TCL_ERROR;
	}
    }

    result = MxGetSelRange(mxwPtr, &start, &dummy);
    if (result != TCL_OK) {
	Tcl_Return(interp, (char *) NULL, TCL_STATIC);
	start = mxwPtr->fileInfoPtr->caretFirst;
    }
    XFlush(Tk_Display(mxwPtr->tkwin));

    fileEnd = Mx_EndOfFile(mxwPtr->fileInfoPtr->file);
    if (regExp) {
	if (!Mx_SearchRegExp(mxwPtr->fileInfoPtr->file,
		Mx_Offset(mxwPtr->fileInfoPtr->file, start, forward?1:-1),
		forward?fileEnd:Mx_ZeroPosition, &newFirst, &newLast)) {
	    if (!Mx_SearchRegExp(mxwPtr->fileInfoPtr->file,
		    forward?Mx_ZeroPosition:fileEnd,
		    start, &newFirst, &newLast)) {
		Tcl_AppendResult(interp, "couldn't find matching pattern", (char *)NULL);
		return TCL_ERROR;
	    }
	}
    }
    else {
	if (!Mx_SearchPattern(mxwPtr->fileInfoPtr->file,
		Mx_Offset(mxwPtr->fileInfoPtr->file, start, forward?1:-1),
		forward?fileEnd:Mx_ZeroPosition,pattern,
		&newFirst, &newLast)) {
	    if (!Mx_SearchPattern(mxwPtr->fileInfoPtr->file,
		    forward?Mx_ZeroPosition:fileEnd,
		    start, pattern, &newFirst, &newLast)) {
		Tcl_AppendResult(interp, "couldn't find matching pattern", (char *)NULL);
		return TCL_ERROR;
	    }
	}
    }
    MxSelectionSet(mxwPtr, newFirst, newLast);
    if (!(mxwPtr->flags & TX_WIDGET)) {
	MxCaretSetPosition(mxwPtr, newFirst, 0);
    }
    MxGetInWindow(mxwPtr, newFirst, mxwPtr->heightLines/2, 0);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_SeeCmd --
 *
 *	This is the top-level procedure that implements the
 *	"see" command.  See the man page for details on what it
 *	does and what it returns.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_SeeCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Mx_Position position;
    int result;

    if ((argc != 2) && (argc != 3)) {
	sprintf(interp->result,
		"wrong # args: should be \"%.50s mark [top|center|bottom]\"",
		argv[0]);
	return TCL_ERROR;
    }
    result = MxGetMark(mxwPtr, argv[1], &position);
    if (result != TCL_OK) {
	return result;
    }
    if (argc == 3) {
	int length, index;

	length = strlen(argv[2]);
	if (strncmp(argv[2], "top", length) == 0) {
	    index = 0;
	} else if (strncmp(argv[2], "center", length) == 0) {
	    index = mxwPtr->heightLines/2;
	} else if (strncmp(argv[2], "bottom", length) == 0) {
	    index = mxwPtr->heightLines-1;
	} else {
	    sprintf(interp->result, "bad \"%.50s\" position \"%.50s\": should be top, center, or bottom\"",
		    argv[0], argv[2]);
	    return TCL_ERROR;
	}
	MxGetInWindow(mxwPtr, position, index, 1);
    } else {
	MxGetInWindow(mxwPtr, position, mxwPtr->heightLines/2, 0);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_SelectionCmd --
 *
 *	This is the top-level procedure that implements the
 *	"selection" command.  See the man page for details on what it
 *	does and what it returns.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_SelectionCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    register Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int result, length;
    char *selMode;
    Mx_Position mark;

    if (argc > 1) {
	length = strlen(argv[1]);
    } else if (argc <= 1) {
	sprintf(interp->result,
	    "need \"%.50s\" option: clear, here, set, anchor, adjust, or f_adjust", argv[0]);
	return TCL_ERROR;
    }
    if (strncmp(argv[1], "here", length) == 0) {
	if (mxwPtr->fileInfoPtr == MxSelectedFile) {
	    Tcl_AppendResult(interp, "1", (char *)NULL);
	} else {
	    Tcl_AppendResult(interp, "0", (char *)NULL);
	}
	return TCL_OK;
    }
    if (strncmp(argv[1], "clear", length) == 0) {
	MxSelectionClear(mxwPtr);
        return TCL_OK;
    }
    if (strncmp(argv[1], "set", length) == 0) {
	Mx_Position first, last;

	if ((argc != 3) && (argc != 4)) {
	    sprintf(interp->result,
		    "wrong # args: should be \"%.50s set mark1 [mark2]\"",
			    argv[0]);
	    return TCL_ERROR;
	}
	result = MxGetMark(mxwPtr, argv[2], &first);
	if (result != TCL_OK) {
	    return result;
	}
	if (argc == 3) {
	    last = first;
	} else {
	    result = MxGetMark(mxwPtr, argv[3], &last);
	    if (result != TCL_OK) {
		return result;
	    }
	}
	if (MX_POS_LEQ(first, last)) {
	    MxSelectionSet(mxwPtr, first, last);
	} else {
	    MxSelectionSet(mxwPtr, last, first);
	}
	return TCL_OK;
    }
    if (argc != 3) {
	sprintf(interp->result,
		    "wrong # args: should be \"%.50s %s mark\"",
			    argv[0], argv[1]);
	return TCL_ERROR;
    }
    /*
     * The following code deals with dynamically setting/dragging the selection
     */
    if (MxGetMark(mxwPtr, argv[2], &mark) != TCL_OK) {
	return TCL_ERROR;
    }
    selMode = Tcl_GetVar(mxwPtr->interp, "mxSelectionMode", TCL_GLOBAL_ONLY);
    if (selMode == NULL) {
	selMode = "char";
	Tcl_SetVar(mxwPtr->interp, "mxSelectionMode", selMode, TCL_GLOBAL_ONLY);
    }
#ifdef notdef
    if (strncmp(argv[1], "float", length) == 0) {
	/*
	 * "float" initializes a floating selection.
	 */
	switch (selMode[0]) {
	    default:
	    case 'c':
		MxSelectionSet(mxwPtr, mark, mark);
		mxwPtr->fileInfoPtr->floatLeft = mark;
		mxwPtr->fileInfoPtr->floatRight = mark;
		return TCL_OK;
	    case 'w': {
		MxSelectWord(mxwPtr, mark);
		MxGetSelRange(mxwPtr, &mxwPtr->fileInfoPtr->floatLeft,
			    &mxwPtr->fileInfoPtr->floatRight);
		return TCL_OK;
	    }
	    case 'l': {
		MxSelectLine(mxwPtr, mark);
		MxGetSelRange(mxwPtr, &mxwPtr->fileInfoPtr->floatLeft,
			    &mxwPtr->fileInfoPtr->floatRight);
		return TCL_OK;
	    }
	}
    }
#endif
    if (strncmp(argv[1], "anchor", length) == 0) {
	/*
	 * "anchor" updates the anchor spot of the floating selection
	 * but doesn't alter the selection itself.  This is useful
	 * for "exchange-dot-and-mark" support.
	 *
	 * Note that anchor + f_adjust is equivalent to the old float op.
	 */
	switch (selMode[0]) {
	    default:
	    case 'c':
		mxwPtr->fileInfoPtr->floatLeft = mark;
		mxwPtr->fileInfoPtr->floatRight = mark;
		return TCL_OK;
	    case 'w': {
		MxFindWord(mxwPtr, mark, &mxwPtr->fileInfoPtr->floatLeft,
			    &mxwPtr->fileInfoPtr->floatRight);
		return TCL_OK;
	    }
	    case 'l': {
		MxFindLine(mxwPtr, mark, &mxwPtr->fileInfoPtr->floatLeft,
			    &mxwPtr->fileInfoPtr->floatRight);
		return TCL_OK;
	    }
	}
    }
    if (strncmp(argv[1], "adjust", length) == 0) {
	/*
	 * Adjust, anchored to the caret.
	 */
	Mx_Position caret;
	caret = mxwPtr->fileInfoPtr->caretFirst;
	switch (selMode[0]) {
	    case 'c':
		if (MX_POS_LEQ(caret, mark)) {
		    MxSelectionSet(mxwPtr, caret, mark);
		} else {
		    caret = Mx_Offset(mxwPtr->fileInfoPtr->file, caret, -1);
		    MxSelectionSet(mxwPtr, mark, caret);
		}
		return TCL_OK;
	    case 'w': {
		Mx_Position wordStart, wordEnd;
		MxFindWord(mxwPtr, mark, &wordStart, &wordEnd);
		if (MX_POS_LEQ(caret, mark)) {
		    MxSelectionSet(mxwPtr, caret, wordEnd);
		} else {
		    caret = Mx_Offset(mxwPtr->fileInfoPtr->file, caret, -1);
		    MxSelectionSet(mxwPtr, wordStart, caret);
		}
		return TCL_OK;
	    }
	    case 'l': {
		Mx_Position lineStart, lineEnd;
		MxFindLine(mxwPtr, mark, &lineStart, &lineEnd);
		if (MX_POS_LEQ(caret, mark)) {
		    MxSelectionSet(mxwPtr, caret, lineEnd);
		} else {
		    caret = Mx_Offset(mxwPtr->fileInfoPtr->file, caret, -1);
		    MxSelectionSet(mxwPtr, lineStart, caret);
		}
		return TCL_OK;
	    }
	    default: {
		return TCL_OK;
	    }
	}
    }
    if (strncmp(argv[1], "f_adjust", length) == 0) {
	/*
	 * Adjust, anchored to start of floating selection.
	 */
	Mx_Position left, right;
	left = mxwPtr->fileInfoPtr->floatLeft;
	right = mxwPtr->fileInfoPtr->floatRight;
	switch (selMode[0]) {
	    case 'c':
		if (MX_POS_LEQ(left, mark)) {
		    MxSelectionSet(mxwPtr, left, mark);
		} else {
		    MxSelectionSet(mxwPtr, mark, right);
		}
		return TCL_OK;
	    case 'w': {
		Mx_Position wordStart, wordEnd;
		MxFindWord(mxwPtr, mark, &wordStart, &wordEnd);
		if (MX_POS_LEQ(left, mark)) {
		    MxSelectionSet(mxwPtr, left, wordEnd);
		} else {
		    MxSelectionSet(mxwPtr, wordStart, right);
		}
		return TCL_OK;
	    }
	    case 'l': {
		Mx_Position lineStart, lineEnd;
		MxFindLine(mxwPtr, mark, &lineStart, &lineEnd);
		if (MX_POS_LEQ(left, mark)) {
		    MxSelectionSet(mxwPtr, left, lineEnd);
		} else {
		    MxSelectionSet(mxwPtr, lineStart, right);
		}
		return TCL_OK;
	    }
	    default: {
		return TCL_OK;
	    }
	}

    }
    sprintf(interp->result,
	    "bad \"%.50s\" option \"%.50s\": must be clear, here, set, start, word, line, or adjust",
	    argv[0], argv[1]);
    return TCL_ERROR;
}

/*
 * MxFindWord --
 *	Return two marks that delimit a word.  The characters in the
 *	word are defined by mxWordMask - see mxSearch.c.  If there
 *	are no word-like characters under the mark, then it is returned.
 */
MxFindWord(mxwPtr, mark, firstPtr, lastPtr)
    MxWidget *mxwPtr;
    Mx_Position mark;
    Mx_Position *firstPtr, *lastPtr;
{
    Mx_Position tmp, tmp2;
    tmp.lineIndex = tmp.charIndex = 0;
    if (! Mx_SearchMask(mxwPtr->fileInfoPtr->file, mark, tmp,
	    mxWordMask, firstPtr)) {
	    *firstPtr = mark;
	    *lastPtr = mark;
	    /*
	     * No word, but try for paren matching.
	     */
	    (void)Mx_SearchParen(mxwPtr->fileInfoPtr->file, mark,
		    Mx_ZeroPosition, Mx_EndOfFile(mxwPtr->fileInfoPtr->file),
		    mxOpenParens, mxCloseParens, firstPtr, &tmp, &tmp2,
		    lastPtr);
    } else {
	(void) Mx_SearchMask(mxwPtr->fileInfoPtr->file, mark,
		Mx_EndOfFile(mxwPtr->fileInfoPtr->file), mxWordMask,
		lastPtr);
    }
}

/*
 * MxSelectWord
 */
MxSelectWord(mxwPtr, mark)
    MxWidget *mxwPtr;
    Mx_Position mark;
{
    Mx_Position first, last;
    MxFindWord(mxwPtr, mark, &first, &last);
    MxSelectionSet(mxwPtr, first, last);
}

/*
 * MxFindLine
 */
MxFindLine(mxwPtr, mark, firstPtr, lastPtr)
    MxWidget *mxwPtr;
    Mx_Position mark;
    Mx_Position *firstPtr, *lastPtr;
{
    Mx_Position endLine, first;
    endLine.lineIndex = first.lineIndex = mark.lineIndex;
    (void) Mx_GetLine(mxwPtr->fileInfoPtr->file, endLine.lineIndex,
	    &endLine.charIndex);
    endLine.charIndex -= 1;		/* Skip newline */
    first.charIndex = 0;		/* Go the beginning of line */
    *lastPtr = endLine;
    *firstPtr = first;
}

/*
 * MxSelectLine
 */
MxSelectLine(mxwPtr, mark)
    MxWidget *mxwPtr;
    Mx_Position mark;
{
    Mx_Position endLine;
    MxFindLine(mxwPtr, mark, &mark, &endLine);
    MxSelectionSet(mxwPtr, mark, endLine);
}


/*
 *----------------------------------------------------------------------
 *
 * Mx_SwitchCmd --
 *
 *	This is the top-level procedure that implements the
 *	"switch" command.  This switchs to a different file.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_SwitchCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int result;
    int force = 0;
    char *fileName;
    Tcl_DString string;

    if (argc > 3) {
	sprintf(interp->result,
		"wrong # args: should be \"%.50s fileName [force]\"",
		argv[0]);
	return TCL_ERROR;
    }
    if (argc == 3) {
	if (strcmp(argv[2], "force") != 0) {
	    sprintf(interp->result, "bad argument: should be \"%.50s [force]\"",
		argv[0]);
	    return TCL_ERROR;
	} else {
	    force = 1;
	}
    }
    fileName = Tcl_TildeSubst(interp, argv[1], &string);
    if (fileName == NULL) {
	return TCL_ERROR;
    }
    if (strcmp(fileName, mxwPtr->fileInfoPtr->name) == 0) {
	/*
	 * SetupFileInfo won't reopen a file that's already
	 * open, so we give a message here.
	 */
	if (mxwPtr->fileInfoPtr->lastMod !=
		MxLastMod(mxwPtr->fileInfoPtr->name)) {
	    MxOutputMsg(mxwPtr, "Warning: current file has been modified");
	}
	Tcl_DStringFree(&string);
	return TCL_OK;
    }
    if (!force) {
	result = MxCheckWritten(interp, mxwPtr, 0);
	if (result != TCL_OK) {
	    return result;
	}
    }
    result = MxSwitchFile(mxwPtr, fileName);
    Tcl_DStringFree(&string);
    return result;
}	    

/*
 *----------------------------------------------------------------------
 *
 * Mx_TaginfoCmd --
 *
 *	This is the top-level procedure that implements the
 *	"taginfo" command.  See the man page for details on what it
 *	does and what it returns.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_TaginfoCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    char *ptrs[2];
    int result;
    int regExp = 1;		/* 1 if regular expression pattern matching */
    char *value;

    value = Tcl_GetVar(mxwPtr->interp, "mxNoRegExps", 1);
    if ((value != NULL) && (*value == '1')) {
	regExp = 0;
    }

    if (argc != 2) {
	sprintf(interp->result, "wrong # args: should be \"%.50s name\"",
		argv[0]);
	return TCL_ERROR;
    }

    result = Mx_GetTag(argv[1], Tcl_GetVar(mxwPtr->interp, "mxTagFiles", 1),
	    regExp, &ptrs[0], &ptrs[1], interp);
    if (result != TCL_OK) {
	return result;
    }
    /*
     * Return a list of two things, the tag and the search string.
     */
    Tcl_AppendElement(interp, ptrs[0]);
    Tcl_AppendElement(interp, ptrs[1]);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_UndoCmd --
 *
 *	This is the top-level procedure that implements the
 *	"undo" command.  See the man page for details on what it
 *	does and what it returns.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_UndoCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (mxwPtr->fileInfoPtr->log == NULL) {
	Tcl_AppendResult(interp, "can't undo: no undo info is kept for this file", (char *)NULL);
	return TCL_ERROR;
    }

    if (argc == 1) {
	return Undo_Last(mxwPtr->fileInfoPtr->log, interp, mxwPtr);
    } else if ((argc == 2) && (strcmp(argv[1], "more") == 0)) {
	return Undo_More(mxwPtr->fileInfoPtr->log, interp, mxwPtr);
    } else if ((argc == 2) && (strcmp(argv[1], "mark") == 0)) {
	Undo_Mark(mxwPtr->fileInfoPtr->log);
	return TCL_OK;
    } else if ((argc == 3) && (strcmp(argv[1], "recover") == 0)) {
	return Undo_Recover(mxwPtr->fileInfoPtr->file, argv[2], interp);
    }
    sprintf(interp->result, "wrong args: must be \"%.50s\", or \"%.50s more\", or \"%.50s recover file\"",
	    argv[0], argv[0], argv[0]);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_ViewCmd --
 *
 *	This is the top-level procedure that implements the
 *	"view" command.  This is for scrollbar interactions.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_ViewCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Mx_Position position;
    int firstUnit = atoi(argv[1]);

    position.lineIndex = firstUnit;
    position.charIndex = 0;
    MxGetInWindow(mxwPtr, position, 0, 1);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_WriteCmd --
 *
 *	This is the top-level procedure that implements the
 *	"write" command.  This saves the file to disk.
 *
 * Results:
 *	Stylized error messages are returned so that the calling
 *	script can put up appropriate dialog boxes:
 *		File exists
 *		File modified
 *	The force option is used to override these kinds of errors.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_WriteCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int result;
    char *name;
    Tcl_DString string;
    Mx_Position eof;
    int forceWrite = 0;
    char msg[200];

    if (argc > 3)  {
	sprintf(interp->result, "too many args: should be \"%.50s [filename [force]]\"",
		argv[0]);
	return TCL_ERROR;
    }
    if (argc == 3) {
	if (strcmp(argv[2], "force") == 0) {
	    forceWrite = 1;
	} else {
	    sprintf(interp->result, "Bad argument, should be \"%.50s [filename [force]]\"",
		argv[0]);
	return TCL_ERROR;
	}
    }
    if (argc == 1) {
	if (*mxwPtr->fileInfoPtr->name == 0) {
	    sprintf(interp->result, "must provide \"name\" arg to \"%.50s\", since the file doesn't have a name yet",
		    argv[0]);
	    return TCL_ERROR;
	}
	name = Tcl_TildeSubst(interp, mxwPtr->fileInfoPtr->name, &string);
	if (name == NULL) {
	    return TCL_ERROR;
	}
    } else {
	char *p;

	/*
	 * Don't allow weird characters in file names.
	 */

	for (p = argv[1]; *p != NULL; p++) {
	    if (!isascii(*p) || iscntrl(*p)) {
		Tcl_AppendResult(interp, "file name contains non-printing characters", (char *)NULL);
		return TCL_ERROR;
	    }
	}
	name = Tcl_TildeSubst(interp, argv[1], &string);
	if (name == NULL) {
	    return TCL_ERROR;
	}

	/*
	 * Don't overwrite an existing file without permission.
	 */

	if (strcmp(mxwPtr->fileInfoPtr->name, name) != 0) {
	    if ((access(name, F_OK) == 0) &&
		(! forceWrite)) {
		int option;

		sprintf(msg, "File exists \"%.50s\"", name);
		Tcl_SetResult(interp, msg, TCL_STATIC);
		Tcl_DStringFree(&string);
		return TCL_ERROR;
	    }
	}
    }

    /* Check if file has changed since we read it in */

    if (!strcmp(mxwPtr->fileInfoPtr->name, name)) {
	if ((mxwPtr->fileInfoPtr->lastMod != MxLastMod(name)) &&
	    (! forceWrite)) {
	    sprintf(msg, "File modified \"%.50s\"",
		    name);
	    Tcl_SetResult(interp, msg, TCL_STATIC);
	    Tcl_DStringFree(&string);
	    return TCL_ERROR;
	}
    }

    /*
     * Get rid of trailing indentation on the current line;  otherwise
     * the next time the user points into the window the indentation will
     * get modified and the file will suddenly appear modified.
     */

    XFlush(Tk_Display(mxwPtr->tkwin));
    MxCleanIndent(mxwPtr->fileInfoPtr->file,
	    mxwPtr->fileInfoPtr->caretFirst.lineIndex);
    result = Mx_FileWrite(mxwPtr->fileInfoPtr->file, name, interp);
    if (result != TCL_OK) {
	Tcl_DStringFree(&string);
	return result;
    }
    mxwPtr->fileInfoPtr->lastMod = MxLastMod(mxwPtr->fileInfoPtr->name);

    if (strcmp(name, mxwPtr->fileInfoPtr->name) == 0) {
	Undo_SetVersion(mxwPtr->fileInfoPtr->log, name);
	mxwPtr->fileInfoPtr->lastMod = MxLastMod(name);
    }
    eof = Mx_EndOfFile(mxwPtr->fileInfoPtr->file);
    sprintf(interp->result, "wrote \"%.50s\": %d lines", name,
	    eof.lineIndex+1);
    Tcl_DStringFree(&string);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_WrittenCmd --
 *
 *	This is the top-level procedure that implements the
 *	"written" command.  This raises an error if the file is
 *	still dirty (i.e., is not written out).
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_WrittenCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int allWindows;

    Tcl_Return(interp, (char *)NULL, TCL_STATIC);
    if (argc > 2)  {
	sprintf(interp->result, "too many args: should be \"%.50s [allWindows]\"",
		argv[0]);
	return TCL_ERROR;
    }
    if (argc == 1) {
	allWindows = 0;
    } else if (strcmp(argv[1], "allWindows") == 0) {
	allWindows = 1;
    } else {
	sprintf(interp->result, "wrong option: should be \"%.50s [allWindows]\"",
		argv[0]);
	return TCL_ERROR;
    }
    return MxCheckWritten(interp, mxwPtr, allWindows);
}
