# mxedit.tk --
# This script constructs and editor based on the mxedit widget.
# This script is sourced by the "mxOpen" TCL command that creates
# a new window (and new interpreter context) in which to edit a file.
# As such, it is re-read each time you create a new window.
#
# A raw mxedit is just a window that you can edit a file in.
# To be useful, it needs associated scrollbars, menus, command entry
# fields, and a feedback mechanism.  This is the top-level script
# that sets these things up.  It sets up the auto_load facility in
# order to access library routines in the mxedit.* files..
#
# Copyright (c) 1992-3 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.

#
# tkerror --
#	This is the handler for background errors that arise
#	from commands bound to keystrokes and menus.  A
#	toplevel message widget is used to display $errorInfo

proc tkerror { msg } {
    global errorInfo
    global paleBackground
    global mxedit
    global entryFont

    if { ! [info exists mxedit] } {
	if {![string match "Quit*" $msg]} {
	    puts stderr $msg
	}
	return
    }
    if [info exists paleBackground] {
	set background $paleBackground
    } else {
	set background white
    }
    if [info exists entryFont] {
	set font $entryFont
    } else {
	set font fixed
    }
    set base ".errorInfo"
    set title "Error Info"
    if [info exists errorInfo] {
	set savedErrorInfo $errorInfo
    } else {
	set savedErrorInfo {no errorInfo}
    }
    # Create a toplevel to contain the error trace back
    if [catch {
	# Choose a unique name by testing for the associated error variable
	# Use the string ".errorInfo-N" as the name of the toplevel
	# and as the name of a variable holding the current errorInfo
	for {set x 1} {$x<10} {set x [expr $x+1]} {
	    global $base-$x
	    if {! [info exists $base-$x]} {
		break
	    }
	}
	global $base-$x ; set $base-$x $errorInfo
	set title $title-$x
	set name $base-$x
	toplevel $name -background $background

	wm title $name $title
    
	buttonFrame $name
    
	packedButton $name.buttons .quit "Dismiss" "destroy $name" left
#	message $name.msg -aspect 300 -font $font 	\
#	    -text $errorInfo -background $paleBackground
	text $name.msg -font $font -background $paleBackground -width 60
	set numLines [llength [split $errorInfo \n]]
	if {$numLines > 20} {
	    set numLines 20
	}
	$name.msg configure -height $numLines
	$name.msg insert end $errorInfo
	pack append $name $name.msg {top expand}
    } oops] {
	set msg [concat $msg "($name: " $oops ")" ]
   }

    if [catch "mxFeedback \{tkerror: $msg\}"] {
	puts stderr "tkrror: $msg"
	puts stderr "*** TCL Trace ***"
	puts stderr $savedErrorInfo
    }
}

# Set things up for autoloading a bunch of utility procs
source [info library]/init.tcl
source $tk_library/tk.tcl
set auto_path "$mxLibrary $auto_path"

# Initialize modules, mainly their global variables
catch utilsInit
catch colorsInit
catch setColorCube

catch mxUtilsInit
catch mxSearchInit
catch mxCommandInit
catch mxEntryBindings

# Turn off auto_exec
set auto_noexec {}

# Source the users initialization file so they can
# set controlling global variables and define hook procedures.
if [file exists ~/.mxedit] {
    if [catch "source ~/.mxedit" msg] {
	puts stderr "source ~/.mxedit failed: $msg"
    }
}

# mxInit --
#	This is called from the "mxOpen" implementation to initialize the editor
#	This assumes there is a top-level main window called "."
#	This fills out "." for the first file on the command line
#	and then calls mxOpen to open a new window on the other files.
#	mxOpen creates a new window and calls back to mxInit.
#
proc mxInit { font geometry args } {
    global argv
    global mxLibrary

    set self "."
    set activeMsg {}

    set haveOneWindow 0
    set hitOtherSession 0
    foreach file [lrange $args 0 end] {
	if { ! $haveOneWindow} {
	    if [catch {mxSetup $self $file $geometry $font} msg] {
		case $msg {
		    "Active: *" {
			# bumped into another editor session
			set hitOtherSession 1
			puts stderr $msg
			set activeMsg $msg
		    }
		    "command aborted*" {
			puts stderr $msg
		    }
		    default {
			error "mxSetup \"$file\" failed: $msg"
		    }
		}
	    } else {
		set haveOneWindow 1
	    }
	} else {
	    # mxOpen is a call back into the application that
	    # will ultimately come back here to mxInit via recursion
	    if [catch {mxOpen $file -geometry $geometry -font $font} msg] {
		if ![string match "Active: *" $msg] {
		    tkerror "mxOpen \"$file\" failed: $msg"
		}
	    }
	}
    }
    if { $hitOtherSession } {
	# raise this error after processing all filename arguments
	error $activeMsg
    } else {
	if { ! $haveOneWindow } {
	    if [catch {mxSetup $self $mxLibrary/mxedit.tutorial \
				    $geometry $font} msg] {
		if [string match "Active: *" $msg] {
		    tkerror "The editor has found another editting session on the tutorial file."
		    error $msg
		} else {
		    error "mxSetup \"tutorial\" failed: $msg"
		}
	    }
	}
    }
}

# mxSetup --
#	Populate a frame (or toplevel) with an editor widget, scrollbar, etc
#	parent is the parent widget (a frame or toplevel)
#	filename is what you're editting
#	geometry is something like 80x20
#	font is an X font name
#
proc mxSetup { parent filename {geometry 80x20} {font fixed} } {
    global mxVersion
    global mxLibrary
    global mxLines
    global mxFile
    global mxGeometry
    global mxedit
    global mxFeedback


#puts stderr [list mxSetup $filename $geometry $font]
    # Look for other interpreters and raise an error if this would
    # establish a conflicting editting session
    mxCheckSessions $filename

    # Set all the widget fonts to be the same
    global entryFont labelFont buttonFont menuFont
    set entryFont $font
    set labelFont $font
    set buttonFont $font
    set menuFont $font

    # Command entry
    if [catch {mxCommandEntry $parent 20 {bottom fillx}} msg] {
	tkerror "mxCommandEntry failed: $msg"
    }

    # Feedback placement
    if ![info exists mxFeedback] {
	set mxFeedback top
    }

    # Feedback entry, between command and main edit window
    if {$mxFeedback == "bottom"} {
	if [catch {pack append $parent [mxFeedbackSetup $parent .feedback 20 2] \
				       {bottom fillx}} msg] {
	    tkerror "mxFeedbackSetup failed: $msg"
	}
	# remember this for repacking the command window
	global mxBottomChild
	set mxBottomChild .feedback
    }
    # Menus
    if [catch {
	    pack append $parent [mxMenuSetup $parent] {top fillx}
	    mxCreateMenus
	} msg] {	
	tkerror "Menu setup failed: $msg"
    }

    # Feedback entry, between menubar and main edit window
    if {$mxFeedback != "bottom"} {
	if [catch {pack append $parent [mxFeedbackSetup $parent .feedback 20 2] \
				       {top fillx}} msg] {
	    tkerror "mxFeedbackSetup failed: $msg"
	}
	# remember this for repacking the command window
	global mxBottomChild
	set mxBottomChild .mx
    }
    # The main editting window coupled with a scrollbar and feedback line
    # It's name will be saved in the mxedit global variable
    pack append $parent \
	[mxScroll $parent $filename mxFeedback $geometry $font] \
	{bottom fillx filly expand}
    focus $mxedit

    # Save file name in global variable.  The mxedit implementation
    # does this for us, except it doesn't do scratch files right.
    # The mxopen implementation keeps a count of scratch windows
    # in order to generate unique interpreter names, so we leverage off that.
    if {[llength $filename] == 0} {
	global mxInterpName
	set mxFile [lindex [set mxInterpName] 1]
	mxFeedback "Mxedit $mxVersion, $mxFile"
    } else {
	set mxFile $filename
	mxFeedback "Mxedit $mxVersion, editing \"$mxFile\": $mxLines lines"
    }

    # Name the window, computing a shortened name for the icon
    mxNameWindow . $mxFile

    # Now that all the decorations have been built up,
    # tell the window manager about a gridded window
    # The widthChars (baseWidth) and heightLines (baseHeight)
    # must agree with what was passed to mxedit.  In turn, the mxedit
    # widget tells us about the gridsize based on font metrics

    if {[scan $mxGeometry "%dx%d" widthChars heightLines] != 2} {
	tkerror "Cannot parse geometry $mxGeometry"
    }
    if [catch {eval {wm grid . $widthChars $heightLines} [$mxedit gridsize]} msg] {
	tkerror "wm grid failed: $msg"
    } else {
	wm geometry . $mxGeometry
    }

    # Do per-site customization
    catch mxLocalBindings

    # Do per-user customization
    if {[llength [info procs mxUserOpenHook]] > 0} {
	if [catch "mxUserOpenHook $mxFile" msg] {
	    mxFeedback "mxUserOpenHook failed: $msg"
	    puts stderr "mxUserOpenHook failed: $msg"
	}
    }

    # Register ourselves in the global interpreter
    global mxInterpName
    mxGlobalEval mxGblNewFile $mxInterpName
}
#
# mxCheckSessions --
#
proc mxCheckSessions { filename } {
    global mxInterpName
    global mxedit

    if [mxFileActive $filename] {
	# Ok to open another window if we are already editing the file
	# with this mxedit process
	return "ok"
    }
    # Get a unique handle on our file
    set myident [mxFileIdent $filename]

    # Look through the set of registered interpreters for one that
    # is editting the same file.
    foreach app [winfo interps] {
	if {[string compare $app $mxInterpName]==0} {
	    continue
	} else {
	    if [string match "mxedit *" $app] {
		scan $app "mxedit %s" f
		if {[string match $f $filename] ||
		    [string match $f tutorial]} {
		    if [catch {send $app {global mxFile ; mxFileIdent $mxFile}} ident] {
			# Can't send => bogus interpreter name
			continue
		    }
#		    puts stderr "Ident? $ident"
		    if {[string compare $myident $ident] == 0} {
			# The other guy is editting the same file
			# Close and open him, then bail out
			global mxOpenCloseDelay
			if ![info exists mxOpenCloseDelay] {
			    set mxOpenCloseDelay 500
			}
			send $app {wm iconify .}
			send $app "update ; after $mxOpenCloseDelay wm deiconify ."
			error "Active: $app"
		    }
		}
	    }
	}
    }
    return ok
}

#
# mxNameWindow --
#	Compute a window name and icon name based on the file name
#	The title is the filename.  The iconname is the last component
#	of the filename.  If the file is dirty, a "!" is appended to both.
#
proc mxNameWindow { window filename } {
    global mxedit

    set title $filename

    set sindex [string last "/" $filename]
    if {$sindex > 0} {
	set iconname [concat "..." [string range "$filename" $sindex end]]
    } else {
	set iconname "$filename"
    }
    if { ! [catch {set mxedit}] } {
	# mxedit is defined so we can ask it if the file is modified
	if [catch "$mxedit written allWindows"] {
	    set title [concat $title " !"]
	    set iconname  [concat $iconname " !"]
	}
    }
    wm title $window $title
    wm iconname $window $iconname
}

# mxWindowNameFix --
#	Update the window and icon name based on global file variable

proc mxWindowNameFix { } {
    global mxFile
    mxNameWindow . $mxFile
}

# mxFeedbackSetup --
# Create an entry widget that is used for feedback
# Create a frame to hold messages, and define a procedure to display them.

proc mxFeedbackSetup { parent name {width 58} {border 6} } {
    global backgroundColor paleBackground foregroundColor
    global entryFont FontWidgets
    global mxFeedbackEntry
    global mxCommand

    set self [selfName $parent $name]

    frame $self -borderwidth 2 -background $backgroundColor -relief raised

    entry $self.entry -width $width -relief flat \
	-font $entryFont \
	-background $paleBackground -foreground $foregroundColor \
	-selectforeground $paleBackground  -selectbackground $foregroundColor
    lappend FontWidgets $self.entry

    pack append $self $self.entry {left fillx expand}
    pack append $parent $self {left fillx expand}

    # Warp focus to the command window
##    bind $self.entry <1> +{ focus $mxCommand.entry }

    # Remember the name of the entry widget for later feedback
    set mxFeedbackEntry $self.entry

    return $self
}

# mxFeedback --
#	Display a message for the user

proc mxFeedback { text } {
    global mxFeedbackEntry
    global mxFN mxLogFeedback

    if ![info exists mxLogFeedback] { set mxLogFeedback 0 }
    if ![info exists mxFN] { set mxFN 0 }

    if [info exists mxFeedbackEntry] {
	$mxFeedbackEntry delete 0 end
	$mxFeedbackEntry insert 0 "$text"
	update
    } else {
	set mxLogFeedback 1
    }
    if { $mxLogFeedback } {
	incr mxFN
	puts stderr "$mxFN: $text"
    }
    return "$text"
}

# mxScroll --
#	Compose an mxedit and a scrollbar

proc mxScroll { parent file feedback geometry font } {
    global mxedit
    global paleBackground
    global mxScroll

#    set mxScrollLeft true
    if ![info exists mxScroll] {
	set mxScroll "left"
    }
    set addSpace 0
    case $mxScroll in {
	"left" {
	    set packScroll {left filly frame w padx 3}
	    set packEdit {right expand fill frame e}
	}
	"right" {
	    set packScroll {right filly frame e }
	    set packEdit {right expand fill frame w}
	    set addSpace 1
	}
	"none" {
	    set packEdit {right expand fill frame e}
	    set addSpace 1
	}
    }

    # Frame to hold mxedit and scrollbar
    set self [selfName $parent .mx]
    frame $self -background white

    # Define a scrollbar and pack it to the left or right of the mxedit widget
    if {$mxScroll != "none"} {
	if [catch { basicScrollbar $self [list $self.edit view]  \
				    $packScroll} msg] {
	    tkerror "basicScrollbar failed: $msg"
	}
    }

    # Define the main editting window
    mxedit $self.edit -file $file -scroll $self.scroll 	\
	    -bg white -fg black -selector black \
	    -feedback $feedback -geometry $geometry -font $font
    # Remember the name of the mxedit widget so that routines
    # in mxedit.utils can easily access it
    set mxedit $self.edit
    pack append $self $mxedit $packEdit

    if $addSpace {
	pack append $self [frame [selfName $self .padding]] {left filly padx 3}
    }

    # Set up keystroke bindings.
    # They are partitioned into three groups: ksys, selection, and scrolling.
    # A variable is used to hold the name of the procedure that
    # sets up each set of bindings.  This variable is set via
    # radio button menu entries.  Changes to the variables are
    # traced and result in the execution of the procedure they name.

    global mxKeyBindingProc mxSelectBindingProc mxScrollBindingProc

    if {![info exists mxKeyBindingProc] ||
	$mxKeyBindingProc == {}} {
	set mxKeyBindingProc mxKeyBindings
    }
    if [catch {$mxKeyBindingProc $mxedit} msg] {
	tkerror "mxKeyBindingProc $mxKeyBindingProc failed: $msg"
    }
    trace variable mxKeyBindingProc w {mxChangeKeyBindings}

    if {![info exists mxSelectBindingProc] ||
	$mxSelectBindingProc == {} } {
	set mxSelectBindingProc mxSelectionBindings
    }
    if [catch {$mxSelectBindingProc $mxedit} msg] {
	tkerror "mxSelectBindingProc $mxSelectBindingProc failed: $msg"
    }
    trace variable mxSelectBindingProc w {mxChangeMouseBindings}

    if {![info exists mxScrollBindingProc] ||
	$mxScrollBindingProc == {}} {
	set mxScrollBindingProc mxScrollBindings
    }
    if [catch {$mxScrollBindingProc $mxedit} msg] {
	tkerror "mxScrollBindingProc $mxScrollBindingProc failed: $msg"
    }
    trace variable mxScrollBindingProc w {mxChangeMouseBindings}

    # Turn on history for redo
    if [catch "$mxedit history on" msg] {
	tkerror "$mxedit history on failed: $msg"
    }

    return $self
}

proc mxSetScrollBindingProc { proc } {
    global mxScrollBindingProc
    set mxScrollBindingProc $proc
}

proc mxSetSelectBindingProc { proc } {
    global mxSelectBindingProc
    set mxSelectBindingProc $proc
}

# mxChangeMouseBindings --
#	Trace procedure usually invoked when the Select or Scroll
#	binding proc variable is set.  Usually this is a side effect of
#	a radio-button menu entry to select a binding set.
#
proc mxChangeMouseBindings { name1 name2 op } {
    global mxedit
    mxClearMouseBindings $mxedit
    mxRestoreMouseBindings $mxedit
}

# mxChangeKeyBindings --
#	Trace procedure usually invoked when the Key
#	binding proc variable is set.  Usually this is a side effect of
#	a radio-button menu entry to select a binding set.
#	The value of the variable is a procedure to execute.
#
proc mxChangeKeyBindings { name1 name2 op } {
    global mxedit
    mxClearKeyBindings $mxedit
    mxRestoreKeyBindings $mxedit
}

# mxEditFocus --
#	Move focus to the editing window

proc mxEditFocus {} {
    global mxedit
    focus $mxedit
}

# mxFont --
#	Set the font of all widgets in the application

proc mxFont { fontname } {
    # Configure all the widgets to have the same font
    global FontWidgets mxFont mxedit
    if [catch {$mxedit configure -font $fontname} msg] {
	mxFeedback "Setting font to $fontname failed: $msg"
    } else {
	global buttonFont entryFont menuFont
	set mxFont $fontname
	set buttonFont $fontname
	set entryFont $fontname
	set menuFont $fontname
	foreach w $FontWidgets {
	    catch {$w configure -font $fontname}
	}
    }
}

