#
# mxedit.file --
#	The file manipulation procedures are contained here.
#	Many of these involve dialogs with the user.


proc mxDialog { frame title text {quitlabel "Dismiss"} } {
    global mxFont paleBackground

    toplevel $frame -background $paleBackground
    wm title $frame $title
    wm transient $frame

    message $frame.msg -font $mxFont -aspect 300 \
	-text $text -background $paleBackground
    pack append $frame $frame.msg {top}

    buttonFrame $frame .buttons
    pack unpack $frame.buttons
    pack append $frame $frame.buttons {top}

    framedButton $frame.buttons .dismiss \
		$quitlabel \
		"destroy $frame" left
    mxPlacePopUp $frame center
    return $frame
}

############## CALLOUTS FROM C CODE

# mxUndoErrorDialog is called from within MxSetupFileInfo if
# there was a problem creating the undo log.  This

proc mxUndoErrorDialog { text } {
    global mxUndoErrorFlag

    set mxUndoErrorFlag abort
    mxDialog .undoerror "Undo Log Error" $text "Quit"
    framedButton .undoerror.buttons .continue \
	"Continue" \
	{global mxUndoErrorFlag ; set mxUndoErrorFlag "ok" ; destroy .undoerror} \
	left
    tkwait window .undoerror
    return $mxUndoErrorFlag
}


# mxUndoErrorNotify is called from within mxUndo.c CheckError if
# there was an I/O error on the undo log.

proc mxUndoErrorNotify { text } {
    mxDialog .undoerror2 "Undo Log Error" $text
    tkwait window .undoerror2
}

# mxRecoveryDialog is called from MxSetupFileInfo if a log file
# is found for the file.

proc mxRecoveryDialog { text } {
    global mxRecoveryFlag

    set mxRecoveryFlag abort
    mxDialog .recovery "Found Recovery Log" $text "Quit"
    framedButton .recovery.buttons .recover \
	"Recover & Delete Log" \
	"global mxRecoveryFlag ; set mxRecoveryFlag recover ; destroy .recovery" \
	left
    framedButton .recovery.buttons .ignore \
	"Ignore Log" \
	{global mxRecoveryFlag ; set mxRecoveryFlag "ignore" ; destroy .recovery} \
	left
    framedButton .recovery.buttons .delete \
	"Delete Log" \
	{global mxRecoveryFlag ; set mxRecoveryFlag "delete"; destroy .recovery} \
	left
    tkwait window .recovery
    return $mxRecoveryFlag
}

proc mxRecoveryFailedDialog { text } {
    mxDialog .recovfailed "Recovery Failed" $text
}

######################## FILE PROCEDURES ########################

# These procedures are layers on the raw mxedit commands
# write, reset, switch, quit
# that protect against losing modified files
# Ordinarily these file-related procedures are accessed via the File menu
# 

# mxSave [filename] --
#	Write out the buffer to the current or named file

proc mxSave { args } {
    global mxedit mxFile
    if {[llength $args]>0} {
	set filename [lindex $args 0]
	mxSaveInner $filename [list $mxedit write $filename]
    } else {
	mxSaveInner $mxFile [list $mxedit write]
    }
}

# mxSaveSel --
#	Write out the file using the selection as the filename

proc mxSaveSel { } {
    global mxedit
    if [catch {selection get} filename] {
	mxFeedback $filename
    } else {
	mxSaveInner $filename [list $mxedit write $filename]
    }
}

proc mxSaveAsInner { top } {
    global mxedit 
    set dir [$top.fields.dir.entry get]
    if {[string length $dir] == 0} {
	set dir "."
    }
    set filename [$top.fields.file.entry get]
    if {[string length $filename] == 0} {
	mxFeedback "Please specify a file name"
	return
    }
    set filename $dir/$filename
    mxSaveInner $filename [list $mxedit write $filename]
    destroy $top
}

proc mxSaveAs { } {
    global paleBackground
    global mxFile
    catch {
	mxDialog .saveas "Save As ..." "Enter file and directory name"
	framedButton .saveas.buttons .save \
		"Save" \
		"mxSaveAsInner .saveas" left
	frame .saveas.fields -background $paleBackground
	labeledEntry .saveas.fields .dir "Dir:"
	.saveas.fields.dir.entry delete 0 end
	.saveas.fields.dir.entry insert 0 [file dirname $mxFile]
	bind .saveas.fields.dir.entry <Return> "mxSaveAsInner .saveas"
	bind .saveas.fields.dir.entry <Tab> "focus .saveas.fields.file.entry"
	labeledEntry .saveas.fields .file "File:"
	.saveas.fields.file.entry delete 0 end
	.saveas.fields.file.entry insert 0 [file tail $mxFile]
	bind .saveas.fields.file.entry <Return> "mxSaveAsInner .saveas"
	bind .saveas.fields.file.entry <Tab> "focus .saveas.fields.dir.entry"
	pack append .saveas .saveas.fields {bottom expand fill}
    }
    mxPlacePopUp .saveas center
}

# mxSaveInner --
#	Inner core of saving a file.  This checks for write errors
#	and puts up various dialog boxes to deal with them.

proc mxSaveInner { filename writeCommand } {
    global mxedit
    global mxFile mxFont
    global paleBackground

    if {$mxFile != $filename} {
	set switchCmd "mxSwitch $filename"
    } else {
	set switchCmd "set foo 0"
    }
    mxFeedback "Writing $filename ..." ;
    if [catch $writeCommand msg] {
	mxFeedback $msg
	case $msg in {
	    {{Cannot write*}} {
		mxDialog .writefailed "Write Failed" $msg
	    }
	    {{File exists*}} {
		mxDialog .fileexists "File Exists" "$msg\nDo you want to overwrite the existing file?"
		framedButton .fileexists.buttons .write \
		    "Overwrite" \
			"mxFeedback \"Writing $filename\" ; \
			$mxedit write $filename force ; \
			$switchCmd ; \
			mxFeedback \"Wrote $filename\" ; \
			destroy .fileexists" left
		tkwait window .fileexists
	    }
	    {{File modified*}} {
		mxDialog .filemodified "File Modified" "$msg\nDo you want to overwrite the newer version?"
		framedButton .filemodified.buttons .write \
		    "Overwrite" \
			"mxFeedback \"Writing $filename\" ; \
			$mxedit write $filename force ; \
			$switchCmd ; \
			mxFeedback \"Wrote $filename\" ; \
			destroy .filemodified" left
		tkwait window .filemodified
	    }
	    default {
		mxDialog .badwrite "Bad Write" "Bad TCL write cmd: $msg"
	    }
	}
	return $msg
    } else {
	if {$mxFile != $filename} {
	    mxSwitch $filename
	} else {
	    mxFeedback $msg
	}
    }
}


# mxReset --
#	Reinitialize the contents of the mxedit window from the disk file.

proc mxReset { } {
    global mxedit
    if [catch {$mxedit written} msg] {
	mxFeedback $msg
	mxResetDialog $msg
	return $msg
    } else {
	return [mxResetAlways]
    }
}

# mxResetAlways --

proc mxResetAlways { } {
    global mxedit
    # Save the current position, even though it won't be perfect
    # after the reset, it probably will be close
    set _c [mxMark caret]
    if [catch {$mxedit reset force} msg] {
	return $msg
    } else {
	mxWindowNameFix
	mxCaret $_c
	mxSee caret
	mxFeedback $msg
	return $msg
    }
}
# mxResetDialog --

proc mxResetDialog { msg } {
	global mxedit mxFont
	global paleBackground

	mxDialog .mxResetDialog "Reset Warning" "$msg\nReset anyway?"
	framedButton .mxResetDialog.buttons .reset \
		"Reset" \
		"mxResetAlways ; destroy .mxResetDialog" left
}


# mxSwitch --
#	Change to a different file

proc mxSwitch { filename } {
    global mxedit
    if [catch {$mxedit written} msg] {
	mxFeedback $msg
	mxSwitchDialog $msg $filename
    } else {
	if [catch {mxCheckSessions $filename} msg] {
	    mxFeedback $msg
	} else {
	    mxSwitchAlways $filename
	}
    }
}

# mxSwitchAlways --
#	Always switch files, even if the current on is dirty
#	This procedure remembers the previous file and position
#	for use with the switchBack command

proc mxSwitchAlways { filename } {
    global mxedit
    global mxFile mxInterpName
    global mxLastFile mxLastCaret
    global mxSwitchDebug
    if [info exists mxSwitchDebug] {
	if {$mxSwitchDebug} {
	    puts stderr "mxSwitchAlways: new $filename old $file"
	}
    }
    set _f $mxFile
    set _c [mxMark caret]
    set oldInterp $mxInterpName
    if [catch {$mxedit switch $filename force} msg] {
	mxFeedback "Switch failed: $msg"
    } else {
	set mxLastFile $_f
	set mxLastCaret $_c
	mxWindowNameFix
	mxGlobalEval mxGblSwitchFile $oldInterp $mxInterpName
	mxFeedback $msg
	return $msg
    }
}

# mxSwitchBack --
#	Switch back to the previous file
proc mxSwitchBack { } {
    global mxLastFile mxLastCaret
    if [catch {set mxLastFile}] {
	mxFeedback "No previous file to switch back to"
    } else {
	set _p $mxLastCaret
	mxSwitch $mxLastFile
	mxCaret $_p
	mxSee caret
    }
}

# mxSwitchDialog --

proc mxSwitchDialog { msg filename } {
    global mxedit
    global paleBackground

    mxDialog .mxSwitchDialog "Switch Conflict" "${msg}\nSwitch anyway?"
    framedButton .mxSwitchDialog.buttons .switch \
	    "Switch" \
	    "mxSwitchAlways $filename ; destroy .mxSwitchDialog" left
    framedButton .mxSwitchDialog.buttons .write \
	    "Save instead" \
	    "mxSave ; destroy .mxSwitchDialog" left

}


# mxQuit --
#	Quit the editing session.  This checks against a modified file
#	and puts up a dialog to ask confirmation in that case.

proc mxQuit { } {
    global mxedit 
    if [catch {$mxedit written} msg] {
	mxFeedback $msg
	mxQuitDialog $msg
    } else {
	mxQuitSafe
    }
}

# mxQuitSafe --
#	Prepare for destruction

proc mxQuitSafe {} {
    global mxedit mxInterpName
    mxGlobalEval mxGblDeleteFile $mxInterpName
    foreach sequence [bind $mxedit] {
	bind $mxedit $sequence {}	;# Unbind everything in sight
    }
    $mxedit history off		;# Prevent logging after destruction
    $mxedit quit		;# Tell the widget about it
    unset mxedit		;# Tell the tkerror proc about it
    destroy .			;# Do it
    return ""			;# Raising an error at this point
				;# causes a reference to a deleted
				;# binding table.  Just die quietly.
}

# mxQuitDialog --

proc mxQuitDialog { msg } {
    global mxedit
    global paleBackground

    mxDialog .mxQuitDialog "Really Quit?" "${msg}\nReally Quit?"
    framedButton .mxQuitDialog.buttons .quit \
	    "Quit" "mxQuitSafe" left
    framedButton .mxQuitDialog.buttons .write \
	    "Save instead" "mxSave ; destroy .mxQuitDialog" left
}

#
# mxPlacePopUp -
#	Place a popup relative to its parent window
#
proc mxPlacePopUp { widget {where center} } {
    global mxedit


    if {[string compare [screenwidth] unknown] == 0} {
	set screenWidth [lindex [exec xwininfo -root | egrep Width:] 2]
	set screenHeight [lindex [exec xwininfo -root | egrep Height:] 2]
    }

    scan [wm geometry .] "%dx%d+%d+%d" charsWide linesHigh xoff yoff
#    puts stderr "Geometry $charsWide $linesHigh $xoff $yoff"

    if ![info exists mxedit] {
	set mainWidth 100
	set mainHeight 100
    } else {
	set gridWidth [lindex [$mxedit gridsize] 0]
	set gridHeight [lindex [$mxedit gridsize] 1]
	set mainWidth [expr {$charsWide * $gridWidth}]
	set mainHeight [expr {$linesHigh * $gridHeight}]
    }

    wm withdraw $widget
    update
    scan [wm geometry $widget] "%dx%d" itsWidth itsHeight    
#    puts stderr "Its $itsWidth $itsHeight"

    set leftRoom $xoff 
    set rightRoom [expr {[screenwidth] - $xoff - $mainWidth}]

    set topRoom $yoff 
    set bottomRoom [expr {[screenheight] - $yoff - $mainHeight}]

    case $where in {
	"off" {
#	    puts stderr "mxPlacePopUp " nonewline
	    if {$leftRoom > $rightRoom} {
		set itsXoff [expr {$xoff - $itsWidth}]
		if {$itsXoff < 0} {
		    set itsXoff 0
		}
#		puts stderr "left $itsXoff " nonewline
	    } else {
		set itsXoff [expr {$xoff + $mainWidth}]
		if {[expr {$itsXoff + $itsWidth}] > [screenwidth]} {
		    set itsXoff [expr {[screenwidth] - $itsWidth}]
		}
#		puts stderr "right $itsXoff " nonewline
	    }
	    if {$topRoom > $bottomRoom} {
		set itsYoff [expr {$yoff - $itsHeight}]
		if {$itsYoff < 0} {
		    set itsYoff 0
		}
#		puts stderr "top $itsYoff " nonewline
	    } else {
		set itsYoff [expr {$yoff + $mainHeight}]
		if {[expr {$itsYoff + $itsHeight}] > [screenheight]} {
		    set itsYoff [expr {[screenheight] - $itsHeight}]
		}
#		puts stderr "bottom $itsYoff " nonewline
	    }
	}
	{ default center } {
	    if {[string compare $where center] == 0} {
		set itsXoff [expr {$xoff + ($mainWidth - $itsWidth) / 2}]
		set itsYoff [expr {$yoff + ($mainHeight - $itsHeight) / 2}]
	    }
	}
    }
    wm geometry $widget +${itsXoff}+${itsYoff}
    wm deiconify $widget
}


