/*
 * Interface to CMU mib routines for traversing the mib tree
 */
#include <stdio.h>
#include <assert.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>


#include <snmp.h>
#include <asn1.h>
#include <snmp_impl.h>
#include <snmp_api.h>
#include <snmp_client.h>
#include <party.h>
#include <context.h>
#include <mib.h>

#include <tcl.h>
#include "../snmplib/parse.h"
#include "tcl_misc.h"

void 
reverse_string(char *inbuf, char *outbuf)
{
	int             i, j, len = strlen(inbuf);
	for (i = len - 1, j = 0; j < len; j++, i--)
		outbuf[j] = inbuf[i];
	outbuf[len] = NULL;
}

char           *
get_oid(struct tree * node)
{
	struct tree    *tp = node;
	char            buf[1024];
	char            tmp1[10], tmp[10], oid_string[1024], 
			correct_oid_string[1024];
	oid_string[0] = NULL;
	while (tp != NULL) {
		sprintf(tmp1, ".%d", tp->subid);
		reverse_string(tmp1, tmp);
		if (*oid_string == NULL)
			strcpy(oid_string, tmp);
		else
			strcat(oid_string, tmp);
		tp = tp->parent;
	}
	reverse_string(oid_string, correct_oid_string);
	strcpy(buf, correct_oid_string);
	return buf;
}

char           *
print_type(struct tree * node)
{
	char            obuf[255];
	switch (node->type) {
	case TYPE_OTHER:
		strcpy(obuf, "other");
		break;
	case TYPE_OBJID:
		strcpy(obuf, "oid");
		break;
	case TYPE_OCTETSTR:
		strcpy(obuf, "string");
		break;
	case TYPE_INTEGER:
		strcpy(obuf, "integer");
		break;
	case TYPE_NETADDR:
		strcpy(obuf, "netaddress");
		break;
	case TYPE_IPADDR:
		strcpy(obuf, "ipaddress");
		break;
	case TYPE_COUNTER:
		strcpy(obuf, "counter");
		break;
	case TYPE_GAUGE:
		strcpy(obuf, "gauge");
		break;
	case TYPE_TIMETICKS:
		strcpy(obuf, "timeticks");
		break;
	case TYPE_OPAQUE:
		strcpy(obuf, "opaque");
		break;
	case TYPE_NULL:
		strcpy(obuf, "null");
		break;
	case TYPE_COUNTER64:
		strcpy(obuf, "counter64");
		break;
	case TYPE_BITSTRING:
		strcpy(obuf, "bitstring");
		break;
	case TYPE_NSAPADDRESS:
		strcpy(obuf, "nsapaddress");
		break;
	case TYPE_UINTEGER:
		strcpy(obuf, "uinteger");
		break;
	case TYPE_SEQUENCE:
		strcpy(obuf, "table");
		break;
	default:
		strcpy(obuf, "unknown");
		break;
	}
	return obuf;
}

void 
find_children(struct tree * root, char *name, char *obuf, int oid_info)
{
	struct tree    *tp, *parent;
	int             i, numeric_name = 0;
	char            tmpbuf[1024];
	assert(name);
	if (name[0] == '.' && name[1] == '1')
		numeric_name = 1;
	for (tp = root->child_list; tp; tp = tp->next_peer) {
		parent = tp->parent;
		if (parent) {
			if (numeric_name) {
				char            str_oid[1024];
				strcpy(str_oid, get_oid(parent));
				if (strcmp(str_oid, name) == 0) {
					if ( oid_info )
						sprintf(tmpbuf, "{%s %s %s} ", tp->label, print_type(tp), get_oid(tp));
					else
						sprintf(tmpbuf, "{%s} ", tp->label);
					strcat(obuf, tmpbuf);
				}
			}
			else if (strcmp(parent->label, name) == 0) {
				if ( oid_info )
					sprintf(tmpbuf, "{%s %s %s} ", tp->label, print_type(tp), get_oid(tp));
				else
					sprintf(tmpbuf, "{%s} ", tp->label);
				strcat(obuf, tmpbuf);
			}
		}
		find_children(tp, name, obuf,oid_info);
	}
}

void 
find_parent(struct tree * root, char *name, char *obuf)
{
	struct tree    *tp, *parent;
	int             i, numeric_name = 0;
	char            tmpbuf[1024];
	assert(name);
	if (name[0] == '.' && name[1] == '1')
		numeric_name = 1;
	for (tp = root->child_list; tp; tp = tp->next_peer) {
		parent = tp->parent;
		if (parent) {
			if (numeric_name) {
				char            str_oid[1024];
				strcpy(str_oid, get_oid(tp));
				if (strcmp(str_oid, name) == 0) {
					sprintf(tmpbuf, "%s", parent->label);
					strcat(obuf, tmpbuf);
					return;
				}
			}
			else if (strcmp(tp->label, name) == 0) {
				sprintf(tmpbuf, "%s", parent->label);
				strcat(obuf, tmpbuf);
				return;
			}
		}
		find_parent(tp, name, obuf);
	}
}

void 
get_info(struct tree * root, char *name, char *obuf)
{
	struct tree    *tp, *parent;
	int             i, numeric_name = 0;
	char            tmpbuf[1024];
	assert(name);
	if (name[0] == '.' && name[1] == '1')
		numeric_name = 1;
	for (tp = root->child_list; tp; tp = tp->next_peer) {
		if (numeric_name) {
			char            str_oid[1024];
			strcpy(str_oid, get_oid(tp));
			if (strcmp(str_oid, name) == 0) {
				sprintf(tmpbuf, "{%s %s %s \"%s\" } ", tp->label, print_type(tp), get_oid(tp),tp->description);
				strcat(obuf, tmpbuf);
				return;
			}
		}
		else if (strcmp(tp->label, name) == 0) {
			sprintf(tmpbuf, "{%s %s %s \"%s\" } ", tp->label, print_type(tp), get_oid(tp),tp->description);
			strcat(obuf, tmpbuf);
			return;
		}
		get_info(tp, name, obuf);
	}
}

/*
 * Creates Tcl command mib which has following sub-commands. 
 * children  :lists all the children of the specified node with their OIDs
 * lchildren :lists all the children of the specified node
 * get_info  :returns the attribute of specified node.
 * parent    :return name of the parent for the specified node.  
 *
 * Here is the sample session output.

./snmptcl
% set lchildren [mib lchildren system]
{sysServices} {sysLocation} {sysName} {sysContact} {sysUpTime} {sysObjectID} {sysDescr}
% set child_list [mib children system]
{sysServices integer .1.3.6.1.2.1.1.7} {sysLocation string .1.3.6.1.2.1.1.6} {sysName string .1.3.6.1.2.1.1.5} {sysContact string .1.3.6.1.2.1.1.4} {sysUpTime timeticks .1.3.6.1.2.1.1.3} {sysObjectID oid .1.3.6.1.2.1.1.2} {sysDescr string .1.3.6.1.2.1.1.1}
% set full_info [mib get_info system]
{system other .1.3.6.1.2.1.1 "(null)" }
% set parent_name [mib parent system]
mib-2
%
 *
 */ 
int 
MibProc(ClientData clientData, Tcl_Interp * interp, int argc, char **argv)
{
	int             i = 0;
	char            obuf[2048], *mib_prefix,buffer[2048];
	struct tree    *root = (struct tree *) clientData;
	obuf[0] = NULL;
	if (argc != 3) {
		interp->result = "wrong # args";
		return TCL_ERROR;
	}
	if ( isdigit(argv[2][0]) ) { /* we are talking digits here. */
		mib_prefix = Tcl_GetVar(interp,"mib_prefix",TCL_GLOBAL_ONLY);
		if ( mib_prefix ){
			sprintf(buffer, "%s.%s", mib_prefix,argv[2]);
			argv[2] = malloc(strlen(buffer)+1);
			strcpy(argv[2],buffer);
		}
	}
	if (strcmp(argv[1], "children") == 0) {
		find_children(root, argv[2], obuf,TRUE);
		Tcl_SetResult(interp,
			      obuf,
			      TCL_DYNAMIC);
		return TCL_OK;
	} else if (strcmp(argv[1], "lchildren") == 0) {
		find_children(root, argv[2], obuf,FALSE);
		Tcl_SetResult(interp,
			      obuf,
			      TCL_DYNAMIC);
		return TCL_OK;
	} else if ( strcmp(argv[1], "get_info") == 0 ) {
		get_info(root, argv[2], obuf);
		Tcl_SetResult(interp,
			      obuf,
			      TCL_DYNAMIC);
		return TCL_OK;
	} else if ( strcmp(argv[1], "parent") == 0 ) {
		find_parent(root, argv[2], obuf);
		Tcl_SetResult(interp,
			      obuf,
			      TCL_DYNAMIC);
		return TCL_OK;
	}
	else {
		interp->result = "wrong arguments";
		return TCL_ERROR;
	}
}

int 
Mib2Proc(ClientData clientData, Tcl_Interp * interp, int argc, char **argv)
{
	int             i = 0;
	char            buffer[255];
	char            obuf[8096];
	struct tree    *root = (struct tree *) clientData;
	obuf[0] = NULL;
	if (argc != 3) {
		interp->result = "wrong # args";
		return TCL_ERROR;
	}
	if (argv[2][0] == '.') {
			interp->result = "wrong parameter, this tree is rooted at mgmt.";
			return TCL_ERROR;
	}
	sprintf(buffer, ".1.3.6.1.2.%s", argv[2]);
	argv[2] = malloc(strlen(buffer)+1);
	strcpy(argv[2],buffer);
	return MibProc(clientData,interp,argc,argv);
}
