
/*
 *
 * bltTreeCmd.c --
 *
 *	Copyright 1998-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
  tree create t0 t1 t2
  tree names
  t0 destroy
     -or-
  tree destroy t0
  tree copy tree@node tree@node -recurse -tags

  tree move node after|before|into t2@node

  $t apply -recurse $root command arg arg			

  $t attach treename				

  $t children $n
  t0 copy node1 node2 node3 node4 node5 destName 
  $t delete $n...				
  $t depth $n
  $t dump $root
  $t dumpfile $root fileName
  $t dup $t2		
  $t find $root -name pat -name pattern
  $t firstchild $n
  $t get $n $key
  $t get $n $key(abc)
  $t index $n
  $t insert $parent $switches?
  $t isancestor $n1 $n2
  $t isbefore $n1 $n2
  $t isleaf $n
  $t lastchild $n
  $t move $n1 after|before|into $n2
  $t next $n
  $t nextsibling $n
  $t path $n1 $n2 $n3...
  $t parent $n
  $t previous $n
  $t prevsibling $n
  $t restore $root data -overwrite
  $t root ?$n?

  $t set $n $key $value ?$key $value?
  $t size $n
  $t slink $n $t2@$node				???
  $t sort -recurse $root		

  $t tag delete tag1 tag2 tag3...
  $t tag names
  $t tag nodes $tag
  $t tag set $n tag1 tag2 tag3...
  $t tag unset $n tag1 tag2 tag3...

  $t trace create $n $key how command		
  $t trace delete id1 id2 id3...
  $t trace names
  $t trace info $id

  $t unset $n key1 key2 key3...
  
  $t notify create -oncreate -ondelete -onmove command 
  $t notify create -oncreate -ondelete -onmove -onsort command arg arg arg 
  $t notify delete id1 id2 id3
  $t notify names
  $t notify info id

  for { set n [$t firstchild $node] } { $n >= 0 } { 
        set n [$t nextsibling $n] } {
  }
  foreach n [$t children $node] { 
	  
  }
  set n [$t next $node]
  set n [$t previous $node]

*/

#include <bltInt.h>

#ifndef NO_TREE
#include <bltTree.h>
#include <bltHash.h>
#include <bltList.h>
#include "bltNsUtil.h"
#include "bltSwitch.h"
#include <ctype.h>

#define TREE_THREAD_KEY "BLT Tree Command Data"
#define TREE_MAGIC ((unsigned int) 0x46170277)

enum TagTypes { TAG_TYPE_NONE, TAG_TYPE_ALL, TAG_TYPE_TAG };

typedef struct {
    Blt_HashTable treeTable;	/* Hash table of trees keyed by address. */
    Tcl_Interp *interp;
} TreeCmdInterpData;

typedef struct {
    Tcl_Interp *interp;
    Tcl_Command cmdToken;	/* Token for tree's Tcl command. */
    Blt_Tree tree;		/* Token holding internal tree. */

    Blt_HashEntry *hashPtr;
    Blt_HashTable *tablePtr;

    TreeCmdInterpData *tdPtr;	/*  */

    int traceCounter;		/* Used to generate trace id strings.  */
    Blt_HashTable traceTable;	/* Table of active traces. Maps trace ids
				 * back to their TraceInfo records. */

    int notifyCounter;		/* Used to generate notify id strings. */
    Blt_HashTable notifyTable;	/* Table of event handlers. Maps notify ids
				 * back to their NotifyInfo records. */
} TreeCmd;

typedef struct {
    TreeCmd *cmdPtr;
    Blt_TreeNode node;

    Blt_TreeTrace traceToken;

    char *withTag;		/* If non-NULL, the event or trace was
				 * specified with this tag.  This
				 * value is saved for informational
				 * purposes.  The tree's trace
				 * matching routines do the real
				 * checking, not the client's
				 * callback.  */

    char command[1];		/* Command prefix for the trace or notify
				 * Tcl callback.  Extra arguments will be
				 * appended to the end. Extra space will
				 * be allocated for the length of the string
				 */
} TraceInfo;

typedef struct {
    TreeCmd *cmdPtr;
    int mask;
    Tcl_Obj **objv;
    int objc;
    Blt_TreeNode node;		/* Node affected by event. */
    Blt_TreeTrace notifyToken;
} NotifyInfo;

typedef struct {
    int mask;
} NotifySwitches;

static Blt_SwitchSpec notifySwitches[] = 
{
    {BLT_SWITCH_FLAG, "-create", Blt_Offset(NotifySwitches, mask), 0, 0, 
	TREE_NOTIFY_CREATE},
    {BLT_SWITCH_FLAG, "-delete", Blt_Offset(NotifySwitches, mask), 0, 0, 
	TREE_NOTIFY_DELETE},
    {BLT_SWITCH_FLAG, "-move", Blt_Offset(NotifySwitches, mask), 0, 0, 
	TREE_NOTIFY_MOVE},
    {BLT_SWITCH_FLAG, "-sort", Blt_Offset(NotifySwitches, mask), 0, 0, 
	TREE_NOTIFY_SORT},
    {BLT_SWITCH_FLAG, "-relabel", Blt_Offset(NotifySwitches, mask), 0, 0, 
	TREE_NOTIFY_RELABEL},
    {BLT_SWITCH_FLAG, "-allevents", Blt_Offset(NotifySwitches, mask), 0, 0, 
	TREE_NOTIFY_ALL},
    {BLT_SWITCH_FLAG, "-whenidle", Blt_Offset(NotifySwitches, mask), 0, 0, 
	TREE_NOTIFY_WHENIDLE},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchParseProc ChildSwitch;
#define INSERT_BEFORE	(ClientData)0
#define INSERT_AFTER	(ClientData)1
static Blt_SwitchCustom beforeSwitch = {
    ChildSwitch, NULL, INSERT_BEFORE,
};
static Blt_SwitchCustom afterSwitch = {
    ChildSwitch, NULL, INSERT_AFTER,
};

typedef struct {
    char *label;
    long position;
    long inode;
    char **tags;
    char **dataPairs;
    Blt_TreeNode parent;
} InsertSwitches;

static Blt_SwitchSpec insertSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-after", Blt_Offset(InsertSwitches, position), 0, 
	&afterSwitch},
    {BLT_SWITCH_LONG_NONNEGATIVE, "-at", Blt_Offset(InsertSwitches, position), 
	0},
    {BLT_SWITCH_CUSTOM, "-before", Blt_Offset(InsertSwitches, position), 0,
	&beforeSwitch},
    {BLT_SWITCH_LIST, "-data", Blt_Offset(InsertSwitches, dataPairs), 0},
    {BLT_SWITCH_STRING, "-label", Blt_Offset(InsertSwitches, label), 0},
    {BLT_SWITCH_LONG_NONNEGATIVE, "-node", Blt_Offset(InsertSwitches, inode), 
	0},
    {BLT_SWITCH_LIST, "-tags", Blt_Offset(InsertSwitches, tags), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

#define PATTERN_EXACT	(1)
#define PATTERN_GLOB	(2)
#define PATTERN_MASK	(0x3)
#define PATTERN_NONE	(0)
#define PATTERN_REGEXP	(3)
#define MATCH_INVERT		(1<<8)
#define MATCH_LEAFONLY		(1<<4)
#define MATCH_NOCASE		(1<<5)
#define MATCH_PATHNAME		(1<<6)

typedef struct {
    TreeCmd *cmdPtr;		/* Tree to examine. */
    Tcl_Obj *listObjPtr;	/* List to accumulate the indices of 
				 * matching nodes. */
    Tcl_Obj **objv;		/* Command converted into an array of 
				 * Tcl_Obj's. */
    int objc;			/* Number of Tcl_Objs in above array. */

    long nMatches;		/* Current number of matches. */

    unsigned int flags;		/* See flags definitions above. */

    /* Integer options. */
    long maxMatches;		/* If > 0, stop after this many matches. */
    long maxDepth;		/* If > 0, don't descend more than this
				 * many levels. */
    int order;			/* Order of search: Can be either
				 * TREE_PREORDER, TREE_POSTORDER,
				 * TREE_INORDER, TREE_BREADTHFIRST. */
    /* String options. */
    Blt_List patternList;	/* List of patterns to compare with labels
				 * or values.  */
    char *addTag;		/* If non-NULL, tag added to selected nodes. */

    char **command;		/* Command split into a Tcl list. */

    Blt_List keyList;		/* List of key name patterns. */
    Blt_List tagList;		/* List of tag names. */
    Blt_HashTable excludeTable;	/* Table of nodes to exclude. */
} FindSwitches;

static Blt_SwitchParseProc OrderSwitch;
static Blt_SwitchCustom orderSwitch = {
    OrderSwitch, NULL, (ClientData)0,
};

static Blt_SwitchParseProc PatternSwitch;
static Blt_SwitchFreeProc FreePatterns;

static Blt_SwitchCustom regexpSwitch = {
    PatternSwitch, FreePatterns, (ClientData)PATTERN_REGEXP,
};
static Blt_SwitchCustom globSwitch = {
    PatternSwitch, FreePatterns, (ClientData)PATTERN_GLOB,
};
static Blt_SwitchCustom exactSwitch = {
    PatternSwitch, FreePatterns, (ClientData)PATTERN_EXACT,
};

static Blt_SwitchCustom tagSwitch = {
    PatternSwitch, FreePatterns, (ClientData)PATTERN_EXACT,
};

static Blt_SwitchParseProc NodesSwitch;
static Blt_SwitchFreeProc FreeNodes;
static Blt_SwitchCustom nodesSwitch = {
    NodesSwitch, FreeNodes, (ClientData)0,
};

static Blt_SwitchSpec findSwitches[] = {
    {BLT_SWITCH_STRING, "-addtag", Blt_Offset(FindSwitches, addTag), 0},
    {BLT_SWITCH_LONG_NONNEGATIVE, "-count", Blt_Offset(FindSwitches, maxMatches),0},
    {BLT_SWITCH_LONG_NONNEGATIVE, "-depth", Blt_Offset(FindSwitches, maxDepth), 0},
    {BLT_SWITCH_CUSTOM, "-exact", Blt_Offset(FindSwitches, patternList), 0,
        &exactSwitch},
    {BLT_SWITCH_CUSTOM, "-excludes", Blt_Offset(FindSwitches, excludeTable), 0, 
	&nodesSwitch},
    {BLT_SWITCH_LIST, "-exec", Blt_Offset(FindSwitches, command), 0},
    {BLT_SWITCH_CUSTOM, "-glob", Blt_Offset(FindSwitches, patternList), 0, 
	&globSwitch},
    {BLT_SWITCH_FLAG, "-invert", Blt_Offset(FindSwitches, flags), 0, 0, 
	MATCH_INVERT},
    {BLT_SWITCH_CUSTOM, "-key", Blt_Offset(FindSwitches, keyList), 0,&exactSwitch},
    {BLT_SWITCH_CUSTOM, "-keyexact", Blt_Offset(FindSwitches, keyList), 0, 
	&exactSwitch},
    {BLT_SWITCH_CUSTOM, "-keyglob", Blt_Offset(FindSwitches, keyList), 0, 
	&globSwitch},
    {BLT_SWITCH_CUSTOM, "-keyregexp", Blt_Offset(FindSwitches, keyList), 0, 
	&regexpSwitch},
    {BLT_SWITCH_FLAG, "-leafonly", Blt_Offset(FindSwitches, flags), 0, 0, 
	MATCH_LEAFONLY},
    {BLT_SWITCH_FLAG, "-nocase", Blt_Offset(FindSwitches, flags), 0, 0, 
	MATCH_NOCASE},
    {BLT_SWITCH_CUSTOM, "-order", Blt_Offset(FindSwitches, order), 0, &orderSwitch},
    {BLT_SWITCH_FLAG, "-path", Blt_Offset(FindSwitches, flags), 0, 0, 
	MATCH_PATHNAME},
    {BLT_SWITCH_CUSTOM, "-regexp", Blt_Offset(FindSwitches, patternList), 0, 
	&regexpSwitch},
    {BLT_SWITCH_CUSTOM, "-tag", Blt_Offset(FindSwitches, tagList), 0, &tagSwitch},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchParseProc NodeSwitch;
static Blt_SwitchCustom nodeSwitch = {
    NodeSwitch, NULL, (ClientData)0,
};

typedef struct {
    TreeCmd *cmdPtr;		/* Tree to move nodes. */
    Blt_TreeNode node;
    long movePos;
} MoveSwitches;

static Blt_SwitchSpec moveSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-after", Blt_Offset(MoveSwitches, node), 0, 
	&nodeSwitch},
    {BLT_SWITCH_LONG_NONNEGATIVE, "-at", Blt_Offset(MoveSwitches, movePos), 0},
    {BLT_SWITCH_CUSTOM, "-before", Blt_Offset(MoveSwitches, node), 0, 
	&nodeSwitch},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    Blt_TreeNode srcNode;
    Blt_Tree srcTree, destTree;
    TreeCmd *srcPtr, *destPtr;
    char *label;
    unsigned int flags;
} CopySwitches;

#define COPY_RECURSE	(1<<0)
#define COPY_TAGS	(1<<1)
#define COPY_OVERWRITE	(1<<2)

static Blt_SwitchSpec copySwitches[] = 
{
    {BLT_SWITCH_STRING, "-label", Blt_Offset(CopySwitches, label), 0},
    {BLT_SWITCH_FLAG, "-recurse", Blt_Offset(CopySwitches, flags), 0, 0, 
	COPY_RECURSE},
    {BLT_SWITCH_FLAG, "-tags", Blt_Offset(CopySwitches, flags), 0, 0, COPY_TAGS},
    {BLT_SWITCH_FLAG, "-overwrite", Blt_Offset(CopySwitches, flags), 0, 0, 
	COPY_OVERWRITE},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    TreeCmd *cmdPtr;		/* Tree to examine. */
    Tcl_Obj **preObjv;		/* Command converted into an array of 
				 * Tcl_Obj's. */
    int preObjc;		/* Number of Tcl_Objs in above array. */

    Tcl_Obj **postObjv;		/* Command converted into an array of 
				 * Tcl_Obj's. */
    int postObjc;		/* Number of Tcl_Objs in above array. */

    unsigned int flags;		/* See flags definitions above. */

    long maxDepth;		/* If > 0, don't descend more than this
				 * many levels. */
    /* String options. */
    Blt_List patternList;	/* List of label or value patterns. */
    char **preCmd;		/* Pre-command split into a Tcl list. */
    char **postCmd;		/* Post-command split into a Tcl list. */

    Blt_List keyList;		/* List of key-name patterns. */
    Blt_List tagList;
} ApplySwitches;

static Blt_SwitchSpec applySwitches[] = 
{
    {BLT_SWITCH_LIST, "-precommand", Blt_Offset(ApplySwitches, preCmd), 0},
    {BLT_SWITCH_LIST, "-postcommand", Blt_Offset(ApplySwitches, postCmd), 0},
    {BLT_SWITCH_LONG_NONNEGATIVE, "-depth", Blt_Offset(ApplySwitches, maxDepth), 0},
    {BLT_SWITCH_CUSTOM, "-exact", Blt_Offset(ApplySwitches, patternList), 0,
	&exactSwitch},
    {BLT_SWITCH_CUSTOM, "-glob", Blt_Offset(ApplySwitches, patternList), 0, 
	&globSwitch},
    {BLT_SWITCH_FLAG, "-invert", Blt_Offset(ApplySwitches, flags), 0, 0, 
	MATCH_INVERT},
    {BLT_SWITCH_CUSTOM, "-key", Blt_Offset(ApplySwitches, keyList), 0, 
	&exactSwitch},
    {BLT_SWITCH_CUSTOM, "-keyexact", Blt_Offset(ApplySwitches, keyList), 0, 
	&exactSwitch},
    {BLT_SWITCH_CUSTOM, "-keyglob", Blt_Offset(ApplySwitches, keyList), 0, 
	&globSwitch},
    {BLT_SWITCH_CUSTOM, "-keyregexp", Blt_Offset(ApplySwitches, keyList), 0, 
	&regexpSwitch},
    {BLT_SWITCH_FLAG, "-leafonly", Blt_Offset(ApplySwitches, flags), 0, 0, 
	MATCH_LEAFONLY},
    {BLT_SWITCH_FLAG, "-nocase", Blt_Offset(ApplySwitches, flags), 0, 0, 
	MATCH_NOCASE},
    {BLT_SWITCH_FLAG, "-path", Blt_Offset(ApplySwitches, flags), 0, 0, 
	MATCH_PATHNAME},
    {BLT_SWITCH_CUSTOM, "-regexp", Blt_Offset(ApplySwitches, patternList), 0,
	&regexpSwitch},
    {BLT_SWITCH_CUSTOM, "-tag", Blt_Offset(ApplySwitches, tagList), 0, &tagSwitch},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    unsigned int flags;
} RestoreSwitches;


#define RESTORE_NO_TAGS		(1<<0)
#define RESTORE_OVERWRITE	(1<<1)

static Blt_SwitchSpec restoreSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-notags", Blt_Offset(RestoreSwitches, flags), 0, 0, 
	RESTORE_NO_TAGS},
    {BLT_SWITCH_FLAG, "-overwrite", Blt_Offset(RestoreSwitches, flags), 0, 0, 
	RESTORE_OVERWRITE},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    unsigned int flags;
} ImportSwitches;

#define IMPORT_OVERWRITE	(1<<0)

static Blt_SwitchSpec importSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-overwrite", Blt_Offset(ImportSwitches, flags), 0, 0, 
	IMPORT_OVERWRITE},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchParseProc FormatSwitch;
static Blt_SwitchCustom formatSwitch =
{
    FormatSwitch, NULL, (ClientData)0,
};

typedef struct {
    int sort;			/* If non-zero, sort the nodes.  */
    int withParent;		/* If non-zero, add the parent node id 
				 * to the output of the command.*/
    int withId;			/* If non-zero, echo the node id in the
				 * output of the command. */
} PositionSwitches;

#define POSITION_SORTED		(1<<0)

static Blt_SwitchSpec positionSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-sort", Blt_Offset(PositionSwitches, sort), 0, 0,
       POSITION_SORTED},
    {BLT_SWITCH_CUSTOM, "-format", 0, 0, &formatSwitch},
    {BLT_SWITCH_END, NULL, 0, 0}
};

#define PARSE_NEWLEAF		(1<<0)
#define PARSE_AUTOCREATE	(1<<1)

typedef struct {
    unsigned int flags;		/* Parse flags. */
    char *separator;		/* Path separator. */
} ParsePathSwitches;

static Blt_SwitchSpec parsePathSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-autocreate", Blt_Offset(ParsePathSwitches, flags), 0, 0,
	PARSE_AUTOCREATE},
    {BLT_SWITCH_FLAG, "-newleaf", Blt_Offset(ParsePathSwitches, flags), 0, 0, 
        PARSE_NEWLEAF},
    {BLT_SWITCH_STRING, "-separator", Blt_Offset(ParsePathSwitches, separator),
	0}, 
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Tcl_InterpDeleteProc TreeInterpDeleteProc;
static Blt_TreeApplyProc MatchNodeProc, SortApplyProc;
static Blt_TreeApplyProc ApplyNodeProc;
static Blt_TreeTraceProc TreeTraceProc;
static Tcl_CmdDeleteProc TreeInstDeleteProc;
static Blt_TreeCompareNodesProc CompareNodes;

static Tcl_ObjCmdProc TreeObjCmd;
static Tcl_ObjCmdProc CompareDictionaryCmd;
static Tcl_ObjCmdProc ExitCmd;
static Blt_TreeNotifyEventProc TreeEventProc;

static int GetNode _ANSI_ARGS_((Tcl_Interp *interp, TreeCmd *cmdPtr, 
	Tcl_Obj *objPtr, Blt_TreeNode *nodePtr));

static int
IsNodeIdOrModifier(CONST char *string)
{
    CONST char *p;

    if (strstr(string, "->") == NULL) {
	for (p = string; *p != '\0'; p++) {
	    if (!isdigit(UCHAR(*p))) {
		return FALSE;
	    }
	}
    }
    return TRUE;
}

static int
IsNodeId(CONST char *string)
{
    CONST char *p;

    for (p = string; *p != '\0'; p++) {
	if (!isdigit(UCHAR(*p))) {
	    return FALSE;
	}
    }
    return TRUE;
}

/*
 *----------------------------------------------------------------------
 *
 * ChildSwitch --
 *
 *	Convert a Tcl_Obj representing the label of a child node into its 
 *	integer node id.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ChildSwitch(
    ClientData clientData,	/* Flag indicating if the node is
				 * considered before or after the
				 * insertion position. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    InsertSwitches *insertPtr = (InsertSwitches *)record;
    Blt_TreeNode node;
    char *string;

    string = Tcl_GetString(objPtr);
    node = Blt_TreeFindChild(insertPtr->parent, string);
    if (node == NULL) {
	Tcl_AppendResult(interp, "can't find a child named \"", string, 
		 "\" in \"", Blt_TreeNodeLabel(insertPtr->parent), "\"",
		 (char *)NULL);	 
	return TCL_ERROR;
    }			  
    insertPtr->position = Blt_TreeNodeDegree(node);
    if (clientData == INSERT_AFTER) {
	insertPtr->position++;
    } 
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NodeSwitch --
 *
 *	Convert a Tcl_Obj representing a node number into its integer
 *	value.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
NodeSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    MoveSwitches *movePtr = (MoveSwitches *)record;
    Blt_TreeNode node;
    TreeCmd *cmdPtr = movePtr->cmdPtr;
    int result;

    result = GetNode(interp, cmdPtr, objPtr, &node);
    if (result != TCL_OK) {
	return TCL_ERROR;
    }
    movePtr->node = node;
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * OrderSwitch --
 *
 *	Convert a string represent a node number into its integer
 *	value.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
OrderSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    int *orderPtr = (int *)(record + offset);
    char c;
    char *string;

    string = Tcl_GetString(objPtr);
    c = string[0];
    if ((c == 'b') && (strcmp(string, "breadthfirst") == 0)) {
	*orderPtr = TREE_BREADTHFIRST;
    } else if ((c == 'i') && (strcmp(string, "inorder") == 0)) {
	*orderPtr = TREE_INORDER;
    } else if ((c == 'p') && (strcmp(string, "preorder") == 0)) {
	*orderPtr = TREE_PREORDER;
    } else if ((c == 'p') && (strcmp(string, "postorder") == 0)) {
	*orderPtr = TREE_POSTORDER;
    } else {
	Tcl_AppendResult(interp, "bad order \"", string, 
		 "\": should be breadthfirst, inorder, preorder, or postorder",
		 (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * PatternSwitch --
 *
 *	Convert a string represent a node number into its integer
 *	value.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
PatternSwitch(
    ClientData clientData,	/* Flag indicating type of pattern. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Blt_List *listPtr = (Blt_List *)(record + offset);

    if (*listPtr == NULL) {
	*listPtr = Blt_ListCreate(BLT_STRING_KEYS);
    }
    Blt_ListAppend(*listPtr, Tcl_GetString(objPtr), clientData);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FreePatterns --
 *
 *	Convert a string represent a node number into its integer
 *	value.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
FreePatterns(char *record, int offset, int flags)
{
    Blt_List *listPtr = (Blt_List *)(record + offset);

    if (*listPtr != NULL) {
	Blt_ListDestroy(*listPtr);
	/* This routine can be called several times for each switch
	 * that appends to this list. Mark it NULL, so we don't try to
	 * destroy the list again. */
	*listPtr = NULL;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * FormatSwitch --
 *
 *	Convert a string represent a node number into its integer
 *	value.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
FormatSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    PositionSwitches *pdPtr = (PositionSwitches *)record;
    char *string;

    string = Tcl_GetString(objPtr);
    if (strcmp(string, "position") == 0) {
	pdPtr->withParent = FALSE;
	pdPtr->withId = FALSE;
    } else if (strcmp(string, "id+position") == 0) {
	pdPtr->withParent = FALSE;
	pdPtr->withId = TRUE;
    } else if (strcmp(string, "parent-at-position") == 0) {
	pdPtr->withParent = TRUE;
	pdPtr->withId = FALSE;
    } else if (strcmp(string, "id+parent-at-position") == 0) {
	pdPtr->withParent = TRUE;
	pdPtr->withId  = TRUE;
    } else {
	Tcl_AppendResult(interp, "bad format \"", string, 
 "\": should be position, parent-at-position, id+position, or id+parent-at-position",
		 (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FreeNodes --
 *
 *	Convert a string represent a node number into its integer
 *	value.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
FreeNodes(char *record, int offset, int flags)
{
    Blt_HashTable *tablePtr = (Blt_HashTable *)(record + offset);

    Blt_DeleteHashTable(tablePtr);
}

/*
 *----------------------------------------------------------------------
 *
 * NodesSwitch --
 *
 *	Convert a Tcl_Obj representing a node number into its integer
 *	value.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
NodesSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    FindSwitches *findPtr = (FindSwitches *)record;
    int objc;
    Tcl_Obj **objv;
    int i;
    
    if (Tcl_ListObjGetElements(interp, objPtr, &objc, &objv) != TCL_OK) {
	return TCL_ERROR;
    }
    for (i = 0; i < objc; i++) {
	int isNew;
	Blt_TreeNode node;

	if (GetNode(interp, findPtr->cmdPtr, objv[i], &node) != TCL_OK) {
	    Blt_DeleteHashTable(&findPtr->excludeTable);
	    return TCL_ERROR;
	}
	Blt_CreateHashEntry(&findPtr->excludeTable, (char *)node, &isNew);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * GetTreeCmdInterpData --
 *
 *---------------------------------------------------------------------- 
 */
static TreeCmdInterpData *
GetTreeCmdInterpData(Tcl_Interp *interp)
{
    TreeCmdInterpData *tdPtr;
    Tcl_InterpDeleteProc *proc;

    tdPtr = (TreeCmdInterpData *)
	Tcl_GetAssocData(interp, TREE_THREAD_KEY, &proc);
    if (tdPtr == NULL) {
	tdPtr = Blt_Malloc(sizeof(TreeCmdInterpData));
	assert(tdPtr);
	tdPtr->interp = interp;
	Tcl_SetAssocData(interp, TREE_THREAD_KEY, TreeInterpDeleteProc,
		 tdPtr);
	Blt_InitHashTable(&tdPtr->treeTable, BLT_ONE_WORD_KEYS);
    }
    return tdPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * GetTreeCmd --
 *
 *	Find the tree command associated with the Tcl command "string".
 *	
 *	We have to do multiple lookups to get this right.  
 *
 *	The first step is to generate a canonical command name.  If an
 *	unqualified command name (i.e. no namespace qualifier) is
 *	given, we should search first the current namespace and then
 *	the global one.  Most Tcl commands (like Tcl_GetCmdInfo) look
 *	only at the global namespace.
 *
 *	Next check if the string is 
 *		a) a Tcl command and 
 *		b) really is a command for a tree object.  
 *	Tcl_GetCommandInfo will get us the objClientData field that 
 *	should be a cmdPtr.  We can verify that by searching our hashtable 
 *	of cmdPtr addresses.
 *
 * Results:
 *	A pointer to the tree command.  If no associated tree command
 *	can be found, NULL is returned.  It's up to the calling routines
 *	to generate an error message.
 *
 *---------------------------------------------------------------------- 
 */
static TreeCmd *
GetTreeCmd(
    TreeCmdInterpData *tdPtr, 
    Tcl_Interp *interp, 
    CONST char *string)
{
    Blt_ObjectName objName;
    Tcl_CmdInfo cmdInfo;
    Blt_HashEntry *hPtr;
    Tcl_DString dString;
    char *treeName;
    int result;

    /* Put apart the tree name and put is back together in a standard
     * format. */
    if (!Blt_ParseObjectName(interp, string, &objName, BLT_NO_ERROR_MSG)) {
	return NULL;		/* No such parent namespace. */
    }
    /* Rebuild the fully qualified name. */
    treeName = Blt_MakeQualifiedName(&objName, &dString);
    result = Tcl_GetCommandInfo(interp, treeName, &cmdInfo);
    Tcl_DStringFree(&dString);

    if (!result) {
	return NULL;
    }
    hPtr = Blt_FindHashEntry(&tdPtr->treeTable, 
			     (char *)(cmdInfo.objClientData));
    if (hPtr == NULL) {
	return NULL;
    }
    return Blt_GetHashValue(hPtr);
}

static Blt_TreeNode 
ParseModifiers(
    Tcl_Interp *interp,
    Blt_Tree tree,
    Blt_TreeNode node,
    char *modifiers)
{
    char *p, *token;

    p = modifiers;
    do {
	p += 2;			/* Skip the initial "->" */
	token = strstr(p, "->");
	if (token != NULL) {
	    *token = '\0';
	}
	if (IsNodeId(p)) {
	    long inode;
	    
	    if (Tcl_GetLong(interp, p, &inode) != TCL_OK) {
		node = NULL;
	    } else {
		node = Blt_TreeGetNode(tree, inode);
	    }
	} else if ((*p == 'p') && (strcmp(p, "parent") == 0)) {
	    node = Blt_TreeNodeParent(node);
	} else if ((*p == 'f') && (strcmp(p, "firstchild") == 0)) {
	    node = Blt_TreeFirstChild(node);
	} else if ((*p == 'l') && (strcmp(p, "lastchild") == 0)) {
	    node = Blt_TreeLastChild(node);
	} else if ((*p == 'n') && (strcmp(p, "next") == 0)) {
	    node = Blt_TreeNextNode(NULL, node);
	} else if ((*p == 'n') && (strcmp(p, "nextsibling") == 0)) {
	    node = Blt_TreeNextSibling(node);
	} else if ((*p == 'p') && (strcmp(p, "previous") == 0)) {
	    node = Blt_TreePrevNode(NULL, node);
	} else if ((*p == 'p') && (strcmp(p, "prevsibling") == 0)) {
	    node = Blt_TreePrevSibling(node);
	} else {
	    int length;

	    length = strlen(p);
	    if (length > 0) {
		char *endp;

		endp = p + length - 1;
		if ((*p == '"') && (*endp == '"')) {
		    *endp = '\0';
		    node = Blt_TreeFindChild(node, p + 1);
		    *endp = '"';
		} else {
		    node = Blt_TreeFindChild(node, p);
		}		
	    }
	}
	if (node == NULL) {
	    goto error;
	}
	if (token != NULL) {
	    *token = '-';	/* Repair the string */
	}
	p = token;
    } while (token != NULL);
    return node;
 error:
    if (token != NULL) {
	*token = '-';		/* Repair the string */
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * GetForeignNode --
 *
 *---------------------------------------------------------------------- 
 */
static int 
GetForeignNode(
    Tcl_Interp *interp,
    Blt_Tree tree,
    Tcl_Obj *objPtr,
    Blt_TreeNode *nodePtr)
{
    Blt_TreeNode node;
    char *string;
    char *p;
    char save;

    save = '\0';		/* Suppress compiler warning. */
    string = Tcl_GetString(objPtr);

    /* 
     * Check if modifiers are present.
     */
    p = strstr(string, "->");
    if (p != NULL) {
	save = *p;
	*p = '\0';
    }
    if (IsNodeId(string)) {
	long inode;

	if (p != NULL) {
	    if (TclGetLong(interp, string, &inode) != TCL_OK) {
		goto error;
	    }
	} else {
	    if (Tcl_GetLongFromObj(interp, objPtr, &inode) != TCL_OK) {
		goto error;
	    }
	}
	node = Blt_TreeGetNode(tree, inode);
	if (p != NULL) {
	    node = ParseModifiers(interp, tree, node, p);
	}
	if (node != NULL) {
	    *nodePtr = node;
	    if (p != NULL) {
		*p = save;
	    }
	    return TCL_OK;
	}
    }
    Tcl_AppendResult(interp, "can't find tag or id \"", string, "\" in ",
	 Blt_TreeName(tree), (char *)NULL);
 error:
    if (p != NULL) {
	*p = save;		/* Restore the string */
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * GetNode --
 *
 *---------------------------------------------------------------------- 
 */
static int 
GetNode(
    Tcl_Interp *interp, 
    TreeCmd *cmdPtr, 
    Tcl_Obj *objPtr, 
    Blt_TreeNode *nodePtr)
{
    Blt_Tree tree = cmdPtr->tree;
    Blt_TreeNode node;
    char *string;
    char *p;
    char save;

    node = NULL;		/* Suppress compiler warnings. */
    save = '\0';

    string = Tcl_GetString(objPtr);
    /* 
     * Check if modifiers are present.
     */
    p = strstr(string, "->");
    if (p != NULL) {
	save = *p;
	*p = '\0';
    }
    if (IsNodeId(string)) {
	long inode;

	if (p != NULL) {
	    if (TclGetLong(interp, string, &inode) != TCL_OK) {
		goto error;
	    }
	} else {
	    if (Tcl_GetLongFromObj(interp, objPtr, &inode) != TCL_OK) {
		goto error;
	    }
	}
	node = Blt_TreeGetNode(tree, inode);
    }  else if (cmdPtr != NULL) {
	if (strcmp(string, "all") == 0) {
	    if (Blt_TreeSize(Blt_TreeRootNode(tree)) > 1) {
		if (interp != NULL) {
		    Tcl_AppendResult(interp, "more than one node tagged as \"", 
				     string, "\"", (char *)NULL);
		}
		goto error;
	    }
	    node = Blt_TreeRootNode(tree);
	} else if (strcmp(string, "root") == 0) {
	    node = Blt_TreeRootNode(tree);
	} else {
	    Blt_HashTable *tablePtr;
	    Blt_HashSearch cursor;
	    Blt_HashEntry *hPtr;

	    node = NULL;
	    tablePtr = Blt_TreeTagHashTable(tree, string);
	    if (tablePtr == NULL) {
		if (interp != NULL) {
		    Tcl_AppendResult(interp, "can't find tag or id \"", string, 
			"\" in ", Blt_TreeName(tree), (char *)NULL);
		}
		goto error;
	    } else if (tablePtr->numEntries > 1) {
		if (interp != NULL) {
		    Tcl_AppendResult(interp, "more than one node tagged as \"", 
			 string, "\"", (char *)NULL);
		}
		goto error;
	    } else if (tablePtr->numEntries > 0) {
		hPtr = Blt_FirstHashEntry(tablePtr, &cursor);
		node = Blt_GetHashValue(hPtr);
		if (p != NULL) {
		    *p = save;
		}
	    }
	}
    }
    if (node != NULL) {
	if (p != NULL) {
	    node = ParseModifiers(interp, tree, node, p);
	    if (node == NULL) {
		if (interp != NULL) {
		    *p = save;	/* Need entire string. */
		    Tcl_AppendResult(interp, "can't find tag or id \"", string, 
			     "\" in ", Blt_TreeName(tree), (char *)NULL);
		}
		goto error;
	    }
	}
	if (p != NULL) {
	    *p = save;
	}
	*nodePtr = node;
	return TCL_OK;
    } 
    if (interp != NULL) {
	Tcl_AppendResult(interp, "can't find tag or id \"", string, "\" in ", 
			 Blt_TreeName(tree), (char *)NULL);
    }
 error:
    if (p != NULL) {
	*p = save;
    }
    return TCL_ERROR;
}

typedef struct {
    int tagType;
    Blt_TreeNode root;
    Blt_HashSearch cursor;
} TagSearch;

/*
 *----------------------------------------------------------------------
 *
 * FirstTaggedNode --
 *
 *	Returns the id of the first node tagged by the given tag in
 *	objPtr.  It basically hides much of the cumbersome special
 *	case details.  For example, the special tags "root" and "all"
 *	always exist, so they don't have entries in the tag hashtable.
 *	If it's a hashed tag (not "root" or "all"), we have to save
 *	the place of where we are in the table for the next call to
 *	NextTaggedNode.
 *
 *---------------------------------------------------------------------- 
 */
static int
FirstTaggedNode(
    Tcl_Interp *interp,
    TreeCmd *cmdPtr,
    Tcl_Obj *objPtr,
    TagSearch *cursorPtr,
    Blt_TreeNode *nodePtr)
{
    char *string;

    *nodePtr = NULL;
    string = Tcl_GetString(objPtr);
    cursorPtr->tagType = TAG_TYPE_NONE;
    cursorPtr->root = Blt_TreeRootNode(cmdPtr->tree);

    /* Process strings with modifiers or digits as simple ids, not
     * tags. */
    if (GetNode((Tcl_Interp *)NULL, cmdPtr, objPtr, nodePtr) == TCL_OK) {
	return TCL_OK;
    }

    if (strcmp(string, "all") == 0) {
	cursorPtr->tagType = TAG_TYPE_ALL;
	*nodePtr = cursorPtr->root;
	return TCL_OK;
    } else if (strcmp(string, "root") == 0)  {
	*nodePtr = cursorPtr->root;
	return TCL_OK;
    } else {
	Blt_HashTable *tablePtr;
	
	tablePtr = Blt_TreeTagHashTable(cmdPtr->tree, string);
	if (tablePtr != NULL) {
	    Blt_HashEntry *hPtr;
	    
	    cursorPtr->tagType = TAG_TYPE_TAG;
	    hPtr = Blt_FirstHashEntry(tablePtr, &cursorPtr->cursor); 
	    if (hPtr == NULL) {
		*nodePtr = NULL;
		return TCL_OK;
	    }
	    *nodePtr = Blt_GetHashValue(hPtr);
	    return TCL_OK;
	}
    }
    Tcl_AppendResult(interp, "can't find tag or id \"", string, "\" in ", 
	Blt_TreeName(cmdPtr->tree), (char *)NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * SkipSeparators --
 *
 *	Moves the character pointer past one of more separators.
 *
 * Results:
 *	Returns the updates character pointer.
 *
 *----------------------------------------------------------------------
 */
static char *
SkipSeparators(char *path, CONST char *separator, int length)
{
    while ((*path == *separator) && (strncmp(path, separator, length) == 0)) {
	path += length;
    }
    return path;
}

/*
 *----------------------------------------------------------------------
 *
 * SplitPath --
 *
 *	Returns the trailing component of the given path.  Trailing
 *	separators are ignored.
 *
 * Results:
 *	Returns the string of the tail component.
 *
 *----------------------------------------------------------------------
 */
static int
SplitPath(
    Tcl_Interp *interp, 
    char *path, 
    CONST char *separator, 
    int *argcPtr, 
    char ***argvPtr)
{
    int skipLen, pathLen;
    int depth;
    size_t listSize;
    char **components;
    char *p;
    char *sp;

    if ((separator == NULL) || (*separator == '\0')) {
	if (Tcl_SplitList(interp, path, argcPtr, argvPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	return TCL_OK;
    }
    pathLen = strlen(path);
    skipLen = strlen(separator);
    path = SkipSeparators(path, separator, skipLen);
    depth = pathLen / skipLen;

    listSize = (depth + 1) * sizeof(char *);
    components = Blt_Malloc(listSize + (pathLen + 1));
    assert(components);
    p = (char *)components + listSize;
    strcpy(p, path);

    depth = 0;
    for (sp = strstr(p, separator); ((*p != '\0') && (sp != NULL)); 
	 sp = strstr(p, separator)) {
	*sp = '\0';
	components[depth++] = p;
	p = SkipSeparators(sp + skipLen, separator, skipLen);
    }
    if (*p != '\0') {
	components[depth++] = p;
    }
    components[depth] = NULL;
    *argcPtr = depth;
    *argvPtr = components;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NextTaggedNode --
 *
 *---------------------------------------------------------------------- 
 */
static Blt_TreeNode
NextTaggedNode(Blt_TreeNode node, TagSearch *cursorPtr)
{
    if (cursorPtr->tagType == TAG_TYPE_ALL) {
	return Blt_TreeNextNode(NULL, node);
    }
    if (cursorPtr->tagType == TAG_TYPE_TAG) {
	Blt_HashEntry *hPtr;

	hPtr = Blt_NextHashEntry(&cursorPtr->cursor);
	if (hPtr == NULL) {
	    return NULL;
	}
	return Blt_GetHashValue(hPtr);
    }
    return NULL;
}

static int
AddTag(TreeCmd *cmdPtr, Blt_TreeNode node, CONST char *tagName)
{
    if (strcmp(tagName, "root") == 0) {
	Tcl_AppendResult(cmdPtr->interp, "can't add reserved tag \"",
			 tagName, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    Blt_TreeAddTag(cmdPtr->tree, node, tagName);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * DeleteNode --
 *
 *---------------------------------------------------------------------- 
 */
static void
DeleteNode(TreeCmd *cmdPtr, Blt_TreeNode node)
{
    Blt_TreeNode root;

    if (!Blt_TreeTagTableIsShared(cmdPtr->tree)) {
	Blt_TreeClearTags(cmdPtr->tree, node);
    }
    root = Blt_TreeRootNode(cmdPtr->tree);
    if (node == root) {
	Blt_TreeNode next;

	/* Don't delete the root node. Simply clean out the tree. */
	for (node = Blt_TreeFirstChild(node); node != NULL; node = next) {
	    next = Blt_TreeNextSibling(node);
	    Blt_TreeDeleteNode(cmdPtr->tree, node);
	}	    
    } else if (Blt_TreeIsAncestor(root, node)) {
	Blt_TreeDeleteNode(cmdPtr->tree, node);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PrintTraceFlags --
 *
 *---------------------------------------------------------------------- 
 */
static void
PrintTraceFlags(unsigned int flags, char *string)
{
    char *s;

    s = string;
    if (flags & TREE_TRACE_READ) {
	*s++ = 'r';
    } 
    if (flags & TREE_TRACE_WRITE) {
	*s++ = 'w';
    } 
    if (flags & TREE_TRACE_UNSET) {
	*s++ = 'u';
    } 
    if (flags & TREE_TRACE_CREATE) {
	*s++ = 'c';
    } 
    *s = '\0';
}

/*
 *----------------------------------------------------------------------
 *
 * GetTraceFlags --
 *
 *---------------------------------------------------------------------- 
 */
static int
GetTraceFlags(char *string)
{
    char *s;
    unsigned int flags;

    flags = 0;
    for (s = string; *s != '\0'; s++) {
	switch (toupper(*s)) {
	case 'R':
	    flags |= TREE_TRACE_READ;
	    break;
	case 'W':
	    flags |= TREE_TRACE_WRITE;
	    break;
	case 'U':
	    flags |= TREE_TRACE_UNSET;
	    break;
	case 'C':
	    flags |= TREE_TRACE_CREATE;
	    break;
	default:
	    return -1;
	}
    }
    return flags;
}

/*
 *----------------------------------------------------------------------
 *
 * SetValues --
 *
 *---------------------------------------------------------------------- 
 */
static int
SetValues(TreeCmd *cmdPtr, Blt_TreeNode node, int objc, Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 0; i < objc; i += 2) {
	char *string;

	string = Tcl_GetString(objv[i]);
	if ((i + 1) == objc) {
	    Tcl_AppendResult(cmdPtr->interp, "missing value for field \"", 
		string, "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	if (Blt_TreeSetValue(cmdPtr->interp, cmdPtr->tree, node, string, 
			     objv[i + 1]) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * UnsetValues --
 *
 *---------------------------------------------------------------------- 
 */
static int
UnsetValues(TreeCmd *cmdPtr, Blt_TreeNode node, int objc, Tcl_Obj *CONST *objv)
{
    if (objc == 0) {
	Blt_TreeKey key;
	Blt_TreeKeySearch cursor;

	for (key = Blt_TreeFirstKey(cmdPtr->tree, node, &cursor); key != NULL;
	     key = Blt_TreeNextKey(cmdPtr->tree, &cursor)) {
	    if (Blt_TreeUnsetValueByKey(cmdPtr->interp, cmdPtr->tree, node, 
			key) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
    } else {
	int i;

	for (i = 0; i < objc; i ++) {
	    if (Blt_TreeUnsetValue(cmdPtr->interp, cmdPtr->tree, node, 
		Tcl_GetString(objv[i])) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }
    return TCL_OK;
}

static int
ComparePatterns(Blt_List patternList, char *string, int nocase)
{
    Blt_ListNode node;
    int result;

    if (nocase) {
	string = Blt_Strdup(string);
	strtolower(string);
    }
    result = FALSE;
    for (node = Blt_ListFirstNode(patternList); node != NULL; 
	node = Blt_ListNextNode(node)) {
	size_t type;
	char *pattern;
		
	type = (size_t)Blt_ListGetValue(node);
	pattern = (char *)Blt_ListGetKey(node);
	switch (type) {
	case PATTERN_EXACT:
	    result = (strcmp(string, pattern) == 0);
	    break;
	    
	case PATTERN_GLOB:
	    result = Tcl_StringMatch(string, pattern);
	    break;
		    
	case PATTERN_REGEXP:
	    result = Tcl_RegExpMatch((Tcl_Interp *)NULL, string, pattern); 
	    break;
	}
    }
    if (nocase) {
	Blt_Free(string);
    }
    return result;
}


static int
CompareTags(
    Blt_Tree tree,
    Blt_TreeNode node,
    Blt_List tagList)
{
    Blt_ListNode tn;

    for (tn = Blt_ListFirstNode(tagList); tn != NULL; 
	 tn = Blt_ListNextNode(tn)) {
	char *tag;

	tag = (char *)Blt_ListGetKey(tn);
	if (Blt_TreeHasTag(tree, node, tag)) {
	    return TRUE;
	}
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * MatchNodeProc --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
MatchNodeProc(Blt_TreeNode node, ClientData clientData, int order)
{
    FindSwitches *findPtr = clientData;
    Tcl_DString dString;
    TreeCmd *cmdPtr = findPtr->cmdPtr;
    Tcl_Interp *interp = findPtr->cmdPtr->interp;
    int result, invert;

    if ((findPtr->flags & MATCH_LEAFONLY) && (!Blt_TreeIsLeaf(node))) {
	return TCL_OK;
    }
    if ((findPtr->maxDepth >= 0) &&
	(findPtr->maxDepth < Blt_TreeNodeDepth(node))) {
	return TCL_OK;
    }
    result = TRUE;
    Tcl_DStringInit(&dString);
    if (findPtr->keyList != NULL) {
	Blt_TreeKey key;
	Blt_TreeKeySearch cursor;

	result = FALSE;		/* It's false if no keys match. */
	for (key = Blt_TreeFirstKey(cmdPtr->tree, node, &cursor); key != NULL;
	     key = Blt_TreeNextKey(cmdPtr->tree, &cursor)) {
	    
	    result = ComparePatterns(findPtr->keyList, key, 0);
	    if (!result) {
		continue;
	    }
	    if (findPtr->patternList != NULL) {
		char *string;
		Tcl_Obj *objPtr;

		Blt_TreeGetValue(interp, cmdPtr->tree, node, key, &objPtr);
		string = (objPtr == NULL) ? "" : Tcl_GetString(objPtr);
		result = ComparePatterns(findPtr->patternList, string, 
			 findPtr->flags & MATCH_NOCASE);
		if (!result) {
		    continue;
		}
	    }
	    break;
	}
    } else if (findPtr->patternList != NULL) {	    
	char *string;

	if (findPtr->flags & MATCH_PATHNAME) {
	    string = Blt_TreeNodePath(node, &dString);
	} else {
	    string = Blt_TreeNodeLabel(node);
	}
	result = ComparePatterns(findPtr->patternList, string, 
		findPtr->flags & MATCH_NOCASE);		     
    }
    if (findPtr->tagList != NULL) {
	result = CompareTags(cmdPtr->tree, node, findPtr->tagList);
    }
    Tcl_DStringFree(&dString);
    invert = (findPtr->flags & MATCH_INVERT) ? TRUE : FALSE;
    if ((result != invert) &&
	/* Check if the matching node is on the exclude list. */	
	((findPtr->excludeTable.numEntries == 0) || 
	(Blt_FindHashEntry(&findPtr->excludeTable, (char *)node) == NULL))) {
	Tcl_Obj *objPtr;

	if (findPtr->addTag != NULL) {
	    if (AddTag(cmdPtr, node, findPtr->addTag) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
	/* Save the node id in our list. */
	objPtr = Tcl_NewLongObj(Blt_TreeNodeId(node));
	Tcl_ListObjAppendElement(interp, findPtr->listObjPtr, objPtr);
	
	/* Execute a procedure for the matching node. */
	if (findPtr->objv != NULL) {
	    findPtr->objv[findPtr->objc - 1] = objPtr;
	    Tcl_IncrRefCount(objPtr);
	    result = Tcl_EvalObjv(interp, findPtr->objc, findPtr->objv, 0);
	    Tcl_DecrRefCount(objPtr);
	    findPtr->objv[findPtr->objc - 1] = NULL;
	    if (result != TCL_OK) {
		return result;
	    }
	}
	findPtr->nMatches++;
	if ((findPtr->maxMatches > 0) && 
	    (findPtr->nMatches >= findPtr->maxMatches)) {
	    return TCL_BREAK;
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ApplyNodeProc --
 *
 *---------------------------------------------------------------------- 
 */
static int
ApplyNodeProc(Blt_TreeNode node, ClientData clientData, int order)
{
    ApplySwitches *applyPtr = clientData;
    TreeCmd *cmdPtr = applyPtr->cmdPtr;
    Tcl_Interp *interp = cmdPtr->interp;
    int invert, result;
    Tcl_DString dString;

    if ((applyPtr->flags & MATCH_LEAFONLY) && (!Blt_TreeIsLeaf(node))) {
	return TCL_OK;
    }
    if ((applyPtr->maxDepth >= 0) &&
	(applyPtr->maxDepth < Blt_TreeNodeDepth(node))) {
	return TCL_OK;
    }
    Tcl_DStringInit(&dString);
    result = TRUE;
    if (applyPtr->keyList != NULL) {
	Blt_TreeKey key;
	Blt_TreeKeySearch cursor;

	result = FALSE;		/* It's false if no keys match. */
	for (key = Blt_TreeFirstKey(cmdPtr->tree, node, &cursor);
	     key != NULL; key = Blt_TreeNextKey(cmdPtr->tree, &cursor)) {
	    
	    result = ComparePatterns(applyPtr->keyList, key, 0);
	    if (!result) {
		continue;
	    }
	    if (applyPtr->patternList != NULL) {
		char *string;
		Tcl_Obj *objPtr;

		Blt_TreeGetValue(interp, cmdPtr->tree, node, key, &objPtr);
		string = (objPtr == NULL) ? "" : Tcl_GetString(objPtr);
		result = ComparePatterns(applyPtr->patternList, string, 
			 applyPtr->flags & MATCH_NOCASE);
		if (!result) {
		    continue;
		}
	    }
	    break;
	}
    } else if (applyPtr->patternList != NULL) {	    
	char *string;

	if (applyPtr->flags & MATCH_PATHNAME) {
	    string = Blt_TreeNodePath(node, &dString);
	} else {
	    string = Blt_TreeNodeLabel(node);
	}
	result = ComparePatterns(applyPtr->patternList, string, 
		applyPtr->flags & MATCH_NOCASE);		     
    }
    Tcl_DStringFree(&dString);
    if (applyPtr->tagList != NULL) {
	result = CompareTags(cmdPtr->tree, node, applyPtr->tagList);
    }
    invert = (applyPtr->flags & MATCH_INVERT) ? 1 : 0;
    if (result != invert) {
	Tcl_Obj *objPtr;

	objPtr = Tcl_NewLongObj(Blt_TreeNodeId(node));
	if (order == TREE_PREORDER) {
	    applyPtr->preObjv[applyPtr->preObjc - 1] = objPtr;
	    return 
		Tcl_EvalObjv(interp, applyPtr->preObjc, applyPtr->preObjv, 0);
	} else if (order == TREE_POSTORDER) {
	    applyPtr->postObjv[applyPtr->postObjc - 1] = objPtr;
	    return 
		Tcl_EvalObjv(interp, applyPtr->postObjc, applyPtr->postObjv, 0);
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ReleaseTreeObject --
 *
 *---------------------------------------------------------------------- 
 */
static void
ReleaseTreeObject(TreeCmd *cmdPtr)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    Blt_TreeReleaseToken(cmdPtr->tree);
    /* 
     * When the tree token is released, all the traces and
     * notification events are automatically removed.  But we still
     * need to clean up the bookkeeping kept for traces. Clear all
     * the tags and trace information.  
     */
    for (hPtr = Blt_FirstHashEntry(&cmdPtr->traceTable, &cursor); hPtr != NULL;
	hPtr = Blt_NextHashEntry(&cursor)) {
	TraceInfo *tracePtr;

	tracePtr = Blt_GetHashValue(hPtr);
	Blt_Free(tracePtr);
    }
    for (hPtr = Blt_FirstHashEntry(&cmdPtr->notifyTable, &cursor); hPtr != NULL;
	hPtr = Blt_NextHashEntry(&cursor)) {
	NotifyInfo *notifyPtr;
	int i;

	notifyPtr = Blt_GetHashValue(hPtr);
	for (i = 0; i < notifyPtr->objc - 2; i++) {
	    Tcl_DecrRefCount(notifyPtr->objv[i]);
	}
	Blt_Free(notifyPtr->objv);
	Blt_Free(notifyPtr);
    }
    cmdPtr->tree = NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeTraceProc --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TreeTraceProc(
    ClientData clientData,
    Tcl_Interp *interp,
    Blt_TreeNode node,		/* Node that has just been updated. */
    Blt_TreeKey key,		/* Field that's updated. */
    unsigned int flags)
{
    TraceInfo *tracePtr = clientData; 
    Tcl_DString dsCmd, dsName;
    char string[5];
    char *qualName;
    int result;
    Blt_ObjectName objName;

    Tcl_DStringInit(&dsCmd);
    Tcl_DStringAppend(&dsCmd, tracePtr->command, -1);
    Tcl_DStringInit(&dsName);
    objName.name = Tcl_GetCommandName(interp, tracePtr->cmdPtr->cmdToken);
    objName.nsPtr = Blt_GetCommandNamespace(tracePtr->cmdPtr->cmdToken);
    qualName = Blt_MakeQualifiedName(&objName, &dsName);
    Tcl_DStringAppendElement(&dsCmd, qualName);
    Tcl_DStringFree(&dsName);
    if (node != NULL) {
	Tcl_DStringAppendElement(&dsCmd, Blt_TreeNodeIdAscii(node));
    } else {
	Tcl_DStringAppendElement(&dsCmd, "");
    }
    Tcl_DStringAppendElement(&dsCmd, key);
    PrintTraceFlags(flags, string);
    Tcl_DStringAppendElement(&dsCmd, string);
    result = Tcl_Eval(interp, Tcl_DStringValue(&dsCmd));
    Tcl_DStringFree(&dsCmd);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeEventProc --
 *
 *---------------------------------------------------------------------- 
 */
static int
TreeEventProc(ClientData clientData, Blt_TreeNotifyEvent *eventPtr)
{
    TreeCmd *cmdPtr = clientData; 
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    Blt_TreeNode node;
    char *string;

    switch (eventPtr->type) {
    case TREE_NOTIFY_CREATE:
	string = "-create";
	break;

    case TREE_NOTIFY_DELETE:
	node = Blt_TreeGetNode(cmdPtr->tree, eventPtr->inode);
	if (node != NULL) {
	    Blt_TreeClearTags(cmdPtr->tree, node);
	}
	string = "-delete";
	break;

    case TREE_NOTIFY_MOVE:
	string = "-move";
	break;

    case TREE_NOTIFY_SORT:
	string = "-sort";
	break;

    case TREE_NOTIFY_RELABEL:
	string = "-relabel";
	break;

    default:
	/* empty */
	string = "???";
	break;
    }	

    for (hPtr = Blt_FirstHashEntry(&(cmdPtr->notifyTable), &cursor);
	 hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	NotifyInfo *notifyPtr;

	notifyPtr = Blt_GetHashValue(hPtr);
	if (notifyPtr->mask & eventPtr->type) {
	    int result;
	    Tcl_Obj *flagObjPtr, *nodeObjPtr;

	    flagObjPtr = Tcl_NewStringObj(string, -1);
	    nodeObjPtr = Tcl_NewLongObj(eventPtr->inode);
	    Tcl_IncrRefCount(flagObjPtr);
	    Tcl_IncrRefCount(nodeObjPtr);
	    notifyPtr->objv[notifyPtr->objc - 2] = flagObjPtr;
	    notifyPtr->objv[notifyPtr->objc - 1] = nodeObjPtr;
	    result = Tcl_EvalObjv(cmdPtr->interp, notifyPtr->objc, 
		notifyPtr->objv, 0);
	    Tcl_DecrRefCount(nodeObjPtr);
	    Tcl_DecrRefCount(flagObjPtr);
	    if (result != TCL_OK) {
		Tcl_BackgroundError(cmdPtr->interp);
		return TCL_ERROR;
	    }
	    Tcl_ResetResult(cmdPtr->interp);
	}
    }
    return TCL_OK;
}


/* Tree command operations. */

/*
 *----------------------------------------------------------------------
 *
 * ApplyOp --
 *
 * t0 apply root -precommand {command} -postcommand {command}
 *
 *---------------------------------------------------------------------- 
 */
static int
ApplyOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int result;
    Blt_TreeNode node;
    int i;
    Tcl_Obj **objArr;
    int count;
    ApplySwitches switches;
    int order;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    memset(&switches, 0, sizeof(switches));
    switches.maxDepth = -1;
    switches.cmdPtr = cmdPtr;
    
    /* Process switches  */
    if (Blt_ParseSwitches(interp, applySwitches, objc - 3, objv + 3, &switches,
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    order = 0;
    if (switches.flags & MATCH_NOCASE) {
	Blt_ListNode ln;

	for (ln = Blt_ListFirstNode(switches.patternList); ln != NULL;
	     ln = Blt_ListNextNode(ln)) {
	    strtolower((char *)Blt_ListGetKey(ln));
	}
    }
    if (switches.preCmd != NULL) {
	char **p;

	count = 0;
	for (p = switches.preCmd; *p != NULL; p++) {
	    count++;
	}
	objArr = Blt_Malloc((count + 1) * sizeof(Tcl_Obj *));
	for (i = 0; i < count; i++) {
	    objArr[i] = Tcl_NewStringObj(switches.preCmd[i], -1);
	    Tcl_IncrRefCount(objArr[i]);
	}
	switches.preObjv = objArr;
	switches.preObjc = count + 1;
	order |= TREE_PREORDER;
    }
    if (switches.postCmd != NULL) {
	char **p;

	count = 0;
	for (p = switches.postCmd; *p != NULL; p++) {
	    count++;
	}
	objArr = Blt_Malloc((count + 1) * sizeof(Tcl_Obj *));
	for (i = 0; i < count; i++) {
	    objArr[i] = Tcl_NewStringObj(switches.postCmd[i], -1);
	    Tcl_IncrRefCount(objArr[i]);
	}
	switches.postObjv = objArr;
	switches.postObjc = count + 1;
	order |= TREE_POSTORDER;
    }
    result = Blt_TreeApplyDFS(node, ApplyNodeProc, &switches, order);
    if (switches.preObjv != NULL) {
	for (i = 0; i < (switches.preObjc - 1); i++) {
	    Tcl_DecrRefCount(switches.preObjv[i]);
	}
	Blt_Free(switches.preObjv);
    }
    if (switches.postObjv != NULL) {
	for (i = 0; i < (switches.postObjc - 1); i++) {
	    Tcl_DecrRefCount(switches.postObjv[i]);
	}
	Blt_Free(switches.postObjv);
    }
    Blt_FreeSwitches(applySwitches, (char *)&switches, 0);
    if (result == TCL_ERROR) {
	return TCL_ERROR;
    }
    return TCL_OK;
}


/*ARGSUSED*/
static int
AncestorOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    long d1, d2, minDepth;
    long i;
    Blt_TreeNode ancestor, node1, node2;

    if ((GetNode(interp, cmdPtr, objv[2], &node1) != TCL_OK) ||
	(GetNode(interp, cmdPtr, objv[3], &node2) != TCL_OK)) {
	return TCL_ERROR;
    }
    if (node1 == node2) {
	ancestor = node1;
	goto done;
    }
    d1 = Blt_TreeNodeDepth(node1);
    d2 = Blt_TreeNodeDepth(node2);
    minDepth = MIN(d1, d2);
    if (minDepth == 0) {	/* One of the nodes is root. */
	ancestor = Blt_TreeRootNode(cmdPtr->tree);
	goto done;
    }
    /* 
     * Traverse back from the deepest node, until the both nodes are
     * at the same depth.  Check if the ancestor node found is the
     * other node.  
     */
    for (i = d1; i > minDepth; i--) {
	node1 = Blt_TreeNodeParent(node1);
    }
    if (node1 == node2) {
	ancestor = node2;
	goto done;
    }
    for (i = d2; i > minDepth; i--) {
	node2 = Blt_TreeNodeParent(node2);
    }
    if (node2 == node1) {
	ancestor = node1;
	goto done;
    }

    /* 
     * First find the mutual ancestor of both nodes.  Look at each
     * preceding ancestor level-by-level for both nodes.  Eventually
     * we'll find a node that's the parent of both ancestors.  Then
     * find the first ancestor in the parent's list of subnodes.  
     */
    for (i = minDepth; i > 0; i--) {
	node1 = Blt_TreeNodeParent(node1);
	node2 = Blt_TreeNodeParent(node2);
	if (node1 == node2) {
	    ancestor = node2;
	    goto done;
	}
    }
    Tcl_AppendResult(interp, "unknown ancestor", (char *)NULL);
    return TCL_ERROR;
 done:
    Tcl_SetLongObj(Tcl_GetObjResult(interp), Blt_TreeNodeId(ancestor));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AttachOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
AttachOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    if (objc == 3) {
	Blt_ObjectName objName;
	CONST char *treeName;
	Blt_Tree token;
	Tcl_DString dString;
	int result;

	if (!Blt_ParseObjectName(interp, Tcl_GetString(objv[2]), &objName, 0)) {
	    return TCL_ERROR;
	}
	treeName = Blt_MakeQualifiedName(&objName, &dString);
	result = Blt_TreeGetToken(interp, treeName, &token);
	Tcl_DStringFree(&dString);
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
	ReleaseTreeObject(cmdPtr);
	cmdPtr->tree = token;
    }
    Tcl_SetStringObj(Tcl_GetObjResult(interp), Blt_TreeName(cmdPtr->tree), -1);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ChildrenOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
ChildrenOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    
    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (objc == 3) {
	Tcl_Obj *listObjPtr;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (node = Blt_TreeFirstChild(node); node != NULL;
	     node = Blt_TreeNextSibling(node)) {
	    Tcl_Obj *objPtr;

	    objPtr = Tcl_NewLongObj(Blt_TreeNodeId(node));
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	Tcl_SetObjResult(interp, listObjPtr);
    } else if (objc == 4) {
	long childPos;
	long inode;
	long count;
	
	/* Get the node at  */
	if (Tcl_GetLongFromObj(interp, objv[3], &childPos) != TCL_OK) {
		return TCL_ERROR;
	}
	count = 0;
	inode = -1;
	for (node = Blt_TreeFirstChild(node); node != NULL;
	     node = Blt_TreeNextSibling(node)) {
	    if (count == childPos) {
		inode = Blt_TreeNodeId(node);
		break;
	    }
	    count++;
	}
	Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
	return TCL_OK;
    } else if (objc == 5) {
	long firstPos, lastPos;
	long count;
	Tcl_Obj *listObjPtr;
	char *string;

	firstPos = lastPos = Blt_TreeNodeDegree(node) - 1;
	string = Tcl_GetString(objv[3]);
	if ((strcmp(string, "end") != 0) &&
	    (Tcl_GetLongFromObj(interp, objv[3], &firstPos) != TCL_OK)) {
	    return TCL_ERROR;
	}
	string = Tcl_GetString(objv[4]);
	if ((strcmp(string, "end") != 0) &&
	    (Tcl_GetLongFromObj(interp, objv[4], &lastPos) != TCL_OK)) {
	    return TCL_ERROR;
	}

	count = 0;
	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (node = Blt_TreeFirstChild(node); node != NULL;
	     node = Blt_TreeNextSibling(node)) {
	    if ((count >= firstPos) && (count <= lastPos)) {
		Tcl_Obj *objPtr;

		objPtr = Tcl_NewLongObj(Blt_TreeNodeId(node));
		Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	    }
	    count++;
	}
	Tcl_SetObjResult(interp, listObjPtr);
    }
    return TCL_OK;
}


static Blt_TreeNode 
CopyNodes(
    CopySwitches *copyPtr,
    Blt_TreeNode node,		/* Node to be copied. */
    Blt_TreeNode parent)	/* New parent for the copied node. */
{
    Blt_TreeNode newNode;	/* Newly created copy. */
    char *label;

    newNode = NULL;
    label = Blt_TreeNodeLabel(node);
    if (copyPtr->flags & COPY_OVERWRITE) {
	newNode = Blt_TreeFindChild(parent, label);
    }
    if (newNode == NULL) {	/* Create node in new parent. */
	newNode = Blt_TreeCreateNode(copyPtr->destTree, parent, label, -1);
    }
    /* Copy the data fields. */
    {
	Blt_TreeKey key;
	Blt_TreeKeySearch cursor;

	for (key = Blt_TreeFirstKey(copyPtr->srcTree, node, &cursor); 
	     key != NULL; key = Blt_TreeNextKey(copyPtr->srcTree, &cursor)) {
	    Tcl_Obj *objPtr;

	    if (Blt_TreeGetValueByKey((Tcl_Interp *)NULL, copyPtr->srcTree, 
			node, key, &objPtr) == TCL_OK) {
		Blt_TreeSetValueByKey((Tcl_Interp *)NULL, copyPtr->destTree, 
			newNode, key, objPtr);
	    } 
	}
    }
    /* Add tags to destination tree command. */
    if ((copyPtr->destPtr != NULL) && (copyPtr->flags & COPY_TAGS)) {
	Blt_HashSearch cursor;
	Blt_HashEntry *hPtr;

	for (hPtr = Blt_TreeFirstTag(copyPtr->srcPtr->tree, &cursor); 
	     hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	    Blt_HashEntry *h2Ptr;
	    Blt_TreeTagEntry *tPtr;

	    tPtr = Blt_GetHashValue(hPtr);
	    h2Ptr = Blt_FindHashEntry(&tPtr->nodeTable, (char *)node);
	    if (h2Ptr != NULL) {
		if (AddTag(copyPtr->destPtr, newNode, tPtr->tagName)!= TCL_OK) {
		    return NULL;
		}
	    }
	}
    }
    if (copyPtr->flags & COPY_RECURSE) {
	Blt_TreeNode child;

	for (child = Blt_TreeFirstChild(node); child != NULL;
	     child = Blt_TreeNextSibling(child)) {
	    if (CopyNodes(copyPtr, child, newNode) == NULL) {
		return NULL;
	    }
	}
    }
    return newNode;
}

/*
 *----------------------------------------------------------------------
 *
 * CopyOp --
 * 
 *	t0 copy node tree node 
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
CopyOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    TreeCmd *srcPtr, *destPtr;
    Blt_Tree srcTree, destTree;
    Blt_TreeNode copyNode;
    Blt_TreeNode parent;
    CopySwitches switches;
    int nArgs, nSwitches;
    Blt_TreeNode root;
    int i;
    
    if (GetNode(interp, cmdPtr, objv[2], &parent) != TCL_OK) {
	return TCL_ERROR;
    }
    srcTree = destTree = cmdPtr->tree;
    srcPtr = destPtr = cmdPtr;

    /* Find the first switch. */
    for(i = 3; i < objc; i++) {
	char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
    }
    nArgs = i - 2;
    nSwitches = objc - i;
    if ((nArgs < 2) || (nArgs > 3)) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", 
			Tcl_GetString(objv[0]), 
			 " copy parent ?tree? node ?switches?", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    if (nArgs == 3) {
	char *string;

	/* 
	 * The tree name is either the name of a tree command (first choice)
	 * or an internal tree object.  
	 */
	string = Tcl_GetString(objv[3]);
	srcPtr = GetTreeCmd(cmdPtr->tdPtr, interp, string);
	if (srcPtr != NULL) {
	    srcTree = srcPtr->tree;
	} else {
	    /* Try to get the tree as an internal tree data object. */
	    if (Blt_TreeGetToken(interp, string, &srcTree) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
	objv++;
    }

    root = NULL;
    if (srcPtr == NULL) {
	if (GetForeignNode(interp, srcTree, objv[3], &copyNode) != TCL_OK) {
	    goto error;
	}
    } else {
	if (GetNode(interp, srcPtr, objv[3], &copyNode) != TCL_OK) {
	    goto error;
	}
    }
    if ((switches.flags & COPY_OVERWRITE) && 
	(Blt_TreeNodeParent(copyNode) == parent)) {
	Tcl_AppendResult(interp, "source and destination nodes are the same",
		 (char *)NULL);	     
	goto error;
    }
    memset((char *)&switches, 0, sizeof(switches));
    /* Process switches  */
    if (Blt_ParseSwitches(interp, copySwitches, nSwitches, objv + 4, &switches,
	BLT_SWITCH_DEFAULTS) < 0) {
	goto error;
    }
    switches.destPtr = destPtr;
    switches.destTree = destTree;
    switches.srcPtr = srcPtr;
    switches.srcTree = srcTree;

    if ((srcTree == destTree) && (switches.flags & COPY_RECURSE) &&
	(Blt_TreeIsAncestor(copyNode, parent))) {    
	Tcl_AppendResult(interp, "can't make cyclic copy: ",
			 "source node is an ancestor of the destination",
			 (char *)NULL);	     
	goto error;
    }

    /* Copy nodes to destination. */
    root = CopyNodes(&switches, copyNode, parent);
    if (root != NULL) {
	if (switches.label != NULL) {
	    Blt_TreeRelabelNode(switches.destTree, root, switches.label);
	}
	Tcl_SetLongObj(Tcl_GetObjResult(interp), Blt_TreeNodeId(root));
    }
 error:
    if (srcPtr == NULL) {
	Blt_TreeReleaseToken(srcTree);
    }
    return (root == NULL) ? TCL_ERROR : TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * DepthOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
DegreeOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), Blt_TreeNodeDegree(node));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DeleteOp --
 *
 *	Deletes one or more nodes from the tree.  Nodes may be
 *	specified by their id (a number) or a tag.
 *	
 *	Tags have to be handled carefully here.  We can't use the
 *	normal GetTaggedNode, NextTaggedNode, etc. routines because
 *	they walk hashtables while we're deleting nodes.  Also,
 *	remember that deleting a node recursively deletes all its
 *	children. If a parent and its children have the same tag, its
 *	possible that the tag list may contain nodes than no longer
 *	exist. So save the node indices in a list and then delete 
 *	then in a second pass.
 *
 *---------------------------------------------------------------------- 
 */
static int
DeleteOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;
    char *string;

    string = NULL;
    for (i = 2; i < objc; i++) {
	string = Tcl_GetString(objv[i]);
	if (IsNodeIdOrModifier(string)) {
	    Blt_TreeNode node;

	    if (GetNode(interp, cmdPtr, objv[i], &node) != TCL_OK) {
		return TCL_ERROR;
	    }
	    DeleteNode(cmdPtr, node);
	} else {
	    Blt_Chain *chainPtr;
	    Blt_ChainLink *linkPtr, *nextPtr;
	    Blt_HashEntry *hPtr;
	    Blt_HashSearch cursor;
	    Blt_HashTable *tablePtr;

	    if ((strcmp(string, "all") == 0) || (strcmp(string, "root") == 0)) {
		Blt_TreeNode node;

		node = Blt_TreeRootNode(cmdPtr->tree);
		DeleteNode(cmdPtr, node);
		continue;
	    }
	    tablePtr = Blt_TreeTagHashTable(cmdPtr->tree, string);
	    if (tablePtr == NULL) {
		goto error;
	    }
	    /* 
	     * Generate a list of tagged nodes. Save the inode instead
	     * of the node itself since a pruned branch may contain
	     * more tagged nodes.  
	     */
	    chainPtr = Blt_ChainCreate();
	    for (hPtr = Blt_FirstHashEntry(tablePtr, &cursor); 
		hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
		Blt_TreeNode node;

		node = Blt_GetHashValue(hPtr);
		Blt_ChainAppend(chainPtr, (ClientData)Blt_TreeNodeId(node));
	    }   
	    /*  
	     * Iterate through this list to delete the nodes.  By
	     * side-effect the tag table is deleted and Uids are
	     * released.  
	     */
	    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
		 linkPtr = nextPtr) {
		Blt_TreeNode node;
		long inode;

		nextPtr = Blt_ChainNextLink(linkPtr);
		inode = (long)Blt_ChainGetValue(linkPtr);
		node = Blt_TreeGetNode(cmdPtr->tree, inode);
		if (node != NULL) {
		    DeleteNode(cmdPtr, node);
		}
	    }
	    Blt_ChainDestroy(chainPtr);
	}
    }
    return TCL_OK;
 error:
    Tcl_AppendResult(interp, "can't find tag or id \"", string, "\" in ", 
		     Blt_TreeName(cmdPtr->tree), (char *)NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DepthOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
DepthOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), Blt_TreeNodeDepth(node));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DumpOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
DumpOp(cmdPtr, interp, objc, objv)
    TreeCmd *cmdPtr;
    Tcl_Interp *interp;
    int objc;			/* Not used. */
    Tcl_Obj *CONST *objv;
{
    Blt_TreeNode top;
    Tcl_DString dString;

    if (GetNode(interp, cmdPtr, objv[2], &top) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_DStringInit(&dString);
    Blt_TreeDump(cmdPtr->tree, top, &dString);
    Tcl_DStringResult(interp, &dString);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DumpfileOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
DumpfileOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode top;

    if (GetNode(interp, cmdPtr, objv[2], &top) != TCL_OK) {
	return TCL_ERROR;
    }
    return Blt_TreeDumpToFile(interp, cmdPtr->tree, top, 
	Tcl_GetString(objv[3]));
}

/*
 *----------------------------------------------------------------------
 *
 * ExistsOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
ExistsOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    int bool;
    
    bool = TRUE;
    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	bool = FALSE;
    } else if (objc == 4) { 
	Tcl_Obj *valueObjPtr;
	char *string;
	
	string = Tcl_GetString(objv[3]);
	if (Blt_TreeGetValue((Tcl_Interp *)NULL, cmdPtr->tree, node, 
			     string, &valueObjPtr) != TCL_OK) {
	    bool = FALSE;
	}
    } 
    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), bool);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FindOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
FindOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    FindSwitches switches;
    int result;
    Tcl_Obj **objArr;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    memset(&switches, 0, sizeof(switches));
    switches.maxDepth = -1;
    switches.order = TREE_POSTORDER;
    switches.cmdPtr = cmdPtr;
    Blt_InitHashTable(&switches.excludeTable, BLT_ONE_WORD_KEYS);
    objArr = NULL;

    /* Process switches  */
    if (Blt_ParseSwitches(interp, findSwitches, objc - 3, objv + 3, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (switches.maxDepth >= 0) {
	switches.maxDepth += Blt_TreeNodeDepth(node);
    }
    if (switches.flags & MATCH_NOCASE) {
	Blt_ListNode ln;

	for (ln = Blt_ListFirstNode(switches.patternList); ln != NULL;
	     ln = Blt_ListNextNode(ln)) {
	    strtolower((char *)Blt_ListGetKey(ln));
	}
    }
    if (switches.command != NULL) {
	int count;
	char **p;
	int i;

	count = 0;
	for (p = switches.command; *p != NULL; p++) {
	    count++;
	}
	/* Leave room for node Id argument to be appended */
	objArr = Blt_Calloc(count + 2, sizeof(Tcl_Obj *));
	for (i = 0; i < count; i++) {
	    objArr[i] = Tcl_NewStringObj(switches.command[i], -1);
	    Tcl_IncrRefCount(objArr[i]);
	}
	switches.objv = objArr;
	switches.objc = count + 1;
    }
    switches.listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    switches.cmdPtr = cmdPtr;
    if (switches.order == TREE_BREADTHFIRST) {
	result = Blt_TreeApplyBFS(node, MatchNodeProc, &switches);
    } else {
	result = Blt_TreeApplyDFS(node, MatchNodeProc, &switches, 
		switches.order);
    }
    if (switches.command != NULL) {
	Tcl_Obj **objPtrPtr;

	for (objPtrPtr = objArr; *objPtrPtr != NULL; objPtrPtr++) {
	    Tcl_DecrRefCount(*objPtrPtr);
	}
	Blt_Free(objArr);
    }
    Blt_FreeSwitches(findSwitches, (char *)&switches, 0);
    if (result == TCL_ERROR) {
	return TCL_ERROR;
    }
    Tcl_SetObjResult(interp, switches.listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FindChildOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
FindChildOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode parent, child;
    long inode;

    if (GetNode(interp, cmdPtr, objv[2], &parent) != TCL_OK) {
	return TCL_ERROR;
    }
    inode = -1;
    child = Blt_TreeFindChild(parent, Tcl_GetString(objv[3]));
    if (child != NULL) {
	inode = Blt_TreeNodeId(child);
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FirstChildOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
FirstChildOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    long inode;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    inode = -1;
    node = Blt_TreeFirstChild(node);
    if (node != NULL) {
	inode = Blt_TreeNodeId(node);
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * GetOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
GetOp(
    TreeCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (objc == 3) {
	Blt_TreeKey key;
	Tcl_Obj *valueObjPtr, *listObjPtr;
	Blt_TreeKeySearch cursor;

	/* Add the key-value pairs to a new Tcl_Obj */
	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (key = Blt_TreeFirstKey(cmdPtr->tree, node, &cursor); key != NULL; 
	     key = Blt_TreeNextKey(cmdPtr->tree, &cursor)) {
	    if (Blt_TreeGetValue((Tcl_Interp *)NULL, cmdPtr->tree, node, key,
				 &valueObjPtr) == TCL_OK) {
		Tcl_Obj *objPtr;

		objPtr = Tcl_NewStringObj(key, -1);
		Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
		Tcl_ListObjAppendElement(interp, listObjPtr, valueObjPtr);
	    }
	}	    
	Tcl_SetObjResult(interp, listObjPtr);
	return TCL_OK;
    } else {
	Tcl_Obj *valueObjPtr;
	char *string;

	string = Tcl_GetString(objv[3]); 
	if (Blt_TreeGetValue((Tcl_Interp *)NULL, cmdPtr->tree, node, string,
		     &valueObjPtr) != TCL_OK) {
	    if (objc == 4) {
		Tcl_DString dString;
		char *path;

		Tcl_DStringInit(&dString);
		path = Blt_TreeNodePath(node, &dString);		
		Tcl_AppendResult(interp, "can't find field \"", string, 
			"\" in \"", path, "\"", (char *)NULL);
		Tcl_DStringFree(&dString);
		return TCL_ERROR;
	    } 
	    /* Default to given value */
	    valueObjPtr = objv[4];
	} 
	Tcl_SetObjResult(interp, valueObjPtr);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * ImportOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
ImportOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode root;		/* Root node of restored subtree. */
    ImportSwitches switches;

    if (GetNode(interp, cmdPtr, objv[2], &root) != TCL_OK) {
	return TCL_ERROR;
    }
    memset((char *)&switches, 0, sizeof(switches));
    if (Blt_ParseSwitches(interp, importSwitches, objc - 4, objv + 4, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
#ifdef HAVE_LIBEXPAT
    if (strcmp(Tcl_GetString(objv[1]), "import") == 0) {
	return Blt_TreeImportData(interp, cmdPtr->tree, root, objv[3], 
		switches.flags);
    } else {
	return Blt_TreeImportFile(interp, cmdPtr->tree, root, objv[3], 
		switches.flags);
    }
#else
    Tcl_AppendResult(interp, "expat library not compiled in", (char *)NULL);
    return TCL_ERROR;
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * IndexOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
IndexOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    long inode;
    char *string;

    inode = -1;
    string = Tcl_GetString(objv[2]);
    if (GetNode(interp, cmdPtr, objv[2], &node) == TCL_OK) {
	if (node != NULL) {
	    inode = Blt_TreeNodeId(node);
	}
    } else {
	int i;
	Blt_TreeNode parent;
	Tcl_Obj **pathv;
	int pathc;

	if (Tcl_ListObjGetElements(interp, objv[2], &pathc, &pathv) != TCL_OK) {
	    goto done;		/* Can't split object. */
	}
	/* Start from the root and verify each path component. */
	parent = Blt_TreeRootNode(cmdPtr->tree);
	for (i = 0; i < pathc; i++) {
	    string = Tcl_GetString(pathv[i]);
	    if (string[0] == '\0') {
		continue;	/* Skip null separators. */
	    }
	    node = Blt_TreeFindChild(parent, string);
	    if (node == NULL) {
		goto done;	/* Can't find component */
	    }
	    parent = node;
	}
	inode = Blt_TreeNodeId(node);
    }
 done:
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * InsertOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
InsertOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode parent, child;
    InsertSwitches switches;

    child = NULL;
    if (GetNode(interp, cmdPtr, objv[2], &parent) != TCL_OK) {
	return TCL_ERROR;
    }
    /* Initialize switch flags */
    memset(&switches, 0, sizeof(switches));
    switches.position = -1;	/* Default to append node. */
    switches.parent = parent;
    switches.inode = -1;

    if (Blt_ParseSwitches(interp, insertSwitches, objc - 3, objv + 3, &switches,
	BLT_SWITCH_DEFAULTS) < 0) {
	goto error;
    }
    if (switches.inode > 0) {
	Blt_TreeNode node;

	node = Blt_TreeGetNode(cmdPtr->tree, switches.inode);
	if (node != NULL) {
	    Tcl_AppendResult(interp, "can't reissue node id \"", 
		Blt_Ltoa(switches.inode), "\": id already exists.", 
		(char *)NULL);
	    goto error;
	}
	child = Blt_TreeCreateNodeWithId(cmdPtr->tree, parent, switches.label, 
		switches.inode, switches.position);
    } else {
	child = Blt_TreeCreateNode(cmdPtr->tree, parent, switches.label, 
		switches.position);
    }
    if (child == NULL) {
	Tcl_AppendResult(interp, "can't allocate new node", (char *)NULL);
	goto error;
    }
    if (switches.label == NULL) {
	char string[200];

	sprintf(string, "node%ld", Blt_TreeNodeId(child));
	Blt_TreeRelabelNodeWithoutNotify(child, string);
    } 
    if (switches.tags != NULL) {
	char **p;

	for (p = switches.tags; *p != NULL; p++) {
	    if (AddTag(cmdPtr, child, *p) != TCL_OK) {
		goto error;
	    }
	}
    }
    if (switches.dataPairs != NULL) {
	char **p;

	for (p = switches.dataPairs; *p != NULL; p++) {
	    Tcl_Obj *objPtr;
	    char *key;

	    key = *p;
	    p++;
	    if (*p == NULL) {
		Tcl_AppendResult(interp, "missing value for \"", key, "\"",
				 (char *)NULL);
		goto error;
	    }
	    objPtr = Tcl_NewStringObj(*p, -1);
	    if (Blt_TreeSetValue(interp, cmdPtr->tree, child, key, objPtr) 
		!= TCL_OK) {
		Tcl_DecrRefCount(objPtr);
		goto error;
	    }
	}
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), Blt_TreeNodeId(child));
    Blt_FreeSwitches(insertSwitches, (char *)&switches, 0);
    return TCL_OK;

 error:
    if (child != NULL) {
	Blt_TreeDeleteNode(cmdPtr->tree, child);
    }
    Blt_FreeSwitches(insertSwitches, (char *)&switches, 0);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * IsAncestorOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
IsAncestorOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node1, node2;
    int bool;

    if ((GetNode(interp, cmdPtr, objv[3], &node1) != TCL_OK) ||
	(GetNode(interp, cmdPtr, objv[4], &node2) != TCL_OK)) {
	return TCL_ERROR;
    }
    bool = Blt_TreeIsAncestor(node1, node2);
    Tcl_SetIntObj(Tcl_GetObjResult(interp), bool);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * IsBeforeOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
IsBeforeOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node1, node2;
    int bool;

    if ((GetNode(interp, cmdPtr, objv[3], &node1) != TCL_OK) ||
	(GetNode(interp, cmdPtr, objv[4], &node2) != TCL_OK)) {
	return TCL_ERROR;
    }
    bool = Blt_TreeIsBefore(node1, node2);
    Tcl_SetIntObj(Tcl_GetObjResult(interp), bool);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * IsLeafOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
IsLeafOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;

    if (GetNode(interp, cmdPtr, objv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetIntObj(Tcl_GetObjResult(interp), Blt_TreeIsLeaf(node));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * IsRootOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
IsRootOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    int bool;

    if (GetNode(interp, cmdPtr, objv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    bool = (node == Blt_TreeRootNode(cmdPtr->tree));
    Tcl_SetIntObj(Tcl_GetObjResult(interp), bool);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * IsOp --
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec isOps[] =
{
    {"ancestor", 1, (Blt_Op)IsAncestorOp, 5, 5, "node1 node2",},
    {"before", 1, (Blt_Op)IsBeforeOp, 5, 5, "node1 node2",},
    {"leaf", 2, (Blt_Op)IsLeafOp, 4, 4, "node",},
    {"root", 1, (Blt_Op)IsRootOp, 4, 4, "node",},
};

static int nIsOps = sizeof(isOps) / sizeof(Blt_OpSpec);

static int
IsOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nIsOps, isOps, BLT_OP_ARG2, objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}


/*
 *----------------------------------------------------------------------
 *
 * KeysOp --
 *
 *	Returns the key names of values for a node or array value.
 *
 *---------------------------------------------------------------------- 
 */
static int
KeysOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_HashTable keyTable;
    int i;

    Blt_InitHashTableWithPool(&keyTable, BLT_ONE_WORD_KEYS);
    for (i = 2; i < objc; i++) {
	Blt_TreeNode node;
	TagSearch cursor;
	int isNew;

	if (FirstTaggedNode(interp, cmdPtr, objv[i], &cursor, &node) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	for (/* empty */; node != NULL; node = NextTaggedNode(node, &cursor)) {
	    Blt_TreeKey key;
	    Blt_TreeKeySearch keyIter;

	    for (key = Blt_TreeFirstKey(cmdPtr->tree, node, &keyIter); 
		 key != NULL; key = Blt_TreeNextKey(cmdPtr->tree, &keyIter)) {
		Blt_CreateHashEntry(&keyTable, key, &isNew);
	    }
	}
    }
    {
	Blt_HashSearch tagSearch;
	Blt_HashEntry *hPtr;
	Tcl_Obj *listObjPtr;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (hPtr = Blt_FirstHashEntry(&keyTable, &tagSearch); hPtr != NULL;
	     hPtr = Blt_NextHashEntry(&tagSearch)) {
	    Tcl_Obj *objPtr;
	    
	    objPtr = Tcl_NewStringObj(Blt_GetHashKey(&keyTable, hPtr), -1);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	Tcl_SetObjResult(interp, listObjPtr);
    }
    Blt_DeleteHashTable(&keyTable);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * LabelOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
LabelOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (objc == 4) {
	Blt_TreeRelabelNode(cmdPtr->tree, node, Tcl_GetString(objv[3]));
    }
    Tcl_SetStringObj(Tcl_GetObjResult(interp), Blt_TreeNodeLabel(node), -1);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * LastChildOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
LastChildOp(cmdPtr, interp, objc, objv)
    TreeCmd *cmdPtr;
    Tcl_Interp *interp;
    int objc;			/* Not used. */
    Tcl_Obj *CONST *objv;
{
    Blt_TreeNode node;
    long inode;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    node = Blt_TreeLastChild(node);
    inode = (node != NULL) ? Blt_TreeNodeId(node) : -1 ;
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * MoveOp --
 *
 *	The big trick here is to not consider the node to be moved in
 *	determining it's new location.  Ideally, you would temporarily
 *	pull from the tree and replace it (back in its old location if
 *	something went wrong), but you could still pick the node by 
 *	its serial number.  So here we make lots of checks for the 
 *	node to be moved.
 * 
 *---------------------------------------------------------------------- 
 */
static int
MoveOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode root, parent, node;
    Blt_TreeNode before;
    MoveSwitches switches;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    if (GetNode(interp, cmdPtr, objv[3], &parent) != TCL_OK) {
	return TCL_ERROR;
    }
    root = Blt_TreeRootNode(cmdPtr->tree);
    if (node == root) {
	Tcl_AppendResult(interp, "can't move root node", (char *)NULL);
	return TCL_ERROR;
    }
    if (parent == node) {
	Tcl_AppendResult(interp, "can't move node to self", (char *)NULL);
	return TCL_ERROR;
    }
    switches.node = NULL;
    switches.cmdPtr = cmdPtr;
    switches.movePos = -1;
    /* Process switches  */
    if (Blt_ParseSwitches(interp, moveSwitches, objc - 4, objv + 4, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    /* Verify they aren't ancestors. */
    if (Blt_TreeIsAncestor(node, parent)) {
	Tcl_AppendResult(interp, "can't move node: \"", 
		 Tcl_GetString(objv[2]), (char *)NULL);
	Tcl_AppendResult(interp, "\" is an ancestor of \"", 
		 Tcl_GetString(objv[3]), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    before = NULL;		/* If before is NULL, this appends the
				 * node to the parent's child list.  */

    if (switches.node != NULL) {	/* -before or -after */
	if (Blt_TreeNodeParent(switches.node) != parent) {
	    Tcl_AppendResult(interp, Tcl_GetString(objv[2]), 
		     " isn't the parent of ", Blt_TreeNodeLabel(switches.node),
		     (char *)NULL);
	    return TCL_ERROR;
	}
	if (Blt_SwitchChanged(moveSwitches, "-before", (char *)NULL)) {
	    before = switches.node;
	    if (before == node) {
		Tcl_AppendResult(interp, "can't move node before itself", 
				 (char *)NULL);
		return TCL_ERROR;
	    }
	} else {
	    before = Blt_TreeNextSibling(switches.node);
	    if (before == node) {
		Tcl_AppendResult(interp, "can't move node after itself", 
				 (char *)NULL);
		return TCL_ERROR;
	    }
	}
    } else if (switches.movePos >= 0) { /* -at */
	int count;		/* Tracks the current list index. */
	Blt_TreeNode child;

	/* 
	 * If the node is in the list, ignore it when determining the
	 * "before" node using the -at index.  An index of -1 means to
	 * append the node to the list.
	 */
	count = 0;
	for(child = Blt_TreeFirstChild(parent); child != NULL; 
	    child = Blt_TreeNextSibling(child)) {
	    if (child == node) {
		continue;	/* Ignore the node to be moved. */
	    }
	    if (count == switches.movePos) {
		before = child;
		break;		
	    }
	    count++;	
	}
    }
    if (Blt_TreeMoveNode(cmdPtr->tree, node, parent, before) != TCL_OK) {
	Tcl_AppendResult(interp, "can't move node ", Tcl_GetString(objv[2]), 
		 " to ", Tcl_GetString(objv[3]), (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NextOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
NextOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    long inode;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    node = Blt_TreeNextNode(NULL, node);
    inode = (node != NULL) ? Blt_TreeNodeId(node) : -1;
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NextSiblingOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
NextSiblingOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    long inode;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    inode = -1;
    node = Blt_TreeNextSibling(node);
    if (node != NULL) {
	inode = Blt_TreeNodeId(node);
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyCreateOp --
 *
 *	tree0 notify create ?flags? command arg
 *---------------------------------------------------------------------- 
 */
static int
NotifyCreateOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    NotifyInfo *notifyPtr;
    NotifySwitches switches;
    char idString[200];
    int isNew, nArgs;
    Blt_HashEntry *hPtr;
    int count;
    int i;

    count = 0;
    for (i = 3; i < objc; i++) {
	char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] != '-') {
	    break;
	}
	count++;
    }
    switches.mask = 0;
    /* Process switches  */
    if (Blt_ParseSwitches(interp, notifySwitches, count, objv + 3, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    notifyPtr = Blt_Malloc(sizeof(NotifyInfo));

    nArgs = objc - i;

    /* Stash away the command in structure and pass that to the notifier. */
    notifyPtr->objv = Blt_Malloc((nArgs + 2) * sizeof(Tcl_Obj *));
    for (count = 0; i < objc; i++, count++) {
	Tcl_IncrRefCount(objv[i]);
	notifyPtr->objv[count] = objv[i];
    }
    notifyPtr->objc = nArgs + 2;
    notifyPtr->cmdPtr = cmdPtr;
    if (switches.mask == 0) {
	switches.mask = TREE_NOTIFY_ALL;
    }
    notifyPtr->mask = switches.mask;

    sprintf(idString, "notify%d", cmdPtr->notifyCounter++);
    hPtr = Blt_CreateHashEntry(&(cmdPtr->notifyTable), idString, &isNew);
    Blt_SetHashValue(hPtr, notifyPtr);

    Tcl_SetStringObj(Tcl_GetObjResult(interp), idString, -1);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyDeleteOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
NotifyDeleteOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 3; i < objc; i++) {
	Blt_HashEntry *hPtr;
	NotifyInfo *notifyPtr;
	char *string;
	int j;

	string = Tcl_GetString(objv[i]);
	hPtr = Blt_FindHashEntry(&(cmdPtr->notifyTable), string);
	if (hPtr == NULL) {
	    Tcl_AppendResult(interp, "unknown notify name \"", string, "\"", 
			     (char *)NULL);
	    return TCL_ERROR;
	}
	notifyPtr = Blt_GetHashValue(hPtr);
	Blt_DeleteHashEntry(&(cmdPtr->notifyTable), hPtr);
	for (j = 0; j < (notifyPtr->objc - 2); j++) {
	    Tcl_DecrRefCount(notifyPtr->objv[j]);
	}
	Blt_Free(notifyPtr->objv);
	Blt_Free(notifyPtr);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyInfoOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
NotifyInfoOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    NotifyInfo *notifyPtr;
    Blt_HashEntry *hPtr;
    Tcl_DString dString;
    char *string;
    int i;

    string = Tcl_GetString(objv[3]);
    hPtr = Blt_FindHashEntry(&(cmdPtr->notifyTable), string);
    if (hPtr == NULL) {
	Tcl_AppendResult(interp, "unknown notify name \"", string, "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    notifyPtr = Blt_GetHashValue(hPtr);

    Tcl_DStringInit(&dString);
    Tcl_DStringAppendElement(&dString, string);	/* Copy notify Id */
    Tcl_DStringStartSublist(&dString);
    if (notifyPtr->mask & TREE_NOTIFY_CREATE) {
	Tcl_DStringAppendElement(&dString, "-create");
    }
    if (notifyPtr->mask & TREE_NOTIFY_DELETE) {
	Tcl_DStringAppendElement(&dString, "-delete");
    }
    if (notifyPtr->mask & TREE_NOTIFY_MOVE) {
	Tcl_DStringAppendElement(&dString, "-move");
    }
    if (notifyPtr->mask & TREE_NOTIFY_SORT) {
	Tcl_DStringAppendElement(&dString, "-sort");
    }
    if (notifyPtr->mask & TREE_NOTIFY_RELABEL) {
	Tcl_DStringAppendElement(&dString, "-relabel");
    }
    if (notifyPtr->mask & TREE_NOTIFY_WHENIDLE) {
	Tcl_DStringAppendElement(&dString, "-whenidle");
    }
    Tcl_DStringEndSublist(&dString);
    Tcl_DStringStartSublist(&dString);
    for (i = 0; i < (notifyPtr->objc - 2); i++) {
	Tcl_DStringAppendElement(&dString, Tcl_GetString(notifyPtr->objv[i]));
    }
    Tcl_DStringEndSublist(&dString);
    Tcl_DStringResult(interp, &dString);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyNamesOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
NotifyNamesOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    Tcl_Obj *listObjPtr;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (hPtr = Blt_FirstHashEntry(&(cmdPtr->notifyTable), &cursor);
	 hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	Tcl_Obj *objPtr;
	char *notifyId;

	notifyId = Blt_GetHashKey(&(cmdPtr->notifyTable), hPtr);
	objPtr = Tcl_NewStringObj(notifyId, -1);
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyOp --
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec notifyOps[] =
{
    {"create", 1, (Blt_Op)NotifyCreateOp, 4, 0, "?flags? command",},
    {"delete", 1, (Blt_Op)NotifyDeleteOp, 3, 0, "notifyId...",},
    {"info", 1, (Blt_Op)NotifyInfoOp, 4, 4, "notifyId",},
    {"names", 1, (Blt_Op)NotifyNamesOp, 3, 3, "",},
};

static int nNotifyOps = sizeof(notifyOps) / sizeof(Blt_OpSpec);

static int
NotifyOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nNotifyOps, notifyOps, BLT_OP_ARG2, objc, 
	objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}


/*ARGSUSED*/
static int
ParentOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    long inode;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    inode = -1;
    node = Blt_TreeNodeParent(node);
    if (node != NULL) {
	inode = Blt_TreeNodeId(node);
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*ARGSUSED*/
static int
ParsePathOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode parent, child;
    ParsePathSwitches switches;
    char **argv;
    int argc;
    int i;
    int result;
    long inode;

    if (GetNode(interp, cmdPtr, objv[2], &parent) != TCL_OK) {
	return TCL_ERROR;
    }
    /* Process switches  */
    memset(&switches, 0, sizeof(switches));
    if (Blt_ParseSwitches(interp, parsePathSwitches, objc - 4, objv + 4, 
			  &switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    result = SplitPath(interp, Tcl_GetString(objv[3]), switches.separator, 
		       &argc, &argv);
    Blt_FreeSwitches(parsePathSwitches, (char *)&switches, 0);
    if (result != TCL_OK) {
	return TCL_ERROR;
    }
    for (i = 0; i < (argc - 1); i++) {
	Blt_TreeNode child;

	child = Blt_TreeFindChild(parent, argv[i]);
	if (child == NULL) {
	    if (switches.flags & PARSE_AUTOCREATE) {
		child = Blt_TreeCreateNode(cmdPtr->tree, parent, argv[i], -1);
	    } else {
		Tcl_DString dString;

		Tcl_DStringInit(&dString);
		Tcl_AppendResult(interp, "can't find \"", argv[i], "\" in ", 
			Blt_TreeNodePath(parent, &dString), "\"", (char *)NULL);
		Tcl_DStringFree(&dString);
		Blt_Free(argv);
		return TCL_ERROR;
	    
	    }
	}
	parent = child;
    }
    child = Blt_TreeFindChild(parent, argv[i]);
    if (child == NULL) {
	if (switches.flags & (PARSE_AUTOCREATE | PARSE_NEWLEAF)) {
	    child = Blt_TreeCreateNode(cmdPtr->tree, parent, argv[i], -1);
	} else {
	    Tcl_DString dString;
	    
	    Tcl_DStringInit(&dString);
	    Tcl_AppendResult(interp, "can't find \"", argv[i], "\" in ", 
		Blt_TreeNodePath(parent, &dString), "\"", (char *)NULL);
	    Tcl_DStringFree(&dString);
	    Blt_Free(argv);
	    return TCL_ERROR;
	}
	parent = child;
    }
    Blt_Free(argv);
    inode = -1;
    if (parent != NULL) {
	inode = Blt_TreeNodeId(parent);
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * PathOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
PathOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    Tcl_DString dString;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_DStringInit(&dString);
    Blt_TreeNodePath(node, &dString);
    Tcl_DStringResult(interp, &dString);
    return TCL_OK;
}


static int
ComparePositions(Blt_TreeNode *n1Ptr, Blt_TreeNode *n2Ptr)
{
    if (*n1Ptr == *n2Ptr) {
        return 0;
    }
    if (Blt_TreeIsBefore(*n1Ptr, *n2Ptr)) {
        return -1;
    }
    return 1;
}

/*
 *----------------------------------------------------------------------
 *
 * PositionOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
PositionOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    PositionSwitches switches;
    Blt_TreeNode *nodeArr, *nodePtr;
    Blt_TreeNode lastParent;
    Tcl_Obj *listObjPtr, *objPtr;
    int i;
    long position;
    Tcl_DString dString;
    int n;

    memset((char *)&switches, 0, sizeof(switches));
    /* Process switches  */
    n = Blt_ParseSwitches(interp, positionSwitches, objc - 2, objv + 2, 
	&switches, BLT_SWITCH_OBJV_PARTIAL);
    if (n < 0) {
	return TCL_ERROR;
    }
    objc -= n + 2, objv += n + 2;

    /* Collect the node ids into an array */
    nodeArr = Blt_Malloc((objc + 1) * sizeof(Blt_TreeNode));
    for (i = 0; i < objc; i++) {
	Blt_TreeNode node;

	if (GetNode(interp, cmdPtr, objv[i], &node) != TCL_OK) {
	    Blt_Free(nodeArr);
	    return TCL_ERROR;
	}
	nodeArr[i] = node;
    }
    nodeArr[i] = NULL;

    if (switches.sort) {		/* Sort the nodes by depth-first order 
				 * if requested. */
	qsort((char *)nodeArr, objc, sizeof(Blt_TreeNode), 
	      (QSortCompareProc *)ComparePositions);
    }

    position = 0;		/* Suppress compiler warning. */
    lastParent = NULL;
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
    Tcl_DStringInit(&dString);
    for (nodePtr = nodeArr; *nodePtr != NULL; nodePtr++) {
	Blt_TreeNode parent;

	parent = Blt_TreeNodeParent(*nodePtr);
	if ((parent != NULL) && (parent == lastParent)) {
	    Blt_TreeNode node;

	    /* 
	     * Since we've sorted the nodes already, we can safely
	     * assume that if two consecutive nodes have the same
	     * parent, the first node came before the second. If
	     * this is the case, use the last node as a starting
	     * point.  
	     */
	    
	    /*
	     * Note that we start comparing from the last node,
	     * not its successor.  Some one may give us the same
	     * node more than once.  
	     */
	    node = *(nodePtr - 1); /* Can't get here unless there's
				    * more than one node. */
	    for(/*empty*/; node != NULL; node = Blt_TreeNextSibling(node)) {
		if (node == *nodePtr) {
		    break;
		}
		position++;
	    }
	} else {
	    /* The fallback is to linearly search through the
	     * parent's list of children, counting the number of
	     * preceding siblings. Except for nodes with many
	     * siblings (100+), this should be okay. */
	    position = Blt_TreeNodePosition(*nodePtr);
	}
	if (switches.sort) {
	    lastParent = parent; /* Update the last parent. */
	}	    
	/* 
	 * Add an element in the form "parent -at position" to the
	 * list that we're generating.
	 */
	if (switches.withId) {
	    objPtr = Tcl_NewLongObj(Blt_TreeNodeId(*nodePtr));
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	if (switches.withParent) {
	    char *string;

	    Tcl_DStringSetLength(&dString, 0); /* Clear the string. */
	    string = (parent == NULL) ? "" : Blt_TreeNodeIdAscii(parent);
	    Tcl_DStringAppendElement(&dString, string);
	    Tcl_DStringAppendElement(&dString, "-at");
	    Tcl_DStringAppendElement(&dString, Blt_Ltoa(position));
	    objPtr = Tcl_NewStringObj(Tcl_DStringValue(&dString), -1);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	} else {
	    objPtr = Tcl_NewLongObj(position);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
    }
    Tcl_DStringFree(&dString);
    Blt_Free(nodeArr);
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * PreviousOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
PreviousOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    long inode;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    node = Blt_TreePrevNode(NULL, node);
    inode = (node != NULL) ? Blt_TreeNodeId(node) : -1;
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*ARGSUSED*/
static int
PrevSiblingOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    long inode;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    inode = -1;
    node = Blt_TreePrevSibling(node);
    if (node != NULL) {
	inode = Blt_TreeNodeId(node);
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), inode);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RestoreOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
RestoreOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode root;		/* Root node of restored subtree. */
    RestoreSwitches switches;
    char *string;

    if (GetNode(interp, cmdPtr, objv[2], &root) != TCL_OK) {
	return TCL_ERROR;
    }
    memset((char *)&switches, 0, sizeof(switches));
    if (Blt_ParseSwitches(interp, restoreSwitches, objc - 4, objv + 4, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    string = Tcl_GetString(objv[3]);
    if (strcmp(Tcl_GetString(objv[1]), "restore") == 0) {
	return Blt_TreeRestore(interp, cmdPtr->tree, root, string, 
		switches.flags);
    } else {
	return Blt_TreeRestoreFromFile(interp, cmdPtr->tree, root, string,
		switches.flags);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RootOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
RootOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)	/* Not used. */
{
    Blt_TreeNode root;

    root = Blt_TreeRootNode(cmdPtr->tree);
    Tcl_SetLongObj(Tcl_GetObjResult(interp), Blt_TreeNodeId(root));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SetOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
SetOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    TagSearch cursor;

    if (FirstTaggedNode(interp, cmdPtr, objv[2], &cursor, &node) != TCL_OK) {
	return TCL_ERROR;
    }
    while (node != NULL) {
	if (SetValues(cmdPtr, node, objc - 3, objv + 3) != TCL_OK) {
	    return TCL_ERROR;
	}
	node = NextTaggedNode(node, &cursor);
    } 
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * SizeOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
SizeOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetLongObj(Tcl_GetObjResult(interp), Blt_TreeSize(node));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TagAddOp --
 *
 *	.t tag add tagName node1 node2 node3
 *
 *---------------------------------------------------------------------- 
 */
static int
TagAddOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    char *string;

    string = Tcl_GetString(objv[3]);
    if (isdigit(UCHAR(string[0]))) {
	Tcl_AppendResult(interp, "bad tag \"", string, 
		 "\": can't start with a digit", (char *)NULL);
	return TCL_ERROR;
    }
    if ((strcmp(string, "all") == 0) || (strcmp(string, "root") == 0)) {
	Tcl_AppendResult(cmdPtr->interp, "can't add reserved tag \"",
			 string, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    if (objc == 4) {
	/* No nodes specified.  Just add the tag. */
	if (AddTag(cmdPtr, NULL, string) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	int i;

	for (i = 4; i < objc; i++) {
	    Blt_TreeNode node;
	    TagSearch cursor;
	    
	    if (FirstTaggedNode(interp, cmdPtr, objv[i], &cursor, &node) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	    for (/* empty */; node != NULL; 
			    node = NextTaggedNode(node, &cursor)) {
		if (AddTag(cmdPtr, node, string) != TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	}
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * TagDeleteOp --
 *
 *	.t add tag node1 node2 node3
 *
 *---------------------------------------------------------------------- 
 */
static int
TagDeleteOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    char *string;
    Blt_HashTable *tablePtr;

    string = Tcl_GetString(objv[3]);
    if (isdigit(UCHAR(string[0]))) {
	Tcl_AppendResult(interp, "bad tag \"", string, 
		 "\": can't start with a digit", (char *)NULL);
	return TCL_ERROR;
    }
    if ((strcmp(string, "all") == 0) || (strcmp(string, "root") == 0)) {
	Tcl_AppendResult(interp, "can't delete reserved tag \"", string, "\"", 
			 (char *)NULL);
        return TCL_ERROR;
    }
    tablePtr = Blt_TreeTagHashTable(cmdPtr->tree, string);
    if (tablePtr != NULL) {
        int i;
      
        for (i = 4; i < objc; i++) {
	    Blt_TreeNode node;
	    TagSearch cursor;

	    if (FirstTaggedNode(interp, cmdPtr, objv[i], &cursor, &node) 
		!= TCL_OK) {
	        return TCL_ERROR;
	    }
	    for (/* empty */; node != NULL; 
			    node = NextTaggedNode(node, &cursor)) {
		Blt_HashEntry *hPtr;

	        hPtr = Blt_FindHashEntry(tablePtr, (char *)node);
	        if (hPtr != NULL) {
		    Blt_DeleteHashEntry(tablePtr, hPtr);
	        }
	   }
       }
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TagDumpOp --
 *
 *	Like "dump", but dumps only the contents of nodes tagged by a
 *	given tag.
 *
 *---------------------------------------------------------------------- 
 */
static int
TagDumpOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode root;
    Tcl_DString dString;
    int i;

    Tcl_DStringInit(&dString);
    root = Blt_TreeRootNode(cmdPtr->tree);
    for (i = 3; i < objc; i++) {
	Blt_TreeNode node;
	TagSearch cursor;

	if (FirstTaggedNode(interp, cmdPtr, objv[i], &cursor, &node) 
	    != TCL_OK) {
	    Tcl_DStringFree(&dString);
	    return TCL_ERROR;
	}
	for (/* empty */; node != NULL; node = NextTaggedNode(node, &cursor)) {
	    Blt_TreeDumpNode(cmdPtr->tree, root, node, &dString);
	}
    }
    Tcl_DStringResult(interp, &dString);
    Tcl_DStringFree(&dString);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TagExistsOp --
 *
 *	Returns the existence of the one or more tags in the given
 *	node.  If the node has any the tags, true is return in the
 *	interpreter.
 *
 *	.t tag exists node tag1 tag2 tag3...
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TagExistsOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    int i;

    if (GetNode(interp, cmdPtr, objv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    for (i = 4; i < objc; i++) {
	char *string;

	string = Tcl_GetString(objv[i]);
	if (Blt_TreeHasTag(cmdPtr->tree, node, string)) {
	    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), TRUE);
	    return TCL_OK;
	}
    }
    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), FALSE);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TagForgetOp --
 *
 *	Removes the given tags from all nodes.
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TagForgetOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 3; i < objc; i++) {
	char *string;

	string = Tcl_GetString(objv[i]);
	if (isdigit(UCHAR(string[0]))) {
	    Tcl_AppendResult(interp, "bad tag \"", string, 
			     "\": can't start with a digit", (char *)NULL);
	    return TCL_ERROR;
	}
	Blt_TreeForgetTag(cmdPtr->tree, string);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TagGetOp --
 *
 *	Returns tag names for a given node.  If one of more pattern
 *	arguments are provided, then only those matching tags are
 *	returned.
 *
 *	.t tag get node pat1 pat2...
 *
 *---------------------------------------------------------------------- 
 */
static int
TagGetOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node; 
    Tcl_Obj *listObjPtr;
   
    if (GetNode(interp, cmdPtr, objv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    if (objc == 4) {
	Blt_HashEntry *hPtr;
	Blt_HashSearch cursor;
	Tcl_Obj *objPtr;

	/* Dump all tags for this node. */
	if (node == Blt_TreeRootNode(cmdPtr->tree)) {
	    objPtr = Tcl_NewStringObj("root", 4);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	for (hPtr = Blt_TreeFirstTag(cmdPtr->tree, &cursor); hPtr != NULL;
	     hPtr = Blt_NextHashEntry(&cursor)) {
	    Blt_TreeTagEntry *tPtr;
	    Blt_HashEntry *h2Ptr;

	    tPtr = Blt_GetHashValue(hPtr);
	    h2Ptr = Blt_FindHashEntry(&tPtr->nodeTable, (char *)node);
	    if (h2Ptr != NULL) {
		objPtr = Tcl_NewStringObj(tPtr->tagName, -1);
		Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	    }
	}
	objPtr = Tcl_NewStringObj("all", 3);
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    } else {
	int i;
	Tcl_Obj *objPtr;

	/* Check if we need to add the special tags "all" and "root" */
	for (i = 4; i < objc; i++) {
	    char *pattern;

	    pattern = Tcl_GetString(objv[i]);
	    if (Tcl_StringMatch("all", pattern)) {
		objPtr = Tcl_NewStringObj("all", 3);
		Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
		break;
	    }
	}
	if (node == Blt_TreeRootNode(cmdPtr->tree)) {
	    for (i = 4; i < objc; i++) {
		char *pattern;

		pattern = Tcl_GetString(objv[i]);
		if (Tcl_StringMatch("root", pattern)) {
		    objPtr = Tcl_NewStringObj("root", 4);
		    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
		    break;
		}
	    }
	}
	/* Now process any standard tags. */
	for (i = 4; i < objc; i++) {
	    Blt_HashEntry *hPtr;
	    Blt_HashSearch cursor;
	    char *pattern;

	    pattern = Tcl_GetString(objv[i]);
	    for (hPtr = Blt_TreeFirstTag(cmdPtr->tree, &cursor); hPtr != NULL;
		 hPtr = Blt_NextHashEntry(&cursor)) {
		Blt_TreeTagEntry *tPtr;

		tPtr = Blt_GetHashValue(hPtr);
		if (Tcl_StringMatch(tPtr->tagName, pattern)) {
		    Blt_HashEntry *h2Ptr;

		    h2Ptr = Blt_FindHashEntry(&tPtr->nodeTable, (char *)node);
		    if (h2Ptr != NULL) {
			objPtr = Tcl_NewStringObj(tPtr->tagName, -1);
			Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
		    }
		}
	    }
	}
    }    
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TagNamesOp --
 *
 *	Returns the names of all the tags in the tree.  If one of more 
 *	node arguments are provided, then only the tags found in those
 *	nodes are returned.
 *
 *	.t tag names node node node...
 *
 *---------------------------------------------------------------------- 
 */
static int
TagNamesOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Tcl_Obj *listObjPtr, *objPtr;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    objPtr = Tcl_NewStringObj("all", -1);
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    if (objc == 3) {
	Blt_HashEntry *hPtr;
	Blt_HashSearch cursor;

	objPtr = Tcl_NewStringObj("root", -1);
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	for (hPtr = Blt_TreeFirstTag(cmdPtr->tree, &cursor); hPtr != NULL; 
	     hPtr = Blt_NextHashEntry(&cursor)) {
	    Blt_TreeTagEntry *tPtr;

	    tPtr = Blt_GetHashValue(hPtr);
	    objPtr = Tcl_NewStringObj(tPtr->tagName, -1);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
    } else {
	Blt_HashTable uniqTable;
	int i;

	Blt_InitHashTable(&uniqTable, BLT_STRING_KEYS);
	for (i = 3; i < objc; i++) {
	    Blt_HashEntry *hPtr;
	    Blt_HashSearch cursor;
	    Blt_TreeNode node;
	    int isNew;

	    if (GetNode(interp, cmdPtr, objv[i], &node) != TCL_OK) {
		goto error;
	    }
	    if (node == Blt_TreeRootNode(cmdPtr->tree)) {
		Blt_CreateHashEntry(&uniqTable, "root", &isNew);
	    }
	    for (hPtr = Blt_TreeFirstTag(cmdPtr->tree, &cursor); hPtr != NULL;
		 hPtr = Blt_NextHashEntry(&cursor)) {
		Blt_TreeTagEntry *tPtr;
		Blt_HashEntry *h2Ptr;

		tPtr = Blt_GetHashValue(hPtr);
		h2Ptr = Blt_FindHashEntry(&tPtr->nodeTable, (char *)node);
		if (h2Ptr != NULL) {
		    Blt_CreateHashEntry(&uniqTable, tPtr->tagName, &isNew);
		}
	    }
	}
	{
	    Blt_HashEntry *hPtr;
	    Blt_HashSearch cursor;

	    for (hPtr = Blt_FirstHashEntry(&uniqTable, &cursor); hPtr != NULL;
		 hPtr = Blt_NextHashEntry(&cursor)) {
		objPtr = Tcl_NewStringObj(Blt_GetHashKey(&uniqTable, hPtr), -1);
		Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	    }
	}
	Blt_DeleteHashTable(&uniqTable);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
 error:
    Tcl_DecrRefCount(listObjPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * TagNodesOp --
 *
 *	Returns the node ids for the given tags.  The ids returned
 *	will represent the union of nodes for all the given tags.
 *
 *	.t nodes tag1 tag2 tag3...
 *
 *---------------------------------------------------------------------- 
 */
static int
TagNodesOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_HashTable nodeTable;
    int i;
	
    Blt_InitHashTable(&nodeTable, BLT_ONE_WORD_KEYS);
    for (i = 3; i < objc; i++) {
	char *string;
	int isNew;

	string = Tcl_GetString(objv[i]);
	if (isdigit(UCHAR(string[0]))) {
	    Tcl_AppendResult(interp, "bad tag \"", string, 
			     "\": can't start with a digit", (char *)NULL);
	    goto error;
	}
	if (strcmp(string, "all") == 0) {
	    break;
	} else if (strcmp(string, "root") == 0) {
	    Blt_CreateHashEntry(&nodeTable, 
		(char *)Blt_TreeRootNode(cmdPtr->tree), &isNew);
	    continue;
	} else {
	    Blt_HashTable *tablePtr;
	    
	    tablePtr = Blt_TreeTagHashTable(cmdPtr->tree, string);
	    if (tablePtr != NULL) {
		Blt_HashEntry *hPtr;
		Blt_HashSearch cursor;

		for (hPtr = Blt_FirstHashEntry(tablePtr, &cursor); 
		     hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
		    Blt_TreeNode node;

		    node = Blt_GetHashValue(hPtr);
		    Blt_CreateHashEntry(&nodeTable, (char *)node, &isNew);
		}
		continue;
	    }
	}
	Tcl_AppendResult(interp, "can't find a tag \"", string, "\"",
			 (char *)NULL);
	goto error;
    }
    {
	Blt_HashEntry *hPtr;
	Blt_HashSearch cursor;
	Tcl_Obj *listObjPtr;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (hPtr = Blt_FirstHashEntry(&nodeTable, &cursor); hPtr != NULL; 
	     hPtr = Blt_NextHashEntry(&cursor)) {
	    Blt_TreeNode node;
	    Tcl_Obj *objPtr;

	    node = (Blt_TreeNode)Blt_GetHashKey(&nodeTable, hPtr);
	    objPtr = Tcl_NewLongObj(Blt_TreeNodeId(node));
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	Tcl_SetObjResult(interp, listObjPtr);
    }
    Blt_DeleteHashTable(&nodeTable);
    return TCL_OK;

 error:
    Blt_DeleteHashTable(&nodeTable);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * TagSetOp --
 *
 *	Sets one or more tags for a given node.  Tag names can't start
 *	with a digit (to distinquish them from node ids) and can't be
 *	a reserved tag ("root" or "all").
 *
 *	.t tag set node tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
TagSetOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    int i;

    if (GetNode(interp, cmdPtr, objv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    for (i = 4; i < objc; i++) {
	char *string;

	string = Tcl_GetString(objv[i]);
	if (isdigit(UCHAR(string[0]))) {
	    Tcl_AppendResult(interp, "bad tag \"", string, 
			     "\": can't start with a digit", (char *)NULL);
	    return TCL_ERROR;
	}
	if ((strcmp(string, "all") == 0) || (strcmp(string, "root") == 0)) {
	    Tcl_AppendResult(interp, "can't add reserved tag \"", string, "\"",
		 (char *)NULL);	
	    return TCL_ERROR;
	}
	if (AddTag(cmdPtr, node, string) != TCL_OK) {
	    return TCL_ERROR;
	}
    }    
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TagUnsetOp --
 *
 *	Removes one or more tags from a given node. If a tag doesn't
 *	exist or is a reserved tag ("root" or "all"), nothing will
 *	be done and no error message will be returned.
 *
 *	.t tag unset node tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
TagUnsetOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    int i;

    if (GetNode(interp, cmdPtr, objv[3], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    for (i = 4; i < objc; i++) {
	Blt_TreeRemoveTag(cmdPtr->tree, node, Tcl_GetString(objv[i]));
    }    
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TagOp --
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec tagOps[] = {
    {"add", 1,    (Blt_Op)TagAddOp, 4, 0, "tag ?node...?",},
    {"delete", 2, (Blt_Op)TagDeleteOp, 5, 0, "tag node...",},
    {"dump", 2,   (Blt_Op)TagDumpOp, 4, 0, "tag...",},
    {"exists", 1, (Blt_Op)TagExistsOp, 4, 0, "node tag...",},
    {"forget", 1, (Blt_Op)TagForgetOp, 4, 0, "tag...",},
    {"get", 1,    (Blt_Op)TagGetOp, 4, 0, "node ?pattern...?",},
    {"names", 2,  (Blt_Op)TagNamesOp, 3, 0, "?node...?",},
    {"nodes", 2,  (Blt_Op)TagNodesOp, 4, 0, "tag ?tag...?",},
    {"set", 1,    (Blt_Op)TagSetOp, 4, 0, "node tag...",},
    {"unset", 1,  (Blt_Op)TagUnsetOp, 4, 0, "node tag...",},
};

static int nTagOps = sizeof(tagOps) / sizeof(Blt_OpSpec);

static int
TagOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nTagOps, tagOps, BLT_OP_ARG2, objc, objv, 
	0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceCreateOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TraceCreateOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_HashEntry *hPtr;
    Blt_TreeNode node;
    TraceInfo *tracePtr;
    char *string, *key, *command;
    char *tagName;
    char idString[200];
    int flags, isNew;
    int length;

    string = Tcl_GetString(objv[3]);
    if (isdigit(UCHAR(*string))) {
	if (GetNode(interp, cmdPtr, objv[3], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	tagName = NULL;
    } else {
	tagName = Blt_Strdup(string);
	node = NULL;
    }
    key = Tcl_GetString(objv[4]);
    string = Tcl_GetString(objv[5]);
    flags = GetTraceFlags(string);
    if (flags < 0) {
	Tcl_AppendResult(interp, "unknown flag in \"", string, "\"", 
		     (char *)NULL);
	return TCL_ERROR;
    }
    command = Tcl_GetStringFromObj(objv[6], &length);
    /* Stash away the command in structure and pass that to the trace. */
    tracePtr = Blt_Malloc(length + sizeof(TraceInfo));
    strcpy(tracePtr->command, command);
    tracePtr->cmdPtr = cmdPtr;
    tracePtr->withTag = tagName;
    tracePtr->node = node;
    tracePtr->traceToken = Blt_TreeCreateTrace(cmdPtr->tree, node, key, tagName,
	flags, TreeTraceProc, tracePtr);

    sprintf(idString, "trace%d", cmdPtr->traceCounter++);
    hPtr = Blt_CreateHashEntry(&(cmdPtr->traceTable), idString, &isNew);
    Blt_SetHashValue(hPtr, tracePtr);

    Tcl_SetStringObj(Tcl_GetObjResult(interp), idString, -1);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceDeleteOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
TraceDeleteOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 3; i < objc; i++) {
	Blt_HashEntry *hPtr;
	TraceInfo *tracePtr;
	char *key;

	key = Tcl_GetString(objv[i]);
	hPtr = Blt_FindHashEntry(&(cmdPtr->traceTable), key);
	if (hPtr == NULL) {
	    Tcl_AppendResult(interp, "unknown trace \"", key, "\"", 
			     (char *)NULL);
	    return TCL_ERROR;
	}
	tracePtr = Blt_GetHashValue(hPtr);
	Blt_DeleteHashEntry(&(cmdPtr->traceTable), hPtr); 
	Blt_TreeDeleteTrace(tracePtr->traceToken);
	if (tracePtr->withTag != NULL) {
	    Blt_Free(tracePtr->withTag);
	}
	Blt_Free(tracePtr);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceNamesOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TraceNamesOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    for (hPtr = Blt_FirstHashEntry(&(cmdPtr->traceTable), &cursor);
	 hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	Tcl_AppendElement(interp, Blt_GetHashKey(&(cmdPtr->traceTable), hPtr));
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceInfoOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TraceInfoOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    TraceInfo *tracePtr;
    struct Blt_TreeTraceStruct *tokenPtr;
    Blt_HashEntry *hPtr;
    Tcl_DString dString;
    char string[5];
    char *key;

    key = Tcl_GetString(objv[3]);
    hPtr = Blt_FindHashEntry(&(cmdPtr->traceTable), key);
    if (hPtr == NULL) {
	Tcl_AppendResult(interp, "unknown trace \"", key, "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    Tcl_DStringInit(&dString);
    tracePtr = Blt_GetHashValue(hPtr);
    if (tracePtr->withTag != NULL) {
	Tcl_DStringAppendElement(&dString, tracePtr->withTag);
    } else {
	Tcl_DStringAppendElement(&dString, Blt_TreeNodeIdAscii(tracePtr->node));
    }
    tokenPtr = (struct Blt_TreeTraceStruct *)tracePtr->traceToken;
    Tcl_DStringAppendElement(&dString, tokenPtr->key);
    PrintTraceFlags(tokenPtr->mask, string);
    Tcl_DStringAppendElement(&dString, string);
    Tcl_DStringAppendElement(&dString, tracePtr->command);
    Tcl_DStringResult(interp, &dString);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceOp --
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec traceOps[] =
{
    {"create", 1, (Blt_Op)TraceCreateOp, 7, 7, "node key how command",},
    {"delete", 1, (Blt_Op)TraceDeleteOp, 3, 0, "id...",},
    {"info", 1, (Blt_Op)TraceInfoOp, 4, 4, "id",},
    {"names", 1, (Blt_Op)TraceNamesOp, 3, 3, "",},
};

static int nTraceOps = sizeof(traceOps) / sizeof(Blt_OpSpec);

static int
TraceOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nTraceOps, traceOps, BLT_OP_ARG2, objc, 
	objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * GetOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TypeOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    Tcl_Obj *valueObjPtr;
    char *string;

    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }

    string = Tcl_GetString(objv[3]);
    if (Blt_TreeGetValue(interp, cmdPtr->tree, node, string, &valueObjPtr) 
	!= TCL_OK) {
	return TCL_ERROR;
    }
    if (valueObjPtr->typePtr != NULL) {
	Tcl_SetStringObj(Tcl_GetObjResult(interp), valueObjPtr->typePtr->name, 
			 -1);
    } else {
	Tcl_SetStringObj(Tcl_GetObjResult(interp), "string", 6);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * UnsetOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
UnsetOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    char *string;
	
    string = Tcl_GetString(objv[2]);
    if (isdigit(UCHAR(*string))) {
	if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (UnsetValues(cmdPtr, node, objc - 3, objv + 3) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	TagSearch cursor;

	if (FirstTaggedNode(interp, cmdPtr, objv[2], &cursor, &node) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	for (/* empty */; node != NULL; node = NextTaggedNode(node, &cursor)) {
	    if (UnsetValues(cmdPtr, node, objc - 3, objv + 3) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }
    return TCL_OK;
}


typedef struct {
    TreeCmd *cmdPtr;
    unsigned int flags;
    int type;
    int mode;
    char *key;
    char *command;
} SortSwitches;

#define SORT_RECURSE		(1<<2)
#define SORT_DECREASING		(1<<3)
#define SORT_PATHNAME		(1<<4)

enum SortTypes { SORT_DICTIONARY, SORT_REAL, SORT_INTEGER, SORT_ASCII, 
	SORT_COMMAND };

enum SortModes { SORT_FLAT, SORT_REORDER };

static Blt_SwitchSpec sortSwitches[] = 
{
    {BLT_SWITCH_VALUE, "-ascii", Blt_Offset(SortSwitches, type), 0, 0, 
	SORT_ASCII},
    {BLT_SWITCH_STRING, "-command", Blt_Offset(SortSwitches, command), 0},
    {BLT_SWITCH_FLAG, "-decreasing", Blt_Offset(SortSwitches, flags), 0, 0, 
	SORT_DECREASING},
    {BLT_SWITCH_VALUE, "-dictionary", Blt_Offset(SortSwitches, type), 0, 0, 
	SORT_DICTIONARY},
    {BLT_SWITCH_VALUE, "-integer", Blt_Offset(SortSwitches, type), 0, 0, 
	SORT_INTEGER},
    {BLT_SWITCH_STRING, "-key", Blt_Offset(SortSwitches, key), 0},
    {BLT_SWITCH_FLAG, "-path", Blt_Offset(SortSwitches, flags), 0, 0, 
	SORT_PATHNAME},
    {BLT_SWITCH_VALUE, "-real", Blt_Offset(SortSwitches, type), 0, 0, 
	SORT_REAL},
    {BLT_SWITCH_VALUE, "-recurse", Blt_Offset(SortSwitches, flags), 0, 0, 
	SORT_RECURSE},
    {BLT_SWITCH_VALUE, "-reorder", Blt_Offset(SortSwitches, mode), 0, 0, 
	SORT_REORDER},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static SortSwitches sortData;

static int
CompareNodes(Blt_TreeNode *n1Ptr, Blt_TreeNode *n2Ptr)
{
    TreeCmd *cmdPtr = sortData.cmdPtr;
    char *s1, *s2;
    int result;
    Tcl_DString dString1, dString2;

    s1 = s2 = "";
    result = 0;

    if (sortData.flags & SORT_PATHNAME) {
	Tcl_DStringInit(&dString1);
	Tcl_DStringInit(&dString2);
    }
    if (sortData.key != NULL) {
	Tcl_Obj *valueObjPtr;

	if (Blt_TreeGetValue((Tcl_Interp *)NULL, cmdPtr->tree, *n1Ptr, 
	     sortData.key, &valueObjPtr) == TCL_OK) {
	    s1 = Tcl_GetString(valueObjPtr);
	}
	if (Blt_TreeGetValue((Tcl_Interp *)NULL, cmdPtr->tree, *n2Ptr, 
	     sortData.key, &valueObjPtr) == TCL_OK) {
	    s2 = Tcl_GetString(valueObjPtr);
	}
    } else if (sortData.flags & SORT_PATHNAME)  {
	Blt_TreeNode root;
	
	root = Blt_TreeRootNode(cmdPtr->tree);
	Tcl_DStringInit(&dString1), Tcl_DStringInit(&dString2);
	s1 = Blt_TreeNodeRelativePath(root, *n1Ptr, NULL, 0, &dString1);
	s2 = Blt_TreeNodeRelativePath(root, *n2Ptr, NULL, 0, &dString2);
    } else {
	s1 = Blt_TreeNodeLabel(*n1Ptr);
	s2 = Blt_TreeNodeLabel(*n2Ptr);
    }
    switch (sortData.type) {
    case SORT_ASCII:
	result = strcmp(s1, s2);
	break;

    case SORT_COMMAND:
	if (sortData.command == NULL) {
	    result = Blt_DictionaryCompare(s1, s2);
	} else {
	    Blt_ObjectName objName;
	    Tcl_DString dsCmd, dsName;
	    char *qualName;

	    result = 0;	/* Hopefully this will be Ok even if the
			 * Tcl command fails to return the correct
			 * result. */
	    Tcl_DStringInit(&dsCmd);
	    Tcl_DStringAppend(&dsCmd, sortData.command, -1);
	    Tcl_DStringInit(&dsName);
	    objName.name = Tcl_GetCommandName(cmdPtr->interp, cmdPtr->cmdToken);
	    objName.nsPtr = Blt_GetCommandNamespace(cmdPtr->cmdToken);
	    qualName = Blt_MakeQualifiedName(&objName, &dsName);
	    Tcl_DStringAppendElement(&dsCmd, qualName);
	    Tcl_DStringFree(&dsName);
	    Tcl_DStringAppendElement(&dsCmd, Blt_TreeNodeIdAscii(*n1Ptr));
	    Tcl_DStringAppendElement(&dsCmd, Blt_TreeNodeIdAscii(*n2Ptr));
	    Tcl_DStringAppendElement(&dsCmd, s1);
	    Tcl_DStringAppendElement(&dsCmd, s2);
	    result = Tcl_GlobalEval(cmdPtr->interp, Tcl_DStringValue(&dsCmd));
	    Tcl_DStringFree(&dsCmd);
	    
	    if ((result != TCL_OK) ||
		(Tcl_GetInt(cmdPtr->interp, 
		    Tcl_GetStringResult(cmdPtr->interp), &result) != TCL_OK)) {
		Tcl_BackgroundError(cmdPtr->interp);
	    }
	    Tcl_ResetResult(cmdPtr->interp);
	}
	break;

    case SORT_DICTIONARY:
	result = Blt_DictionaryCompare(s1, s2);
	break;

    case SORT_INTEGER:
	{
	    int i1, i2;

	    if (Tcl_GetInt(NULL, s1, &i1) == TCL_OK) {
		if (Tcl_GetInt(NULL, s2, &i2) == TCL_OK) {
		    result = i1 - i2;
		} else {
		    result = -1;
		} 
	    } else if (Tcl_GetInt(NULL, s2, &i2) == TCL_OK) {
		result = 1;
	    } else {
		result = Blt_DictionaryCompare(s1, s2);
	    }
	}
	break;

    case SORT_REAL:
	{
	    double r1, r2;

	    if (Tcl_GetDouble(NULL, s1, &r1) == TCL_OK) {
		if (Tcl_GetDouble(NULL, s2, &r2) == TCL_OK) {
		    result = (r1 < r2) ? -1 : (r1 > r2) ? 1 : 0;
		} else {
		    result = -1;
		} 
	    } else if (Tcl_GetDouble(NULL, s2, &r2) == TCL_OK) {
		result = 1;
	    } else {
		result = Blt_DictionaryCompare(s1, s2);
	    }
	}
	break;
    }
    if (result == 0) {
	result = Blt_TreeNodeId(*n1Ptr) - Blt_TreeNodeId(*n2Ptr);
    }
    if (sortData.flags & SORT_DECREASING) {
	result = -result;
    } 
    if (sortData.flags & SORT_PATHNAME) {
	Tcl_DStringFree(&dString1);
	Tcl_DStringFree(&dString2);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * SortApplyProc --
 *
 *	Sorts the subnodes at a given node.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SortApplyProc(
    Blt_TreeNode node,
    ClientData clientData,
    int order)			/* Not used. */
{
    TreeCmd *cmdPtr = clientData;

    if (!Blt_TreeIsLeaf(node)) {
	Blt_TreeSortNode(cmdPtr->tree, node, CompareNodes);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * SortOp --
 *  
 *---------------------------------------------------------------------- 
 */
static int
SortOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode top;
    SortSwitches switches;
    int result;

    if (GetNode(interp, cmdPtr, objv[2], &top) != TCL_OK) {
	return TCL_ERROR;
    }
    /* Process switches  */
    memset(&switches, 0, sizeof(switches));
    switches.cmdPtr = cmdPtr;
    if (Blt_ParseSwitches(interp, sortSwitches, objc - 3, objv + 3, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (switches.command != NULL) {
	switches.type = SORT_COMMAND;
    }
    switches.cmdPtr = cmdPtr;
    sortData = switches;
    if (switches.mode == SORT_FLAT) {
	Blt_TreeNode *nodeArr;
	long nNodes;
	long i;

	if (switches.flags & SORT_RECURSE) {
	    nNodes = Blt_TreeSize(top);
	} else {
	    nNodes = Blt_TreeNodeDegree(top);
	}
	nodeArr = Blt_Malloc(nNodes * sizeof(Blt_TreeNode));
	assert(nodeArr);
	if (switches.flags & SORT_RECURSE) {
	    Blt_TreeNode node, *p;

	    p = nodeArr;
	    for(p = nodeArr, node = top; node != NULL; 
		node = Blt_TreeNextNode(top, node)) {
		*p++ = node;
	    }
	} else {
	    Blt_TreeNode node, *p;

	    for (p = nodeArr, node = Blt_TreeFirstChild(top); node != NULL;
		 node = Blt_TreeNextSibling(node)) {
		*p++ = node;
	    }
	}
	qsort((char *)nodeArr, nNodes, sizeof(Blt_TreeNode),
	      (QSortCompareProc *)CompareNodes);
	{
	    Tcl_Obj *listObjPtr;
	    Blt_TreeNode *p;

	    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	    for (p = nodeArr, i = 0; i < nNodes; i++, p++) {
		Tcl_Obj *objPtr;

		objPtr = Tcl_NewLongObj(Blt_TreeNodeId(*p));
		Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	    }
	    Tcl_SetObjResult(interp, listObjPtr);
	}
	Blt_Free(nodeArr);
	result = TCL_OK;
    } else {
	if (switches.flags & SORT_RECURSE) {
	    result = Blt_TreeApply(top, SortApplyProc, cmdPtr);
	} else {
	    result = SortApplyProc(top, cmdPtr, TREE_PREORDER);
	}
    }
    Blt_FreeSwitches(sortSwitches, (char *)&switches, 0);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ValuesOp --
 *
 *	Returns the names of values for a node or array value.
 *
 *---------------------------------------------------------------------- 
 */
static int
ValuesOp(
    TreeCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_TreeNode node;
    Tcl_Obj *listObjPtr;
    
    if (GetNode(interp, cmdPtr, objv[2], &node) != TCL_OK) {
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    if (objc == 4) { 
	char *string;

	string = Tcl_GetString(objv[3]);
	if (Blt_TreeArrayNames(interp, cmdPtr->tree, node, string, listObjPtr)
	    != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	Blt_TreeKey key;
	Blt_TreeKeySearch keyIter;

	for (key = Blt_TreeFirstKey(cmdPtr->tree, node, &keyIter); key != NULL; 
	     key = Blt_TreeNextKey(cmdPtr->tree, &keyIter)) {
	    Tcl_Obj *objPtr;

	    objPtr = Tcl_NewStringObj(key, -1);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}	    
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}


/*
 * --------------------------------------------------------------
 *
 * TreeInstObjCmd --
 *
 * 	This procedure is invoked to process commands on behalf of
 *	the tree object.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 * --------------------------------------------------------------
 */
static Blt_OpSpec treeOps[] =
{
    {"ancestor", 2, (Blt_Op)AncestorOp, 4, 4, "node1 node2",},
    {"apply", 1, (Blt_Op)ApplyOp, 3, 0, "node ?switches?",},
    {"attach", 2, (Blt_Op)AttachOp, 2, 3, "?tree?",},
    {"children", 2, (Blt_Op)ChildrenOp, 3, 5, "node ?first? ?last?",},
    {"copy", 2, (Blt_Op)CopyOp, 4, 0, "parent ?tree? node ?switches?",},
    {"degree", 2, (Blt_Op)DegreeOp, 3, 0, "node",},
    {"delete", 2, (Blt_Op)DeleteOp, 3, 0, "node ?node...?",},
    {"depth", 3, (Blt_Op)DepthOp, 3, 3, "node",},
    {"dump", 4, (Blt_Op)DumpOp, 3, 3, "node",},
    {"dumpfile", 5, (Blt_Op)DumpfileOp, 4, 4, "node fileName",},
    {"exists", 1, (Blt_Op)ExistsOp, 3, 4, "node ?key?",},
    {"find", 4, (Blt_Op)FindOp, 3, 0, "node ?switches?",},
    {"findchild", 5, (Blt_Op)FindChildOp, 4, 4, "node label",},
    {"firstchild", 3, (Blt_Op)FirstChildOp, 3, 3, "node",},
    {"get", 1, (Blt_Op)GetOp, 3, 5, "node ?key? ?defaultValue?",},
    {"import", 6, (Blt_Op)ImportOp, 4, 5, "node data ?switches?",},
    {"importfile", 7, (Blt_Op)ImportOp, 4, 5, "node fileName ?switches?",},
    {"index", 3, (Blt_Op)IndexOp, 3, 3, "label|list",},
    {"insert", 3, (Blt_Op)InsertOp, 3, 0, "parent ?switches?",},
    {"is", 2, (Blt_Op)IsOp, 2, 0, "oper args...",},
    {"keys", 1, (Blt_Op)KeysOp, 3, 0, "node ?node...?",},
    {"label", 3, (Blt_Op)LabelOp, 3, 4, "node ?newLabel?",},
    {"lastchild", 3, (Blt_Op)LastChildOp, 3, 3, "node",},
    {"move", 1, (Blt_Op)MoveOp, 4, 0, "node newParent ?switches?",},
    {"next", 4, (Blt_Op)NextOp, 3, 3, "node",},
    {"nextsibling", 5, (Blt_Op)NextSiblingOp, 3, 3, "node",},
    {"notify", 2, (Blt_Op)NotifyOp, 2, 0, "args...",},
    {"parent", 4, (Blt_Op)ParentOp, 3, 3, "node",},
    {"parsepath", 4, (Blt_Op)ParsePathOp, 4, 5, "node string ?separator?",},
    {"path", 3, (Blt_Op)PathOp, 3, 3, "node",},
    {"position", 2, (Blt_Op)PositionOp, 3, 0, "?switches? node...",},
    {"previous", 5, (Blt_Op)PreviousOp, 3, 3, "node",},
    {"prevsibling", 5, (Blt_Op)PrevSiblingOp, 3, 3, "node",},
    {"restore", 7, (Blt_Op)RestoreOp, 4, 5, "node data ?switches?",},
    {"restorefile", 8, (Blt_Op)RestoreOp, 4, 5, "node fileName ?switches?",},
    {"root", 2, (Blt_Op)RootOp, 2, 2, "",},
    {"set", 3, (Blt_Op)SetOp, 3, 0, "node ?key value...?",},
    {"size", 2, (Blt_Op)SizeOp, 3, 3, "node",},
    {"sort", 2, (Blt_Op)SortOp, 3, 0, "node ?flags...?",},
    {"tag", 2, (Blt_Op)TagOp, 3, 0, "args...",},
    {"trace", 2, (Blt_Op)TraceOp, 2, 0, "args...",},
    {"type", 2, (Blt_Op)TypeOp, 4, 4, "node key",},
    {"unset", 3, (Blt_Op)UnsetOp, 3, 0, "node ?key...?",},
    {"values", 1, (Blt_Op)ValuesOp, 3, 4, "node ?key?",},
};

static int nTreeOps = sizeof(treeOps) / sizeof(Blt_OpSpec);

static int
TreeInstObjCmd(
    ClientData clientData,	/* Information about the widget. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Vector of argument strings. */
{
    Blt_Op proc;
    TreeCmd *cmdPtr = clientData;
    int result;

    proc = Blt_GetOpFromObj(interp, nTreeOps, treeOps, BLT_OP_ARG1, objc, objv,
 	BLT_OP_LINEAR_SEARCH);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    Tcl_Preserve(cmdPtr);
    result = (*proc) (cmdPtr, interp, objc, objv);
    Tcl_Release(cmdPtr);
    return result;
}

/*
 * ----------------------------------------------------------------------
 *
 * TreeInstDeleteProc --
 *
 *	Deletes the command associated with the tree.  This is
 *	called only when the command associated with the tree is
 *	destroyed.
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
TreeInstDeleteProc(ClientData clientData)
{
    TreeCmd *cmdPtr = clientData;

    ReleaseTreeObject(cmdPtr);
    if (cmdPtr->hashPtr != NULL) {
	Blt_DeleteHashEntry(cmdPtr->tablePtr, cmdPtr->hashPtr);
    }
    Blt_DeleteHashTable(&(cmdPtr->traceTable));
    Blt_Free(cmdPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * GenerateName --
 *
 *	Generates an unique tree command name.  Tree names are in
 *	the form "treeN", where N is a non-negative integer. Check 
 *	each name generated to see if it is already a tree. We want
 *	to recycle names if possible.
 *	
 * Results:
 *	Returns the unique name.  The string itself is stored in the
 *	dynamic string passed into the routine.
 *
 * ----------------------------------------------------------------------
 */
static CONST char *
GenerateName(
    Tcl_Interp *interp,
    CONST char *prefix, 
    CONST char *suffix,
    Tcl_DString *resultPtr)
{

    int i;
    CONST char *treeName;

    /* 
     * Parse the command and put back so that it's in a consistent
     * format.  
     *
     *	t1         <current namespace>::t1
     *	n1::t1     <current namespace>::n1::t1
     *	::t1	   ::t1
     *  ::n1::t1   ::n1::t1
     */
    treeName = NULL;		/* Suppress compiler warning. */
    for (i = 0; i < INT_MAX; i++) {
	Blt_ObjectName objName;
	Tcl_CmdInfo cmdInfo;
	Tcl_DString dString;
	char string[200];

	Tcl_DStringInit(&dString);
	Tcl_DStringAppend(&dString, prefix, -1);
	sprintf(string, "tree%d", i);
	Tcl_DStringAppend(&dString, string, -1);
	Tcl_DStringAppend(&dString, suffix, -1);
	if (!Blt_ParseObjectName(interp, Tcl_DStringValue(&dString), 
				 &objName, 0)) {
	    Tcl_DStringFree(&dString);
	    return NULL;
	}
	treeName = Blt_MakeQualifiedName(&objName, resultPtr);
	Tcl_DStringFree(&dString);
	/* 
	 * Check if the command already exists. 
	 */
	if (Tcl_GetCommandInfo(interp, (char *)treeName, &cmdInfo)) {
	    continue;
	}
	if (!Blt_TreeExists(interp, treeName)) {
	    /* 
	     * We want the name of the tree command and the underlying
	     * tree object to be the same. Check that the free command
	     * name isn't an already a tree object name.  
	     */
	    break;
	}
    }
    return treeName;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeCreateOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TreeCreateOp(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    TreeCmdInterpData *tdPtr = clientData;
    CONST char *treeName;
    Tcl_DString dString;
    Blt_Tree token;

    treeName = NULL;
    if (objc == 3) {
	treeName = Tcl_GetString(objv[2]);
    }
    Tcl_DStringInit(&dString);
    if (treeName == NULL) {
	treeName = GenerateName(interp, "", "", &dString);
    } else {
	char *p;

	p = strstr(treeName, "#auto");
	if (p != NULL) {
	    *p = '\0';
	    treeName = GenerateName(interp, treeName, p + 5, &dString);
	    *p = '#';
	} else {
	    Blt_ObjectName objName;
	    Tcl_CmdInfo cmdInfo;

	    /* 
	     * Parse the command and put back so that it's in a consistent
	     * format.  
	     *
	     *	t1         <current namespace>::t1
	     *	n1::t1     <current namespace>::n1::t1
	     *	::t1	   ::t1
	     *  ::n1::t1   ::n1::t1
	     */
	    if (!Blt_ParseObjectName(interp, treeName, &objName, 0)) {
		return TCL_ERROR;
	    }
	    treeName = Blt_MakeQualifiedName(&objName, &dString);
	    /* 
	     * Check if the command already exists. 
	     */
	    if (Tcl_GetCommandInfo(interp, (char *)treeName, &cmdInfo)) {
		Tcl_AppendResult(interp, "a command \"", treeName,
				 "\" already exists", (char *)NULL);
		goto error;
	    }
	    if (Blt_TreeExists(interp, treeName)) {
		Tcl_AppendResult(interp, "a tree \"", treeName, 
			"\" already exists", (char *)NULL);
		goto error;
	    }
	} 
    } 
    if (treeName == NULL) {
	goto error;
    }
    if (Blt_TreeCreate(interp, treeName, &token) == TCL_OK) {
	int isNew;
	TreeCmd *cmdPtr;

	cmdPtr = Blt_Calloc(1, sizeof(TreeCmd));
	assert(cmdPtr);
	cmdPtr->tdPtr = tdPtr;
	cmdPtr->tree = token;
	cmdPtr->interp = interp;
	Blt_InitHashTable(&(cmdPtr->traceTable), BLT_STRING_KEYS);
	Blt_InitHashTable(&(cmdPtr->notifyTable), BLT_STRING_KEYS);
	cmdPtr->cmdToken = Tcl_CreateObjCommand(interp, (char *)treeName, 
		(Tcl_ObjCmdProc *)TreeInstObjCmd, cmdPtr, TreeInstDeleteProc);
	cmdPtr->tablePtr = &tdPtr->treeTable;
	cmdPtr->hashPtr = Blt_CreateHashEntry(cmdPtr->tablePtr, (char *)cmdPtr,
	      &isNew);
	Blt_SetHashValue(cmdPtr->hashPtr, cmdPtr);
	Tcl_SetStringObj(Tcl_GetObjResult(interp), (char *)treeName, -1);
	Tcl_DStringFree(&dString);
	Blt_TreeCreateEventHandler(cmdPtr->tree, TREE_NOTIFY_ALL, 
	     TreeEventProc, cmdPtr);
	return TCL_OK;
    }
 error:
    Tcl_DStringFree(&dString);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeDestroyOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TreeDestroyOp(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    TreeCmdInterpData *tdPtr = clientData;
    int i;

    for (i = 2; i < objc; i++) {
	TreeCmd *cmdPtr;
	char *string;

	string = Tcl_GetString(objv[i]);
	cmdPtr = GetTreeCmd(tdPtr, interp, string);
	if (cmdPtr == NULL) {
	    Tcl_AppendResult(interp, "can't find a tree named \"", string,
			     "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	Tcl_DeleteCommandFromToken(interp, cmdPtr->cmdToken);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TreeNamesOp --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TreeNamesOp(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    TreeCmdInterpData *tdPtr = clientData;
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    Tcl_Obj *listObjPtr;
    Tcl_DString dString;

    Tcl_DStringInit(&dString);
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (hPtr = Blt_FirstHashEntry(&tdPtr->treeTable, &cursor);
	 hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_ObjectName objName;
	TreeCmd *cmdPtr;
	char *qualName;
	Tcl_Obj *objPtr;

	cmdPtr = Blt_GetHashValue(hPtr);
	objName.name = Tcl_GetCommandName(interp, cmdPtr->cmdToken);
	objName.nsPtr = Blt_GetCommandNamespace(cmdPtr->cmdToken);
	qualName = Blt_MakeQualifiedName(&objName, &dString);
	if (objc == 3) {
	    if (!Tcl_StringMatch(qualName, Tcl_GetString(objv[2]))) {
		continue;
	    }
	}
	objPtr = Tcl_NewStringObj(qualName, -1);
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    Tcl_DStringFree(&dString);
    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * TreeObjCmd --
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec treeCmdOps[] =
{
    {"create", 1, (Blt_Op)TreeCreateOp, 2, 3, "?name?",},
    {"destroy", 1, (Blt_Op)TreeDestroyOp, 3, 0, "name...",},
    {"names", 1, (Blt_Op)TreeNamesOp, 2, 3, "?pattern?...",},
};

static int nCmdOps = sizeof(treeCmdOps) / sizeof(Blt_OpSpec);

/*ARGSUSED*/
static int
TreeObjCmd(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;

    proc = Blt_GetOpFromObj(interp, nCmdOps, treeCmdOps, BLT_OP_ARG1, objc, 
	objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    return (*proc) (clientData, interp, objc, objv);
}

/*
 * -----------------------------------------------------------------------
 *
 * TreeInterpDeleteProc --
 *
 *	This is called when the interpreter hosting the "tree" command
 *	is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Removes the hash table managing all tree names.
 *
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
static void
TreeInterpDeleteProc(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp)
{
    TreeCmdInterpData *tdPtr = clientData;

    /* All tree instances should already have been destroyed when
     * their respective Tcl commands were deleted. */
    Blt_DeleteHashTable(&tdPtr->treeTable);
    Tcl_DeleteAssocData(interp, TREE_THREAD_KEY);
    Blt_Free(tdPtr);
}

/*ARGSUSED*/
static int
CompareDictionaryCmd(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int result;
    char *s1, *s2;

    s1 = Tcl_GetString(objv[1]);
    s2 = Tcl_GetString(objv[2]);
    result = Blt_DictionaryCompare(s1, s2);
    Tcl_SetIntObj(Tcl_GetObjResult(interp), result);
    return TCL_OK;
}

/*ARGSUSED*/
static int
ExitCmd(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int code;

    if (Tcl_GetIntFromObj(interp, objv[1], &code) != TCL_OK) {
	return TCL_ERROR;
    }
#ifdef TCL_THREADS
    Tcl_Exit(code);
#else 
    exit(code);
#endif
    /*NOTREACHED*/
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_TreeInit --
 *
 *	This procedure is invoked to initialize the "tree" command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Creates the new command and adds a new entry into a global Tcl
 *	associative array.
 *
 * ------------------------------------------------------------------------
 */
int
Blt_TreeInit(Tcl_Interp *interp)
{
    static Blt_InitCmdSpec cmdSpec = { 
	"tree", TreeObjCmd, 
    };
    static Blt_InitCmdSpec utilSpecs[] = { 
	{ "compare", CompareDictionaryCmd, },
	{ "exit", ExitCmd, }
    };
    if (Blt_InitCmds(interp, "blt::util", utilSpecs, 2) != TCL_OK) {
	return TCL_ERROR;
    }
    cmdSpec.clientData = GetTreeCmdInterpData(interp);
    return Blt_InitCmd(interp, "blt", &cmdSpec);
}

int
Blt_TreeCmdGetToken(
    Tcl_Interp *interp,
    CONST char *string,
    Blt_Tree  *treePtr)
{
    TreeCmdInterpData *tdPtr;
    TreeCmd *cmdPtr;

    tdPtr = GetTreeCmdInterpData(interp);
    cmdPtr = GetTreeCmd(tdPtr, interp, string);
    if (cmdPtr == NULL) {
	Tcl_AppendResult(interp, "can't find a tree associated with \"",
		 string, "\"", (char *)NULL);
	return TCL_ERROR;
    }
    *treePtr = cmdPtr->tree;
    return TCL_OK;
}

/* Dump tree to dbm */
/* Convert node data to datablock */

#endif /* NO_TREE */

