
/*
 * tkScrollbar.c --
 *
 *	This module implements a scrollbar widgets for the Tk
 *	toolkit.  A scrollbar displays a slider and two arrows;
 *	mouse clicks on features within the scrollbar cause
 *	scrolling commands to be invoked.
 *
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkScrollbar.c 1.79 96/02/15 18:52:40
 */

#include "bltInt.h"
#ifndef NO_TILESCROLLBAR
#include "bltTile.h"

#define NORMAL_BG	"#d9d9d9"
#define ACTIVE_BG	"#ececec"
#define SELECT_BG	"#c3c3c3"
#define TROUGH		"#c3c3c3"
#define INDICATOR	"#b03060"
#define DISABLED	"#a3a3a3"

/*
 * Defaults for scrollbars:
 */
#define DEF_SCROLLBAR_ACTIVE_BACKGROUND	ACTIVE_BG
#define DEF_SCROLLBAR_ACTIVE_BG_MONO	RGB_BLACK
#define DEF_SCROLLBAR_ACTIVE_RELIEF	"raised"
#define DEF_SCROLLBAR_BACKGROUND		NORMAL_BG
#define DEF_SCROLLBAR_BG_MONO		RGB_WHITE
#define DEF_SCROLLBAR_BORDERWIDTH	"2"
#define DEF_SCROLLBAR_COMMAND		""
#define DEF_SCROLLBAR_CURSOR		""
#define DEF_SCROLLBAR_EL_BORDERWIDTH	"-1"
#define DEF_SCROLLBAR_HIGHLIGHT_BG	NORMAL_BG
#define DEF_SCROLLBAR_HIGHLIGHT		RGB_BLACK
#define DEF_SCROLLBAR_HIGHLIGHT_WIDTH	"2"
#define DEF_SCROLLBAR_JUMP		"0"
#define DEF_SCROLLBAR_ORIENT		"vertical"
#define DEF_SCROLLBAR_RELIEF		"sunken"
#define DEF_SCROLLBAR_REPEAT_DELAY	"300"
#define DEF_SCROLLBAR_REPEAT_INTERVAL	"100"
#define DEF_SCROLLBAR_TAKE_FOCUS	(char *)NULL
#define DEF_SCROLLBAR_TROUGH_COLOR	TROUGH
#define DEF_SCROLLBAR_TROUGH_MONO	RGB_WHITE
#define DEF_SCROLLBAR_WIDTH		"15"
#define DEF_SCROLLBAR_MIN_SLIDER_LENGTH	"12"

/*
 * A data structure of the following type is kept for each scrollbar
 * widget managed by this file:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the scrollbar.  NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
    Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
    Tcl_Interp *interp;		/* Interpreter associated with scrollbar. */
    Tcl_Command widgetCmd;	/* Token for scrollbar's widget command. */
    char *orientation;		/* Orientation for window ("vertical" or
				 * "horizontal"). */
    int vertical;		/* Non-zero means vertical orientation
				 * requested, zero means horizontal. */
    int width;			/* Desired narrow dimension of scrollbar,
				 * in pixels. */
    char *command;		/* Command prefix to use when invoking
				 * scrolling commands.  NULL means don't
				 * invoke commands.  Malloc'ed. */
    int commandSize;		/* Number of non-NULL bytes in command. */
    int repeatDelay;		/* How long to wait before auto-repeating
				 * on scrolling actions (in ms). */
    int repeatInterval;		/* Interval between autorepeats (in ms). */
    int jump;			/* Value of -jump option. */

    /*
     * Information used when displaying widget:
     */

    int borderWidth;		/* Width of 3-D borders. */
    Tk_3DBorder bgBorder;	/* Used for drawing background (all flat
				 * surfaces except for trough). */
    Tk_3DBorder activeBorder;	/* For drawing backgrounds when active (i.e.
				 * when mouse is positioned over element). */
    XColor *troughColorPtr;	/* Color for drawing trough. */
    GC troughGC;		/* For drawing trough. */
    GC copyGC;			/* Used for copying from pixmap onto screen. */
    GC arrowGC;			/* Used for drawing the arrow. */
    int relief;			/* Indicates whether window as a whole is
				 * raised, sunken, or flat. */
    int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * <= 0 means don't draw a highlight. */
    XColor *highlightBgColorPtr;
    /* Color for drawing traversal highlight
				 * area when highlight is off. */
    XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
    int inset;			/* Total width of all borders, including
				 * traversal highlight and 3-D border.
				 * Indicates how much interior stuff must
				 * be offset from outside edges to leave
				 * room for borders. */
    int minSliderLength;	/* Minimum size of thumb. */
    int elementBorderWidth;	/* Width of border to draw around elements
				 * inside scrollbar (arrows and slider).
				 * -1 means use borderWidth. */
    int arrowLength;		/* Length of arrows along long dimension of
				 * scrollbar, including space for a small gap
				 * between the arrow and the slider.
				 * Recomputed on window size changes. */
    int sliderFirst;		/* Pixel coordinate of top or left edge
				 * of slider area, including border. */
    int sliderLast;		/* Coordinate of pixel just after bottom
				 * or right edge of slider area, including
				 * border. */
    int activeField;		/* Names field to be displayed in active
				 * colors, such as TOP_ARROW, or 0 for
				 * no field. */
    int activeRelief;		/* Value of -activeRelief option: relief
				 * to use for active element. */

    /*
     * Information describing the application related to the scrollbar.
     * This information is provided by the application by invoking the
     * "set" widget command.  This information can now be provided in
     * two ways:  the "old" form (totalUnits, windowUnits, firstUnit,
     * and lastUnit), or the "new" form (firstFraction and lastFraction).
     * FirstFraction and lastFraction will always be valid, but
     * the old-style information is only valid if the NEW_STYLE_COMMANDS
     * flag is 0.
     */

    int totalUnits;		/* Total dimension of application, in
				 * units.  Valid only if the NEW_STYLE_COMMANDS
				 * flag isn't set. */
    int windowUnits;		/* Maximum number of units that can be
				 * displayed in the window at once.  Valid
				 * only if the NEW_STYLE_COMMANDS flag isn't
				 * set. */
    int firstUnit;		/* Number of last unit visible in
				 * application's window.  Valid only if the
				 * NEW_STYLE_COMMANDS flag isn't set. */
    int lastUnit;		/* Index of last unit visible in window.
				 * Valid only if the NEW_STYLE_COMMANDS
				 * flag isn't set. */
    double firstFraction;	/* Position of first visible thing in window,
				 * specified as a fraction between 0 and
				 * 1.0. */
    double lastFraction;	/* Position of last visible thing in window,
				 * specified as a fraction between 0 and
				 * 1.0. */

    /*
     * Miscellaneous information:
     */

    Tk_Cursor cursor;		/* Current cursor for window, or None. */
    char *takeFocus;		/* Value of -takefocus option;  not used in
				 * the C code, but used by keyboard traversal
				 * scripts.  Malloc'ed, but may be NULL. */
    int flags;			/* Various flags;  see below for
				 * definitions. */
    Blt_Tile tile, activeTile;
} Scrollbar;

/*
 * Legal values for "activeField" field of Scrollbar structures.  These
 * are also the return values from the ScrollbarPosition procedure.
 */

#define OUTSIDE		0
#define TOP_ARROW	1
#define TOP_GAP		2
#define SLIDER		3
#define BOTTOM_GAP	4
#define BOTTOM_ARROW	5

/*
 * Flag bits for scrollbars:
 *
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler
 *				has already been queued to redraw
 *				this window.
 * NEW_STYLE_COMMANDS:		Non-zero means the new style of commands
 *				should be used to communicate with the
 *				widget:  ".t yview scroll 2 lines", instead
 *				of ".t yview 40", for example.
 * GOT_FOCUS:			Non-zero means this window has the input
 *				focus.
 */

#define REDRAW_PENDING		1
#define NEW_STYLE_COMMANDS	2
#define GOT_FOCUS		4

/*
 * Minimum slider length, in pixels (designed to make sure that the slider
 * is always easy to grab with the mouse).
 */

#define MIN_SLIDER_LENGTH	12

/*
 * Information used for objv parsing.
 */

static Blt_ConfigSpec configSpecs[] =
{
    {BLT_CONFIG_BORDER, "-activebackground", "activeBackground", "Foreground",
	DEF_SCROLLBAR_ACTIVE_BACKGROUND, Blt_Offset(Scrollbar, activeBorder),
	BLT_CONFIG_COLOR_ONLY},
    {BLT_CONFIG_BORDER, "-activebackground", "activeBackground", "Foreground",
	DEF_SCROLLBAR_ACTIVE_BG_MONO, Blt_Offset(Scrollbar, activeBorder),
	BLT_CONFIG_MONO_ONLY},
    {BLT_CONFIG_RELIEF, "-activerelief", "activeRelief", "Relief",
	DEF_SCROLLBAR_ACTIVE_RELIEF, Blt_Offset(Scrollbar, activeRelief), 0},
    {BLT_CONFIG_TILE, "-activetile", "activeTile", "Tile",
	(char *)NULL, Blt_Offset(Scrollbar, activeTile), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_BORDER, "-background", "background", "Background",
	DEF_SCROLLBAR_BACKGROUND, Blt_Offset(Scrollbar, bgBorder),
	BLT_CONFIG_COLOR_ONLY},
    {BLT_CONFIG_BORDER, "-background", "background", "Background",
	DEF_SCROLLBAR_BG_MONO, Blt_Offset(Scrollbar, bgBorder),
	BLT_CONFIG_MONO_ONLY},
    {BLT_CONFIG_SYNONYM, "-bd", "borderWidth", (char *)NULL,
	(char *)NULL, 0, 0},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL,
	(char *)NULL, 0, 0},
    {BLT_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_SCROLLBAR_BORDERWIDTH, Blt_Offset(Scrollbar, borderWidth), 0},
    {BLT_CONFIG_STRING, "-command", "command", "Command",
	DEF_SCROLLBAR_COMMAND, Blt_Offset(Scrollbar, command),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
	DEF_SCROLLBAR_CURSOR, Blt_Offset(Scrollbar, cursor), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_PIXELS_ANY, "-elementborderwidth", "elementBorderWidth",
	"BorderWidth", DEF_SCROLLBAR_EL_BORDERWIDTH,
	Blt_Offset(Scrollbar, elementBorderWidth), 0},
    {BLT_CONFIG_COLOR, "-highlightbackground", "highlightBackground",
	"HighlightBackground", DEF_SCROLLBAR_HIGHLIGHT_BG,
	Blt_Offset(Scrollbar, highlightBgColorPtr), 0},
    {BLT_CONFIG_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
	DEF_SCROLLBAR_HIGHLIGHT,
	Blt_Offset(Scrollbar, highlightColorPtr), 0},
    {BLT_CONFIG_PIXELS, "-highlightthickness", "highlightThickness",
	"HighlightThickness",
	DEF_SCROLLBAR_HIGHLIGHT_WIDTH, Blt_Offset(Scrollbar, highlightWidth), 0},
    {BLT_CONFIG_BOOLEAN, "-jump", "jump", "Jump",
	DEF_SCROLLBAR_JUMP, Blt_Offset(Scrollbar, jump), 0},
    {BLT_CONFIG_PIXELS_POSITIVE, "-minsliderlength", "minSliderLength",
	"MinSliderLength", DEF_SCROLLBAR_MIN_SLIDER_LENGTH, 
	Blt_Offset(Scrollbar, minSliderLength), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STRING, "-orient", "orient", "Orient",
	DEF_SCROLLBAR_ORIENT, Blt_Offset(Scrollbar, orientation), 0},
    {BLT_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_SCROLLBAR_RELIEF, Blt_Offset(Scrollbar, relief), 0},
    {BLT_CONFIG_INT, "-repeatdelay", "repeatDelay", "RepeatDelay",
	DEF_SCROLLBAR_REPEAT_DELAY, Blt_Offset(Scrollbar, repeatDelay), 0},
    {BLT_CONFIG_INT, "-repeatinterval", "repeatInterval", "RepeatInterval",
	DEF_SCROLLBAR_REPEAT_INTERVAL, Blt_Offset(Scrollbar, repeatInterval), 0},
    {BLT_CONFIG_STRING, "-takefocus", "takeFocus", "TakeFocus",
	DEF_SCROLLBAR_TAKE_FOCUS, Blt_Offset(Scrollbar, takeFocus),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_TILE, "-tile", "tile", "Tile",
	(char *)NULL, Blt_Offset(Scrollbar, tile), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_COLOR, "-troughcolor", "troughColor", "Background",
	DEF_SCROLLBAR_TROUGH_COLOR, Blt_Offset(Scrollbar, troughColorPtr),
	BLT_CONFIG_COLOR_ONLY},
    {BLT_CONFIG_COLOR, "-troughcolor", "troughColor", "Background",
	DEF_SCROLLBAR_TROUGH_MONO, Blt_Offset(Scrollbar, troughColorPtr),
	BLT_CONFIG_MONO_ONLY},
    {BLT_CONFIG_PIXELS, "-width", "width", "Width",
	DEF_SCROLLBAR_WIDTH, Blt_Offset(Scrollbar, width), 0},
    {BLT_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL,
	(char *)NULL, 0, 0}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static void ComputeScrollbarGeometry _ANSI_ARGS_((
	Scrollbar *scrollPtr));
static int ConfigureScrollbar _ANSI_ARGS_((Tcl_Interp *interp,
	Scrollbar *scrollPtr, int objc, Tcl_Obj *CONST *objv,
	int flags));
static void DestroyScrollbar _ANSI_ARGS_((DestroyData *memPtr));
static void DisplayScrollbar _ANSI_ARGS_((ClientData clientData));
static void EventuallyRedraw _ANSI_ARGS_((Scrollbar *scrollPtr));
static void ScrollbarCmdDeletedProc _ANSI_ARGS_((
	ClientData clientData));
static void ScrollbarEventProc _ANSI_ARGS_((ClientData clientData,
	XEvent *eventPtr));
static int ScrollbarPosition _ANSI_ARGS_((Scrollbar *scrollPtr,
	int x, int y));
static int ScrollbarWidgetCmd _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *, int objc, Tcl_Obj *CONST *objv));

static Blt_TileChangedProc TileChangedProc;
static Tcl_ObjCmdProc ScrollbarCmd;
/*
 *--------------------------------------------------------------
 *
 * ScrollbarCmd --
 *
 *	This procedure is invoked to process the "scrollbar" Tcl
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

/*ARGSUSED*/
static int
ScrollbarCmd(
    ClientData clientData,	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument strings. */
{
    Scrollbar *scrollPtr;
    Tk_Window tkwin;

    if (objc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
	    Tcl_GetString(objv[0]), " pathName ?options?\"", (char *)NULL);
	return TCL_ERROR;
    }
    tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp), 
	Tcl_GetString(objv[1]), (char *)NULL);
    if (tkwin == NULL) {
	Tcl_AppendResult(interp, 
			 "can't create window from ", Tcl_GetString(objv[1]),
			 (char *)NULL);

	return TCL_ERROR;
    }
    /*
     * Initialize fields that won't be initialized by ConfigureScrollbar,
     * or which ConfigureScrollbar expects to have reasonable values
     * (e.g. resource pointers).
     */

    scrollPtr = Blt_Calloc(1, sizeof(Scrollbar));
    scrollPtr->tkwin = tkwin;
    scrollPtr->display = Tk_Display(tkwin);
    scrollPtr->interp = interp;
    scrollPtr->widgetCmd = Tcl_CreateObjCommand(interp,
	Tk_PathName(scrollPtr->tkwin), ScrollbarWidgetCmd,
	(ClientData)scrollPtr, ScrollbarCmdDeletedProc);
    scrollPtr->relief = TK_RELIEF_FLAT;
    scrollPtr->elementBorderWidth = -1;
    scrollPtr->activeRelief = TK_RELIEF_RAISED;
    scrollPtr->minSliderLength = MIN_SLIDER_LENGTH;

    Tk_SetClass(scrollPtr->tkwin, "Scrollbar");
    Tk_CreateEventHandler(scrollPtr->tkwin,
	ExposureMask | StructureNotifyMask | FocusChangeMask,
	ScrollbarEventProc, (ClientData)scrollPtr);
    if (ConfigureScrollbar(interp, scrollPtr, objc - 2, objv + 2, 0) 
	!= TCL_OK) {
	goto error;
    }
    Tcl_SetObjResult(interp, objv[1]);
    return TCL_OK;

  error:
    Tk_DestroyWindow(scrollPtr->tkwin);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ScrollbarWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
ScrollbarWidgetCmd(clientData, interp, objc, objv)
    ClientData clientData;	/* Information about scrollbar
					 * widget. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST *objv;		/* Argument strings. */
{
    Scrollbar *scrollPtr = clientData;
    int result = TCL_OK;
    size_t length;
    int c;
    char *string;

    if (objc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
	    objv[0], " option ?arg arg ...?\"", (char *)NULL);
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData)scrollPtr);
    string = Tcl_GetString(objv[1]);
    c = string[0];
    length = strlen(string);
    if ((c == 'a') && (strncmp(string, "activate", length) == 0)) {
	if (objc == 2) {
	    switch (scrollPtr->activeField) {
	    case TOP_ARROW:
		Tcl_SetStringObj(Tcl_GetObjResult(interp), "arrow1", 6);
		break;
	    case SLIDER:
		Tcl_SetStringObj(Tcl_GetObjResult(interp), "slider", 6);
		break;
	    case BOTTOM_ARROW:
		Tcl_SetStringObj(Tcl_GetObjResult(interp), "arrow2", 6);
		break;
	    }
	    goto done;
	}
	if (objc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " activate element\"", (char *)NULL);
	    goto error;
	}
	string = Tcl_GetString(objv[2]);
	c = string[0];
	length = strlen(string);
	if ((c == 'a') && (strcmp(string, "arrow1") == 0)) {
	    scrollPtr->activeField = TOP_ARROW;
	} else if ((c == 'a') && (strcmp(string, "arrow2") == 0)) {
	    scrollPtr->activeField = BOTTOM_ARROW;
	} else if ((c == 's') && (strncmp(string, "slider", length) == 0)) {
	    scrollPtr->activeField = SLIDER;
	} else {
	    scrollPtr->activeField = OUTSIDE;
	}
	EventuallyRedraw(scrollPtr);
    } else if ((c == 'c') && (length >= 2) && 
	       (strncmp(string, "cget", length) == 0)) {
	if (objc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " cget option\"", (char *)NULL);
	    goto error;
	}
	result = Blt_ConfigureValueFromObj(interp, scrollPtr->tkwin, 
		configSpecs, (char *)scrollPtr, objv[2], 0);
    } else if ((c == 'c') && (length >= 2) && 
	       (strncmp(string, "configure", length) == 0)) {
	if (objc == 2) {
	    result = Blt_ConfigureInfoFromObj(interp, scrollPtr->tkwin, 
		configSpecs, (char *)scrollPtr, (Tcl_Obj *)NULL, 0);
	} else if (objc == 3) {
	    result = Blt_ConfigureInfoFromObj(interp, scrollPtr->tkwin, 
		configSpecs, (char *)scrollPtr, objv[2], 0);
	} else {
	    result = ConfigureScrollbar(interp, scrollPtr, objc - 2, objv + 2,
		BLT_CONFIG_OBJV_ONLY);
	}
    } else if ((c == 'd') && (strncmp(string, "delta", length) == 0)) {
	int xDelta, yDelta, pixels, barWidth;
	double fraction;

	if (objc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " delta xDelta yDelta\"", (char *)NULL);
	    goto error;
	}
	if ((Tcl_GetIntFromObj(interp, objv[2], &xDelta) != TCL_OK) || 
	    (Tcl_GetIntFromObj(interp, objv[3], &yDelta) != TCL_OK)) {
	    goto error;
	}
	if (scrollPtr->vertical) {
	    pixels = yDelta;
	    barWidth = Tk_Height(scrollPtr->tkwin) - 1
		- 2 * (scrollPtr->arrowLength + scrollPtr->inset);
	} else {
	    pixels = xDelta;
	    barWidth = Tk_Width(scrollPtr->tkwin) - 1
		- 2 * (scrollPtr->arrowLength + scrollPtr->inset);
	}
	if (barWidth == 0) {
	    fraction = 0.0;
	} else {
	    fraction = ((double)pixels / (double)barWidth);
	}
	Tcl_SetDoubleObj(Tcl_GetObjResult(interp), fraction);
    } else if ((c == 'f') && (strncmp(string, "fraction", length) == 0)) {
	int x, y, pos, barWidth;
	double fraction;

	if (objc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " fraction x y\"", (char *)NULL);
	    goto error;
	}
	if ((Tcl_GetIntFromObj(interp, objv[2], &x) != TCL_OK) || 
	    (Tcl_GetIntFromObj(interp, objv[3], &y) != TCL_OK)) {
	    goto error;
	}
	if (scrollPtr->vertical) {
	    pos = y - (scrollPtr->arrowLength + scrollPtr->inset);
	    barWidth = Tk_Height(scrollPtr->tkwin) - 1
		- 2 * (scrollPtr->arrowLength + scrollPtr->inset);
	} else {
	    pos = x - (scrollPtr->arrowLength + scrollPtr->inset);
	    barWidth = Tk_Width(scrollPtr->tkwin) - 1
		- 2 * (scrollPtr->arrowLength + scrollPtr->inset);
	}
	if (barWidth == 0) {
	    fraction = 0.0;
	} else {
	    fraction = ((double)pos / (double)barWidth);
	}
	if (fraction < 0) {
	    fraction = 0;
	} else if (fraction > 1.0) {
	    fraction = 1.0;
	}
	Tcl_SetDoubleObj(Tcl_GetObjResult(interp), fraction);
    } else if ((c == 'g') && (strncmp(string, "get", length) == 0)) {
	if (objc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " get\"", (char *)NULL);
	    goto error;
	}
	if (scrollPtr->flags & NEW_STYLE_COMMANDS) {
	    Tcl_Obj *listObjPtr;

	    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewDoubleObj(scrollPtr->firstFraction));
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewDoubleObj(scrollPtr->lastFraction));
	    Tcl_SetObjResult(interp, listObjPtr);
	} else {
	    Tcl_Obj *listObjPtr;

	    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewIntObj(scrollPtr->totalUnits));
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewIntObj(scrollPtr->windowUnits));
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewIntObj(scrollPtr->firstUnit));
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewIntObj(scrollPtr->lastUnit));
	    Tcl_SetObjResult(interp, listObjPtr);
	}
    } else if ((c == 'i') && (strncmp(string, "identify", length) == 0)) {
	int x, y, thing;

	if (objc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " identify x y\"", (char *)NULL);
	    goto error;
	}
	if ((Tcl_GetIntFromObj(interp, objv[2], &x) != TCL_OK) || 
	    (Tcl_GetIntFromObj(interp, objv[3], &y) != TCL_OK)) {
	    goto error;
	}
	thing = ScrollbarPosition(scrollPtr, x, y);
	switch (thing) {
	case TOP_ARROW:
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), "arrow1", 6);
	    break;
	case TOP_GAP:
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), "trough1", 7);
	    break;
	case SLIDER:
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), "slider", 6);
	    break;
	case BOTTOM_GAP:
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), "trough2", 7);
	    break;
	case BOTTOM_ARROW:
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), "arrow2", 6);
	    break;
	}
    } else if ((c == 's') && (strncmp(string, "set", length) == 0)) {
	int totalUnits, windowUnits, firstUnit, lastUnit;

	if (objc == 4) {
	    double first, last;

	    if ((Tcl_GetDoubleFromObj(interp, objv[2], &first) != TCL_OK) ||
		(Tcl_GetDoubleFromObj(interp, objv[3], &last) != TCL_OK)) {
		goto error;
	    }
	    if (first < 0) {
		scrollPtr->firstFraction = 0;
	    } else if (first > 1.0) {
		scrollPtr->firstFraction = 1.0;
	    } else {
		scrollPtr->firstFraction = first;
	    }
	    if (last < scrollPtr->firstFraction) {
		scrollPtr->lastFraction = scrollPtr->firstFraction;
	    } else if (last > 1.0) {
		scrollPtr->lastFraction = 1.0;
	    } else {
		scrollPtr->lastFraction = last;
	    }
	    scrollPtr->flags |= NEW_STYLE_COMMANDS;
	} else if (objc == 6) {
	    if (Tcl_GetIntFromObj(interp, objv[2], &totalUnits) != TCL_OK) {
		goto error;
	    }
	    if (totalUnits < 0) {
		totalUnits = 0;
	    }
	    if (Tcl_GetIntFromObj(interp, objv[3], &windowUnits) != TCL_OK) {
		goto error;
	    }
	    if (windowUnits < 0) {
		windowUnits = 0;
	    }
	    if (Tcl_GetIntFromObj(interp, objv[4], &firstUnit) != TCL_OK) {
		goto error;
	    }
	    if (Tcl_GetIntFromObj(interp, objv[5], &lastUnit) != TCL_OK) {
		goto error;
	    }
	    if (totalUnits > 0) {
		if (lastUnit < firstUnit) {
		    lastUnit = firstUnit;
		}
	    } else {
		firstUnit = lastUnit = 0;
	    }
	    scrollPtr->totalUnits = totalUnits;
	    scrollPtr->windowUnits = windowUnits;
	    scrollPtr->firstUnit = firstUnit;
	    scrollPtr->lastUnit = lastUnit;
	    if (scrollPtr->totalUnits == 0) {
		scrollPtr->firstFraction = 0.0;
		scrollPtr->lastFraction = 1.0;
	    } else {
		scrollPtr->firstFraction = ((double)firstUnit) / totalUnits;
		scrollPtr->lastFraction = ((double)(lastUnit + 1)) / totalUnits;
	    }
	    scrollPtr->flags &= ~NEW_STYLE_COMMANDS;
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), 
		" set firstFraction lastFraction\" or \"",
		Tcl_GetString(objv[0]),
		" set totalUnits windowUnits firstUnit lastUnit\"",
		(char *)NULL);
	    goto error;
	}
	ComputeScrollbarGeometry(scrollPtr);
	EventuallyRedraw(scrollPtr);
    } else {
	Tcl_AppendResult(interp, "bad option \"", Tcl_GetString(objv[1]),
	    "\": must be activate, cget, configure, delta, fraction, ",
	    "get, identify, or set", (char *)NULL);
	goto error;
    }
  done:
    Tcl_Release((ClientData)scrollPtr);
    return result;

  error:
    Tcl_Release((ClientData)scrollPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyScrollbar --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a scrollbar at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the scrollbar is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyScrollbar(DestroyData *memPtr) /* Info about scrollbar widget. */
{
    Scrollbar *scrollPtr = (Scrollbar *)memPtr;

    /*
     * Free up all the stuff that requires special handling, then
     * let Blt_FreeOptions handle all the standard option-related
     * stuff.
     */

    if (scrollPtr->troughGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->troughGC);
    }
    if (scrollPtr->copyGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->copyGC);
    }
    if (scrollPtr->arrowGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->arrowGC);
    }
    if (scrollPtr->activeTile != NULL) {
	Blt_FreeTile(scrollPtr->activeTile);
    }
    if (scrollPtr->tile != NULL) {
	Blt_FreeTile(scrollPtr->tile);
    }
    Blt_FreeOptions(configSpecs, (char *)scrollPtr, scrollPtr->display, 0);
    Blt_Free(scrollPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * TileChangedProc
 *
 *	Routine for tile change notifications.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
TileChangedProc(clientData, tile)
    ClientData clientData;
    Blt_Tile tile;		/* Not used. */
{
    Scrollbar *scrollPtr = clientData;

    if (scrollPtr->tkwin != NULL) {
	EventuallyRedraw(scrollPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureScrollbar --
 *
 *	This procedure is called to process an objv/objc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a scrollbar widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for scrollPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */

static int
ConfigureScrollbar(
    Tcl_Interp *interp,		/* Used for error reporting. */
    Scrollbar *scrollPtr,	/* Information about widget; may or
				 * may not already have values for
				 * some fields. */
    int objc,			/* Number of valid entries in objv. */
    Tcl_Obj *CONST *objv,	/* Arguments. */
    int flags)			/* Flags to pass to
				 * Blt_ConfigureWidget. */
{
    size_t length;
    XGCValues gcValues;
    GC new;

    if (Blt_ConfigureWidgetFromObj(interp, scrollPtr->tkwin, configSpecs,
	    objc, objv, (char *)scrollPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * A few options need special processing, such as parsing the
     * orientation or setting the background from a 3-D border.
     */

    length = strlen(scrollPtr->orientation);
    if (strncmp(scrollPtr->orientation, "vertical", length) == 0) {
	scrollPtr->vertical = 1;
    } else if (strncmp(scrollPtr->orientation, "horizontal", length) == 0) {
	scrollPtr->vertical = 0;
    } else {
	Tcl_AppendResult(interp, "bad orientation \"", scrollPtr->orientation,
	    "\": must be vertical or horizontal", (char *)NULL);
	return TCL_ERROR;
    }

    if (scrollPtr->command != NULL) {
	scrollPtr->commandSize = strlen(scrollPtr->command);
    } else {
	scrollPtr->commandSize = 0;
    }
    if (scrollPtr->activeTile != NULL) {
	Blt_SetTileChangedProc(scrollPtr->activeTile, TileChangedProc,
	    (ClientData)scrollPtr);
    }
    if (scrollPtr->tile != NULL) {
	Blt_SetTileChangedProc(scrollPtr->tile, TileChangedProc,
	    (ClientData)scrollPtr);
    }
    Tk_SetBackgroundFromBorder(scrollPtr->tkwin, scrollPtr->bgBorder);

    gcValues.foreground = scrollPtr->troughColorPtr->pixel;
    new = Tk_GetGC(scrollPtr->tkwin, GCForeground, &gcValues);
    if (scrollPtr->troughGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->troughGC);
    }
    scrollPtr->troughGC = new;
    if (scrollPtr->copyGC == None) {
	gcValues.graphics_exposures = False;
	scrollPtr->copyGC = Tk_GetGC(scrollPtr->tkwin, GCGraphicsExposures,
	    &gcValues);
    }
    gcValues.foreground = BlackPixel(
	Tk_Display(scrollPtr->tkwin), Tk_ScreenNumber(scrollPtr->tkwin));
    new = Tk_GetGC(scrollPtr->tkwin, GCForeground, &gcValues);
    if (scrollPtr->arrowGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->arrowGC);
    }
    scrollPtr->arrowGC = new;
    /*
     * Register the desired geometry for the window (leave enough space
     * for the two arrows plus a minimum-size slider, plus border around
     * the whole window, if any).  Then arrange for the window to be
     * redisplayed.
     */

    ComputeScrollbarGeometry(scrollPtr);
    EventuallyRedraw(scrollPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayScrollbar --
 *
 *	This procedure redraws the contents of a scrollbar window.
 *	It is invoked as a do-when-idle handler, so it only runs
 *	when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

static void
DisplayScrollbar(ClientData clientData)	/* Information about window. */
{
    Scrollbar *scrollPtr = clientData;
    Tk_Window tkwin = scrollPtr->tkwin;
    XPoint points[7];
    Tk_3DBorder border;
    int relief, width, elementBorderWidth;
    Pixmap pixmap;
    Blt_Tile tile;

    if ((scrollPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
	goto done;
    }
    if (scrollPtr->vertical) {
	width = Tk_Width(tkwin) - 2 * scrollPtr->inset;
    } else {
	width = Tk_Height(tkwin) - 2 * scrollPtr->inset;
    }
    elementBorderWidth = scrollPtr->elementBorderWidth;
    if (elementBorderWidth < 0) {
	elementBorderWidth = scrollPtr->borderWidth;
    }

    /*
     * In order to avoid screen flashes, this procedure redraws
     * the scrollbar in a pixmap, then copies the pixmap to the
     * screen in a single operation.  This means that there's no
     * point in time where the on-sreen image has been cleared.
     */

    pixmap = Tk_GetPixmap(scrollPtr->display, Tk_WindowId(tkwin),
	Tk_Width(tkwin), Tk_Height(tkwin), Tk_Depth(tkwin));

    if (scrollPtr->highlightWidth != 0) {
	GC gc;

	if (scrollPtr->flags & GOT_FOCUS) {
	    gc = Tk_GCForColor(scrollPtr->highlightColorPtr, pixmap);
	} else {
	    gc = Tk_GCForColor(scrollPtr->highlightBgColorPtr, pixmap);
	}
	Tk_DrawFocusHighlight(tkwin, gc, scrollPtr->highlightWidth, pixmap);
    }
    Blt_Draw3DRectangle(tkwin, pixmap, scrollPtr->bgBorder,
	scrollPtr->highlightWidth, scrollPtr->highlightWidth,
	Tk_Width(tkwin) - 2 * scrollPtr->highlightWidth,
	Tk_Height(tkwin) - 2 * scrollPtr->highlightWidth,
	scrollPtr->borderWidth, scrollPtr->relief);

    if (scrollPtr->tile != NULL) {
	Blt_SetTileOrigin(tkwin, scrollPtr->tile, 0, 0);
	Blt_TileRectangle(tkwin, pixmap, scrollPtr->tile, scrollPtr->inset, 
		scrollPtr->inset,
		(unsigned)(Tk_Width(tkwin) - 2 * scrollPtr->inset),
		(unsigned)(Tk_Height(tkwin) - 2 * scrollPtr->inset));
    } else {
	XFillRectangle(scrollPtr->display, pixmap, scrollPtr->troughGC,
		scrollPtr->inset, scrollPtr->inset,
		(unsigned)(Tk_Width(tkwin) - 2 * scrollPtr->inset),
		(unsigned)(Tk_Height(tkwin) - 2 * scrollPtr->inset));
    }

    /*
     * Draw the top or left arrow.  The coordinates of the polygon
     * points probably seem odd, but they were carefully chosen with
     * respect to X's rules for filling polygons.  These point choices
     * cause the arrows to just fill the narrow dimension of the
     * scrollbar and be properly centered.
     */
    tile = NULL;
    if (scrollPtr->activeField == TOP_ARROW) {
	border = scrollPtr->activeBorder;
	relief = scrollPtr->activeField == TOP_ARROW ? scrollPtr->activeRelief
	    : TK_RELIEF_RAISED;
	if (scrollPtr->activeTile != NULL) {
	    Blt_SetTileOrigin(tkwin, scrollPtr->activeTile, 0, 0);
	    tile = scrollPtr->activeTile;
	}
    } else {
	border = scrollPtr->bgBorder;
	relief = TK_RELIEF_RAISED;
	if (scrollPtr->tile != NULL) {
	    tile = scrollPtr->tile;
	}
    }
    if (scrollPtr->vertical) {
	points[0].x = scrollPtr->inset - 1;
	points[0].y = scrollPtr->arrowLength + scrollPtr->inset - 1;
	points[1].x = width + scrollPtr->inset;
	points[1].y = points[0].y;
	points[2].x = width / 2 + scrollPtr->inset;
	points[2].y = scrollPtr->inset - 1;
    } else {
	points[0].x = scrollPtr->arrowLength + scrollPtr->inset - 1;
	points[0].y = scrollPtr->inset - 1;
	points[1].x = scrollPtr->inset;
	points[1].y = width / 2 + scrollPtr->inset;
	points[2].x = points[0].x;
	points[2].y = width + scrollPtr->inset;
    }
#ifdef notdef
    if (tile != NULL) {
	Blt_TilePolygon(tkwin, pixmap, tile, points, 3);
	Tk_Draw3DPolygon(tkwin, pixmap, border, points, 3,
	    elementBorderWidth, relief);
    } else {
	Tk_Fill3DPolygon(tkwin, pixmap, border, points, 3,
	    elementBorderWidth, relief);
    }
#else
    Blt_Fill3DRectangle(tkwin, pixmap, border, scrollPtr->inset, 
       scrollPtr->inset, width, width, elementBorderWidth, relief); 
    
    Blt_DrawArrow(scrollPtr->display, pixmap, scrollPtr->arrowGC, 
		  scrollPtr->inset + width / 2, 
		  scrollPtr->inset + width / 2, 
		  STD_ARROW_HEIGHT, 
		  (scrollPtr->vertical) ? ARROW_UP : ARROW_LEFT);
#endif    
    /*
     * Display the bottom or right arrow.
     */
    tile = NULL;
    if (scrollPtr->activeField == BOTTOM_ARROW) {
	border = scrollPtr->activeBorder;
	relief = scrollPtr->activeField == BOTTOM_ARROW
	    ? scrollPtr->activeRelief : TK_RELIEF_RAISED;
	if (scrollPtr->activeTile != NULL) {
	    Blt_SetTileOrigin(tkwin, scrollPtr->activeTile, 0, 0);
	    tile = scrollPtr->activeTile;
	}
    } else {
	border = scrollPtr->bgBorder;
	relief = TK_RELIEF_RAISED;
	if (scrollPtr->tile != NULL) {
	    tile = scrollPtr->tile;
	}
    }
    if (scrollPtr->vertical) {
	points[0].x = scrollPtr->inset;
	points[0].y = Tk_Height(tkwin) - scrollPtr->arrowLength
	    - scrollPtr->inset + 1;
	points[1].x = width / 2 + scrollPtr->inset;
	points[1].y = Tk_Height(tkwin) - scrollPtr->inset;
	points[2].x = width + scrollPtr->inset;
	points[2].y = points[0].y;
    } else {
	points[0].x = Tk_Width(tkwin) - scrollPtr->arrowLength
	    - scrollPtr->inset + 1;
	points[0].y = scrollPtr->inset - 1;
	points[1].x = points[0].x;
	points[1].y = width + scrollPtr->inset;
	points[2].x = Tk_Width(tkwin) - scrollPtr->inset;
	points[2].y = width / 2 + scrollPtr->inset;
    }
#ifdef notdef
    if (tile != NULL) {
	Blt_TilePolygon(tkwin, pixmap, tile, points, 3);
	Tk_Draw3DPolygon(tkwin, pixmap, border, points, 3,
	    elementBorderWidth, relief);
    } else {
	Tk_Fill3DPolygon(tkwin, pixmap, border, points, 3,
	    elementBorderWidth, relief);
    }
#else
    Blt_Fill3DRectangle(tkwin, pixmap, border, 
		       Tk_Width(tkwin) - (width + scrollPtr->inset), 
		       Tk_Height(tkwin) - (width + scrollPtr->inset),
		       width, width, elementBorderWidth, relief); 
    
    Blt_DrawArrow(scrollPtr->display, pixmap, scrollPtr->arrowGC, 
		  Tk_Width(tkwin) - (scrollPtr->inset + width / 2) - 1, 
		  Tk_Height(tkwin) - (scrollPtr->inset + width / 2) - 1, 
		  STD_ARROW_HEIGHT, 
		  (scrollPtr->vertical) ? ARROW_DOWN : ARROW_RIGHT);
#endif    
    /*
     * Display the slider.
     */
    tile = NULL;
    if (scrollPtr->activeField == SLIDER) {
	border = scrollPtr->activeBorder;
	relief = scrollPtr->activeField == SLIDER ? scrollPtr->activeRelief
	    : TK_RELIEF_RAISED;
	if (scrollPtr->activeTile != NULL) {
	    Blt_SetTileOrigin(tkwin, scrollPtr->activeTile, 0, 0);
	    tile = scrollPtr->activeTile;
	}
    } else {
	border = scrollPtr->bgBorder;
	relief = TK_RELIEF_RAISED;
	if (scrollPtr->tile != NULL) {
	    tile = scrollPtr->tile;
	}
    }
    if (scrollPtr->vertical) {
	if (tile != NULL) {
	    Blt_TileRectangle(tkwin, pixmap, tile, scrollPtr->inset, 
		scrollPtr->sliderFirst, width - 1,
		scrollPtr->sliderLast - scrollPtr->sliderFirst - 1);
	    Blt_Draw3DRectangle(tkwin, pixmap, border,
		scrollPtr->inset, scrollPtr->sliderFirst,
		width, scrollPtr->sliderLast - scrollPtr->sliderFirst,
		elementBorderWidth, relief);
	} else {
	    Blt_Fill3DRectangle(tkwin, pixmap, border,
		scrollPtr->inset, scrollPtr->sliderFirst,
		width, scrollPtr->sliderLast - scrollPtr->sliderFirst,
		elementBorderWidth, relief);
	}
    } else {
	if (tile != NULL) {
	    Blt_TileRectangle(tkwin, pixmap, tile, scrollPtr->sliderFirst, 
		scrollPtr->inset,
		scrollPtr->sliderLast - scrollPtr->sliderFirst - 1, width - 1);
	    Blt_Draw3DRectangle(tkwin, pixmap, border,
		scrollPtr->sliderFirst, scrollPtr->inset,
		scrollPtr->sliderLast - scrollPtr->sliderFirst, width,
		elementBorderWidth, relief);
	} else {
	    Blt_Fill3DRectangle(tkwin, pixmap, border,
		scrollPtr->sliderFirst, scrollPtr->inset,
		scrollPtr->sliderLast - scrollPtr->sliderFirst, width,
	       elementBorderWidth, relief);
	}
    }

    /*
     * Copy the information from the off-screen pixmap onto the screen,
     * then delete the pixmap.
     */

    XCopyArea(scrollPtr->display, pixmap, Tk_WindowId(tkwin),
	scrollPtr->copyGC, 0, 0, (unsigned)Tk_Width(tkwin),
	(unsigned)Tk_Height(tkwin), 0, 0);
    Tk_FreePixmap(scrollPtr->display, pixmap);

  done:
    scrollPtr->flags &= ~REDRAW_PENDING;
}

/*
 *--------------------------------------------------------------
 *
 * ScrollbarEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on scrollbars.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */
static void
ScrollbarEventProc(
    ClientData clientData,	/* Information about window. */
    XEvent *eventPtr)		/* Information about event. */
{
    Scrollbar *scrollPtr = clientData;

    if ((eventPtr->type == Expose) && (eventPtr->xexpose.count == 0)) {
	EventuallyRedraw(scrollPtr);
    } else if (eventPtr->type == DestroyNotify) {
	if (scrollPtr->tkwin != NULL) {
	    scrollPtr->tkwin = NULL;
	    Tcl_DeleteCommandFromToken(scrollPtr->interp,scrollPtr->widgetCmd);
	}
	if (scrollPtr->flags & REDRAW_PENDING) {
	    Tcl_CancelIdleCall(DisplayScrollbar, (ClientData)scrollPtr);
	}
	Tcl_EventuallyFree((ClientData)scrollPtr,
	    (Tcl_FreeProc *)DestroyScrollbar);
    } else if (eventPtr->type == ConfigureNotify) {
	ComputeScrollbarGeometry(scrollPtr);
	EventuallyRedraw(scrollPtr);
    } else if (eventPtr->type == FocusIn) {
	if (eventPtr->xfocus.detail != NotifyInferior) {
	    scrollPtr->flags |= GOT_FOCUS;
	    if (scrollPtr->highlightWidth > 0) {
		EventuallyRedraw(scrollPtr);
	    }
	}
    } else if (eventPtr->type == FocusOut) {
	if (eventPtr->xfocus.detail != NotifyInferior) {
	    scrollPtr->flags &= ~GOT_FOCUS;
	    if (scrollPtr->highlightWidth > 0) {
		EventuallyRedraw(scrollPtr);
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ScrollbarCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */
static void
ScrollbarCmdDeletedProc(ClientData clientData)
{
    Scrollbar *scrollPtr = clientData;
    Tk_Window tkwin = scrollPtr->tkwin;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (tkwin != NULL) {
	scrollPtr->tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeScrollbarGeometry --
 *
 *	After changes in a scrollbar's size or configuration, this
 *	procedure recomputes various geometry information used in
 *	displaying the scrollbar.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The scrollbar will be displayed differently.
 *
 *----------------------------------------------------------------------
 */

static void
ComputeScrollbarGeometry(Scrollbar *sp)	
{
    int width, fieldLength;

    if (sp->highlightWidth < 0) {
	sp->highlightWidth = 0;
    }
    sp->inset = sp->highlightWidth + sp->borderWidth;
    width = (sp->vertical) ? Tk_Width(sp->tkwin) : Tk_Height(sp->tkwin);
    sp->arrowLength = width - (2 * sp->inset + 1);
    fieldLength = (sp->vertical ? Tk_Height(sp->tkwin) : 
		   Tk_Width(sp->tkwin)) - 2 * (sp->arrowLength + sp->inset);
    if (fieldLength < 0) {
	fieldLength = 0;
    }
    sp->sliderFirst = (int)(fieldLength * sp->firstFraction);
    sp->sliderLast = (int)(fieldLength * sp->lastFraction);

    /*
     * Adjust the slider so that some piece of it is always
     * displayed in the scrollbar and so that it has at least
     * a minimal width (so it can be grabbed with the mouse).
     */
    {
	int minSliderLength, sliderLength;

	minSliderLength = sp->minSliderLength;
	if (minSliderLength > fieldLength) {
	    minSliderLength = fieldLength;
	}
	sliderLength = sp->sliderLast - sp->sliderFirst;
	if (sliderLength < minSliderLength) {
	    fieldLength -= minSliderLength - sliderLength;
	    sp->sliderFirst = (int)(fieldLength * sp->firstFraction);
	    sp->sliderLast = sp->sliderFirst + minSliderLength;
	} else {
	    if (sp->sliderFirst > (fieldLength - 2 * sp->borderWidth)) {
		sp->sliderFirst = fieldLength - 2 * sp->borderWidth;
	    }
	    if (sp->sliderFirst < 0) {
		sp->sliderFirst = 0;
	    }
	    if (sp->sliderLast > fieldLength) {
		sp->sliderLast = fieldLength;
	    }
	}
    }
    sp->sliderFirst += sp->arrowLength + sp->inset;
    sp->sliderLast += sp->arrowLength + sp->inset;

    /*
     * Register the desired geometry for the window (leave enough space
     * for the two arrows plus a minimum-size slider, plus border around
     * the whole window, if any).  Then arrange for the window to be
     * redisplayed.
     */

    if (sp->vertical) {
	Tk_GeometryRequest(sp->tkwin,
	    sp->width + 2 * sp->inset,
	    2 * (sp->arrowLength + sp->borderWidth
		+ sp->inset));
    } else {
	Tk_GeometryRequest(sp->tkwin,
	    2 * (sp->arrowLength + sp->borderWidth
		+ sp->inset), sp->width + 2 * sp->inset);
    }
    Tk_SetInternalBorder(sp->tkwin, sp->inset);
}

/*
 *--------------------------------------------------------------
 *
 * ScrollbarPosition --
 *
 *	Determine the scrollbar element corresponding to a
 *	given position.
 *
 * Results:
 *	One of TOP_ARROW, TOP_GAP, etc., indicating which element
 *	of the scrollbar covers the position given by (x, y).  If
 *	(x,y) is outside the scrollbar entirely, then OUTSIDE is
 *	returned.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static int
ScrollbarPosition(
    Scrollbar *scrollPtr,	/* Scrollbar widget record. */
    int x, int y)		/* Coordinates within scrollPtr's 
				 * window. */
{
    int length, width, tmp;

    if (scrollPtr->vertical) {
	length = Tk_Height(scrollPtr->tkwin);
	width = Tk_Width(scrollPtr->tkwin);
    } else {
	tmp = x;
	x = y;
	y = tmp;
	length = Tk_Width(scrollPtr->tkwin);
	width = Tk_Height(scrollPtr->tkwin);
    }

    if ((x < scrollPtr->inset) || (x >= (width - scrollPtr->inset)) || 
	(y < scrollPtr->inset) || (y >= (length - scrollPtr->inset))) {
	return OUTSIDE;
    }
    /*
     * All of the calculations in this procedure mirror those in
     * DisplayScrollbar.  Be sure to keep the two consistent.
     */

    if (y < (scrollPtr->inset + scrollPtr->arrowLength)) {
	return TOP_ARROW;
    }
    if (y < scrollPtr->sliderFirst) {
	return TOP_GAP;
    }
    if (y < scrollPtr->sliderLast) {
	return SLIDER;
    }
    if (y >= (length - (scrollPtr->arrowLength + scrollPtr->inset))) {
	return BOTTOM_ARROW;
    }
    return BOTTOM_GAP;
}

/*
 *--------------------------------------------------------------
 *
 * EventuallyRedraw --
 *
 *	Arrange for one or more of the fields of a scrollbar
 *	to be redrawn.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
EventuallyRedraw(Scrollbar *scrollPtr) /* Information about widget. */
{
    if ((scrollPtr->tkwin == NULL) || (!Tk_IsMapped(scrollPtr->tkwin))) {
	return;
    }
    if ((scrollPtr->flags & REDRAW_PENDING) == 0) {
	Tcl_DoWhenIdle(DisplayScrollbar, (ClientData)scrollPtr);
	scrollPtr->flags |= REDRAW_PENDING;
    }
}

int
Blt_ScrollbarInit(Tcl_Interp *interp)
{
    static Blt_InitCmdSpec cmdSpec = { "scrollbar", ScrollbarCmd, };

    return Blt_InitCmd(interp, "blt::tile", &cmdSpec);
}

#endif /* NO_TILESCROLLBAR */
