
/*
 * bltPictureGif.c --
 *
 * This module implements GIF file format conversion routines for
 * the picture image type in the BLT toolkit.
 *
 *	Copyright 2003-2005 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * The GIF reader is from converters/other/giftopnm.c in the netpbm
 * (version 10.19) distribution.
 *
 *	Copyright 1990, 1991, 1993, David Koblas.  (koblas@netcom.com)
 *	Permission to use, copy, modify, and distribute this software
 *	and its documentation for any purpose and without fee is
 *	hereby granted, provided that the above copyright notice
 *	appear in all copies and that both that copyright notice and
 *	this permission notice appear in supporting documentation.
 *	This software is provided "as is" without express or implied
 *	warranty.
 *
 */

#include "bltInt.h"
#include <bltSink.h>
#include <bltHash.h>
#include "bltPicture.h"
#include "bltPictureFormats.h"

#include <X11/Xutil.h>

#include <setjmp.h>

typedef struct Blt_PictureStruct Picture;


#define MAX_LZW_BITS  12
#define MAX_LZW_CODE (1<<MAX_LZW_BITS)

#define GIF87A	1
#define GIF89A	2

typedef struct {
    jmp_buf jmpbuf;
    Tcl_DString errors;
    Tcl_DString warnings;
    int nWarnings, nErrors;
} GifMessage;

typedef struct {
    int version;		/* Eithe GIF87A or GIF89A */
    Pix32 globalColorTable[256];
    Pix32 localColorTable[256];

    /* Data to be parsed */
    Blt_DataSink data;

    /* Global parameters. */
    int logicalScreenWidth, logicalScreenHeight;
    int globalColorTableFlag;
    int colorResolution;
    int sortFlag;
    unsigned int globalColorTableSize;
    int bgColorIndex;
    int pixelAspectRatio;

    /* Local parameters. */
    int imageLeftPosition, imageTopPosition;
    int imageWidth, imageHeight;
    int localColorTableFlag;
    int interlaceFlag;
    int imageSortFlag;
    unsigned int localColorTableSize;

    /* Graphic control extension. */

    int disposalMethod;
    int userInputFlag;
    int transparentColorFlag;
    int delayTime;
    int transparentColorIndex;

    int lzwMinimumCodeSize;

} Gif;

/* Stack grows from low addresses to high addresses */
typedef struct {
    int *stack;			/* Malloc'ed array */
    int *sp;			/* Stack pointer */
    int *top;			/* Next word above top of stack */
} LzwStack;

typedef struct {
    unsigned char buf[280];	/* This is the buffer through which we
				 * read the data from the stream.  We
				 * must buffer it because we have to
				 * read whole data blocks at a time,
				 * but our client wants one code at a
				 * time.  The buffer typically
				 * contains the contents of one data
				 * block plus two bytes from the
				 * previous data block. */
    int bufCount;		/* # of bytes of contents in buf[]. */
    int curbit;			/* The bit number (starting at 0)
				 * within buf[] of the next bit to be
				 * returned.  If the next bit to be
				 * returned is not yet in buf[] (we've
				 * already returned everything in
				 * there), this points one beyond the
				 * end of the buffer contents.
				 */
    int streamExhausted;	/* The last time we read from the
				 * input stream, we got an EOD marker
				 * or EOF */
} LzwCodeState;


/*----------------------------------------------------------------------------
   Some notes on LZW.

   LZW is an extension of Limpel-Ziv.  The two extensions are:

     1) in Limpel-Ziv, codes are all the same number of bits.  In
        LZW, they start out small and increase as the stream progresses.

     2) LZW has a clear code that resets the string table and code
        size.

   The LZW code space is allocated as follows:

   The true data elements are dataWidth bits wide, so the maximum
   value of a true data element is 2**dataWidth-1.  We call that
   max_dataVal.  The first byte in the stream tells you what dataWidth
   is.

   LZW codes 0 - max_dataVal are direct codes.  Each on represents
   the true data element whose value is that of the LZW code itself.
   No decompression is required.

   max_dataVal + 1 and up are compression codes.  They encode
   true data elements:

   max_dataVal + 1 is the clear code.
         
   max_dataVal + 2 is the end code.

   max_dataVal + 3 and up are string codes.  Each string code 
   represents a string of true data elements.  The translation from a
   string code to the string of true data elements varies as the stream
   progresses.  In the beginning and after every clear code, the
   translation table is empty, so no string codes are valid.  As the
   stream progresses, the table gets filled and more string codes 
   become valid.

-----------------------------------------------------------------------------*/

typedef struct {
    LzwStack stack;
    int fresh;			/* The stream is right after a clear
				   code or at the very beginning */
    int codeSize;		/* Current code size -- each LZW code
				 * in this part of the image is this
				 * many bits.  Ergo, we read this many
				 * bits at a time from the stream. */

    int maxnum_code;		/* Maximum number of LZW codes that
				 * can be represented with the current
				 * code size. (1 << codeSize) */

    int next_tableSlot;		/* Index in the code translation table
				 * of the next free entry */
    int firstcode;		/* This is always a true data element
				 * code */
    int prevcode;		/* The code just before, in the image,
				 * the one we're processing now */

    int table[2][(1 << MAX_LZW_BITS)];

    /* The following are constant for the life of the decompressor */
    int init_codeSize;
    int max_dataVal;
    int clear_code;
    int end_code; 

    Blt_DataSink data;
} LzwDecompressor;

static int zeroDataBlock = 0;

static GifMessage *gifMessagePtr;

/*ARGSUSED*/
static void
GifError TCL_VARARGS_DEF(char *, arg1)
{
    char *fmt;
    char string[BUFSIZ+4];
    int length;
    va_list args;

    fmt = TCL_VARARGS_START(char *, arg1, args);
    length = vsnprintf(string, BUFSIZ, fmt, args);
    if (length > BUFSIZ) {
	strcat(string, "...");
    }
    Tcl_DStringAppend(&gifMessagePtr->errors, string, -1);
    va_end(args);
    longjmp(gifMessagePtr->jmpbuf, 0);
}

/*ARGSUSED*/
static void
GifWarning TCL_VARARGS_DEF(char *, arg1)
{
    char *fmt;
    char string[BUFSIZ+4];
    int length;
    va_list args;

    fmt = TCL_VARARGS_START(char *, arg1, args);
    length = vsnprintf(string, BUFSIZ, fmt, args);
    if (length > BUFSIZ) {
	strcat(string, "...");
    }
    Tcl_DStringAppend(&gifMessagePtr->warnings, string, -1);
    va_end(args);
    gifMessagePtr->nWarnings++;
}

/*
 *       7 6 5 4 3 2 1 0        Field Name                    Type 
 *      +----------------+ 
 *    0 |                |      Signature                     3 Bytes 
 *    1 |                | 
 *    2 |                | 
 *      +----------------+ 
 *    3 |                |      Version                       3 Bytes 
 *    4 |                | 
 *    5 |                | 
 *      +----------------+ 
 */
static int
GifHeader(Blt_DataSink data, Gif *gifPtr)
{
    unsigned char *bp;

    Blt_SinkResetCursor(data);
    if (Blt_SinkBytesLeft(data) < 6) {
	return FALSE;
    }
    bp = Blt_SinkPointer(data);
    if ((bp[0] != 'G') || (bp[1] != 'I') || (bp[2] != 'F') || (bp[3] != '8') ||
	(bp[5] != 'a')) {
	return FALSE;
    }
    if (bp[4] == '7') {
	gifPtr->version = GIF87A;
    } else if (bp[4] == '9') {
	gifPtr->version = GIF89A;
    }
    Blt_SinkIncrCursor(data, 6);
    return TRUE;
}

/*
 *        7 6 5 4 3 2 1 0        Field Name                    Type 
 *      +----------------+ 
 *   0  |                |       Logical Screen Width          Unsigned 
 *   1  |                | 
 *      +----------------+ 
 *   2  |                |       Logical Screen Height         Unsigned 
 *   3  |                | 
 *      +----------------+ 
 *   4  | |     | |      |       See below.
 *      +----------------+ 
 *   5  |                |       Background Color Index        Byte 
 *      +----------------+ 
 *   6  |                |       Pixel Aspect Ratio            Byte 
 *      +----------------+ 
 *
 *                       =       Global Color Table Flag       1 Bit 
 *                               Color Resolution              3 Bits 
 *                               Sort Flag                     1 Bit 
 *                               Size of Global Color Table    3 Bits 
 */
static int
GifLogicalScreenDescriptor(Blt_DataSink data, Gif *gifPtr)
{
    unsigned char *bp;

    if (Blt_SinkBytesLeft(data) <  7) {
	return TCL_ERROR;
    }
    bp = Blt_SinkPointer(data);
    /* Logical Screen Descriptor. */
    gifPtr->logicalScreenWidth = (bp[1] << 8) | bp[0];
    gifPtr->logicalScreenHeight = (bp[3] << 8) | bp[2];
    gifPtr->globalColorTableFlag = (bp[4] & 0x80);
    gifPtr->colorResolution =	((bp[4] & 0x70) >> 4);
    gifPtr->sortFlag = (bp[4] & 0x08) >> 3;
    gifPtr->globalColorTableSize = 1 << ((bp[4] & 0x07) + 1);
    gifPtr->bgColorIndex = bp[5];	/* Index of bgcolor color */
    gifPtr->pixelAspectRatio = bp[6];
    Blt_SinkIncrCursor(data, 7);
    return TCL_OK;
}

/*
 *        7 6 5 4 3 2 1 0        Field Name                    Type 
 *      +================+ 
 *   0  |                |       Red 0                         Byte 
 *   1  |                |       Green 0                       Byte 
 *   2  |                |       Blue 0                        Byte 
 *   3  |                |       Red 1                         Byte 
 *      |                |       Green 1                       Byte 
 *  up  |                | 
 *  to  |                | 
 *      |                |       Green 255                     Byte 
 * 767  |                |       Blue 255                      Byte 
 *      +================+ 
 */
static int
GifGlobalColorTable(Blt_DataSink data, Gif *gifPtr)
{
    Pix32 *dp;
    unsigned int i;
    unsigned char *bp;

    if (Blt_SinkBytesLeft(data) < (gifPtr->globalColorTableSize*3)) {
	return TCL_ERROR;
    }
    bp = Blt_SinkPointer(data);
    dp = gifPtr->globalColorTable;
    for (i = 0; i < gifPtr->globalColorTableSize; i++) {
	dp->Red = bp[0];
	dp->Green = bp[1];
	dp->Blue = bp[2];
	dp->Alpha = ALPHA_OPAQUE;
	dp++;
	bp += 3;
    }	    
    Blt_SinkSetPointer(data, bp);
    return TCL_OK;
}

/*
 *       7 6 5 4 3 2 1 0        Field Name                    Type
 *     +----------------+
 *  0  |                |       Image Separator               Byte
 *     +----------------+
 *  1  |                |       Image Left Position           Unsigned
 *  2  |                |
 *     +----------------+
 *  3  |                |       Image Top Position            Unsigned
 *  4  |                |
 *     +----------------+
 *  5  |                |       Image Width                   Unsigned
 *  6  |                |
 *     +----------------+
 *  7  |                |       Image Height                  Unsigned
 *  8  |                |
 *     +----------------+
 *  9  | | | |   |      |	See below
 *     +----------------+
 *
 *			=	Local Color Table Flag        1 Bit
 *				Interlace Flag                1 Bit
 *				Sort Flag                     1 Bit
 *				Reserved                      2 Bits
 *				Size of Local Color Table     3 Bits
 */
static int
GifImageDescriptor(Blt_DataSink data, Gif *gifPtr)
{
    unsigned char *bp;
    int size;

    if (Blt_SinkBytesLeft(data) < 9) {
	return TCL_ERROR;
    }
    bp = Blt_SinkPointer(data);
    gifPtr->imageLeftPosition =   (bp[1] << 8) | bp[0];
    gifPtr->imageTopPosition =    (bp[3] << 8) | bp[2];
    gifPtr->imageWidth =          (bp[5] << 8) | bp[4];
    gifPtr->imageHeight =         (bp[7] << 8) | bp[6];
    gifPtr->localColorTableFlag = (bp[8] & 0x80) >> 7;
    gifPtr->interlaceFlag =       (bp[8] & 0x40) >> 6;
    gifPtr->imageSortFlag =       (bp[8] & 0x20) >> 5;
    size =                        (bp[8] & 0x07);
    if (size > 0) {
        size = 1 << (size + 1);
    }
    gifPtr->localColorTableSize = size;
    Blt_SinkIncrCursor(data, 9);
    return TCL_OK;
}

/*
 *        7 6 5 4 3 2 1 0        Field Name                    Type 
 *      +================+ 
 *   0  |                |       Red 0                         Byte 
 *   1  |                |       Green 0                       Byte 
 *   2  |                |       Blue 0                        Byte 
 *   3  |                |       Red 1                         Byte 
 *      |                |       Green 1                       Byte 
 *  up  |                | 
 *  to  |                | 
 *      |                |       Green 255                     Byte 
 * 767  |                |       Blue 255                      Byte 
 *      +================+ 
 */
static int 
GifLocalColorTable(Blt_DataSink data, Gif *gifPtr)
{
    unsigned char *bp;
    Pix32 *dp;
    unsigned int i;

    if (Blt_SinkBytesLeft(data) < (gifPtr->localColorTableSize*3)) {
	return TCL_ERROR;
    }
    bp = Blt_SinkPointer(data);
    dp = gifPtr->localColorTable;
    for (i = 0; i < gifPtr->localColorTableSize; i++) {
	dp->Red = bp[0];
	dp->Green = bp[1];
	dp->Blue = bp[2];
	dp->Alpha = ALPHA_OPAQUE;
	dp++;
	bp += 3;
    }	    
    Blt_SinkSetPointer(data, bp);
    return TCL_OK;
}


/*
 *        7 6 5 4 3 2 1 0        Field Name                    Type 
 *      +----------------+ 
 *   0  |                |       Extension Introducer          Byte 
 *      +----------------+ 
 *   1  |                |       Graphic Control Label         Byte 
 *      +----------------+ 
 *
 *      +----------------+ 
 *   0  |                |       Block Size                    Byte 
 *      +----------------+ 
 *   1  |      |     | | |                      See below 
 *      +----------------+ 
 *   2  |                |       Delay Time                    Unsigned 
 *   3  |                | 
 *      +----------------+ 
 *   4  |                |       Transparent Color Index       Byte 
 *      +----------------+ 
 *
 *      +----------------+ 
 *   0  |                |       Block Terminator              Byte 
 *      +----------------+ 
 *
 *
 *      		  =     Reserved                      3 Bits 
 *                              Disposal Method               3 Bits 
 *                              User Input Flag               1 Bit 
 *                              Transparent Color Flag        1 Bit 
 */
static int
GifGraphicControlExtension(Blt_DataSink data, Gif *gifPtr)
{
    unsigned int length;

    while ((length = Blt_SinkNextByte(data)) > 0) {
	unsigned char *bp;

	bp = Blt_SinkPointer(data);
	gifPtr->disposalMethod = (bp[0] & 0x1C) >> 2;
	gifPtr->userInputFlag = bp[0] & 0x02;
	gifPtr->transparentColorFlag = bp[0] & 0x01;
	gifPtr->delayTime = (bp[2] << 8) | bp[1];
	gifPtr->transparentColorIndex = bp[3];
	Blt_SinkIncrCursor(data, length);
    }
    return TCL_OK;
}

static int
GifExtensions(Blt_DataSink data, Gif *gifPtr)
{
    /* 
     * Handle only the "Graphic control extension" block, ignoring
     * all others (text, comment, and application).
     */
    if (Blt_SinkNextByte(data) == 0xF9) {	
	if (GifGraphicControlExtension(data, gifPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	unsigned int length;
	
	/* Skip the data sub-blocks */
	while ((length = Blt_SinkNextByte(data)) > 0) {
	    if (Blt_SinkBytesLeft(data) < length) {
		return TCL_ERROR;
	    }
	    Blt_SinkIncrCursor(data, length);
	}
    }
    return TCL_OK;
}

static void
GifDataBlock(Blt_DataSink data, unsigned char *out, int *lengthPtr) 
{
    unsigned int length;

    length = Blt_SinkNextByte(data);
    if (Blt_SinkBytesLeft(data) < length) {
	*lengthPtr = -1;
    } else {
	if (length > 0) {
	    memcpy(out, Blt_SinkPointer(data), length);
	}
	Blt_SinkIncrCursor(data, length);
	*lengthPtr = (int)length;
    }
}

static void
LzwInitCodeState(LzwCodeState *statePtr) 
{
    /* Fake a previous data block */
    statePtr->buf[0] = statePtr->buf[1] = 0;
    statePtr->bufCount = 2;
    statePtr->curbit = statePtr->bufCount * 8;
    statePtr->streamExhausted = 0;
}

static void
LzwGetAnotherBlock(Blt_DataSink data, LzwCodeState *statePtr) 
{
    unsigned int count;
    unsigned int assumed_count;

    /* 
     * Shift buffer down so last two bytes are now the first two
     * bytes.  Shift 'curbit' cursor, which must be somewhere in or
     * immediately after those two bytes, accordingly. 
     */
    statePtr->buf[0] = statePtr->buf[statePtr->bufCount-2];
    statePtr->buf[1] = statePtr->buf[statePtr->bufCount-1];

    statePtr->curbit -= (statePtr->bufCount-2)*8;
    statePtr->bufCount = 2;
        
    /* Add the next block to the buffer */
    GifDataBlock(data, statePtr->buf  + statePtr->bufCount, &count);
    if (count == -1) {
        GifWarning("EOF encountered in image before EOD marker.  The GIF file is malformed, but we are proceeding anyway as if an EOD marker were at the end of the file.");
        assumed_count = 0;
    } else {
        assumed_count = count;
    }
    statePtr->streamExhausted = (assumed_count == 0);
    statePtr->bufCount += assumed_count;
}

static int
LzwNextCode(Blt_DataSink data, int codeSize, LzwCodeState *statePtr) 
{
    int result;

    if (((statePtr->curbit + codeSize) > (statePtr->bufCount * 8)) && 
	(!statePtr->streamExhausted)) {
        /* 
	 * Not enough left in buffer to satisfy request.  Get the next
	 * data block into the buffer.
	 */
        LzwGetAnotherBlock(data, statePtr);
    }
    if ((statePtr->curbit + codeSize) > (statePtr->bufCount * 8)) {
        /* 
	 * If the buffer still doesn't have enough bits in it, that means
	 * there were no data blocks left to read.
	 */
	result = -1;  /* EOF */

        {
            int const bitsUnused = (statePtr->bufCount * 8) - statePtr->curbit;
            if (bitsUnused > 0) {
                GifWarning("Stream ends with a partial code (%d bits left in file; expected a %d bit code). Ignoring.", bitsUnused, codeSize);
	    }
        }
    } else {
        int i, j;
        int code;
        unsigned char *bp;

        bp = statePtr->buf;
        code = 0;		/* Initial value */
        for (i = statePtr->curbit, j = 0; j < codeSize; ++i, ++j) {
            code |= ((bp[ i / 8 ] & (1 << (i % 8))) != 0) << j;
	}
        statePtr->curbit += codeSize;
        result = code;
    }
    return result;
}

/*-------------------------------------------------------------------------
 * 
 * LzwGetCode --
 *
 *	If 'init', initialize the code getter.  Otherwise, read and
 *	return the next LZW code from the buffer.
 *
 * Results:
 *	Retunes the retrieved code.  Otherwise -1 if we encounter the
 *	end of the file.
 *
 *--------------------------------------------------------------------------
 */
static int
LzwGetCode(
    Blt_DataSink data,		/* Buffer to read from. */
    int codeSize,		/* # of bits in the code we are to get. */
    int init)			/* Indicates to initial the code reader. */
{
    static LzwCodeState state;

    if (init) {
        LzwInitCodeState(&state);
        return 0;
    } 
    return LzwNextCode(data, codeSize, &state);
}

static void 
LzwInitStack(
    LzwDecompressor *dcPtr, 
    size_t size) 
{
    dcPtr->stack.stack = Blt_Malloc(size * sizeof(int));
    if (dcPtr->stack.stack == NULL) {
        GifError("Unable to allocate %d -word stack.", size);
    }
    dcPtr->stack.sp = dcPtr->stack.stack;
    dcPtr->stack.top = dcPtr->stack.stack + size;
}

static void
LzwPushStack(LzwDecompressor *dcPtr, int const value) 
{
    if (dcPtr->stack.sp >= dcPtr->stack.top) {
        GifError("stack overflow");
    }
    *(dcPtr->stack.sp++) = value;
}

static int
LzwStackIsEmpty(LzwDecompressor *dcPtr) 
{
    return (dcPtr->stack.sp == dcPtr->stack.stack);
}

static int
LzwPopStack(LzwDecompressor *dcPtr) 
{
    if (dcPtr->stack.sp <= dcPtr->stack.stack) {
        GifError("stack underflow");
    }
    return *(--dcPtr->stack.sp);
}

static void
LzwTermStack(LzwDecompressor *dcPtr) 
{
    Blt_Free(dcPtr->stack.stack);
    dcPtr->stack.stack = NULL;
}

static void
LzwResetDecompressor(LzwDecompressor *dcPtr) 
{
    dcPtr->codeSize = dcPtr->init_codeSize + 1;
    dcPtr->maxnum_code = 1 << dcPtr->codeSize;
    dcPtr->next_tableSlot = dcPtr->max_dataVal + 3;
    dcPtr->fresh = 1;
}

static void
LzwInit(Blt_DataSink data, int codeSize, LzwDecompressor *dcPtr) 
{
    dcPtr->init_codeSize = codeSize;

    dcPtr->max_dataVal = (1 << codeSize) - 1;
    dcPtr->clear_code = dcPtr->max_dataVal + 1;
    dcPtr->end_code = dcPtr->max_dataVal + 2;
    
    /* 
     * The entries in the translation table for true data codes are
     * constant throughout the stream.  We set them now and they never
     * change.
     */
    {
        int i;

        for (i = 0; i <= dcPtr->max_dataVal; ++i) {
            dcPtr->table[0][i] = 0;
            dcPtr->table[1][i] = i;
        }
    }
    LzwResetDecompressor(dcPtr);
    LzwGetCode(data, 0, TRUE);
    dcPtr->data = data;
    dcPtr->fresh = TRUE;
    LzwInitStack(dcPtr, MAX_LZW_CODE * 2);
}

static void
LzwTerm(LzwDecompressor *dcPtr) 
{
    LzwTermStack(dcPtr);
}

/*
 * -------------------------------------------------------------------------
 * 
 * LzwExpandCodeOntoStack --
 *
 *	'incode' is an LZW string code.  It represents a string of
 *	true data elements, as defined by the string translation
 *	table in *decompP.
 *
 *	Expand the code to a string of LZW direct codes and push them
 *	onto the * stack such that the leftmost code is on top.
 *
 *	Also add to the translation table where appropriate.
 *
 * Results:
 *	If successful, return TRUE. Iff the translation table contains a
 *	cycle (which means the LZW stream from which it was built is
 *	invalid), return FALSE.
 *	
 *------------------------------------------------------------------------
 */
static int
LzwExpandCodeOntoStack(LzwDecompressor *dcPtr, int incode) 
{
    int code;
    int error;

    error = FALSE;
    if (incode < dcPtr->next_tableSlot) {
        code = incode;
    } else {
        /* It's a code that isn't in our translation table yet */
        LzwPushStack(dcPtr, dcPtr->firstcode);
        code = dcPtr->prevcode;
    }

    {
        /* 
	 * Get the whole string that this compression code represents
	 * and push it onto the code stack so the leftmost code is
	 * on top.  Set dcPtr->firstcode to the first (leftmost)
	 * code in that string. 
	 */
        unsigned int stringCount;
        stringCount = 0;

        while ((code > dcPtr->max_dataVal) && (!error)) {
            if (stringCount > MAX_LZW_CODE) {
                GifError("contains LZW string loop");
            } 
	    ++stringCount;
	    LzwPushStack(dcPtr, dcPtr->table[1][code]);
	    code = dcPtr->table[0][code];
        }
        dcPtr->firstcode = dcPtr->table[1][code];
        LzwPushStack(dcPtr, dcPtr->firstcode);
    }

    if (dcPtr->next_tableSlot < MAX_LZW_CODE) {

        dcPtr->table[0][dcPtr->next_tableSlot] = dcPtr->prevcode;
        dcPtr->table[1][dcPtr->next_tableSlot] = dcPtr->firstcode;
        ++dcPtr->next_tableSlot;

        if (dcPtr->next_tableSlot >= dcPtr->maxnum_code) {
            /* 
	     * We've used up all the codes of the current code size.
	     * Future codes in the stream will have codes one bit
	     * longer.  But there's an exception if we're already at
	     * the LZW maximum, in which case the codes will simply
	     * continue the same size.
	     */
            if (dcPtr->codeSize < MAX_LZW_BITS) {
                ++dcPtr->codeSize;
                dcPtr->maxnum_code = 1 << dcPtr->codeSize;
            }
        }
    }
    dcPtr->prevcode = incode;
    return error == 0;
}

/*
 * ------------------------------------------------------------------------
 *
 * LzwReadByte --
 *
 *	Returns the next data element of the decompressed image.  In
 *	the context of a GIF, a data element is the color table index
 *	of one pixel.
 *
 *	We read and return the next byte of the decompressed image:
 *
 * Results:
 *	Returns -1, if we hit EOF prematurely (i.e. before an "end"
 *	code.   We forgive the case that the "end" code is followed
 *	by EOF instead of an EOD marker (zero length DataBlock)).
 *
 *	Returns -2 if there are no more bytes in the image.
 *
 *	Returns -3 if we encounter errors in the LZW stream. 
 *
 *------------------------------------------------------------------------
 */
static int
LzwReadByte(LzwDecompressor *dcPtr) 
{
    int code;

    if (!LzwStackIsEmpty(dcPtr)) {
        return LzwPopStack(dcPtr);
    }
    if (dcPtr->fresh) {
        dcPtr->fresh = FALSE;
        /* 
	 * Read off all initial clear codes, read the first non-clear code,
	 * and return it.  There are no strings in the table yet, so the next
	 * code must be a direct true data code.
	 */
        do {
            dcPtr->firstcode = LzwGetCode(dcPtr->data, dcPtr->codeSize, 
		FALSE);
            dcPtr->prevcode = dcPtr->firstcode; 
	} while (dcPtr->firstcode == dcPtr->clear_code);

        return dcPtr->firstcode;
    } 
    code = LzwGetCode(dcPtr->data, dcPtr->codeSize, FALSE);
    if (code < 0) {
	return code;
    } 
    if (code == dcPtr->clear_code) {
	LzwResetDecompressor(dcPtr);
	return LzwReadByte(dcPtr);
    } 
    if (code == dcPtr->end_code) {
	if (!zeroDataBlock) {
	    /* readThroughEod(dcPtr->ifP); */
	}
	return -2;
    } 
    if (!LzwExpandCodeOntoStack(dcPtr, code)) {
	return -3;
    }
    return LzwPopStack(dcPtr);
}


static Blt_Picture
GifCreatePictureFromData(Blt_DataSink data, Gif *gifPtr) 
{
    Blt_Picture picture;
    LzwDecompressor dc;
    Pix32 *colormap;
    int v;

    colormap = (gifPtr->localColorTableFlag) ? gifPtr->localColorTable :
	gifPtr->globalColorTable;
    LzwInit(data, gifPtr->lzwMinimumCodeSize, &dc);

    picture = Blt_CreatePicture(gifPtr->imageWidth, gifPtr->imageHeight);
    if (!gifPtr->transparentColorFlag) {
	gifPtr->transparentColorIndex = -1;
    }
    if (gifPtr->interlaceFlag) {
	static int istart[] = { 0, 4, 2, 1 };
	static int istep[] = { 8, 8, 4, 2 };
	int x, y, pass;
	
	pass = 0;
	x = y = 0;
	while ((v = LzwReadByte(&dc)) >= 0) {
	    Pix32 *dp;
	    
	    dp = Blt_PicturePixel(picture, x, y);
	    dp->color = colormap[v].color;
	    if (v == gifPtr->transparentColorIndex) {
		dp->Alpha = 0x0;
		picture->flags |= BLT_PICTURE_BLEND;
	    }
	    ++x;
	    if (x == gifPtr->imageWidth) {
		x = 0;
		y += istep[pass]; /* 8, 8, 4, 2 */
		if (y >= gifPtr->imageHeight) {
		    ++pass;
		    y = istart[pass]; /* 0, 4, 2, 1 */
		    if (y == 0) {
			goto done;
		    }
		}
	    }
	    if (y >= gifPtr->imageHeight) {
		break;
	    }
	}
    } else {
	Pix32 *destRowPtr;
	int y;

	v = 0;			/* Suppress compiler warning. */
	destRowPtr = Blt_PictureBits(picture);
	for (y = 0; y < gifPtr->imageHeight; y++) {
	    Pix32 *dp;
	    int x;

	    dp = destRowPtr;
	    for (x = 0; x < gifPtr->imageWidth; x++) {
		v = LzwReadByte(&dc);
		if (v < 0) {
		    goto done;
		}
		dp->color = colormap[v].color;
		if (v == gifPtr->transparentColorIndex) { 
		    dp->Alpha = 0x0;
		    picture->flags |= BLT_PICTURE_BLEND;
		}
		dp++;
	    }
	    destRowPtr += Blt_PictureStride(picture);
	}
    }
 done:
    if (v == -3) {
        GifError("Error in GIF input stream");
    } 
    if (LzwReadByte(&dc) >= 0) {
	GifWarning("too much input data, ignoring extra...");
    }
    LzwTerm(&dc);
    return picture;
}

static int
GifImageData(
    Blt_DataSink data, 
    Gif *gifPtr, 
    Blt_Picture *picturePtr, 
    int skip)
{
    /* Table based image data. */

    /* Get LZW minimum code size. */
    gifPtr->lzwMinimumCodeSize = Blt_SinkNextByte(data);

    /* Get Image data */
    if (skip) {
	unsigned int length;

	while ((length = Blt_SinkNextByte(data)) > 0) {
	    if (Blt_SinkBytesLeft(data) < length) {
		return TCL_ERROR;
	    }
	    Blt_SinkIncrCursor(data, length); /* Skip blocks */
	}
    } else {
	/* Read and decode image blocks. */
	*picturePtr = GifCreatePictureFromData(data, gifPtr);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_GifToPicture --
 *
 *      Reads a GIF file and converts it into a picture.
 *
 * Results:
 *      The picture is returned.  If an error occured, such
 *	as the designated file could not be opened, NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Blt_Picture 
Blt_GifToPicture(
    Tcl_Interp *interp, 
    char *fileName,
    Blt_DataSink data,
    int imageIndex)
{
    Gif gif;
    Blt_Picture picture;
    GifMessage message;

    gifMessagePtr = &message;
    memset(&gif, 0, sizeof(gif)); /* Clear the structure. */
    
    message.nWarnings = 0;
    Tcl_DStringInit(&message.errors);
    Tcl_DStringInit(&message.warnings);
    Tcl_DStringAppend(&message.errors, "error reading \"", -1);
    Tcl_DStringAppend(&message.errors, fileName, -1);
    Tcl_DStringAppend(&message.errors, "\": ", -1);

    if (setjmp(message.jmpbuf)) {
	Tcl_DStringResult(interp, &message.errors);
	Tcl_DStringFree(&message.warnings);
	return NULL;
    }

    picture = NULL;
    if (!GifHeader(data, &gif)) {
	GifError("bad GIF header");
    }
    if (GifLogicalScreenDescriptor(data, &gif) != TCL_OK) {
	GifError("bad GIF logical screen descriptor");
    }
    if (gif.globalColorTableFlag) {
	if (GifGlobalColorTable(data, &gif) != TCL_OK) {
	    GifError("bad GIF global color table");
	}
    }
    for(;;) {
	int byte;

	byte = Blt_SinkNextByte(data);
	if (byte == '!') {
	    if (GifExtensions(data, &gif) != TCL_OK) {
		GifError("bad GIF extensions block");
	    }
	} else if (byte == ',') {
	    if (GifImageDescriptor(data, &gif) != TCL_OK) {
		GifError("bad GIF image descriptor");
	    }
	    if (gif.localColorTableFlag) {
		if (GifLocalColorTable(data, &gif) != TCL_OK) {
		    GifError("bad GIF local color table");
		}
	    }
	    imageIndex--;
	    if (GifImageData(data, &gif, &picture, imageIndex != 0) 
		!= TCL_OK) {
		GifError("bad GIF image data");
	    } 
	} else if (byte == ';') {
	    break;		/* Done. */
	}
    }
    if (message.nWarnings > 0) {
	Tcl_SetErrorCode(interp, "PICTURE", "GIF_READ_WARNINGS", 
		Tcl_DStringValue(&message.warnings), (char *)NULL);
    }
    Tcl_DStringFree(&message.warnings);
    Tcl_DStringFree(&message.errors);
    return picture;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_IsGif --
 *
 *      Attempts to parse a JPG file header.
 *
 * Results:
 *      Returns 1 is the header is GIF and 0 otherwise.  Note that
 *      the validity of the header contents is not checked here.  That's
 *	done in Blt_GifToPicture.
 *
 *----------------------------------------------------------------------
 */
int
Blt_IsGif(Blt_DataSink data)
{
    Gif gif;
    int bool;

    bool = GifHeader(data, &gif);
    return bool;
}
