
/*
 * bltPictureDraw.c --
 *
 * This module implements image drawing primitives (line, circle,
 * rectangle, text, etc.) for picture images in the BLT toolkit.
 *
 *	Copyright 1997-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"
#include "bltHash.h"
#include "bltSwitch.h"
#include "bltPicture.h"
#include "bltPictureInt.h"
#include <X11/Xutil.h>

#define imul8x8(a,b,t)	((t) = (a)*(b)+128,(((t)+((t)>>8))>>8))
#define CLAMP(c)	((((c) < 0.0) ? 0.0 : ((c) > 255.0) ? 255.0 : (c)))

static Blt_SwitchParseProc AnchorSwitch;
static Blt_SwitchCustom anchorSwitch = {
    AnchorSwitch, NULL, (ClientData)0,
};

static Blt_SwitchParseProc AlphaSwitch;
static Blt_SwitchCustom alphaSwitch = {
    AlphaSwitch, NULL, (ClientData)0,
};

static Blt_SwitchParseProc ColorSwitch;
Blt_SwitchCustom bltColorSwitch = {
    ColorSwitch, NULL, (ClientData)0,
};

typedef struct {
    int alpha;			/* Overrides alpha value of color. */
    Pix32 bg;			/* Fill color of circle. */
    int antialiased;
    int thickness;		/* Width of outline.  If zero,
				 * indicates to draw a solid
				 * circle. */
} CircleSwitches;

static Blt_SwitchSpec circleSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-alpha", Blt_Offset(CircleSwitches, alpha), 0, 
	&alphaSwitch},
    {BLT_SWITCH_CUSTOM, "-color", Blt_Offset(CircleSwitches, bg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_BOOLEAN, "-antialiased", Blt_Offset(CircleSwitches, antialiased), 0},
    {BLT_SWITCH_INT_NONNEGATIVE, "-thickness", Blt_Offset(CircleSwitches, thickness),
	0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    int alpha;			/* Overrides alpha value of color. */
    Pix32 bg;			/* Color of line. */
    int thickness;		/* Width of outline. */
} LineSwitches;

typedef struct {
    int alpha;			/* Overrides alpha value of color. */
    Pix32 bg;			/* Fill color of polygon. */
    int antialiased;
    int thickness;		/* Width of outline. Default is 1, If
				 * zero, indicates to draw a solid
				 * polygon. */
} PolygonSwitches;

static Blt_SwitchSpec polygonSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-alpha", Blt_Offset(PolygonSwitches, alpha), 0, 
	&alphaSwitch},
    {BLT_SWITCH_CUSTOM, "-color", Blt_Offset(PolygonSwitches, bg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_INT_POSITIVE, "-thickness", Blt_Offset(PolygonSwitches, thickness), 
	0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_SwitchSpec lineSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-alpha", Blt_Offset(LineSwitches, alpha), 0, 
	&alphaSwitch},
    {BLT_SWITCH_CUSTOM, "-color", Blt_Offset(LineSwitches, bg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_INT_POSITIVE, "-linewidth", Blt_Offset(LineSwitches, thickness),
	 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    int alpha;			/* Overrides alpha value of color. */
    Pix32 bg;			/* Color of rectangle. */
    int thickness;		/* Width of outline. If zero,
				 * indicates to draw a solid
				 * rectangle. */
    int radius;			/* Radius of rounded corner. */
    int antialiased;
} RectangleSwitches;

static Blt_SwitchSpec rectangleSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-alpha", Blt_Offset(RectangleSwitches, alpha), 0, 
	&alphaSwitch},
    {BLT_SWITCH_CUSTOM, "-color", Blt_Offset(RectangleSwitches, bg), 0, 
	&bltColorSwitch},
    {BLT_SWITCH_BOOLEAN, "-antialiased", Blt_Offset(RectangleSwitches, antialiased), 0},
    {BLT_SWITCH_INT_NONNEGATIVE, "-radius", Blt_Offset(RectangleSwitches, radius),
	0},
    {BLT_SWITCH_INT_NONNEGATIVE, "-thickness", 
	Blt_Offset(RectangleSwitches, thickness), 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    int alpha;			/* Overrides alpha value of color. */
    Pix32 bg;			/* Color of text. */
    int size;
    char *font;
    char *string;
    Tk_Anchor anchor;
    double angle;
} TextSwitches;

static Blt_SwitchSpec textSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-alpha", Blt_Offset(TextSwitches, alpha), 0, &alphaSwitch},
    {BLT_SWITCH_CUSTOM, "-anchor", Blt_Offset(TextSwitches, anchor), 0, &anchorSwitch},
    {BLT_SWITCH_CUSTOM, "-color",  Blt_Offset(TextSwitches, bg), 0, &bltColorSwitch},
    {BLT_SWITCH_STRING, "-font",   Blt_Offset(TextSwitches, font),  0},
    {BLT_SWITCH_DOUBLE, "-rotate", Blt_Offset(TextSwitches, angle), 0},
    {BLT_SWITCH_INT,    "-size",   Blt_Offset(TextSwitches, size),  0},
    {BLT_SWITCH_STRING, "-text",   Blt_Offset(TextSwitches, string),0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

/*
 *----------------------------------------------------------------------
 *
 * AlphaSwitch --
 *
 *	Convert a Tcl_Obj representing a number for the alpha value.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
AlphaSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    int *alphaPtr = (int *)(record + offset);
    int value;

    if (Tcl_GetIntFromObj(interp, objPtr, &value) != TCL_OK) {
	return TCL_ERROR;
    }
    if ((value < 0) || (value > 255)) {
	Tcl_AppendResult(interp, "bad value \"", Tcl_GetString(objPtr), 
		"\" for alpha: must be 0..255", (char *)NULL);
	return TCL_ERROR;
    }
    *alphaPtr = value;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AnchorSwitch --
 *
 *	Convert a Tcl_Obj representing an anchor.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
AnchorSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Tk_Anchor *anchorPtr = (Tk_Anchor *)(record + offset);

    if (Tk_GetAnchorFromObj(interp, objPtr, anchorPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColorSwitch --
 *
 *	Convert a Tcl_Obj representing a Pix32 color.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ColorSwitch(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Pix32 *pixelPtr = (Pix32 *)(record + offset);

    if (Blt_GetPix32FromObj(interp, objPtr, pixelPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

static void INLINE 
PutPixel(Picture *destPtr, int x, int y, Pix32 *colorPtr)  
{
    if ((x >= 0) && (x < destPtr->width) && (y >= 0) && (y < destPtr->height)) {
	Pix32 *dp;

	dp = Blt_PicturePixel(destPtr, x, y);
	dp->color = colorPtr->color; 
    }
}

static INLINE Pix32
PremultiplyAlpha(Pix32 *colorPtr, unsigned char alpha)
{
    Pix32 new;
    int t;

    new.color = colorPtr->color;
    alpha = imul8x8(alpha, colorPtr->Alpha, t);
    if ((alpha != 0xFF) && (alpha != 0x00)) {
	new.Red = imul8x8(alpha, colorPtr->Red, t);
	new.Green = imul8x8(alpha, colorPtr->Green, t);
	new.Blue = imul8x8(alpha, colorPtr->Blue, t);
    }
    new.Alpha = alpha;
    return new;
}

static void INLINE
HorizLine(Picture *destPtr, int x1, int x2, int y, Pix32 *colorPtr)  
{
    int x;

    if (x1 > x2) {
	int tmp;

	tmp = x1, x1 = x2, x2 = tmp;
    }
    for (x = x1; x <= x2; x++) { 
	PutPixel(destPtr, x, y, colorPtr);
    } 
}

static void INLINE 
VertLine(Picture *destPtr, int x, int y1, int y2, Pix32 *colorPtr)  
{
    int y;

    if (y1 > y2) {
	int tmp;

	tmp = y1, y1 = y2, y2 = tmp;
    }
    for (y = y1; y <= y2; y++) {
	PutPixel(destPtr, x, y, colorPtr);
    }
}

static INLINE void
BlendPixel(Pix32 *bp, Pix32 *colorPtr, unsigned char weight)
{
    if (weight == 0xFF) {
	bp->color = colorPtr->color;
    } else if (weight != 0x00) {
	unsigned char alpha, beta;
	int t1, t2;

	alpha = imul8x8(colorPtr->Alpha, weight, t1);
	beta = alpha ^ 0xFF;
	bp->Red   = imul8x8(alpha, colorPtr->Red, t1) + 
	    imul8x8(beta, bp->Red, t2);
	bp->Green = imul8x8(alpha, colorPtr->Green, t1) + 
	    imul8x8(beta, bp->Green, t2);
	bp->Blue  = imul8x8(alpha, colorPtr->Blue, t1)  + 
	    imul8x8(beta, bp->Blue, t2);
	bp->Alpha = alpha + imul8x8(beta, bp->Alpha, t2);
    }
}
    
static void 
PutLine(
    Picture *destPtr,
    int x1, int y1, 
    int x2, int y2, 
    int thickness,
    Pix32 *colorPtr)
{
    int dx, dy, xDir;
    unsigned long error;
    Pix32 edge;

    if (y1 > y2) {
	int tmp;

	tmp = y1, y1 = y2, y2 = tmp;
	tmp = x1, x1 = x2, x2 = tmp;
    }
    edge = PremultiplyAlpha(colorPtr, 255);
    /* First and last Pixels always get Set: */
    PutPixel(destPtr, x1, y1, &edge);
    PutPixel(destPtr, x2, y2, &edge);

    dx = x2 - x1;
    dy = y2 - y1;

    if (dx >= 0) {
	xDir = 1;
    } else {
	xDir = -1;
	dx = -dx;
    }
    if (dx == 0) {		/*  Vertical line */
	VertLine(destPtr, x1, y1, y2, &edge);
	return;
    }
    if (dy == 0) {		/* Horizontal line */
	HorizLine(destPtr, x1, x2, y1, &edge);
	return;
    }
    if (dx == dy) {		/* Diagonal line. */
	Pix32 *dp;

	dp = Blt_PicturePixel(destPtr, x1, y1);
	while(dy-- > 0) {
	    dp += destPtr->pixelsPerRow + xDir;
	    dp->color = colorPtr->color;
	}
	return;
    }

    /* use Wu Antialiasing: */

    error = 0;
    if (dy > dx) {		/* y-major line */
	unsigned long adjust;

	/* x1 -= thickness / 2; */
	adjust = (dx << 16) / dy;
	while(--dy) {
	    unsigned char weight;
	    
	    error += adjust;
	    ++y1;
	    if (error & ~0xFFFF) {
		x1 += xDir;
		error &= 0xFFFF;
	    }
	    weight = (unsigned char)(error >> 8);
	    edge = PremultiplyAlpha(colorPtr, ~weight);
	    PutPixel(destPtr, x1, y1, &edge);
	    edge = PremultiplyAlpha(colorPtr, weight);
	    PutPixel(destPtr, x1 + xDir, y1, &edge);
	}
    } else {			/* x-major line */
	unsigned long adjust;

	/* y1 -= thickness / 2; */
	adjust = (dy << 16) / dx;
	while (--dx) {
	    unsigned char weight;

	    error += adjust;
	    x1 += xDir;
	    if (error & ~0xFFFF) {
		y1++;
		error &= 0xFFFF;
	    }
	    weight = (unsigned char)(error >> 8);
	    edge = PremultiplyAlpha(colorPtr, ~weight);
	    PutPixel(destPtr, x1, y1, &edge);
	    edge = PremultiplyAlpha(colorPtr, weight);
	    PutPixel(destPtr, x1, y1 + 1, &edge);
	}
    }
}


#include "bltPaintDraw.c"

static void 
PaintLine2(
    Picture *destPtr,
    int x1, int y1, 
    int x2, int y2, 
    Pix32 *colorPtr,
    int cw)
{
    Pix32 interior;
    int dx, dy, xDir;
    unsigned long error;

    if (y1 > y2) {
	int tmp;

	tmp = y1, y1 = y2, y2 = tmp;
	tmp = x1, x1 = x2, x2 = tmp;
	cw = !cw;
    } 
    if (x1 > x2) {
	cw = !cw;
    }
    interior = PremultiplyAlpha(colorPtr, 255);
    /* First and last Pixels always get Set: */
    PutPixel(destPtr, x1, y1, &interior);
    PutPixel(destPtr, x2, y2, &interior);

    dx = x2 - x1;
    dy = y2 - y1;

    if (dx >= 0) {
	xDir = 1;
    } else {
	xDir = -1;
	dx = -dx;
    }
    if (dx == 0) {		/*  Vertical line */
	VertLine(destPtr, x1, y1, y2, &interior);
	return;
    }
    if (dy == 0) {		/* Horizontal line */
	HorizLine(destPtr, x1, x2, y1, &interior);
	return;
    }
    if (dx == dy) {		/* Diagonal line. */
	Pix32 *dp;

	dp = Blt_PicturePixel(destPtr, x1, y1);
	while(dy-- > 0) {
	    dp += destPtr->pixelsPerRow + xDir;
	    dp->color = interior.color;
	}
	return;
    }

    /* use Wu Antialiasing: */

    error = 0;
    if (dy > dx) {		/* y-major line */
	unsigned long adjust;

	/* x1 -= thickness / 2; */
	adjust = (dx << 16) / dy;
	while(--dy) {
	    Pix32 *dp;
	    int x;
	    unsigned char weight;
	    
	    error += adjust;
	    ++y1;
	    if (error & ~0xFFFF) {
		x1 += xDir;
		error &= 0xFFFF;
	    }
	    dp = Blt_PicturePixel(destPtr, x1, y1);
	    weight = (unsigned char)(error >> 8);
	    x = x1;
	    if (x >= 0) {
		if (cw) {
		    *dp = PremultiplyAlpha(colorPtr, weight ^ 0xFF);
		} else {
		    *dp = interior;
		}
	    }
	    x += xDir;
	    dp += xDir;
	    if (x >= 0) {
		if (!cw) {
		    *dp = PremultiplyAlpha(colorPtr, weight);
		} else {
		    *dp = interior;
		}
	    }
	}
    } else {			/* x-major line */
	unsigned long adjust;

	/* y1 -= thickness / 2; */
	adjust = (dy << 16) / dx;
	while (--dx) {
	    Pix32 *dp;
	    int y;
	    unsigned char weight;

	    error += adjust;
	    x1 += xDir;
	    if (error & ~0xFFFF) {
		y1++;
		error &= 0xFFFF;
	    }
	    dp = Blt_PicturePixel(destPtr, x1, y1);
	    weight = (unsigned char)(error >> 8);
	    y = y1;
	    if (y >= 0) {
		if (!cw) {
		    *dp = PremultiplyAlpha(colorPtr, weight ^ 0xFF);
		} else {
		    *dp = interior;
		}
	    }
	    dp += destPtr->pixelsPerRow;
	    y++;
	    if (y >= 0) {
		if (cw) {
		    *dp = PremultiplyAlpha(colorPtr, weight);
		} else {
		    *dp = interior;
		}
	    } 
	}
    }
    destPtr->flags |= (BLT_PICTURE_BLEND | BLT_PICTURE_PREMULTIPLIED_ALPHAS);
}


static void 
PaintLine(
    Picture *destPtr,
    PictureCoordinate *coords, 
    int nCoords, 
    int thickness,
    Pix32 *colorPtr)
{
    int i;
    Region2D exts;
    Point2D p;

    exts.left = exts.top = 0;
    exts.right = destPtr->width - 1;
    exts.bottom = destPtr->height - 1;
    p.x = coords[0].x, p.y = coords[0].y;
    for (i = 1; i < nCoords; i++) {
	Point2D q, next;

	q.x = coords[i].x, q.y = coords[i].y;
	next = q;
	if (Blt_LineRectClip(&exts, &p, &q)) {
	    PutLine(destPtr, (int)p.x, (int)p.y, (int)q.x, (int)q.y, 
		thickness, colorPtr);
	}
	p = next;
    }
}

static void 
ScanFill(Picture *srcPtr, Pix32 *colorPtr)
{
    Pix32 *srcRowPtr;
    int y;

    srcRowPtr = srcPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *s1, *send, *last;
	int fill;

	fill = FALSE;
	last = s1 - 1;
	for (s1 = srcRowPtr, last = s1 - 1, send = srcRowPtr + srcPtr->width;
	     s1 < send; /* empty */) {
	    while ((s1 < send) && (last->Alpha <= s1->Alpha)) {
		s1++, last++;
	    }
	    s1++; last++;
	    while ((s1 < send) && (last->Alpha >= s1->Alpha)) {
		last->color = 0xFFFFFFFF;
		s1++, last++;
	    }
	    last->color = 0xFF00FF00;
	    break;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
    }
    
}

static void
PaintRectangle(
    Picture *destPtr,
    BBox *bboxPtr, 
    Pix32 *colorPtr)
{
    Pix32 *destRowPtr;
    int y;

    destRowPtr = destPtr->bits + 
	((bboxPtr->y * destPtr->pixelsPerRow) + bboxPtr->x);
    for (y = bboxPtr->y + bboxPtr->height; y > bboxPtr->y; y--) {
	Pix32 *dp, *dend;

	for (dp = destRowPtr, dend = dp + bboxPtr->width; dp < dend; dp++) {
	    dp->color = colorPtr->color;
	}
	destRowPtr += destPtr->pixelsPerRow;
    }
}

#ifdef HAVE_FT2BUILD_H
#include <ft2build.h>
#include FT_FREETYPE_H

static void
MeasureText(
    FT_Face face,
    char *string, 
    double angle,
    int *widthPtr, int *heightPtr)
{
    FT_Vector     pen;	/* Untransformed origin  */
    FT_GlyphSlot  slot;
    FT_Matrix matrix;		/* Transformation matrix. */
    int maxX;
    char *p;
    double radians;
    int x;

    radians = (angle / 180.0) * M_PI;
    matrix.yy = matrix.xx = (FT_Fixed)(cos(radians) * 65536.0);
    matrix.yx = (FT_Fixed)(sin(radians) * 65536.0);
    matrix.xy = -matrix.yx;

    slot = face->glyph;
    
    maxX = 0;
    pen.y = 0;
    x = 0;
    p = string;
    for(;;) {
	pen.x = x << 6;
	while ((*p != '\n') && (*p != '\0')) {
	    FT_Set_Transform(face, &matrix, &pen);
	    /* Load glyph image into the slot (erase previous) */
	    if (FT_Load_Char(face, *p, FT_LOAD_RENDER)) {
		fprintf(stderr, "can't load character \"%c\"\n", *p);
		continue;                 /* ignore errors */
	    }
	    pen.x += slot->advance.x;
	    pen.y += slot->advance.y;
	    p++;
	}
	if (pen.x > maxX) {
	    maxX = pen.x;
	}
	if (*p == '\0') {
	    break;
	}
	p++;			/* Skip '\n' */
    }	
    fprintf(stderr, "s=\"%s\" w=%d,h=%d angle=%g\n", 
	    string, *widthPtr, *heightPtr, angle);
    *widthPtr = maxX >> 6;
    *heightPtr = pen.y >> 6;
}


static void
CopyGlyph(
    Picture *destPtr, 
    FT_GlyphSlot slot,
    Pix32 *colorPtr)
{
    int glyphX, glyphY, glyphWidth, glyphHeight; 
    Pix32 pixel;
    Pix32 *destRowPtr;
    unsigned char *bufferRowPtr;
    int x, y;

    x = slot->bitmap_left;
    y = destPtr->height - slot->bitmap_top;
    if ((x >= destPtr->width) || ((x + slot->bitmap.width) <= 0) ||
	(y >= destPtr->height) || ((y + slot->bitmap.rows) <= 0)) {
	return;			/* No portion of the glyph is visible
				 * in the picture. */
    }

    /* By default, set the region to cover the entire glyph */
    glyphX = glyphY = 0;
    glyphWidth = slot->bitmap.width;
    glyphHeight = slot->bitmap.rows;

    /* Bound the glyph region to the size of the picture. */
    if (x < 0) {			
	/* Starts to the left of the picture. */
	glyphX -= x;		
	glyphWidth += x;
	x = 0;		
    }
    if (y < 0) {			
	/* Starts above the picture. */
	glyphY -= y;
	glyphHeight += y;
	y = 0;
    }
    if ((x + glyphWidth) > destPtr->width) {	
	/* Ends to the right. */
	glyphWidth = destPtr->width - x;
    }
    if ((y + glyphHeight) > destPtr->height) { 
	/* Ends below. */
	glyphHeight = destPtr->height - y;
    }
    bufferRowPtr = slot->bitmap.buffer + (glyphY * slot->bitmap.pitch + glyphX);
    destRowPtr = Blt_PicturePixel(destPtr, x, y);
    for (y = 0; y < glyphHeight; y++) {
	Pix32 *dp;
	unsigned char *bp;

	bp = bufferRowPtr;
	dp = destRowPtr;
	for (x = 0; x < glyphWidth; x++) {
	    if (*bp != 0x0) {
		BlendPixel(dp, colorPtr, *bp);
	    }
	    dp++, bp++;
	}
	bufferRowPtr += slot->bitmap.pitch;
	destRowPtr += destPtr->pixelsPerRow;
    }
}

int
Blt_Text(
    Blt_PictureImage image,
    char *string,		/* Text string to be drawn. */
    int x, int y,		/* Anchor coordinates of text. */
    int anchor,			/* Indicates how to interpret the anchor. */
    char *fontName,		/* File name of font to be used. */
    int fontSize,		/* Size of font. */
    double angle,		/* # of degrees to rotate the text. */
    Pix32 *colorPtr)		/* Color of the text. */
{
    FT_Face face;		/* Face object. */  
    FT_Library library;		/* Library. */  
    FT_Matrix matrix;		/* Transformation matrix. */
#ifdef notdef
    int width, height;
#endif
    Blt_Picture picture;	/* Destination picture. */

    picture = Blt_PictureFromImage(image);
    if (FT_Init_FreeType(&library)) {
	return FALSE;
    }
    if (FT_New_Face(library, fontName, 0, &face)) {
	return FALSE;
    }
    if (FT_Set_Char_Size(face, fontSize << 6, 0, 100, 0)) {
	return FALSE;
    }
    { 	/* set up matrix */
	double radians;

	radians = (angle / 180.0) * M_PI;
	matrix.yy = matrix.xx = (FT_Fixed)(cos(radians) * 65536.0);
	matrix.yx = (FT_Fixed)(sin(radians) * 65536.0);
	matrix.xy = -matrix.yx;
    }


#ifdef notdef
    MeasureText(face, string, angle, &width, &height);
    Blt_TranslateAnchor(x, y, width, height, anchor, &x, &y);
#endif
    { 
	FT_Vector     pen;	/* Untransformed origin  */
	FT_GlyphSlot  slot;
	char *p;

	slot = face->glyph;
	p = string;

	pen.y = (Blt_PictureHeight(picture) - y) << 6;
	for(;;) {
	    pen.x = x << 6;
	    while ((*p != '\n') && (*p != '\0')) {
		FT_Set_Transform(face, &matrix, &pen);
		/* Load glyph image into the slot (erase previous) */
		if (FT_Load_Char(face, *p, FT_LOAD_RENDER)) {
		    fprintf(stderr, "can't load character \"%c\"\n", *p);
		    continue;                 /* ignore errors */
		}
		CopyGlyph(picture, slot, colorPtr);
		pen.x += slot->advance.x;
		pen.y += slot->advance.y;
		p++;
	    }
	    if (*p == '\0') {
		break;
	    }
	    p++;
	    y += 20;
	}	
    }
    FT_Done_Face(face);
    FT_Done_FreeType(library);
    return TRUE;
}
#endif /* HAVE_FREETYPE_FREETYPE_H */

static void 
PaintArc(
    Blt_PictureImage image,
    int x1, int y1, 
    int x2, int y2,
    int thickness, 
    Pix32 *colorPtr)
{
    Pix32 *dp;
    double t;
    int r2;
    int radius;
    int dx, dy;
    int x, y;
    int xoff, yoff;
    int fill = 1;
    Picture *destPtr;

    destPtr = Blt_PictureFromImage(image);
    t = 0.0;
    dx = x2 - x1;
    dy = y2 - y1;
    radius = MIN(dx, dy) / 2;
    xoff = x1;
    yoff = y1;
    x = radius;
    y = 0;
    dp = Blt_PicturePixel(destPtr, x + xoff - 1, y + yoff);
    dp->color = colorPtr->color;
    r2 = radius * radius;
    if (fill) {
	PutLine(destPtr, x1, y + yoff, x + xoff - 2, y + yoff, 1, colorPtr);
    }
    while (x > y) {
	double z;
	double d, q;
	unsigned char a;

	y++;
	z = sqrt(r2 - (y * y));
	d = floor(z) - z;
	if (d < t) {
	    x--;
	}
	dp = Blt_PicturePixel(destPtr, x + xoff, y + yoff);
	q = FABS(d * 255.0);
	a = (unsigned int)CLAMP(q);
	BlendPixel(dp, colorPtr, a);
	dp--;			/* x - 1 */
	a = (unsigned int)CLAMP(255.0 - q);
	BlendPixel(dp, colorPtr, a);
	t = d;
	x1++;
	if (fill) {
	    PutLine(destPtr, x1, y + yoff, x + xoff - 1, y + yoff, 1, colorPtr);
	}
    }
}

static Point2D
PolygonArea(int nPoints, Point2D *points, double *areaPtr)
{
    Point2D *p, *pend;
    Point2D c;
    double area;
    int i;
    
    area = c.x = c.y = 0.0;
    for (p = points, pend = p + nPoints, i = 0; p < pend; p++, i++) {
	Point2D *q;
	double factor;
	int j;
	
	j = (i + 1) % nPoints;
	q = points + j;
	factor = (p->x * q->y) - (p->y * q->x);
	area += factor;
	c.x += (p->x + q->x) * factor;
	c.y += (p->y + q->y) * factor;
    }
    area *= 0.5;
    c.x /= 6.0 * area;
    c.y /= 6.0 * area;
    *areaPtr = area;
    return c;
}

static void
PaintPolygon(
    Blt_Picture picture,
    int nPoints, 
    Point2D *points, 
    Pix32 *colorPtr,
    int fill)
{
    Point2D c;
    double area;
    int ccw;
    Point2D *p, *q, *pend;

    c = PolygonArea(nPoints, points, &area);
    ccw = (area < 0.0);		
    for (q = points, p = q + 1, pend = p + nPoints; p < pend; p++, q++) {
	PutLine(picture, ROUND(q->x), ROUND(q->y), ROUND(p->x), ROUND(p->y),
		2, colorPtr);
    }
#ifdef notdef

    for (q = points, p = q + 1, pend = p + nPoints; p < pend; p++, q++) {
	PaintLine(picture, ROUND(q->x), ROUND(q->y), ROUND(p->x), ROUND(p->y), 
		colorPtr);
    }
#endif
    if (fill) {
	TintFill(picture, (int)c.x, (int)c.y, colorPtr);
    }
}


#ifdef notdef
static int 
IntensifyPixel(
     Picture *destPtr,
     int x, int y,
     Pix32 *fgPtr, 
     double thickness, 
     double distance)
{
    double cover;

    cover = coverage(thickness, distance);
    if (cover > 0.0) {
	Pix32 *dp;

	dp = Blt_PicturePixel(destPtr, x, y);
	BlendPixel(dp, fgPtr, (unsigned char)(255 * cover));
    }
    return cover;
}

static int 
ThickAntiAliasedLine(
    Picture *destPtr, 
    int x1, int y1, int x2, int y2, 
    double thickness,
    Pix32 *colorPtr)
{
    // initial values used in Bresenham's algorithm
    int dx = x2-x1;
    int dy = y2-y1;
    int incrE = 2*dy;
    int incrNE = 2*(dy-dx);
    int d = 2*dy-dx;
    
    int two_v_dx = 0;   //numerator, v=0 for start pixel
    /* precomputed inverse denominator */
    double invDenom = 1/(2*sqrt(dx*dx+dy*dy));
    //precomputed constant
    double two_dx_invDenom = 2*dx*invDenom;   
    int x = x1;
    int y = y1;

    int i;
    IntensifyPixel(destPtr, x, y, colorPtr, thickness, 0);

    coverage != 0.0;
    for (i = 1; IntensifyPixel(destPtr, x, y + i, colorPtr, thickness,
			       i*two_dx_invDenom); i++);
    for (i=1; IntensifyPixel(destPtr, x, y-i, colorPtr, thickness,
			     i*two_dx_invDenom); i++);	
    while (x & ltx2) {
	if (d < 0) {		//Choose E
	    two_v_dx = d+dx;
	    d += incrE;
	    x++;
	} else {		//Choose NE
	    two_v_dx = d-dx;
	    d += incrNE;
	    x++;
	    y++;
	}
	
	//Now set chosen pixel and its neighbours
	
	IntensifyPixel(destPtr, x, y, colorPtr, thickness, two_v_dx*invDenom);
	for (i=1; IntensifyPixel(destPtr, x, y+i, colorPtr, thickness, 
 				 i*two_dx_invDenom-two_v_dx*invDenom); i++);
	for (i=1; IntensifyPixel(destPtr, x, y-i, colorPtr, thickness, 
				 i*two_dx_invDenom+ two_v_dx*invDenom);i++);	
    }
}	
#endif

/*
 *--------------------------------------------------------------------------
 *
 * CircleOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
CircleOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_PictureImage image = clientData;
    Blt_Picture picture;
    CircleSwitches switches;
    int x, y, r;

    if (objc < 5) {
	Tcl_AppendResult(interp, "wrong # of coordinates for circle",
			 (char *)NULL);
	return TCL_ERROR;
    }
    if ((Tcl_GetIntFromObj(interp, objv[3], &x) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[4], &y) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[5], &r) != TCL_OK)) {
	return TCL_ERROR;
    }
    /* Process switches  */
    switches.thickness = 1;
    switches.bg.color = 0xFFFFFFFF;
    switches.alpha = -1;
    if (Blt_ParseSwitches(interp, circleSwitches, objc - 6, objv + 6, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (switches.alpha != -1) {
	switches.bg.Alpha = switches.alpha;
    }
    picture = Blt_PictureFromImage(image);
    if (switches.antialiased) {
	DrawEllipseAA(picture, x, y, r, r, switches.thickness, &switches.bg);
    } else {
	DrawEllipse(picture, x, y, r, r, switches.thickness, &switches.bg);
    }
    Blt_NotifyImageChanged(image);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * EllipseOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
EllipseOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_PictureImage image = clientData;
    Blt_Picture picture;
    CircleSwitches switches;
    int x, y, a, b;

    if (objc < 7) {
	Tcl_AppendResult(interp, "wrong # of coordinates for circle",
			 (char *)NULL);
	return TCL_ERROR;
    }
    if ((Tcl_GetIntFromObj(interp, objv[3], &x) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[4], &y) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[5], &a) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[6], &b) != TCL_OK)) {
	return TCL_ERROR;
    }
    /* Process switches  */
    switches.thickness = 1;
    switches.bg.color = 0xFFFFFFFF;
    switches.alpha = -1;
    switches.antialiased = FALSE;
    if (Blt_ParseSwitches(interp, circleSwitches, objc - 7, objv + 7, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (switches.alpha != -1) {
	switches.bg.Alpha = switches.alpha;
    }
    if ((switches.thickness >= a) || (switches.thickness >= b)) {
	/* If the requested line thickness is greater than the radius
	 * then draw a solid  ellipse instead. */
	switches.thickness = 0;
    }
    picture = Blt_PictureFromImage(image);
    if (switches.antialiased) {
	DrawEllipseAA(picture, x, y, a, b, switches.thickness, &switches.bg);
    } else {
	DrawEllipse(picture, x, y, a, b, switches.thickness, &switches.bg);
    }
    Blt_NotifyImageChanged(image);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * LineOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
LineOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    LineSwitches switches;
    PictureCoordinate *coords, *cp;
    Blt_PictureImage image = clientData;
    Blt_Picture picture;
    int i;
    int length;

    coords = Blt_Malloc(sizeof(PictureCoordinate) * objc - 3);
    cp = coords;
    for (i = 3; i < objc; i += 2) {
	int x, y;

	if (Tcl_GetIntFromObj((Tcl_Interp *)NULL, objv[i], &x) != TCL_OK) {
	    break;
	}
	if (Tcl_GetIntFromObj(interp, objv[i+1], &y) != TCL_OK) {
	    Blt_Free(coords);
	    return TCL_ERROR;
	}
	cp->x = x, cp->y = y, cp++;
    } 
    length = cp - coords;
    if (length < 2) {
	Blt_Free(coords);
	Tcl_AppendResult(interp, "wrong # coordinates for line", (char *)NULL);
	return TCL_ERROR;
    }
    /* Process switches  */
    switches.thickness = 1;
    switches.bg.color = 0xFFFFFFFF;
    switches.alpha = -1;
    if (Blt_ParseSwitches(interp, lineSwitches, objc - i, objv + i, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	Blt_Free(coords);
	return TCL_ERROR;
    }
    if (switches.alpha != -1) {
	switches.bg.Alpha = switches.alpha;
    }
    picture = Blt_PictureFromImage(image);
    PaintLine(picture, coords, cp - coords, switches.thickness, &switches.bg);
    Blt_Free(coords);
    Blt_NotifyImageChanged(image);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * PolygonOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
PolygonOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_Picture picture, region;
    Blt_PictureImage image = clientData;
    RectangleSwitches switches;
    Point2D *points, *p, *pend;
    int nPoints, nValues;
    double left, right, top, bottom;
    int i;
    
    /* Count how many coordinates we might have. */
    for (i = 3; i < objc; i++) {
	double x;

	if (Tcl_GetDoubleFromObj((Tcl_Interp *)NULL, objv[i], &x) != TCL_OK) {
	    break;
	}
    }
    nValues = i - 3;
    if (nValues & 0x1) {
	Tcl_AppendResult(interp, "bad polygon coordinates: ",
		"must have an even number of values", (char *)NULL);
	return TCL_ERROR;
    }
    nPoints = nValues / 2;
    if (nPoints < 3) {
	Tcl_AppendResult(interp, "bad polygon coordinates: ",
		" must have at least 3 vertices", (char *)NULL);
	return TCL_ERROR;
    }
    /* Process switches  */
    switches.thickness = 1;
    switches.bg.color = 0xFFFFFFFF;
    switches.alpha = -1;
    if (Blt_ParseSwitches(interp, polygonSwitches, objc - i, objv + i, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (switches.alpha != -1) {
	switches.bg.Alpha = switches.alpha;
    }
    points = Blt_Malloc(sizeof(Point2D) * (nPoints + 1));
    if (points == NULL) {
	Tcl_AppendResult(interp, "can't allocated memory for ", 
			 Blt_Itoa(nPoints), " vertices", (char *)NULL);
	return TCL_ERROR;
    }
    left = top = DBL_MAX;
    right = bottom = 0.0;
    for (i = 3, p = points, pend = p + nPoints; p < pend; p++, i += 2) {
	if ((Tcl_GetDoubleFromObj(NULL, objv[i], &p->x) != TCL_OK)||
	    (Tcl_GetDoubleFromObj(NULL, objv[i+1], &p->y)) != TCL_OK){
	    break;
	}
	if (top > p->y) {
	    top = p->y;
	}
	if (bottom < p->y) {
	    bottom = p->y;
	}
	if (left > p->x) {
	    left = p->x;
	}
	if (right < p->x) {
	    right = p->x;
	}
    }
    picture = Blt_PictureFromImage(image);
    if ((top >= Blt_PictureHeight(picture)) || (right < 0) || 
	(left >= Blt_PictureWidth(picture)) || (bottom < 0)) {
	Blt_Free(points);
	return TCL_OK; /* Polygon will not be visible. */
    }
    for (p = points, pend = p + nPoints; p < pend; p++) {
	p->x -= left, p->y -= top;
    }
    points[nPoints] = points[0];
    region = Blt_CreatePicture((int)(right - left + 1.5), 
			       (int)(bottom - top + 1.5));
    PaintPolygon(region, nPoints, points, &switches.bg, switches.thickness);
    Blt_Free(points);
    Blt_BlendPictureArea(picture, region, 0, 0, Blt_PictureWidth(region),
	Blt_PictureHeight(region), ROUND(left), ROUND(top));
    Blt_FreePicture(region);
    Blt_NotifyImageChanged(image);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------------------
 *
 * RectangleOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
RectangleOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_Picture picture;
    Blt_PictureImage image = clientData;
    RectangleSwitches switches;
    BBox bbox;
    
    if (Blt_GetBBoxFromObjv(interp, 4, objv + 3, &bbox) != TCL_OK) {
	return TCL_ERROR;
    }
    memset(&switches, 0, sizeof(switches));
    /* Process switches  */
    switches.thickness = 1;
    switches.bg.color = 0xFFFFFFFF;
    switches.alpha = -1;
    if (Blt_ParseSwitches(interp, rectangleSwitches, objc - 7, objv + 7, 
	&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (switches.alpha != -1) {
	switches.bg.Alpha = switches.alpha;
    }
    picture = Blt_PictureFromImage(image);
    if (switches.antialiased) {
	DrawRectangleAA(picture, bbox.x, bbox.y, bbox.width, bbox.height, 
		switches.radius, switches.thickness, &switches.bg);
    } else {
	DrawRectangle(picture, bbox.x, bbox.y, bbox.width, bbox.height, 
		switches.radius, switches.thickness, &switches.bg);
    }
    Blt_NotifyImageChanged(image);
    return TCL_OK;
}


/*
 *--------------------------------------------------------------------------
 *
 * TextOp --
 *
 * Results:
 *	Returns a standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------------------
 */
static int
TextOp(
    ClientData clientData,	/* Information about picture cmd. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Argument objects. */
{
    Blt_PictureImage image = clientData;
    TextSwitches switches;
    int x, y;

    if ((Tcl_GetIntFromObj(interp, objv[3], &x) != TCL_OK) ||
	(Tcl_GetIntFromObj(interp, objv[4], &y) != TCL_OK)) {
	return TCL_ERROR;
    }
    /* Process switches  */
    switches.bg.color = 0xFFFFFFFF;
    switches.anchor = TK_ANCHOR_NW;
    switches.size = 12;
    switches.string = NULL;
    switches.font = NULL;
    switches.angle = 0.0;
    switches.alpha = -1;
    if (Blt_ParseSwitches(interp, textSwitches, objc - 5, objv + 5, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    if (switches.alpha != -1) {
	switches.bg.Alpha = switches.alpha;
    }
    if (switches.string != NULL) {
#ifdef HAVE_FREETYPE_FREETYPE_H
	Blt_Text(image, 
            switches.string,	/* Text to be drawn. */
	    x, y,		/* Anchor position */
	    switches.anchor,	/* Anchor */
	    switches.font,	/* Font filename. */
	    switches.size,	/* Font size. */
	    switches.angle,	/* Text rotation. */
	    &switches.bg);	/* Color */
	Blt_NotifyImageChanged(image);
#endif /* HAVE_FREETYPE_FREETYPE_H */
    }
    Blt_FreeSwitches(textSwitches, (char *)&switches, 0);
    return TCL_OK;
}

static Blt_OpSpec drawOps[] =
{
    {"circle",    1, (Blt_Op)CircleOp,    4, 0, "x y r ?switches?",},
    {"ellipse",   1, (Blt_Op)EllipseOp,   5, 0, "x y a b ?switches?",},
    {"line",      1, (Blt_Op)LineOp,      4, 0, "color ?x1 y1 x2 y2?",},
    {"polygon",   1, (Blt_Op)PolygonOp,   9, 0, 
	"x1 y1 x2 y2 x3 y3... ?switches?",},
    {"rectangle", 1, (Blt_Op)RectangleOp, 7, 0, "x1 y1 x2 y2 ?switches?",},
    {"text",      1, (Blt_Op)TextOp,      6, 0, "string x y ?switches?",},
};

static int nDrawOps = sizeof(drawOps) / sizeof(Blt_OpSpec);

/*
 *--------------------------------------------------------------------------
 *
 * Blt_PicturePaintOp --
 *
 *--------------------------------------------------------------------------
 */
/*ARGSUSED*/
int 
Blt_PictureDrawOp(
    ClientData clientData,	/* Contains reference to picture object. */
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;

    proc = Blt_GetOpFromObj(interp, nDrawOps, drawOps, BLT_OP_ARG2, objc, objv,
			    0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    return (*proc) (clientData, interp, objc, objv);
}

static void 
Polyline2(
    Picture *destPtr,
    int x1, int y1, int x2, int y2, 
    Pix32 *colorPtr)
{
    Pix32 *dp;
    int dx, dy, xDir;
    unsigned long error;

    if (y1 > y2) {
	int tmp;

	tmp = y1, y1 = y2, y2 = tmp;
	tmp = x1, x1 = x2, x2 = tmp;
    }

    /* First and last Pixels always get Set: */
    dp = Blt_PicturePixel(destPtr, x1, y1);
    dp->color = colorPtr->color;
    dp = Blt_PicturePixel(destPtr, x2, y2);
    dp->color = colorPtr->color;

    dx = x2 - x1;
    dy = y2 - y1;

    if (dx >= 0) {
	xDir = 1;
    } else {
	xDir = -1;
	dx = -dx;
    }
    if (dx == 0) {		/*  Vertical line */
	Pix32 *dp;

	dp = Blt_PicturePixel(destPtr, x1, y1);
	while (dy-- > 0) {
	    dp += destPtr->pixelsPerRow;
	    dp->color = colorPtr->color;
	}
	return;
    }
    if (dy == 0) {		/* Horizontal line */
	Pix32 *dp;

	dp = Blt_PicturePixel(destPtr, x1, y1);
	while(dx-- > 0) {
	    dp += xDir;
	    dp->color = colorPtr->color;
	}
	return;
    }
    if (dx == dy) {		/* Diagonal line. */
	Pix32 *dp;

	dp = Blt_PicturePixel(destPtr, x1, y1);
	while(dy-- > 0) {
	    dp += destPtr->pixelsPerRow + xDir;
	    dp->color = colorPtr->color;
	}
	return;
    }

    /* use Wu Antialiasing: */

    error = 0;
    if (dy > dx) {		/* y-major line */
	unsigned long adjust;

	/* x1 -= thickness / 2; */
	adjust = (dx << 16) / dy;
	while(--dy) {
	    Pix32 *dp;
	    int x;
	    unsigned char weight;
	    
	    error += adjust;
	    ++y1;
	    if (error & ~0xFFFF) {
		x1 += xDir;
		error &= 0xFFFF;
	    }
	    dp = Blt_PicturePixel(destPtr, x1, y1);
	    weight = (unsigned char)(error >> 8);
	    x = x1;
	    if (x >= 0) {
		dp->color = colorPtr->color;
		dp->Alpha = ~weight;
	    }
	    x += xDir;
	    dp += xDir;
	    if (x >= 0) {
		dp->color = colorPtr->color;
		dp->Alpha = weight;
	    }
	}
    } else {			/* x-major line */
	unsigned long adjust;

	/* y1 -= thickness / 2; */
	adjust = (dy << 16) / dx;
	while (--dx) {
	    Pix32 *dp;
	    int y;
	    unsigned char weight;

	    error += adjust;
	    x1 += xDir;
	    if (error & ~0xFFFF) {
		y1++;
		error &= 0xFFFF;
	    }
	    dp = Blt_PicturePixel(destPtr, x1, y1);
	    weight = (unsigned char)(error >> 8);
	    y = y1;
	    if (y >= 0) {
		dp->color = colorPtr->color;
		dp->Alpha = ~weight;
	    }
	    dp += destPtr->pixelsPerRow;
	    y++;
	    if (y >= 0) {
		dp->color = colorPtr->color;
		dp->Alpha = weight;
	    } 
	}
    }
}
