
/*
 * bltPicture.c --
 *
 * This module implements basic picture processing routines for the
 * BLT toolkit.
 *
 *	Copyright 1997-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"
#include "bltHash.h"
#include "bltPicture.h"
#include "bltPictureInt.h"
#include <X11/Xutil.h>

#define JITTER(x)	((x) * (0.05 - drand48() * 0.10))
#define JCLAMP(c)	((((c) < 0.0) ? 0.0 : ((c) > 1.0) ? 1.0 : (c)))

#define CLAMP(c)	((((c) < 0.0) ? 0.0 : ((c) > 255.0) ? 255.0 : (c)))

#define imul8x8(a,b,t)	((t) = (a)*(b)+128,(((t)+((t)>>8))>>8))

#ifdef HAVE_X86_ASM
extern int bltUseMMX;

void Blt_ApplyPictureToPictureMMX(Blt_Picture dest, Blt_Picture src,
	Blt_PictureArithOps op);
void Blt_ApplyScalarToPictureMMX(Blt_Picture dest, Pix32 *colorPtr, 
	Blt_PictureArithOps op);
void Blt_TentHorizontallyMMX(Blt_Picture dest, Blt_Picture src);
void Blt_TentVerticallyMMX(Blt_Picture dest, Blt_Picture src);
void Blt_ZoomHorizontallyMMX(Blt_Picture dest, Blt_Picture src, 
	Blt_PictureFilter filter);
void Blt_ZoomVerticallyMMX(Blt_Picture dest, Blt_Picture src, 
	Blt_PictureFilter filter);
void Blt_BlendPictureAreaMMX(Blt_Picture dest, Blt_Picture src, int srcX, 
	int srcY, int width, int height, int destX, int destY);
void Blt_SelectPixelsMMX(Blt_Picture dest, Blt_Picture src, 
	Pix32 *lowPtr , Pix32 *highPtr);
#endif /* HAVE_X86_ASM */

static PictureFilterProc DefaultFilter;
static PictureFilterProc BellFilter;
static PictureFilterProc BesselFilter;
static PictureFilterProc BoxFilter;
static PictureFilterProc BSplineFilter;
static PictureFilterProc CatRomFilter;
static PictureFilterProc DummyFilter;
static PictureFilterProc GaussianFilter;
static PictureFilterProc GiFilter;
static PictureFilterProc Lanczos3Filter;
static PictureFilterProc MitchellFilter;
static PictureFilterProc SincFilter;
static PictureFilterProc TriangleFilter;

extern int Blt_ResetPicture(Tcl_Interp *interp, char *imageName, 
	Blt_Picture picture);

/*
 *----------------------------------------------------------------------
 *
 * Blt_CreatePicture --
 *
 *      Allocates a picture of a designated height and width.
 *
 *	This routine will be augmented with other types of information
 *	such as a color table, etc.
 *
 * Results:
 *      Returns the new color pict.
 *
 *----------------------------------------------------------------------
 */
Picture *
Blt_CreatePicture(int width, int height) /* Dimensions of new picture. */
{
    Picture *destPtr;
    size_t stride;

    assert((width > 0) && (height > 0));
    destPtr = Blt_Malloc(sizeof(Picture));
#ifdef notdef
    fprintf(stderr, "alignof(destPtr->bits) = %d\n", 
		__alignof__(destPtr->bits));
#endif
    assert(destPtr);

    /* 
     * Be careful. There's a bunch of picture handling routines that
     * assume an even number of pixels per row. 
     */
    stride = (width + 3) & ~3;	/* Align each row on a 16-byte boundary. */

    destPtr->bits = Blt_Calloc(stride * height, sizeof(Pix32));
    assert(destPtr->bits);
    destPtr->align = 0;
    destPtr->pixelsPerRow = stride;
    destPtr->width  = width;
    destPtr->height = height;
    destPtr->flags  = BLT_PICTURE_DIRTY;
    return destPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_FreePicture --
 *
 *      Deallocates the given picture.
 *
 * Results:
 *      None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_FreePicture(Picture *pictPtr)
{
    Blt_Free(pictPtr->bits);
    Blt_Free(pictPtr);
}

int
Blt_ReallocatePicture(
    Picture *srcPtr,
    int width,			/* New dimensions of picture. */
    int height,
    int doSave)	
{
    if ((width != srcPtr->width) || (height != srcPtr->height)) {
	int pixelsPerRow;
	Pix32 *bits;

	/* 
	 * Be careful. There's a bunch of picture handling routines
	 * that assume an even number of pixels per row.
	 */
	pixelsPerRow = (width + 3) & ~3; /* Align each row on a
					  * 16-byte boundary. */
	
	bits = Blt_Calloc(pixelsPerRow * height, sizeof(Pix32));
	if (bits == NULL) {
	    return FALSE;
	}
	if (doSave) {
	    int bytesPerRow;
	    Pix32 *srcRowPtr, *destRowPtr;
	    int y, nRows;

	    bytesPerRow = sizeof(Pix32) * 
		MIN(pixelsPerRow, srcPtr->pixelsPerRow);
	    nRows = MIN(srcPtr->height, height);

	    srcRowPtr = srcPtr->bits, destRowPtr = bits;
	    for (y = 0; y < nRows; y++) {
		memcpy(destRowPtr, srcRowPtr, bytesPerRow);
		destRowPtr += pixelsPerRow;
		srcRowPtr  += srcPtr->pixelsPerRow;
	    }
	}
	if (srcPtr->bits != NULL) {
	    Blt_Free(srcPtr->bits);
	}
	srcPtr->pixelsPerRow = pixelsPerRow;
	srcPtr->width = width;
	srcPtr->height = height;
	srcPtr->bits = bits;
	srcPtr->flags = BLT_PICTURE_DIRTY;
    }
    return TRUE;
}

void
Blt_BlankPicture(
    Picture *destPtr, 
    Pix32 *colorPtr) 
{
    Pix32 *destRowPtr;
    int y;

    destRowPtr = destPtr->bits;
    for (y = 0; y < destPtr->height; y++) {
	int n;
	Pix32 *dp;

	dp = destRowPtr;
	n = (destPtr->width + 7) / 8; /* width > 0 assumed */
	switch (destPtr->width % 8) {
	case 0:        do {  dp->color = colorPtr->color, dp++;
	case 7:              dp->color = colorPtr->color, dp++;
	case 6:              dp->color = colorPtr->color, dp++;
	case 5:              dp->color = colorPtr->color, dp++;
	case 4:              dp->color = colorPtr->color, dp++;
	case 3:              dp->color = colorPtr->color, dp++;
	case 2:              dp->color = colorPtr->color, dp++;
	case 1:              dp->color = colorPtr->color, dp++;
			} while (--n > 0);
	}
	destRowPtr += destPtr->pixelsPerRow;
    }
    destPtr->flags |= BLT_PICTURE_DIRTY;
    destPtr->flags &= ~(BLT_PICTURE_BLEND | BLT_PICTURE_MASK);
    if (colorPtr->Alpha == 0x00) {
	destPtr->flags |= BLT_PICTURE_MASK;
    } else if (colorPtr->Alpha != 0xFF) {
	destPtr->flags |= BLT_PICTURE_BLEND;
    }
}

void
Blt_GammaCorrectPicture(Picture *destPtr, Picture *srcPtr, double gamma)
{
    Pix32 *srcRowPtr, *destRowPtr;
    int i, y;
    unsigned char lut[256];
    double iGamma;

    iGamma = 1.0 / gamma;
    for (i = 0; i < 256; i++) {
	double value;

	value = 255.0 * pow((double)i / 255.0, iGamma);
	lut[i] = (unsigned char)CLAMP(value);
    }
    srcRowPtr = srcPtr->bits;
    destRowPtr = destPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp, *dp;
	int x;

	sp = srcRowPtr, dp = destRowPtr;
	for (x = 0; x < srcPtr->width; x++) {
	    sp->Red = lut[(size_t)sp->Red];
	    sp->Green = lut[(size_t)sp->Green];
	    sp->Blue = lut[(size_t)sp->Blue];
	}
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_GreyscalePicture --
 *
 *	Returns a new picture converted to grey scale.  All three
 *	picture color components will have the same value.
 *
 *	Luminosity is computed using the formula,
 *
 *	    Y = 0.212671 * Red + 0.715160 * Green + 0.072169 * Blue
 *
 *	which is supposed to better represent contemporary monitors.
 *
 * Results:
 *	Returns a new greyscale picture.
 *
 *----------------------------------------------------------------------
 */

Blt_Picture 
Blt_GreyscalePicture(Picture *srcPtr)
{
    Picture *destPtr;
    Pix32 *srcRowPtr, *destRowPtr;
    int y;

    /*
     * We can use scaled integers (20-bit fraction) to compute the
     * luminosity with reasonable accuracy considering it's stored
     * in an 8-bit result.
     */
#define YR	223002		/* 0.212671 */
#define YG	749900		/* 0.715160 */
#define YB	75675		/* 0.072169 */
#define YMAX	267386880	/* 255.0 */
#define YCLAMP(s) \
	(unsigned char)((s) > YMAX) ? 255 : ((((s) + 524288) >> 20))

    destPtr = Blt_CreatePicture(srcPtr->width, srcPtr->height);
    srcRowPtr = srcPtr->bits;
    destRowPtr = destPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp, *dp, *send;

	sp = srcRowPtr, dp = destRowPtr;
	for (send = sp + srcPtr->width; sp < send; sp++) {
	    unsigned int Y;

	    Y = YR * sp->Red + YG * sp->Green + YB * sp->Blue;
	    dp->Red = dp->Green = dp->Blue = YCLAMP(Y);
	    dp->Alpha = sp->Alpha;
	    dp++;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
    destPtr->flags = srcPtr->flags;
    destPtr->flags |= BLT_PICTURE_DIRTY;
    destPtr->flags &= ~BLT_PICTURE_COLOR;
    return destPtr;
}

void
Blt_FadePicture(
    Picture *destPtr, 
    Picture *srcPtr, 
    double fade)
{
    Pix32 *srcRowPtr, *destRowPtr;
    double ifade;
    int y;

    if (srcPtr != destPtr) {
	if (!Blt_ReallocatePicture(destPtr, srcPtr->width, srcPtr->height, 0)) {
	    fprintf(stderr, "can't resize picture");
	    return;
	} 
    }
    destRowPtr = destPtr->bits;
    srcRowPtr = srcPtr->bits;

    if (srcPtr->flags & BLT_PICTURE_PREMULTIPLIED_ALPHAS) {
	Blt_UnmultiplyAlphas(srcPtr);
    }
    ifade = 1.0 / fade;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp, *dp, *send;
	
	dp = destRowPtr;
	for (sp = srcRowPtr, send = sp + srcPtr->width; sp < send; sp++, dp++) {
	    double a;
	    
	    a = ifade * + sp->Alpha;
	    dp->Alpha = (unsigned char)CLAMP(a);
	}
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
}

void
Blt_PremultiplyAlphas(Picture *srcPtr) /* (in/out) picture */
{
    Pix32 *srcRowPtr;
    int y;

    srcRowPtr = srcPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp;
	int x;

	sp = srcRowPtr;
	for (x = 0; x < srcPtr->width; x++) {
	    /* No conversion necessary if 100% transparent or opaque. */
	    if ((sp->Alpha != 0xFF) && (sp->Alpha != 0x00)) {
		int t;

		sp->Red = imul8x8(sp->Alpha, sp->Red, t);
		sp->Green = imul8x8(sp->Alpha, sp->Green, t);
		sp->Blue = imul8x8(sp->Alpha, sp->Blue, t);
	    }
	    sp++;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
    }
    srcPtr->flags |= BLT_PICTURE_PREMULTIPLIED_ALPHAS;
}

void
Blt_UnmultiplyAlphas(Picture *srcPtr) /* (in/out) Source picture with */
{
    Pix32 *srcRowPtr;
    int y;

    srcRowPtr = srcPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp;
	int x;

	sp = srcRowPtr;
	for (x = 0; x < srcPtr->width; x++) {
	    /* No conversion necessary if 100% transparent or opaque. */
	    if ((sp->Alpha != 0xFF) && (sp->Alpha != 0x00)) {
		int bias;
		    
		bias = sp->Alpha >> 1;
		sp->Red   = (mul255(sp->Red) + bias)   / sp->Alpha;
		sp->Green = (mul255(sp->Green) + bias) / sp->Alpha;
		sp->Blue  = (mul255(sp->Blue) + bias)  / sp->Alpha;
	    }
	    sp++;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
    }
    srcPtr->flags &= ~BLT_PICTURE_PREMULTIPLIED_ALPHAS;
}

static void
BlendPictureArea(
    Picture *bgPtr,		/* Foreground picture. */
    Picture *fgPtr,		/* (in/out) Background picture. Composite
				 * overwrites region in background. */
    int fgX, int fgY,		/* Origin of foreground region in source. */
    int fgWidth, int fgHeight,	/* Dimension of area in foreground to
				 * be blended. */
    int bgX, int bgY)		/* Start of blend area in background. */
{
    Pix32 *srcRowPtr, *destRowPtr;
    int y;

    if ((fgPtr->flags & (BLT_PICTURE_PREMULTIPLIED_ALPHAS | BLT_PICTURE_BLEND))
	== BLT_PICTURE_BLEND) {
	Blt_PremultiplyAlphas(fgPtr);
    }
    if ((bgPtr->flags & (BLT_PICTURE_PREMULTIPLIED_ALPHAS | BLT_PICTURE_BLEND))
	== BLT_PICTURE_BLEND) {
	Blt_PremultiplyAlphas(bgPtr);
    }
    destRowPtr = bgPtr->bits + ((bgY * bgPtr->pixelsPerRow) + bgX);
    srcRowPtr = fgPtr->bits + ((fgY * fgPtr->pixelsPerRow) + fgX);

    for (y = 0; y < fgHeight; y++) {
	Pix32 *sp, *dp;
	int x;

	sp = srcRowPtr, dp = destRowPtr;
	for (x = 0; x < fgWidth; x++) {

	    /* Blend the foreground and background together. */
	    if (sp->Alpha == 0xFF) {
		*dp = *sp;
	    } else if (sp->Alpha != 0x00) {
		int beta, t;
		int r,g, b, a;

		beta = sp->Alpha ^ 0xFF; /* beta = 1 - alpha */
		r = sp->Red   + imul8x8(beta, dp->Red, t);
		g = sp->Green + imul8x8(beta, dp->Green, t);
		b = sp->Blue  + imul8x8(beta, dp->Blue, t);
		a = sp->Alpha + imul8x8(beta, dp->Alpha, t);
		dp->Red = (r > 255) ? 255 : ((r < 0) ? 0 : r);
		dp->Green = (g > 255) ? 255 : ((g < 0) ? 0 : g);
		dp->Blue = (b > 255) ? 255 : ((b < 0) ? 0 : b);
		dp->Alpha = (a > 255) ? 255 : ((a < 0) ? 0 : a);
	    }
	    sp++, dp++;
	}
	srcRowPtr += fgPtr->pixelsPerRow;
	destRowPtr += bgPtr->pixelsPerRow;
    }
}

static void
BlendPictureArea2(
    Picture *bgPtr,		/* Foreground picture. */
    Picture *fgPtr,		/* (in/out) Background picture. Composite
				 * overwrites region in background. */
    Blt_BlendingMode mode,	/* Blend mode. */
    int fgX, int fgY,		/* Origin of foreground region in source. */
    int width, int height,	/* Dimension of region to be blended. */
    int bgX, int bgY)	/* Origin of background region in
				 * destination. */
{
    Pix32 *srcRowPtr, *destRowPtr;
    int y;

    if ((fgPtr->flags & (BLT_PICTURE_PREMULTIPLIED_ALPHAS | BLT_PICTURE_BLEND))
	== BLT_PICTURE_BLEND) {
	Blt_PremultiplyAlphas(fgPtr);
    }
    if ((bgPtr->flags & (BLT_PICTURE_PREMULTIPLIED_ALPHAS | BLT_PICTURE_BLEND))
	== BLT_PICTURE_BLEND) {
	Blt_PremultiplyAlphas(bgPtr);
    }
    destRowPtr = bgPtr->bits + ((bgY * bgPtr->pixelsPerRow) + bgX);
    srcRowPtr = fgPtr->bits + ((fgY * fgPtr->pixelsPerRow) + fgX);

    for (y = 0; y < height; y++) {
	Pix32 *sp, *send, *dp;

	dp = destRowPtr;
	for (sp = srcRowPtr, send = sp + width; sp < send; sp++ , dp++) {
	    int r,g, b, a;
	    double fa, z;

	    /*
	     * C = ((1 - Fa) * B) + ((1 - Ba) * F) + (Fa * Ba * BLEND(F,B));
	     */
	    {
		int fgBeta, bgBeta;

		bgBeta = dp->Alpha ^ 0xFF;
		fgBeta = sp->Alpha ^ 0xFF;
		r = (fgBeta * dp->Red)   + (bgBeta * sp->Red);
		g = (fgBeta * dp->Green) + (bgBeta * sp->Green);
		b = (fgBeta * dp->Blue)  + (bgBeta * sp->Blue);
		a = (fgBeta * dp->Alpha) + (bgBeta * sp->Alpha);
	    }
	    switch (mode) {
	    case BLT_BLEND_NORMAL: /* C = F */
		r += dp->Alpha * sp->Red;
		g += dp->Alpha * sp->Green;
		b += dp->Alpha * sp->Blue;
		a += dp->Alpha * sp->Alpha;
		break;
		
	    case BLT_BLEND_MULTIPLY: /* C = F * B */
		r += dp->Red * sp->Red;
		g += dp->Green * sp->Green;
		b += dp->Blue * sp->Blue;
		a += dp->Alpha * sp->Alpha;
		break;
		
	    case BLT_BLEND_COLORDODGE: /* C = B / (1 - F) */
		if (sp->Alpha > 0) {
		    fa = 1.0 / sp->Alpha;
		    z = (sp->Alpha * dp->Red) / (1.0 - (sp->Red * fa));
		    r += z + 0.5 ;
		    z = (sp->Alpha * dp->Green) / (1.0 - (sp->Green * fa));
		    g += z + 0.5 ;
		    z = (sp->Alpha * dp->Blue) / (1.0 - (sp->Blue * fa));
		    b += z + 0.5;
		    a += dp->Alpha * sp->Alpha;
		} else {
		    r += dp->Alpha * sp->Red;
		    g += dp->Alpha * sp->Green;
		    b += dp->Alpha * sp->Blue;
		    a += dp->Alpha * sp->Alpha;
		}
		break;
		
	    case BLT_BLEND_COLORBURN: /* C = B / F; */
		if (sp->Alpha > 0) {
		    fa = 1.0 / sp->Alpha;
		    z = 511.0 * ((dp->Alpha * sp->Red * fa) - 
				 ((sp->Alpha * dp->Red) / (sp->Red + 1)));
		    r += z + 0.5;
		    z = 511.0 * ((dp->Alpha * sp->Green * fa) - 
				 ((sp->Alpha * dp->Green) / (sp->Green + 1)));
		    g += z + 0.5;
		    z = 511.0 * ((dp->Alpha * sp->Blue * fa) - 
				 ((sp->Alpha * dp->Blue) / (sp->Blue + 1)));
		    b += z + 0.5;
		    a += dp->Alpha * sp->Alpha;
		} else {
		    r += dp->Alpha * sp->Red;
		    g += dp->Alpha * sp->Green;
		    b += dp->Alpha * sp->Blue;
		    a += dp->Alpha * sp->Alpha;
		}
		break;
		
	    case BLT_BLEND_OVERLAY: /* Multiplies or screens depending
				     * upon background color. */
		if (dp->Red < 128) {
		    r += dp->Red * sp->Red;
		} else {
		    r += ((sp->Alpha * dp->Red) + (dp->Alpha * sp->Red) - 
			  (sp->Red * dp->Red));
		}
		if (dp->Green < 128) {
		    g += dp->Green * sp->Green;
		} else {
		    g += ((sp->Alpha * dp->Green) + (dp->Alpha * sp->Green) - 
			  (sp->Green * dp->Green));
		}
		if (dp->Blue < 128) {
		    b += dp->Blue * sp->Blue;
		} else {
		    b += ((sp->Alpha * dp->Blue) + (dp->Alpha * sp->Blue) - 
			  (sp->Blue * dp->Blue));
		}
		a += dp->Alpha * sp->Alpha;
		break;
		
	    case BLT_BLEND_SCREEN: /* C = 1 - ((1 - F) * B) */
		r += ((sp->Alpha * dp->Red) + (dp->Alpha * sp->Red) - 
		      (sp->Red * dp->Red));
		g += ((sp->Alpha * dp->Green) + (dp->Alpha * sp->Green) - 
		      (sp->Green * dp->Green));
		b += ((sp->Alpha * dp->Blue) + (dp->Alpha * sp->Blue) - 
		      (sp->Blue * dp->Blue));
		a += dp->Alpha * sp->Alpha;
		break;
		
	    case BLT_BLEND_DARKEN:	/* C = min(F,B) */
		r += MIN(dp->Alpha * sp->Red,   sp->Alpha * dp->Red);
		g += MIN(dp->Alpha * sp->Green, sp->Alpha * dp->Green);
		b += MIN(dp->Alpha * sp->Blue,  sp->Alpha * dp->Blue);
		a += dp->Alpha * sp->Alpha;
		break;
		
	    case BLT_BLEND_LIGHTEN:	/* C = max(F,B) */
		r += MAX(dp->Alpha * sp->Red,   sp->Alpha * dp->Red);
		g += MAX(dp->Alpha * sp->Green, sp->Alpha * dp->Green);
		b += MAX(dp->Alpha * sp->Blue,  sp->Alpha * dp->Blue);
		a += dp->Alpha * sp->Alpha;
		break;
		
	    case BLT_BLEND_DIFFERENCE: /* C = |F - B| */
		/* C = |(F * Ba) - (B * Fa)| */
		r += ABS((dp->Alpha * sp->Red) -   (sp->Alpha * dp->Red));
		g += ABS((dp->Alpha * sp->Green) - (sp->Alpha * dp->Green));
		b += ABS((dp->Alpha * sp->Blue) -  (sp->Alpha * dp->Blue));
		a += dp->Alpha * sp->Alpha;
		break;
	    case BLT_BLEND_HARDLIGHT:
		if (sp->Red > 127) {
		    r += ((sp->Alpha * dp->Red) + (dp->Alpha * sp->Red) - 
			  (sp->Red * dp->Red));
		} else {
		    r += dp->Red * sp->Red;
		}
		if (sp->Green > 127) {
		    g += ((sp->Alpha * dp->Green) + 
			  (dp->Alpha * sp->Green) - 
			  (sp->Green * dp->Green));
		} else {
		    g += dp->Green * sp->Green;
		}
		if (sp->Blue > 127) {
		    b += ((sp->Alpha * dp->Blue) + (dp->Alpha * sp->Blue) - 
			  (sp->Blue * dp->Blue));
		} else {
		    b += dp->Blue * sp->Blue;
		}
		a += MIN(sp->Alpha, dp->Alpha);
		break;
	    case BLT_BLEND_SOFTLIGHT:
		fa = 1.0 / sp->Alpha;
		if (sp->Red > 127) {
		    z = (sp->Alpha * dp->Red) / (1.0 - (sp->Red * fa));
		} else {
		    z = 511.0 * ((dp->Alpha * sp->Red * fa) - 
			((sp->Alpha * dp->Red) / (sp->Red + 1)));
		}
		r += z + 0.5;
		if (sp->Green > 127) {
		    z = (sp->Alpha * dp->Green) / (1.0 - (sp->Green * fa));
		} else {
		    z = 511.0 * ((dp->Alpha * sp->Green * fa) - 
			((sp->Alpha * dp->Green) / (sp->Green + 1)));
		}
		g += z + 0.5 ;
		if (sp->Blue > 127) {
		    z = (sp->Alpha * dp->Blue) / (1.0 - (sp->Blue * fa));
		} else {
		    z = 511.0 * ((dp->Alpha * sp->Blue * fa) - 
				 ((sp->Alpha * dp->Blue) / (sp->Blue + 1)));
		}
		b += z + 0.5 ;
		a += sp->Alpha * dp->Alpha;
		break;
	    }
	    dp->Red =   (r >= 65025) ? 255 : ((r < 0) ? 0 : div255(r));
	    dp->Green = (g >= 65025) ? 255 : ((g < 0) ? 0 : div255(g));
	    dp->Blue =  (b >= 65025) ? 255 : ((b < 0) ? 0 : div255(b));
	    dp->Alpha = (a >= 65025) ? 255 : ((a < 0) ? 0 : div255(a));
	}
	srcRowPtr += fgPtr->pixelsPerRow;
	destRowPtr += bgPtr->pixelsPerRow;
    }
}

void
Blt_BlendPictureArea(
    Picture *bgPtr,		/* Foreground picture. */
    Picture *fgPtr,		/* (in/out) Background picture. Composite
				 * overwrites region in background. */
    int fgX, int fgY,		/* Origin of area in foreground picture. */
    int fgWidth, int fgHeight,	/* Dimension of foreground area to be
				 * blended. */
    int bgX, int bgY)		/* Origin of area in background. */
{
    if ((fgPtr->flags & (BLT_PICTURE_PREMULTIPLIED_ALPHAS | BLT_PICTURE_BLEND))
	== BLT_PICTURE_BLEND) {
	Blt_PremultiplyAlphas(fgPtr);
    }
    if ((bgX + fgWidth) > bgPtr->width) {
	fgWidth = bgPtr->width - bgX;
    }
    if ((bgY + fgHeight) > bgPtr->height) {
	fgHeight = bgPtr->height - bgY;
    }
#ifdef HAVE_X86_ASM
    if (bltUseMMX) {
	Blt_BlendPictureAreaMMX(bgPtr, fgPtr, fgX, fgY, fgWidth, fgHeight, 
		bgX, bgY);
    } else {
	BlendPictureArea(bgPtr, fgPtr, fgX, fgY, fgWidth, fgHeight, bgX, bgY);
    }
#else 
    BlendPictureArea(bgPtr, fgPtr, fgX, fgY, fgWidth, fgHeight, bgX, bgY);
#endif /* HAVE_X86_ASM */
}

void
Blt_BlendPictures(
    Picture *bgPtr,		/* Foreground picture. */
    Picture *fgPtr)		/* (in/out) Background picture. Composite
				 * overwrites region in background. */
{
    Blt_BlendPictureArea(bgPtr, fgPtr, 0, 0, fgPtr->width, 
	fgPtr->height, 0, 0);
}

void
Blt_BlendPictures2(
    Picture *bgPtr,		/* Foreground picture. */
    Picture *fgPtr,		/* (in/out) Background picture. Composite
				 * overwrites region in background. */
    Blt_BlendingMode mode)
{
    BlendPictureArea2(bgPtr, fgPtr, mode, 0, 0, fgPtr->width, fgPtr->height, 
	0, 0);
}

/*
 *------------------------------------------------------------------------
 *
 * Blt_BitmapToPicture --
 *
 *	Converts the given bitmap into a picture.
 *
 * Results:
 *	Returns a new picture containing the bitmap image.
 *
 *-------------------------------------------------------------------------
 */
Blt_Picture
Blt_BitmapToPicture(
    Display *display,
    Pixmap bitmap,		/* Source bitmap. */
    int width, int height,	/* Width and height of the bitmap */
    Pix32 *fgColorPtr,		/* Foreground pixel color */
    Pix32 *bgColorPtr)		/* Background pixel color */
{
    XImage *imagePtr;
    Pix32 *destRowPtr;
    Picture *destPtr;
    int y;
    unsigned int flags;

    imagePtr = XGetImage(display, bitmap, 0, 0, width, height, 1, ZPixmap);

    destPtr = Blt_CreatePicture(width, height);
    destRowPtr = destPtr->bits;
    for (y = 0; y < height; y++) {
	Pix32 *dp;
	int x;

	/*
	 * The slow but robust brute force method of converting an X image:
	 * FIXME: update to avoid XGetPixel calls.
	 */
	dp = destRowPtr;
	for (x = 0; x < width; x++) {
	    unsigned int pixel;

	    pixel = XGetPixel(imagePtr, x, y);
	    dp->color = (pixel) ? fgColorPtr->color : bgColorPtr->color;
	    dp++;
	}
	destRowPtr += destPtr->pixelsPerRow;
    }
    XDestroyImage(imagePtr);
    flags = 0;
    if (bgColorPtr->Alpha == 0x00) { 
	flags |= BLT_PICTURE_MASK;
    } else if (bgColorPtr->Alpha != 0xFF) {
	flags |= BLT_PICTURE_BLEND;
    }
    if (fgColorPtr->Alpha == 0x00) { 
	flags |= BLT_PICTURE_MASK;
    } else if (fgColorPtr->Alpha != 0xFF) {
	flags |= BLT_PICTURE_BLEND;
    }
    destPtr->flags |= flags;
    return destPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PictureToPhoto --
 *
 *      Translates a picture into a Tk photo.
 *
 * Results:
 *      The photo is re-written with the new picture.
 *
 *----------------------------------------------------------------------
 */
void
Blt_PictureToPhoto(
    Picture *srcPtr,		/* Image to use as source */
    Tk_PhotoHandle photo)	/* Photo to write picture into */
{
    Tk_PhotoImageBlock destImgBlock;

    Tk_PhotoGetImage(photo, &destImgBlock);
    destImgBlock.pixelSize = sizeof(Pix32);
    destImgBlock.pitch = sizeof(Pix32) * srcPtr->pixelsPerRow;
    destImgBlock.width = srcPtr->width;
    destImgBlock.height = srcPtr->height;
    destImgBlock.offset[0] = Blt_Offset(Pix32, Red);
    destImgBlock.offset[1] = Blt_Offset(Pix32, Green);
    destImgBlock.offset[2] = Blt_Offset(Pix32, Blue);
    destImgBlock.offset[3] = Blt_Offset(Pix32, Alpha); 
    Tk_PhotoSetSize(photo, srcPtr->width, srcPtr->height);
    if (srcPtr->flags & BLT_PICTURE_BLEND) {
	Picture *tmpPtr;

	/* Divide out the alphas from picture's pre-multipled RGB
	 * values. */
	tmpPtr = Blt_ClonePicture(srcPtr);
	Blt_UnmultiplyAlphas(tmpPtr);
	destImgBlock.pixelPtr = (unsigned char *)tmpPtr->bits;
	Tk_PhotoSetSize(photo, tmpPtr->width, tmpPtr->height);
	Tk_PhotoPutBlock(photo, &destImgBlock, 0, 0, tmpPtr->width, 
		tmpPtr->height);
	Blt_FreePicture(tmpPtr);
    } else {
	destImgBlock.pixelPtr = (unsigned char *)srcPtr->bits;
	Tk_PhotoPutBlock(photo, &destImgBlock, 0, 0, srcPtr->width, 
		srcPtr->height);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PhotoAreaToPicture --
 *
 *      Create a photo to a picture.
 *
 * Results:
 *      The new picture is returned.
 *
 *----------------------------------------------------------------------
 */
Picture *
Blt_PhotoAreaToPicture(
    Tk_PhotoHandle photo,	/* Source photo image to scale */
    int x, int y,
    int width, int height)
{
    Tk_PhotoImageBlock srcImgBlock;
    Picture *destPtr;
    int offset;
    int ir, ib, ig, ia;

    if (x < 0) {
	x = 0;
    } 
    if (y < 0) {
	y = 0;
    }
    Tk_PhotoGetImage(photo, &srcImgBlock);
    if (width < 0) {
	width = srcImgBlock.width;
    }
    if (height < 0) {
	height = srcImgBlock.height;
    }
    if ((x + width) > srcImgBlock.width) {
	width = srcImgBlock.width - x;
    }
    if ((height + y) > srcImgBlock.height) {
	height = srcImgBlock.width - y;
    }
    offset = (x * srcImgBlock.pixelSize) + (y * srcImgBlock.pitch);

    destPtr = Blt_CreatePicture(width, height);
    ir = srcImgBlock.offset[0];
    ig = srcImgBlock.offset[1];
    ib = srcImgBlock.offset[2];
    ia = srcImgBlock.offset[3];

    if (srcImgBlock.pixelSize == 4) {
	Pix32 *destRowPtr;
	int x, y;

	destRowPtr = destPtr->bits;
        for (y = 0; y < height; y++) {
	    Pix32 *dp;
	    unsigned char *bits;

	    dp = destRowPtr;
	    bits = srcImgBlock.pixelPtr + offset;
	    for (x = 0; x < width; x++) {

	        dp->Alpha = bits[ia];

		if (dp->Alpha == 0xFF) {
		    dp->Red = bits[ir];
		    dp->Green = bits[ig];
		    dp->Blue = bits[ib];
		} else if (dp->Alpha == 0x00) {
		    dp->Red = bits[ir];
		    dp->Green = bits[ig];
		    dp->Blue = bits[ib];
		    destPtr->flags |= BLT_PICTURE_MASK;
		} else {
		    int t;

		    /* 
		     * Premultiple the alpha into each component. 
		     * (0..255 * 0..255) / 255.0 
		     */
		    dp->Red = imul8x8(dp->Alpha, bits[ir], t);
		    dp->Green = imul8x8(dp->Alpha, bits[ig], t);
		    dp->Blue = imul8x8(dp->Alpha, bits[ib], t);
		    destPtr->flags |= 
			(BLT_PICTURE_BLEND | BLT_PICTURE_PREMULTIPLIED_ALPHAS);
		}
		bits += srcImgBlock.pixelSize;
	        dp++;
	    }
	    offset += srcImgBlock.pitch;
	    destRowPtr += destPtr->pixelsPerRow;
        }
    } else if (srcImgBlock.pixelSize == 3) {
	Pix32 *destRowPtr;
	int x, y;

	destRowPtr = destPtr->bits;
        for (y = 0; y < height; y++) {
	    Pix32 *dp;
	    unsigned char *bits;

	    dp = destRowPtr;
	    bits = srcImgBlock.pixelPtr + offset;
	    for (x = 0; x < width; x++) {
	        dp->Red = bits[ir];
	        dp->Green = bits[ig];
	        dp->Blue = bits[ib];
	        dp->Alpha = ALPHA_OPAQUE;
	        bits += srcImgBlock.pixelSize;
	        dp++;
	    }
	    offset += srcImgBlock.pitch;
	    destRowPtr += destPtr->pixelsPerRow;
        }
    } else {
	Pix32 *destRowPtr;
	int x, y;

	destRowPtr = destPtr->bits;
        for (y = 0; y < height; y++) {
	    Pix32 *dp;
	    unsigned char *bits;

	    dp = destRowPtr;
	    bits = srcImgBlock.pixelPtr + offset;
	    for (x = 0; x < width; x++) {
	        dp->Red = dp->Green = dp->Blue = bits[ir];
	        dp->Alpha = ALPHA_OPAQUE;
	        bits += srcImgBlock.pixelSize;
	        dp++;
	    }
	    offset += srcImgBlock.pitch;
	    destRowPtr += destPtr->pixelsPerRow;
        }
    } 
    return destPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PhotoToPicture --
 *
 *      Create a photo to a picture.
 *
 * Results:
 *      The new picture is returned.
 *
 *----------------------------------------------------------------------
 */
Picture *
Blt_PhotoToPicture(Tk_PhotoHandle photo) /* Source photo to scale */
{
    Picture *destPtr;
    Tk_PhotoImageBlock srcImgBlock;
    int bytesPerRow;
    int srcWidth, srcHeight;
    int ir, ib, ig, ia;

    Tk_PhotoGetImage(photo, &srcImgBlock);
    srcWidth = srcImgBlock.width;
    srcHeight = srcImgBlock.height;
    bytesPerRow = srcImgBlock.pixelSize * srcWidth;
    ir = srcImgBlock.offset[0];
    ig = srcImgBlock.offset[1];
    ib = srcImgBlock.offset[2];
    ia = srcImgBlock.offset[3];

    destPtr = Blt_CreatePicture(srcWidth, srcHeight);
    if (srcImgBlock.pixelSize == 4) {
	Pix32 *destRowPtr;
	int y;
	unsigned char *srcRowPtr;
	
	srcRowPtr = srcImgBlock.pixelPtr;
	destRowPtr = destPtr->bits;
        for (y = 0; y < srcHeight; y++) {
	    Pix32 *dp;
	    unsigned char *bits, *bend;

	    dp = destRowPtr;
	    for (bits = srcRowPtr, bend = bits + bytesPerRow; bits < bend; 
		bits += srcImgBlock.pixelSize) {
	        dp->Alpha = bits[ia];
		if (dp->Alpha == 0xFF) {
		    dp->Red = bits[ir];
		    dp->Green = bits[ig];
		    dp->Blue = bits[ib];
		} else if (dp->Alpha == 0x00) {
		    dp->Red = bits[ir];
		    dp->Green = bits[ig];
		    dp->Blue = bits[ib];
		    destPtr->flags |= BLT_PICTURE_MASK;
		} else {
		    int t;

		    /* 
		     * Premultiple the alpha into each component. 
		     * (0..255 * 0..255) / 255.0 
		     */
		    dp->Red = imul8x8(dp->Alpha, bits[ir], t);
		    dp->Green = imul8x8(dp->Alpha, bits[ig], t);
		    dp->Blue = imul8x8(dp->Alpha, bits[ib], t);
		    destPtr->flags |= BLT_PICTURE_BLEND;
		}
	        dp++;
	    }
	    srcRowPtr += srcImgBlock.pitch;
	    destRowPtr += destPtr->pixelsPerRow;
        }
    } else if (srcImgBlock.pixelSize == 3) {
	Pix32 *destRowPtr;
	int y;
	unsigned char *srcRowPtr;

	srcRowPtr = srcImgBlock.pixelPtr;
	destRowPtr = destPtr->bits;
        for (y = 0; y < srcHeight; y++) {
	    Pix32 *dp;
	    unsigned char *bits, *bend;

	    dp = destRowPtr;
	    for (bits = srcRowPtr, bend = bits + bytesPerRow; bits < bend; 
		bits += srcImgBlock.pixelSize) {
	        dp->Red = bits[ir];
	        dp->Green = bits[ig];
	        dp->Blue = bits[ib];
	        dp->Alpha = ALPHA_OPAQUE;
	        dp++;
	    }
	    srcRowPtr += srcImgBlock.pitch;
	    destRowPtr += destPtr->pixelsPerRow;
        }
    } else {
	Pix32 *destRowPtr;
	int y;
	unsigned char *srcRowPtr;

	srcRowPtr = srcImgBlock.pixelPtr;
	destRowPtr = destPtr->bits;
        for (y = 0; y < srcHeight; y++) {
	    Pix32 *dp;
	    unsigned char *bits, *bend;

	    dp = destRowPtr;
	    for (bits = srcRowPtr, bend = bits + bytesPerRow; bits < bend; 
		bits += srcImgBlock.pixelSize) {
	        dp->Red = dp->Green = dp->Blue = bits[ir];
	        dp->Alpha = ALPHA_OPAQUE;
	        dp++;
	    }
	    srcRowPtr += srcImgBlock.pitch;
	    destRowPtr += destPtr->pixelsPerRow;
        }
    } 
    return destPtr;
}

/*
 *	filter function definitions
 */

/*
 *----------------------------------------------------------------------
 *
 * ByteToHex --
 *
 *	Convert a byte to its ASCII hexidecimal equivalent.
 *
 * Results:
 *	The converted 2 ASCII character string is returned.
 *
 *----------------------------------------------------------------------
 */
INLINE static void
ByteToHex(unsigned char byte, char *string)
{
    static char hexDigits[] = "0123456789ABCDEF";

    string[0] = hexDigits[byte >> 4];
    string[1] = hexDigits[byte & 0x0F];
}


/*
 *----------------------------------------------------------------------
 *
 * Blt_PictureToPsData --
 *
 *	Converts a picture to PostScript RGB (3 components)
 *	or Greyscale (1 component) output.  With 3 components, we
 *	assume the "colorimage" operator is available.
 *
 *	Note: The picture is converted from bottom to top, to conform
 *	      with the PostScript coordinate system.
 *
 * Results:
 *	The PostScript data comprising the picture is written
 *	into the dynamic string.
 *
 *----------------------------------------------------------------------
 */
int
Blt_PictureToPsData(
    Picture *srcPtr,		/* Picture to be represented in PostScript. */
    int nComponents,		/* # of color components (either 1 or
				 * 3).  If it's 1, we only look at
				 * red for color information. */
    Tcl_DString *resultPtr,	/* (out) Holds the generated postscript */
    char *prefix)		/* Indicates how to prefix the start
				 * of each line of output. This is
				 * normally used for PostScript
				 * previews, where each line is
				 * comment "% ". */
{
    int count, nLines;

    nLines = count = 0;
    if (nComponents == 3) {
	Pix32 *srcRowPtr;
	int y;

	srcRowPtr = srcPtr->bits + ((srcPtr->height-1) * srcPtr->pixelsPerRow);
	for (y = (srcPtr->height - 1); y >= 0; y--) {
	    Pix32 *sp;
	    int x;

	    sp = srcRowPtr;
	    for (x = 0; x < srcPtr->width; x++, sp++) {
		char string[10];

		if (count == 0) {
		    Tcl_DStringAppend(resultPtr, prefix, -1);
		    Tcl_DStringAppend(resultPtr, " ", -1);
		}
		count += 6;
		ByteToHex(sp->Red, string);
		ByteToHex(sp->Green, string + 2);
		ByteToHex(sp->Blue, string + 4);
		string[6] = '\0';
		if (count >= 60) {
		    string[6] = '\n';
		    string[7] = '\0';
		    count = 0;
		    nLines++;
		}
		Tcl_DStringAppend(resultPtr, string, -1);
	    }
	    srcRowPtr -= srcPtr->pixelsPerRow;
	}
    } else if (nComponents == 1) {
	int y;
	Pix32 *srcRowPtr;

	srcRowPtr = srcPtr->bits + ((srcPtr->height-1) * srcPtr->pixelsPerRow);
	for (y = (srcPtr->height - 1); y >= 0; y--) {
	    Pix32 *sp;
	    int x;

	    sp = srcRowPtr;
	    for (x = 0; x < srcPtr->width; x++, sp++) {
		char string[10];
		char byte;

		if (count == 0) {
		    Tcl_DStringAppend(resultPtr, prefix, -1);
		    Tcl_DStringAppend(resultPtr, " ", -1);
		}
		count += 2;
		byte = ~(sp->Red);
		ByteToHex(byte, string);
		string[2] = '\0';
		if (count >= 60) {
		    string[2] = '\n';
		    string[3] = '\0';
		    count = 0;
		    nLines++;
		}
		Tcl_DStringAppend(resultPtr, string, -1);
	    }
	    srcRowPtr -= srcPtr->pixelsPerRow;
	}
    }
    if (count != 0) {
	Tcl_DStringAppend(resultPtr, "\n", -1);
	nLines++;
    }
    return nLines;
}

static double
DefaultFilter(double x)
{
    if (x < 0.0) {
	x = -x;
    }
    if (x < 1.0) {
	/* f(x) = 2x^3 - 3x^2 + 1, -1 <= x <= 1 */
	return (2.0 * x - 3.0) * x * x + 1.0;
    }
    return 0.0;
}

/* Just for testing */
static double
DummyFilter(double x)
{
    return FABS(x);
}

/*
 *
 * Finite filters in increasing order:
 *	Box (constant)
 *	Triangle (linear)
 *	Bell
 *	BSpline (cubic)
 *
 */
static double
BoxFilter(double x)
{
    if ((x < -0.5) || (x > 0.5)) {
	return 0.0;
    }
    return 1.0;
}

static double
TriangleFilter(double x)
{
    if (x < 0.0) {
	x = -x;
    }
    if (x < 1.0) {
	return (1.0 - x);
    }
    return 0.0;
}

static double
BellFilter(double x)
{
    if (x < 0.0) {
	x = -x;
    }
    if (x < 0.5) {
	return (0.75 - (x * x));
    }
    if (x < 1.5) {
	x = (x - 1.5);
	return (0.5 * (x * x));
    }
    return 0.0;
}

static double
BSplineFilter(double x)
{
    double x2;

    if (x < 0.0) {
	x = -x;
    }
    if (x < 1) {
	x2 = x * x;
	return ((.5 * x2 * x) - x2 + (2.0 / 3.0));
    } else if (x < 2) {
	x = 2 - x;
	return ((x * x * x) / 6.0);
    }
    return 0.0;
}

/*
 *
 * Infinite Filters:
 *	Sinc		perfect lowpass filter
 *	Bessel		circularly symmetric 2-D filter
 *	Gaussian
 *	Lanczos3
 *	Mitchell
 */

static double
SincFilter(double x)
{
    x *= M_PI;
    if (x == 0.0) {
	return 1.0;
    }
    return (sin(x) / x);
}

static double
BesselFilter(double x)
{
    /*
     * See Pratt "Digital Image Processing" p. 97 for Bessel functions
     * zeros are at approx x=1.2197, 2.2331, 3.2383, 4.2411, 5.2428, 6.2439,
     * 7.2448, 8.2454
     */
#ifdef __BORLANDC__
    return 0.0;
#else
    return (x == 0.0) ? M_PI / 4.0 : j1(M_PI * x) / (x + x);
#endif
}

#define SQRT_2PI	0.79788456080286541	/* sqrt(2.0 / M_PI) */

static double
GaussianFilter(double x)
{
    return exp(-2.0 * x * x) * SQRT_2PI;
}

static double
Lanczos3Filter(double x)
{
    if (x < 0) {
	x = -x;
    }
    if (x < 3.0) {
	return (SincFilter(x) * SincFilter(x / 3.0));
    }
    return 0.0;
}

#define	B		0.3333333333333333	/* (1.0 / 3.0) */
#define	C		0.3333333333333333	/* (1.0 / 3.0) */

static double
MitchellFilter(double x)
{
    double x2;

    x2 = x * x;
    if (x < 0) {
	x = -x;
    }
    if (x < 1.0) {
	x = (((12.0 - 9.0 * B - 6.0 * C) * (x * x2)) +
	    ((-18.0 + 12.0 * B + 6.0 * C) * x2) + (6.0 - 2 * B));
	return (x / 6.0);
    } else if (x < 2.0) {
	x = (((-1.0 * B - 6.0 * C) * (x * x2)) + ((6.0 * B + 30.0 * C) * x2) +
	    ((-12.0 * B - 48.0 * C) * x) + (8.0 * B + 24 * C));
	return (x / 6.0);
    }
    return 0.0;
}

/*
 * Catmull-Rom spline
 */
static double
CatRomFilter(double x)
{
    if (x < -2.) {
	return 0.0;
    }
    if (x < -1.0) {
	return 0.5 * (4.0 + x * (8.0 + x * (5.0 + x)));
    }
    if (x < 0.0) {
	return 0.5 * (2.0 + x * x * (-5.0 + x * -3.0));
    }
    if (x < 1.0) {
	return 0.5 * (2.0 + x * x * (-5.0 + x * 3.0));
    }
    if (x < 2.0) {
	return 0.5 * (4.0 + x * (-8.0 + x * (5.0 - x)));
    }
    return 0.0;
}

/* approximation to the gaussian integral [x, inf) */
static double
GiFilter(double x)
{
    if (x > 1.5) {
	return 0.0;
    } else if (x < -1.5) {
	return 1.0;
    } else {
#define I6 0.166666666666667
#define I4 0.25
#define I3 0.333333333333333
	double x2 = x * x;
	double x3 = x2 * x;

	if (x > 0.5) {
	    return .5625  - ( x3 * I6 - 3 * x2 * I4 + 1.125 * x);
	} else if (x > -0.5) {
	    return 0.5    - (0.75 * x - x3 * I3);
	} else {
	    return 0.4375 + (-x3 * I6 - 3 * x2 * I4 - 1.125 * x);
	}
    }
}

static double filterValues[11];
static int filterSize;

static double
TableFilter(double x)
{
    int i;

    if (filterSize == 0) {
	return 0.0;
    }
    /* 
     * convert floating point distance x 
     *     x = -r..r where r = filterSize / 2 
     * to interger index i 
     *	   i = 0..filterSize
     */
    i = (int)(x + (filterSize * 0.5));
    if (i < 0) {
	i = 0;
    } else if (i > filterSize) {
	i = filterSize - 1;
    }
    return filterValues[i];
}

static PictureFilter filterTable[] =
{
    /* Name,     Function,		     Support */
    {"bell",     BellFilter,		     1.5	 },
    {"bessel",   BesselFilter,		     3.2383      },
    {"box",      BoxFilter,		     0.5         },
    {"bspline",  BSplineFilter,		     2.0	 },
    {"catrom",   CatRomFilter,		     2.0	 },
    {"default",  DefaultFilter,		     1.0	 },
    {"dummy",    DummyFilter,		     0.5	 },
    {"gauss8",   GaussianFilter,	     8.0	 },
    {"gaussian", GaussianFilter,	     1.25	 },
    {"gi",	 GiFilter,		     4.0	 },
    {"lanczos3", Lanczos3Filter,	     3.0	 },
    {"mitchell", MitchellFilter,	     2.0	 },
    {"none",     (PictureFilterProc *)NULL, 0.0	 },
    {"sinc",     SincFilter,		     4.0	 },
    {"tent",	 TriangleFilter,	     1.0	 },
    {"triangle", TriangleFilter,	     1.0	 },
};

static int nFilters = sizeof(filterTable) / sizeof(PictureFilter);

static PictureFilter tableFilter = { "table", TableFilter, 1.0 };

Blt_PictureFilter bltBoxFilter = &(filterTable[2]);
Blt_PictureFilter bltMitchellFilter = &(filterTable[11]);

char *
Blt_NameOfPictureFilter(PictureFilter *filterPtr)
{
    return filterPtr->name;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_GetPictureFilterFromObj --
 *
 *      Finds a 1-D filter associated by the given filter name.
 *
 * Results:
 *      A standard Tcl result.  Returns TCL_OK is the filter was
 *	found.  The filter information (proc and support) is returned
 *	via filterPtrPtr. Otherwise TCL_ERROR is returned and an error
 *	message is left in interp->result.
 *
 *----------------------------------------------------------------------
 */
int
Blt_GetPictureFilterFromObj(
    Tcl_Interp *interp,
    Tcl_Obj *objPtr,
    PictureFilter **filterPtrPtr)
{
    PictureFilter *fp, *fend;
    Tcl_Obj **objv;
    int objc;
    char *string;

    /* First check if the filter string is a symbolic name. */
    string = Tcl_GetString(objPtr);
    if (isalpha(string[0])) {
	for (fp = filterTable, fend = fp + nFilters; fp < fend; fp++) {
	    if (strcmp(string, fp->name) == 0) {
		*filterPtrPtr = (fp->proc == NULL) ? NULL : fp;
		return TCL_OK;
	    }
	}
	Tcl_AppendResult(interp, "can't find filter \"", string, "\"", 
			 (char *)NULL);
	return TCL_ERROR;
    }
    /* Next see if it's a table (list of numbers) */
    if (Tcl_ListObjGetElements(interp, objPtr, &objc, &objv) != TCL_OK) {
	return TCL_ERROR;	/* Can't split object. */
    }
    if ((objc & 0x1) == 0) {
	Tcl_AppendResult(interp, "# of values in filter table \"", string, 
		"\" must be odd", (char *)NULL);
	return TCL_ERROR;
    }
    if ((objc < 3) || (objc > 11)) {
	Tcl_AppendResult(interp, 
		"# of values in filter table must be between 3 and 11",
		(char *)NULL);
	return TCL_ERROR;
    }
    filterSize = 0;
    { 
	int i;
	double maxWeight;

	maxWeight = 0.0;
	for (i = 0; i < objc; i++) {
	    double value;

	    if (Tcl_GetDoubleFromObj(interp, objv[i], &value) != TCL_OK) {
		return TCL_ERROR;
	    }
	    filterValues[i] = value;
	    value = FABS(value);
	    if (value > maxWeight) {
		maxWeight = value;
	    }
	}
	if (maxWeight > 1.0) {
	    double s;

	    s = 1.0 / (maxWeight * 0.5);
	    for (i = 0; i < objc; i++) {
		filterValues[i] *= s;
	    }
	}
	filterSize = i;
	tableFilter.support = (double)i * 0.5;
    }
    *filterPtrPtr = &tableFilter;
    return TCL_OK;
}

size_t
Blt_ComputeWeights(
    int srcWidth, int destWidth,
    PictureFilter *filterPtr,
    Sample **samplePtrPtr)
{
    Sample *samples;
    double scale;
    int bytesPerSample;

    /* Pre-calculate filter contributions for a row */
    scale = (double)destWidth / (double)srcWidth;
    if (scale < 1.0) {
	Sample *samplePtr;
	double radius, fscale;
	int filterSize, x;

	/* Downsample */

	radius = filterPtr->support / scale;
	fscale = 1.0 / scale;
	filterSize = (int)(radius * 2 + 2);

	bytesPerSample = sizeof(Sample) + 
	    ((filterSize - 1) * sizeof(PixelWeight));
	samples = Blt_Calloc(destWidth, bytesPerSample);
	assert(samples);

	samplePtr = samples;
#define DEBUG 0
#if DEBUG
	fprintf(stderr, "downscale=%g, fscale=%g, radius=%g\n",
		    scale, fscale, radius);
#endif
	for (x = 0; x < destWidth; x++) {
	    PixelWeight *wp;
	    double sum, center, factor;
	    int i, left, right;	/* filter bounds */

	    center = ((double)x + 0.5) * fscale;

	    /* Determine bounds of filter and its density */
	    left = (int)/*floor*/(center - radius);
	    if (left < 0) {
		left = 0;
	    }
	    right = (int)/*floor*/(center + radius);
	    if (right >= srcWidth) {
		right = srcWidth - 1;
	    }
	    samplePtr->start = left;
	    samplePtr->wend = samplePtr->weights + (right - left + 1);

	    sum = 0.0;
	    for (wp = samplePtr->weights, i = left; i <= right; i++, wp++) {
		wp->f32 = (float)
		    (*filterPtr->proc)(((double)i - center) * scale);
		sum += wp->f32;
	    }

	    factor = (sum == 0.0) ? 1.0 : (1.0 / sum);
	    for (wp = samplePtr->weights; wp < samplePtr->wend; wp++) {
		wp->f32 = (float)(wp->f32 * factor);
		wp->i32 = float2si(wp->f32);
	    }
	    samplePtr = (Sample *)((char *)samplePtr + bytesPerSample);
	}
    } else {
	Sample *samplePtr;
	double fscale;
	int filterSize, x;

	/* Upsample */

	filterSize = (int)(filterPtr->support * 2 + 2);
	bytesPerSample = sizeof(Sample) + 
	    ((filterSize - 1) * sizeof(PixelWeight));
	samples = Blt_Calloc(destWidth, bytesPerSample);
	assert(samples);

	fscale = 1.0 / scale;

	samplePtr = samples;
#if DEBUG
	fprintf(stderr, "upscale=%g, fscale=%g, radius=%g\n",
		    scale, fscale, filterPtr->support);
#endif
	for (x = 0; x < destWidth; x++) {
	    PixelWeight *wp;
	    double sum, center, factor;
	    int i, left, right;	/* filter bounds */

	    center = ((double)x + 0.5) * fscale;
	    left = (int)(center - filterPtr->support);
	    if (left < 0) {
		left = 0;
	    }
	    right = (int)(center + filterPtr->support);
	    if (right >= srcWidth) {
		right = srcWidth - 1;
	    }
	    samplePtr->start = left;
	    samplePtr->wend = samplePtr->weights + (right - left + 1);

	    /* Sum the contributions for each pixel in the filter. */
	    sum = 0.0;
	    for (wp = samplePtr->weights, i = left; i <= right;i++, wp++) {
		wp->f32 = (float) (*filterPtr->proc) ((double)i - center);
		sum += wp->f32;
	    }

	    /* The sum of the contributions should not be greater than 1.0 */
	    factor = (sum == 0.0) ? 1.0 : (1.0 / sum);

	    for (wp = samplePtr->weights; wp < samplePtr->wend; wp++) {
#ifdef notdef
		fprintf(stderr, "w[%d]=%g, %g sum=%g\n", 
			wp - samplePtr->weights, 
			wp->f32, wp->f32 * factor, sum);
#endif
		wp->f32 = (float)(wp->f32 * factor);
		/* Convert each weight into a fixed-point scaled integer */
		wp->i32 = float2si(wp->f32);

	    }
	    /* Go to the next sample. */
	    samplePtr = (Sample *)((char *)samplePtr + bytesPerSample);
	}
    }
    *samplePtrPtr = samples;
    return bytesPerSample;
}

static void
ZoomVertically(
    Picture *destPtr,
    Picture *srcPtr, 
    Blt_PictureFilter filter)
{
    Sample *samples, *send;
    int x;
    int bytesPerSample;		/* Size of sample. */

    /* Pre-calculate filter contributions for each row. */
    bytesPerSample = Blt_ComputeWeights(srcPtr->height, destPtr->height, 
	filter, &samples);
    send = (Sample *)((char *)samples + (destPtr->height * bytesPerSample));

    /* Apply filter to each row. */
    for (x = 0; x < srcPtr->width; x++) {
	Pix32 *dp, *srcColumnPtr;
	Sample *samplePtr;

	srcColumnPtr = srcPtr->bits + x;
	dp = destPtr->bits + x;
	for (samplePtr = samples; samplePtr < send; 
	     samplePtr = (Sample *)((char *)samplePtr + bytesPerSample)) {
	    Pix32 *sp;
	    int r, g, b, a;
	    PixelWeight *wp;

	    r = g = b = a = 0;
	    sp = srcColumnPtr + (samplePtr->start * srcPtr->pixelsPerRow);
	    for (wp = samplePtr->weights; wp < samplePtr->wend; wp++) {
		a += wp->i32 * sp->Alpha;
		r += wp->i32 * sp->Red;
		g += wp->i32 * sp->Green;
		b += wp->i32 * sp->Blue;
		sp += srcPtr->pixelsPerRow;
	    }
	    dp->Red = SICLAMP(r);
	    dp->Green = SICLAMP(g);
	    dp->Blue = SICLAMP(b);
	    dp->Alpha = SICLAMP(a);
#ifdef notdef
	    if (dp->Alpha != 0xFF) {
		fprintf(stdout, "v1: alpha=0x%x\n", dp->Alpha);
	    }
#endif
	    dp += destPtr->pixelsPerRow;

	}
    }
    /* Free the memory allocated for filter weights. */
    Blt_Free(samples);
}

static void
ZoomHorizontally(
    Picture *destPtr, 
    Picture *srcPtr,
    Blt_PictureFilter filter)
{
    Sample *samples, *send;
    int y;
    Pix32 *srcRowPtr, *destRowPtr;
    int bytesPerSample;		/* Size of sample. */

    /* Pre-calculate filter contributions for each column. */
    bytesPerSample = Blt_ComputeWeights(srcPtr->width, destPtr->width, 
	filter, &samples);
    send = (Sample *)((char *)samples + (destPtr->width * bytesPerSample));

    /* Apply filter to each column. */
    srcRowPtr = srcPtr->bits;
    destRowPtr = destPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *dp;
	Sample *samplePtr;

	dp = destRowPtr;
	for (samplePtr = samples; samplePtr < send; 
	     samplePtr = (Sample *)((char *)samplePtr + bytesPerSample)) {
	    Pix32 *sp;
	    int r, g, b, a;
	    PixelWeight *wp;

	    r = g = b = a = 0;
	    sp = srcRowPtr + samplePtr->start;
	    for (wp = samplePtr->weights; wp < samplePtr->wend; wp++) {
		a += wp->i32 * sp->Alpha;
		r += wp->i32 * sp->Red;
		g += wp->i32 * sp->Green;
		b += wp->i32 * sp->Blue;
		sp++;
	    }
	    dp->Red = SICLAMP(r);
	    dp->Green = SICLAMP(g);
	    dp->Blue = SICLAMP(b);
	    dp->Alpha = SICLAMP(a);
#ifdef notdef
	    if (dp->Alpha != 0xFF) {
		fprintf(stdout, "h1: alpha=0x%x\n", dp->Alpha);
	    }
#endif
	    dp++;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
    /* Free the memory allocated for horizontal filter weights. */
    Blt_Free(samples);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_ResamplePicture --
 *
 *      Resamples a given picture using 1-D filters and returns
 *	a new picture of the designated size.
 *
 * Results:
 *      Returns the resampled picture. The original picture
 *	is left intact.
 *
 *----------------------------------------------------------------------
 */
void
Blt_ResamplePicture(
    Picture *destPtr,
    Picture *srcPtr,
    Blt_PictureFilter hFilter, 
    Blt_PictureFilter vFilter)
{
    Picture *tmpPtr;

    tmpPtr = Blt_CreatePicture(destPtr->width, srcPtr->height);

    /* 
     * It's usually faster to zoom vertically last.  This has to
     * do with the fact that pictures are stored in contiguous
     * rows.
     */
#ifdef HAVE_X86_ASM
    if (bltUseMMX == 0) {
	Blt_ZoomHorizontallyMMX(tmpPtr, srcPtr, hFilter);
    } else {
	ZoomHorizontally(tmpPtr, srcPtr, hFilter);
    }
#else 
    ZoomHorizontally(tmpPtr, srcPtr, hFilter);
#endif /* HAVE_X86_ASM */
#ifdef HAVE_X86_ASM
    if (bltUseMMX) {
	Blt_ZoomVerticallyMMX(destPtr, tmpPtr, vFilter);
    } else {
	ZoomVertically(destPtr, tmpPtr, vFilter);
    }
#else 
    ZoomVertically(destPtr, tmpPtr, vFilter);
#endif /* HAVE_X86_ASM */
    Blt_FreePicture(tmpPtr);
    destPtr->flags = srcPtr->flags;
    destPtr->flags |= BLT_PICTURE_DIRTY;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_ResamplePhoto --
 *
 *      Resamples a Tk photo image using 1-D filters and writes the
 *      image into another Tk photo.  It is possible for the
 *      source and destination to be the same photo.
 *
 * Results:
 *      The designated destination photo will contain the resampled
 *	picture. The original photo is left intact.
 *
 *---------------------------------------------------------------------- 
 */
void
Blt_ResamplePhoto(
    Tk_PhotoHandle srcPhoto,	/* Source photo image to scale */
    int x, int y,
    int width, int height,
    Tk_PhotoHandle destPhoto,	/* Resulting scaled photo image */
    Blt_PictureFilter hFilter, 
    Blt_PictureFilter vFilter)
{
    Blt_Picture src, dest;
    Tk_PhotoImageBlock destImgBlock;

    Tk_PhotoGetImage(destPhoto, &destImgBlock);
    src = Blt_PhotoAreaToPicture(srcPhoto, x, y, width, height);
    dest = Blt_CreatePicture(destImgBlock.width, destImgBlock.height);
    Blt_ResamplePicture(dest, src, hFilter, vFilter);
    Blt_FreePicture(src);
    Blt_PictureToPhoto(dest, destPhoto);
    Blt_FreePicture(dest);
}

static void
FillScaleTables(
    int srcWidth, int srcHeight, /* Dimension of source. */
    int areaX, int areaY,	/* Origin of requested area. */
    int areaWidth, int areaHeight, /* Dimension of requested area. */
    int destWidth, int destHeight, /* Desired new dimension. */
    int *mapX, int *mapY)	/* (out) Resulting mapping tables. */
{
    int left, right, top, bottom;
    double xScale, yScale;
    int dx, dy;

    left = areaX;
    top = areaY;
    right = MIN(srcWidth, areaX + areaWidth) - 1;
    bottom = MIN(srcHeight, areaY + areaHeight) - 1;

    xScale = (double)areaWidth / (double)destWidth;
    for (dx = 0; dx < destWidth; dx++) {
	int sx;
	
	sx = (int)(xScale * (double)dx);
	sx += left;
	if (sx > right) {
	    sx = right;
	}
	mapX[dx] = sx;
    }
    yScale = (double)areaHeight / (double)destHeight;
    for (dy = 0; dy < destHeight; dy++) {
	int sy;
	
	sy = (int)(yScale * (double)dy);
	sy += top;
	if (sy > bottom) {
	    sy = bottom;
	}
	mapY[dy] = sy;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_ResizePhoto --
 *
 *	Scales the region of the source image to the size of the
 *	destination image.  This routine performs raw scaling of
 *	the image and unlike Blt_ResamplePhoto does not handle
 *	aliasing effects from subpixel sampling. It is possible
 *	for the source and destination to be the same photo.
 *
 * Results:
 *      The designated destination photo will contain the resampled
 *	picture. The original photo is left intact.
 *
 *----------------------------------------------------------------------
 */
void
Blt_ResizePhoto(
    Tk_PhotoHandle srcPhoto,	/* Source photo image to scaled. */
    int srcX, int srcY,		/* Area of source photo to be
				 * scaled. */
    int srcWidth, int srcHeight,
    Tk_PhotoHandle destPhoto)	/* (out) Resulting scaled photo image.
				 * Scaling factors are derived from
				 * the destination photo's
				 * dimensions. */
{
    Picture *destPtr;
    Tk_PhotoImageBlock srcImgBlock, destImgBlock;
    int *mapX, *mapY;
    int ir, ib, ig, ia;

    Tk_PhotoGetImage(destPhoto, &destImgBlock);
    mapX = (int *)Blt_Malloc(sizeof(int) * destImgBlock.width);
    mapY = (int *)Blt_Malloc(sizeof(int) * destImgBlock.height);
    FillScaleTables(srcImgBlock.width, srcImgBlock.height, srcX, srcY, srcWidth, 
	srcHeight, destImgBlock.width, destImgBlock.height, mapX, mapY);

    destPtr = Blt_CreatePicture(destImgBlock.width, destImgBlock.height);
    Tk_PhotoGetImage(srcPhoto, &srcImgBlock);
    ir = srcImgBlock.offset[0];
    ig = srcImgBlock.offset[1];
    ib = srcImgBlock.offset[2];
    ia = srcImgBlock.offset[3];

    if (srcImgBlock.pixelSize == 4) {
	Pix32 *destRowPtr;
	int x, y;

	destRowPtr = destPtr->bits;
	for (y = 0; y < destImgBlock.height; y++) {
	    Pix32 *dp;
	    unsigned char *srcRowPtr;

	    dp = destRowPtr;
	    srcRowPtr = srcImgBlock.pixelPtr + (mapY[y] * srcImgBlock.pitch);
	    for (x = 0; x < destImgBlock.width; x++) {
		unsigned char *sp;

		sp = srcRowPtr + (mapX[x] * srcImgBlock.pixelSize);
		dp->Red = sp[ir];
		dp->Green = sp[ig];
		dp->Blue = sp[ib];
		dp->Alpha = sp[ia];
		dp++;
	    }
	    destRowPtr += destPtr->pixelsPerRow;
	}
    } else if (srcImgBlock.pixelSize == 3) {
	Pix32 *destRowPtr;
	int x, y;

	destRowPtr = destPtr->bits;
	for (y = 0; y < destImgBlock.height; y++) {
	    Pix32 *dp;
	    unsigned char *srcRowPtr;

	    dp = destRowPtr;
	    srcRowPtr = srcImgBlock.pixelPtr + (mapY[y] * srcImgBlock.pitch);
	    for (x = 0; x < destImgBlock.width; x++) {
		unsigned char *sp;

		sp = srcRowPtr + (mapX[x] * srcImgBlock.pixelSize);
		dp->Red = sp[ir];
		dp->Green = sp[ig];
		dp->Blue = sp[ib];
		dp->Alpha = ALPHA_OPAQUE;
		dp++;
	    }
	    destRowPtr += destPtr->pixelsPerRow;
	}
    } else  {
	Pix32 *destRowPtr;
	int x, y;

	destRowPtr = destPtr->bits;
	for (y = 0; y < destImgBlock.height; y++) {
	    Pix32 *dp;
	    unsigned char *srcRowPtr;

	    dp = destRowPtr;
	    srcRowPtr = srcImgBlock.pixelPtr + (mapY[y] * srcImgBlock.pitch);
	    for (x = 0; x < destImgBlock.width; x++) {
		unsigned char *sp;

		sp = srcRowPtr + (mapX[x] * srcImgBlock.pixelSize);
		dp->Red = dp->Green = dp->Blue = sp[ir];
		dp->Alpha = ALPHA_OPAQUE;
		dp++;
	    }
	    destRowPtr += destPtr->pixelsPerRow;
	}
    }	
    Blt_Free(mapX);
    Blt_Free(mapY);
    Blt_PictureToPhoto(destPtr, destPhoto);
    Blt_FreePicture(destPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_ScalePicture --
 *
 *	Scales the region of the source picture to the size requested.
 *	This routine performs raw scaling of the image and unlike
 *	Blt_ResamplePhoto does not do any filtering.
 *
 *          src picture
 *      +===================+
 *      |                   |              dest picture
 *      |                   |            +==============+
 *      | x,y               |            |              |
 *      |  *-------*        |            |              |
 *      |  | area  |        |            |              |
 *      |  |       |        |            |              |
 *      |  |       | height |            |              | 
 *      |  |       |        |            |              |
 *      |  *-------*        |            |              |
 *      |    width          |            |              |
 *      |                   |            +==============+
 *      |                   |
 *      +===================+
 *
 *		x ratio = dest width / area width
 *		y ratio = dest height / area height
 *		
 *
 * Results:
 *      Returns the new resized picture.  The original picture is
 *      left intact.
 *
 *----------------------------------------------------------------------
 */
Picture *
Blt_ScalePicture(
    Picture *srcPtr,		/* Source picture to be scaled. */
    int srcX, int srcY,		/* Area of source picture to scaled. */
    int srcWidth, int srcHeight,
    int reqWidth, int reqHeight)/* Requested dimensions of the scaled
				 * picture. */
{
    Picture *destPtr;
    Pix32 *destRowPtr;
    int *mapX, *mapY;
    int y;

    mapX = (int *)Blt_Malloc(sizeof(int) * reqWidth);
    mapY = (int *)Blt_Malloc(sizeof(int) * reqHeight);
    FillScaleTables(srcPtr->width, srcPtr->height, srcX, srcY, srcWidth, 
	srcHeight, reqWidth, reqHeight, mapX, mapY);

    destPtr = Blt_CreatePicture(reqWidth, reqHeight);
    destRowPtr = destPtr->bits;
    for (y = 0; y < reqHeight; y++) {
	Pix32 *dp;
	Pix32 *srcRowPtr;
	int x;

	dp = destRowPtr;
	srcRowPtr = srcPtr->bits + (srcPtr->pixelsPerRow * mapY[y]);
	for (x = 0; x < reqWidth; x++) {
	    Pix32 *sp;

	    sp = srcRowPtr + mapX[x];
	    dp->color = sp->color; /* Copy the pixel. */
	    dp++;
	}
	destRowPtr += destPtr->pixelsPerRow;
    }
    Blt_Free(mapX), Blt_Free(mapY);
    destPtr->flags = (srcPtr->flags | BLT_PICTURE_DIRTY);
    return destPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_ScalePictureArea --
 *
 *	Scales the region of the source picture to the size of the
 *	destination image.  This routine performs raw scaling of the
 *	image and unlike Blt_ResamplePhoto does not perform any
 *	antialiasing.
 *
 * Results:
 *      Returns the new resized picture.  The original picture is left
 *      intact.
 *
 *----------------------------------------------------------------------
 */
Picture *
Blt_ScalePictureArea(
    Picture *srcPtr,		/* Source picture to be scaled. */
    int areaX,			/* Origin of subimage in destination. */
    int areaY,			
    int areaWidth,		/* Dimension of subimage. */
    int areaHeight,
    int destWidth, 
    int destHeight)		/* Dimensions of the entire scaled
				   image. */
{
    Picture *destPtr;
    Pix32 *srcRowPtr, *destRowPtr;
    double xScale, yScale;
    int *mapX, *mapY;
    int x, y;

    xScale = (double)srcPtr->width / (double)destWidth;
    yScale = (double)srcPtr->height / (double)destHeight;
    mapX = Blt_Malloc(sizeof(int) * areaWidth);
    mapY = Blt_Malloc(sizeof(int) * areaHeight);

    /* Precompute scaling factors for each row and column. */
    for (x = 0; x < areaWidth; x++) {
	int sx;

	sx = (int)(xScale * (double)(x + areaX));
	if (sx >= srcPtr->width) {
	    sx = srcPtr->width - 1;
	} 
	mapX[x] = sx;
    }
    for (y = 0; y < areaHeight; y++) {
	int sy;

	sy = (int)(yScale * (double)(y + areaY));
	if (sy > srcPtr->height) {
	    sy = srcPtr->height - 1;
	} 
	mapY[y] = sy;
    }

    destPtr = Blt_CreatePicture(areaWidth, areaHeight);
    destRowPtr = destPtr->bits;
    for (y = 0; y < areaHeight; y++) {
	Pix32 *dp;

	dp = destRowPtr;
	srcRowPtr = srcPtr->bits + (srcPtr->pixelsPerRow * mapY[y]);
	for (x = 0; x < areaWidth; x++) {
	    Pix32 *sp;

	    sp = srcRowPtr + mapX[x];
	    dp->color = sp->color; /* Copy the pixel. */
	    dp++;
	}
	destRowPtr += destPtr->pixelsPerRow;
    }
    Blt_Free(mapX), Blt_Free(mapY);
    return destPtr;
}

#ifdef notdef
/*
 * FIXME: Boundary handling could be better (pixels are replicated).
 *	  It's slow. Take boundary tests out of inner loop.
 */
Picture *
Blt_ConvolvePicture(
    Blt_Picture srcPict,
    Filter2D *filterPtr)
{
    Blt_Picture destPict;
    Pix32 *sp, *dp;
#define MAXROWS	24
    int sx, sy, dx, dy;
    int x, y;
    double red, green, blue;
    int width, height;
    int radius;
    register double *valuePtr;

    width = Blt_PictureWidth(srcPict);
    height = Blt_PictureHeight(srcPict);

    destPict = Blt_CreatePicture(width, height);
    radius = (int)filterPtr->support;
    if (radius < 1) {
	radius = 1;
    }
    dp = Blt_PictureBits(destPict);
    for (dy = 0; dy < height; dy++) {
	for (dx = 0; dx < width; dx++) {
	    red = green = blue = 0.0;
	    valuePtr = filterPtr->kernel;
	    for (sy = (dy - radius); sy <= (dy + radius); sy++) {
		y = sy;
		if (y < 0) {
		    y = 0;
		} else if (y >= height) {
		    y = height - 1;
		}
		for (sx = (dx - radius); sx <= (dx + radius); sx++) {
		    x = sx;
		    if (x < 0) {
			x = 0;
		    } else if (sx >= width) {
			x = width - 1;
		    }
		    sp = Blt_PicturePixel(srcPict, x, y);
		    red += *valuePtr * (double)sp->Red;
		    green += *valuePtr * (double)sp->Green;
		    blue += *valuePtr * (double)sp->Blue;
#if DEBUG
		    fprintf(stderr, "%d,%d = r=%f,g=%f,b=%f\n", x, y,
			red, green, blue);
#endif
		    valuePtr++;
		}
	    }
	    red /= filterPtr->sum;
	    green /= filterPtr->sum;
	    blue /= filterPtr->sum;
	    dp->Red = (unsigned char)CLAMP(red);
	    dp->Green = (unsigned char)CLAMP(green);
	    dp->Blue = (unsigned char)CLAMP(blue);
	    dp->Alpha = ALPHA_OPAQUE;
	    dp++;
	}
    }
    return destPict;
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * Blt_SnapPhoto --
 *
 *      Takes a snapshot of an X drawable (pixmap or window) and
 *	writes it to an existing Tk photo image.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side Effects:
 *	The named Tk photo is updated with the snapshot.
 *
 *----------------------------------------------------------------------
 */
int
Blt_SnapPhoto(
    Tcl_Interp *interp,		/* Interpreter to report errors back to */
    Tk_Window tkwin,
    Drawable drawable,		/* Window or pixmap to be snapped */
    int x, int y,		/* Offset of image from drawable origin. */
    int width, int height,	/* Dimension of the drawable */
    int destWidth, 
    int destHeight,		/* Desired size of the Tk photo */
    char *photoName,		/* Name of an existing Tk photo image. */
    double inputGamma)
{
    Tk_PhotoHandle photo;	/* The photo image to write into. */
    Blt_Picture picture;

    photo = Tk_FindPhoto(interp, photoName);
    if (photo == NULL) {
	Tcl_AppendResult(interp, "can't find photo \"", photoName, "\"", 
		(char *)NULL);
	return TCL_ERROR;
    }
    picture = Blt_DrawableToPicture(tkwin, drawable, x, y, width, height, 
	inputGamma);
    if (picture == NULL) {
	Tcl_AppendResult(interp,
	    "can't grab window or pixmap (possibly obscured?)", (char *)NULL);
	return TCL_ERROR;	/* Can't grab window image */
    }
    if ((destWidth != width) || (destHeight != height)) {
	Blt_Picture dest;

	/*
	 * The requested size for the destination image is different than
	 * that of the source snapshot.  Resample the image as necessary.
	 * We'll use a cheap box filter. I'm assuming that the destination
	 * image will typically be smaller than the original.
	 */
	dest = Blt_CreatePicture(destWidth, destHeight);
	Blt_ResamplePicture(dest, picture, bltBoxFilter, bltBoxFilter);
	Blt_FreePicture(picture);
	picture = dest;
    }
    Blt_PictureToPhoto(picture, photo);
    Blt_FreePicture(picture);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_SnapPhoto --
 *
 *      Takes a snapshot of an X drawable (pixmap or window) and
 *	writes it to an existing Tk photo image.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side Effects:
 *	The named Tk photo is updated with the snapshot.
 *
 *----------------------------------------------------------------------
 */
int
Blt_SnapPicture(
    Tcl_Interp *interp,		/* Interpreter to report errors back to */
    Tk_Window tkwin,
    Drawable drawable,		/* Window or pixmap to be snapped */
    int x, int y,		/* Offset of image from drawable origin. */
    int width, int height,	/* Dimension of the drawable */
    int destWidth, 
    int destHeight,		/* Desired size of the picture. */
    char *imageName,		/* Name of an existing picture image. */
    double inputGamma)
{
    Blt_Picture picture;

    picture = Blt_DrawableToPicture(tkwin, drawable, x, y, width, height, 
	inputGamma);
    if (picture == NULL) {
	Tcl_AppendResult(interp,
	    "can't grab window or pixmap (possibly obscured?)", (char *)NULL);
	return TCL_ERROR;	/* Can't grab window image */
    }
    if ((destWidth != width) || (destHeight != height)) {
	Blt_Picture dest;

	/*
	 * The requested size for the destination image is different than
	 * that of the source snapshot.  Resample the image as necessary.
	 * We'll use a cheap box filter. I'm assuming that the destination
	 * image will typically be smaller than the original.
	 */
	dest = Blt_CreatePicture(destWidth, destHeight);
	Blt_ResamplePicture(dest, picture, bltBoxFilter, bltBoxFilter);
	Blt_FreePicture(picture);
	picture = dest;
    }
    if (Blt_ResetPicture(interp, imageName, picture) == TCL_OK) {
	return TCL_OK;
    }
    Blt_FreePicture(picture);
    return TCL_ERROR;
}

/* 
 * --------------------------------------------------------------------------
 *
 * ShearY --
 *
 *	Shears a row horizontally.  Antialiasing limited to filtering
 *	two adjacent pixels.  So the shear angle must be between +-45
 *	degrees.
 *	
 * Results:   
 *	None.
 *
 * Side Effects:
 *	The sheared image is drawn into the destination picture.
 *
 * --------------------------------------------------------------------------
 */
static void 
ShearY(
    Picture *destPtr, 
    Picture *srcPtr,
    int y,			/* Designates the row to be sheared */
    int offset,			/* Difference between  of  */
    double frac,
    Pix32 *bgColorPtr)
{
    Pix32 *sp, *dp;
    Pix32 *srcRowPtr, *destRowPtr;
    int x, dx;
    int oldRed, oldGreen, oldBlue, oldAlpha;
    int ifrac;

    destRowPtr = destPtr->bits + (y * destPtr->pixelsPerRow);
    srcRowPtr = srcPtr->bits + (y * srcPtr->pixelsPerRow);

    for (dp = destRowPtr, x = 0; x < offset; x++, dp++) { 
        dp->color = bgColorPtr->color;
    }

    dp = destRowPtr + offset;
    sp = srcRowPtr;
    dx = offset;

    oldRed = uchar2si(bgColorPtr->Red);
    oldGreen = uchar2si(bgColorPtr->Green);
    oldBlue = uchar2si(bgColorPtr->Blue);
    oldAlpha = uchar2si(bgColorPtr->Alpha);

    ifrac = float2si(frac);
    for (x = 0; x < srcPtr->width; x++, dx++) { /* Loop through row pixels */
	int leftRed, leftGreen, leftBlue, leftAlpha;

	leftRed = sp->Red * ifrac;
	leftGreen = sp->Green * ifrac;
	leftBlue = sp->Blue * ifrac;
	leftAlpha = sp->Alpha * ifrac;
        if ((dx >= 0) && (dx < destPtr->width)) {
	    int r, b, g, a;

	    r = uchar2si(sp->Red) - (leftRed - oldRed);
	    g = uchar2si(sp->Green) - (leftGreen - oldGreen);
	    b = uchar2si(sp->Blue) - (leftBlue - oldBlue);
	    a = uchar2si(sp->Alpha) - (leftAlpha - oldAlpha);
	    dp->Red = SICLAMP(r);
	    dp->Green = SICLAMP(g);
	    dp->Blue = SICLAMP(b);
	    dp->Alpha = SICLAMP(a);
        }
	oldRed = leftRed;
	oldGreen = leftGreen;
	oldBlue = leftBlue;
	oldAlpha = leftAlpha;
	sp++, dp++;
    }
    x = srcPtr->width + offset;  
    dp = destPtr->bits + (y * destPtr->pixelsPerRow) + x;
    if (x < destPtr->width) {
	int leftRed, leftGreen, leftBlue, leftAlpha;
	int r, b, g, a;

	leftRed = uchar2si(bgColorPtr->Red);
	leftGreen = uchar2si(bgColorPtr->Green);
	leftBlue = uchar2si(bgColorPtr->Blue);
	leftAlpha = uchar2si(bgColorPtr->Alpha);

	r = leftRed + oldRed - (bgColorPtr->Red * ifrac);
	g = leftGreen + oldGreen - (bgColorPtr->Green * ifrac);
	b = leftBlue + oldBlue - (bgColorPtr->Blue * ifrac);
	a = leftAlpha + oldAlpha - (bgColorPtr->Alpha * ifrac);
	dp->Red = SICLAMP(r);
	dp->Green = SICLAMP(g);
	dp->Blue = SICLAMP(b);
	dp->Alpha = SICLAMP(a);
	dp++;
    }
    for (x++; x < destPtr->width; x++, dp++) {
        dp->color = bgColorPtr->color;
    }
}  

/* 
 * --------------------------------------------------------------------------
 *
 * ShearX --
 *
 *	Shears a column. Antialiasing is limited to filtering two
 *	adjacent pixels.  So the shear angle must be between +-45
 *	degrees.
 *	
 * Results:   
 *	None.
 *
 * Side Effects:
 *	The sheared image is drawn into the destination picture.
 *
 * -------------------------------------------------------------------------- 
 */
static void
ShearX(
    Picture *destPtr, 
    Picture *srcPtr,
    int x,			/* Column in source image to be sheared. */
    int offset,			/* Offset of */
    double frac,		/* Fraction of subpixel. */
    Pix32 *bgColorPtr)
{
    Pix32 *sp, *dp;
    int y, destY;
    int oldRed, oldGreen, oldBlue, oldAlpha;
    int ifrac;

    dp = destPtr->bits + x;
    for (y = 0; y < offset; y++) {
        dp->color = bgColorPtr->color;
	dp += destPtr->pixelsPerRow;
    }

    oldRed = uchar2si(bgColorPtr->Red);
    oldGreen = uchar2si(bgColorPtr->Green);
    oldBlue = uchar2si(bgColorPtr->Blue);
    oldAlpha = uchar2si(bgColorPtr->Alpha);

    dp = destPtr->bits + (offset * destPtr->pixelsPerRow) + x;
    sp = srcPtr->bits + x;
    ifrac = float2si(frac);
    for (destY = offset, y = 0; y < srcPtr->height; y++, destY++) {
	int leftRed, leftGreen, leftBlue, leftAlpha;

	leftRed = sp->Red * ifrac;
	leftGreen = sp->Green * ifrac;
	leftBlue = sp->Blue * ifrac;
	leftAlpha = sp->Alpha * ifrac;
        if ((destY >= 0) && (destY < destPtr->height)) {
	    int r, g, b, a;

	    r = uchar2si(sp->Red) - (leftRed - oldRed);
	    g = uchar2si(sp->Green) - (leftGreen - oldGreen);
	    b = uchar2si(sp->Blue) - (leftBlue - oldBlue);
	    a = uchar2si(sp->Alpha) - (leftAlpha - oldAlpha);
	    dp->Red = SICLAMP(r);
	    dp->Green = SICLAMP(g);
	    dp->Blue = SICLAMP(b);
	    dp->Alpha = SICLAMP(a);
        }
	oldRed = leftRed;
	oldGreen = leftGreen;
	oldBlue = leftBlue;
	oldAlpha = leftAlpha;
	sp += srcPtr->pixelsPerRow; 
	dp += destPtr->pixelsPerRow;
    }
    y = srcPtr->height + offset;  

    dp = destPtr->bits + (y * destPtr->pixelsPerRow) + x;
    if (y < destPtr->height) {
	int leftRed, leftGreen, leftBlue, leftAlpha;
	int r, g, b, a;

	leftRed = uchar2si(bgColorPtr->Red);
	leftGreen = uchar2si(bgColorPtr->Green);
	leftBlue = uchar2si(bgColorPtr->Blue);
	leftAlpha = uchar2si(bgColorPtr->Alpha);

	r = leftRed + oldRed - (bgColorPtr->Red * ifrac);
	g = leftGreen + oldGreen - (bgColorPtr->Green * ifrac);
	b = leftBlue + oldBlue - (bgColorPtr->Blue  * ifrac);
	a = leftAlpha + oldAlpha - (bgColorPtr->Alpha  * ifrac);
	dp->Red = SICLAMP(r);
	dp->Green = SICLAMP(g);
	dp->Blue = SICLAMP(b);
	dp->Alpha = SICLAMP(a);
	dp += destPtr->pixelsPerRow;
    }
    
    for (y++; y < destPtr->height; y++) {
	dp->color = bgColorPtr->color;
	dp += destPtr->pixelsPerRow; 
    }
}  

/* 
 * ---------------------------------------------------------------------------
 *
 * Rotate45 --
 *
 *	Rotates an image by a given angle.  The angle must be in the
 *	range -45.0 to 45.0 inclusive.  Anti-aliasing filtering is
 *	performed on two adjacent pixels, so the angle can't be so
 *	great as to force a sheared pixel to occupy 3 destination
 *	pixels.  Performs a three shear rotation described below.
 *
 *	Reference: Alan W. Paeth, "A Fast Algorithm for General Raster
 *		   Rotation", Graphics Gems, pp 179-195.  
 *
 *
 * Results:  
 *	Returns a newly allocated rotated image.
 *
 * --------------------------------------------------------------------------- 
 */
static Picture *
Rotate45(
    Picture *srcPtr,
    double angle,
    Pix32 *bgColorPtr)
{
    Picture *shear1Ptr, *shear2Ptr, *destPtr;
    double  skewf;
    double sinTheta, cosTheta, tanTheta;
    int skewi;
    int shearWidth, shearHeight;
    int x, y;

    sinTheta = sin(angle);
    cosTheta = cos(angle);
    tanTheta = tan(angle * 0.5);
    
    shearWidth = srcPtr->width + (int)(srcPtr->height * FABS(tanTheta));
    shearHeight = srcPtr->height;

    /* 1st shear */

    shear1Ptr = Blt_CreatePicture(shearWidth, shearHeight);
    assert(shear1Ptr);

    if (tanTheta >= 0.0) {	/* Positive angle */
	for (y = 0; y < shearHeight; y++) {  
	    skewf = (y + 0.5) * tanTheta;
	    skewi = (int)floor(skewf);
	    ShearY(shear1Ptr, srcPtr, y, skewi, skewf - skewi, bgColorPtr);
	}
    } else {			/* Negative angle */
	for (y = 0; y < shearHeight; y++) {  
	    skewf = ((y - srcPtr->height) + 0.5) * tanTheta;
	    skewi = (int)floor(skewf);
	    ShearY(shear1Ptr, srcPtr, y, skewi, skewf - skewi, bgColorPtr);
	}
    }
    shearHeight = 
	(int)(srcPtr->width * FABS(sinTheta) + srcPtr->height * cosTheta) + 1;

    shear2Ptr = Blt_CreatePicture(shearWidth, shearHeight);
    assert(shear2Ptr);

    /* 2nd shear */

    if (sinTheta > 0.0) {	/* Positive angle */
        skewf = (srcPtr->width - 1) * sinTheta;
    } else {			/* Negative angle */
        skewf = (srcPtr->width - shearWidth) * -sinTheta;
    }
    for (x = 0; x < shearWidth; x++) {
	skewi = (int)floor(skewf);
        ShearX(shear2Ptr, shear1Ptr, x, skewi, skewf - skewi, bgColorPtr);
	skewf -= sinTheta;
    }

    Blt_FreePicture(shear1Ptr);

    /* 3rd shear */

    shearWidth = 
	(int)(srcPtr->height * FABS(sinTheta) + srcPtr->width * cosTheta) + 1;

    destPtr = Blt_CreatePicture(shearWidth, shearHeight);
    assert(destPtr);

    if (sinTheta >= 0.0) {	/* Positive angle */
        skewf = (srcPtr->width - 1) * sinTheta * -tanTheta;
    } else {			/* Negative angle */
        skewf = tanTheta * ((srcPtr->width - 1) * -sinTheta - (shearHeight-1));
    }
    for (y = 0; y < shearHeight; y++) {
	skewi = (int)floor(skewf);
        ShearY(destPtr, shear2Ptr, y, skewi, skewf - skewi, bgColorPtr);
	skewf += tanTheta;
    }
    Blt_FreePicture(shear2Ptr);
    destPtr->flags |= BLT_PICTURE_BLEND;
    return destPtr;      
}

/* 
 * ---------------------------------------------------------------------------
 *
 * Blt_CopyPicture --
 *
 *	Creates a copy of the given picture.  
 *
 * Results:  
 *	Returns the new copy.
 *
 * --------------------------------------------------------------------------- 
 */
void
Blt_CopyPicture(Picture *destPtr, Picture *srcPtr)
{
    if (srcPtr != destPtr) {
	if (!Blt_ReallocatePicture(destPtr, srcPtr->width, srcPtr->height, 0)) {
	    fprintf(stderr, "can't resize picture");
	    return;
	} 
	memcpy(destPtr->bits, srcPtr->bits, 
	       srcPtr->pixelsPerRow * srcPtr->height * sizeof(Pix32));
	destPtr->flags = (srcPtr->flags | BLT_PICTURE_DIRTY);
    }
}

/* 
 * ---------------------------------------------------------------------------
 *
 * Blt_ClonePicture --
 *
 *	Creates a copy of the given picture.  
 *
 * Results:  
 *	Returns the new copy.
 *
 * --------------------------------------------------------------------------- 
 */
Picture *
Blt_ClonePicture(Picture *srcPtr)
{
    Picture *destPtr;

    destPtr = Blt_CreatePicture(srcPtr->width, srcPtr->height);
    Blt_CopyPicture(destPtr, srcPtr);
    return destPtr;
}

/* 
 * ---------------------------------------------------------------------------
 *
 * Rotate90 --
 *
 *	Rotates the given picture by 90 degrees.  This is part
 *	of the special case right-angle rotations that do not create
 *	subpixel aliasing.
 *
 * Results:  
 *	Returns a newly allocated, rotated picture.
 *
 * --------------------------------------------------------------------------- 
 */
static Picture *
Rotate90(Picture *srcPtr)
{
    Picture *destPtr;
    Pix32 *srcRowPtr;
    int offset;
    int x, y;

    destPtr = Blt_CreatePicture(srcPtr->height, srcPtr->width);
    offset = (destPtr->height - 1) * destPtr->pixelsPerRow;
    srcRowPtr = srcPtr->bits;
    for (x = 0; x < destPtr->width; x++) {
	Pix32 *dp, *sp;

	sp = srcRowPtr;
	dp = destPtr->bits + offset + x;
	for (y = 0; y < destPtr->height; y++) {
	    *dp = *sp++;
	    dp -= destPtr->pixelsPerRow;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
    }
    destPtr->flags = srcPtr->flags;
    return destPtr;
}

/* 
 * ---------------------------------------------------------------------------
 *
 * Rotate180 --
 *
 *	Rotates the given picture by 180 degrees.  This is one of
 *	the special case orthogonal rotations that do not create
 *	subpixel aliasing.
 *
 * Results:  
 *	Returns a newly allocated, rotated picture.
 *
 * --------------------------------------------------------------------------- 
 */
static Picture *
Rotate180(Picture *srcPtr)
{
    Picture *destPtr;
    Pix32 *srcRowPtr;
    int offset;
    int x, y;

    destPtr = Blt_CreatePicture(srcPtr->width, srcPtr->height);
    offset = (destPtr->height - 1) * destPtr->pixelsPerRow;
    srcRowPtr = srcPtr->bits;
    for (y = 0; y < destPtr->height; y++) {
	Pix32 *dp, *sp;

	sp = srcRowPtr;
	dp = destPtr->bits + offset + destPtr->width - 1;
	for (x = 0; x < destPtr->width; x++) {
	    *dp-- = *sp++;
	}
	offset -= destPtr->pixelsPerRow;
	srcRowPtr += srcPtr->pixelsPerRow;
    }
    destPtr->flags = srcPtr->flags;
    return destPtr;
}

/* 
 * ---------------------------------------------------------------------------
 *
 * Rotate270 --
 *
 *	Rotates the given picture by 270 degrees.  This is part
 *	of the special case right-angle rotations that do not create
 *	subpixel aliasing.
 *
 * Results:  
 *	Returns a newly allocated, rotated picture.
 *
 * --------------------------------------------------------------------------- 
 */
static Picture *
Rotate270(Picture *srcPtr)
{
    Picture *destPtr;
    Pix32 *srcRowPtr;
    int x, y;

    destPtr = Blt_CreatePicture(srcPtr->height, srcPtr->width);

    srcRowPtr = srcPtr->bits;
    for (x = destPtr->width - 1; x >= 0; x--) {
	Pix32 *sp, *dp;

	sp = srcRowPtr;
	dp = destPtr->bits + x;
	for (y = 0; y < destPtr->height; y++) {
	    *dp = *sp++;
	    dp += destPtr->pixelsPerRow;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
    }
    destPtr->flags = srcPtr->flags;
    return destPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_RotatePicture --
 *
 *	Rotates a picture by a given # of degrees.
 *
 * Results:
 *	Returns a newly allocated, rotated picture.
 *
 *----------------------------------------------------------------------
 */
Picture *
Blt_RotatePicture(
    Picture *srcPtr,
    double angle)
{
    Picture *destPtr, *tmpPtr;
    int quadrant;
    
    tmpPtr = srcPtr;		/* Suppress compiler warning. */

    /* Make the angle positive between 0 and 360 degrees. */ 
    angle = FMOD(angle, 360.0);
    if (angle < 0.0) {
	angle += 360.0;
    }
    quadrant = ROTATE_0;
    if ((angle > 45.0) && (angle <= 135.0)) {
        quadrant = ROTATE_90;
        angle -= 90.0;
    } else if ((angle > 135.0) && (angle <= 225.0)) { 
        quadrant = ROTATE_180;
        angle -= 180.0;
    } else if ((angle > 225.0) && (angle <= 315.0)) { 
        quadrant = ROTATE_270;
        angle -= 270.0;
    } else if (angle > 315.0) {
	angle -= 360.0;
    }
    /* 
     * If necessary, create a temporary image that's rotated by a
     * right-angle.  We'll then rotate this picture between -45 to
     * 45 degrees to arrive at its final angle.
     */
    switch (quadrant) {
    case ROTATE_270:		/* 270 degrees */
	tmpPtr = Rotate270(srcPtr);
	break;

    case ROTATE_90:		/* 90 degrees */
	tmpPtr = Rotate90(srcPtr);
	break;

    case ROTATE_180:		/* 180 degrees */
	tmpPtr = Rotate180(srcPtr);
	break;

    case ROTATE_0:		/* 0 degrees */
	if (angle == 0.0) {
	    tmpPtr = Blt_ClonePicture(srcPtr); /* Make a copy of the source. */
	} 
	break;
    }

    assert((angle >= -45.0) && (angle <= 45.0));

    destPtr = tmpPtr;
    if (angle != 0.0) {
	angle = (angle / 180.0) * M_PI;
	destPtr = Rotate45(tmpPtr, angle, 0x00000000);
	if (tmpPtr != srcPtr) {
	    Blt_FreePicture(tmpPtr);
	}
    } 
    return destPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_RotatePicture --
 *
 *	Rotates a picture by a given # of degrees.
 *
 * Results:
 *	Returns a newly allocated, rotated picture.
 *
 *----------------------------------------------------------------------
 */
void
Blt_FlipPicture(Picture *srcPtr, int vertically)
{
    if (vertically) {
	Pix32 *s1RowPtr, *s2RowPtr;
	int y;
	
	s1RowPtr = srcPtr->bits;
	s2RowPtr = srcPtr->bits + ((srcPtr->height - 1) * srcPtr->pixelsPerRow);
	for (y = 0; y < srcPtr->height / 2; y++) {
	    Pix32 *s1, *s2, *send;
	    
	    s1 = s1RowPtr, s2 = s2RowPtr;
	    for (send = s1 + srcPtr->width; s1 < send; s1++, s2++) {
		Pix32 tmp;
		
		tmp.color = s1->color;
		s1->color = s2->color;
		s2->color = tmp.color;
	    }
	    s1RowPtr += srcPtr->pixelsPerRow;
	    s2RowPtr -= srcPtr->pixelsPerRow;
	}
    } else {
	Pix32 *s1ColumnPtr, *s2ColumnPtr;
	int x;
	
	s1ColumnPtr = srcPtr->bits;
	s2ColumnPtr = srcPtr->bits + (srcPtr->width - 1);
	for (x = 0; x < srcPtr->width / 2; x++) {
	    Pix32 *s1, *s2, *send;
	    
	    s1 = s1ColumnPtr, s2 = s2ColumnPtr;
	    for (send = s1 + (srcPtr->height * srcPtr->pixelsPerRow); s1 < send;
		 s1 += srcPtr->pixelsPerRow, s2 += srcPtr->pixelsPerRow) {
		Pix32 tmp;
		
		tmp.color = s1->color;
		s1->color = s2->color;
		s2->color = tmp.color;
	    }
	    s1ColumnPtr++;
	    s2ColumnPtr--;
	}
    }
    srcPtr->flags |= BLT_PICTURE_DIRTY;
}

#define NC		256
enum ColorIndices { RED, GREEN, BLUE };

#define R0	(cubePtr->r0)
#define R1	(cubePtr->r1)
#define G0	(cubePtr->g0)
#define G1	(cubePtr->g1)
#define B0	(cubePtr->b0)
#define B1	(cubePtr->b1)

typedef struct {
    int r0, r1;			/* min, max values: 
				 * min exclusive max inclusive */
    int g0, g1;
    int b0, b1;
    int vol;
} Cube;

typedef unsigned int (*ColorLookupTable)[33][33];

/*
 *----------------------------------------------------------------------
 *
 * Histogram is in elements 1..HISTSIZE along each axis,
 * element 0 is for base or marginal value
 * NB: these must start out 0!
 *----------------------------------------------------------------------
 */
typedef struct {
    long int wt[33][33][33];	/* # pixels in voxel */
    long int mR[33][33][33];	/* Sum over voxel of red pixel values */
    long int mG[33][33][33];	/* Sum over voxel of green pixel values */
    long int mB[33][33][33];	/* Sum over voxel of blue pixel values */
    long int gm2[33][33][33];	/* Variance */
} PictureStatistics;

/*
 * Build 3-D color histogram of counts, R/G/B, c^2
 */
static PictureStatistics *
GetPictureStatistics(Picture *srcPtr)
{
#define MAX_INTENSITIES	256
    unsigned int sqr[MAX_INTENSITIES];
    Pix32 *srcRowPtr;
    int i, x, y;
    PictureStatistics *s;

    s = Blt_Calloc(1, sizeof(PictureStatistics));
    assert(s);

    /* Precompute table of squares. */
    for (i = 0; i < MAX_INTENSITIES; i++) {
	sqr[i] = i * i;
    }

    srcRowPtr = srcPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp;

	for (sp = srcRowPtr, x = 0; x < srcPtr->width; x++, sp++) {
	    int r, g, b;

	    /*
	     * Reduce the number of bits (5) per color component. This
	     * will keep the table size (2^15) reasonable without
	     * perceptually affecting the final image.
	     */
	    r = (sp->Red >> 3) + 1;
	    g = (sp->Green >> 3) + 1;
	    b = (sp->Blue >> 3) + 1;
	    
	    s->wt[r][g][b] += 1;
	    s->mR[r][g][b] += sp->Red;
	    s->mG[r][g][b] += sp->Green;
	    s->mB[r][g][b] += sp->Blue;
	    s->gm2[r][g][b] += sqr[sp->Red] + sqr[sp->Green] + sqr[sp->Blue];
	}
	srcRowPtr += srcPtr->pixelsPerRow;
    }
    return s;
}

/*
 *----------------------------------------------------------------------
 *
 *  At conclusion of the histogram step, we can interpret
 *	wt[r][g][b] = sum over voxel of P(c)
 *	mR[r][g][b] = sum over voxel of r*P(c)  ,  similarly for mG, mB
 *	m2[r][g][b] = sum over voxel of c^2*P(c)
 * 
 *  Actually each of these should be divided by 'size' to give the
 *  usual interpretation of P() as ranging from 0 to 1, but we needn't
 *  do that here.
 * 
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 * 
 *     We now convert histogram into moments so that we can rapidly
 *     calculate the sums of the above quantities over any desired
 *     box.
 * 
 *----------------------------------------------------------------------
 */

/* Compute cumulative moments. */
static void
M3d(PictureStatistics *s)
{
    unsigned char i, r, g, b, r0;
    long int area[33], rArea[33], gArea[33], bArea[33];
    unsigned int area2[33];

    for (r = 1, r0 = 0; r < 33; r++, r0++) {
	for (i = 0; i <= 32; ++i) {
	    area2[i] = area[i] = rArea[i] = gArea[i] = bArea[i] = 0;
	}
	for (g = 1; g < 33; g++) {
	    long int line, rLine, gLine, bLine;
	    unsigned int line2;

	    line2 = line = rLine = gLine = bLine = 0;
	    for (b = 1; b < 33; b++) {
		/* ind1 = RGBIndex(r, g, b); */

		line += s->wt[r][g][b];
		rLine += s->mR[r][g][b];
		gLine += s->mG[r][g][b];
		bLine += s->mB[r][g][b];
		line2 += s->gm2[r][g][b];

		area[b] += line;
		rArea[b] += rLine;
		gArea[b] += gLine;
		bArea[b] += bLine;
		area2[b] += line2;

		/* ind2 = ind1 - 1089; [r0][g][b] */
		s->wt[r][g][b] = s->wt[r0][g][b] + area[b];
		s->mR[r][g][b] = s->mR[r0][g][b] + rArea[b];
		s->mG[r][g][b] = s->mG[r0][g][b] + gArea[b];
		s->mB[r][g][b] = s->mB[r0][g][b] + bArea[b];
		s->gm2[r][g][b] = s->gm2[r0][g][b] + area2[b];
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 *	Compute sum over a box of any given statistic
 *
 *----------------------------------------------------------------------
 */
static INLINE 
long int
Volume(
    Cube *cubePtr,
    long int m[33][33][33])
{
    return (m[R1][G1][B1] - m[R1][G1][B0] - m[R1][G0][B1] + m[R1][G0][B0] -
	    m[R0][G1][B1] + m[R0][G1][B0] + m[R0][G0][B1] - m[R0][G0][B0]);
}

/*
 *----------------------------------------------------------------------
 * 
 * Bottom -- 
 *
 *	The next two routines allow a slightly more efficient
 *	calculation of Volume() for a proposed subbox of a given box.
 *	The sum of Top() and Bottom() is the Volume() of a subbox
 *	split in the given direction and with the specified new upper
 *	bound.
 *
 *----------------------------------------------------------------------
 */
/* Compute part of Volume(cubePtr, mmt) that doesn't depend on r1, g1, or b1 */
/* (depending on dir) */
static long int
Bottom(
    Cube *cubePtr,
    unsigned char dir,
    long int m[33][33][33])	/* Moment */
{
    switch (dir) {
    case RED:
	return -m[R0][G1][B1] + m[R0][G1][B0] + m[R0][G0][B1] - m[R0][G0][B0];
    case GREEN:
	return -m[R1][G0][B1] + m[R1][G0][B0] + m[R0][G0][B1] - m[R0][G0][B0];
    case BLUE:
	return -m[R1][G1][B0] + m[R1][G0][B0] + m[R0][G1][B0] - m[R0][G0][B0];
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * Top --
 *
 *	Compute remainder of Volume(cubePtr, mmt), substituting pos
 *	for r1, g1, or b1 (depending on dir).
 *
 *----------------------------------------------------------------------
 */
static long int
Top(
    Cube *cubePtr,
    unsigned char dir,
    int pos,
    long int m[33][33][33])
{
    switch (dir) {
    case RED:
	return (m[pos][G1][B1] - m[pos][G1][B0] - 
		m[pos][G0][B1] + m[pos][G0][B0]);

    case GREEN:
	return (m[R1][pos][B1] - m[R1][pos][B0] - 
		m[R0][pos][B1] + m[R0][pos][B0]);

    case BLUE:
	return (m[R1][G1][pos] - m[R1][G0][pos] -
		m[R0][G1][pos] + m[R0][G0][pos]);
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 * 
 * Variance --
 *
 *	Compute the weighted variance of a box NB: as with the raw
 *	statistics, this is really the (variance * size)
 *
 *----------------------------------------------------------------------
 */
static double
Variance(
    Cube *cubePtr,
    PictureStatistics *s)
{
    double dR, dG, dB, xx;

    dR = Volume(cubePtr, s->mR);
    dG = Volume(cubePtr, s->mG);
    dB = Volume(cubePtr, s->mB);
    xx = (s->gm2[R1][G1][B1] - s->gm2[R1][G1][B0] -
	  s->gm2[R1][G0][B1] + s->gm2[R1][G0][B0] -
	  s->gm2[R0][G1][B1] + s->gm2[R0][G1][B0] +
	  s->gm2[R0][G0][B1] - s->gm2[R0][G0][B0]);
    return (xx - (dR * dR + dG * dG + dB * dB) / Volume(cubePtr, s->wt));
}

/*
 *----------------------------------------------------------------------
 *
 * Maximize --
 *
 *	We want to minimize the sum of the variances of two subboxes.
 *	The sum(c^2) terms can be ignored since their sum over both
 *	subboxes is the same (the sum for the whole box) no matter
 *	where we split.  The remaining terms have a minus sign in the
 *	variance formula, so we drop the minus sign and MAXIMIZE the
 *	sum of the two terms.
 *
 *----------------------------------------------------------------------
 */
static double
Maximize(
    Cube *cubePtr,
    unsigned char dir,
    int first, int last, 
    int *cut,
    long int rWhole, 
    long int gWhole, 
    long int bWhole, 
    long int wWhole,
    PictureStatistics *s)
{
    long int rHalf, gHalf, bHalf, wHalf;
    long int rBase, gBase, bBase, wBase;
    int i;
    double temp, max;

    rBase = Bottom(cubePtr, dir, s->mR);
    gBase = Bottom(cubePtr, dir, s->mG);
    bBase = Bottom(cubePtr, dir, s->mB);
    wBase = Bottom(cubePtr, dir, s->wt);
    max = 0.0;
    *cut = -1;
    for (i = first; i < last; i++) {
	rHalf = rBase + Top(cubePtr, dir, i, s->mR);
	gHalf = gBase + Top(cubePtr, dir, i, s->mG);
	bHalf = bBase + Top(cubePtr, dir, i, s->mB);
	wHalf = wBase + Top(cubePtr, dir, i, s->wt);

	/* Now half_x is sum over lower half of box, if split at i */
	if (wHalf == 0) {	/* subbox could be empty of pixels! */
	    continue;		/* never split into an empty box */
	} else {
	    temp = ((double)rHalf * rHalf + (float)gHalf * gHalf +
		    (double)bHalf * bHalf) / wHalf;
	}
	rHalf = rWhole - rHalf;
	gHalf = gWhole - gHalf;
	bHalf = bWhole - bHalf;
	wHalf = wWhole - wHalf;
	if (wHalf == 0) {	/* Subbox could be empty of pixels! */
	    continue;		/* never split into an empty box */
	} else {
	    temp += ((double)rHalf * rHalf + (float)gHalf * gHalf +
		(double)bHalf * bHalf) / wHalf;
	}
	if (temp > max) {
	    max = temp;
	    *cut = i;
	}
    }
    return max;
}

/*
 *----------------------------------------------------------------------
 *
 * Cut --
 *
 *----------------------------------------------------------------------
 */
static int
Cut(Cube *set1, Cube *set2, PictureStatistics *s)
{
    unsigned char dir;
    int rCut, gCut, bCut;
    double rMax, gMax, bMax;
    long int rWhole, gWhole, bWhole, wWhole;

    rWhole = Volume(set1, s->mR);
    gWhole = Volume(set1, s->mG);
    bWhole = Volume(set1, s->mB);
    wWhole = Volume(set1, s->wt);

    rMax = Maximize(set1, RED, set1->r0 + 1, set1->r1, &rCut,
	rWhole, gWhole, bWhole, wWhole, s);
    gMax = Maximize(set1, GREEN, set1->g0 + 1, set1->g1, &gCut,
	rWhole, gWhole, bWhole, wWhole, s);
    bMax = Maximize(set1, BLUE, set1->b0 + 1, set1->b1, &bCut,
	rWhole, gWhole, bWhole, wWhole, s);

    if ((rMax >= gMax) && (rMax >= bMax)) {
	dir = RED;
	if (rCut < 0) {
	    return 0;		/* can't split the box */
	}
    } else {
	dir = ((gMax >= rMax) && (gMax >= bMax)) ? GREEN : BLUE;
    }
    set2->r1 = set1->r1;
    set2->g1 = set1->g1;
    set2->b1 = set1->b1;

    switch (dir) {
    case RED:
	set2->r0 = set1->r1 = rCut;
	set2->g0 = set1->g0;
	set2->b0 = set1->b0;
	break;

    case GREEN:
	set2->g0 = set1->g1 = gCut;
	set2->r0 = set1->r0;
	set2->b0 = set1->b0;
	break;

    case BLUE:
	set2->b0 = set1->b1 = bCut;
	set2->r0 = set1->r0;
	set2->g0 = set1->g0;
	break;
    }
    set1->vol = (set1->r1 - set1->r0) * (set1->g1 - set1->g0) *
	(set1->b1 - set1->b0);
    set2->vol = (set2->r1 - set2->r0) * (set2->g1 - set2->g0) *
	(set2->b1 - set2->b0);
    return 1;
}


/*
 *----------------------------------------------------------------------
 *
 * SplitColorSpace --
 *
 *----------------------------------------------------------------------
 */
static int
SplitColorSpace(
    PictureStatistics *s, 
    Cube *cubes, 
    int nColors)
{
    double *vv, temp;
    int i;
    int n, k;

    vv = Blt_Malloc(sizeof(double) * nColors);
    assert(vv);
    
    cubes[0].r0 = cubes[0].g0 = cubes[0].b0 = 0;
    cubes[0].r1 = cubes[0].g1 = cubes[0].b1 = 32;
    for (i = 1, n = 0; i < nColors; i++) {
	if (Cut(cubes + n, cubes + i, s)) {
	    /*
	     * Volume test ensures we won't try to cut one-cell box
	     */
	    vv[n] = vv[i] = 0.0;
	    if (cubes[n].vol > 1) {
		vv[n] = Variance(cubes + n, s);
	    }
	    if (cubes[i].vol > 1) {
		vv[i] = Variance(cubes + i, s);
	    }
	} else {
	    vv[n] = 0.0;	/* don't try to split this box again */
	    i--;		/* didn't create box i */
	}
	
	n = 0;
	temp = vv[0];
	for (k = 1; k <= i; k++) {
	    if (vv[k] > temp) {
		temp = vv[k];
		n = k;
	    }
	}
	if (temp <= 0.0) {
	    i++;
	    fprintf(stderr, "Only got %d boxes\n", i);
	    break;
	}
    }
    Blt_Free(vv);
    return i;
}

/*
 *----------------------------------------------------------------------
 *
 * Mark --
 *
 *--------------------------------------------------------------------
 */
static void
Mark(
    Cube *cubePtr, 
    int label, 
    ColorLookupTable tag)
{
    int r, g, b;

    for (r = R0 + 1; r <= R1; r++) {
	for (g = G0 + 1; g <= G1; g++) {
	    for (b = B0 + 1; b <= B1; b++) {
		tag[r][g][b] = label;
	    }
	}
    }
}

static ColorLookupTable
CreateColorLookupTable(
    PictureStatistics *s, 
    Cube *cubes, 
    int nColors)
{
    ColorLookupTable clut;
    Cube *cp, *cend;

    clut = Blt_Calloc(sizeof(unsigned int), 33 * 33 * 33);
    assert(clut);

    for (cp = cubes, cend = cp + nColors; cp < cend; cp++) {
	unsigned int r, g, b;
	unsigned int weight;
	Pix32 pixel;

	weight = Volume(cp, s->wt);
	if (weight) {
	    r = (Volume(cp, s->mR) / weight) * (NC + 1);
	    g = (Volume(cp, s->mG) / weight) * (NC + 1);
	    b = (Volume(cp, s->mB) / weight) * (NC + 1);
	} else {
	    r = g = b = 0;
	}
	pixel.Red = r / 257;
	pixel.Green = g / 257;
	pixel.Blue = b / 257;
	pixel.Alpha = ALPHA_OPAQUE;
	Mark(cp, pixel.color, clut);
    }
    return clut;
}

static void
MapColors(Picture *destPtr, Picture *srcPtr, ColorLookupTable clut)
{
    Pix32 *srcRowPtr, *destRowPtr;
    int y;

    /* Apply the color lookup table against the original image */
    srcRowPtr = srcPtr->bits;    
    destRowPtr = destPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *dp, *sp, *send;
	
	sp = srcRowPtr, dp = destRowPtr;
	for (sp = srcRowPtr, send = sp + srcPtr->width; sp < send; sp++) {
	    dp->color = clut[sp->Red>>3][sp->Green>>3][sp->Blue>>3];
	    dp->Alpha = sp->Alpha;
	    dp++;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_QuantizePicture --
 *
 *	C Implementation of Wu's Color Quantizer (v. 2) (see Graphics Gems
 *	vol. II, pp. 126-133)
 *
 *	Author: Xiaolin Wu
 *		Dept. of Computer Science Univ. of Western
 *		Ontario London, Ontario
 *		N6A 5B7
 *		wu@csd.uwo.ca
 *
 *		Greedy orthogonal bipartition of RGB space for variance
 *		minimization aided by inclusion-exclusion tricks.  For
 *		speed no nearest neighbor search is done. Slightly
 *		better performance can be expected by more
 *		sophisticated but more expensive versions.
 *
 *		The author thanks Tom Lane at Tom_Lane@G.GP.CS.CMU.EDU
 *		for much of additional documentation and a cure to a
 *		previous bug.
 *
 *		Free to distribute, comments and suggestions are
 *		appreciated.
 *
 *----------------------------------------------------------------------
 */
Blt_Picture
Blt_QuantizePicture(
    Picture *srcPtr, 		/* Source image. */
    int reduceColors)		/* Reduced number of colors. */
{
    Cube *cubes;
    PictureStatistics *statistics;
    int nColors;
    ColorLookupTable clut;
    Picture *destPtr;

    /*
     * Allocated a structure to hold color statistics.
     */
    statistics = GetPictureStatistics(srcPtr);
    M3d(statistics);

    cubes = Blt_Malloc(sizeof(Cube) * reduceColors);
    assert(cubes);

    nColors = SplitColorSpace(statistics, cubes, reduceColors);
    assert(nColors <= reduceColors);

    clut = CreateColorLookupTable(statistics, cubes, nColors);
    Blt_Free(statistics);
    Blt_Free(cubes);
    destPtr = Blt_CreatePicture(srcPtr->width, srcPtr->height);
    MapColors(destPtr, srcPtr, clut);
    Blt_Free(clut);
    return destPtr;
}

/* 
 * ---------------------------------------------------------------------------
 *
 * Blt_CopyPictureArea --
 *
 *	Creates a copy of the given picture.  
 *
 * Results:  
 *	Returns the new copy.
 *
 * --------------------------------------------------------------------------- 
 */
void
Blt_CopyPictureArea(
    Picture *destPtr, 
    Picture *srcPtr,
    int srcX, int srcY, 
    int srcWidth, int srcHeight,
    int destX, int destY)
{
    int *srcRowPtr, *destRowPtr;
    int right, bottom;
    int destWidth, destHeight;
    int width, height;

    destWidth  = destPtr->width - destX;
    destHeight = destPtr->height - destY;

    width  = MIN(destWidth, srcWidth);
    height = MIN(destHeight, srcHeight);
    
    bottom = height + srcY;
    right  = width + srcX;

    srcRowPtr = (int *)(srcPtr->bits + ((srcPtr->pixelsPerRow * srcY) + srcX));
    destRowPtr = (int *)
	(destPtr->bits + ((destPtr->pixelsPerRow * destY) + destX));

    for (/*empty*/; srcY < bottom; srcY++) {
	int n;
	int *sp, *dp;

	sp = srcRowPtr, dp = destRowPtr;
	n = (width + 7) / 8;      /* count > 0 assumed */
	switch (width % 8) {
	case 0:        do {  *dp++ = *sp++;
	case 7:              *dp++ = *sp++;
	case 6:              *dp++ = *sp++;
	case 5:              *dp++ = *sp++;
	case 4:              *dp++ = *sp++;
	case 3:              *dp++ = *sp++;
	case 2:              *dp++ = *sp++;
	case 1:              *dp++ = *sp++;
			} while (--n > 0);
	}
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
}

/*
 * Image dithering routines.
 *
 * Reference: 
 *	Victor Ostromoukhov http://www.iro.umontreal.ca/~ostrom/.  
 *
 *	Victor Ostromoukhov, "A Simple and Efficient Error-Diffusion
 *	Algorithm" SIGGRAPH'01
 *
 *	University of Montreal, http://www.iro.umontreal.ca/~ostrom/
 *
 */
typedef struct {
    short int r;		/* Right */
    short int dl;		/* Down-left */
    short int d;		/* Down */
    short int sum;		/* Sum */
} VarCoefs;

static VarCoefs coefTable[256] = {
    {    13,     0,     5,    18,   },  /*    0 */
    {    13,     0,     5,    18,   },  /*    1 */
    {    21,     0,    10,    31,   },  /*    2 */
    {     7,     0,     4,    11,   },  /*    3 */
    {     8,     0,     5,    13,   },  /*    4 */
    {    47,     3,    28,    78,   },  /*    5 */
    {    23,     3,    13,    39,   },  /*    6 */
    {    15,     3,     8,    26,   },  /*    7 */
    {    22,     6,    11,    39,   },  /*    8 */
    {    43,    15,    20,    78,   },  /*    9 */
    {     7,     3,     3,    13,   },  /*   10 */
    {   501,   224,   211,   936,   },  /*   11 */
    {   249,   116,   103,   468,   },  /*   12 */
    {   165,    80,    67,   312,   },  /*   13 */
    {   123,    62,    49,   234,   },  /*   14 */
    {   489,   256,   191,   936,   },  /*   15 */
    {    81,    44,    31,   156,   },  /*   16 */
    {   483,   272,   181,   936,   },  /*   17 */
    {    60,    35,    22,   117,   },  /*   18 */
    {    53,    32,    19,   104,   },  /*   19 */
    {   237,   148,    83,   468,   },  /*   20 */
    {   471,   304,   161,   936,   },  /*   21 */
    {     3,     2,     1,     6,   },  /*   22 */
    {   459,   304,   161,   924,   },  /*   23 */
    {    38,    25,    14,    77,   },  /*   24 */
    {   453,   296,   175,   924,   },  /*   25 */
    {   225,   146,    91,   462,   },  /*   26 */
    {   149,    96,    63,   308,   },  /*   27 */
    {   111,    71,    49,   231,   },  /*   28 */
    {    63,    40,    29,   132,   },  /*   29 */
    {    73,    46,    35,   154,   },  /*   30 */
    {   435,   272,   217,   924,   },  /*   31 */
    {   108,    67,    56,   231,   },  /*   32 */
    {    13,     8,     7,    28,   },  /*   33 */
    {   213,   130,   119,   462,   },  /*   34 */
    {   423,   256,   245,   924,   },  /*   35 */
    {     5,     3,     3,    11,   },  /*   36 */
    {   281,   173,   162,   616,   },  /*   37 */
    {   141,    89,    78,   308,   },  /*   38 */
    {   283,   183,   150,   616,   },  /*   39 */
    {    71,    47,    36,   154,   },  /*   40 */
    {   285,   193,   138,   616,   },  /*   41 */
    {    13,     9,     6,    28,   },  /*   42 */
    {    41,    29,    18,    88,   },  /*   43 */
    {    36,    26,    15,    77,   },  /*   44 */
    {   289,   213,   114,   616,   },  /*   45 */
    {   145,   109,    54,   308,   },  /*   46 */
    {   291,   223,   102,   616,   },  /*   47 */
    {    73,    57,    24,   154,   },  /*   48 */
    {   293,   233,    90,   616,   },  /*   49 */
    {    21,    17,     6,    44,   },  /*   50 */
    {   295,   243,    78,   616,   },  /*   51 */
    {    37,    31,     9,    77,   },  /*   52 */
    {    27,    23,     6,    56,   },  /*   53 */
    {   149,   129,    30,   308,   },  /*   54 */
    {   299,   263,    54,   616,   },  /*   55 */
    {    75,    67,    12,   154,   },  /*   56 */
    {    43,    39,     6,    88,   },  /*   57 */
    {   151,   139,    18,   308,   },  /*   58 */
    {   303,   283,    30,   616,   },  /*   59 */
    {    38,    36,     3,    77,   },  /*   60 */
    {   305,   293,    18,   616,   },  /*   61 */
    {   153,   149,     6,   308,   },  /*   62 */
    {   307,   303,     6,   616,   },  /*   63 */
    {     1,     1,     0,     2,   },  /*   64 */
    {   101,   105,     2,   208,   },  /*   65 */
    {    49,    53,     2,   104,   },  /*   66 */
    {    95,   107,     6,   208,   },  /*   67 */
    {    23,    27,     2,    52,   },  /*   68 */
    {    89,   109,    10,   208,   },  /*   69 */
    {    43,    55,     6,   104,   },  /*   70 */
    {    83,   111,    14,   208,   },  /*   71 */
    {     5,     7,     1,    13,   },  /*   72 */
    {   172,   181,    37,   390,   },  /*   73 */
    {    97,    76,    22,   195,   },  /*   74 */
    {    72,    41,    17,   130,   },  /*   75 */
    {   119,    47,    29,   195,   },  /*   76 */
    {     4,     1,     1,     6,   },  /*   77 */
    {     4,     1,     1,     6,   },  /*   78 */
    {     4,     1,     1,     6,   },  /*   79 */
    {     4,     1,     1,     6,   },  /*   80 */
    {     4,     1,     1,     6,   },  /*   81 */
    {     4,     1,     1,     6,   },  /*   82 */
    {     4,     1,     1,     6,   },  /*   83 */
    {     4,     1,     1,     6,   },  /*   84 */
    {     4,     1,     1,     6,   },  /*   85 */
    {    65,    18,    17,   100,   },  /*   86 */
    {    95,    29,    26,   150,   },  /*   87 */
    {   185,    62,    53,   300,   },  /*   88 */
    {    30,    11,     9,    50,   },  /*   89 */
    {    35,    14,    11,    60,   },  /*   90 */
    {    85,    37,    28,   150,   },  /*   91 */
    {    55,    26,    19,   100,   },  /*   92 */
    {    80,    41,    29,   150,   },  /*   93 */
    {   155,    86,    59,   300,   },  /*   94 */
    {     5,     3,     2,    10,   },  /*   95 */
    {     5,     3,     2,    10,   },  /*   96 */
    {     5,     3,     2,    10,   },  /*   97 */
    {     5,     3,     2,    10,   },  /*   98 */
    {     5,     3,     2,    10,   },  /*   99 */
    {     5,     3,     2,    10,   },  /*  100 */
    {     5,     3,     2,    10,   },  /*  101 */
    {     5,     3,     2,    10,   },  /*  102 */
    {     5,     3,     2,    10,   },  /*  103 */
    {     5,     3,     2,    10,   },  /*  104 */
    {     5,     3,     2,    10,   },  /*  105 */
    {     5,     3,     2,    10,   },  /*  106 */
    {     5,     3,     2,    10,   },  /*  107 */
    {   305,   176,   119,   600,   },  /*  108 */
    {   155,    86,    59,   300,   },  /*  109 */
    {   105,    56,    39,   200,   },  /*  110 */
    {    80,    41,    29,   150,   },  /*  111 */
    {    65,    32,    23,   120,   },  /*  112 */
    {    55,    26,    19,   100,   },  /*  113 */
    {   335,   152,   113,   600,   },  /*  114 */
    {    85,    37,    28,   150,   },  /*  115 */
    {   115,    48,    37,   200,   },  /*  116 */
    {    35,    14,    11,    60,   },  /*  117 */
    {   355,   136,   109,   600,   },  /*  118 */
    {    30,    11,     9,    50,   },  /*  119 */
    {   365,   128,   107,   600,   },  /*  120 */
    {   185,    62,    53,   300,   },  /*  121 */
    {    25,     8,     7,    40,   },  /*  122 */
    {    95,    29,    26,   150,   },  /*  123 */
    {   385,   112,   103,   600,   },  /*  124 */
    {    65,    18,    17,   100,   },  /*  125 */
    {   395,   104,   101,   600,   },  /*  126 */
    {     4,     1,     1,     6,   },  /*  127 */
    {     4,     1,     1,     6,   },  /*  128 */
    {   395,   104,   101,   600,   },  /*  129 */
    {    65,    18,    17,   100,   },  /*  130 */
    {   385,   112,   103,   600,   },  /*  131 */
    {    95,    29,    26,   150,   },  /*  132 */
    {    25,     8,     7,    40,   },  /*  133 */
    {   185,    62,    53,   300,   },  /*  134 */
    {   365,   128,   107,   600,   },  /*  135 */
    {    30,    11,     9,    50,   },  /*  136 */
    {   355,   136,   109,   600,   },  /*  137 */
    {    35,    14,    11,    60,   },  /*  138 */
    {   115,    48,    37,   200,   },  /*  139 */
    {    85,    37,    28,   150,   },  /*  140 */
    {   335,   152,   113,   600,   },  /*  141 */
    {    55,    26,    19,   100,   },  /*  142 */
    {    65,    32,    23,   120,   },  /*  143 */
    {    80,    41,    29,   150,   },  /*  144 */
    {   105,    56,    39,   200,   },  /*  145 */
    {   155,    86,    59,   300,   },  /*  146 */
    {   305,   176,   119,   600,   },  /*  147 */
    {     5,     3,     2,    10,   },  /*  148 */
    {     5,     3,     2,    10,   },  /*  149 */
    {     5,     3,     2,    10,   },  /*  150 */
    {     5,     3,     2,    10,   },  /*  151 */
    {     5,     3,     2,    10,   },  /*  152 */
    {     5,     3,     2,    10,   },  /*  153 */
    {     5,     3,     2,    10,   },  /*  154 */
    {     5,     3,     2,    10,   },  /*  155 */
    {     5,     3,     2,    10,   },  /*  156 */
    {     5,     3,     2,    10,   },  /*  157 */
    {     5,     3,     2,    10,   },  /*  158 */
    {     5,     3,     2,    10,   },  /*  159 */
    {     5,     3,     2,    10,   },  /*  160 */
    {   155,    86,    59,   300,   },  /*  161 */
    {    80,    41,    29,   150,   },  /*  162 */
    {    55,    26,    19,   100,   },  /*  163 */
    {    85,    37,    28,   150,   },  /*  164 */
    {    35,    14,    11,    60,   },  /*  165 */
    {    30,    11,     9,    50,   },  /*  166 */
    {   185,    62,    53,   300,   },  /*  167 */
    {    95,    29,    26,   150,   },  /*  168 */
    {    65,    18,    17,   100,   },  /*  169 */
    {     4,     1,     1,     6,   },  /*  170 */
    {     4,     1,     1,     6,   },  /*  171 */
    {     4,     1,     1,     6,   },  /*  172 */
    {     4,     1,     1,     6,   },  /*  173 */
    {     4,     1,     1,     6,   },  /*  174 */
    {     4,     1,     1,     6,   },  /*  175 */
    {     4,     1,     1,     6,   },  /*  176 */
    {     4,     1,     1,     6,   },  /*  177 */
    {     4,     1,     1,     6,   },  /*  178 */
    {   119,    47,    29,   195,   },  /*  179 */
    {    72,    41,    17,   130,   },  /*  180 */
    {    97,    76,    22,   195,   },  /*  181 */
    {   172,   181,    37,   390,   },  /*  182 */
    {     5,     7,     1,    13,   },  /*  183 */
    {    83,   111,    14,   208,   },  /*  184 */
    {    43,    55,     6,   104,   },  /*  185 */
    {    89,   109,    10,   208,   },  /*  186 */
    {    23,    27,     2,    52,   },  /*  187 */
    {    95,   107,     6,   208,   },  /*  188 */
    {    49,    53,     2,   104,   },  /*  189 */
    {   101,   105,     2,   208,   },  /*  190 */
    {     1,     1,     0,     2,   },  /*  191 */
    {   307,   303,     6,   616,   },  /*  192 */
    {   153,   149,     6,   308,   },  /*  193 */
    {   305,   293,    18,   616,   },  /*  194 */
    {    38,    36,     3,    77,   },  /*  195 */
    {   303,   283,    30,   616,   },  /*  196 */
    {   151,   139,    18,   308,   },  /*  197 */
    {    43,    39,     6,    88,   },  /*  198 */
    {    75,    67,    12,   154,   },  /*  199 */
    {   299,   263,    54,   616,   },  /*  200 */
    {   149,   129,    30,   308,   },  /*  201 */
    {    27,    23,     6,    56,   },  /*  202 */
    {    37,    31,     9,    77,   },  /*  203 */
    {   295,   243,    78,   616,   },  /*  204 */
    {    21,    17,     6,    44,   },  /*  205 */
    {   293,   233,    90,   616,   },  /*  206 */
    {    73,    57,    24,   154,   },  /*  207 */
    {   291,   223,   102,   616,   },  /*  208 */
    {   145,   109,    54,   308,   },  /*  209 */
    {   289,   213,   114,   616,   },  /*  210 */
    {    36,    26,    15,    77,   },  /*  211 */
    {    41,    29,    18,    88,   },  /*  212 */
    {    13,     9,     6,    28,   },  /*  213 */
    {   285,   193,   138,   616,   },  /*  214 */
    {    71,    47,    36,   154,   },  /*  215 */
    {   283,   183,   150,   616,   },  /*  216 */
    {   141,    89,    78,   308,   },  /*  217 */
    {   281,   173,   162,   616,   },  /*  218 */
    {     5,     3,     3,    11,   },  /*  219 */
    {   423,   256,   245,   924,   },  /*  220 */
    {   213,   130,   119,   462,   },  /*  221 */
    {    13,     8,     7,    28,   },  /*  222 */
    {   108,    67,    56,   231,   },  /*  223 */
    {   435,   272,   217,   924,   },  /*  224 */
    {    73,    46,    35,   154,   },  /*  225 */
    {    63,    40,    29,   132,   },  /*  226 */
    {   111,    71,    49,   231,   },  /*  227 */
    {   149,    96,    63,   308,   },  /*  228 */
    {   225,   146,    91,   462,   },  /*  229 */
    {   453,   296,   175,   924,   },  /*  230 */
    {    38,    25,    14,    77,   },  /*  231 */
    {   459,   304,   161,   924,   },  /*  232 */
    {     3,     2,     1,     6,   },  /*  233 */
    {   471,   304,   161,   936,   },  /*  234 */
    {   237,   148,    83,   468,   },  /*  235 */
    {    53,    32,    19,   104,   },  /*  236 */
    {    60,    35,    22,   117,   },  /*  237 */
    {   483,   272,   181,   936,   },  /*  238 */
    {    81,    44,    31,   156,   },  /*  239 */
    {   489,   256,   191,   936,   },  /*  240 */
    {   123,    62,    49,   234,   },  /*  241 */
    {   165,    80,    67,   312,   },  /*  242 */
    {   249,   116,   103,   468,   },  /*  243 */
    {   501,   224,   211,   936,   },  /*  244 */
    {     7,     3,     3,    13,   },  /*  245 */
    {    43,    15,    20,    78,   },  /*  246 */
    {    22,     6,    11,    39,   },  /*  247 */
    {    15,     3,     8,    26,   },  /*  248 */
    {    23,     3,    13,    39,   },  /*  249 */
    {    47,     3,    28,    78,   },  /*  250 */
    {     8,     0,     5,    13,   },  /*  251 */
    {     7,     0,     4,    11,   },  /*  252 */
    {    21,     0,    10,    31,   },  /*  253 */
    {    13,     0,     5,    18,   },  /*  254 */
    {    13,     0,     5,    18,   },  /*  255 */
};

static void 
DistributeError(
    double (*cl0)[3], double (*cl1)[3],
    int x, 
    double rDiff, double gDiff, double bDiff,
    int dir, Pix32 *sp)
{
    VarCoefs *coefsPtr;
    double sum, r, dl, d;

    coefsPtr = coefTable + sp->Red;
    sum = coefsPtr->sum;
    r = coefsPtr->r * rDiff / sum;
    dl = coefsPtr->dl * rDiff / sum;
    d = rDiff - (r + dl);

    cl0[x + dir][0] += r;
    cl1[x - dir][0] += dl;
    cl1[x][0]       += d;

    coefsPtr = coefTable + sp->Green;
    sum = coefsPtr->sum;
    r = coefsPtr->r * gDiff / sum;
    dl = coefsPtr->dl * gDiff / sum;
    d = gDiff - (r + dl);

    cl0[x + dir][1] += r;
    cl1[x - dir][1] += dl;
    cl1[x][1]       += d;

    coefsPtr = coefTable + sp->Blue;
    sum = coefsPtr->sum;
    r = coefsPtr->r * bDiff / sum;
    dl = coefsPtr->dl * bDiff / sum;
    d = bDiff - (r + dl);

    cl0[x + dir][2] += r;
    cl1[x - dir][2] += dl;
    cl1[x][2]       += d;
}

void 
DistributeErrorFS(
    double (*cl0)[3], double (*cl1)[3],
    int x, 
    double rDiff, double gDiff, double bDiff,
    int dir, Pix32 *sp)
{
    double d2, d3, d5, d7;

    d2 = rDiff + rDiff;
    d3 = rDiff + d2;
    d5 = d3 + d2;
    d7 = d5 + d2;

    cl0[x + dir][0] += (d7 / 16);
    cl1[x - dir][0] += (d3 / 16);
    cl1[x][0]       += (d5 / 16);
    cl1[x + dir][0] += (rDiff / 16);
    
    d2 = gDiff + gDiff;
    d3 = gDiff + d2;
    d5 = d3 + d2;
    d7 = d5 + d2;

    cl0[x + dir][1] += (d7 / 16);
    cl1[x - dir][1] += (d3 / 16);
    cl1[x][1]       += (d5 / 16);
    cl1[x + dir][1] += (gDiff / 16);

    d2 = bDiff + bDiff;
    d3 = bDiff + d2;
    d5 = d3 + d2;
    d7 = d5 + d2;

    cl0[x + dir][2] += (d7 / 16);
    cl1[x - dir][2] += (d3 / 16);
    cl1[x][2]       += (d5 / 16);
    cl1[x + dir][2] += (bDiff / 16);
}


static void 
ShiftCarryBuffers(
    double (**cl0)[3],
    double (**cl1)[3],
    int width)
{
    double (*tmp)[3];
    int i;

    /* Swap cl0 and cl1 */
    tmp = *cl0, *cl0 = *cl1, *cl1 = tmp;

    /* Clear c11 */
    for (i = 0; i < width; ++i) {
        (*cl1)[i][0] = (*cl1)[i][1] = (*cl1)[i][2] = 0.0;
    }
}

/* 
 *--------------------------------------------------------------------------- 
 *
 * Blt_DitherPicture --
 *
 *	Dithers a 24-bit picture using a given reduced-color palette. 
 *
 *	Reference: 
 *		Victor Ostromoukhov http://www.iro.umontreal.ca/~ostrom/.  
 *
 *		Victor Ostromoukhov, "A Simple and Efficient
 *		Error-Diffusion Algorithm" SIGGRAPH'01
 *
 *		University of Montreal, http://www.iro.umontreal.ca/~ostrom/
 *
 * Results:
 *	A new picture is allocated, dithered and returned. Returns NULL
 *	only is memory can't be allocated for the dithered picture.
 *
 *--------------------------------------------------------------------------- 
 */
Blt_Picture 
Blt_DitherPicture(
    Picture *srcPtr,		/* Source picture to be dithered. */
    Pix32 *palette)		/* Palette of available colors. */
{
    Picture *destPtr;
    Pix32 *srcRowPtr, *destRowPtr;
    double (*cl0)[3];
    double (*cl1)[3];
    int y;
    
    /* allocate carry_line_0 and carry_line_1 */
    cl0 = Blt_Calloc(srcPtr->width + 2, sizeof(*cl0));
    if (cl0 == NULL) {
	return NULL;
    }
    cl1 = Blt_Calloc(srcPtr->width + 2, sizeof(*cl1));
    if (cl1 == NULL) {
	return NULL;
    }
    ++cl0;
    ++cl1;

    destPtr = Blt_CreatePicture(srcPtr->width, srcPtr->height);

    srcRowPtr = srcPtr->bits, destRowPtr = destPtr->bits;
    for (y = 0; y < srcPtr->height; ++y) {
	Pix32 *sp, *dp;
	int start, finish, step;
	int x;

        if (y & 1) {
            start = srcPtr->width - 1;
            finish = -1;
            step = -1;
        } else {
            start = 0;
            finish = srcPtr->width;
            step = 1;
        }
	sp = srcRowPtr + start, dp = destRowPtr + start;
        for (x = start; x != finish; x += step) {
            double rCorrected, gCorrected, bCorrected;
            double rDiff, gDiff, bDiff;
	    int rIntensity, gIntensity, bIntensity;

            rCorrected = sp->Red +   cl0[x][0];
            gCorrected = sp->Green + cl0[x][1];
            bCorrected = sp->Blue +  cl0[x][2];

	    rIntensity = palette[(int)CLAMP(rCorrected)].Red;
	    gIntensity = palette[(int)CLAMP(gCorrected)].Green;
	    bIntensity = palette[(int)CLAMP(bCorrected)].Blue;

            rDiff = rCorrected - rIntensity;
            gDiff = gCorrected - gIntensity;
            bDiff = bCorrected - bIntensity;

            DistributeError(cl0, cl1, x, rDiff, gDiff, bDiff, step, sp); 
	    dp->Red = rIntensity;
	    dp->Green = gIntensity;
	    dp->Blue = bIntensity;
	    dp += step, sp += step;
        }
        ShiftCarryBuffers(&cl0, &cl1, srcPtr->width);
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
    Blt_Free(cl0 - 1);
    Blt_Free(cl1 - 1);
    return destPtr;
}

INLINE static void
filt2(Pix32 *ap, Pix32 *bp, Pix32 *dp, Pix32 *fp, Pix32 *gp, Pix32 *out)
{
    int bf, i;
    
#ifdef notdef
    mm1 = b;			/* Copy b pixel into 4 16-bit slots. */
    mm2 = f;			/* Copy f pixel into 4 16-bit slots. */
    mm2 = b + f;		/* Add b + f.  */
    mm1 = d;			/* Copy d pixel into 4 16-bit slots. */
    mm1 = d << 1;		/* Mult d * 2. */
    mm3 = a;			/* Copy a pixel into 4 16-bit slots. */
    mm4 = g;			/* Copy g pixel info 4 16-bit slots. */
    mm4 = a + g;		/* Add a + g. */
    mm1 = d + bf;		/* Add mm1 + mm2 */
    mm1 = mm1 << 2;
    mm1 = mm1 + bf;
    mm1 = mm1 - mm4;
    mm1 = mm1 >> 4;
    out = mm1;
#endif

    bf = bp->Red + fp->Red;
    i = (((((dp->Red << 1) + bf) << 2) + bf - ap->Red - gp->Red) >> 4);
    out->Red = ((i > 255) ? 255 : (i < 0) ? 0 : i);

    bf = bp->Green + fp->Green;
    i = (((((dp->Green << 1) + bf) << 2) + bf - ap->Green - gp->Green) >> 4);
    out->Green = ((i > 255) ? 255 : (i < 0) ? 0 : i);

    bf = bp->Blue + fp->Blue;
    i = (((((dp->Blue << 1) + bf) << 2) + bf - ap->Blue - gp->Blue) >> 4);
    out->Blue = ((i > 255) ? 255 : (i < 0) ? 0 : i);

    bf = bp->Alpha + fp->Alpha;
    i = (((((dp->Alpha << 1) + bf) << 2) + bf - ap->Alpha - gp->Alpha) >> 4);
    out->Alpha = ((i > 255) ? 255 : (i < 0) ? 0 : i);
}

#define FILT2(A0,A2,A3,A4,A6,out) \
   value = ((((((A3)->Red << 1) + (A2)->Red + (A4)->Red) <<2) + \
	(A2)->Red + (A4)->Red - (A0)->Red - (A6)->Red) >> 4), \
   out->Red = ((value > 255) ? 255 : (value < 0) ? 0 : value), \
   value = ((((((A3)->Green << 1) + (A2)->Green + (A4)->Green) <<2) + \
	(A2)->Green + (A4)->Green - (A0)->Green - (A6)->Green) >> 4), \
   out->Green = ((value > 255) ? 255 : (value < 0) ? 0 : value), \
   value = ((((((A3)->Blue << 1) + (A2)->Blue + (A4)->Blue) <<2) + \
	(A2)->Blue + (A4)->Blue - (A0)->Blue - (A6)->Blue) >> 4),  \
   out->Blue = ((value > 255) ? 255 : (value < 0) ? 0 : value), \
   value = ((((((A3)->Alpha << 1) + (A2)->Alpha + (A4)->Alpha) <<2) + \
	(A2)->Alpha + (A4)->Alpha - (A0)->Alpha - (A6)->Alpha) >> 4), \
   out->Alpha = ((value > 255) ? 255 : (value < 0) ? 0 : value) 

#define FILT3(A0,A2,A3,A4,A6,out) \
   value = ((((((A3)->Red << 1) - (A2)->Red - (A4)->Red) <<2) - \
	(A2)->Red - (A4)->Red + (A0)->Red + (A6)->Red) >> 4), \
   out->Red = ((value > 255) ? 255 : (value < 0) ? 0 : value), \
   value = ((((((A3)->Green << 1) - (A2)->Green - (A4)->Green) <<2) - \
	(A2)->Green - (A4)->Green + (A0)->Green + (A6)->Green) >> 4), \
   out->Green = ((value > 255) ? 255 : (value < 0) ? 0 : value), \
   value = ((((((A3)->Blue << 1) - (A2)->Blue - (A4)->Blue) <<2) - \
	(A2)->Blue - (A4)->Blue + (A0)->Blue + (A6)->Blue) >> 4),  \
   out->Blue = ((value > 255) ? 255 : (value < 0) ? 0 : value), \
   value = ((((((A3)->Alpha << 1) - (A2)->Alpha - (A4)->Alpha) <<2) - \
	(A2)->Alpha - (A4)->Alpha + (A0)->Alpha + (A6)->Alpha) >> 4), \
   out->Alpha = ((value > 255) ? 255 : (value < 0) ? 0 : value) 

static void
FastMin2X(
    Picture *destPtr, 
    Picture *srcPtr)
{
    int y;
    Pix32 *srcRowPtr, *destRowPtr, *send;

    /* Apply filter to each column. */
    srcRowPtr = srcPtr->bits;
    destRowPtr  = destPtr->bits;
    assert(srcPtr->width > 7);
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp, *dp;

	sp = srcRowPtr;
	dp = destRowPtr;
	/* 0,x,0,0,1,x,3 */
	filt2(sp, sp, sp, sp+1, sp+3, dp);
	dp->Alpha = ALPHA_OPAQUE;
	dp++;
	/* 0,x,1,2,3,x,5 */
	filt2(sp, sp+1, sp+2, sp+3, sp+5, dp);
	dp->Alpha = ALPHA_OPAQUE;
	dp++;
	for (sp = srcRowPtr + 3, send = sp + (srcPtr->width - 7); sp < send; 
	     sp += 2) {
	    filt2(sp-3, sp-1, sp, sp+1, sp+3, dp);
	    dp->Alpha = ALPHA_OPAQUE;
	    dp++;
	}
	/* -3,x,-1,0,1,x,3 */
	filt2(sp-3, sp-1, sp, sp+1, sp+2, dp);
	dp->Alpha = ALPHA_OPAQUE;
	dp++;
	filt2(sp-1, sp+1, sp+2, sp+3, sp+3, dp);
	dp->Alpha = ALPHA_OPAQUE;
	dp++;
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
}
	    
static void
FastMin2Y(
    Picture *destPtr, 
    Picture *srcPtr)
{
    int x, y;

    for (x = 0; x < srcPtr->width; x++) {
	Pix32 *dp;

	dp = destPtr->bits + x;
	/* 0,x,0,0,1,x,3 */
	filt2(Blt_PicturePixel(srcPtr, x, 0),
	      Blt_PicturePixel(srcPtr, x, 0),
	      Blt_PicturePixel(srcPtr, x, 0),
	      Blt_PicturePixel(srcPtr, x, 1),
	      Blt_PicturePixel(srcPtr, x, 3),
	      dp);
	dp->Alpha = ALPHA_OPAQUE;
	dp += destPtr->pixelsPerRow;

	/* 0,x,1,2,3,x,5 */
	filt2(Blt_PicturePixel(srcPtr, x, 0),
	      Blt_PicturePixel(srcPtr, x, 1),
	      Blt_PicturePixel(srcPtr, x, 2),
	      Blt_PicturePixel(srcPtr, x, 3),
	      Blt_PicturePixel(srcPtr, x, 5),
	      dp);
	dp->Alpha = ALPHA_OPAQUE;
	dp += destPtr->pixelsPerRow;

	for (y = 3; y < (srcPtr->height - 7); y += 2) {
	    filt2(Blt_PicturePixel(srcPtr, x, y - 3),
		  Blt_PicturePixel(srcPtr, x, y - 1),
		  Blt_PicturePixel(srcPtr, x, y),
		  Blt_PicturePixel(srcPtr, x, y + 1),
		  Blt_PicturePixel(srcPtr, x, y + 3),
		  dp);
	    dp->Alpha = ALPHA_OPAQUE;
	    dp += destPtr->pixelsPerRow;
	}
	filt2(Blt_PicturePixel(srcPtr, x, y - 3),
	      Blt_PicturePixel(srcPtr, x, y - 1),
	      Blt_PicturePixel(srcPtr, x, y),
	      Blt_PicturePixel(srcPtr, x, y + 1),
	      Blt_PicturePixel(srcPtr, x, y + 3),
	      dp);
	dp->Alpha = ALPHA_OPAQUE;
	y += 2;
	dp += destPtr->pixelsPerRow;
	filt2(Blt_PicturePixel(srcPtr, x, y - 3),
	      Blt_PicturePixel(srcPtr, x, y - 1),
	      Blt_PicturePixel(srcPtr, x, y),
	      Blt_PicturePixel(srcPtr, x, y + 1),
	      Blt_PicturePixel(srcPtr, x, y + 3),
	      dp);
	dp->Alpha = ALPHA_OPAQUE;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_Min2Picture --
 *
 *      Resamples a given picture using 1-D filters and returns
 *	a new picture of the designated size.
 *
 * Results:
 *      Returns the resampled picture. The original picture
 *	is left intact.
 *
 *----------------------------------------------------------------------
 */
Blt_Picture
Blt_Min2Picture(Picture *srcPtr)
{
    Picture *tmpPtr, *destPtr;
    int width, height;

    /* 
     * It's usually faster to zoom vertically last.  This has to do
     * with the fact that images are stored in contiguous rows.
     */
    width = srcPtr->width / 2;
    height = srcPtr->height / 2;
    tmpPtr = Blt_CreatePicture(width, srcPtr->height);
    FastMin2X(srcPtr, tmpPtr);
    destPtr = Blt_CreatePicture(width, height);
    FastMin2Y(tmpPtr, destPtr);
    Blt_FreePicture(tmpPtr);
    return destPtr;
}

static void
BoxX(Picture *destPtr, Picture *srcPtr)
{
    Pix32 *srcRowPtr, *destRowPtr;
    int y;

    srcRowPtr = srcPtr->bits;
    destRowPtr = destPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *dp, *send;
	Pix32 *lp, *cp, *rp;	/* Pointers to left, center, and right
				 * pixels.  */
	Pix32 hold;
	double r, g, b, a;

	dp = destRowPtr;
	cp = lp = srcRowPtr, rp = srcRowPtr + 1;
	r = (double)(lp->Red + cp->Red + rp->Red) * 0.333333333333333;
	g = (double)(lp->Green + cp->Green + rp->Green) * 0.333333333333333;
	b = (double)(lp->Blue + cp->Blue + rp->Blue) * 0.333333333333333;
	a = (double)(lp->Alpha + cp->Alpha + rp->Alpha) * 0.333333333333333;
	hold = *lp;
	dp->Red = (unsigned char)CLAMP(r);
	dp->Green = (unsigned char)CLAMP(g);
	dp->Blue = (unsigned char)CLAMP(b);
	dp->Alpha = (unsigned char)CLAMP(a);
	dp++, cp++, rp++;

	for (send = srcRowPtr + srcPtr->width; rp < send; rp++, cp++, lp++) {
	    r = (double)(lp->Red + cp->Red + rp->Red) * 0.333333333333333;
	    g = (double)(lp->Green + cp->Green + rp->Green) * 0.333333333333333;
	    b = (double)(lp->Blue + cp->Blue + rp->Blue) * 0.333333333333333;
	    a = (double)(lp->Alpha + cp->Alpha + rp->Alpha) * 0.333333333333333;
	    dp->Red = (unsigned char)CLAMP(r);
	    dp->Green = (unsigned char)CLAMP(g);
	    dp->Blue = (unsigned char)CLAMP(b);
	    dp->Alpha = (unsigned char)CLAMP(a);
	    hold = *lp;
	    dp++;
	}

	rp = cp;
	r = (double)(lp->Red + cp->Red + rp->Red) * 0.333333333333333;
	g = (double)(lp->Green + cp->Green + rp->Green) * 0.333333333333333;
	b = (double)(lp->Blue + cp->Blue + rp->Blue) * 0.333333333333333;
	a = (double)(lp->Alpha + cp->Alpha + rp->Alpha) * 0.333333333333333;
	dp->Red = (unsigned char)CLAMP(r);
	dp->Green = (unsigned char)CLAMP(g);
	dp->Blue = (unsigned char)CLAMP(b);
	dp->Alpha = (unsigned char)CLAMP(a);

	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
}
    
static void
BoxY(Picture *destPtr, Picture *srcPtr)
{
    Pix32 *srcColumnPtr, *destColumnPtr;
    int x;

    srcColumnPtr = srcPtr->bits;
    destColumnPtr = destPtr->bits;
    for (x = 0; x < srcPtr->width; x++) {
	Pix32 *dp, *rp, *lp, *cp, *send;
	double r, g, b, a;
	Pix32 hold;

	dp = destColumnPtr;
	cp = lp = srcColumnPtr, rp = srcColumnPtr + srcPtr->pixelsPerRow;
	r = (lp->Red + cp->Red + rp->Red) * 0.333333333333333;
	g = (lp->Green + cp->Green + rp->Green) * 0.333333333333333;
	b = (lp->Blue + cp->Blue + rp->Blue) * 0.333333333333333;
	a = (lp->Alpha + cp->Alpha + rp->Alpha) * 0.333333333333333;
	hold = *lp;
	dp->Red = (unsigned char)CLAMP(r);
	dp->Green = (unsigned char)CLAMP(g);
	dp->Blue = (unsigned char)CLAMP(b);
	dp->Alpha = (unsigned char)CLAMP(a);
	dp += destPtr->pixelsPerRow;
	cp += srcPtr->pixelsPerRow;
	rp += srcPtr->pixelsPerRow;

	for (send = srcColumnPtr + (srcPtr->height * srcPtr->pixelsPerRow); 
	     rp < send; /* empty */) {
	    r = (lp->Red + cp->Red + rp->Red) * 0.333333333333333;
	    g = (lp->Green + cp->Green + rp->Green) * 0.333333333333333;
	    b = (lp->Blue + cp->Blue + rp->Blue) * 0.333333333333333;
	    a = (lp->Alpha + cp->Alpha + rp->Alpha) * 0.333333333333333;
	    hold = *lp;
	    dp->Red = (unsigned char)CLAMP(r);
	    dp->Green = (unsigned char)CLAMP(g);
	    dp->Blue = (unsigned char)CLAMP(b);
	    dp->Alpha = (unsigned char)CLAMP(a);
	    dp += destPtr->pixelsPerRow;
	    rp += srcPtr->pixelsPerRow;
	    lp += srcPtr->pixelsPerRow; 
	    cp += srcPtr->pixelsPerRow;
	}	
	rp = cp;
	r = (lp->Red + cp->Red + rp->Red) * 0.333333333333333;
	g = (lp->Green + cp->Green + rp->Green) * 0.333333333333333;
	b = (lp->Blue + cp->Blue + rp->Blue) * 0.333333333333333;
	a = (lp->Alpha + cp->Alpha + rp->Alpha) * 0.333333333333333;
	dp->Red = (unsigned char)CLAMP(r);
	dp->Green = (unsigned char)CLAMP(g);
	dp->Blue = (unsigned char)CLAMP(b);
	dp->Alpha = (unsigned char)CLAMP(a);
	srcColumnPtr++, destColumnPtr++;
    }
}

static void
TentHorizontally(
    Picture *destPtr,
    Picture *srcPtr)
{
    Pix32 *srcRowPtr, *destRowPtr;
    int y;

    srcRowPtr = srcPtr->bits;
    destRowPtr = destPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 left, center, right;
	Pix32 *dp, *sp, *send;

	dp = destRowPtr;
	sp = srcRowPtr + 1;
	left = *srcRowPtr, center = left, right = *sp;
	dp->Red = (left.Red + (center.Red << 1) + right.Red) >> 2;
	dp->Green = (left.Green + (center.Green << 1) + right.Green) >> 2;
	dp->Blue = (left.Blue + (center.Blue << 1) + right.Blue) >> 2;
	dp->Alpha = (left.Alpha + (center.Alpha << 1) + right.Alpha) >> 2;
	center = right;
	dp++, sp++;

	for (send = srcRowPtr + srcPtr->width; sp < send; /*empty*/) {
	    right = *sp;
	    dp->Red = (left.Red + (center.Red << 1) + right.Red) >> 2;
	    dp->Green = (left.Green + (center.Green << 1) + right.Green) >> 2;
	    dp->Blue = (left.Blue + (center.Blue << 1) + right.Blue) >> 2;
	    dp->Alpha = (left.Alpha + (center.Alpha << 1) + right.Alpha) >> 2;
	    left = center;
	    center = right;
	    dp++, sp++;
	}

	right = center;
	dp->Red = (left.Red + (center.Red << 1) + right.Red) >> 2;
	dp->Green = (left.Green + (center.Green << 1) + right.Green) >> 2;
	dp->Blue = (left.Blue + (center.Blue << 1) + right.Blue) >> 2;
	dp->Alpha = (left.Alpha + (center.Alpha << 1) + right.Alpha) >> 2;

	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
    return;
}

static void
TentVertically(
    Picture *destPtr,
    Picture *srcPtr)
{
    Pix32 *srcColumnPtr, *destColumnPtr;
    int x;

    srcColumnPtr = srcPtr->bits;
    destColumnPtr = destPtr->bits;
    for (x = 0; x < srcPtr->width; x++) {
	Pix32 left, center, right;
	Pix32 *dp, *sp, *send;

	dp = destColumnPtr;
	sp = srcColumnPtr + srcPtr->pixelsPerRow;
	left = *srcColumnPtr, center = left, right = *sp;
	dp->Red = (left.Red + (center.Red << 1) + right.Red) >> 2;
	dp->Green = (left.Green + (center.Green << 1) + right.Green) >> 2;
	dp->Blue = (left.Blue + (center.Blue << 1) + right.Blue) >> 2;
	dp->Alpha = (left.Alpha + (center.Alpha << 1) + right.Alpha) >> 2;
	center = right;
	dp += destPtr->pixelsPerRow;
	sp += srcPtr->pixelsPerRow;

	for (send = srcColumnPtr + (srcPtr->height * srcPtr->pixelsPerRow); 
	     sp < send; /* empty */) {
	    right = *sp;
	    dp->Red = (left.Red + (center.Red << 1) + right.Red) >> 2;
	    dp->Green = (left.Green + (center.Green << 1) + right.Green) >> 2;
	    dp->Blue = (left.Blue + (center.Blue << 1) + right.Blue) >> 2;
	    dp->Alpha = (left.Alpha + (center.Alpha << 1) + right.Alpha) >> 2;
	    left = center;
	    center = right;
	    dp += destPtr->pixelsPerRow;
	    sp += srcPtr->pixelsPerRow;
	}	
	right = center;
	dp->Red = (left.Red + (center.Red << 1) + right.Red) >> 2;
	dp->Green = (left.Green + (center.Green << 1) + right.Green) >> 2;
	dp->Blue = (left.Blue + (center.Blue << 1) + right.Blue) >> 2;
	dp->Alpha = (left.Alpha + (center.Alpha << 1) + right.Alpha) >> 2;

	srcColumnPtr++, destColumnPtr++;
    }
}

static void
DarkenPicture(
    Picture *destPtr,
    Picture *s1Ptr, 
    Picture *s2Ptr)
{
    Pix32 *s1RowPtr, *s2RowPtr, *destRowPtr;
    int y;
    
    s1RowPtr = s1Ptr->bits;
    s2RowPtr = s2Ptr->bits;
    destRowPtr = destPtr->bits;

    for (y = 0; y < destPtr->height; y++) {
	Pix32 *s1, *s2, *dp, *dend;

	s1 = s1RowPtr, s2 = s2RowPtr;
	for (dp = destRowPtr, dend = dp + destPtr->width; dp < dend; 
	     /*empty*/) {
	    int i;

	    i = s1->Red - s2->Red;
	    dp->Red =   (i < 0) ? 0 : i;
	    i = s1->Green - s2->Green;
	    dp->Green = (i < 0) ? 0 : i;
	    i = s1->Blue - s2->Blue;
	    dp->Blue =  (i < 0) ? 0 : i;
	    i = s1->Alpha - s2->Alpha;
	    dp->Alpha = (i < 0) ? 0 : i;
	    s1++, s2++, dp++;
	}
	destRowPtr += destPtr->pixelsPerRow;
	s1RowPtr += s1Ptr->pixelsPerRow;
	s2RowPtr += s2Ptr->pixelsPerRow;
    }
}

static void
ApplyPictureToPicture(
    Picture *destPtr, 
    Picture *srcPtr, 
    Blt_PictureArithOps op)
{
    Pix32 *srcRowPtr, *destRowPtr;
    int width, height;
    int y;

    /* If the picture sizes are different use the smaller dimension. */
    width = MIN(srcPtr->width, destPtr->width);
    height = MIN(srcPtr->height, destPtr->height);

    srcRowPtr = srcPtr->bits;
    destRowPtr = destPtr->bits;
    for (y = 0; y < height; y++) {
	Pix32 *sp, *dp, *dend;
	
	sp = srcRowPtr;
	dp = destRowPtr, dend = dp + width;
	switch(op) {
	case PICTURE_ARITH_ADD:
	    while (dp < dend) {
		int i;
		i = dp->Red + sp->Red;
		dp->Red = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		i = dp->Green + sp->Green;
		dp->Green = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		i = dp->Blue + sp->Blue;
		dp->Blue =  (i > MAXINTENSITY) ? MAXINTENSITY : i;
		i = dp->Alpha + sp->Alpha;
		dp->Alpha = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		sp++, dp++;
	    }
	    break;
	    
	case PICTURE_ARITH_SUB:
	    while (dp < dend) {
		int i;

		i = dp->Red - sp->Red;
		dp->Red =   (i < 0) ? 0 : i;
		i = dp->Green - sp->Green;
		dp->Green = (i < 0) ? 0 : i;
		i = dp->Blue - sp->Blue;
		dp->Blue =  (i < 0) ? 0 : i;
		i = dp->Alpha - sp->Alpha;
		dp->Alpha = (i < 0) ? 0 : i;
		sp++, dp++;
	    }
	    break;

	case PICTURE_ARITH_RSUB:
	    while (dp < dend) {
		int i;

		i = sp->Red - dp->Red;
		dp->Red =   (i < 0) ? 0 : i;
		i = sp->Green - dp->Green;
		dp->Green = (i < 0) ? 0 : i;
		i = sp->Blue - dp->Blue;
		dp->Blue =  (i < 0) ? 0 : i;
		i = sp->Alpha - dp->Alpha;
		dp->Alpha = (i < 0) ? 0 : i;
		sp++, dp++;
	    }
	    break;

	case PICTURE_ARITH_AND:
	    while (dp < dend) {
		dp->Red = (dp->Red & sp->Red);
		dp->Green = (dp->Green & sp->Green);
		dp->Blue = (dp->Blue & sp->Blue);
		dp->Alpha = (dp->Alpha & sp->Alpha);
		sp++, dp++;
	    }
	    break;

	case PICTURE_ARITH_OR:
	    while (dp < dend) {
		dp->Red = (dp->Red | sp->Red);
		dp->Green = (dp->Green | sp->Green);
		dp->Blue = (dp->Blue | sp->Blue);
		dp->Alpha = (dp->Alpha | sp->Alpha);
		sp++, dp++;
	    }
	    break;

	case PICTURE_ARITH_NAND:
	    while (dp < dend) {
		dp->Red = ~(dp->Red & sp->Red);
		dp->Green = ~(dp->Green & sp->Green);
		dp->Blue = ~(dp->Blue & sp->Blue);
		dp->Alpha = ~(dp->Alpha & sp->Alpha);
		sp++, dp++;
	    }
	    break;

	case PICTURE_ARITH_NOR:
	    while (dp < dend) {
		dp->Red = ~(dp->Red | sp->Red);
		dp->Green = ~(dp->Green | sp->Green);
		dp->Blue = ~(dp->Blue | sp->Blue);
		dp->Alpha = ~(dp->Alpha | sp->Alpha);
		sp++, dp++;
	    }
	    break;

	case PICTURE_ARITH_XOR:
	    while (dp < dend) {
		dp->Red = (dp->Red ^ sp->Red);
		dp->Green = (dp->Green ^ sp->Green);
		dp->Blue = (dp->Blue ^ sp->Blue);
		dp->Alpha = (dp->Alpha ^ sp->Alpha);
		sp++, dp++;
	    }
	    break;

	case PICTURE_ARITH_MIN:
	    while (dp < dend) {
		dp->Red = MIN(dp->Red, sp->Red);
		dp->Green = MIN(dp->Green, sp->Green);
		dp->Blue = MIN(dp->Blue, sp->Blue);
		dp->Alpha = MIN(dp->Alpha, sp->Alpha);
		sp++, dp++;
	    }
	    break;

	case PICTURE_ARITH_MAX:
	    while (dp < dend) {
		dp->Red = MAX(dp->Red, sp->Red);
		dp->Green = MAX(dp->Green, sp->Green);
		dp->Blue = MAX(dp->Blue, sp->Blue);
		dp->Alpha = MAX(dp->Alpha, sp->Alpha);
		sp++, dp++;
	    }
	    break;

	}
	destRowPtr += destPtr->pixelsPerRow;
	srcRowPtr += srcPtr->pixelsPerRow;
    }
}


static void
ApplyScalarToPicture(
    Picture *srcPtr, 
    Pix32 *colorPtr,
    Blt_PictureArithOps op)
{
    Pix32 *srcRowPtr;
    int y;
    
    srcRowPtr = srcPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp, *send;
	
	sp = srcRowPtr;
	send = sp + srcPtr->width;
	switch(op) {
	case PICTURE_ARITH_ADD:
	    while (sp < send) {
		int i;
		
		i = sp->Red + colorPtr->Red;
		sp->Red = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		i = sp->Green + colorPtr->Green;
		sp->Green = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		i = sp->Blue + colorPtr->Blue;
		sp->Blue =  (i > MAXINTENSITY) ? MAXINTENSITY : i;
		i = sp->Alpha + colorPtr->Alpha;
		sp->Alpha = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		sp++;
	    }
	    break;

	case PICTURE_ARITH_SUB:
	    while (sp < send) {
		int i;

		i = sp->Red - colorPtr->Red;
		sp->Red =   (i < 0) ? 0 : i;
		i = sp->Green - colorPtr->Green;
		sp->Green = (i < 0) ? 0 : i;
		i = sp->Blue - colorPtr->Blue;
		sp->Blue =  (i < 0) ? 0 : i;
		i = sp->Alpha - colorPtr->Alpha;
		sp->Alpha = (i < 0) ? 0 : i;
		sp++;
	    }
	    break;

	case PICTURE_ARITH_RSUB:
	    while (sp < send) {
		int i;

		i = colorPtr->Red - sp->Red;
		sp->Red =   (i < 0) ? 0 : i;
		i = colorPtr->Green - sp->Green;
		sp->Green = (i < 0) ? 0 : i;
		i = colorPtr->Blue - sp->Blue;
		sp->Blue =  (i < 0) ? 0 : i;
		i = colorPtr->Alpha - sp->Alpha;
		sp->Alpha = (i < 0) ? 0 : i;
		sp++;
	    }
	    break;

	case PICTURE_ARITH_AND:
	    while (sp < send) {
		sp->Red = (sp->Red & colorPtr->Red);
		sp->Green = (sp->Green & colorPtr->Green);
		sp->Blue = (sp->Blue & colorPtr->Blue);
		sp->Alpha = (sp->Alpha & colorPtr->Alpha);
		sp++;
	    }
	    break;

	case PICTURE_ARITH_OR:
	    while (sp < send) {
		sp->Red = (sp->Red | colorPtr->Red);
		sp->Green = (sp->Green | colorPtr->Green);
		sp->Blue = (sp->Blue | colorPtr->Blue);
		sp->Alpha = (sp->Alpha | colorPtr->Alpha);
		sp++;
	    }
	    break;

	case PICTURE_ARITH_NAND:
	    while (sp < send) {
		sp->Red = ~(sp->Red & colorPtr->Red);
		sp->Green = ~(sp->Green & colorPtr->Green);
		sp->Blue = ~(sp->Blue & colorPtr->Blue);
		sp->Alpha = ~(sp->Alpha & colorPtr->Alpha);
		sp++;
	    }
	    break;

	case PICTURE_ARITH_NOR:
	    while (sp < send) {
		sp->Red = ~(sp->Red | colorPtr->Red);
		sp->Green = ~(sp->Green | colorPtr->Green);
		sp->Blue = ~(sp->Blue | colorPtr->Blue);
		sp->Alpha = ~(sp->Alpha | colorPtr->Alpha);
		sp++;
	    }
	    break;

	case PICTURE_ARITH_XOR:
	    while (sp < send) {
		sp->Red = ~(sp->Red ^ colorPtr->Red);
		sp->Green = ~(sp->Green ^ colorPtr->Green);
		sp->Blue = ~(sp->Blue ^ colorPtr->Blue);
		sp->Alpha = ~(sp->Alpha ^ colorPtr->Alpha);
		sp++;
	    }
	    break;

	case PICTURE_ARITH_MIN:
	    while (sp < send) {
		sp->Red = MIN(sp->Red, colorPtr->Red);
		sp->Green = MIN(sp->Green, colorPtr->Green);
		sp->Blue = MIN(sp->Blue, colorPtr->Blue);
		sp->Alpha = MIN(sp->Alpha, colorPtr->Alpha);
		sp++;
	    }
	    break;
	case PICTURE_ARITH_MAX:
	    while (sp < send) {
		sp->Red = MAX(sp->Red, colorPtr->Red);
		sp->Green = MAX(sp->Green, colorPtr->Green);
		sp->Blue = MAX(sp->Blue, colorPtr->Blue);
		sp->Alpha = MAX(sp->Alpha, colorPtr->Alpha);
		sp++;
	    }
	    break;

	}
	srcRowPtr += srcPtr->pixelsPerRow;
    }
}


void
Blt_ApplyPictureToPicture(
    Picture *destPtr, 
    Picture *srcPtr, 
    Blt_PictureArithOps op)
{
#ifdef HAVE_X86_ASM
    if (bltUseMMX == 0) {
	Blt_ApplyPictureToPictureMMX(destPtr, srcPtr, op);
    } else {
	ApplyPictureToPicture(destPtr, srcPtr, op);
    }
#else
    ApplyPictureToPicture(destPtr, srcPtr, op);
#endif /* HAVE_X86_ASM */
}

void
Blt_ApplyScalarToPicture(
    Blt_Picture picture, 
    Pix32 *colorPtr, 
    Blt_PictureArithOps op)
{
#ifdef HAVE_X86_ASM
    if (bltUseMMX) {
	Blt_ApplyScalarToPictureMMX(picture, colorPtr, op);
    } else {
	ApplyScalarToPicture(picture, colorPtr, op);
    }
#else
    ApplyScalarToPicture(picture, colorPtr, op);
#endif /* HAVE_X86_ASM */
}

void
Blt_ApplyPictureToPictureWithMatte(
    Picture *destPtr, 
    Picture *srcPtr, 
    Picture *mattePtr,
    int invert,
    Blt_PictureArithOps op)
{
    Pix32 *srcRowPtr, *destRowPtr, *matteRowPtr;
    int width, height;
    int y;
    unsigned int off;

    /* If the picture sizes are different use the smaller dimension. */
    width = MIN3(srcPtr->width, destPtr->width, mattePtr->width);
    height = MIN3(srcPtr->height, destPtr->height, mattePtr->height);

    off = (invert) ? (unsigned int)-1 : 0;
    srcRowPtr = srcPtr->bits;
    destRowPtr = destPtr->bits;
    matteRowPtr = mattePtr->bits;
    for (y = 0; y < height; y++) {
	Pix32 *sp, *dp, *mp, *mend;
	
	sp = srcRowPtr, dp = destRowPtr;
	mp = matteRowPtr, mend = mp + width;
	switch(op) {
	case PICTURE_ARITH_ADD:
	    while (mp < mend) {
		if (mp->color != off) {
		    int i;

		    i = dp->Red + sp->Red;
		    dp->Red = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		    i = dp->Green + sp->Green;
		    dp->Green = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		    i = dp->Blue + sp->Blue;
		    dp->Blue =  (i > MAXINTENSITY) ? MAXINTENSITY : i;
		    i = dp->Alpha + sp->Alpha;
		    dp->Alpha = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		}
		sp++, dp++, mp++;
	    }
	    break;
	    
	case PICTURE_ARITH_SUB:
	    while (mp < mend) {
		if (mp->color != off) {
		    int i;
		    
		    i = dp->Red - sp->Red;
		    dp->Red =   (i < 0) ? 0 : i;
		    i = dp->Green - sp->Green;
		    dp->Green = (i < 0) ? 0 : i;
		    i = dp->Blue - sp->Blue;
		    dp->Blue =  (i < 0) ? 0 : i;
		    i = dp->Alpha - sp->Alpha;
		    dp->Alpha = (i < 0) ? 0 : i;
		}
		sp++, dp++, mp++;
	    }
	    break;
	    
	case PICTURE_ARITH_RSUB:
	    while (mp < mend) {
		if (mp->color != off) {
		    int i;
		    
		    i = sp->Red - dp->Red;
		    dp->Red =   (i < 0) ? 0 : i;
		    i = sp->Green - dp->Green;
		    dp->Green = (i < 0) ? 0 : i;
		    i = sp->Blue - dp->Blue;
		    dp->Blue =  (i < 0) ? 0 : i;
		    i = sp->Alpha - dp->Alpha;
		    dp->Alpha = (i < 0) ? 0 : i;
		}
		sp++, dp++, mp++;
	    }
	    break;

	case PICTURE_ARITH_AND:
	    while (mp < mend) {
		if (mp->color != off) {
		    dp->Red = (dp->Red & sp->Red);
		    dp->Green = (dp->Green & sp->Green);
		    dp->Blue = (dp->Blue & sp->Blue);
		    dp->Alpha = (dp->Alpha & sp->Alpha);
		}
		sp++, dp++, mp++;
	    }
	    break;
	    
	case PICTURE_ARITH_OR:
	    while (mp < mend) {
		if (mp->color != off) {
		    dp->Red = (dp->Red | sp->Red);
		    dp->Green = (dp->Green | sp->Green);
		    dp->Blue = (dp->Blue | sp->Blue);
		    dp->Alpha = (dp->Alpha | sp->Alpha);
		}
		sp++, dp++, mp++;
	    }
	    break;
		
	case PICTURE_ARITH_NAND:
	    while (mp < mend) {
		if (mp->color != off) {
		    dp->Red = ~(dp->Red & sp->Red);
		    dp->Green = ~(dp->Green & sp->Green);
		    dp->Blue = ~(dp->Blue & sp->Blue);
		    dp->Alpha = ~(dp->Alpha & sp->Alpha);
		}
		sp++, dp++, mp++;
	    }
	    break;
	    
	case PICTURE_ARITH_NOR:
	    while (mp < mend) {
		if (mp->color != off) {
		    dp->Red = ~(dp->Red | sp->Red);
		    dp->Green = ~(dp->Green | sp->Green);
		    dp->Blue = ~(dp->Blue | sp->Blue);
		    dp->Alpha = ~(dp->Alpha | sp->Alpha);
		}
		sp++, dp++, mp++;
	    }
	    break;
	    
	case PICTURE_ARITH_XOR:
	    while (mp < mend) {
		if (mp->color != off) {
		    dp->Red = (dp->Red ^ sp->Red);
		    dp->Green = (dp->Green ^ sp->Green);
		    dp->Blue = (dp->Blue ^ sp->Blue);
		    dp->Alpha = (dp->Alpha ^ sp->Alpha);
		}
		sp++, dp++, mp++;
	    }
	    break;
	    
	case PICTURE_ARITH_MIN:
	    while (mp < mend) {
		if (mp->color != off) {
		    dp->Red = MIN(dp->Red, sp->Red);
		    dp->Green = MIN(dp->Green, sp->Green);
		    dp->Blue = MIN(dp->Blue, sp->Blue);
		    dp->Alpha = MIN(dp->Alpha, sp->Alpha);
		}
		sp++, dp++, mp++;
	    }
	    break;
	    
	case PICTURE_ARITH_MAX:
	    while (mp < mend) {
		if (mp->color != off) {
		    dp->Red = MAX(dp->Red, sp->Red);
		    dp->Green = MAX(dp->Green, sp->Green);
		    dp->Blue = MAX(dp->Blue, sp->Blue);
		    dp->Alpha = MAX(dp->Alpha, sp->Alpha);
		}
		sp++, dp++, mp++;
	    }
	    break;
	    
	}
	destRowPtr += destPtr->pixelsPerRow;
	srcRowPtr += srcPtr->pixelsPerRow;
	matteRowPtr += mattePtr->pixelsPerRow;
    }
}

void
Blt_ApplyScalarToPictureWithMatte(
    Picture *srcPtr, 
    Pix32 *colorPtr,
    Picture *mattePtr, 
    int invert,
    Blt_PictureArithOps op)
{
    Pix32 *srcRowPtr, *matteRowPtr;
    int width, height;
    int y;
    unsigned int off;

    width = MIN(srcPtr->width, mattePtr->width);
    height = MIN(srcPtr->height, mattePtr->height);
    srcRowPtr = srcPtr->bits;
    matteRowPtr = mattePtr->bits;
    off = (invert) ? (unsigned int)-1 : 0;
    for (y = 0; y < height; y++) {
	Pix32 *sp, *mp, *mend;
	
	sp = srcRowPtr;
	mp = matteRowPtr, mend = mp + width;
	switch(op) {
	case PICTURE_ARITH_ADD:
	    while (mp < mend) {
		if (mp->color != off) {
		    int i;
		    
		    i = sp->Red + colorPtr->Red;
		    sp->Red = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		    i = sp->Green + colorPtr->Green;
		    sp->Green = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		    i = sp->Blue + colorPtr->Blue;
		    sp->Blue =  (i > MAXINTENSITY) ? MAXINTENSITY : i;
		    i = sp->Alpha + colorPtr->Alpha;
		    sp->Alpha = (i > MAXINTENSITY) ? MAXINTENSITY : i;
		}
		sp++, mp++;
	    }
	    break;

	case PICTURE_ARITH_SUB:
	    while (mp < mend) {
		if (mp->color != off) {
		    int i;
		    
		    i = sp->Red - colorPtr->Red;
		    sp->Red =   (i < 0) ? 0 : i;
		    i = sp->Green - colorPtr->Green;
		    sp->Green = (i < 0) ? 0 : i;
		    i = sp->Blue - colorPtr->Blue;
		    sp->Blue =  (i < 0) ? 0 : i;
		    i = sp->Alpha - colorPtr->Alpha;
		    sp->Alpha = (i < 0) ? 0 : i;
		}
		sp++, mp++;
	    }
	    break;

	case PICTURE_ARITH_RSUB:
	    while (mp < mend) {
		if (mp->color != off) {
		    int i;
		    
		    i = colorPtr->Red - sp->Red;
		    sp->Red =   (i < 0) ? 0 : i;
		    i = colorPtr->Green - sp->Green;
		    sp->Green = (i < 0) ? 0 : i;
		    i = colorPtr->Blue - sp->Blue;
		    sp->Blue =  (i < 0) ? 0 : i;
		    i = colorPtr->Alpha - sp->Alpha;
		    sp->Alpha = (i < 0) ? 0 : i;
		}
		sp++, mp++;
	    }
	    break;

	case PICTURE_ARITH_AND:
	    while (mp < mend) {
		if (mp->color != off) {
		    sp->Red = (sp->Red & colorPtr->Red);
		    sp->Green = (sp->Green & colorPtr->Green);
		    sp->Blue = (sp->Blue & colorPtr->Blue);
		    sp->Alpha = (sp->Alpha & colorPtr->Alpha);
		}
	    }
	    sp++, mp++;
	    break;

	case PICTURE_ARITH_OR:
	    while (mp < mend) {
		if (mp->color != off) {
		    sp->Red = (sp->Red | colorPtr->Red);
		    sp->Green = (sp->Green | colorPtr->Green);
		    sp->Blue = (sp->Blue | colorPtr->Blue);
		    sp->Alpha = (sp->Alpha | colorPtr->Alpha);
		}
		sp++, mp++;
	    }
	    break;

	case PICTURE_ARITH_NAND:
	    while (mp < mend) {
		if (mp->color != off) {
		    sp->Red = ~(sp->Red & colorPtr->Red);
		    sp->Green = ~(sp->Green & colorPtr->Green);
		    sp->Blue = ~(sp->Blue & colorPtr->Blue);
		    sp->Alpha = ~(sp->Alpha & colorPtr->Alpha);
		}
		sp++, mp++;
	    }
	    break;

	case PICTURE_ARITH_NOR:
	    while (mp < mend) {
		if (mp->color != off) {
		    sp->Red = ~(sp->Red | colorPtr->Red);
		    sp->Green = ~(sp->Green | colorPtr->Green);
		    sp->Blue = ~(sp->Blue | colorPtr->Blue);
		    sp->Alpha = ~(sp->Alpha | colorPtr->Alpha);
		}
		sp++, mp++;
	    }
	    break;

	case PICTURE_ARITH_XOR:
	    while (mp < mend) {
		if (mp->color != off) {
		    sp->Red = ~(sp->Red ^ colorPtr->Red);
		    sp->Green = ~(sp->Green ^ colorPtr->Green);
		    sp->Blue = ~(sp->Blue ^ colorPtr->Blue);
		    sp->Alpha = ~(sp->Alpha ^ colorPtr->Alpha);
		}
		sp++, mp++;
	    }
	    break;

	case PICTURE_ARITH_MIN:
	    while (mp < mend) {
		if (mp->color != off) {
		    sp->Red = MIN(sp->Red, colorPtr->Red);
		    sp->Green = MIN(sp->Green, colorPtr->Green);
		    sp->Blue = MIN(sp->Blue, colorPtr->Blue);
		    sp->Alpha = MIN(sp->Alpha, colorPtr->Alpha);
		}
		sp++, mp++;
	    }
	    break;
	case PICTURE_ARITH_MAX:
	    while (mp < mend) {
		if (mp->color != off) {
		    sp->Red = MAX(sp->Red, colorPtr->Red);
		    sp->Green = MAX(sp->Green, colorPtr->Green);
		    sp->Blue = MAX(sp->Blue, colorPtr->Blue);
		    sp->Alpha = MAX(sp->Alpha, colorPtr->Alpha);
		}
		sp++, mp++;
	    }
	    break;

	}
	srcRowPtr += srcPtr->pixelsPerRow;
	matteRowPtr += mattePtr->pixelsPerRow;
    }
}

void
Blt_MultiplyPixels(
    Picture *srcPtr, 
    float scalar)
{
    Pix32 *srcRowPtr;
    short int s15;
    int y;
    float x;
    int nBits, bias;

    x = FABS(scalar);
    if (FABS(x) > 127.0) {
	return;
    }
    /* Figure out how many bits we need. */
    nBits = 0;
    while ((1 << nBits) < x) {
	nBits++;
    }
    nBits = 15 - nBits;
    bias = (1 << nBits) / 2;
    if (scalar < 0) {
	bias = -bias;
    }
    
    s15 = (short int)((scalar * (float)(1 << nBits)) + 0.5);

    srcRowPtr = srcPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp, *send;
	
	sp = srcRowPtr;
	for (sp = srcRowPtr, send = sp + srcPtr->width; sp < send; sp++) {
	    short int r15, g15, b15, a15;
	    int i31;
	    short int i15;
	    
	    r15 = ((sp->Red << 8) + sp->Red) >> 1;
	    g15 = ((sp->Green << 8) + sp->Green) >> 1;
	    b15 = ((sp->Blue << 8) + sp->Blue) >> 1;
	    a15 = ((sp->Alpha << 8) + sp->Alpha) >> 1;
	    
	    i31 = r15 * s15;
	    i15 = i31 >> 16;	/* Truncate lower 16 bits */
	    i15 =+ bias;
	    i15 >>= nBits;
	    sp->Red = (i15 > 255) ? 255 : i15;
	    
	    i31 = g15 * s15;
	    i15 = i31 >> 16;	/* Truncate lower 16 bits */
	    i15 =+ bias;
	    i15 >>= nBits;
	    sp->Green = (i15 > 255) ? 255 : i15;
	    
	    i31 = b15 * s15;
	    i15 = i31 >> 16;	/* Truncate lower 16 bits */
	    i15 =+ bias;
	    i15 >>= nBits;
	    sp->Green = (i15 > 255) ? 255 : i15;
	    
	    i31 = a15 * s15;
	    i15 = i31 >> 16;	/* Truncate lower 16 bits */
	    i15 =+ bias;
	    i15 >>= nBits;
	    sp->Alpha = (i15 > 255) ? 255 : i15;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
    }
}

static void
SelectPixels(
    Picture *destPtr,
    Picture *srcPtr, 
    Pix32 *lowPtr,
    Pix32 *highPtr)
{
    Pix32 *srcRowPtr, *destRowPtr;
    int y;

    destRowPtr = destPtr->bits, srcRowPtr = srcPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *dp, *sp, *send;

	dp = destRowPtr;
	for(sp = srcRowPtr, send = sp + srcPtr->width; sp < send; sp++, dp++) {
	    if ((sp->Red >= lowPtr->Red) && (sp->Red <= highPtr->Red) &&
		(sp->Green >= lowPtr->Green) && (sp->Green <= highPtr->Green) &&
		(sp->Blue >= lowPtr->Blue) && (sp->Blue <= highPtr->Blue) &&
		(sp->Alpha >= lowPtr->Alpha) && (sp->Alpha <= highPtr->Alpha)) {
		dp->color = 0xFFFFFFFF;
	    } else {
		dp->color = 0;
	    }
	}
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
}

void
Blt_SelectPixels(
    Picture *destPtr,		/* (out) Resulting mask. */
    Picture *srcPtr,		/* Picture to be examined. */
    Pix32 *lowPtr,		/* Lower bound of test. */
    Pix32 *highPtr)		/* Upper bound of test. */
{
    if (srcPtr != destPtr) {
	if (!Blt_ReallocatePicture(destPtr, srcPtr->width, srcPtr->height, 0)) {
	    return;
	}
    }
#ifdef HAVE_X86_ASM
    if (bltUseMMX) {
	Blt_SelectPixelsMMX(destPtr, srcPtr, lowPtr, highPtr);
    } else {
	SelectPixels(destPtr, srcPtr, lowPtr, highPtr);
    }
#else
    SelectPixels(destPtr, srcPtr, lowPtr, highPtr);
#endif /* HAVE_X86_ASM */
    destPtr->flags &= ~BLT_PICTURE_BLEND;
    destPtr->flags |= BLT_PICTURE_MASK;
}

void
Blt_BoxFilterPicture(Picture *destPtr, Picture *srcPtr)
{
    if (Blt_ReallocatePicture(destPtr, srcPtr->width, srcPtr->height, FALSE)) {
	BoxY(destPtr, srcPtr);
	BoxX(destPtr, destPtr);    
    }
}

void
Blt_TentFilterPicture(Picture *destPtr, Picture *srcPtr)
{
    if (Blt_ReallocatePicture(destPtr, srcPtr->width, srcPtr->height, FALSE)) {
#ifdef HAVE_X86_ASM
	if (bltUseMMX) {
	    Blt_TentVerticallyMMX(destPtr, srcPtr);
	    Blt_TentHorizontallyMMX(destPtr, destPtr);    
	    asm volatile ("emms");
	} else {
	    TentVertically(destPtr, srcPtr);
	    TentHorizontally(destPtr, destPtr);    
	}
#else
	TentVertically(destPtr, srcPtr);
	TentHorizontally(destPtr, destPtr);    
#endif /* HAVE_X86_ASM */
    }
}

void
Blt_BlurPicture(Blt_Picture picture)
{
    Blt_Picture tmp;

    tmp = Blt_CreatePicture(Blt_PictureWidth(picture), 
			    Blt_PictureHeight(picture));
    BoxX(tmp, picture), BoxY(picture, tmp);
    BoxX(tmp, picture), BoxY(picture, tmp); 
    BoxX(tmp, picture), BoxY(picture, tmp);
    BoxX(tmp, picture), BoxY(picture, tmp);
    Blt_FreePicture(tmp);
}


void
Blt_SharpenPicture(Blt_Picture dest, Blt_Picture src)
{
    Blt_Picture blur1, blur2, tmp;

    blur1 = Blt_CreatePicture(Blt_PictureWidth(src), Blt_PictureHeight(src));
    blur2 = Blt_CreatePicture(Blt_PictureWidth(src), Blt_PictureHeight(src));

    BoxY(blur1, src),   BoxX(blur2, blur1);	
    BoxY(blur1, blur2), BoxX(blur2, blur1);
    BoxY(blur1, blur2), BoxX(blur2, blur1);	
    BoxY(blur1, blur2), BoxX(blur2, blur1);
    Blt_FreePicture(blur1);

    /* 
     * tmp = src - blur;
     * dest = src + tmp
     */
    tmp = Blt_ClonePicture(src);
    Blt_SubtractPictures(tmp, blur2);
    Blt_AddPictures(tmp, src);
    Blt_FreePicture(blur2);
    Blt_CopyPicture(dest, tmp); 
    Blt_FreePicture(tmp);
}

void
Blt_SharpenPicture2(Blt_Picture dest, Blt_Picture src)
{
    Blt_Picture blur, tmp;

    blur = Blt_CreatePicture(Blt_PictureWidth(src), Blt_PictureHeight(src));
#ifdef HAVE_X86_ASM
    if (bltUseMMX) {
	Blt_TentVerticallyMMX(blur, src), Blt_TentHorizontallyMMX(blur, blur);  
	Blt_TentVerticallyMMX(blur, blur), Blt_TentHorizontallyMMX(blur, blur); 
	Blt_TentVerticallyMMX(blur, blur), Blt_TentHorizontallyMMX(blur, blur); 
	Blt_TentVerticallyMMX(blur, blur), Blt_TentHorizontallyMMX(blur, blur); 
    } else {
	TentVertically(blur, src),  TentHorizontally(blur, blur);    
	TentVertically(blur, blur), TentHorizontally(blur, blur);    
	TentVertically(blur, blur), TentHorizontally(blur, blur);    
	TentVertically(blur, blur), TentHorizontally(blur, blur);    
    }
#else
    TentVertically(blur, src),  TentHorizontally(blur, blur);    
    TentVertically(blur, blur), TentHorizontally(blur, blur);    
    TentVertically(blur, blur), TentHorizontally(blur, blur);    
    TentVertically(blur, blur), TentHorizontally(blur, blur);    
#endif
    /* 
     * tmp = src - blur;
     * dest = src + tmp
     */
    tmp = Blt_ClonePicture(src);
    Blt_SubtractPictures(tmp, blur);
    Blt_AddPictures(tmp, src);
    Blt_FreePicture(blur);
    Blt_CopyPicture(dest, tmp); 
    Blt_FreePicture(tmp);
}

size_t
ComputeWeights(
    int srcWidth, int destWidth,
    PictureFilter *filterPtr,
    Sample **samplePtrPtr)
{
    Sample *samples;
    double scale;
    int bytesPerSample;

    /* Pre-calculate filter contributions for a row */
    scale = (double)destWidth / (double)srcWidth;
    if (scale < 1.0) {
	Sample *samplePtr;
	double radius, fscale;
	int filterSize, x;

	/* Downsample */

	radius = filterPtr->support / scale;
	fscale = 1.0 / scale;
	filterSize = (int)(radius * 2 + 2);

	bytesPerSample = sizeof(Sample) + 
	    ((filterSize - 1) * sizeof(PixelWeight));
	samples = Blt_Calloc(destWidth, bytesPerSample);
	assert(samples);

	samplePtr = samples;
#define DEBUG 0
#if DEBUG
	fprintf(stderr, "downscale=%g, fscale=%g, radius=%g\n",
		    scale, fscale, radius);
#endif
	for (x = 0; x < destWidth; x++) {
	    PixelWeight *wp;
	    double center;
	    int i, left, right;	/* Filter bounds. */

	    center = ((double)x + 0.5) * fscale;

	    /* Determine bounds of filter and its density. */
	    left = (int)/*floor*/(center - radius);
	    if (left < 0) {
		left = 0;
	    }
	    right = (int)/*floor*/(center + radius);
	    if (right >= srcWidth) {
		right = srcWidth - 1;
	    }
	    samplePtr->start = left;
	    samplePtr->wend = samplePtr->weights + (right - left + 1);

	    for (wp = samplePtr->weights, i = left; i <= right; i++, wp++) {
		wp->f32 = (float)
		    (*filterPtr->proc)(((double)i - center) * scale);
		wp->i32 = float2si(wp->f32);
	    }
	    samplePtr = (Sample *)((char *)samplePtr + bytesPerSample);
	}
    } else {
	Sample *samplePtr;
	double fscale;
	int filterSize, x;

	/* Upsample */

	filterSize = (int)(filterPtr->support * 2 + 2);
	bytesPerSample = sizeof(Sample) + 
	    ((filterSize - 1) * sizeof(PixelWeight));
	samples = Blt_Calloc(destWidth, bytesPerSample);
	assert(samples);

	fscale = 1.0 / scale;

	samplePtr = samples;
#if DEBUG
	fprintf(stderr, "upscale=%g, fscale=%g, radius=%g\n",
		    scale, fscale, filterPtr->support);
#endif
	for (x = 0; x < destWidth; x++) {
	    PixelWeight *wp;
	    double center;
	    int i, left, right;	/* Filter bounds. */

	    center = ((double)x + 0.5) * fscale;
	    left = (int)(center - filterPtr->support);
	    if (left < 0) {
		left = 0;
	    }
	    right = (int)(center + filterPtr->support);
	    if (right >= srcWidth) {
		right = srcWidth - 1;
	    }
	    samplePtr->start = left;
	    samplePtr->wend = samplePtr->weights + (right - left + 1);

	    /* Sum the contributions for each pixel in the filter. */
	    for (wp = samplePtr->weights, i = left; i <= right;i++, wp++) {
		wp->f32 = (float) (*filterPtr->proc)((double)i - center);
		wp->i32 = float2si(wp->f32);
	    }
	    /* Go get the next sample. */
	    samplePtr = (Sample *)((char *)samplePtr + bytesPerSample);
	}
    }
    *samplePtrPtr = samples;
    return bytesPerSample;
}

static void
ConvolvePictureVertically(
    Picture *destPtr, Picture *srcPtr, 
    Blt_PictureFilter filter)
{
    Sample *samples, *send;
    int x;
    int bytesPerSample;		/* Size of sample. */

    /* Pre-calculate filter contributions for each row. */
    bytesPerSample = ComputeWeights(destPtr->height, destPtr->height, 
	filter, &samples);
    send = (Sample *)((char *)samples + (destPtr->height * bytesPerSample));

    /* Apply filter to each row. */
    for (x = 0; x < srcPtr->width; x++) {
	Pix32 *dp, *srcColumnPtr;
	Sample *samplePtr;

	srcColumnPtr = srcPtr->bits + x;
	dp = destPtr->bits + x;
	for (samplePtr = samples; samplePtr < send; 
	     samplePtr = (Sample *)((char *)samplePtr + bytesPerSample)) {
	    Pix32 *sp;
	    int r, g, b, a;
	    PixelWeight *wp;

	    r = g = b = a = 0;
	    sp = srcColumnPtr + (samplePtr->start * srcPtr->pixelsPerRow);
	    for (wp = samplePtr->weights; wp < samplePtr->wend; wp++) {
		a += wp->i32 * sp->Alpha;
		r += wp->i32 * sp->Red;
		g += wp->i32 * sp->Green;
		b += wp->i32 * sp->Blue;
		sp += srcPtr->pixelsPerRow;
	    }
	    dp->Red = SICLAMP(r);
	    dp->Green = SICLAMP(g);
	    dp->Blue = SICLAMP(b);
	    dp->Alpha = SICLAMP(a);
#ifdef notdef
	    if (dp->Alpha != 0xFF) {
		fprintf(stdout, "v1: alpha=0x%x\n", dp->Alpha);
	    }
#endif
	    dp += destPtr->pixelsPerRow;
	}
    }
    /* Free the memory allocated for filter weights. */
    Blt_Free(samples);
}

static void
ConvolvePictureHorizontally(
    Picture *destPtr, Picture *srcPtr, 
    Blt_PictureFilter filter)
{
    Sample *samples, *send;
    int y;
    Pix32 *srcRowPtr, *destRowPtr;
    int bytesPerSample;		/* Size of sample. */

    /* Pre-calculate filter contributions for each column. */
    bytesPerSample = ComputeWeights(destPtr->width, destPtr->width, 
	filter, &samples);
    send = (Sample *)((char *)samples + (destPtr->width * bytesPerSample));

    /* Apply filter to each column. */
    srcRowPtr = srcPtr->bits;
    destRowPtr = destPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *dp;
	Sample *samplePtr;

	dp = destRowPtr;
	for (samplePtr = samples; samplePtr < send; 
	     samplePtr = (Sample *)((char *)samplePtr + bytesPerSample)) {
	    Pix32 *sp;
	    int r, g, b, a;
	    PixelWeight *wp;

	    r = g = b = a = 0;
	    sp = srcRowPtr + samplePtr->start;
	    for (wp = samplePtr->weights; wp < samplePtr->wend; wp++) {
		a += wp->i32 * sp->Alpha;
		r += wp->i32 * sp->Red;
		g += wp->i32 * sp->Green;
		b += wp->i32 * sp->Blue;
		sp++;
	    }
	    dp->Red   = SICLAMP(r);
	    dp->Green = SICLAMP(g);
	    dp->Blue  = SICLAMP(b);
	    dp->Alpha = SICLAMP(a);
#ifdef notdef
	    if (dp->Alpha != 0xFF) {
		fprintf(stdout, "h1: alpha=0x%x\n", dp->Alpha);
	    }
#endif
	    dp++;
	}
	srcRowPtr += srcPtr->pixelsPerRow;
	destRowPtr += destPtr->pixelsPerRow;
    }
    /* Free the memory allocated for horizontal filter weights. */
    Blt_Free(samples);
}


/*
 *----------------------------------------------------------------------
 *
 * Blt_ConvolvePicture --
 *
 *      Resamples a given picture using 1-D filters and returns
 *	a new picture of the designated size.
 *
 * Results:
 *      Returns the resampled picture. The original picture
 *	is left intact.
 *
 *----------------------------------------------------------------------
 */
void
Blt_ConvolvePicture(
    Picture *destPtr,
    Picture *srcPtr,
    Blt_PictureFilter hFilter, 
    Blt_PictureFilter vFilter)
{
    if (srcPtr != destPtr) {
	if (!Blt_ReallocatePicture(destPtr, srcPtr->width, srcPtr->height, 0)) {
	    return;
	}
    }
#ifdef HAVE_X86_ASM
    if (bltUseMMX) {
	Blt_ZoomHorizontallyMMX(destPtr, srcPtr, hFilter);
	Blt_ZoomVerticallyMMX(destPtr, destPtr, vFilter);
    } else {
	ZoomHorizontally(destPtr, srcPtr, hFilter);
	ZoomVertically(destPtr, destPtr, vFilter);
    }
#else 
    ZoomHorizontally(destPtr, srcPtr, hFilter);
    ZoomVertically(destPtr, destPtr, vFilter);
#endif /* HAVE_X86_ASM */
    destPtr->flags = (srcPtr->flags | BLT_PICTURE_DIRTY);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_TilePicture --
 *
 *      Tiles the designated region in the destination picture with
 *      the source picture.  
 *	
 *	Please note that it is the responsibility of the caller to
 *	verify the region is valid (i.e. wholly contained by the
 *	destination picture).
 *
 * Results:
 *      The destination picture is tiled. The original picture
 *	is left intact.
 *
 *----------------------------------------------------------------------
 */
void
Blt_TilePicture(
    Picture *destPtr, 
    Picture *srcPtr, 
    int xOrigin, int yOrigin,	/* Tile origin. */
    int x, int y,		/* Area of destination to be tiled. */
    int width, int height)	
{
    int startX, startY;		/* Starting upper left corner of region. */
    int delta;

    /* Compute the starting x and y offsets of the tile from the
     * coordinates of the origin. */
    startX = x;
    if (x < xOrigin) {
	delta = (xOrigin - x) % srcPtr->width;
	if (delta > 0) {
	    startX -= (srcPtr->width - delta);
	}
    } else if (x > xOrigin) {
	delta = (x - xOrigin) % srcPtr->width;
	if (delta > 0) {
	    startX -= delta;
	}
    }
    startY = y;
    if (y < yOrigin) {
	delta = (yOrigin - y) % srcPtr->height;
	if (delta > 0) {
	    startY -= (srcPtr->height - delta);
	}
    } else if (y >= yOrigin) {
	delta = (y - yOrigin) % srcPtr->height;
	if (delta > 0) {
	    startY -= delta;
	}
    }

#ifdef notdef
    fprintf(stderr, "tile is (%d,%d,%d,%d)\n", 
		 xOrigin, yOrigin, srcPtr->width, srcPtr->height);
    fprintf(stderr, "region is (%d,%d,%d,%d)\n", x, y, width, height);
    fprintf(stderr, "starting at %d,%d\n", startX, startY);
#endif
    {
	int left, top, right, bottom;

	/* The region to be tiled. */
	left = x;
	right = x + width;
	top = y;
	bottom = y + height;

	for (y = startY; y < bottom; y += srcPtr->height) {
	    int sy, dy;
	    int tileHeight;
	    
	    sy = 0;
	    dy = y;
	    tileHeight = srcPtr->height;
	    if (y < top) {
		sy = (top - y);
		tileHeight = srcPtr->height - sy;
		dy = top;
	    } 
	    if ((dy + tileHeight) > bottom) {
		tileHeight = (bottom - dy);
	    }
	    for (x = startX; x < right; x += srcPtr->width) {
		int sx, dx;
		int tileWidth;
		
		sx = 0;
		dx = x;
		tileWidth = srcPtr->width;
		if (x < left) {
		    sx = (left - x);
		    tileWidth = srcPtr->width - sx;
		    dx = left;
		} 
		if ((dx + tileWidth) > right) {
		    tileWidth = (right - dx);
		}
#ifdef notdef
		fprintf(stder, "drawing pattern (%d,%d,%d,%d) at %d,%d\n",
			sx, sy, tileWidth, tileHeight, dx, dy);
#endif
		Blt_BlendPictureArea(destPtr, srcPtr, sx, sy, tileWidth, 
			tileHeight, dx, dy);
	    }
	}
    }
}

void
Blt_GradientPicture(
    Picture *destPtr,		/* (out) Picture to contain the new gradient. */
    Pix32 *maxPtr,		/* Color to represent 1.0 */
    Pix32 *minPtr,		/* Color to represent 0.0 */
    Blt_Gradient *gradientPtr)
{
    double rRange, gRange, bRange, aRange;

    /* Compute the ranges for each color component. */
    rRange = (double)(maxPtr->Red - minPtr->Red);
    gRange = (double)(maxPtr->Green - minPtr->Green);
    bRange = (double)(maxPtr->Blue - minPtr->Blue);
    aRange = (double)(maxPtr->Alpha - minPtr->Alpha);

    switch (gradientPtr->shape) {
    case BLT_GRADIENT_SHAPE_LINEAR:
	switch (gradientPtr->path) {
	case BLT_GRADIENT_PATH_X:
	    {
		Pix32 *copyRowPtr, *destRowPtr, *dp;
		int x, y;
		
		/* Draw the gradient on the first row, then copy the row
		 * to each subsequent row. */
		
		destRowPtr = destPtr->bits;
		for (dp = destRowPtr, x = 0; x < destPtr->width; x++, dp++) {
		    double t;
		    
		    t = (double)x / (double)(destPtr->width - 1);
		    if (gradientPtr->jitter) {
			t += JITTER(t);
			t = JCLAMP(t);
		    }
		    if (gradientPtr->logScale) {
			t = log10(9.0 * t + 1.0);
		    }
		    dp->Red = (unsigned char)(minPtr->Red + t * rRange);
		    dp->Green = (unsigned char)(minPtr->Green +  t * gRange);
		    dp->Blue = (unsigned char)(minPtr->Blue +  t * bRange);
		    dp->Alpha = (unsigned char)(minPtr->Alpha +  t * aRange);
		}
		destRowPtr += destPtr->pixelsPerRow;;
		copyRowPtr = destPtr->bits;
		for (y = 1; y < destPtr->height; y++) {
		    Pix32 *dp, *sp, *send;
		    
		    for (dp = destRowPtr, sp = copyRowPtr, 
			send = sp + destPtr->width; sp < send; dp++, sp++) {
			dp->color = sp->color;
		    }
		    copyRowPtr += destPtr->pixelsPerRow;
		    destRowPtr += destPtr->pixelsPerRow;
		}
	    }
	    break;

	case BLT_GRADIENT_PATH_Y:
	    {
		Pix32 *destRowPtr;
		int y;
		
		destRowPtr = destPtr->bits;
		for (y = 0; y < destPtr->height; y++) {
		    Pix32 *dp, *dend;
		    double t;
		    Pix32 color;
		    
		    /* Compute the color value for the row and then
		     * replicate it in every pixel in the row. */
		    
		    dp = destRowPtr;
		    t = (double)y / (double)(destPtr->height - 1);
		    if (gradientPtr->jitter) {
			t += JITTER(t);
			t = JCLAMP(t);
		    }
		    if (gradientPtr->logScale) {
			t = log10(9.0 * t + 1.0);
		    }
		    color.Red =   (unsigned char)(minPtr->Red + t * rRange);
		    color.Green = (unsigned char)(minPtr->Green +  t * gRange);
		    color.Blue =  (unsigned char)(minPtr->Blue +  t * bRange);
		    color.Alpha = (unsigned char)(minPtr->Alpha +  t * aRange);
		    for (dp = destRowPtr, dend = dp + destPtr->width; dp < dend;
			 dp++) {
			dp->color = color.color;
		    }
		    destRowPtr += destPtr->pixelsPerRow;
		}
	    }
	    break;

	case BLT_GRADIENT_PATH_YX:
	    {
		Pix32 *destRowPtr;
		int y;
		
		destRowPtr = destPtr->bits;
		for (y = 0; y < destPtr->height; y++) {
		    Pix32 *dp;
		    int x;
		    double ty;

		    ty = (double)y / (double)(destPtr->height - 1);
		    ty = 1.0 - ty;
		    for (dp = destRowPtr, x=0; x < destPtr->width; x++, dp++) {
			double t;
			double tx;
			
			tx = (double)x / (double)(destPtr->width - 1);
			t = (tx + ty) * 0.5;
			if (gradientPtr->jitter) {
			    t += JITTER(t);
			    t = JCLAMP(t);
			}
			if (gradientPtr->logScale) {
			    t = log10(9.0 * t + 1.0);
			}
			dp->Red =   (unsigned char)(minPtr->Red + t * rRange);
			dp->Green = (unsigned char)(minPtr->Green + t * gRange);
			dp->Blue =  (unsigned char)(minPtr->Blue + t * bRange);
			dp->Alpha = (unsigned char)(minPtr->Alpha + t * aRange);
		    }
		    destRowPtr += destPtr->pixelsPerRow;
		}
	    }
	    break;
	case BLT_GRADIENT_PATH_XY:
	    {
		Pix32 *destRowPtr;
		int y;
		
		destRowPtr = destPtr->bits;
		for (y = 0; y < destPtr->height; y++) {
		    Pix32 *dp;
		    int x;
		    double ty;

		    ty = (double)y / (double)(destPtr->height - 1);
		    for (dp = destRowPtr, x=0; x < destPtr->width; x++, dp++) {
			double t, tx;
			
			tx = (double)x / (double)(destPtr->width - 1);
			t = (tx + ty) * 0.5;
			if (gradientPtr->jitter) {
			    t += JITTER(t);
			    t = JCLAMP(t);
			}
			if (gradientPtr->logScale) {
			    t = log10(9.0 * t + 1.0);
			}
			dp->Red =   (unsigned char)(minPtr->Red + t * rRange);
			dp->Green = (unsigned char)(minPtr->Green + t * gRange);
			dp->Blue =  (unsigned char)(minPtr->Blue + t * bRange);
			dp->Alpha = (unsigned char)(minPtr->Alpha + t * aRange);
		    }
		    destRowPtr += destPtr->pixelsPerRow;
		}
	    }
	    break;
	}
	break;

    case BLT_GRADIENT_SHAPE_BILINEAR:
	switch (gradientPtr->path) {
	case BLT_GRADIENT_PATH_X:
	    {
		Pix32 *copyRowPtr, *destRowPtr, *dp;
		int x, y;
		
		/* Draw the gradient on the first row, then copy the row
		 * to each subsequent row. */
		
		destRowPtr = destPtr->bits;
		for (dp = destRowPtr, x = 0; x < destPtr->width; x++, dp++) {
		    double t;
		    
		    t = (double)x / (double)(destPtr->width - 1);
		    if (t > 0.5) {
			t = 1.0 - t;
		    }
		    t += t;
		    if (gradientPtr->jitter) {
			t += JITTER(t);
			t = JCLAMP(t);
		    }
		    if (gradientPtr->logScale) {
			t = log10(9.0 * t + 1.0);
		    }
		    dp->Red = (unsigned char)(minPtr->Red + t * rRange);
		    dp->Green = (unsigned char)(minPtr->Green +  t * gRange);
		    dp->Blue = (unsigned char)(minPtr->Blue +  t * bRange);
		    dp->Alpha = (unsigned char)(minPtr->Alpha +  t * aRange);
		}
		destRowPtr += destPtr->pixelsPerRow;;
		copyRowPtr = destPtr->bits;
		for (y = 1; y < destPtr->height; y++) {
		    Pix32 *dp, *sp, *send;
		    
		    for (dp = destRowPtr, sp = copyRowPtr, 
			send = sp + destPtr->width; sp < send; dp++, sp++) {
			dp->color = sp->color;
		    }
		    copyRowPtr += destPtr->pixelsPerRow;
		    destRowPtr += destPtr->pixelsPerRow;
		}
	    }
	    break;

	case BLT_GRADIENT_PATH_Y:
	    {
		Pix32 *destRowPtr;
		int y;
		
		destRowPtr = destPtr->bits;
		for (y = 0; y < destPtr->height; y++) {
		    Pix32 *dp, *dend;
		    double t;
		    Pix32 color;
		    
		    /* Compute the color value for the row and then
		     * replicate it in every pixel in the row. */
		    
		    dp = destRowPtr;
		    t = (double)y / (double)(destPtr->height - 1);
		    if (t > 0.5) {
			t = 1.0 - t;
		    }
		    t += t;
		    if (gradientPtr->jitter) {
			t += JITTER(t);
			t = JCLAMP(t);
		    }
		    if (gradientPtr->logScale) {
			t = log10(9.0 * t + 1.0);
		    }
		    color.Red =   (unsigned char)(minPtr->Red + t * rRange);
		    color.Green = (unsigned char)(minPtr->Green +  t * gRange);
		    color.Blue =  (unsigned char)(minPtr->Blue +  t * bRange);
		    color.Alpha = (unsigned char)(minPtr->Alpha +  t * aRange);
		    for (dp = destRowPtr, dend = dp + destPtr->width; dp < dend;
			 dp++) {
			dp->color = color.color;
		    }
		    destRowPtr += destPtr->pixelsPerRow;
		}
	    }
	    break;

	case BLT_GRADIENT_PATH_YX:
	    {
		Pix32 *destRowPtr;
		int y;
		
		destRowPtr = destPtr->bits;
		for (y = 0; y < destPtr->height; y++) {
		    Pix32 *dp;
		    int x;
		    double ty;

		    ty = (double)y / (double)(destPtr->height - 1);
		    for (dp = destRowPtr, x = 0; x < destPtr->width; x++, dp++) {
			double t, tx;

			tx = (double)x / (double)(destPtr->width - 1);
			t = 1.0 - ABS(tx - ty);
			if (gradientPtr->jitter) {
			    t += JITTER(t);
			    t = JCLAMP(t);
			}
			if (gradientPtr->logScale) {
			    t = log10(9.0 * t + 1.0);
			}
			dp->Red =   (unsigned char)(minPtr->Red + t * rRange);
			dp->Green = (unsigned char)(minPtr->Green + t * gRange);
			dp->Blue =  (unsigned char)(minPtr->Blue + t * bRange);
			dp->Alpha = (unsigned char)(minPtr->Alpha + t * aRange);
		    }
		    destRowPtr += destPtr->pixelsPerRow;
		}
	    }
	    break;

	case BLT_GRADIENT_PATH_XY:
	    {
		Pix32 *destRowPtr;
		int y;
		
		destRowPtr = destPtr->bits;
		for (y = 0; y < destPtr->height; y++) {
		    Pix32 *dp;
		    int x;
		    double ty;

		    ty = (double)y / (double)(destPtr->height - 1);
		    ty = 1.0 - ty;
		    for (dp = destRowPtr, x=0; x < destPtr->width; x++, dp++) {
			double t, tx;

			tx = (double)x / (double)(destPtr->width - 1);
			t = 1.0 - ABS(tx - ty);
			if (gradientPtr->jitter) {
			    t += JITTER(t);
			    t = JCLAMP(t);
			}
			if (gradientPtr->logScale) {
			    t = log10(9.0 * t + 1.0);
			}
			dp->Red =   (unsigned char)(minPtr->Red + t * rRange);
			dp->Green = (unsigned char)(minPtr->Green + t * gRange);
			dp->Blue =  (unsigned char)(minPtr->Blue + t * bRange);
			dp->Alpha = (unsigned char)(minPtr->Alpha + t * aRange);
		    }
		    destRowPtr += destPtr->pixelsPerRow;
		}
	    }
	    break;
	}
	break;

    case BLT_GRADIENT_SHAPE_RADIAL:
	{
	    Pix32 *destRowPtr;
	    int y;
	    int cx, cy;
	    int range;
	    destRowPtr = destPtr->bits;

	    /* Center coordinates. */
	    cx = destPtr->width / 2, cy = destPtr->height / 2;
	    range = cx * cx + cy * cy;
	    for (y = 0; y < destPtr->height; y++) {
		int x;
		double dy, dy2;
		Pix32 *dp;
		
		dy = ABS(cy - y);
		dy2 = dy * dy;
		dp = destRowPtr;
		for (x = 0; x < destPtr->width; x++) {
		    double dx;
		    double t;
		    
		    dx = ABS(cx - x);
		    t = (dx * dx + dy2) / range;
		    t = 1.0 - t;
		    if (gradientPtr->jitter) {
			t += JITTER(t);
			t = JCLAMP(t);
		    }
		    if (gradientPtr->logScale) {
			t = log10(9.0 * t + 1.0);
		    }
		    dp->Red = (unsigned char)(minPtr->Red + t * rRange);
		    dp->Green = (unsigned char)(minPtr->Green + t * gRange);
		    dp->Blue = (unsigned char)(minPtr->Blue + t * bRange);
		    dp->Alpha = (unsigned char)(t * aRange);
		    dp++;
		}
		destRowPtr += destPtr->pixelsPerRow;
	    }
	}
	break;
	
    case BLT_GRADIENT_SHAPE_RECTANGULAR:
	{
	    Pix32 *destRowPtr;
	    int y;
	    int midX, midY;
	    
	    midX = destPtr->width / 2, midY = destPtr->height / 2;
	    destRowPtr = destPtr->bits;
	    for (y = 0; y < destPtr->height; y++) {
		Pix32 *dp;
		int x;
		double ty;
		
		ty = (double)y / (double)(destPtr->height - 1);
		if (ty > 0.5) {
		    ty = 1.0 - ty;
		}
		dp = destRowPtr;
		for (x = 0; x < destPtr->width; x++) {
		    double t;
		    double tx;

		    tx = (double)x / (double)(destPtr->width - 1);
		    if (tx > 0.5) {
			tx = 1.0 - tx;
		    }
		    t = MIN(tx, ty);
		    t += t;
		    if (gradientPtr->jitter) {
			t += JITTER(t);
			t = JCLAMP(t);
		    }
		    if (gradientPtr->logScale) {
			t = log10(9.0 * t + 1.0);
		    }
		    dp->Red = (unsigned char)(minPtr->Red + t * rRange);
		    dp->Green = (unsigned char)(minPtr->Green + t * gRange);
		    dp->Blue = (unsigned char)(minPtr->Blue + t * bRange);
		    dp->Alpha = (unsigned char)(t * aRange);
		    dp++;
		}
		destRowPtr += destPtr->pixelsPerRow;
	    }
	}
	break;
    }
}


void
Blt_TexturePicture(
    Picture *destPtr,
    Pix32 *lowPtr, 
    Pix32 *highPtr, 
    int type)
{
#define TEXTURE_STRIPED 0
    switch (type) {
    case TEXTURE_STRIPED:
	{
	    Pix32 *destRowPtr;
	    int y;

	    destRowPtr = destPtr->bits;
	    for (y = 0; y < destPtr->height; y++) {
		Pix32 *dp, *dend;
		Pix32 color;

		color = ((y / 2) & 0x1) ? *lowPtr : *highPtr;
		if (0) {
		    double t;
		    t = 0.8;
		    t += JITTER(t);
		    t = JCLAMP(t);
		    color.Blue = (unsigned char)(color.Blue + t * 255.0);
		    color.Red = (unsigned char)(color.Red + t * 255.0);
		    color.Green = (unsigned char)(color.Green + t * 255.0);
		}
		for (dp = destRowPtr, dend = dp + destPtr->width; dp < dend; 
		     dp++) {
		    dp->color = color.color;
		}
		destRowPtr += destPtr->pixelsPerRow;
	    }
	}
	break;
    }
}

int
Blt_QueryColors(
    Picture *srcPtr, 		/* Picture to be queried. */
    Blt_HashTable *tablePtr)	/* (out) If non-NULL, hash table to be
				 * filled with the unique colors of
				 * the picture. */
{
    Blt_HashTable colorTable;
    Pix32 *srcRowPtr;
    int y;
    int nColors;
    unsigned int flags;

    flags = 0;
    if (tablePtr == NULL) {
	tablePtr = &colorTable;
    }
    Blt_InitHashTable(tablePtr, BLT_ONE_WORD_KEYS);
    srcRowPtr = srcPtr->bits;
    for (y = 0; y < srcPtr->height; y++) {
	Pix32 *sp, *send;
	
	for (sp = srcRowPtr, send = sp + srcPtr->width; sp < send; sp++) {
	    int isNew;
	    Pix32 color;
	    unsigned long key;

	    if ((sp->Red != sp->Green) || (sp->Green != sp->Blue)) {
		flags |= BLT_PICTURE_COLOR;
	    }
	    if (sp->Alpha != 0xFF) {
		if (sp->Alpha == 0x00) {
		    flags |= BLT_PICTURE_MASK;
		} else {
		    flags |= BLT_PICTURE_BLEND;
		}
	    }
	    color.color = sp->color;
	    color.Alpha = 0xFF;
	    key = (unsigned long)color.color;
	    Blt_CreateHashEntry(tablePtr, (char *)key, &isNew);
	}
	srcRowPtr += srcPtr->pixelsPerRow;
    }
    nColors = tablePtr->numEntries;
    if (tablePtr == &colorTable) {
	Blt_DeleteHashTable(&colorTable);
    }
    srcPtr->flags |= flags;
    return nColors;
}
