
/*
 * bltGrBar.c --
 *
 * This module implements barchart elements for the BLT graph widget.
 *
 *	Copyright 1993-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltGraph.h"
#include <X11/Xutil.h>

#include "bltGrElem.h"

typedef struct {
    Pen base;

    XColor *fgColor;		/* Foreground color of bar */
    Tk_3DBorder border;		/* 3D border and background color */
    int borderWidth;		/* 3D border width of bar */
    int relief;			/* Relief of the bar */
    Pixmap stipple;		/* Stipple */
    GC gc;			/* Graphics context */

    /* Error bar attributes. */
    int errorBarShow;		/* Describes which error bars to
				 * display: none, x, y, or * both. */

    int errorBarLineWidth;	/* Width of the error bar segments. */

    int errorBarCapWidth;
    XColor *errorBarColor;	/* Color of the error bar. */

    GC errorBarGC;		/* Error bar graphics context. */

    /* Show value attributes. */
    int valueShow;		/* Indicates whether to display data value.
				 * Values are x, y, or none. */

    char *valueFormat;		/* A printf format string. */
    TextStyle valueStyle;	/* Text attributes (color, font,
				 * rotation, etc.) of the value. */
    
} BarPen;

typedef struct {
    PenStyle base;

    XRectangle *bars;		/* Indicates starting location in bar
				 * array for this pen. */
    int nBars;			/* Number of bar segments for this pen. */

} BarPenStyle;

typedef struct {
    Element base;

    double barWidth;

    int *barToData;
    XRectangle *bars;	        /* Array of rectangles comprising the bar
				 * segments of the element. */
    int *activeToData;
    XRectangle *activeRects;

    int nBars;			/* # of visible bar segments for element */
    int nActive;

    int padX;			/* Spacing on either side of bar */

    BarPen builtinPen;
} Bar;

extern Blt_CustomOption bltBarPenOption;
extern Blt_CustomOption bltDataOption;
extern Blt_CustomOption bltDataPairsOption;
extern Blt_CustomOption bltXAxisOption;
extern Blt_CustomOption bltYAxisOption;
extern Blt_CustomOption bltColorOption;
extern Blt_CustomOption bltBarStylesOption;

static Blt_OptionParseProc ObjToBarMode;
static Blt_OptionPrintProc BarModeToObj;
Blt_CustomOption bltBarModeOption =
{
    ObjToBarMode, BarModeToObj, NULL, (ClientData)0
};

#define DEF_BAR_ACTIVE_PEN		"activeBar"
#define DEF_BAR_AXIS_X			"x"
#define DEF_BAR_AXIS_Y			"y"
#define DEF_BAR_BACKGROUND		"navyblue"
#define DEF_BAR_BORDERWIDTH		"2"
#define DEF_BAR_ERRORBAR_COLOR		"defcolor"
#define DEF_BAR_ERRORBAR_LINE_WIDTH	"1"
#define DEF_BAR_ERRORBAR_CAP_WIDTH	"1"
#define DEF_BAR_FOREGROUND		"blue"
#define DEF_BAR_HIDE			"no"
#define DEF_BAR_LABEL_RELIEF		"flat"
#define DEF_BAR_NORMAL_STIPPLE		""
#define DEF_BAR_RELIEF			"raised"
#define DEF_BAR_SHOW_ERRORBARS		"both"
#define DEF_BAR_STATE			"normal"
#define DEF_BAR_STYLES			""
#define DEF_BAR_TAGS			"all"
#define DEF_BAR_WIDTH			"0.0"

#define DEF_PEN_ACTIVE_BACKGROUND	"red"
#define DEF_PEN_ACTIVE_FOREGROUND     	"pink"
#define DEF_PEN_BORDERWIDTH		"2"
#define DEF_PEN_NORMAL_BACKGROUND	"navyblue"
#define DEF_PEN_NORMAL_FOREGROUND	"blue"
#define DEF_PEN_RELIEF			"raised"
#define DEF_PEN_STIPPLE			""
#define DEF_PEN_TYPE			"bar"
#define	DEF_PEN_VALUE_ANCHOR		"s"
#define	DEF_PEN_VALUE_COLOR		RGB_BLACK
#define	DEF_PEN_VALUE_FONT		STD_FONT_SMALL
#define	DEF_PEN_VALUE_FORMAT		"%g"
#define DEF_PEN_SHOW_VALUES		"no"

static Blt_ConfigSpec barPenConfigSpecs[] =
{
    {BLT_CONFIG_BORDER, "-background", "background", "Background",
	DEF_PEN_ACTIVE_BACKGROUND, Blt_Offset(BarPen, border),
	BLT_CONFIG_NULL_OK | ACTIVE_PEN},
    {BLT_CONFIG_BORDER, "-background", "background", "Background",
	DEF_PEN_NORMAL_BACKGROUND, Blt_Offset(BarPen, border),
	BLT_CONFIG_NULL_OK | NORMAL_PEN},
    {BLT_CONFIG_SYNONYM, "-bd", "borderWidth", (char *)NULL,
	(char *)NULL, 0, ALL_PENS},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL,
	(char *)NULL, 0, ALL_PENS},
    {BLT_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_PEN_BORDERWIDTH, Blt_Offset(BarPen, borderWidth), ALL_PENS},
    {BLT_CONFIG_CUSTOM, "-errorbarcolor", "errorBarColor", "ErrorBarColor",
	DEF_BAR_ERRORBAR_COLOR, Blt_Offset(BarPen, errorBarColor), ALL_PENS, 
	&bltColorOption},
    {BLT_CONFIG_PIXELS, "-errorbarwidth", "errorBarWidth", "ErrorBarWidth",
	DEF_BAR_ERRORBAR_LINE_WIDTH, Blt_Offset(BarPen, errorBarLineWidth),
        ALL_PENS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS, "-errorbarcap", "errorBarCap", "ErrorBarCap", 
	DEF_BAR_ERRORBAR_CAP_WIDTH, Blt_Offset(BarPen, errorBarCapWidth),
        ALL_PENS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_SYNONYM, "-fg", "foreground", (char *)NULL,
	(char *)NULL, 0, ALL_PENS},
    {BLT_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_PEN_ACTIVE_FOREGROUND, Blt_Offset(BarPen, fgColor),
	ACTIVE_PEN | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_PEN_NORMAL_FOREGROUND, Blt_Offset(BarPen, fgColor),
	NORMAL_PEN |  BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_PEN_RELIEF, Blt_Offset(BarPen, relief), ALL_PENS},
    {BLT_CONFIG_FILL, "-showerrorbars", "showErrorBars", "ShowErrorBars",
	DEF_BAR_SHOW_ERRORBARS, Blt_Offset(BarPen, errorBarShow),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_FILL, "-showvalues", "showValues", "ShowValues",
	DEF_PEN_SHOW_VALUES, Blt_Offset(BarPen, valueShow),
        ALL_PENS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BITMAP, "-stipple", "stipple", "Stipple", DEF_PEN_STIPPLE, 
	Blt_Offset(BarPen, stipple), ALL_PENS | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_STRING, "-type", (char *)NULL, (char *)NULL, DEF_PEN_TYPE, 
	Blt_Offset(BarPen, base.typeId), ALL_PENS | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_ANCHOR, "-valueanchor", "valueAnchor", "ValueAnchor",
	DEF_PEN_VALUE_ANCHOR, Blt_Offset(BarPen, valueStyle.anchor), 
	ALL_PENS},
    {BLT_CONFIG_COLOR, "-valuecolor", "valueColor", "ValueColor",
	DEF_PEN_VALUE_COLOR, Blt_Offset(BarPen, valueStyle.color), 
	ALL_PENS},
    {BLT_CONFIG_FONT, "-valuefont", "valueFont", "ValueFont",
	DEF_PEN_VALUE_FONT, Blt_Offset(BarPen, valueStyle.font), 
	ALL_PENS},
    {BLT_CONFIG_STRING, "-valueformat", "valueFormat", "ValueFormat",
	DEF_PEN_VALUE_FORMAT, Blt_Offset(BarPen, valueFormat),
	ALL_PENS | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_DOUBLE, "-valuerotate", "valueRotate", "ValueRotate",
	(char *)NULL, Blt_Offset(BarPen, valueStyle.angle), ALL_PENS},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};


static Blt_ConfigSpec barElemConfigSpecs[] =
{
    {BLT_CONFIG_CUSTOM, "-activepen", "activePen", "ActivePen",
	DEF_BAR_ACTIVE_PEN, Blt_Offset(Bar, base.activePenPtr), 
	BLT_CONFIG_NULL_OK, &bltBarPenOption},
    {BLT_CONFIG_BORDER, "-background", "background", "Background",
	DEF_BAR_BACKGROUND, Blt_Offset(Bar, builtinPen.border),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_DOUBLE, "-barwidth", "barWidth", "BarWidth",
	DEF_BAR_WIDTH, Blt_Offset(Bar, barWidth),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_SYNONYM, "-bd", "borderWidth", (char *)NULL,
	(char *)NULL, 0, 0},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL,
	(char *)NULL, 0, 0},
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags",
	DEF_BAR_TAGS, Blt_Offset(Bar, base.object.tags), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_BAR_BORDERWIDTH, Blt_Offset(Bar, builtinPen.borderWidth), 0},
    {BLT_CONFIG_CUSTOM, "-errorbarcolor", "errorBarColor", "ErrorBarColor",
	DEF_BAR_ERRORBAR_COLOR, Blt_Offset(Bar, builtinPen.errorBarColor), 0, 
	&bltColorOption},
    {BLT_CONFIG_PIXELS, "-errorbarwidth", "errorBarWidth", "ErrorBarWidth",
	DEF_BAR_ERRORBAR_LINE_WIDTH, 
	Blt_Offset(Bar, builtinPen.errorBarLineWidth), 
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_PIXELS, "-errorbarcap", "errorBarCap", "ErrorBarCap", 
	DEF_BAR_ERRORBAR_CAP_WIDTH, 
	Blt_Offset(Bar, builtinPen.errorBarCapWidth),
        BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_SYNONYM, "-fg", "foreground", (char *)NULL, (char *)NULL, 0, 0},
    {BLT_CONFIG_CUSTOM, "-data", "data", "Data", (char *)NULL, 0, 0, 
	&bltDataPairsOption},
    {BLT_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_BAR_FOREGROUND, Blt_Offset(Bar, builtinPen.fgColor), 
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_STRING, "-label", "label", "Label", (char *)NULL, 
	Blt_Offset(Bar, base.label), BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_RELIEF, "-labelrelief", "labelRelief", "LabelRelief",
	DEF_BAR_LABEL_RELIEF, Blt_Offset(Bar, base.labelRelief),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide", DEF_BAR_HIDE, 
	Blt_Offset(Bar, base.object.hidden), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-mapx", "mapX", "MapX", DEF_BAR_AXIS_X, 
	Blt_Offset(Bar, base.axes.x), 0, &bltXAxisOption},
    {BLT_CONFIG_CUSTOM, "-mapy", "mapY", "MapY", DEF_BAR_AXIS_Y, 
	Blt_Offset(Bar, base.axes.y), 0, &bltYAxisOption},
    {BLT_CONFIG_CUSTOM, "-pen", "pen", "Pen", (char *)NULL, 
	Blt_Offset(Bar, base.normalPenPtr), BLT_CONFIG_NULL_OK, 
	&bltBarPenOption},
    {BLT_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_BAR_RELIEF, Blt_Offset(Bar, builtinPen.relief), 0},
    {BLT_CONFIG_FILL, "-showerrorbars", "showErrorBars", "ShowErrorBars",
	DEF_BAR_SHOW_ERRORBARS, Blt_Offset(Bar, builtinPen.errorBarShow),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_FILL, "-showvalues", "showValues", "ShowValues",
	DEF_PEN_SHOW_VALUES, Blt_Offset(Bar, builtinPen.valueShow),
	BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STATE, "-state", "state", "State", DEF_BAR_STATE, 
	Blt_Offset(Bar, base.state), BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BITMAP, "-stipple", "stipple", "Stipple",
	DEF_BAR_NORMAL_STIPPLE, Blt_Offset(Bar, builtinPen.stipple),
	BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_CUSTOM, "-styles", "styles", "Styles", DEF_BAR_STYLES, 
	Blt_Offset(Bar, base.stylePalette), 0, &bltBarStylesOption},
    {BLT_CONFIG_ANCHOR, "-valueanchor", "valueAnchor", "ValueAnchor",
	DEF_PEN_VALUE_ANCHOR, Blt_Offset(Bar, builtinPen.valueStyle.anchor), 0},
    {BLT_CONFIG_COLOR, "-valuecolor", "valueColor", "ValueColor",
	DEF_PEN_VALUE_COLOR, Blt_Offset(Bar, builtinPen.valueStyle.color), 0},
    {BLT_CONFIG_FONT, "-valuefont", "valueFont", "ValueFont",
	DEF_PEN_VALUE_FONT, Blt_Offset(Bar, builtinPen.valueStyle.font), 0},
    {BLT_CONFIG_STRING, "-valueformat", "valueFormat", "ValueFormat",
	DEF_PEN_VALUE_FORMAT, Blt_Offset(Bar, builtinPen.valueFormat),
        BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_DOUBLE, "-valuerotate", "valueRotate", "ValueRotate",
	(char *)NULL, Blt_Offset(Bar, builtinPen.valueStyle.angle), 0},
    {BLT_CONFIG_CUSTOM, "-weights", "weights", "Weights", (char *)NULL, 
	Blt_Offset(Bar, base.w), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-x", "xdata", "Xdata", (char *)NULL, 
	Blt_Offset(Bar, base.x), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-y", "ydata", "Ydata", (char *)NULL, 
	Blt_Offset(Bar, base.y), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xdata", "xdata", "Xdata", (char *)NULL, 
	Blt_Offset(Bar, base.x), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-ydata", "ydata", "Ydata", (char *)NULL, 
	Blt_Offset(Bar, base.y), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xerror", "xError", "XError", (char *)NULL, 
	Blt_Offset(Bar, base.xError), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xhigh", "xHigh", "XHigh", (char *)NULL, 
	Blt_Offset(Bar, base.xHigh), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-xlow", "xLow", "XLow", (char *)NULL, 
	Blt_Offset(Bar, base.xLow), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-yerror", "yError", "YError", (char *)NULL, 
	Blt_Offset(Bar, base.yError), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-yhigh", "yHigh", "YHigh", (char *)NULL, 
	Blt_Offset(Bar, base.yHigh), 0, &bltDataOption},
    {BLT_CONFIG_CUSTOM, "-ylow", "yLow", "YLow", (char *)NULL, 
	Blt_Offset(Bar, base.yLow), 0, &bltDataOption},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/* Forward declarations */
static PenConfigureProc ConfigureBarPen;
static PenDestroyProc DestroyBarPen;
static ElementClosestProc ClosestBar;
static ElementConfigProc ConfigureBar;
static ElementDestroyProc DestroyBar;
static ElementDrawProc DrawActiveBar;
static ElementDrawProc DrawNormalBar;
static ElementDrawSymbolProc DrawSymbol;
static ElementExtentsProc GetBarExtents;
static ElementToPostScriptProc ActiveBarToPostScript;
static ElementToPostScriptProc NormalBarToPostScript;
static ElementSymbolToPostScriptProc SymbolToPostScript;
static ElementMapProc MapBar;

INLINE static int
Round(double x)
{
    return (int) (x + ((x < 0.0) ? -0.5 : 0.5));
}

/*
 * ----------------------------------------------------------------------
 * Custom option parse and print procedures
 * ----------------------------------------------------------------------
 */

/*
 * ----------------------------------------------------------------------
 *
 * NameOfBarMode --
 *
 *	Converts the integer representing the mode style into a string.
 *
 * ----------------------------------------------------------------------
 */
static char *
NameOfBarMode(BarMode mode)
{
    switch (mode) {
    case MODE_INFRONT:
	return "infront";
    case MODE_OVERLAP:
	return "overlap";
    case MODE_STACKED:
	return "stacked";
    case MODE_ALIGNED:
	return "aligned";
    default:
	return "unknown mode value";
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ObjToMode --
 *
 *	Converts the mode string into its numeric representation.
 *
 *	Valid mode strings are:
 *
 *      "infront"   Draw a full bar at each point in the element.
 *
 * 	"stacked"   Stack bar segments vertically. Each stack is defined
 *		    by each ordinate at a particular abscissa. The height
 *		    of each segment is represented by the sum the previous
 *		    ordinates.
 *
 *	"aligned"   Align bar segments as smaller slices one next to
 *		    the other.  Like "stacks", aligned segments are
 *		    defined by each ordinate at a particular abscissa.
 *
 * Results:
 *	A standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToBarMode(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* Mode style string */
    char *widgRec,		/* Cubicle structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    BarMode *modePtr = (BarMode *)(widgRec + offset);
    size_t length;
    char c;
    char *string;
    
    string = Tcl_GetString(objPtr);
    c = string[0];
    length = strlen(string);
    if ((c == 'n') && (strncmp(string, "normal", length) == 0)) {
	*modePtr = MODE_INFRONT;
    } else if ((c == 'i') && (strncmp(string, "infront", length) == 0)) {
	*modePtr = MODE_INFRONT;
    } else if ((c == 's') && (strncmp(string, "stacked", length) == 0)) {
	*modePtr = MODE_STACKED;
    } else if ((c == 'a') && (strncmp(string, "aligned", length) == 0)) {
	*modePtr = MODE_ALIGNED;
    } else if ((c == 'o') && (strncmp(string, "overlap", length) == 0)) {
	*modePtr = MODE_OVERLAP;
    } else {
	Tcl_AppendResult(interp, "bad mode argument \"", string,
	    "\": should be \"infront\", \"stacked\", \"overlap\", or \"aligned\"",
	    (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * BarModeToObj --
 *
 *	Returns the mode style string based upon the mode flags.
 *
 * Results:
 *	The mode style string is returned.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
BarModeToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Row/column structure record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    BarMode mode = *(BarMode *)(widgRec + offset);

    return Tcl_NewStringObj(NameOfBarMode(mode), -1);
}


/* 
 * Zero out the style's number of bars and errorbars. 
 */
static void
ResetStylePalette(Blt_Chain *stylePalette)
{
    Blt_ChainLink *lp;

    for (lp = Blt_ChainFirstLink(stylePalette); lp != NULL; 
	 lp = Blt_ChainNextLink(lp)) {
	BarPenStyle *stylePtr;

	stylePtr = Blt_ChainGetValue(lp);
	stylePtr->base.xErrorBarCnt = stylePtr->base.yErrorBarCnt = 0;
	stylePtr->nBars = 0;
    }
}

static int
ConfigureBarPen(Graph *graphPtr, Pen *penPtr)
{
    BarPen *bpPtr = (BarPen *)penPtr;
    XGCValues gcValues;
    unsigned long gcMask;
    int fillStyle;
    GC newGC;
    long defColor;

    gcMask = GCForeground;
    if (bpPtr->fgColor != NULL) {
	defColor = bpPtr->fgColor->pixel;
	gcValues.foreground = bpPtr->fgColor->pixel;
    } else if (bpPtr->border != NULL) {
	defColor = Tk_3DBorderColor(bpPtr->border)->pixel;
	gcValues.foreground = Tk_3DBorderColor(bpPtr->border)->pixel;
    } else {
	defColor = BlackPixel(graphPtr->display, 
			      Tk_ScreenNumber(graphPtr->tkwin));
    }
    if ((bpPtr->fgColor != NULL) && (bpPtr->border != NULL)) {
	gcMask |= GCBackground;
	gcValues.background = Tk_3DBorderColor(bpPtr->border)->pixel;
	fillStyle = FillOpaqueStippled;
    } else {
	fillStyle = FillStippled;
    }
    if (bpPtr->stipple != None) {
	gcValues.stipple = bpPtr->stipple;
	gcValues.fill_style = fillStyle;
	gcMask |= (GCStipple | GCFillStyle);
    }
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (bpPtr->gc != NULL) {
	Tk_FreeGC(graphPtr->display, bpPtr->gc);
    }
    bpPtr->gc = newGC;

    gcMask = GCForeground | GCLineWidth;
    if (bpPtr->errorBarColor == COLOR_DEFAULT) {
	gcValues.foreground = defColor;
    } else {
	gcValues.foreground = bpPtr->errorBarColor->pixel;
    }
    gcValues.line_width = LineWidth(bpPtr->errorBarLineWidth);
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (bpPtr->errorBarGC != NULL) {
	Tk_FreeGC(graphPtr->display, bpPtr->errorBarGC);
    }
    bpPtr->errorBarGC = newGC;

    return TCL_OK;
}

static void
DestroyBarPen(Graph *graphPtr, Pen *penPtr)
{
    BarPen *bpPtr = (BarPen *)penPtr;

    Blt_TextFreeStyle(graphPtr->display, &bpPtr->valueStyle);
    if (bpPtr->gc != NULL) {
	Tk_FreeGC(graphPtr->display, bpPtr->gc);
    }
    if (bpPtr->errorBarGC != NULL) {
	Tk_FreeGC(graphPtr->display, bpPtr->errorBarGC);
    }
}

static void
InitializeBarPen(BarPen *penPtr)
{
    /* Generic fields common to all pen types. */
    penPtr->base.configProc = ConfigureBarPen;
    penPtr->base.destroyProc = DestroyBarPen;
    penPtr->base.flags = NORMAL_PEN;
    penPtr->base.configSpecs = barPenConfigSpecs;

    /* Initialize fields specific to bar pens. */
    Blt_TextInitStyle(penPtr->valueStyle);
    penPtr->relief = TK_RELIEF_RAISED;
    penPtr->valueShow = SHOW_NONE;
    penPtr->borderWidth = 2;
    penPtr->errorBarShow = SHOW_BOTH;
}

Pen *
Blt_BarPen(char *penName)
{
    BarPen *penPtr;

    penPtr = Blt_Calloc(1, sizeof(BarPen));
    assert(penPtr);
    InitializeBarPen(penPtr);
    penPtr->base.name = Blt_Strdup(penName);
    if (strcmp(penName, "activeBar") == 0) {
	penPtr->base.flags = ACTIVE_PEN;
    }
    return (Pen *) penPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * CheckStacks --
 *
 *	Check that the data limits are not superseded by the heights
 *	of stacked bar segments.  The heights are calculated by
 *	Blt_ComputeStacks.
 *
 * Results:
 *	If the y-axis limits need to be adjusted for stacked segments,
 *	*minPtr* or *maxPtr* are updated.
 *
 * Side effects:
 *	Autoscaling of the y-axis is affected.
 *
 * ----------------------------------------------------------------------
 */
static void
CheckStacks(
    Graph *graphPtr,
    Axis2D *pairPtr,
    double *minPtr, 
    double *maxPtr)	/* Current minimum maximum for y-axis */
{
    FreqInfo *infoPtr;
    int i;

    if ((graphPtr->mode != MODE_STACKED) || (graphPtr->nStacks == 0)) {
	return;
    }
    infoPtr = graphPtr->freqArr;
    for (i = 0; i < graphPtr->nStacks; i++) {
	if ((infoPtr->axes.x == pairPtr->x) && 
	    (infoPtr->axes.y == pairPtr->y)) {
	    /*

	     * Check if any of the y-values (because of stacking) are
	     * greater than the current limits of the graph.
	     */
	    if (infoPtr->sum < 0.0) {
		if (*minPtr > infoPtr->sum) {
		    *minPtr = infoPtr->sum;
		}
	    } else {
		if (*maxPtr < infoPtr->sum) {
		    *maxPtr = infoPtr->sum;
		}
	    }
	}
	infoPtr++;
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureBar --
 *
 *	Sets up the appropriate configuration parameters in the GC.
 *      It is assumed the parameters have been previously set by
 *	a call to Blt_ConfigureWidget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information such as bar foreground/background
 *	color and stipple etc. get set in a new GC.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigureBar(Graph *graphPtr, Element *elemPtr)
{
    Blt_ChainLink *lp;
    PenStyle *stylePtr;

    if (ConfigureBarPen(graphPtr, elemPtr->builtinPenPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * Point to the static normal pen if no external pens have
     * been selected.
     */
    lp = Blt_ChainFirstLink(elemPtr->stylePalette);
    if (lp == NULL) {
	lp = Blt_ChainAllocLink(sizeof(BarPenStyle));
	Blt_ChainLinkBefore(elemPtr->stylePalette, lp, NULL);
    }
    stylePtr = Blt_ChainGetValue(lp);
    stylePtr->penPtr = NORMALPEN(elemPtr);

    if (Blt_ConfigModified(elemPtr->configSpecs, "-barwidth", "-*data",
	    "-map*", "-label", "-hide", "-x", "-y", (char *)NULL)) {
	elemPtr->flags |= MAP_ITEM;
    }
    return TCL_OK;
}

static void
GetBarExtents(Element *elemPtr, Region2D *extsPtr)
{
    Graph *graphPtr = elemPtr->object.graphPtr;
    Bar *barPtr = (Bar *)elemPtr;
    double middle, barWidth;
    int nPoints;

    extsPtr->top = extsPtr->left = DBL_MAX;
    extsPtr->bottom = extsPtr->right = -DBL_MAX;

    nPoints = NUMBEROFPOINTS(elemPtr);
    if (nPoints < 1) {
	return;			/* No data points */
    }
    barWidth = graphPtr->barWidth;
    if (barPtr->barWidth > 0.0) {
	barWidth = barPtr->barWidth;
    }
    middle = barWidth * 0.5;
    extsPtr->left = elemPtr->x.min - middle;
    extsPtr->right = elemPtr->x.max + middle;

    extsPtr->top = elemPtr->y.min;
    extsPtr->bottom = elemPtr->y.max;
    if (extsPtr->bottom < graphPtr->baseline) {
	extsPtr->bottom = graphPtr->baseline;
    }
    /*
     * Handle "stacked" bar elements specially.
     *
     * If element is stacked, the sum of its ordinates may be outside
     * the minimum/maximum limits of the element's data points.
     */
    if ((graphPtr->mode == MODE_STACKED) && (graphPtr->nStacks > 0)) {
	CheckStacks(graphPtr, &elemPtr->axes, &extsPtr->top, &extsPtr->bottom);
    }
    /* Warning: You get what you deserve if the x-axis is logScale */
    if (elemPtr->axes.x->logScale) {
	extsPtr->left = Blt_FindElemVectorMinimum(&elemPtr->x, DBL_MIN) + 
	    middle;
    }
    /* Fix y-min limits for barchart */
    if (elemPtr->axes.y->logScale) {
 	if ((extsPtr->top <= 0.0) || (extsPtr->top > 1.0)) {
	    extsPtr->top = 1.0;
	}
    } else {
	if (extsPtr->top > 0.0) {
	    extsPtr->top = 0.0;
	}
    }
    /* Correct the extents for error bars if they exist. */
    if (elemPtr->xError.nValues > 0) {
	int i;
	
	/* Correct the data limits for error bars */
	nPoints = MIN(elemPtr->xError.nValues, nPoints);
	for (i = 0; i < nPoints; i++) {
	    double x;

	    x = elemPtr->x.valueArr[i] + elemPtr->xError.valueArr[i];
	    if (x > extsPtr->right) {
		extsPtr->right = x;
	    }
	    x = elemPtr->x.valueArr[i] - elemPtr->xError.valueArr[i];
	    if (elemPtr->axes.x->logScale) {
		if (x < 0.0) {
		    x = -x;	/* Mirror negative values, instead
				 * of ignoring them. */
		}
		if ((x > DBL_MIN) && (x < extsPtr->left)) {
		    extsPtr->left = x;
		}
	    } else if (x < extsPtr->left) {
		extsPtr->left = x;
	    }
	}		     
    } else {
	if ((elemPtr->xHigh.nValues > 0) && 
	    (elemPtr->xHigh.max > extsPtr->right)) {
	    extsPtr->right = elemPtr->xHigh.max;
	}
	if (elemPtr->xLow.nValues > 0) {
	    double left;
	    
	    if ((elemPtr->xLow.min <= 0.0) && 
		(elemPtr->axes.x->logScale)) {
		left = Blt_FindElemVectorMinimum(&elemPtr->xLow, DBL_MIN);
	    } else {
		left = elemPtr->xLow.min;
	    }
	    if (left < extsPtr->left) {
		extsPtr->left = left;
	    }
	}
    }
    if (elemPtr->yError.nValues > 0) {
	int i;
	
	nPoints = MIN(elemPtr->yError.nValues, nPoints);
	for (i = 0; i < nPoints; i++) {
	    double y;

	    y = elemPtr->y.valueArr[i] + elemPtr->yError.valueArr[i];
	    if (y > extsPtr->bottom) {
		extsPtr->bottom = y;
	    }
	    y = elemPtr->y.valueArr[i] - elemPtr->yError.valueArr[i];
	    if (elemPtr->axes.y->logScale) {
		if (y < 0.0) {
		    y = -y;	/* Mirror negative values, instead
				 * of ignoring them. */
		}
		if ((y > DBL_MIN) && (y < extsPtr->left)) {
		    extsPtr->top = y;
		}
	    } else if (y < extsPtr->top) {
		extsPtr->top = y;
	    }
	}		     
    } else {
	if ((elemPtr->yHigh.nValues > 0) && 
	    (elemPtr->yHigh.max > extsPtr->bottom)) {
	    extsPtr->bottom = elemPtr->yHigh.max;
	}
	if (elemPtr->yLow.nValues > 0) {
	    double top;
	    
	    if ((elemPtr->yLow.min <= 0.0) && 
		(elemPtr->axes.y->logScale)) {
		top = Blt_FindElemVectorMinimum(&elemPtr->yLow, DBL_MIN);
	    } else {
		top = elemPtr->yLow.min;
	    }
	    if (top < extsPtr->top) {
		extsPtr->top = top;
	    }
	}
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ClosestBar --
 *
 *	Find the bar segment closest to the window coordinates	point
 *	specified.
 *
 *	Note:  This does not return the height of the stacked segment
 *	       (in graph coordinates) properly.
 *
 * Results:
 *	Returns 1 if the point is width any bar segment, otherwise 0.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
ClosestBar(
    Graph *graphPtr,		/* Graph widget record */
    Element *elemPtr,		/* Bar element */
    ClosestSearch *searchPtr)	/* Info of closest point in element */
{
    Bar *barPtr = (Bar *)elemPtr;
    XRectangle *rp;
    double minDist;
    int imin;
    int i;

    minDist = searchPtr->dist;
    imin = 0;
    
    for (rp = barPtr->bars, i = 0; i < barPtr->nBars; i++, rp++) {
	Point2D *pp, *pend;
	Point2D outline[5];
	double left, right, top, bottom;

	if (PointInRectangle(rp, searchPtr->x, searchPtr->y)) {
	    imin = barPtr->barToData[i];
	    minDist = 0.0;
	    break;
	}
	left = rp->x, top = rp->y;
	right = (double)(rp->x + rp->width);
	bottom = (double)(rp->y + rp->height);
	outline[4].x = outline[3].x = outline[0].x = left;
	outline[4].y = outline[1].y = outline[0].y = top;
	outline[2].x = outline[1].x = right;
	outline[3].y = outline[2].y = bottom;

	for (pp = outline, pend = outline + 4; pp < pend; pp++) {
	    Point2D t;
	    double dist;

	    t = Blt_GetProjection(searchPtr->x, searchPtr->y, pp, pp + 1);
	    if (t.x > right) {
		t.x = right;
	    } else if (t.x < left) {
		t.x = left;
	    }
	    if (t.y > bottom) {
		t.y = bottom;
	    } else if (t.y < top) {
		t.y = top;
	    }
	    dist = hypot((t.x - searchPtr->x), (t.y - searchPtr->y));
	    if (dist < minDist) {
		minDist = dist;
		imin = barPtr->barToData[i];
	    }
	}
    }
    if (minDist < searchPtr->dist) {
	searchPtr->elemPtr = (Element *)elemPtr;
	searchPtr->dist = minDist;
	searchPtr->index = imin;
	searchPtr->point.x = (double)elemPtr->x.valueArr[imin];
	searchPtr->point.y = (double)elemPtr->y.valueArr[imin];
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MergePens --
 *
 *	Reorders the both arrays of points and errorbars to merge pens.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The old arrays are freed and new ones allocated containing
 *	the reordered points and errorbars.
 *
 *----------------------------------------------------------------------
 */
static void
MergePens(Element *elemPtr, PenStyle **dataToStyle)
{
    Bar *barPtr = (Bar *)elemPtr;

    if (Blt_ChainGetLength(elemPtr->stylePalette) < 2) {
	Blt_ChainLink *lp;
	BarPenStyle *stylePtr;

	lp = Blt_ChainFirstLink(elemPtr->stylePalette);

	stylePtr = Blt_ChainGetValue(lp);
	stylePtr->nBars = barPtr->nBars;
	stylePtr->bars = barPtr->bars;
	stylePtr->base.symbolSize = barPtr->bars->width / 2;
	stylePtr->base.xErrorBarCnt = elemPtr->xErrorBarCnt;
	stylePtr->base.xErrorBars = elemPtr->xErrorBars;
	stylePtr->base.yErrorBarCnt = elemPtr->yErrorBarCnt;
	stylePtr->base.yErrorBars = elemPtr->yErrorBars;
	return;
    }
    /* We have more than one style. Group bar segments of like pen
     * styles together.  */

    if (barPtr->nBars > 0) {
	Blt_ChainLink *lp;
	XRectangle *bars, *rp;
	int *ip, *barToData;

	bars = Blt_Malloc(barPtr->nBars * sizeof(XRectangle));
	barToData = Blt_Malloc(barPtr->nBars * sizeof(int));
	assert(bars && barToData);

	rp = bars, ip = barToData;
	for (lp = Blt_ChainFirstLink(elemPtr->stylePalette); lp != NULL; 
	     lp = Blt_ChainNextLink(lp)) {
	    BarPenStyle *stylePtr;
	    int i;

	    stylePtr = Blt_ChainGetValue(lp);
	    stylePtr->base.symbolSize = rp->width / 2;
	    stylePtr->bars = rp;
	    for (i = 0; i < barPtr->nBars; i++) {
		int iData;

		iData = barPtr->barToData[i];
		if (dataToStyle[iData] == (PenStyle *)stylePtr) {
		    *rp++ = barPtr->bars[i];
		    *ip++ = iData;
		}
	    }
	    stylePtr->nBars = rp - stylePtr->bars;
	}
	Blt_Free(barPtr->bars);
	Blt_Free(barPtr->barToData);
	barPtr->bars = bars;
	barPtr->barToData = barToData;
    }

    if (elemPtr->xErrorBarCnt > 0) {
	Blt_ChainLink *lp;
	Segment2D *errorBars, *sp;
	int *errorToData, *ip;

	errorBars = Blt_Malloc(elemPtr->xErrorBarCnt * sizeof(Segment2D));
	errorToData = Blt_Malloc(elemPtr->xErrorBarCnt * sizeof(int));
	assert(errorBars);
	sp = errorBars, ip = errorToData;
	for (lp = Blt_ChainFirstLink(elemPtr->stylePalette); 
	     lp != NULL; lp = Blt_ChainNextLink(lp)) {
	    PenStyle *stylePtr;
	    int i;

	    stylePtr = Blt_ChainGetValue(lp);
	    stylePtr->xErrorBars = sp;
	    for (i = 0; i < elemPtr->xErrorBarCnt; i++) {
		int iData;

		iData = elemPtr->xErrorToData[i];
		if (dataToStyle[iData] == stylePtr) {
		    *sp++ = elemPtr->xErrorBars[i];
		    *ip++ = iData;
		}
	    }
	    stylePtr->xErrorBarCnt = sp - stylePtr->xErrorBars;
	}
	Blt_Free(elemPtr->xErrorBars);
	elemPtr->xErrorBars = errorBars;
	Blt_Free(elemPtr->xErrorToData);
	elemPtr->xErrorToData = errorToData;
    }
    if (elemPtr->yErrorBarCnt > 0) {
	Blt_ChainLink *lp;
	Segment2D *errorBars, *sp;
	int *errorToData, *ip;

	errorBars = Blt_Malloc(elemPtr->yErrorBarCnt * sizeof(Segment2D));
	errorToData = Blt_Malloc(elemPtr->yErrorBarCnt * sizeof(int));
	assert(errorBars);
	sp = errorBars, ip = errorToData;
	for (lp = Blt_ChainFirstLink(elemPtr->stylePalette); lp != NULL; 
	     lp = Blt_ChainNextLink(lp)) {
	    PenStyle *stylePtr;
	    int i;

	    stylePtr = Blt_ChainGetValue(lp);
	    stylePtr->yErrorBars = sp;
	    for (i = 0; i < elemPtr->yErrorBarCnt; i++) {
		int iData;

		iData = elemPtr->yErrorToData[i];
		if (dataToStyle[iData] == stylePtr) {
		    *sp++ = elemPtr->yErrorBars[i];
		    *ip++ = iData;
		}
	    }
	    stylePtr->yErrorBarCnt = sp - stylePtr->yErrorBars;
	}
	Blt_Free(elemPtr->yErrorBars);
	elemPtr->yErrorBars = errorBars;
	Blt_Free(elemPtr->yErrorToData);
	elemPtr->yErrorToData = errorToData;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MapActiveBars --
 *
 *	Creates an array of points of the active graph coordinates.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is freed and allocated for the active point array.
 *
 *----------------------------------------------------------------------
 */
static void
MapActiveBars(Element *elemPtr)
{
    Bar *barPtr = (Bar *)elemPtr;

    if (barPtr->activeRects != NULL) {
	Blt_Free(barPtr->activeRects);
	barPtr->activeRects = NULL;
    }
    if (barPtr->activeToData != NULL) {
	Blt_Free(barPtr->activeToData);
	barPtr->activeToData = NULL;
    }
    barPtr->nActive = 0;

    if (elemPtr->nActiveIndices > 0) {
	XRectangle *activeRects;
	int *activeToData;
	int i;
	int count;

	activeRects = Blt_Malloc(sizeof(XRectangle) * elemPtr->nActiveIndices);
	assert(activeRects);
	activeToData = Blt_Malloc(sizeof(int) * elemPtr->nActiveIndices);
	assert(activeToData);
	count = 0;
	for (i = 0; i < barPtr->nBars; i++) {
	    int *ip, *iend;

	    for (ip = elemPtr->activeIndices, 
		     iend = ip + elemPtr->nActiveIndices; ip < iend; ip++) {
		if (barPtr->barToData[i] == *ip) {
		    activeRects[count] = barPtr->bars[i];
		    activeToData[count] = i;
		    count++;
		}
	    }
	}
	barPtr->nActive = count;
	barPtr->activeRects = activeRects;
	barPtr->activeToData = activeToData;
    }
    elemPtr->flags &= ~ACTIVE_PENDING;
}

static void
ResetBar(Element *elemPtr)
{
    Bar *barPtr = (Bar *)elemPtr;

    /* Release any storage associated with the display of the bar */
    ResetStylePalette(elemPtr->stylePalette);
    if (barPtr->activeRects != NULL) {
	Blt_Free(barPtr->activeRects);
    }
    if (barPtr->activeToData != NULL) {
	Blt_Free(barPtr->activeToData);
    }
    if (elemPtr->xErrorBars != NULL) {
	Blt_Free(elemPtr->xErrorBars);
    }
    if (elemPtr->xErrorToData != NULL) {
	Blt_Free(elemPtr->xErrorToData);
    }
    if (elemPtr->yErrorBars != NULL) {
	Blt_Free(elemPtr->yErrorBars);
    }
    if (elemPtr->yErrorToData != NULL) {
	Blt_Free(elemPtr->yErrorToData);
    }
    if (barPtr->bars != NULL) {
	Blt_Free(barPtr->bars);
    }
    if (barPtr->barToData != NULL) {
	Blt_Free(barPtr->barToData);
    }
    barPtr->activeToData = elemPtr->xErrorToData = elemPtr->yErrorToData = 
	barPtr->barToData = NULL;
    barPtr->activeRects = barPtr->bars = NULL;
    elemPtr->xErrorBars = elemPtr->yErrorBars = NULL;
    barPtr->nActive = elemPtr->xErrorBarCnt = elemPtr->yErrorBarCnt = 
	barPtr->nBars = 0;
}

/*
 * ----------------------------------------------------------------------
 *
 * MapBar --
 *
 *	Calculates the actual window coordinates of the bar element.
 *	The window coordinates are saved in the bar element structure.
 *
 * Results:
 *	None.
 *
 * Notes:
 *	A bar can have multiple segments (more than one x,y pairs).
 *	In this case, the bar can be represented as either a set of
 *	non-contiguous bars or a single multi-segmented (stacked) bar.
 *
 *	The x-axis layout for a barchart may be presented in one of
 *	two ways.  If abscissas are used, the bars are placed at those
 *	coordinates.  Otherwise, the range will represent the number
 *	of values.
 *
 * ----------------------------------------------------------------------
 */
static void
MapBar(Graph *graphPtr, Element *elemPtr)
{
    Bar *barPtr = (Bar *)elemPtr;
    PenStyle **dataToStyle;
    double *x, *y;
    double barWidth, barOffset;
    double baseline;
    int *barToData;		/* Maps bars to data point indices */
    int invertBar;
    int nPoints, count;
    XRectangle *rp, *bars;
    int i;
    int size;

    ResetBar(elemPtr);
    nPoints = NUMBEROFPOINTS(elemPtr);
    if (nPoints < 1) {
	return;			/* No data points */
    }
    barWidth = graphPtr->barWidth;
    if (barPtr->barWidth > 0.0) {
	barWidth = barPtr->barWidth;
    }
    baseline = (elemPtr->axes.y->logScale) ? 1.0 : graphPtr->baseline;
    barOffset = barWidth * 0.5;

    /*
     * Create an array of bars representing the screen coordinates
     * of all the segments in the bar.
     */
    bars = Blt_Calloc(nPoints, sizeof(XRectangle));
    assert(bars);
    barToData = Blt_Calloc(nPoints, sizeof(int));
    assert(barToData);

    x = elemPtr->x.valueArr, y = elemPtr->y.valueArr;
    count = 0;
    for (rp = bars, i = 0; i < nPoints; i++) {
	Point2D c1, c2;		/* Two opposite corners of the rectangle
				 * in graph coordinates. */
	double dx, dy;
	int height;

	if (((x[i] - barWidth) > elemPtr->axes.x->axisRange.max) ||
	    ((x[i] + barWidth) < elemPtr->axes.x->axisRange.min)) {
	    continue;		/* Abscissa is out of range of the x-axis */
	}
	c1.x = x[i] - barOffset;
	c1.y = y[i];
	c2.x = c1.x + barWidth;
	c2.y = baseline;

	/*
	 * If the mode is "aligned" or "stacked" we need to adjust the
	 * x or y coordinates of the two corners.
	 */

	if ((graphPtr->nStacks > 0) && (graphPtr->mode != MODE_INFRONT)) {
	    Blt_HashEntry *hPtr;
	    FreqKey key;

	    key.value = x[i];
	    key.axes = elemPtr->axes;
	    hPtr = Blt_FindHashEntry(&graphPtr->freqTable, (char *)&key);
	    if (hPtr != NULL) {
		FreqInfo *infoPtr;
		double slice, width;

		infoPtr = (FreqInfo *)Blt_GetHashValue(hPtr);
		switch (graphPtr->mode) {
		case MODE_STACKED:
		    c2.y = infoPtr->lastY;
		    c1.y += c2.y;
		    infoPtr->lastY = c1.y;
		    break;

		case MODE_ALIGNED:
		    infoPtr->count++;
		    slice = barWidth / (double)infoPtr->freq;
		    c1.x += slice * (infoPtr->freq - infoPtr->count);
		    c2.x = c1.x + slice;
		    break;

		case MODE_OVERLAP:
		    infoPtr->count++;
		    slice = barWidth / (double)(infoPtr->freq * 2);
		    width = slice * (infoPtr->freq + 1);
		    c1.x += slice * (infoPtr->freq - infoPtr->count);
		    c2.x = c1.x + width;
		    break;
		case MODE_INFRONT:
		    break;
		}
	    }
	}
	invertBar = FALSE;
	if (c1.y < c2.y) {
	    double temp;

	    /* Handle negative bar values by swapping ordinates */
	    temp = c1.y, c1.y = c2.y, c2.y = temp;
	    invertBar = TRUE;
	}
	/*
	 * Get the two corners of the bar segment and compute the rectangle
	 */
	c1 = Blt_Map2D(graphPtr, c1.x, c1.y, &elemPtr->axes);
	c2 = Blt_Map2D(graphPtr, c2.x, c2.y, &elemPtr->axes);

	/* Bound the bars horizontally by the width of the graph window */
	/* Bound the bars vertically by the position of the axis. */
	if (graphPtr->stackAxes) {
	    int right, left, top, bottom;

	    if (graphPtr->inverted) {
		left = elemPtr->axes.y->screenMin;
		right = elemPtr->axes.y->screenMin + 
		    elemPtr->axes.y->screenRange;
		top = graphPtr->top;
		bottom = graphPtr->bottom;
	    } else {
		top = elemPtr->axes.y->screenMin;
		bottom = elemPtr->axes.y->screenMin + 
		    elemPtr->axes.y->screenRange;
		left = graphPtr->left;
		right = graphPtr->right;
	    }
	    if (c1.y < (double)top) {
		c1.y = (double)top;
	    } else if (c1.y > (double)bottom) {
		c1.y = (double)bottom;
	    }
	    if (c2.y < (double)top) {
		c2.y = (double)top;
	    } else if (c2.y > (double)bottom) {
		c2.y = (double)bottom;
	    }
	    if (c1.x < (double)left) {
		c1.x = (double)left;
	    } else if (c1.x > (double)right) {
		c1.x = (double)right;
	    }
	    if (c2.x < (double)left) {
		c2.x = (double)left;
	    } else if (c2.x > (double)right) {
		c2.x = (double)right;
	    }
	} else {
	    int right, left, top, bottom;

	    if (graphPtr->inverted) {
		top = graphPtr->left;
		bottom = graphPtr->right;
		left = graphPtr->top;
		right = graphPtr->bottom;
	    } else {
		top = graphPtr->top;
		bottom = graphPtr->bottom;
		left = graphPtr->left;
		right = graphPtr->right;
	    }

	    if (c1.y < 0.0) {
		c1.y = 0.0;
	    } else if (c1.y > (double)graphPtr->height) {
		c1.y = (double)graphPtr->height;
	    }
	    if (c2.y < 0.0) {
		c2.y = 0.0;
	    } else if (c2.y > (double)graphPtr->height) {
		c2.y = (double)graphPtr->height;
	    }
	}
	dx = FABS(c1.x - c2.x);
	dy = FABS(c1.y - c2.y);
	if ((dx == 0) || (dy == 0)) {
	    continue;
	}
	height = (int)dy;
	if (invertBar) {
	    rp->y = (short int)MIN(c1.y, c2.y);
	} else {
	    rp->y = (short int)(MAX(c1.y, c2.y)) - height;
	}
	rp->x = (short int)MIN(c1.x, c2.x);
	rp->width = (short int)dx + 1;
	if (rp->width < 1) {
	    rp->width = 1;
	}
	rp->height = height + 1;
	if (rp->height < 1) {
	    rp->height = 1;
	}
	barToData[count] = i;	/* Save the data index corresponding to the
				 * rectangle */
	count++;
	rp++;
    }
    barPtr->nBars = count;
    barPtr->bars = bars;
    barPtr->barToData = barToData;
    if (elemPtr->nActiveIndices > 0) {
	MapActiveBars(elemPtr);
    }
	
    size = 20;
    if (count > 0) {
	size = bars->width;
    }
    {
	Blt_ChainLink *lp;

	/* Set the symbol size of all the pen styles. */
	for (lp = Blt_ChainFirstLink(elemPtr->stylePalette); lp != NULL;
	     lp = Blt_ChainNextLink(lp)) {
	    PenStyle *stylePtr;
	    BarPen *penPtr;
	    
	    stylePtr = Blt_ChainGetValue(lp);
	    penPtr = (BarPen *)stylePtr->penPtr;
	    stylePtr->symbolSize = size;
	    stylePtr->errorBarCapWidth = (penPtr->errorBarCapWidth > 0) 
		? penPtr->errorBarCapWidth : (size * 66666) / 100000;
	    stylePtr->errorBarCapWidth /= 2;
	}
    }
    dataToStyle = Blt_StyleMap(elemPtr);
    if (((elemPtr->yHigh.nValues > 0) && (elemPtr->yLow.nValues > 0)) ||
	((elemPtr->xHigh.nValues > 0) && (elemPtr->xLow.nValues > 0)) ||
	(elemPtr->xError.nValues > 0) || (elemPtr->yError.nValues > 0)) {
	Blt_MapErrorBars(graphPtr, elemPtr, dataToStyle);
    }
    MergePens(elemPtr, dataToStyle);
    Blt_Free(dataToStyle);
}

/*
 * -----------------------------------------------------------------
 *
 * DrawSymbol --
 *
 * 	Draw a symbol centered at the given x,y window coordinate
 *	based upon the element symbol type and size.
 *
 * Results:
 *	None.
 *
 * Problems:
 *	Most notable is the round-off errors generated when
 *	calculating the centered position of the symbol.
 * -----------------------------------------------------------------
 */
/*ARGSUSED*/
static void
DrawSymbol(
    Graph *graphPtr,
    Drawable drawable,		/* Pixmap or window to draw into */
    Element *elemPtr,
    int x, int y,
    int size)
{
    BarPen *penPtr;
    int radius;

    penPtr = (BarPen *)NORMALPEN(elemPtr);
    if ((penPtr->border == NULL) && (penPtr->fgColor == NULL)) {
	return;
    }
    radius = (size / 2);
    size--;

    x -= radius;
    y -= radius;
    XSetTSOrigin(graphPtr->display, penPtr->gc, x, y);
    XFillRectangle(graphPtr->display, drawable, penPtr->gc, x, y, size, size);
    XSetTSOrigin(graphPtr->display, penPtr->gc, 0, 0);
}

/*
 * -----------------------------------------------------------------
 *
 * DrawBarSegments --
 *
 * 	Draws each of the rectangular segments for the element.
 *
 * Results:
 *	None.
 *
 * -----------------------------------------------------------------
 */
static void
DrawBarSegments(
    Graph *graphPtr,
    Drawable drawable,		/* Pixmap or window to draw into */
    BarPen *penPtr,
    XRectangle *bars,
    int nBars)
{

    if ((penPtr->border == NULL) && (penPtr->fgColor == NULL)) {
	return;
    }
    XFillRectangles(graphPtr->display, drawable, penPtr->gc, bars, 
		    nBars);
    if ((penPtr->border != NULL) && (penPtr->borderWidth > 0) && 
	(penPtr->relief != TK_RELIEF_FLAT)) {
	XRectangle *rp, *rend;

	for (rp = bars, rend = rp + nBars; rp < rend; rp++) {
	    Blt_Draw3DRectangle(graphPtr->tkwin, drawable, penPtr->border,
		rp->x, rp->y, rp->width, rp->height, penPtr->borderWidth, 
		penPtr->relief);
	}
    }
}

/*
 * -----------------------------------------------------------------
 *
 * DrawBarValues --
 *
 * 	Draws the numeric value of the bar.
 *
 * Results:
 *	None.
 *
 * -----------------------------------------------------------------
 */
static void
DrawBarValues(
    Graph *graphPtr, 
    Drawable drawable, 
    Element *elemPtr,
    BarPen *penPtr,
    XRectangle *bars,
    int nBars,
    int *barToData)
{
    XRectangle *rp, *rend;
    int count;
    char *fmt;
    
    fmt = penPtr->valueFormat;
    if (fmt == NULL) {
	fmt = "%g";
    }
    count = 0;
    for (rp = bars, rend = rp + nBars; rp < rend; rp++) {
	Point2D anchorPos;
	double x, y;
	char string[TCL_DOUBLE_SPACE * 2 + 2];

	x = elemPtr->x.valueArr[barToData[count]];
	y = elemPtr->y.valueArr[barToData[count]];

	count++;
	if (penPtr->valueShow == SHOW_X) {
	    sprintf(string, fmt, x); 
	} else if (penPtr->valueShow == SHOW_Y) {
	    sprintf(string, fmt, y); 
	} else if (penPtr->valueShow == SHOW_BOTH) {
	    sprintf(string, fmt, x);
	    strcat(string, ",");
	    sprintf(string + strlen(string), fmt, y);
	}
	if (graphPtr->inverted) {
	    anchorPos.y = rp->y + rp->height * 0.5;
	    anchorPos.x = rp->x + rp->width;
	    if (y < graphPtr->baseline) {
		anchorPos.x -= rp->width;
	    } 
	} else {
	    anchorPos.x = rp->x + rp->width * 0.5;
	    anchorPos.y = rp->y;
	    if (y < graphPtr->baseline) {			
		anchorPos.y += rp->height;
	    }
	}
	Blt_DrawText(graphPtr->tkwin, drawable, string, &penPtr->valueStyle, 
		     (int)anchorPos.x, (int)anchorPos.y);
    }
}


/*
 * ----------------------------------------------------------------------
 *
 * DrawNormalBar --
 *
 *	Draws the rectangle representing the bar element.  If the
 *	relief option is set to "raised" or "sunken" and the bar
 *	borderwidth is set (borderwidth > 0), a 3D border is drawn
 *	around the bar.
 *
 *	Don't draw bars that aren't visible (i.e. within the limits
 *	of the axis).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	X drawing commands are output.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawNormalBar(Graph *graphPtr, Drawable drawable, Element *elemPtr)
{
    Bar *barPtr = (Bar *)elemPtr;
    int count;
    Blt_ChainLink *lp;

    count = 0;
    for (lp = Blt_ChainFirstLink(elemPtr->stylePalette); lp != NULL;
	 lp = Blt_ChainNextLink(lp)) {
	PenStyle *stylePtr;
	BarPenStyle *barStylePtr;
	BarPen *penPtr;

	stylePtr = Blt_ChainGetValue(lp);
	barStylePtr = (BarPenStyle *)stylePtr;
	penPtr = (BarPen *)stylePtr->penPtr;
	if (barStylePtr->nBars > 0) {
	    DrawBarSegments(graphPtr, drawable, penPtr, barStylePtr->bars,
		barStylePtr->nBars);
	}
	if ((stylePtr->xErrorBarCnt > 0) && (penPtr->errorBarShow & SHOW_X)) {
	    Blt_Draw2DSegments(graphPtr->display, drawable, penPtr->errorBarGC, 
		       stylePtr->xErrorBars, stylePtr->xErrorBarCnt);
	}
	if ((stylePtr->yErrorBarCnt > 0) && (penPtr->errorBarShow & SHOW_Y)) {
	    Blt_Draw2DSegments(graphPtr->display, drawable, penPtr->errorBarGC, 
		       stylePtr->yErrorBars, stylePtr->yErrorBarCnt);
	}
	if (penPtr->valueShow != SHOW_NONE) {
	    DrawBarValues(graphPtr, drawable, elemPtr, penPtr, 
			barStylePtr->bars, barStylePtr->nBars, 
			barPtr->barToData + count);
	}
	count += barStylePtr->nBars;
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DrawActiveBar --
 *
 *	Draws bars representing the active segments of the
 *	bar element.  If the -relief option is set (other than "flat")
 *	and the borderwidth is greater than 0, a 3D border is drawn
 *	around the each bar segment.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	X drawing commands are output.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawActiveBar(Graph *graphPtr, Drawable drawable, Element *elemPtr)
{
    Bar *barPtr = (Bar *)elemPtr;

    if (elemPtr->activePenPtr != NULL) {
	BarPen *penPtr = (BarPen *)elemPtr->activePenPtr;

	if (elemPtr->nActiveIndices > 0) {
	    if (elemPtr->flags & ACTIVE_PENDING) {
		MapActiveBars(elemPtr);
	    }
	    DrawBarSegments(graphPtr, drawable, penPtr, barPtr->activeRects, 
			 barPtr->nActive);
	    if (penPtr->valueShow != SHOW_NONE) {
		DrawBarValues(graphPtr, drawable, elemPtr, penPtr, 
			barPtr->activeRects, barPtr->nActive, 
			barPtr->activeToData);
	    }
	} else if (elemPtr->nActiveIndices < 0) {
	    DrawBarSegments(graphPtr, drawable, penPtr, barPtr->bars, 
			 barPtr->nBars);
	    if (penPtr->valueShow != SHOW_NONE) {
		DrawBarValues(graphPtr, drawable, elemPtr, penPtr, 
			barPtr->bars, barPtr->nBars, barPtr->barToData);
	    }
	}
    }
}

/*
 * -----------------------------------------------------------------
 *
 * SymbolToPostScript --
 *
 * 	Draw a symbol centered at the given x,y window coordinate
 *	based upon the element symbol type and size.
 *
 * Results:
 *	None.
 *
 * Problems:
 *	Most notable is the round-off errors generated when
 *	calculating the centered position of the symbol.
 *
 * -----------------------------------------------------------------
 */
/*ARGSUSED*/
static void
SymbolToPostScript(
    Graph *graphPtr,
    Blt_PostScript ps,
    Element *elemPtr,
    double x, double y,
    int size)
{
    BarPen *penPtr;

    penPtr = (BarPen *)NORMALPEN(elemPtr);
    if ((penPtr->border == NULL) && (penPtr->fgColor == NULL)) {
	return;
    }
    /*
     * Build a PostScript procedure to draw the fill and outline of
     * the symbol after the path of the symbol shape has been formed
     */
    Blt_AppendToPostScript(ps, "\n",
	"/DrawSymbolProc {\n",
	"  gsave\n    ", (char *)NULL);
    if (penPtr->stipple != None) {
	if (penPtr->border != NULL) {
	    Blt_BackgroundToPostScript(ps, Tk_3DBorderColor(penPtr->border));
	    Blt_AppendToPostScript(ps, "    Fill\n    ", (char *)NULL);
	}
	if (penPtr->fgColor != NULL) {
	    Blt_ForegroundToPostScript(ps, penPtr->fgColor);
	} else {
	    Blt_ForegroundToPostScript(ps, Tk_3DBorderColor(penPtr->border));
	}
	Blt_StippleToPostScript(ps, graphPtr->display, penPtr->stipple);
    } else if (penPtr->fgColor != NULL) {
	Blt_ForegroundToPostScript(ps, penPtr->fgColor);
	Blt_AppendToPostScript(ps, "    fill\n", (char *)NULL);
    }
    Blt_AppendToPostScript(ps, "  grestore\n", (char *)NULL);
    Blt_AppendToPostScript(ps, "} def\n\n", (char *)NULL);
    Blt_FormatToPostScript(ps, "%g %g %d Sq\n", x, y, size);
}

static void
SegmentsToPostScript(
    Graph *graphPtr,
    Blt_PostScript ps,
    BarPen *penPtr,
    XRectangle *bars,
    int nBars)
{
    XRectangle *rp, *rend;

    if ((penPtr->border == NULL) && (penPtr->fgColor == NULL)) {
	return;
    }
    for (rp = bars, rend = rp + nBars; rp < rend; rp++) {
	if ((rp->width < 1) || (rp->height < 1)) {
	    continue;
	}
	if (penPtr->stipple != None) {
	    Blt_RegionToPostScript(ps, (double)rp->x, (double)rp->y, 
		(int)rp->width - 1, (int)rp->height - 1);
	    if (penPtr->border != NULL) {
		Blt_BackgroundToPostScript(ps, 
			Tk_3DBorderColor(penPtr->border));
		Blt_AppendToPostScript(ps, "Fill\n", (char *)NULL);
	    }
	    if (penPtr->fgColor != NULL) {
		Blt_ForegroundToPostScript(ps, penPtr->fgColor);
	    } else {
		Blt_ForegroundToPostScript(ps, 
			Tk_3DBorderColor(penPtr->border));
	    }
	    Blt_StippleToPostScript(ps, graphPtr->display, penPtr->stipple);
	} else if (penPtr->fgColor != NULL) {
	    Blt_ForegroundToPostScript(ps, penPtr->fgColor);
	    Blt_RectangleToPostScript(ps, (double)rp->x, (double)rp->y, 
		(int)rp->width - 1, (int)rp->height - 1);
	}
	if ((penPtr->border != NULL) && (penPtr->borderWidth > 0) && 
	    (penPtr->relief != TK_RELIEF_FLAT)) {
	    Blt_Draw3DRectangleToPostScript(ps, penPtr->border, 
		(double)rp->x, (double)rp->y, (int)rp->width, (int)rp->height,
		penPtr->borderWidth, penPtr->relief);
	}
    }
}

static void
BarValuesToPostScript(
    Graph *graphPtr,
    Blt_PostScript ps,
    Element *elemPtr,
    BarPen *penPtr,
    XRectangle *bars,
    int nBars,
    int *barToData)
{
    XRectangle *rp, *rend;
    int count;
    char *fmt;
    char string[TCL_DOUBLE_SPACE * 2 + 2];
    double x, y;
    Point2D anchorPos;
    
    count = 0;
    fmt = penPtr->valueFormat;
    if (fmt == NULL) {
	fmt = "%g";
    }
    for (rp = bars, rend = rp + nBars; rp < rend; rp++) {
	x = elemPtr->x.valueArr[barToData[count]];
	y = elemPtr->y.valueArr[barToData[count]];
	count++;
	if (penPtr->valueShow == SHOW_X) {
	    sprintf(string, fmt, x); 
	} else if (penPtr->valueShow == SHOW_Y) {
	    sprintf(string, fmt, y); 
	} else if (penPtr->valueShow == SHOW_BOTH) {
	    sprintf(string, fmt, x);
	    strcat(string, ",");
	    sprintf(string + strlen(string), fmt, y);
	}
	if (graphPtr->inverted) {
	    anchorPos.y = rp->y + rp->height * 0.5;
	    anchorPos.x = rp->x + rp->width;
	    if (y < graphPtr->baseline) {
		anchorPos.x -= rp->width;
	    } 
	} else {
	    anchorPos.x = rp->x + rp->width * 0.5;
	    anchorPos.y = rp->y;
	    if (y < graphPtr->baseline) {			
		anchorPos.y += rp->height;
	    }
	}
	Blt_TextToPostScript(ps, string, &penPtr->valueStyle, anchorPos.x, 
		anchorPos.y);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ActiveBarToPostScript --
 *
 *	Similar to the NormalBarToPostScript procedure, generates
 *	PostScript commands to display the bars representing the
 *	active bar segments of the element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript pen width, dashes, and color settings are changed.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
ActiveBarToPostScript(
    Graph *graphPtr,
    Blt_PostScript ps,
    Element *elemPtr)
{
    Bar *barPtr = (Bar *)elemPtr;

    if (elemPtr->activePenPtr != NULL) {
	BarPen *penPtr = (BarPen *)elemPtr->activePenPtr;
	
	if (elemPtr->nActiveIndices > 0) {
	    if (elemPtr->flags & ACTIVE_PENDING) {
		MapActiveBars(elemPtr);
	    }
	    SegmentsToPostScript(graphPtr, ps, penPtr, barPtr->activeRects,
		barPtr->nActive);
	    if (penPtr->valueShow != SHOW_NONE) {
		BarValuesToPostScript(graphPtr, ps, elemPtr, penPtr, 
		   barPtr->activeRects, barPtr->nActive, barPtr->activeToData);
	    }
	} else if (elemPtr->nActiveIndices < 0) {
	    SegmentsToPostScript(graphPtr, ps, penPtr, barPtr->bars, 
		barPtr->nBars);
	    if (penPtr->valueShow != SHOW_NONE) {
		BarValuesToPostScript(graphPtr, ps, elemPtr, penPtr, 
		   barPtr->bars, barPtr->nBars, barPtr->barToData);
	    }
	}
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * NormalBarToPostScript --
 *
 *	Generates PostScript commands to form the bars
 *	representing the segments of the bar element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript pen width, dashes, and color settings are changed.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
NormalBarToPostScript(
    Graph *graphPtr,
    Blt_PostScript ps,
    Element *elemPtr)
{
    Bar *barPtr = (Bar *)elemPtr;
    Blt_ChainLink *lp;
    int count;

    count = 0;
    for (lp = Blt_ChainFirstLink(elemPtr->stylePalette); lp != NULL;
	 lp = Blt_ChainNextLink(lp)) {
	PenStyle *stylePtr;
	BarPen *penPtr;
	BarPenStyle *barStylePtr;
	XColor *colorPtr;

	stylePtr = Blt_ChainGetValue(lp);
	barStylePtr = (BarPenStyle *)stylePtr;
	penPtr = (BarPen *)stylePtr->penPtr;
	if (barStylePtr->nBars > 0) {
	    SegmentsToPostScript(graphPtr, ps, penPtr, barStylePtr->bars, 
		barStylePtr->nBars);
	}
	colorPtr = penPtr->errorBarColor;
	if (colorPtr == COLOR_DEFAULT) {
	    colorPtr = penPtr->fgColor;
	}
	if ((stylePtr->xErrorBarCnt > 0) && (penPtr->errorBarShow & SHOW_X)) {
	    Blt_LineAttributesToPostScript(ps, colorPtr, 
		penPtr->errorBarLineWidth, NULL, CapButt, JoinMiter);
	    Blt_2DSegmentsToPostScript(ps, stylePtr->xErrorBars,
		stylePtr->xErrorBarCnt);
	}
	if ((stylePtr->yErrorBarCnt > 0) && (penPtr->errorBarShow & SHOW_Y)) {
	    Blt_LineAttributesToPostScript(ps, colorPtr, 
		penPtr->errorBarLineWidth, NULL, CapButt, JoinMiter);
	    Blt_2DSegmentsToPostScript(ps, stylePtr->yErrorBars,
		stylePtr->yErrorBarCnt);
	}
	if (penPtr->valueShow != SHOW_NONE) {
	    BarValuesToPostScript(graphPtr, ps, elemPtr, penPtr, 
		barStylePtr->bars, barStylePtr->nBars, 
		barPtr->barToData + count);
	}
	count += barStylePtr->nBars;
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyBar --
 *
 *	Release memory and resources allocated for the bar element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the bar element is freed up.
 *
 * ----------------------------------------------------------------------
 */
#define FreeElemVector(v) \
    if ((v).clientId != NULL) { \
	Blt_FreeVectorId((v).clientId); \
    } else if ((v).valueArr != NULL) { \
	Blt_Free((v).valueArr); \
    } 

static void
DestroyBar(Graph *graphPtr, Element *elemPtr)
{
    DestroyBarPen(graphPtr, elemPtr->builtinPenPtr);
    ResetBar(elemPtr);
    if (elemPtr->activeIndices != NULL) {
	Blt_Free(elemPtr->activeIndices);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_BarElement --
 *
 *	Allocate memory and initialize methods for the new bar element.
 *
 * Results:
 *	The pointer to the newly allocated element structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the bar element structure.
 *
 * ----------------------------------------------------------------------
 */

static ElementProcs barProcs =
{
    ClosestBar,
    ConfigureBar,
    DestroyBar,
    DrawActiveBar,
    DrawNormalBar,
    DrawSymbol,
    GetBarExtents,
    ActiveBarToPostScript,
    NormalBarToPostScript,
    SymbolToPostScript,
    MapBar,
};


Element *
Blt_BarElement(Graph *graphPtr, char *name, int classId)
{
    Bar *barPtr;

    barPtr = Blt_Calloc(1, sizeof(Bar));
    assert(barPtr);
    barPtr->base.procsPtr = &barProcs;
    barPtr->base.configSpecs = barElemConfigSpecs;
    barPtr->base.labelRelief = TK_RELIEF_FLAT;
    Blt_GraphSetObjectClass(&barPtr->base.object, classId);
    barPtr->base.object.name = Blt_Strdup(name);
    barPtr->base.object.graphPtr = graphPtr;
    barPtr->base.object.hidden = FALSE;

    /* By default, an element's name and label are the same. */
    barPtr->base.label = Blt_Strdup(name);
    barPtr->base.builtinPenPtr = (Pen *)&barPtr->builtinPen;

    InitializeBarPen(&barPtr->builtinPen);
    barPtr->base.stylePalette = Blt_ChainCreate();
    bltBarStylesOption.clientData = (ClientData)sizeof(BarPenStyle);
    return (Element *)barPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_InitFreqTable --
 *
 *	Generate a table of abscissa frequencies.  Duplicate
 *	x-coordinates (depending upon the bar drawing mode) indicate
 *	that something special should be done with each bar segment
 *	mapped to the same abscissa (i.e. it should be stacked,
 *	aligned, or overlay-ed with other segments)
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is allocated for the bar element structure.
 *
 * ----------------------------------------------------------------------
 */
void
Blt_InitFreqTable(Graph *graphPtr)
{
    Blt_ChainLink *lp;
    int nStacks, nSegs;
    Blt_HashTable freqTable;

    /*
     * Free resources associated with a previous frequency table. This
     * includes the array of frequency information and the table itself
     */
    if (graphPtr->freqArr != NULL) {
	Blt_Free(graphPtr->freqArr);
	graphPtr->freqArr = NULL;
    }
    if (graphPtr->nStacks > 0) {
	Blt_DeleteHashTable(&graphPtr->freqTable);
	graphPtr->nStacks = 0;
    }
    if (graphPtr->mode == MODE_INFRONT) {
	return;			/* No frequency table is needed for
				 * "infront" mode */
    }
    Blt_InitHashTable(&graphPtr->freqTable, sizeof(FreqKey) / sizeof(int));

    /*
     * Initialize a hash table and fill it with unique abscissas.
     * Keep track of the frequency of each x-coordinate and how many
     * abscissas have duplicate mappings.
     */
    Blt_InitHashTable(&freqTable, sizeof(FreqKey) / sizeof(int));
    nSegs = nStacks = 0;
    for (lp = Blt_ChainFirstLink(graphPtr->elements.displayList);
	lp != NULL; lp = Blt_ChainNextLink(lp)) {
	Bar *barPtr;
	Element *elemPtr;
	double *xArr;
	int nPoints;
	int i;

	elemPtr = Blt_ChainGetValue(lp);
	if ((elemPtr->object.hidden) || 
	    (elemPtr->object.classId != OBJECT_CLASS_BAR_ELEMENT)) {
	    continue;
	}
	nSegs++;
	barPtr = (Bar *)elemPtr;
	xArr = elemPtr->x.valueArr;
	nPoints = NUMBEROFPOINTS(elemPtr);
	for (i = 0; i < nPoints; i++) {
	    Blt_HashEntry *hPtr;
	    FreqKey key;
	    int isNew;
	    size_t count;

	    key.value = xArr[i];
	    key.axes = elemPtr->axes;
	    hPtr = Blt_CreateHashEntry(&freqTable, (char *)&key, &isNew);
	    assert(hPtr != NULL);
	    if (isNew) {
		count = 1;
	    } else {
		count = (size_t)Blt_GetHashValue(hPtr);
		if (count == 1) {
		    nStacks++;
		}
		count++;
	    }
	    Blt_SetHashValue(hPtr, (ClientData)count);
	}
    }
    if (nSegs == 0) {
	return;			/* No bar elements to be displayed */
    }
    if (nStacks > 0) {
	FreqInfo *fp;
	Blt_HashEntry *h1;
	Blt_HashSearch cursor;

	graphPtr->freqArr = Blt_Calloc(nStacks, sizeof(FreqInfo));
	assert(graphPtr->freqArr);

	fp = graphPtr->freqArr;
	for (h1 = Blt_FirstHashEntry(&freqTable, &cursor); h1 != NULL;
	    h1 = Blt_NextHashEntry(&cursor)) {
	    FreqKey *keyPtr;
	    size_t count;

	    count = (size_t)Blt_GetHashValue(h1);
	    keyPtr = (FreqKey *)Blt_GetHashKey(&freqTable, h1);
	    if (count > 1) {
		Blt_HashEntry *h2;
		int isNew;

		h2 = Blt_CreateHashEntry(&graphPtr->freqTable, (char *)keyPtr, 
			&isNew);
		count = (size_t)Blt_GetHashValue(h1);
		fp->freq = count;
		fp->axes = keyPtr->axes;
		Blt_SetHashValue(h2, fp);
		fp++;
	    }
	}
    }
    Blt_DeleteHashTable(&freqTable);
    graphPtr->nStacks = nStacks;
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_ComputeStacks --
 *
 *	Determine the height of each stack of bar segments.  A stack
 *	is created by designating two or more points with the same
 *	abscissa.  Each ordinate defines the height of a segment in
 *	the stack.  This procedure simply looks at all the data points
 *	summing the heights of each stacked segment. The sum is saved
 *	in the frequency information table.  This value will be used
 *	to calculate the y-axis limits (data limits aren't sufficient).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The heights of each stack is computed. CheckStacks will
 *	use this information to adjust the y-axis limits if necessary.
 *
 * ----------------------------------------------------------------------
 */
void
Blt_ComputeStacks(Graph *graphPtr)
{
    Blt_ChainLink *lp;

    if ((graphPtr->mode != MODE_STACKED) || (graphPtr->nStacks == 0)) {
	return;
    }

    /* Reset the sums for all duplicate values to zero. */
    {
	FreqInfo *infoPtr;
	int i;

	infoPtr = graphPtr->freqArr;
	for (i = 0; i < graphPtr->nStacks; i++) {
	    infoPtr->sum = 0.0;
	    infoPtr++;
	}
    }

    /* Look at each bar point, adding the ordinates of duplicate abscissas */

    for (lp = Blt_ChainFirstLink(graphPtr->elements.displayList); lp != NULL; 
	lp = Blt_ChainNextLink(lp)) {
	Bar *barPtr;
	Element *elemPtr;
	double *xArr, *yArr;
	int nPoints;
	int i;

	elemPtr = Blt_ChainGetValue(lp);
	if ((elemPtr->object.hidden) || 
	    (elemPtr->object.classId != OBJECT_CLASS_BAR_ELEMENT)) {
	    continue;
	}
	barPtr = (Bar *)elemPtr;
	xArr = elemPtr->x.valueArr;
	yArr = elemPtr->y.valueArr;
	nPoints = NUMBEROFPOINTS(elemPtr);
	for (i = 0; i < nPoints; i++) {
	    Blt_HashEntry *hPtr;
	    FreqKey key;
	    FreqInfo *infoPtr;

	    key.value = xArr[i];
	    key.axes = elemPtr->axes;
	    hPtr = Blt_FindHashEntry(&graphPtr->freqTable, (char *)&key);
	    if (hPtr == NULL) {
		continue;
	    }
	    infoPtr = (FreqInfo *)Blt_GetHashValue(hPtr);
	    infoPtr->sum += yArr[i];
	}
    }
}

void
Blt_ResetStacks(Graph *graphPtr)
{
    FreqInfo *fp, *fend;

    for (fp = graphPtr->freqArr, fend = fp+graphPtr->nStacks; fp < fend; fp++) {
	fp->lastY = 0.0;
	fp->count = 0;
    }
}

