
/*
 *
 * bltDataTableCmd.c --
 *
 *	Copyright 1998-2005 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
  datatable create t0 t1 t2
  datatable names
  t0 destroy
     -or-
  datatable destroy t0
  datatable copy table@node table@node -recurse -tags

  table move node after|before|into t2@node

  $t apply -recurse $root command arg arg			

  $t attach tablename				

  $t children $n
  t0 copy node1 node2 node3 node4 node5 destName 
  $t delete $n...				
  $t delete 0 
  $t delete 0 10
  $t delete all
  $t depth $n
  $t dumpdata $root
  $t dumpfile $root fileName
  $t dup $t2		
  $t find $root -name pat -name pattern
  $t firstchild $n
  $t get $n $key
  $t get $n $key(abc)
  $t index $n
  $t insert $parent $switches?
  $t isancestor $n1 $n2
  $t isbefore $n1 $n2
  $t isleaf $n
  $t lastchild $n
  $t move $n1 after|before|into $n2
  $t next $n
  $t nextsibling $n
  $t path $n1 $n2 $n3...
  $t parent $n
  $t previous $n
  $t prevsibling $n
  $t restore $root data -overwrite
  $t root ?$n?

  $t set $n $key $value ?$key $value?
  $t size $n
  $t slink $n $t2@$node				???
  $t sort -recurse $root		

  $t tag delete tag1 tag2 tag3...
  $t tag names
  $t tag nodes $tag
  $t tag set $n tag1 tag2 tag3...
  $t tag unset $n tag1 tag2 tag3...

  $t trace create $n $key how command		
  $t trace delete id1 id2 id3...
  $t trace names
  $t trace info $id

  $t unset $n key1 key2 key3...
  
  $t row notify $row ?flags? command 
  $t column notify $column ?flags? command arg arg arg 
  $t notify create -oncreate -ondelete -onmove command 
  $t notify create -oncreate -ondelete -onmove -onsort command arg arg arg 
  $t notify delete id1 id2 id3
  $t notify names
  $t notify info id

  for { set n [$t firstchild $node] } { $n >= 0 } { 
        set n [$t nextsibling $n] } {
  }
  foreach n [$t children $node] { 
	  
  }
  set n [$t next $node]
  set n [$t previous $node]
*/

/*
 *
 *  datatable create ?name?
 *  datatable names
 *  datatable destroy $table
 *  
 *  $table array get $row $column $key $value ?defValue?
 *  $table array set $row $column $key $value
 *  $table array unset $row $column $key $value
 *  $table column delete row row row row 
 *  $table column extend 1
 *  $table column get $column
 *  $table column insert $column "label"
 *  $table column select $expr
 *  $table column set {1 20} $list
 *  $table column set 1-20 $list
 *  $table column set 1:20 $list
 *  $table column trace $row rwu proc
 *  $table column unset $column
 *  $table copy fromTable
 *  $table copy newTable
 *  $table dup newTable -includerows $tag $tag $tag -excludecolumns $tag $tag 
 *  $table get $row $column $value ?$defValue?
 *  $table pack 
 *  $table row delete row row row row 
 *  $table row insert -tags $tags -label $label -before -after 
 *  $table row extend 1
 *  $table row get $row
 *  $table row insert $column "label" "label"
 *  $table row select $expr
 *  $table row set $row $list
 *  $table row trace $row  rwu proc
 *  $table row unset $row
 *  $table row tag add where expr 
 *  $table row tag forget $tag $tag
 *  $table row tag delete $tag $row $row $row
 *  $table row tag find $row ?pattern*?
 *  $table row tag add $row tag tag tag tag 
 *  $table row tag unset $row tag tag tag 
 *  $table row find $expr
 *  $table select $expr
 *  $table set $row $column $value 
 *  $table trace $row $column rwu proc
 *  $table unset $row $column 
 *  $table column find $expr
 *  $table import -format {csv} -overwrite fileName
 *  $table export -format {csv,xml} fileName
 *  $table dumpdata -notags -nolabels -columns "tags" -rows "tags" string
 *  $table dump -notags -nolabels -columns "tags" -rows "tags" fileName
 *  $table dup $destTable -notags -nolabels -columns "tags" -rows "tags"
 *  $table restore -overwrite -notags -nolabels string
 *  $table restoredata -overwrite -notags -nolabels -data string
 *  $table restore -overwrite -notags -nolabels fileName
 *
 *  $table row hide label
 *  $table row unhide label
 *
 *  $table emptyvalue ?value?
 */
/* 
 * $table import -format tree {tree 10} 
 * $table import -format csv $fileName.csv
 *
 * $table import tree $tree 10 ?switches?
 * $table import csv ?switches? fileName.xml
 * $table importdata csv -separator , -quote " $fileName.csv
 * $table import csv -separator , -quote " -data string
 * $table exportdata xml ?switches? ?$fileName.xml?
 * $table export csv -separator \t -quote ' $fileName.csv
 * $table export csv -separator \t -quote ' 
 * $table export -format dbm $fileName.dbm
 * $table export -format db $fileName.db
 * $table export -format csv $fileName.csv
 */
/*
 * $vector attach "$table c $column"
 * $vector detach 
 * $graph element create x -x "${table} column ${column}" \
 *		"datatable0 r abc"
 * $tree attach 0 $table
 */

#include <bltInt.h>

#ifndef NO_DATATABLE

#include <bltNsUtil.h>
#include <bltSwitch.h>
#include <bltHash.h>
#include <bltList.h>
#include "bltSwitch.h"
#include <bltDataTable.h>
#include <ctype.h>

#define DATATABLE_THREAD_KEY "BLT DataTable Command Interface"

/*
 * DataTableCmdInterpData --
 *
 *	Structure containing global data, used on a interpreter by
 *	interpreter basis.
 *
 *	This structure holds the hash table of instances of datatable
 *	commands associated with a particular interpreter.  
 */
typedef struct {
    Blt_HashTable instTable;	/* Tracks datatables in use. */
    Tcl_Interp *interp;
} DataTableCmdInterpData;

/*
 * DataTableCmd --
 *
 *	Structure representing the Tcl command used to manipulate the
 *	underlying table object.  
 *
 *	This structure acts as a shell for a table object.  A table
 *	object maybe shared by more than one client.  This shell
 *	provides Tcl commands to access and change values as well as
 *	the structure of the table itself.  It manages the traces and
 *	notifier events that it creates, providing a Tcl interface to
 *	those facilities. It also provides a user-selectable value for
 *	empty-cell values.
 */
typedef struct {
    Tcl_Interp *interp;		/* Interpreter this command is
				 * associated with. */

    Blt_DataTable table;	/* Token representing the client
				 * table. */

    Tcl_Command cmdToken;	/* Token for Tcl command representing
				 * this datatable. */

    Tcl_Obj *emptyValueObjPtr;	/* Obj representing an empty value in
				 * the table. */

    Blt_HashTable *tablePtr;	/* Pointer to hash table containing a
				 * pointer to this structure.  Used to
				 * delete this datatable entry from
				 * the table. */
    Blt_HashEntry *hPtr;	/* Pointer to the hash table entry for
				 * this datatable in the
				 * interpreter-specific hash table. */

    int nextTraceId;		/* Used to generate trace id strings.  */

    Blt_HashTable traceTable;	/* Table of active traces. Maps trace ids
				 * back to their TraceInfo records. */

    int nextNotifyId;		/* Used to generate notify id strings. */

    Blt_HashTable notifyTable;	/* Table of event handlers. Maps notify ids
				 * back to their NotifyInfo records. */
} DataTableCmd;

typedef struct {
    Tcl_Obj *cmd0;
    Tcl_Interp *interp;
} CompareData;

static Blt_DataTableSortProc DataTableCompareProc;

/*
 * TraceInfo --
 *
 *	Structure containing information about a trace set from this
 *	command shell.
 *
 *	This auxillary structure houses data to be used for a callback
 *	to a Tcl procedure when a table object trace fires.  It is
 *	stored in a hash table in the DataTableCmd structure to keep track
 *	of traces issued by this shell.
 */
typedef struct {
    Blt_DataTableTrace trace;
    DataTableCmd *cmdPtr;
    Blt_HashEntry *hPtr;
    Blt_HashTable *tablePtr;
    int type;
    int cmdc;
    Tcl_Obj **cmdv;
} TraceInfo;

/*
 * NotifierInfo --
 *
 *	Structure containing information about a notifier set from
 *	this command shell.
 *
 *	This auxillary structure houses data to be used for a callback
 *	to a Tcl procedure when a table object notify event fires.  It
 *	is stored in a hash table in the DataTableCmd structure to keep
 *	track of notifiers issued by this shell.
 */
typedef struct {
    Blt_DataTableNotifier notifier;
    DataTableCmd *cmdPtr;
    Blt_HashEntry *hPtr;
    int cmdc;
    Tcl_Obj **cmdv;
} NotifierInfo;

static Blt_SwitchParseProc PositionSwitch;

#define INSERT_BEFORE	(ClientData)(1<<0)
#define INSERT_AFTER	(ClientData)(1<<1)

#define INSERT_ROW	(BLT_SWITCH_USER_BIT<<1)
#define INSERT_COLUMN	(BLT_SWITCH_USER_BIT<<2)

static Blt_SwitchCustom beforeSwitch = {
    PositionSwitch, NULL, INSERT_BEFORE,
};
static Blt_SwitchCustom afterSwitch = {
    PositionSwitch, NULL, INSERT_AFTER,
};

typedef struct {
    DataTableCmd *cmdPtr;
    long insertPos;		/* Index where to install new row or
				 * column. */

    char *label;		/* New label. */

    Tcl_Obj *tags;		/* List of tags to be applied to this
				 * row or column. */
} InsertSwitches;

static Blt_SwitchSpec insertSwitches[] = 
{
    {BLT_SWITCH_CUSTOM, "-after", Blt_Offset(InsertSwitches, insertPos), 
        INSERT_COLUMN, &afterSwitch},
    {BLT_SWITCH_CUSTOM, "-after", Blt_Offset(InsertSwitches, insertPos), 
        INSERT_ROW, &afterSwitch},
    {BLT_SWITCH_CUSTOM, "-before", Blt_Offset(InsertSwitches, insertPos), 
	INSERT_COLUMN, &beforeSwitch},
    {BLT_SWITCH_CUSTOM, "-before", Blt_Offset(InsertSwitches, insertPos), 
	INSERT_ROW, &beforeSwitch},
    {BLT_SWITCH_STRING, "-label", Blt_Offset(InsertSwitches, label), 
        INSERT_ROW | INSERT_COLUMN},
    {BLT_SWITCH_OBJ, "-tags", Blt_Offset(InsertSwitches, tags), 
	INSERT_ROW | INSERT_COLUMN},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    unsigned int flags;
} AppendData;

#define APPEND_NOTAGS	(1<<1)
#define APPEND_NOLABELS	(1<<2)
#define APPEND_FORCE	(1<<3)

static Blt_SwitchSpec appendSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-nolabels", Blt_Offset(AppendData, flags), 0, 0, 
	APPEND_NOLABELS},
    {BLT_SWITCH_FLAG, "-notags", Blt_Offset(AppendData, flags), 0, 0, 
	APPEND_NOTAGS},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    unsigned int flags;
    char *label;
} CopyData;

#define COPY_NOTAGS	(1<<1)
#define COPY_LABEL	(1<<3)

static Blt_SwitchSpec copySwitches[] = 
{
    {BLT_SWITCH_STRING, "-label", Blt_Offset(CopyData, label), 0, 0},
    {BLT_SWITCH_FLAG, "-notags", Blt_Offset(CopyData, flags), 0, 0, 
	COPY_NOTAGS},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    unsigned int flags;
} DupData;

#define DUP_NOTAGS	(1<<1)

static Blt_SwitchSpec dupSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-notags", Blt_Offset(DupData, flags), 0, 0, DUP_NOTAGS},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    Tcl_Obj *rows, *cols;
    Blt_Chain *rowChainPtr, *colChainPtr;
    Tcl_Channel channel;
    Tcl_DString *dsPtr;
    unsigned int flags;
} DumpData;

static Blt_SwitchSpec dumpSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-rows", Blt_Offset(DumpData, rows), 0, 0},
    {BLT_SWITCH_OBJ, "-columns", Blt_Offset(DumpData, cols), 0, 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    unsigned int flags;
    Blt_HashTable idTable;
} RestoreData;

static Blt_SwitchSpec restoreSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-notags", Blt_Offset(RestoreData, flags), 0, 0, 
	TABLE_RESTORE_NO_TAGS},
    {BLT_SWITCH_FLAG, "-overwrite", Blt_Offset(RestoreData, flags), 0, 0, 
	TABLE_RESTORE_OVERWRITE},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    unsigned int flags;
} SortSwitches;

#define SORT_DECREASING		(1<<0)
#define SORT_LIST		(1<<1)

static Blt_SwitchSpec sortSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-decreasing", Blt_Offset(SortSwitches, flags), 0, 0, 
	SORT_DECREASING},
    {BLT_SWITCH_FLAG, "-list", Blt_Offset(SortSwitches, flags), 0, 0, 
	SORT_LIST},
    {BLT_SWITCH_END, NULL, 0, 0}
};

typedef struct {
    unsigned int flags;
} NotifyData;

static Blt_SwitchSpec notifySwitches[] = 
{
    {BLT_SWITCH_FLAG, "-allevents", Blt_Offset(NotifyData, flags), 0, 0, 
	TABLE_NOTIFY_ALL},
    {BLT_SWITCH_FLAG, "-create", Blt_Offset(NotifyData, flags), 0, 0, 
	TABLE_NOTIFY_CREATE},
    {BLT_SWITCH_FLAG, "-delete", Blt_Offset(NotifyData, flags), 0, 0, 
	TABLE_NOTIFY_DELETE},
    {BLT_SWITCH_FLAG, "-move", Blt_Offset(NotifyData, flags), 0, 0, 
	TABLE_NOTIFY_MOVE},
    {BLT_SWITCH_FLAG, "-pack", Blt_Offset(NotifyData, flags), 0, 0, 
	TABLE_NOTIFY_PACK},
    {BLT_SWITCH_FLAG, "-relabel", Blt_Offset(NotifyData, flags), 0, 0, 
	TABLE_NOTIFY_RELABEL},
    {BLT_SWITCH_FLAG, "-sort", Blt_Offset(NotifyData, flags), 0, 0, 
	TABLE_NOTIFY_SORT},
    {BLT_SWITCH_FLAG, "-whenidle", Blt_Offset(NotifyData, flags), 0, 0, 
	TABLE_NOTIFY_WHENIDLE},
    {BLT_SWITCH_END, NULL, 0, 0}
};

/*
 * ExportCsvData --
 */
typedef struct {
    Blt_Chain *rowChainPtr;
    Blt_Chain *colChainPtr;
    Tcl_Obj *rows, *cols;	/* Selected rows and columns to export. */
    unsigned int flags;

    Tcl_Channel channel;	/* If non-NULL, channel to write output to. */
    Tcl_DString *dsPtr;
    int length;			/* Length of dynamic string. */
    int count;			/* Number of fields in current record. */
    Tcl_Interp *interp;
    char *quote;		/* Quoted string delimiter. */
    char *sep;			/* Separator character. */
} ExportCsvData;

static Blt_SwitchSpec exportCsvSwitches[] = 
{
    {BLT_SWITCH_OBJ, "-columns", Blt_Offset(ExportCsvData, cols), 0, 0},
    {BLT_SWITCH_STRING, "-quote", Blt_Offset(ExportCsvData, quote), 0, 0},
    {BLT_SWITCH_OBJ, "-rows", Blt_Offset(ExportCsvData, rows), 0, 0},
    {BLT_SWITCH_STRING, "-separator", Blt_Offset(ExportCsvData, sep), 0, 0},
    {BLT_SWITCH_END, NULL, 0, 0}
};

/*
 * ExportCsvData --
 */
typedef struct {
    unsigned int flags;
    Tcl_Channel channel;	/* If non-NULL, channel to read from. */
    char *fileName;		/* Name of file representing the channel. */
    char *buffer;		/* Buffer to read data into. */
    int nBytes;			/* # of bytes in the buffer. */
    Tcl_Interp *interp;
    char *quote;		/* Quoted string delimiter. */
    char *sep;			/* Separator character. */
} ImportCsvData;

#define INDEX_LIST	1
typedef struct {
    int flags;
} IndexSwitches;

static Blt_SwitchSpec indexSwitches[] = 
{
    {BLT_SWITCH_FLAG, "-list", Blt_Offset(IndexSwitches, flags), 0, 0,
       INDEX_LIST},
    {BLT_SWITCH_END, NULL, 0, 0}
};

static Blt_DataTableTraceProc TraceProc;
static Blt_DataTableTraceDeleteProc TraceDeleteProc;

static Blt_DataTableNotifierEventProc NotifierEventProc;
static Blt_DataTableNotifierDeleteProc NotifierDeleteProc;

static Tcl_CmdDeleteProc DataTableInstDeleteProc;
static Tcl_InterpDeleteProc DataTableInterpDeleteProc;
static Tcl_ObjCmdProc DataTableInstObjCmd;
static Tcl_ObjCmdProc DataTableObjCmd;


/*
 *----------------------------------------------------------------------
 *
 * PositionSwitch --
 *
 *	Convert a Tcl_Obj representing an offset in the table.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
PositionSwitch(
    ClientData clientData,	/* Flag indicating if the node is
				 * considered before or after the
				 * insertion position. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    char *switchName,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representation */
    char *record,		/* Structure record */
    int offset,			/* Offset to field in structure */
    int flags)			/* Indicates whether this is a row or
				 * column index. */
{
    InsertSwitches *insertPtr = (InsertSwitches *)record;
    Blt_DataTable table;

    table = insertPtr->cmdPtr->table;
    if (flags & INSERT_COLUMN) {
	long col;

	if (Blt_DataTableGetColumn(interp, table, objPtr, &col) != TCL_OK) {
	    return TCL_ERROR;
	}
	insertPtr->insertPos = Blt_DataTableColumnIndex(table, col);
    } else if (flags & INSERT_ROW) {
	long row;

	if (Blt_DataTableGetRow(interp, table, objPtr, &row) != TCL_OK) {
	    return TCL_ERROR;
	}
	insertPtr->insertPos = Blt_DataTableRowIndex(table, row);
    }
    if (clientData == INSERT_AFTER) {
	insertPtr->insertPos++;
    } 
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * GetDataTableCmdInterpData --
 *
 *---------------------------------------------------------------------- 
 */
static DataTableCmdInterpData *
GetDataTableCmdInterpData(Tcl_Interp *interp)
{
    DataTableCmdInterpData *dataPtr;
    Tcl_InterpDeleteProc *proc;

    dataPtr = (DataTableCmdInterpData *)
	Tcl_GetAssocData(interp, DATATABLE_THREAD_KEY, &proc);
    if (dataPtr == NULL) {
	dataPtr = Blt_Malloc(sizeof(DataTableCmdInterpData));
	assert(dataPtr);
	dataPtr->interp = interp;
	Tcl_SetAssocData(interp, DATATABLE_THREAD_KEY, 
		DataTableInterpDeleteProc, dataPtr);
	Blt_InitHashTable(&dataPtr->instTable, BLT_STRING_KEYS);
    }
    return dataPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * NewDataTableCmd --
 *
 *	This is a helper routine used by DataTableCreateOp.  It create a
 *	new instance of a table command.  Memory is allocated for the
 *	command structure and a new Tcl command is created (same as
 *	the instance name).  All table commands have hash table
 *	entries in a global (interpreter-specific) registry.
 *	
 * Results:
 *	Returns a pointer to the newly allocated table command structure.
 *
 * Side Effects:
 *	Memory is allocated for the structure and a hash table entry is
 *	added.  
 *
 *---------------------------------------------------------------------- 
 */
static DataTableCmd *
NewDataTableCmd(Tcl_Interp *interp, Blt_DataTable table, CONST char *name)
{
    DataTableCmd *cmdPtr;
    DataTableCmdInterpData *dataPtr;
    int isNew;

    cmdPtr = Blt_Calloc(1, sizeof(DataTableCmd));
    assert(cmdPtr);
    cmdPtr->table = table;
    cmdPtr->interp = interp;
    cmdPtr->emptyValueObjPtr = Tcl_NewStringObj("", -1);
    Tcl_IncrRefCount(cmdPtr->emptyValueObjPtr);

    Blt_InitHashTable(&cmdPtr->traceTable, BLT_STRING_KEYS);
    Blt_InitHashTable(&cmdPtr->notifyTable, BLT_STRING_KEYS);

    cmdPtr->cmdToken = Tcl_CreateObjCommand(interp, name, 
	DataTableInstObjCmd, cmdPtr, DataTableInstDeleteProc);

    dataPtr = GetDataTableCmdInterpData(interp);
    cmdPtr->tablePtr = &dataPtr->instTable;
    cmdPtr->hPtr = Blt_CreateHashEntry(&dataPtr->instTable, name, &isNew);
    Blt_SetHashValue(cmdPtr->hPtr, cmdPtr);
    return cmdPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * GenerateName --
 *
 *	Generates an unique table command name.  Table names are in
 *	the form "datatableN", where N is a non-negative integer. Check 
 *	each name generated to see if it is already a table. We want
 *	to recycle names if possible.
 *	
 * Results:
 *	Returns the unique name.  The string itself is stored in the
 *	dynamic string passed into the routine.
 *
 * ----------------------------------------------------------------------
 */
static CONST char *
GenerateName(
    Tcl_Interp *interp,
    CONST char *prefix, 
    CONST char *suffix,
    Tcl_DString *resultPtr)
{

    int n;
    CONST char *instName;

    /* 
     * Parse the command and put back so that it's in a consistent
     * format.  
     *
     *	t1         <current namespace>::t1
     *	n1::t1     <current namespace>::n1::t1
     *	::t1	   ::t1
     *  ::n1::t1   ::n1::t1
     */
    instName = NULL;		/* Suppress compiler warning. */
    for (n = 0; n < INT_MAX; n++) {
	Blt_ObjectName objName;
	Tcl_CmdInfo cmdInfo;
	Tcl_DString ds;
	char string[200];

	Tcl_DStringInit(&ds);
	Tcl_DStringAppend(&ds, prefix, -1);
	sprintf(string, "datatable%d", n);
	Tcl_DStringAppend(&ds, string, -1);
	Tcl_DStringAppend(&ds, suffix, -1);
	if (!Blt_ParseObjectName(interp, Tcl_DStringValue(&ds), 
				 &objName, 0)) {
	    return NULL;
	}
	instName = Blt_MakeQualifiedName(&objName, resultPtr);
	Tcl_DStringFree(&ds);
	/* 
	 * Check if the command already exists. 
	 */
	if (Tcl_GetCommandInfo(interp, (char *)instName, &cmdInfo)) {
	    continue;
	}
	if (!Blt_DataTableExists(interp, instName)) {
	    /* 
	     * We want the name of the table command and the underlying
	     * table object to be the same. Check that the free command
	     * name isn't an already a table object name.  
	     */
	    break;
	}
    }
    return instName;
}


/*
 *----------------------------------------------------------------------
 *
 * GetDataTableCmd --
 *
 *	Find the table command associated with the Tcl command "string".
 *	
 *	We have to perform multiple lookups to get this right.  
 *
 *	The first step is to generate a canonical command name.  If an
 *	unqualified command name (i.e. no namespace qualifier) is
 *	given, we should search first the current namespace and then
 *	the global one.  Most Tcl commands (like Tcl_GetCmdInfo) look
 *	only at the global namespace.
 *
 *	Next check if the string is 
 *		a) a Tcl command and 
 *		b) really is a command for a table object.  
 *	Tcl_GetCommandInfo will get us the objClientData field that 
 *	should be a cmdPtr.  We verify that by searching our hashtable 
 *	of cmdPtr addresses.
 *
 * Results:
 *	A pointer to the table command.  If no associated table command
 *	can be found, NULL is returned.  It's up to the calling routines
 *	to generate an error message.
 *
 *---------------------------------------------------------------------- 
 */
static DataTableCmd *
GetDataTableCmd(Tcl_Interp *interp, CONST char *name)
{
    Blt_HashEntry *hPtr;
    Tcl_DString ds;
    DataTableCmdInterpData *dataPtr;
    Blt_ObjectName objName;
    char *qualName;

    /* Put apart the table name and put is back together in a standard
     * format. */
    if (!Blt_ParseObjectName(interp, name, &objName, BLT_NO_ERROR_MSG)) {
	return NULL;		/* No such parent namespace. */
    }
    /* Rebuild the fully qualified name. */
    qualName = Blt_MakeQualifiedName(&objName, &ds);
    dataPtr = GetDataTableCmdInterpData(interp);
    hPtr = Blt_FindHashEntry(&dataPtr->instTable, qualName);
    Tcl_DStringFree(&ds);
    if (hPtr == NULL) {
	return NULL;
    }
    return Blt_GetHashValue(hPtr);
}

static char *
ColumnLabel(Blt_DataTable table, long column)
{
    char *label;

    label = (char *)Blt_DataTableColumnLabel(table, column);
    if (label == NULL) {
	return "";
    } 
    return label;
}

static char *
RowLabel(Blt_DataTable table, long row)
{
    char *label;

    label = (char *)Blt_DataTableRowLabel(table, row);
    if (label == NULL) {
	return "";
    } 
    return label;
}

/*
 *----------------------------------------------------------------------
 *
 * GetTraceFlags --
 *
 *	Parses a string representation of the trace bit flags and
 *	returns the mask.
 *
 * Results:
 *	The trace mask is returned.
 *
 *---------------------------------------------------------------------- 
 */
static int
GetTraceFlags(char *string)
{
    char *p;
    unsigned int flags;

    flags = 0;
    for (p = string; *p != '\0'; p++) {
	switch (toupper(UCHAR(*p))) {
	case 'R':
	    flags |= TABLE_TRACE_READS;		break;
	case 'W':
	    flags |= TABLE_TRACE_WRITES;	break;
	case 'U':
	    flags |= TABLE_TRACE_UNSETS;	break;
	case 'C':
	    flags |= TABLE_TRACE_CREATES;	break;
	default:
	    return -1;
	}
    }
    return flags;
}

/*
 *----------------------------------------------------------------------
 *
 * PrintTraceFlags --
 *
 *	Generates a string representation of the trace bit flags.
 *	It's assumed that the provided string is at least 5 bytes.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The bitflag information is written to the provided string.
 *
 *---------------------------------------------------------------------- 
 */
static void
PrintTraceFlags(unsigned int flags, char *string)
{
    char *p;

    p = string;
    if (flags & TABLE_TRACE_READS) {
	*p++ = 'r';
    } 
    if (flags & TABLE_TRACE_WRITES) {
	*p++ = 'w';
    } 
    if (flags & TABLE_TRACE_UNSETS) {
	*p++ = 'u';
    } 
    if (flags & TABLE_TRACE_CREATES) {
	*p++ = 'c';
    } 
    *p = '\0';
}

static void
PrintTraceInfo(
    Tcl_Interp *interp, 
    TraceInfo *tiPtr, 
    Tcl_Obj *listObjPtr)
{
    Tcl_Obj *objPtr;
    int i;
    char string[5];
    struct Blt_DataTableTraceStruct *tracePtr;
    Blt_DataTable table;

    table = tiPtr->cmdPtr->table;
    Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewStringObj("id", 2));
    Tcl_ListObjAppendElement(interp, listObjPtr, 
			     Tcl_NewStringObj(tiPtr->hPtr->key.string, -1));
    tracePtr = tiPtr->trace;
    if (tracePtr->rowTag != NULL) {
	Tcl_ListObjAppendElement(interp, listObjPtr, 
				 Tcl_NewStringObj("row", 3));
	Tcl_ListObjAppendElement(interp, listObjPtr, 
				 Tcl_NewStringObj(tracePtr->rowTag, -1));
    }
    if (tracePtr->row >= 0) {
	Tcl_ListObjAppendElement(interp, listObjPtr, 
				 Tcl_NewStringObj("row", 3));
	Tcl_ListObjAppendElement(interp, listObjPtr, 
	    Tcl_NewLongObj(Blt_DataTableRowIndex(table, tracePtr->row)));
    }
    if (tracePtr->colTag != NULL) {
	Tcl_ListObjAppendElement(interp, listObjPtr, 
				 Tcl_NewStringObj("column", 6));
	Tcl_ListObjAppendElement(interp, listObjPtr, 
				 Tcl_NewStringObj(tracePtr->colTag, -1));
    }
    if (tracePtr->column >= 0) {
	Tcl_ListObjAppendElement(interp, listObjPtr, 
				 Tcl_NewStringObj("column", 6));
	Tcl_ListObjAppendElement(interp, listObjPtr, 
	    Tcl_NewLongObj(Blt_DataTableColumnIndex(table, tracePtr->column)));
    }
    PrintTraceFlags(tracePtr->flags, string);
    Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewStringObj("flags", 5));
    Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewStringObj(string, -1));

    Tcl_ListObjAppendElement(interp, listObjPtr, 
			     Tcl_NewStringObj("command", 7));
    objPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (i = 0; i < tiPtr->cmdc; i++) {
	Tcl_ListObjAppendElement(interp, objPtr, tiPtr->cmdv[i]);
    }
    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * FreeNotifierInfo --
 *
 *	This is a helper routine used to delete notifiers.  It
 *	releases the Tcl_Objs used in the notification callback
 *	command and the actual table notifier.  Memory for the
 *	notifier is also freed.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Memory is deallocated and the notitifer is no longer 
 *	active.
 *
 *---------------------------------------------------------------------- 
 */
static void
FreeNotifierInfo(NotifierInfo *niPtr)
{
    int i;

    for (i = 0; i < niPtr->cmdc; i++) {
	Tcl_DecrRefCount(niPtr->cmdv[i]);
    }
    Blt_DataTableDeleteNotifier(niPtr->notifier);
    Blt_Free(niPtr->cmdv);
    Blt_Free(niPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * TraceProc --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TraceProc(
    ClientData clientData,
    Tcl_Interp *interp,
    long row, long column,	/* Index of cell that's updated. */
    unsigned int flags)
{
    TraceInfo *tracePtr = clientData; 
    char string[5];
    int result;
    int i;

    i = tracePtr->cmdc;		/* Add extra command arguments
				 * starting at this index. */
    tracePtr->cmdv[i+1] = Tcl_NewLongObj(row);
    tracePtr->cmdv[i+2] = Tcl_NewLongObj(column);
    PrintTraceFlags(flags, string);
    tracePtr->cmdv[i+3] = Tcl_NewStringObj(string, -1);

    Tcl_IncrRefCount(tracePtr->cmdv[i+1]);
    Tcl_IncrRefCount(tracePtr->cmdv[i+2]);
    Tcl_IncrRefCount(tracePtr->cmdv[i+3]);
    result = Tcl_EvalObjv(interp, tracePtr->cmdc + 4, tracePtr->cmdv, 0);
    Tcl_DecrRefCount(tracePtr->cmdv[i+3]);
    Tcl_DecrRefCount(tracePtr->cmdv[i+2]);
    Tcl_DecrRefCount(tracePtr->cmdv[i+1]);
    if (result != TCL_OK) {
	Tcl_BackgroundError(interp);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceDeleteProc --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static void
TraceDeleteProc(ClientData clientData)
{
    TraceInfo *tiPtr = clientData; 
    int i;

    for (i = 0; i <= tiPtr->cmdc; i++) {
	Tcl_DecrRefCount(tiPtr->cmdv[i]);
    }
    Blt_Free(tiPtr->cmdv);
    if (tiPtr->hPtr != NULL) {
	Blt_DeleteHashEntry(tiPtr->tablePtr, tiPtr->hPtr);
    }
    Blt_Free(tiPtr);
}

static char *
GetEventName(type)
{
    switch (type & TABLE_NOTIFY_EVENT_MASK) {
    case TABLE_NOTIFY_CREATE:
	return "-create";
    case TABLE_NOTIFY_DELETE:
	return "-delete";
    case TABLE_NOTIFY_RELABEL:
	return "-relabel";
    case TABLE_NOTIFY_SORT:
	return "-sort";
    case TABLE_NOTIFY_MOVE:
	return "-move";
    case TABLE_NOTIFY_PACK:
	return "-pack";
    }	
    return "???";
}

/*
 *----------------------------------------------------------------------
 *
 * NotifierDeleteProc --
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static void
NotifierDeleteProc(ClientData clientData)
{
    NotifierInfo *niPtr; 

    niPtr = clientData; 
    FreeNotifierInfo(niPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * NotifierEventProc --
 *
 *---------------------------------------------------------------------- 
 */
static int
NotifierEventProc(ClientData clientData, Blt_DataTableNotifierEvent *eventPtr)
{
    NotifierInfo *niPtr; 
    Tcl_Interp *interp;
    int i, result;
    long index;

    niPtr = clientData; 
    interp = niPtr->cmdPtr->interp;
    i = niPtr->cmdc;
    niPtr->cmdv[i] = Tcl_NewStringObj(GetEventName(eventPtr->type), -1);
    if (eventPtr->type & TABLE_NOTIFY_ROW) {
	index = Blt_DataTableRowIndex(niPtr->cmdPtr->table, eventPtr->offset);
    } else {
	index = Blt_DataTableColumnIndex(niPtr->cmdPtr->table, 
					 eventPtr->offset);
    }	
    niPtr->cmdv[i+1] = Tcl_NewLongObj(index);
    Tcl_IncrRefCount(niPtr->cmdv[i]);
    Tcl_IncrRefCount(niPtr->cmdv[i+1]);
    Tcl_IncrRefCount(niPtr->cmdv[i+2]);
    result = Tcl_EvalObjv(interp, niPtr->cmdc + 3, niPtr->cmdv, 0);
    Tcl_DecrRefCount(niPtr->cmdv[i+2]);
    Tcl_DecrRefCount(niPtr->cmdv[i+1]);
    Tcl_DecrRefCount(niPtr->cmdv[i]);
    if (result != TCL_OK) {
	Tcl_BackgroundError(interp);
	return TCL_ERROR;
    }
    Tcl_ResetResult(interp);
    return TCL_OK;
}

static int
DataTableCompareProc(
    ClientData clientData,
    Tcl_Obj *valueObjPtr1, 
    Tcl_Obj *valueObjPtr2)
{
    Tcl_Obj *objv[3];
    CompareData *dataPtr;
    int result;

    if (valueObjPtr1 == NULL) {
	valueObjPtr1 = Tcl_NewStringObj("", 0);
    }
    if (valueObjPtr2 == NULL) {
	valueObjPtr2 = Tcl_NewStringObj("", 0);
    }
    dataPtr = clientData;
    objv[0] = dataPtr->cmd0;
    objv[1] = valueObjPtr1;
    objv[2] = valueObjPtr2;
    Tcl_IncrRefCount(objv[1]);
    Tcl_IncrRefCount(objv[2]);
    result = Tcl_EvalObjv(dataPtr->interp, 3, objv, 0);
    Tcl_DecrRefCount(objv[1]);
    Tcl_DecrRefCount(objv[2]);
    if (result == TCL_ERROR) {
	Tcl_BackgroundError(dataPtr->interp);
	return 0;
    }
    if (Tcl_GetIntFromObj(dataPtr->interp, Tcl_GetObjResult(dataPtr->interp),
			  &result) != TCL_OK) {
	Tcl_BackgroundError(dataPtr->interp);
	return 0;
    }
    if ((result > 1) || (result < -1)) {
	Tcl_AppendResult(dataPtr->interp, 
			 "invalid result from sort procedure \"", 
			 Tcl_GetString(objv[0]), "\"", (char *)NULL);
	Tcl_BackgroundError(dataPtr->interp);
	return 0;
    }
    return result;
}

static char *
CopyOfColumnLabel(Blt_DataTable table, char *label, Tcl_DString *dsPtr)
{
    int i;
    int oldLen;

    oldLen = strlen(label) + 2;
    Tcl_DStringSetLength(dsPtr, oldLen);
    sprintf(Tcl_DStringValue(dsPtr), "%s #", label);
    for (i = 2; i < INT_MAX; i++) {
	Tcl_DStringAppend(dsPtr, Blt_Itoa(i), -1);
	if (Blt_DataTableFindColumn(table, Tcl_DStringValue(dsPtr)) < 0) {
	    return Tcl_DStringValue(dsPtr);
	}
	Tcl_DStringSetLength(dsPtr, oldLen);
    }
    return NULL;
}

static char *
CopyOfRowLabel(Blt_DataTable table, char *label, Tcl_DString *dsPtr)
{
    int i;
    int oldLen;

    oldLen = strlen(label) + 2;
    Tcl_DStringSetLength(dsPtr, oldLen);
    sprintf(Tcl_DStringValue(dsPtr), "%s #", label);
    for (i = 2; i < INT_MAX; i++) {
	Tcl_DStringAppend(dsPtr, Blt_Itoa(i), -1);
	if (Blt_DataTableFindRow(table, Tcl_DStringValue(dsPtr)) < 0) {
	    return Tcl_DStringValue(dsPtr);
	}
	Tcl_DStringSetLength(dsPtr, oldLen);
    }
    return NULL;
}

static int
GetColumns(
    Tcl_Interp *interp,
    Blt_DataTable table, 
    int objc, 
    Tcl_Obj *CONST *objv, 
    Blt_Chain *chainPtr)
{
    int i;
    Blt_HashTable colTable;

    Blt_InitHashTable(&colTable, BLT_ONE_WORD_KEYS);
    /* Collect the column offsets into a table. */
    for (i = 0; i < objc; i++) {
	Blt_DataTableIterator colIter;
	long col;

	if (Blt_DataTableGetColumns(interp, table, objv[i], &colIter) !=TCL_OK){
	    Blt_DeleteHashTable(&colTable);
	    return TCL_ERROR;
	}
	for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
	     col = Blt_DataTableNextColumn(&colIter)) {
	    int isNew;

	    Blt_CreateHashEntry(&colTable, (char *)col, &isNew);
	    if (isNew) {
		Blt_ChainAppend(chainPtr, (ClientData)col);
	    }
	}
    }
    Blt_DeleteHashTable(&colTable);
    return TCL_OK;
}

static Blt_Chain *
AllColumns(Blt_DataTable table)
{
    long iCol, nCols;
    Blt_Chain *chainPtr;
    
    chainPtr = Blt_ChainCreate();
    nCols = Blt_DataTableNumColumns(table);
    for (iCol = 1; iCol <= nCols; iCol++) {
	long col;

	col = Blt_DataTableColumnOffset(table, iCol);
	Blt_ChainAppend(chainPtr, (ClientData)col);
			
    }
    return chainPtr;
}

static int
GetRows(
    Tcl_Interp *interp,
    Blt_DataTable table, 
    int objc, 
    Tcl_Obj *CONST *objv, 
    Blt_Chain *chainPtr)
{
    int i;
    Blt_HashTable rows;

    Blt_InitHashTable(&rows, BLT_ONE_WORD_KEYS);
    /* Collect the row offsets into a table. */
    for (i = 0; i < objc; i++) {
	Blt_DataTableIterator rowIter;
	long row;

	if (Blt_DataTableGetRows(interp, table, objv[i], &rowIter) != TCL_OK) {
	    Blt_DeleteHashTable(&rows);
	    return TCL_ERROR;
	}
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    int isNew;

	    Blt_CreateHashEntry(&rows, (char *)row, &isNew);
	    if (isNew) {
		Blt_ChainAppend(chainPtr, (ClientData)row);
	    }
	}
    }
    Blt_DeleteHashTable(&rows);
    return TCL_OK;
}

static Blt_Chain *
AllRows(Blt_DataTable table)
{
    long ri, nRows;
    Blt_Chain *chainPtr;

    chainPtr = Blt_ChainCreate();
    nRows = Blt_DataTableNumRows(table);
    for (ri = 1; ri <= nRows; ri++) {
	long row;

	row = 	Blt_DataTableRowOffset(table, ri);
	Blt_ChainAppend(chainPtr, (ClientData)row);
    }
    return chainPtr;
}

static int
CopyColumn(
    Tcl_Interp *interp,
    Blt_DataTable src,
    Blt_DataTable dest,
    long srcCol,		/* Column offset in the source table. */
    long destCol)		/* Column offset in the destination. */
{
    long nSrcRows, nDestRows;
    long iRow;

    if ((Blt_DataTableSameTableObject(src, dest)) && (srcCol == destCol)) {
	return TCL_OK;		/* Source and destination are the same. */
    }
    nSrcRows = Blt_DataTableNumRows(src);
    nDestRows = Blt_DataTableNumRows(dest);
    if (nDestRows < nSrcRows) {
	if (Blt_DataTableExtendRows(interp, dest, nSrcRows - nDestRows, NULL)
	    != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    for (iRow = 0; iRow < nSrcRows; iRow++) {
	Tcl_Obj *objPtr;
	long row;

	row = Blt_DataTableRowIndex(src, iRow);
	if (Blt_DataTableGetValue(interp, src, row, srcCol, &objPtr)!=TCL_OK) {
	    return TCL_ERROR;
	}
	row = Blt_DataTableRowIndex(dest, iRow);
	if (Blt_DataTableSetValue(interp, dest, row, destCol, objPtr)!=TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}	    

static void
CopyColumnTags(
    Blt_DataTable src,
    Blt_DataTable dest,
    long srcCol,		/* Column offset in the source table. */
    long destCol)		/* Column offset in the destination. */
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    /* Find all tags for with this column index. */
    for (hPtr = Blt_DataTableFirstColumnTag(src, &cursor); hPtr != NULL; 
	 hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_HashTable *tablePtr;
	Blt_HashEntry *h2Ptr;
	
	tablePtr = Blt_GetHashValue(hPtr);
	h2Ptr = Blt_FindHashEntry(tablePtr, (char *)srcCol);
	if (h2Ptr != NULL) {
	    /* We know the tag tables are keyed by strings, so we
	     * don't need to call Blt_GetHashKey or the hash table
	     * pointer to retrieve the key. */
	    Blt_DataTableSetColumnTag(NULL, dest, destCol, hPtr->key.string);
	}
    }
}	    


/************* Column Operations ****************/

/*
 *----------------------------------------------------------------------
 *
 * ColumnAppendOp --
 *
 *	Appends the specified columns from another table.
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$dest column append $table $col $col ?switches?
 *	$dest column append $table ?switches?
 *	$dest column append $table 1:20 -notags -nolabels
 *	$dest column append 1:20 -notags -nolabels
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnAppendOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    AppendData switches;
    Blt_DataTable src, dest;
    Tcl_Obj *listObjPtr;
    Blt_ChainLink *linkPtr;
    Blt_Chain *chainPtr;
    long i;
    int result;

    /* Get the source table.  It may also be the destination.  */
    dest = cmdPtr->table;
    result = TCL_ERROR;
    if (Blt_DataTableGetToken(interp, Tcl_GetString(objv[3]), &src) == TCL_OK) {
	objv++, objc--;
    } else {
	src = dest;
	Tcl_ResetResult(interp);
    }
    /* Count the number of column arguments. */
    for (i = 3; i < objc; i++) {
	CONST char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    if (i == 3) {
	chainPtr = AllColumns(dest);
    } else {
	chainPtr = Blt_ChainCreate();
	if (GetColumns(interp, src, i - 3, objv + 3, chainPtr) != TCL_OK) {
	    goto error;
	}
    }
    /* Process switches following the column names. */
    switches.flags = 0;
    if (Blt_ParseSwitches(interp, appendSwitches, objc - i, objv + i, 
	(char *)&switches, BLT_SWITCH_DEFAULTS) < 0) {
	goto error;
    }
    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	char *label;
	long n;
	long srcCol, destCol;

	srcCol = (long)Blt_ChainGetValue(linkPtr);
	label = (char *)Blt_DataTableColumnLabel(src, srcCol);
	result = Blt_DataTableExtendColumns(interp, dest, 1, &destCol);
	if (result != TCL_OK) {
	    goto error;
	}
	result = CopyColumn(interp, src, dest, srcCol, destCol);
	if (result != TCL_OK) {
	    goto error;
	}
	if ((label != NULL) && ((switches.flags & APPEND_NOLABELS) == 0)) {
	    Tcl_DString ds;
	    
	    Tcl_DStringInit(&ds);
	    if (Blt_DataTableFindColumn(dest, label) >= 0) {
		label = CopyOfColumnLabel(dest, label, &ds);
	    }
	    result = Blt_DataTableSetColumnLabel(interp, dest, destCol, label);
	    Tcl_DStringFree(&ds);
	    if (result != TCL_OK) {
		goto error;
	    }
	}
	if ((switches.flags & APPEND_NOTAGS) == 0) {
	    CopyColumnTags(src, dest, srcCol, destCol);
	}
	n = Blt_DataTableColumnIndex(dest, destCol);
	Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewLongObj(n));
    }
 error:
    Blt_ChainDestroy(chainPtr);
    if (result != TCL_OK) {
	Tcl_DecrRefCount(listObjPtr);
    } else {
	Tcl_SetObjResult(interp, listObjPtr);
    }
    Blt_FreeSwitches(copySwitches, (char *)&switches, 0);
    if (src != dest) {
	Blt_DataTableReleaseToken(src);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnCopyOp --
 *
 *	Copies the specified columns to the table.  A different table
 *	may be selected as the source.
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$dest column copy ?$table? $from $to ?switches?
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnCopyOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    CopyData switches;
    Blt_DataTable src, dest;
    long i;
    int result;
    long srcCol, destCol;
    char *label;

    result = TCL_ERROR;
    dest = cmdPtr->table;
    /* Get the source table.  It may also be the destination.  */
    if (Blt_DataTableGetToken(interp, Tcl_GetString(objv[3]), &src) == TCL_OK) {
	objv++, objc--;
    } else {
	src = dest;
	Tcl_ResetResult(interp);
    }
    /* Count the number of column arguments. */
    for (i = 3; i < objc; i++) {
	CONST char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
    }
    if (i < 5) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", 
		Tcl_GetString(objv[0]), 
		" column copy ?table? from to ?switches?", (char *)NULL);
	return TCL_ERROR;
    }
    if (Blt_DataTableGetColumn(interp, src, objv[3], &srcCol) != TCL_OK) {
	goto error;
    }
    if (Blt_DataTableGetColumn(interp, dest, objv[4], &destCol) != TCL_OK) {
	goto error;
    }
    /* Process switches following the column names. */
    switches.flags = 0;
    if (Blt_ParseSwitches(interp, copySwitches, objc - i, objv + i, 
	(char *)&switches, BLT_SWITCH_DEFAULTS) < 0) {
	goto error;
    }
    if (CopyColumn(interp, src, dest, srcCol, destCol) != TCL_OK) {
	goto error;
    }
    label = switches.label;
    if (label == NULL) {
	Tcl_DString ds;

	label = (char *)Blt_DataTableColumnLabel(src, srcCol);
	Tcl_DStringInit(&ds);
	if (Blt_DataTableFindColumn(dest, label) != destCol) {
	    label = CopyOfColumnLabel(dest, label, &ds);
	}
	result = Blt_DataTableSetColumnLabel(interp, dest, destCol, label);
	Tcl_DStringFree(&ds);
    } else {
	result = Blt_DataTableSetColumnLabel(interp, dest, destCol, label);
    }
    if (result != TCL_OK) {
	goto error;
    }
    if ((switches.flags & COPY_NOTAGS) == 0) {
	CopyColumnTags(src, dest, srcCol, destCol);
    }
 error:
    Blt_FreeSwitches(copySwitches, (char *)&switches, 0);
    if (src != dest) {
	Blt_DataTableReleaseToken(src);
    }
    return result;
    
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnDeleteOp --
 *
 *	Deletes the columns designated.  One or more columns may be
 *	deleted using a tag.  
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t column delete ?column?...
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnDeleteOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;
    Blt_DataTable table;

    table = cmdPtr->table;
    chainPtr = Blt_ChainCreate();
    if (GetColumns(interp, table, objc - 3, objv + 3, chainPtr) != TCL_OK) {
	Blt_ChainDestroy(chainPtr);
	return TCL_ERROR;
    }
    /* 
     * Walk through the list of column offsets, deleting each column.
     */
    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	long col;

	col = (long)Blt_ChainGetValue(linkPtr);
	if (Blt_DataTableDeleteColumn(interp, table, col) != TCL_OK) {
	    Blt_ChainDestroy(chainPtr);
	    return TCL_ERROR;
	}
    }
    Blt_ChainDestroy(chainPtr);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * ColumnDupOp --
 *
 *	Duplicates the specified columns in the table.  This differs
 *	from ColumnCopyOp, since the same table is the source and
 *	destination.
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$dest column dup label ?label?... ?switches?
 *	$dest column dup column... ?switches?
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnDupOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    DupData switches;
    Blt_DataTable table;
    Tcl_Obj *listObjPtr;
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;
    long i;

    table = cmdPtr->table;
    /* Count the number of column arguments. */
    for (i = 3; i < objc; i++) {
	CONST char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
    }
    chainPtr = Blt_ChainCreate();
    if (GetColumns(interp, table, i - 3, objv + (i + 3), chainPtr) != TCL_OK) {
	Blt_ChainDestroy(chainPtr);
	return TCL_ERROR;
    }
    /* Process switches following the column names. */
    switches.flags = 0;
    if (Blt_ParseSwitches(interp, dupSwitches, objc - i, objv + i, 
	(char *)&switches, BLT_SWITCH_DEFAULTS) < 0) {
	Blt_ChainDestroy(chainPtr);
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	char *label;
	int result;
	long n;
	long src, dest;

	src = (long)Blt_ChainGetValue(linkPtr);
	if (Blt_DataTableExtendColumns(interp, table, 1, &dest) != TCL_OK) {
	    goto error;
	}
	if (CopyColumn(interp, table, table, src, dest)!= TCL_OK) {
	    goto error;
	}
	label = (char *)Blt_DataTableColumnLabel(table, src);
	if (label != NULL) {
	    Tcl_DString ds;

	    Tcl_DStringInit(&ds);
	    label = CopyOfColumnLabel(table, label, &ds);
	    result = Blt_DataTableSetColumnLabel(interp, table, dest, label);
	    Tcl_DStringFree(&ds);
	    if (result != TCL_OK) {
		goto error;
	    }
	}
	if ((switches.flags & DUP_NOTAGS) == 0) {
	    CopyColumnTags(table, table, src, dest);
	}
	n = Blt_DataTableColumnIndex(table, dest);
	Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewLongObj(n));
    }
    Blt_ChainDestroy(chainPtr);
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
 error:
    Blt_ChainDestroy(chainPtr);
    Tcl_DecrRefCount(listObjPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnExistsOp --
 *
 *	Indicates is the given column exists.  The column description
 *	can be either an index, label, or single tag.  
 *
 *	Problem: The Blt_DataTableGetColumns function checks both for
 *		 1) valid/invalid indices, labels, and tags and 2) 
 *		 syntax errors.
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t column exists n
 *	
 *---------------------------------------------------------------------- 
 */
static int
ColumnExistsOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int result;
    int bool;
    long col;

    table = cmdPtr->table;
    result = Blt_DataTableGetColumn(NULL, table, objv[3], &col);
    bool = (result == TCL_OK);
    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), bool);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnExtendOp --
 *
 *	Extends the table by the given number of columns.
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t column extend n 
 *	
 *---------------------------------------------------------------------- 
 */
static int
ColumnExtendOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Tcl_Obj *listObjPtr;
    long *cols;
    long i, n;

    table = cmdPtr->table;
    if (Tcl_GetLongFromObj(interp, objv[3], &n) != TCL_OK) {
	return TCL_ERROR;	
    }	    
    if (n == 0) {
	return TCL_OK;
    }
    if (n < 0) {
	Tcl_AppendResult(interp, "bad count \"", Blt_Itoa(n), 
		"\": can't be negative.", (char *)NULL);
	return TCL_ERROR;
    }
    cols = Blt_Malloc(sizeof(long) * n);
    if (Blt_DataTableExtendColumns(interp, table, n, cols) != TCL_OK) {
	Blt_Free(cols);
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (i = 0; i < n; i++) {
	Tcl_Obj *objPtr;

	objPtr = Tcl_NewLongObj(Blt_DataTableColumnIndex(table, cols[i]));
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    Blt_Free(cols);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnGetOp --
 *
 *	Retrieves a column of values.  The column argument can be
 *	either a tag, label, or column index.  If it is a tag, it must
 *	refer to exactly one column.
 * 
 * Results:
 *	A standard Tcl result.  If successful, a list of values is
 *	returned in the interpreter result.  If the column index is
 *	invalid, TCL_ERROR is returned and an error message is left in
 *	the interpreter result.
 *	
 *	$t column get column
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnGetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Tcl_Obj *listObjPtr;
    long col;
    long iRow;
    long nRows;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumn(interp, table, objv[3], &col) != TCL_OK) {
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    nRows = Blt_DataTableNumRows(table);
    for (iRow = 1; iRow <= nRows; iRow++) {
	Tcl_Obj *objPtr;
	long row;

	row = Blt_DataTableRowOffset(table, iRow);
	if (Blt_DataTableGetValue(interp, table, row, col, &objPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (objPtr == NULL) {
	    if (objc == 5) {
		objPtr = objv[4];
	    } else {
		objPtr = cmdPtr->emptyValueObjPtr;
	    }
	}
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnIndexOp --
 *
 *	Returns the column index of the given column tag, label, or 
 *	index.  A tag can't represent more than one column.
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t column index $column
 *	
 *---------------------------------------------------------------------- 
 */
static int
ColumnIndexOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    IndexSwitches switches;

    /* Process switches following the column names. */
    switches.flags = 0;
    if (Blt_ParseSwitches(interp, indexSwitches, objc - 4, objv + 4, 
	(char *)&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    table = cmdPtr->table;
    if (switches.flags & INDEX_LIST) {
	Blt_DataTableIterator colIter;
	Tcl_Obj *listObjPtr;
	long col;

	if (Blt_DataTableGetColumns(NULL, table, objv[3], &colIter) != TCL_OK) {
	    return TCL_OK;
	}
	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
	     col = Blt_DataTableNextColumn(&colIter)) {
	    Tcl_Obj *objPtr;
	    
	    objPtr = Tcl_NewLongObj(Blt_DataTableColumnIndex(table, col));
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	Tcl_SetObjResult(interp, listObjPtr);
    } else {
	long col, iCol;

	iCol = -1;
	if (Blt_DataTableGetColumn(NULL, table, objv[3], &col) == TCL_OK) {
	    iCol = Blt_DataTableColumnIndex(table, col);
	}
	Tcl_SetLongObj(Tcl_GetObjResult(interp), iCol);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnInsertOp --
 *
 *	Inserts a single new column into the table.  The location of
 *	the new column may be specified by -before or -after switches.
 *	By default, the new column is appended.
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t column insert -before 0 -after 1 -label label
 *	
 *---------------------------------------------------------------------- 
 */
static int
ColumnInsertOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    InsertSwitches switches;
    long col;
    unsigned int flags;

    table = cmdPtr->table;
    switches.insertPos = Blt_DataTableNumColumns(table) + 1;
    switches.label = NULL;
    switches.tags = NULL;
    switches.cmdPtr = cmdPtr;

    flags = INSERT_COLUMN;
    if (Blt_ParseSwitches(interp, insertSwitches, objc - 3, objv + 3, 
		&switches, flags) < 0) {
	goto error;
    }
    if (Blt_DataTableExtendColumns(interp, table, 1, &col) != TCL_OK) {
	goto error;
    }
    if (switches.label != NULL) {
	if (Blt_DataTableSetColumnLabel(interp, table, col, switches.label) 
	    != TCL_OK) {
	    goto error;
	}
    }
    if (Blt_DataTableMoveColumns(interp, table, 
		Blt_DataTableColumnIndex(table, col), 
		switches.insertPos, 1) != TCL_OK) {
	goto error;
    }
    if (switches.tags != NULL) {
	Tcl_Obj **elv;
	int elc;
	int i;

	if (Tcl_ListObjGetElements(interp, switches.tags, &elc, &elv) 
	    != TCL_OK) {
	    goto error;
	}
	for (i = 0; i < elc; i++) {
	    if (Blt_DataTableSetColumnTag(interp, table, col, 
			Tcl_GetString(elv[i])) != TCL_OK) {
		goto error;
	    }
	}
    }
    Tcl_SetObjResult(interp, 
	Tcl_NewLongObj(Blt_DataTableColumnIndex(table, col)));
    Blt_FreeSwitches(insertSwitches, (char *)&switches, flags);
    return TCL_OK;
 error:
    Blt_FreeSwitches(insertSwitches, (char *)&switches, flags);
    return TCL_ERROR;
}


/*
 *----------------------------------------------------------------------
 *
 * ColumnLabelOp --
 *
 *	Sets a label for a column.  
 * 
 * Results:
 *	A standard Tcl result.  If successful, the old column label is
 *	returned in the interpreter result.  If the column index is
 *	invalid, TCL_ERROR is returned and an error message is left in
 *	the interpreter result.
 *	
 *	$t column label column ?newLabel? 
 *	$t column label 1
 *	$t column label 1 newLabel
 *	$t column label 1:5 { c1 c2 c3 c4 c5 }
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnLabelOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;

    table = cmdPtr->table;
    chainPtr = Blt_ChainCreate();
    if (GetColumns(interp, table, 1, objv + 3, chainPtr)!= TCL_OK) {
	Blt_ChainDestroy(chainPtr);
	return TCL_ERROR;
    }
    if (objc == 4) {
	linkPtr = Blt_ChainFirstLink(chainPtr);
	if (Blt_ChainGetLength(chainPtr) == 1) {
	    Tcl_Obj *objPtr;
	    long col;
	    
	    col = (long)Blt_ChainGetValue(linkPtr);
	    objPtr = Tcl_NewStringObj(ColumnLabel(table, col), -1);
	    Tcl_SetObjResult(interp, objPtr);
	} else {
	    Tcl_Obj *listObjPtr;

	    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	    for (/*empty*/; linkPtr != NULL; 
			  linkPtr = Blt_ChainNextLink(linkPtr)) {
		Tcl_Obj *objPtr;
		long col;
		
		col = (long)Blt_ChainGetValue(linkPtr);
		objPtr = Tcl_NewStringObj(ColumnLabel(table, col), -1);
		Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	    }
	    Tcl_SetObjResult(interp, listObjPtr);
	}
    } else if (objc == 5) {
	long nCols, i, col;
	int elc;
	Tcl_Obj **elv;

	nCols = Blt_ChainGetLength(chainPtr);
	if (nCols == 1) {
	    linkPtr = Blt_ChainFirstLink(chainPtr);
	    col = (long)Blt_ChainGetValue(linkPtr);
	    if (Blt_DataTableSetColumnLabel(interp, table, col, 
			Tcl_GetString(objv[4])) != TCL_OK) {
		goto error;
	    } 
	}
	if (Tcl_ListObjGetElements(interp, objv[4], &elc, &elv) != TCL_OK) {
	    goto error;
	}
	if (elc != nCols) {
	    Tcl_AppendResult(interp, "# of labels does not match # of columns",
			     (char *)NULL);
	    goto error;
	}
	for (i = 0, linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	     linkPtr = Blt_ChainNextLink(linkPtr), i++) {
	    long col;
	    
	    col = (long)Blt_ChainGetValue(linkPtr);
	    if (Blt_DataTableSetColumnLabel(interp, table, col, 
				Tcl_GetString(elv[i])) != TCL_OK) {
		goto error;
	    }
	}
    }
    Blt_ChainDestroy(chainPtr);
    return TCL_OK;
 error:
    Blt_ChainDestroy(chainPtr);
    return TCL_ERROR;

}

/*
 *----------------------------------------------------------------------
 *
 * ColumnLengthOp --
 *
 *	Returns the number of columns the client sees.
 * 
 * Results:
 *	A standard Tcl result.  If successful, the old column label is
 *	returned in the interpreter result.  If the column index is
 *	invalid, TCL_ERROR is returned and an error message is left in
 *	the interpreter result.
 *	
 *	$t column label column ?newLabel?
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnLengthOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Tcl_SetLongObj(Tcl_GetObjResult(interp),
		   Blt_DataTableNumColumns(cmdPtr->table));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnMoveOp --
 *
 *	Moves the given number of columns to another location
 *	in the table.
 * 
 * Results:
 *	A standard Tcl result. If the column index is invalid,
 *	TCL_ERROR is returned and an error message is left in the
 *	interpreter result.
 *	
 *	$t column move from to ?n?
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnMoveOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    long from, to, count;
    Blt_DataTable table;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumn(interp, table, objv[3], &from) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetColumn(interp, table, objv[4], &to) != TCL_OK) {
	return TCL_ERROR;
    }
    count = 1;
    if (objc == 6) {
	if (Tcl_GetLongFromObj(interp, objv[5], &count) != TCL_OK) {
	    return TCL_ERROR;

	}
	if (count == 0) {
	    return TCL_OK;
	}
	if (count < 0) {
	    Tcl_AppendResult(interp, "# of columns can't be negative",
			     (char *)NULL);
	    return TCL_ERROR;
	}
    }
    return Blt_DataTableMoveColumns(interp, table, 
		Blt_DataTableColumnIndex(table, from),
		Blt_DataTableColumnIndex(table, to), count);
}


/*
 *----------------------------------------------------------------------
 *
 * ColumnNamesOp --
 *
 *	Reports the labels of all columns.  
 * 
 * Results:
 *	Always returns TCL_OK.  The interpreter result is a list of
 *	column labels.
 *	
 *	$t column names pattern 
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnNamesOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Tcl_Obj *listObjPtr;
    long iCol, nCols;

    table = cmdPtr->table;
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    nCols = Blt_DataTableNumColumns(table);
    for (iCol = 1; iCol <= nCols; iCol++) {
	long col;
	char *label;
	int match;
	int i;

	col = Blt_DataTableColumnOffset(table, iCol);
	label = ColumnLabel(table, col);
	match = (objc == 3);
	for (i = 3; i < objc; i++) {
	    char *pattern;

	    pattern = Tcl_GetString(objv[i]);
	    if (Tcl_StringMatch(label, pattern)) {
		match = TRUE;
		break;
	    }
	}
	if (match) {
	    Tcl_Obj *objPtr;

	    objPtr = Tcl_NewStringObj(label, -1);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnNotifyOp --
 *
 *	Creates a notifier for this instance.  Notifiers represent
 *	a bitmask of events and a command prefix to be invoked 
 *	when a matching event occurs.  
 *
 *	The command prefix is parsed and saved in an array of
 *	Tcl_Objs. Extra slots are allocated for the 
 *
 * Results:
 *	A standard Tcl result.  The name of the new notifier is
 *	returned in the interpreter result.  Otherwise, if it failed
 *	to parse a switch, then TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * Example:
 *	table0 column notify col ?flags? command arg
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnNotifyOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator colIter;
    NotifierInfo *niPtr;
    NotifyData switches;
    char *tag;
    int count, i;
    int nArgs;
    long column;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumns(interp, table, objv[3], &colIter) != TCL_OK) {
	return TCL_ERROR;
    }
    column = -1;
    tag = NULL;
    if (colIter.type == TABLE_ITER_RANGE) {
	Tcl_AppendResult(interp, "can't notify range of columns: use a tag", 
		(char *)NULL);
	return TCL_ERROR;
    }
    if (colIter.type == TABLE_ITER_INDEX) {
	column = Blt_DataTableFirstColumn(&colIter);
    } else {
	tag = (char *)colIter.tagName;
    }
    count = 0;
    for (i = 4; i < objc; i++) {
	char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] != '-') {
	    break;
	}
	count++;
    }
    switches.flags = 0;
    /* Process switches  */
    if (Blt_ParseSwitches(interp, notifySwitches, count, objv + 4, 
	     (char *)&switches, 0) < 0) {
	return TCL_ERROR;
    }
    niPtr = Blt_Malloc(sizeof(NotifierInfo));
    niPtr->cmdPtr = cmdPtr;
    niPtr->notifier = Blt_DataTableColumnNotifier(interp, table, switches.flags,
	column, tag, NotifierEventProc, NotifierDeleteProc,niPtr);
    nArgs = (objc - i) + 2;
    /* Stash away the command in structure and pass that to the notifier. */
    niPtr->cmdv = Blt_Malloc(nArgs * sizeof(Tcl_Obj *));
    for (count = 0; i < objc; i++, count++) {
	Tcl_IncrRefCount(objv[i]);
	niPtr->cmdv[count] = objv[i];
    }
    niPtr->cmdc = nArgs;
    if (switches.flags == 0) {
	switches.flags = TABLE_NOTIFY_ALL;
    }
    {
	char notifyId[200];
	Blt_HashEntry *hPtr;
	int isNew;

	sprintf(notifyId, "notify%d", cmdPtr->nextNotifyId++);
	hPtr = Blt_CreateHashEntry(&cmdPtr->notifyTable, notifyId, &isNew);
	assert(hPtr != NULL);
	assert(isNew);
	Blt_SetHashValue(hPtr, niPtr);
	Tcl_SetStringObj(Tcl_GetObjResult(interp), notifyId, -1);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnSetOp --
 *
 *	Sets a column of values.  One or more columns may be set using a
 *	tag.  The row order is always the table's current
 *	view of the table.  There may be less values than needed.
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *	$t column set column valueList
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnSetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator colIter;
    Tcl_Obj **elv;
    int elc;
    long n;
    long col;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumns(interp, table, objv[3], &colIter) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Tcl_ListObjGetElements(interp, objv[4], &elc, &elv) != TCL_OK) {
	return TCL_ERROR;
    }
    n = elc - Blt_DataTableNumRows(table);
    if (n > 0) {
	if (Blt_DataTableExtendRows(interp, table, n, NULL) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
	 col = Blt_DataTableNextColumn(&colIter)) {
	int i;

	for (i = 0; i < elc; i++) {
	    long row;

	    row = Blt_DataTableRowOffset(table, i + 1);
	    if (Blt_DataTableSetValue(interp, table, row, col, elv[i]) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnTagAddOp --
 *
 *	Adds one or more tags for a given column.  Tag names can't
 *	start with a digit (to distinquish them from node ids) and
 *	can't be a reserved tag ("all" or "end").
 *
 *	.t column tag add $column tag1 tag2...
 *	.t column tag range $from $to tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnTagAddOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator colIter;
    int i;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumns(interp, table, objv[4], &colIter) != TCL_OK) {
	return TCL_ERROR;
    }
    for (i = 5; i < objc; i++) {
	CONST char *tagName;
	long col;

	tagName = Tcl_GetString(objv[i]);
	for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
	     col = Blt_DataTableNextColumn(&colIter)) {
	    if (Blt_DataTableSetColumnTag(interp, table, col, tagName) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}    
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * ColumnTagDeleteOp --
 *
 *	Removes one or more tags from a given column. If a tag doesn't
 *	exist or is a reserved tag ("all" or "end"), nothing will be
 *	done and no error message will be returned.
 *
 *	.t column tag delete $column tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnTagDeleteOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator colIter;
    int i;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumns(interp, table, objv[4], &colIter) != TCL_OK) {
	return TCL_ERROR;
    }
    for (i = 5; i < objc; i++) {
	CONST char *tagName;
	long col;

	tagName = Tcl_GetString(objv[i]);
	for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
	     col = Blt_DataTableNextColumn(&colIter)) {
	    if (Blt_DataTableUnsetColumnTag(interp, table, col, tagName) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }    
    return TCL_OK;
}
/*
 *----------------------------------------------------------------------
 *
 * ColumnTagForgetOp --
 *
 *	Removes the given tags from all nodes.
 *
 *	$t column tag forget tag1 tag2 tag3...
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
ColumnTagForgetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,		/* Not used. */
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 4; i < objc; i++) {
	if (Blt_DataTableForgetColumnTag(interp, cmdPtr->table, 
		Tcl_GetString(objv[i])) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnTagIndicesOp --
 *
 *	Returns column indices names for the given tags.  If one of
 *	more tag names are provided, then only those matching indices
 *	are returned.
 *
 *	.t column tag indices tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnTagIndicesOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;
    long *matches;
    long nCols;

    table = cmdPtr->table;
    nCols = Blt_DataTableNumColumns(table);
    matches = Blt_Calloc(nCols + 1, sizeof(long));
    assert(matches);

    /* Handle the reserved tags "all" or "end". */
    for (i = 4; i < objc; i++) {
	char *tagName;
	long n;

	tagName = Tcl_GetString(objv[i]);
	if (strcmp("all", tagName) == 0) {
	    for (n = 1; n <= nCols; n++) {
		matches[n] = TRUE;
	    }
	    goto done;		/* Don't care other tags. */
	} 
	if (strcmp("end", tagName) == 0) {
	    matches[nCols] = TRUE;
	}
    }
    /* Now check user-defined tags. */
    for (i = 4; i < objc; i++) {
	Blt_HashEntry *hPtr;
	Blt_HashSearch cursor;
	Blt_HashTable *tagTablePtr;
	CONST char *tagName;
	
	tagName = Tcl_GetString(objv[i]);
	if ((strcmp("all", tagName) == 0) || (strcmp("end", tagName) == 0)) {
	    continue;
	}
	tagTablePtr = Blt_DataTableColumnTagTable(table, tagName);
	if (tagTablePtr == NULL) {
	    Tcl_AppendResult(interp, "unknown column tag \"", tagName, "\"",
		(char *)NULL);
	    Blt_Free(matches);
	    return TCL_ERROR;
	}
	for (hPtr = Blt_FirstHashEntry(tagTablePtr, &cursor); hPtr != NULL; 
	     hPtr = Blt_NextHashEntry(&cursor)) {
	    long col, n;

	    col = (long)Blt_GetHashValue(hPtr);
	    n = Blt_DataTableColumnIndex(table, col);
	    assert(n >= 0);
	    matches[n] = TRUE;
	}
    }

 done:
    {
	Tcl_Obj *listObjPtr;
	long n;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (n = 1; n <= nCols; n++) {
	    if (matches[n]) {
		Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewLongObj(n));
	    }
	}
	Tcl_SetObjResult(interp, listObjPtr);
    }
    Blt_Free(matches);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * ColumnTagRangeOp --
 *
 *	Adds one or more tags for a given column.  Tag names can't
 *	start with a digit (to distinquish them from node ids) and
 *	can't be a reserved tag ("all" or "end").
 *
 *	.t column tag range $from $to tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnTagRangeOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;
    long from, to, offset;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumn(interp, table, objv[4], &offset) != TCL_OK) {
	return TCL_ERROR;
    }
    from = Blt_DataTableColumnIndex(table, offset);
    if (Blt_DataTableGetColumn(interp, table, objv[5], &offset) != TCL_OK) {
	return TCL_ERROR;
    }
    to = Blt_DataTableColumnIndex(table, offset);
    if (from > to) {
	long tmp;
	tmp = to, to = from, from = tmp;
    }
    for (i = 6; i < objc; i++) {
	CONST char *tagName;
	long iCol;
	
	tagName = Tcl_GetString(objv[i]);
	for (iCol = from; iCol <= to; iCol++) {
	    long col;

	    col = Blt_DataTableColumnOffset(table, iCol);
	    assert(col >= 0);
	    if (Blt_DataTableSetColumnTag(interp, table, col, tagName) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}    
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * ColumnTagSearchOp --
 *
 *	Returns tag names for a given  column.  If one of more pattern
 *	arguments are  provided,   then only those  matching  tags are
 *	returned.
 *
 *	.t column tag find $column pat1 pat2...
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnTagSearchOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    Blt_DataTableIterator colIter;
    Tcl_Obj *listObjPtr;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumns(interp, table, objv[4], &colIter) != TCL_OK) {
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (hPtr = Blt_DataTableFirstColumnTag(table, &cursor); hPtr != NULL; 
	hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_HashTable *tablePtr;
	long col;

	tablePtr = Blt_GetHashValue(hPtr);
	for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
	     col = Blt_DataTableNextColumn(&colIter)) {
	    Blt_HashEntry *h2Ptr;
	
	    h2Ptr = Blt_FindHashEntry(tablePtr, (char *)col);
	    if (h2Ptr != NULL) {
		CONST char *tagName;
		int match;
		int i;

		match = (objc == 5);
		tagName = hPtr->key.string;
		for (i = 5; i < objc; i++) {
		    if (Tcl_StringMatch(tagName, Tcl_GetString(objv[i]))) {
			match = TRUE;
			break;	/* Found match. */
		    }
		}
		if (match) {
		    Tcl_Obj *objPtr;

		    objPtr = Tcl_NewStringObj(tagName, -1);
		    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
		    break;	/* Tag matches this column. Don't care
				 * if it matches any other columns. */
		}
	    }
	}
    }

    /* Handle reserved tags specially. */
    {
	int i;
	int allMatch, endMatch;

	endMatch = allMatch = (objc == 5);
	for (i = 5; i < objc; i++) {
	    if (Tcl_StringMatch("all", Tcl_GetString(objv[i]))) {
		allMatch = TRUE;
	    }
	    if (Tcl_StringMatch("end", Tcl_GetString(objv[i]))) {
		endMatch = TRUE;
	    }
	}
	if (allMatch) {
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewStringObj("all", 3));
	}
	if (endMatch) {
	    long col, lastCol, iCol;
	    
	    iCol = Blt_DataTableNumColumns(table);
	    lastCol = Blt_DataTableColumnOffset(table, iCol);
	    for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
		 col = Blt_DataTableNextColumn(&colIter)) {
		if (col == lastCol) {
		    Tcl_ListObjAppendElement(interp, listObjPtr,
			Tcl_NewStringObj("end", 3));
		    break;
		}
	    }
	}
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnTagOp --
 *
 * 	This procedure is invoked to process tag operations.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec columnTagOps[] =
{
    {"add",     1, (Blt_Op)ColumnTagAddOp,     5, 0, "column ?tag...?",},
    {"delete",  1, (Blt_Op)ColumnTagDeleteOp,  5, 0, "column ?tag...?",},
    {"forget",  1, (Blt_Op)ColumnTagForgetOp,  4, 0, "?tag...?",},
    {"indices", 1, (Blt_Op)ColumnTagIndicesOp, 4, 0, "?tag...?",},
    {"range",   1, (Blt_Op)ColumnTagRangeOp,   6, 0, "from to ?tag...?",},
    {"search",  1, (Blt_Op)ColumnTagSearchOp,  5, 6, "column ?pattern?",},
};

static int nColumnTagOps = sizeof(columnTagOps) / sizeof(Blt_OpSpec);

static int
ColumnTagOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nColumnTagOps, columnTagOps, BLT_OP_ARG3, 
	objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc)(cmdPtr, interp, objc, objv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnTraceOp --
 *
 *	Creates a trace for this instance.  Traces represent
 *	list of keys, a bitmask of trace flags, and a command prefix 
 *	to be invoked when a matching trace event occurs.  
 *
 *	The command prefix is parsed and saved in an array of
 *	Tcl_Objs. The qualified name of the instance is saved
 *	also.
 *
 * Results:
 *	A standard Tcl result.  The name of the new trace is
 *	returned in the interpreter result.  Otherwise, if it failed
 *	to parse a switch, then TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * $t column trace tag rwx proc 
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
ColumnTraceOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator colIter;
    Blt_DataTableTrace trace;
    TraceInfo *tiPtr;
    char *tag;
    int flags;
    long col;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumns(interp, table, objv[3], &colIter) != TCL_OK) {
	return TCL_ERROR;
    }
    flags = GetTraceFlags(Tcl_GetString(objv[4]));
    if (flags < 0) {
	Tcl_AppendResult(interp, "unknown flag in \"", Tcl_GetString(objv[4]), 
		"\"", (char *)NULL);
	return TCL_ERROR;
    }
    col = -1;
    tag = NULL;
    if (colIter.type == TABLE_ITER_RANGE) {
	Tcl_AppendResult(interp, "can't trace range of columns: use a tag", 
		(char *)NULL);
	return TCL_ERROR;
    }
    if (colIter.type == TABLE_ITER_INDEX) {
	col = Blt_DataTableFirstColumn(&colIter);
    } else {
	tag = (char *)colIter.tagName;
    } 
    tiPtr = Blt_Malloc(sizeof(TraceInfo));
    if (tiPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate trace: out of memory", 
		(char *)NULL);
	return TCL_ERROR;
    }
    trace = Blt_DataTableCreateTrace(table, -1, col, NULL, tag, flags, 
	TraceProc, TraceDeleteProc, tiPtr);
    if (trace == NULL) {
	Tcl_AppendResult(interp, "can't create column trace: out of memory", 
		(char *)NULL);
	Blt_Free(tiPtr);
	return TCL_ERROR;
    }
    /* Initialize the trace information structure. */
    tiPtr->cmdPtr = cmdPtr;
    tiPtr->trace = trace;
    tiPtr->tablePtr = &cmdPtr->traceTable;
    {
	Tcl_Obj **elv, **cmdv;
	int elc, i;

	if (Tcl_ListObjGetElements(interp, objv[5], &elc, &elv) != TCL_OK) {
	    return TCL_ERROR;
	}
	cmdv = Blt_Calloc(elc + 1 + 3 + 1, sizeof(Tcl_Obj *));
	for(i = 0; i < elc; i++) {
	    cmdv[i] = elv[i];
	    Tcl_IncrRefCount(cmdv[i]);
	}
	cmdv[i] = Tcl_NewStringObj(cmdPtr->hPtr->key.string, -1);
	Tcl_IncrRefCount(cmdv[i]);
	tiPtr->cmdc = elc;
	tiPtr->cmdv = cmdv;
    }
    {
	char traceId[200];
	int isNew;

	sprintf(traceId, "trace%d", cmdPtr->nextTraceId++);
	tiPtr->hPtr = Blt_CreateHashEntry(&cmdPtr->traceTable, traceId, &isNew);
	Blt_SetHashValue(tiPtr->hPtr, tiPtr);
	Tcl_SetStringObj(Tcl_GetObjResult(interp), traceId, -1);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnTypeOp --
 *
 *	Reports and/or sets the type of a column.  
 * 
 * Results:
 *	A standard Tcl result.  If successful, the old column label is
 *	returned in the interpreter result.  If the column index is
 *	invalid, TCL_ERROR is returned and an error message is left in
 *	the interpreter result.
 *	
 *	$t column type column ?newType?
 *
 *---------------------------------------------------------------------- 
 */
static int
ColumnTypeOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator colIter;
    Tcl_Obj *listObjPtr;
    long col;

    table = cmdPtr->table;
    if (Blt_DataTableGetColumns(interp, table, objv[3], &colIter) != TCL_OK) {
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
	 col = Blt_DataTableNextColumn(&colIter)) {
	int type;
	Tcl_Obj *objPtr;

	type = Blt_DataTableColumnType(table, col);
	if (objc == 5) {
	    int newType;

	    newType = Blt_DataTableParseColumnType(Tcl_GetString(objv[4]));
	    if (newType == TABLE_COLUMN_TYPE_UNKNOWN) {
		return TCL_ERROR;
	    }
	    if (Blt_DataTableSetColumnType(interp, table, col, newType) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}
	objPtr = Tcl_NewStringObj(Blt_DataTableNameOfColumnType(type), -1);
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnUnsetOp --
 *
 *	$t column unset column ?column?
 *
 *	Unsets one or more columns of values.  One or more columns may be
 *	unset (using tags or multiple arguments).
 * 
 * Results:
 *	A standard Tcl result. If the tag or column index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *---------------------------------------------------------------------- 
 */
static int
ColumnUnsetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;
    long nRows;

    table = cmdPtr->table;
    nRows = Blt_DataTableNumRows(table);
    for (i = 3; i < objc; i++) {
	Blt_DataTableIterator colIter;
	long col;

	if (Blt_DataTableGetColumns(interp, table, objv[i], &colIter)!=TCL_OK) {
	    return TCL_ERROR;
	}
	for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
	     col = Blt_DataTableNextColumn(&colIter)) {
	    long iRow;

	    for (iRow = 1; iRow <= nRows; iRow++) {
		long row;

		row = Blt_DataTableRowOffset(table, iRow);
		if (Blt_DataTableUnsetValue(interp, table, row, col)!=TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ColumnOp --
 *
 *	Parses the given command line and calls one of several
 *	column-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec columnOps[] =
{
    {"append", 1, (Blt_Op)ColumnAppendOp, 3, 0, 
	"?table? ?column...? ?switches?",},
    {"copy",   1, (Blt_Op)ColumnCopyOp,   4, 0, "?table? from to ?switches?",},
    {"delete", 2, (Blt_Op)ColumnDeleteOp, 4, 0, "column...",},
    {"dup",    2, (Blt_Op)ColumnDupOp,    3, 0, "column... ?switches?",},
    {"exists", 3, (Blt_Op)ColumnExistsOp, 4, 4, "column",},
    {"extend", 3, (Blt_Op)ColumnExtendOp, 4, 4, "count",},
    {"get",    1, (Blt_Op)ColumnGetOp,    4, 5, "column ?defValue?",},
    {"index",  2, (Blt_Op)ColumnIndexOp,  4, 0, "column ?switches?",},
    {"insert", 2, (Blt_Op)ColumnInsertOp, 3, 0, "?switches?",},
    {"label",  2, (Blt_Op)ColumnLabelOp,  4, 5, "column ?label?",},
    {"length", 2, (Blt_Op)ColumnLengthOp, 3, 3, "",},
    {"move",   1, (Blt_Op)ColumnMoveOp,   5, 6, "from to ?count?",},
    {"names",  2, (Blt_Op)ColumnNamesOp,  3, 0, "?pattern...?",},
    {"notify", 2, (Blt_Op)ColumnNotifyOp, 5, 0, "column ?flags? command",},
    {"set",    1, (Blt_Op)ColumnSetOp,    5, 5, "column valueList",},
    {"tag",    2, (Blt_Op)ColumnTagOp,    3, 0, "op args",},
    {"trace",  2, (Blt_Op)ColumnTraceOp,  6, 6, "column how command",},
    {"type",   2, (Blt_Op)ColumnTypeOp,   4, 5, "column ?type?",},
    {"unset",  1, (Blt_Op)ColumnUnsetOp,  4, 0, "column...",},
};

static int nColumnOps = sizeof(columnOps) / sizeof(Blt_OpSpec);

static int
ColumnOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nColumnOps, columnOps, BLT_OP_ARG2, objc, 
	objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

/************ Row Operations ***************/


static int
CopyRow(
    Tcl_Interp *interp,
    Blt_DataTable src,
    Blt_DataTable dest,
    long srcRow,		/* Row offset in the source table. */
    long destRow)		/* Row offset in the destination. */
{
    long srcLen, destLen;
    long iCol;

    if ((Blt_DataTableSameTableObject(src, dest)) && (srcRow == destRow)) {
	return TCL_OK;		/* Source and destination are the same. */
    }
    srcLen = Blt_DataTableNumColumns(src);
    destLen = Blt_DataTableNumColumns(dest);
    if (destLen < srcLen) {
	if (Blt_DataTableExtendColumns(interp, dest, srcLen - destLen, NULL) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    for (iCol = 1; iCol <= srcLen; iCol++) {
	Tcl_Obj *objPtr;
	long srcCol, destCol;

	srcCol = Blt_DataTableColumnOffset(src, iCol);
	destCol = Blt_DataTableColumnOffset(dest, iCol);
	if (Blt_DataTableGetValue(interp, src, srcRow, srcCol, &objPtr) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Blt_DataTableSetValue(interp, dest, destRow, destCol, objPtr)
	    != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}	    

static void
CopyRowTags(
    Blt_DataTable src,
    Blt_DataTable dest,
    long srcRow,		/* Row offset in the source table. */
    long destRow)		/* Row offset in the destination. */
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    /* Find all tags for with this row index. */
    for (hPtr = Blt_DataTableFirstRowTag(src, &cursor); hPtr != NULL; 
	 hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_HashTable *tablePtr;
	Blt_HashEntry *h2Ptr;
	
	tablePtr = Blt_GetHashValue(hPtr);
	h2Ptr = Blt_FindHashEntry(tablePtr, (char *)srcRow);
	if (h2Ptr != NULL) {
	    /* We know the tag tables are keyed by strings, so we
	     * don't need to call Blt_GetHashKey or the hash table
	     * pointer to retrieve the key. */
	    Blt_DataTableSetRowTag(NULL, dest, destRow, hPtr->key.string);
	}
    }
}	    



/*
 *----------------------------------------------------------------------
 *
 * RowAppendOp --
 *
 *	Appends the specified rows from another table.
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$dest row append $table $col $col ?switches?
 *	$dest row append $table ?switches?
 *	$dest row append $table 1:20 -notags -nolabels
 *	$dest row append 1:20 -notags -nolabels
 *
 *---------------------------------------------------------------------- 
 */
static int
RowAppendOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    AppendData switches;
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;
    Blt_DataTable src, dest;
    Tcl_Obj *listObjPtr;
    int result;
    long i;

    /* Get the source table.  It may also be the destination.  */
    dest = cmdPtr->table;
    result = TCL_ERROR;
    if (Blt_DataTableGetToken(interp, Tcl_GetString(objv[3]), &src) == TCL_OK) {
	objv++, objc--;
    } else {
	src = dest;
	Tcl_ResetResult(interp);
    }
    /* Count the number of row arguments. */
    for (i = 3; i < objc; i++) {
	CONST char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    if (i == 3) {
	chainPtr = AllRows(dest);
    } else {
	chainPtr = Blt_ChainCreate();
	if (GetRows(interp, src, i - 3, objv + 3, chainPtr) != TCL_OK) {
	    goto error;
	}
    }
    /* Process switches following the row names. */
    switches.flags = 0;
    if (Blt_ParseSwitches(interp, appendSwitches, objc - i, objv + i, 
	(char *)&switches, BLT_SWITCH_DEFAULTS) < 0) {
	goto error;
    }
    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	char *label;
	long n;
	long srcCol, destCol;

	srcCol = (long)Blt_ChainGetValue(linkPtr);
	label = (char *)Blt_DataTableRowLabel(src, srcCol);
	result = Blt_DataTableExtendRows(interp, dest, 1, &destCol);
	if (result != TCL_OK) {
	    goto error;
	}
	result = CopyRow(interp, src, dest, srcCol, destCol);
	if (result != TCL_OK) {
	    goto error;
	}
	if ((label != NULL) && ((switches.flags & APPEND_NOLABELS) == 0)) {
	    Tcl_DString ds;
	    
	    Tcl_DStringInit(&ds);
	    if (Blt_DataTableFindRow(dest, label) >= 0) {
		label = CopyOfRowLabel(dest, label, &ds);
	    }
	    result = Blt_DataTableSetRowLabel(interp, dest, destCol, label);
	    Tcl_DStringFree(&ds);
	    if (result != TCL_OK) {
		goto error;
	    }
	}
	if ((switches.flags & APPEND_NOTAGS) == 0) {
	    CopyRowTags(src, dest, srcCol, destCol);
	}
	n = Blt_DataTableRowIndex(dest, destCol);
	Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewLongObj(n));
    }
 error:
    Blt_ChainDestroy(chainPtr);
    if (result != TCL_OK) {
	Tcl_DecrRefCount(listObjPtr);
    } else {
	Tcl_SetObjResult(interp, listObjPtr);
    }
    Blt_FreeSwitches(copySwitches, (char *)&switches, 0);
    if (src != dest) {
	Blt_DataTableReleaseToken(src);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * RowCopyOp --
 *
 *	Copies the specified rows to the table.  A different table
 *	may be selected as the source.
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$dest row copy ?$table? $from $to ?switches?
 *
 *---------------------------------------------------------------------- 
 */
static int
RowCopyOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    CopyData switches;
    Blt_DataTable src, dest;
    long i;
    int result;
    long srcCol, destCol;
    char *label;

    result = TCL_ERROR;
    dest = cmdPtr->table;
    /* Get the source table.  It may also be the destination.  */
    if (Blt_DataTableGetToken(interp, Tcl_GetString(objv[3]), &src) == TCL_OK) {
	objv++, objc--;
    } else {
	src = dest;
	Tcl_ResetResult(interp);
    }
    /* Count the number of row arguments. */
    for (i = 3; i < objc; i++) {
	CONST char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
    }
    if (i < 5) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", 
		Tcl_GetString(objv[0]), 
		" row copy ?table? from to ?switches?", (char *)NULL);
	return TCL_ERROR;
    }
    if (Blt_DataTableGetRow(interp, src, objv[3], &srcCol) != TCL_OK) {
	goto error;
    }
    if (Blt_DataTableGetRow(interp, dest, objv[4], &destCol) != TCL_OK) {
	goto error;
    }
    /* Process switches following the row names. */
    switches.flags = 0;
    if (Blt_ParseSwitches(interp, copySwitches, objc - i, objv + i, 
	(char *)&switches, BLT_SWITCH_DEFAULTS) < 0) {
	goto error;
    }
    if (CopyRow(interp, src, dest, srcCol, destCol) != TCL_OK) {
	goto error;
    }
    label = switches.label;
    if (label == NULL) {
	Tcl_DString ds;

	label = (char *)Blt_DataTableRowLabel(src, srcCol);
	Tcl_DStringInit(&ds);
	if (Blt_DataTableFindRow(dest, label) != destCol) {
	    label = CopyOfRowLabel(dest, label, &ds);
	}
	result = Blt_DataTableSetRowLabel(interp, dest, destCol, label);
	Tcl_DStringFree(&ds);
    } else {
	result = Blt_DataTableSetRowLabel(interp, dest, destCol, label);
    }
    if (result != TCL_OK) {
	goto error;
    }
    if ((switches.flags & COPY_NOTAGS) == 0) {
	CopyRowTags(src, dest, srcCol, destCol);
    }
 error:
    Blt_FreeSwitches(copySwitches, (char *)&switches, 0);
    if (src != dest) {
	Blt_DataTableReleaseToken(src);
    }
    return result;
    
}

/*
 *----------------------------------------------------------------------
 *
 * RowDeleteOp --
 *
 *	Deletes the rows designated.  One or more rows may be
 *	deleted using a tag.  
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t row delete ?row?...
 *
 *---------------------------------------------------------------------- 
 */
static int
RowDeleteOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;
    Blt_DataTable table;

    table = cmdPtr->table;
    chainPtr = Blt_ChainCreate();
    if (GetRows(interp, table, objc - 3, objv + 3, chainPtr) != TCL_OK) {
	goto error;
    }
    /* 
     * Walk through the list of row offsets, deleting each row.
     */
    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	long row;

	row = (long)Blt_ChainGetValue(linkPtr);
	if (Blt_DataTableDeleteRow(interp, table, row) != TCL_OK) {
	    goto error;
	}
    }
    Blt_ChainDestroy(chainPtr);
    return TCL_OK;
 error:
    Blt_ChainDestroy(chainPtr);
    return TCL_ERROR;
}


/*
 *----------------------------------------------------------------------
 *
 * RowDupOp --
 *
 *	Duplicates the specified rows in the table.  This differs from
 *	RowCopyOp, since the same table is always the source and
 *	destination.
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$dest row dup label ?label?... ?switches?
 *	$dest row dup row... ?switches?
 *
 *---------------------------------------------------------------------- 
 */
static int
RowDupOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    DupData switches;
    Blt_DataTable table;
    Tcl_Obj *listObjPtr;
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;
    long i;

    table = cmdPtr->table;
    /* Count the number of row arguments. */
    for (i = 3; i < objc; i++) {
	CONST char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
    }
    chainPtr = Blt_ChainCreate();
    if (GetRows(interp, table, i - 3, objv + (i + 3), chainPtr) != TCL_OK) {
	Blt_ChainDestroy(chainPtr);
	return TCL_ERROR;
    }
    /* Process switches following the row names. */
    switches.flags = 0;
    if (Blt_ParseSwitches(interp, dupSwitches, objc - i, objv + i, 
	(char *)&switches, BLT_SWITCH_DEFAULTS) < 0) {
	Blt_ChainDestroy(chainPtr);
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	char *label;
	int result;
	long n;
	long src, dest;

	src = (long)Blt_ChainGetValue(linkPtr);
	if (Blt_DataTableExtendRows(interp, table, 1, &dest) != TCL_OK) {
	    goto error;
	}
	if (CopyRow(interp, table, table, src, dest)!= TCL_OK) {
	    goto error;
	}
	label = (char *)Blt_DataTableRowLabel(table, src);
	if (label != NULL) {
	    Tcl_DString ds;

	    Tcl_DStringInit(&ds);
	    label = CopyOfRowLabel(table, label, &ds);
	    result = Blt_DataTableSetRowLabel(interp, table, dest, label);
	    Tcl_DStringFree(&ds);
	    if (result != TCL_OK) {
		goto error;
	    }
	}
	if ((switches.flags & DUP_NOTAGS) == 0) {
	    CopyRowTags(table, table, src, dest);
	}
	n = Blt_DataTableRowIndex(table, dest);
	Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewLongObj(n));
    }
    Blt_ChainDestroy(chainPtr);
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
 error:
    Blt_ChainDestroy(chainPtr);
    Tcl_DecrRefCount(listObjPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * RowExistsOp --
 *
 *	Indicates is the given row exists.
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t row exists n
 *	
 *---------------------------------------------------------------------- 
 */
static int
RowExistsOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int result;
    int bool;
    long row;

    table = cmdPtr->table;
    result = Blt_DataTableGetRow(NULL, table, objv[3], &row);
    bool = (result == TCL_OK);
    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), bool);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowExtendOp --
 *
 *	Extends the table by the given number of rows.
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t row extend n
 *	
 *---------------------------------------------------------------------- 
 */
static int
RowExtendOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Tcl_Obj *listObjPtr;
    long *rows;
    long i, n;

    table = cmdPtr->table;
    if (Tcl_GetLongFromObj(interp, objv[3], &n) != TCL_OK) {
	return TCL_ERROR;	
    }	    
    if (n == 0) {
	return TCL_OK;
    }
    if (n < 0) {
	Tcl_AppendResult(interp, "bad count \"", Blt_Itoa(n), 
		"\": can't be negative.", (char *)NULL);
	return TCL_ERROR;
    }
    rows = Blt_Malloc(sizeof(long) * n);
    if (Blt_DataTableExtendRows(interp, table, n, rows) != TCL_OK) {
	Blt_Free(rows);
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (i = 0; i < n; i++) {
	Tcl_Obj *objPtr;

	objPtr = Tcl_NewLongObj(Blt_DataTableRowIndex(table, rows[i]));
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    Blt_Free(rows);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowGetOp --
 *
 *	Retrieves a row of values.  The row argument can be
 *	either a tag, label, or row index.  If it is a tag, it must
 *	refer to exactly one row.
 * 
 * Results:
 *	A standard Tcl result.  If successful, a list of values is
 *	returned in the interpreter result.  If the row index is
 *	invalid, TCL_ERROR is returned and an error message is left in
 *	the interpreter result.
 *	
 *	$t row get row
 *
 *---------------------------------------------------------------------- 
 */
static int
RowGetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Tcl_Obj *listObjPtr;
    long iCol;
    long nCols;
    long row;

    table = cmdPtr->table;
    if (Blt_DataTableGetRow(interp, table, objv[3], &row) != TCL_OK) {
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    nCols = Blt_DataTableNumColumns(table);
    for (iCol = 1; iCol <= nCols; iCol++) {
	Tcl_Obj *objPtr;
	long col;

	col = Blt_DataTableColumnOffset(table, iCol);
	if (Blt_DataTableGetValue(interp, table, row, col, &objPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (objPtr == NULL) {
	    if (objc == 5) {
		objPtr = objv[4];
	    } else {
		objPtr = cmdPtr->emptyValueObjPtr;
	    }
	}
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowIndexOp --
 *
 *	Returns the row index of the given row tag, label, or 
 *	index.  A tag can't represent more than one row.
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t row index $row
 *	
 *---------------------------------------------------------------------- 
 */
static int
RowIndexOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    IndexSwitches switches;
    
    /* Process switches following the column names. */
    switches.flags = 0;
    if (Blt_ParseSwitches(interp, indexSwitches, objc - 4, objv + 4, 
	(char *)&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    table = cmdPtr->table;
    if (switches.flags & INDEX_LIST) {
	Blt_DataTableIterator rowIter;
	Tcl_Obj *listObjPtr;
	long row;

	if (Blt_DataTableGetRows(NULL, table, objv[3], &rowIter) != TCL_OK) {
	    return TCL_OK;
	}
	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    Tcl_Obj *objPtr;
	    
	    objPtr = Tcl_NewLongObj(Blt_DataTableRowIndex(table, row));
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	Tcl_SetObjResult(interp, listObjPtr);
    } else {
	long row, iRow;

	iRow = -1;
	if (Blt_DataTableGetRow(NULL, table, objv[3], &row) == TCL_OK) {
	    iRow = Blt_DataTableRowIndex(table, row);
	}
	Tcl_SetLongObj(Tcl_GetObjResult(interp), iRow);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowInsertOp --
 *
 *	Inserts a single new row into the table.  The location of
 *	the new row may be specified by -before or -after switches.
 *	By default, the new row is appended.
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t row insert -before 0 -after 1 -label label
 *	
 *---------------------------------------------------------------------- 
 */
static int
RowInsertOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    InsertSwitches switches;
    long row;
    unsigned int flags;

    table = cmdPtr->table;
    switches.insertPos = Blt_DataTableNumRows(table) + 1;
    switches.label = NULL;
    switches.tags = NULL;
    switches.cmdPtr = cmdPtr;

    flags = INSERT_ROW;
    if (Blt_ParseSwitches(interp, insertSwitches, objc - 3, objv + 3, 
		&switches, flags) < 0) {
	goto error;
    }
    if (Blt_DataTableExtendRows(interp, table, 1, &row) != TCL_OK) {
	goto error;
    }
    if (switches.label != NULL) {
	if (Blt_DataTableSetRowLabel(interp, table, row, switches.label) 
	    != TCL_OK) {
	    goto error;
	}
    }
    if (Blt_DataTableMoveRows(interp, table, 
	Blt_DataTableRowIndex(table, row), switches.insertPos, 1) 
	!= TCL_OK) {
	goto error;
    }
    if (switches.tags != NULL) {
	Tcl_Obj **elv;
	int elc;
	int i;

	if (Tcl_ListObjGetElements(interp, switches.tags, &elc, &elv) 
	    != TCL_OK) {
	    goto error;
	}
	for (i = 0; i < elc; i++) {
	    if (Blt_DataTableSetRowTag(interp, table, row, 
			Tcl_GetString(elv[i])) != TCL_OK) {
		goto error;
	    }
	}
    }
    Tcl_SetObjResult(interp, 
	Tcl_NewLongObj(Blt_DataTableRowIndex(table, row)));
    Blt_FreeSwitches(insertSwitches, (char *)&switches, flags);
    return TCL_OK;
 error:
    Blt_FreeSwitches(insertSwitches, (char *)&switches, flags);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * RowLabelOp --
 *
 *	Sets a label for a row.  
 * 
 * Results:
 *	A standard Tcl result.  If successful, the old row label is
 *	returned in the interpreter result.  If the row index is
 *	invalid, TCL_ERROR is returned and an error message is left in
 *	the interpreter result.
 *	
 *	$t row label row ?newLabel?
 *
 *---------------------------------------------------------------------- 
 */
static int
RowLabelOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;

    table = cmdPtr->table;
    chainPtr = Blt_ChainCreate();
    if (GetRows(interp, table, 1, objv + 3, chainPtr) != TCL_OK) {
	Blt_ChainDestroy(chainPtr);
	return TCL_ERROR;
    }
    if (objc == 4) {
	Tcl_Obj *listObjPtr;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	     linkPtr = Blt_ChainNextLink(linkPtr)) {
	    Tcl_Obj *objPtr;
	    long row;
	    
	    row = (long)Blt_ChainGetValue(linkPtr);
	    objPtr = Tcl_NewStringObj(RowLabel(table, row), -1);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	Tcl_SetObjResult(interp, listObjPtr);
    } else if (objc == 5) {
	long nRows, i, row;
	int elc;
	Tcl_Obj **elv;

	nRows = Blt_ChainGetLength(chainPtr);
	if (nRows == 1) {
	    linkPtr = Blt_ChainFirstLink(chainPtr);
	    row = (long)Blt_ChainGetValue(linkPtr);
	    if (Blt_DataTableSetRowLabel(interp, table, row, 
			Tcl_GetString(objv[4])) != TCL_OK) {
		goto error;
	    } 
	}
	if (Tcl_ListObjGetElements(interp, objv[4], &elc, &elv) != TCL_OK) {
	    goto error;
	}
	if (elc != nRows) {
	    Tcl_AppendResult(interp, "# of labels does not match # of rows",
			     (char *)NULL);
	    goto error;
	}
	for (i = 0, linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	     linkPtr = Blt_ChainNextLink(linkPtr), i++) {
	    long row;
	    
	    row = (long)Blt_ChainGetValue(linkPtr);
	    if (Blt_DataTableSetRowLabel(interp, table, row, 
				Tcl_GetString(elv[i])) != TCL_OK) {
		goto error;
	    }
	}
    }
    Blt_ChainDestroy(chainPtr);
    return TCL_OK;
 error:
    Blt_ChainDestroy(chainPtr);
    return TCL_ERROR;

}

/*
 *----------------------------------------------------------------------
 *
 * RowLengthOp --
 *
 *	Returns the number of rows the client sees.
 * 
 * Results:
 *	A standard Tcl result.  If successful, the old row label is
 *	returned in the interpreter result.  If the row index is
 *	invalid, TCL_ERROR is returned and an error message is left in
 *	the interpreter result.
 *	
 *	$t row label row ?newLabel?
 *
 *---------------------------------------------------------------------- 
 */
static int
RowLengthOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;

    table = cmdPtr->table;
    Tcl_SetLongObj(Tcl_GetObjResult(interp), Blt_DataTableNumRows(table));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowMoveOp --
 *
 *	Moves the given number of rows to another location
 *	in the table.
 * 
 * Results:
 *	A standard Tcl result.  If successful, a list of values is
 *	returned in the interpreter result.  If the row index is
 *	invalid, TCL_ERROR is returned and an error message is left in
 *	the interpreter result.
 *	
 *	$t row move from to ?n?
 *
 *---------------------------------------------------------------------- 
 */
static int
RowMoveOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    long from, to, count;

    table = cmdPtr->table;
    if (Blt_DataTableGetRow(interp, table, objv[3], &from) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetRow(interp, table, objv[4], &to) != TCL_OK) {
	return TCL_ERROR;
    }
    count = 1;
    if (objc == 6) {
	if (Tcl_GetLongFromObj(interp, objv[5], &count) != TCL_OK) {
	    return TCL_ERROR;

	}
	if (count == 0) {
	    return TCL_OK;
	}
	if (count < 0) {
	    Tcl_AppendResult(interp, "# of rows can't be negative",
			     (char *)NULL);
	    return TCL_ERROR;
	}
    }
    return Blt_DataTableMoveRows(interp, table, 
		Blt_DataTableRowIndex(table, from),
		Blt_DataTableRowIndex(table, to), count);
}


/*
 *----------------------------------------------------------------------
 *
 * RowNamesOp --
 *
 *	Reports the labels of all rows.  
 * 
 * Results:
 *	Always returns TCL_OK.  The interpreter result is a list of
 *	row labels.
 *	
 *	$t row names pattern...
 *
 *---------------------------------------------------------------------- 
 */
static int
RowNamesOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Tcl_Obj *listObjPtr;
    long iRow, nRows;

    table = cmdPtr->table;
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    nRows = Blt_DataTableNumRows(table);
    for (iRow = 1; iRow <= nRows; iRow++) {
	long row;
	char *label;
	int match;
	int i;

	row = Blt_DataTableRowOffset(table, iRow);
	label = RowLabel(table, row);
	match = (objc == 3);
	for (i = 3; i < objc; i++) {
	    char *pattern;

	    pattern = Tcl_GetString(objv[i]);
	    if (Tcl_StringMatch(label, pattern)) {
		match = TRUE;
		break;
	    }
	}
	if (match) {
	    Tcl_Obj *objPtr;

	    objPtr = Tcl_NewStringObj(label, -1);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowNotifyOp --
 *
 *	Creates a notifier for this instance.  Notifiers represent
 *	a bitmask of events and a command prefix to be invoked 
 *	when a matching event occurs.  
 *
 *	The command prefix is parsed and saved in an array of
 *	Tcl_Objs. Extra slots are allocated for the 
 *
 * Results:
 *	A standard Tcl result.  The name of the new notifier is
 *	returned in the interpreter result.  Otherwise, if it failed
 *	to parse a switch, then TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * Example:
 *	table0 row notify row ?flags? command arg
 *
 *---------------------------------------------------------------------- 
 */
static int
RowNotifyOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator rowIter;
    NotifierInfo *niPtr;
    NotifyData switches;
    char *tag;
    int count;
    int i;
    int nArgs;
    long row;

    table = cmdPtr->table;
    if (Blt_DataTableGetRows(interp, table, objv[3], &rowIter) != TCL_OK) {
	return TCL_ERROR;
    }
    row = -1;
    tag = NULL;
    if (rowIter.type == TABLE_ITER_RANGE) {
	Tcl_AppendResult(interp, "can't notify range of rows: use a tag", 
		(char *)NULL);
	return TCL_ERROR;
    }
    if (rowIter.type == TABLE_ITER_INDEX) {
	row = Blt_DataTableFirstRow(&rowIter);
    } else {
	tag = (char *)rowIter.tagName;
    }
    count = 0;
    for (i = 4; i < objc; i++) {
	char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] != '-') {
	    break;
	}
	count++;
    }
    switches.flags = 0;
    /* Process switches  */
    if (Blt_ParseSwitches(interp, notifySwitches, count, objv + 4, 
	     (char *)&switches, 0) < 0) {
	return TCL_ERROR;
    }
    niPtr = Blt_Malloc(sizeof(NotifierInfo));
    niPtr->cmdPtr = cmdPtr;
    niPtr->notifier = Blt_DataTableRowNotifier(interp, table, switches.flags, 
	row, tag, NotifierEventProc, NotifierDeleteProc, niPtr);
    nArgs = (objc - i) + 2;
    /* Stash away the command in structure and pass that to the notifier. */
    niPtr->cmdv = Blt_Malloc(nArgs * sizeof(Tcl_Obj *));
    for (count = 0; i < objc; i++, count++) {
	Tcl_IncrRefCount(objv[i]);
	niPtr->cmdv[count] = objv[i];
    }
    niPtr->cmdc = nArgs;
    if (switches.flags == 0) {
	switches.flags = TABLE_NOTIFY_ALL;
    }
    {
	char notifyId[200];
	Blt_HashEntry *hPtr;
	int isNew;

	sprintf(notifyId, "notify%d", cmdPtr->nextNotifyId++);
	hPtr = Blt_CreateHashEntry(&cmdPtr->notifyTable, notifyId, &isNew);
	assert(hPtr != NULL);
	assert(isNew);
	Blt_SetHashValue(hPtr, niPtr);
	Tcl_SetStringObj(Tcl_GetObjResult(interp), notifyId, -1);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RowSetOp --
 *
 *	Sets a row of values.  One or more rows may be set using a
 *	tag.  The column order is always the table's current
 *	view of the table.  There may be less values than needed.
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *	$t row set row value...
 *
 *---------------------------------------------------------------------- 
 */
static int
RowSetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator rowIter;
    Tcl_Obj **elv;
    int elc;
    long n;
    long row;

    table = cmdPtr->table;
    if (Blt_DataTableGetRows(interp, table, objv[3], &rowIter) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Tcl_ListObjGetElements(interp, objv[4], &elc, &elv) != TCL_OK) {
	return TCL_ERROR;
    }
    n = elc - Blt_DataTableNumColumns(table);
    if (n > 0) {
	if (Blt_DataTableExtendColumns(interp, table, n, NULL) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	 row = Blt_DataTableNextRow(&rowIter)) {
	long i;

	for (i = 0; i < elc; i++) {
	    long col;

	    col = Blt_DataTableColumnOffset(table, i + 1);
	    if (Blt_DataTableSetValue(interp, table, row, col, elv[i]) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowSortOp --
 *  
 *	$table row sort { a type } b c  -decreasing -list
 *
 *---------------------------------------------------------------------- 
 */
static int
RowSortOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableSortOrder *sp, *send, *order;
    SortSwitches switches;
    int i;
    int result;
    long nRows, n, *map;

    table = cmdPtr->table;
    result = TCL_ERROR;
    /* Create an array of column indices representing the order in
     * which to compare two rows. */
    sp = order = Blt_Calloc((objc - 3), sizeof(Blt_DataTableSortOrder));
    n = objc - 3;
    nRows = Blt_DataTableNumRows(table);
    for (i = 3; i < objc; i++) {
	long col;
	Tcl_Obj **elv;
	int elc;
	char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
	if (Tcl_ListObjGetElements(interp, objv[i], &elc, &elv) != TCL_OK) {
	    goto error;
	}
	if ((elc != 1) && (elc != 2)) {
	    Tcl_AppendResult(interp, "wrong # elements in column sort spec",
			     (char *)NULL);
	    goto error;
	}
	if (Blt_DataTableGetColumn(interp, table, elv[0], &col) != TCL_OK) {
	    goto error;
	}
	sp->offset = col;
	if (elc == 2) {
	    char *string;
	    char c;

	    string = Tcl_GetString(elv[1]);
	    c = string[0];
	    if ((c == 'a') && (strcmp(string, "ascii") == 0)) {
		sp->type = TABLE_SORT_ASCII;
	    } else if ((c == 'd') && (strcmp(string, "dictionary") == 0)) {
		sp->type = TABLE_SORT_DICTIONARY;
	    } else if ((c == 'i') && (strcmp(string, "integer") == 0)) {
		sp->type = TABLE_SORT_INTEGER;
	    } else if ((c == 'd') && (strcmp(string, "double") == 0)) {
		sp->type = TABLE_SORT_DOUBLE;
	    } else {
		CompareData *dataPtr;

		sp->type = TABLE_SORT_CUSTOM;
		dataPtr = Blt_Malloc(sizeof(CompareData));
		assert(dataPtr);
		dataPtr->cmd0 = elv[1];
		dataPtr->interp = interp;
		Tcl_IncrRefCount(dataPtr->cmd0);
		sp->proc = DataTableCompareProc;
		sp->clientData = dataPtr;
	    }
	}
	sp++;
    }
    /* Process switches  */
    switches.flags = 0;
    if (Blt_ParseSwitches(interp, sortSwitches, objc - i, objv + i, &switches, 
	BLT_SWITCH_DEFAULTS) < 0) {
	goto error;
    }
    map = Blt_DataTableSortRows(table, order, sp - order, switches.flags);
    if (map == NULL) {
	Tcl_AppendResult(interp, "out of memory: can't allocate sort map",
			 (char *)NULL);
	goto error;
    }
    if (switches.flags & SORT_LIST) {
	Tcl_Obj *listObjPtr;
	long *ip, *iend;
	
	/* Return the new row order as a list. */
	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (ip = map, iend = ip + nRows; ip < iend; ip++) {
	    Tcl_Obj *objPtr;

	    /* Convert the table offset back to a client index. */
	    objPtr = Tcl_NewLongObj(Blt_DataTableRowIndex(table, *ip));
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	Tcl_SetObjResult(interp, listObjPtr);
    } else {
	/* Make row order permanent. */
	Blt_DataTableSetRowMap(table, map);
    }
    Blt_Free(map);
    result = TCL_OK;
 error:
    for (sp = order, send = sp + n; sp < order; sp++) {
	if (sp->clientData != NULL) {
	    CompareData *dataPtr;

	    dataPtr = sp->clientData;
	    Tcl_DecrRefCount(dataPtr->cmd0);
	    Blt_Free(dataPtr);
	}
    }
    Blt_Free(order);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * RowTagAddOp --
 *
 *	Adds one or more tags for a given row.  Tag names can't
 *	start with a digit (to distinquish them from node ids) and
 *	can't be a reserved tag ("all" or "end").
 *
 *	.t row tag add $row tag1 tag2...
 *	.t row tag range $from $to tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
RowTagAddOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator rowIter;
    int i;

    table = cmdPtr->table;
    if (Blt_DataTableGetRows(interp, table, objv[4], &rowIter) != TCL_OK) {
	return TCL_ERROR;
    }
    for (i = 5; i < objc; i++) {
	CONST char *tagName;
	long row;

	tagName = Tcl_GetString(objv[i]);
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    if (Blt_DataTableSetRowTag(interp, table, row, tagName) != TCL_OK) {
		return TCL_ERROR;
	    }
	}    
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RowTagDeleteOp --
 *
 *	Removes one or more tags from a given row. If a tag doesn't
 *	exist or is a reserved tag ("all" or "end"), nothing will be
 *	done and no error message will be returned.
 *
 *	.t row tag delete $row tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
RowTagDeleteOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator rowIter;
    int i;

    table = cmdPtr->table;
    if (Blt_DataTableGetRows(interp, table, objv[4], &rowIter) != TCL_OK) {
	return TCL_ERROR;
    }
    for (i = 5; i < objc; i++) {
	CONST char *tagName;
	long row;

	tagName = Tcl_GetString(objv[i]);
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    if (Blt_DataTableUnsetRowTag(interp, table, row, tagName)!=TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }    
    return TCL_OK;
}
/*
 *----------------------------------------------------------------------
 *
 * RowTagForgetOp --
 *
 *	Removes the given tags from all nodes.
 *
 *	$t row tag forget tag1 tag2 tag3...
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
RowTagForgetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,		/* Not used. */
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;

    table = cmdPtr->table;
    for (i = 4; i < objc; i++) {
	if (Blt_DataTableForgetRowTag(interp, table, Tcl_GetString(objv[i])) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowTagIndicesOp --
 *
 *	Returns row indices names for the given tags.  If one of
 *	more tag names are provided, then only those matching indices
 *	are returned.
 *
 *	.t row tag indices tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
RowTagIndicesOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;
    long *matches;
    long nRows;

    table = cmdPtr->table;
    nRows = Blt_DataTableNumRows(table);
    matches = Blt_Calloc(nRows + 1, sizeof(long));
    assert(matches);

    /* Handle the reserved tags "all" or "end". */
    for (i = 4; i < objc; i++) {
	char *tagName;
	long n;

	tagName = Tcl_GetString(objv[i]);
	if (strcmp("all", tagName) == 0) {
	    for (n = 1; n <= nRows; n++) {
		matches[n] = TRUE;
	    }
	    goto done;		/* Don't care other tags. */
	} 
	if (strcmp("end", tagName) == 0) {
	    matches[nRows] = TRUE;
	}
    }
    /* Now check user-defined tags. */
    for (i = 4; i < objc; i++) {
	Blt_HashEntry *hPtr;
	Blt_HashSearch cursor;
	Blt_HashTable *tagTablePtr;
	CONST char *tagName;
	
	tagName = Tcl_GetString(objv[i]);
	if ((strcmp("all", tagName) == 0) || (strcmp("end", tagName) == 0)) {
	    continue;
	}
	tagTablePtr = Blt_DataTableRowTagTable(table, tagName);
	if (tagTablePtr == NULL) {
	    Tcl_AppendResult(interp, "unknown row tag \"", tagName, "\"",
		(char *)NULL);
	    Blt_Free(matches);
	    return TCL_ERROR;
	}
	for (hPtr = Blt_FirstHashEntry(tagTablePtr, &cursor); hPtr != NULL; 
	     hPtr = Blt_NextHashEntry(&cursor)) {
	    long row, n;

	    row = (long)Blt_GetHashValue(hPtr);
	    n = Blt_DataTableRowIndex(table, row);
	    assert(n >= 0);
	    matches[n] = TRUE;
	}
    }

 done:
    {
	Tcl_Obj *listObjPtr;
	long n;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
	for (n = 1; n <= nRows; n++) {
	    if (matches[n]) {
		Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewLongObj(n));
	    }
	}
	Tcl_SetObjResult(interp, listObjPtr);
    }
    Blt_Free(matches);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowTagRangeOp --
 *
 *	Adds one or more tags for a given row.  Tag names can't
 *	start with a digit (to distinquish them from node ids) and
 *	can't be a reserved tag ("all" or "end").
 *
 *	.t row tag range $from $to tag1 tag2...
 *
 *---------------------------------------------------------------------- 
 */
static int
RowTagRangeOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;
    long from, to, offset;

    table = cmdPtr->table;
    if (Blt_DataTableGetRow(interp, table, objv[4], &offset) != TCL_OK) {
	return TCL_ERROR;
    }
    from = Blt_DataTableRowIndex(table, offset);
    if (Blt_DataTableGetRow(interp, table, objv[5], &offset) != TCL_OK) {
	return TCL_ERROR;
    }
    to = Blt_DataTableRowIndex(table, offset);
    if (from > to) {
	long tmp;
	tmp = to, to = from, from = tmp;
    }
    for (i = 6; i < objc; i++) {
	CONST char *tagName;
	long iRow;
	
	tagName = Tcl_GetString(objv[i]);
	for (iRow = from; iRow <= to; iRow++) {
	    long row;

	    row = Blt_DataTableRowOffset(table, iRow);
	    assert(row >= 0);
	    if (Blt_DataTableSetRowTag(interp, table, row, tagName) != TCL_OK) {
		return TCL_ERROR;
	    }
	}    
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RowTagSearchOp --
 *
 *	Returns tag names for a given  row.  If one of more pattern
 *	arguments are  provided,   then only those  matching  tags are
 *	returned.
 *
 *	.t row tag find $row pat1 pat2...
 *
 *---------------------------------------------------------------------- 
 */
static int
RowTagSearchOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    Blt_DataTableIterator rowIter;
    Tcl_Obj *listObjPtr;

    table = cmdPtr->table;
    if (Blt_DataTableGetRows(interp, table, objv[4], &rowIter) != TCL_OK) {
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (hPtr = Blt_DataTableFirstRowTag(table, &cursor); hPtr != NULL; 
	hPtr = Blt_NextHashEntry(&cursor)) {
	Blt_HashTable *tablePtr;
	long row;

	tablePtr = Blt_GetHashValue(hPtr);
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    Blt_HashEntry *h2Ptr;
	
	    h2Ptr = Blt_FindHashEntry(tablePtr, (char *)row);
	    if (h2Ptr != NULL) {
		CONST char *tagName;
		int match;
		int i;

		match = (objc == 5);
		tagName = hPtr->key.string;
		for (i = 5; i < objc; i++) {
		    if (Tcl_StringMatch(tagName, Tcl_GetString(objv[i]))) {
			match = TRUE;
			break;	/* Found match. */
		    }
		}
		if (match) {
		    Tcl_Obj *objPtr;

		    objPtr = Tcl_NewStringObj(tagName, -1);
		    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
		    break;	/* Tag matches this row. Don't care
				 * if it matches any other rows. */
		}
	    }
	}
    }

    /* Handle reserved tags specially. */
    {
	int i;
	int allMatch, endMatch;

	endMatch = allMatch = (objc == 5);
	for (i = 5; i < objc; i++) {
	    if (Tcl_StringMatch("all", Tcl_GetString(objv[i]))) {
		allMatch = TRUE;
	    }
	    if (Tcl_StringMatch("end", Tcl_GetString(objv[i]))) {
		endMatch = TRUE;
	    }
	}
	if (allMatch) {
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewStringObj("all", 3));
	}
	if (endMatch) {
	    long row, lastRow, index;
	    
	    index = Blt_DataTableNumRows(table);
	    lastRow = Blt_DataTableRowOffset(table, index);
	    for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
		 row = Blt_DataTableNextRow(&rowIter)) {
		if (row == lastRow) {
		    Tcl_ListObjAppendElement(interp, listObjPtr,
			Tcl_NewStringObj("end", 3));
		    break;
		}
	    }
	}
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowTagOp --
 *
 * 	This procedure is invoked to process tag operations.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec rowTagOps[] =
{
    {"add",     1, (Blt_Op)RowTagAddOp,     5, 0, "row ?tag...?",},
    {"delete",  1, (Blt_Op)RowTagDeleteOp,  5, 0, "row ?tag...?",},
    {"forget",  1, (Blt_Op)RowTagForgetOp,  4, 0, "?tag...?",},
    {"indices", 1, (Blt_Op)RowTagIndicesOp, 4, 0, "?tag...?",},
    {"range",   1, (Blt_Op)RowTagRangeOp,   6, 0, "from to ?tag...?",},
    {"search",  1, (Blt_Op)RowTagSearchOp,  5, 6, "row ?pattern?",},
};

static int nRowTagOps = sizeof(rowTagOps) / sizeof(Blt_OpSpec);

static int
RowTagOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nRowTagOps, rowTagOps, BLT_OP_ARG3, 
			    objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}


/*
 *----------------------------------------------------------------------
 *
 * ColumnTraceOp --
 *
 *	Creates a trace for this instance.  Traces represent
 *	list of keys, a bitmask of trace flags, and a command prefix 
 *	to be invoked when a matching trace event occurs.  
 *
 *	The command prefix is parsed and saved in an array of
 *	Tcl_Objs. The qualified name of the instance is saved
 *	also.
 *
 * Results:
 *	A standard Tcl result.  The name of the new trace is
 *	returned in the interpreter result.  Otherwise, if it failed
 *	to parse a switch, then TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * $t column trace tag rwx proc 
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
RowTraceOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator rowIter;
    Blt_DataTableTrace trace;
    TraceInfo *tiPtr;
    char *tag;
    int flags;
    long row;

    table = cmdPtr->table;
    if (Blt_DataTableGetRows(interp, table, objv[3], &rowIter) != TCL_OK) {
	return TCL_ERROR;
    }
    flags = GetTraceFlags(Tcl_GetString(objv[4]));
    if (flags < 0) {
	Tcl_AppendResult(interp, "unknown flag in \"", Tcl_GetString(objv[4]), 
		"\"", (char *)NULL);
	return TCL_ERROR;
    }
    row = -1;
    tag = NULL;
    if (rowIter.type == TABLE_ITER_RANGE) {
	Tcl_AppendResult(interp, "can't trace range of rows: use a tag", 
		(char *)NULL);
	return TCL_ERROR;
    }
    if (rowIter.type == TABLE_ITER_INDEX) {
	row = Blt_DataTableFirstRow(&rowIter);
    } else {
	tag = (char *)rowIter.tagName;
    }
    tiPtr = Blt_Malloc(sizeof(TraceInfo));
    if (tiPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate trace: out of memory", 
		(char *)NULL);
	return TCL_ERROR;
    }
    trace = Blt_DataTableCreateTrace(table, row, -1, tag, NULL, flags, 
	TraceProc, TraceDeleteProc, tiPtr);
    if (trace == NULL) {
	Tcl_AppendResult(interp, "can't create row trace: out of memory", 
		(char *)NULL);
	Blt_Free(tiPtr);
	return TCL_ERROR;
    }
    /* Initialize the trace information structure. */
    tiPtr->cmdPtr = cmdPtr;
    tiPtr->trace = trace;
    tiPtr->tablePtr = &cmdPtr->traceTable;
    {
	Tcl_Obj **elv, **cmdv;
	int elc, i;

	if (Tcl_ListObjGetElements(interp, objv[5], &elc, &elv) != TCL_OK) {
	    return TCL_ERROR;
	}
	cmdv = Blt_Calloc(elc + 1 + 3 + 1, sizeof(Tcl_Obj *));
	for(i = 0; i < elc; i++) {
	    cmdv[i] = elv[i];
	    Tcl_IncrRefCount(cmdv[i]);
	}
	cmdv[i] = Tcl_NewStringObj(cmdPtr->hPtr->key.string, -1);
	Tcl_IncrRefCount(cmdv[i]);
	tiPtr->cmdc = elc;
	tiPtr->cmdv = cmdv;
    }
    {
	char traceId[200];
	int isNew;

	sprintf(traceId, "trace%d", cmdPtr->nextTraceId++);
	tiPtr->hPtr = Blt_CreateHashEntry(&cmdPtr->traceTable, traceId, &isNew);
	Blt_SetHashValue(tiPtr->hPtr, tiPtr);
	Tcl_SetStringObj(Tcl_GetObjResult(interp), traceId, -1);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowUnsetOp --
 *
 *	$t row unset row ?row?
 *
 *	Unsets one or more rows of values.  One or more rows may be
 *	unset (using tags or multiple arguments).
 * 
 * Results:
 *	A standard Tcl result. If the tag or row index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *---------------------------------------------------------------------- 
 */
static int
RowUnsetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;
    long nCols;

    table = cmdPtr->table;
    nCols = Blt_DataTableNumColumns(table);
    for (i = 3; i < objc; i++) {
	Blt_DataTableIterator rowIter;
	long row;

	if (Blt_DataTableGetRows(interp, table, objv[i], &rowIter) != TCL_OK) {
	    return TCL_ERROR;
	}
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    long iCol;

	    for (iCol = 1; iCol <= nCols; iCol++) {
		long col;

		col = Blt_DataTableColumnOffset(table, iCol);
		if (Blt_DataTableUnsetValue(interp, table, col, row)!=TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	}
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RowOp --
 *
 *	Parses the given command line and calls one of several
 *	row-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec rowOps[] =
{
    {"append", 1, (Blt_Op)RowAppendOp, 3, 0, "?table? ?row...? ?switches?",},
    {"copy",   1, (Blt_Op)RowCopyOp,   4, 0, "?table? from to ?switches?",},
    {"delete", 2, (Blt_Op)RowDeleteOp, 4, 0, "row...",},
    {"dup",    2, (Blt_Op)RowDupOp,    3, 0, "row... ?switches?",},
    {"exists", 3, (Blt_Op)RowExistsOp, 4, 4, "row",},
    {"extend", 3, (Blt_Op)RowExtendOp, 4, 4, "count",},
    {"get",    1, (Blt_Op)RowGetOp,    4, 5, "row ?defValue?",},
    {"index",  2, (Blt_Op)RowIndexOp,  4, 0, "row ?switches?",},
    {"insert", 2, (Blt_Op)RowInsertOp, 3, 0, "?switches?",},
    {"label",  2, (Blt_Op)RowLabelOp,  4, 5, "row ?label?",},
    {"length", 2, (Blt_Op)RowLengthOp, 3, 3, "",},
    {"move",   1, (Blt_Op)RowMoveOp,   5, 6, "from to ?count?",},
    {"names",  2, (Blt_Op)RowNamesOp,  3, 0, "?pattern...?",},
    {"notify", 2, (Blt_Op)RowNotifyOp, 5, 0, "row ?flags? command",},
    {"set",    2, (Blt_Op)RowSetOp,    5, 5, "row valueList",},
    {"sort",   2, (Blt_Op)RowSortOp,   3, 0, "?flags...?",},
    {"tag",    2, (Blt_Op)RowTagOp,    3, 0, "op args",},
    {"trace",  2, (Blt_Op)RowTraceOp,  6, 6, "row how command",},
    {"unset",  1, (Blt_Op)RowUnsetOp,  4, 0, "row...",},
};

static int nRowOps = sizeof(rowOps) / sizeof(Blt_OpSpec);

static int
RowOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nRowOps, rowOps, BLT_OP_ARG2, objc, 
	objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

/****** Array Operations *********/

/*
 *----------------------------------------------------------------------
 *
 * ArrayExistsOp --
 *
 *	Retrieves the value from a given table for a designated
 *	column.  It's an error if the column key or row table is
 *	invalid.  If no key argument is provided then, all key-value
 *	pairs in the table are returned.  If a value is empty (the
 *	Tcl_Obj value is NULL), then the table command's "empty"
 *	string representation will be used.
 *	
 *	A third argument may be provided as the default return value.
 *	If no value exists for the given key, then this value is
 *	returned.
 * 
 * Results:
 *	A standard Tcl result. If the tag or index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *	$t array exists row column key
 *
 *---------------------------------------------------------------------- 
 */
static int
ArrayExistsOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int bool;
    long row, col;

    table = cmdPtr->table;
    if (Blt_DataTableGetRow(interp, table, objv[3], &row) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetColumn(interp, table, objv[4], &col) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableArrayValueExists(interp, table, row, col, 
	Tcl_GetString(objv[5]), &bool) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), bool);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * ArrayGetOp --
 *	
 *	Retrieves the value from a given table for a designated
 *	column.  It's an error if the column key or row table is
 *	invalid.  If no key argument is provided then, all key-value
 *	pairs in the table are returned.  If a value is empty (the
 *	Tcl_Obj value is NULL), then the table command's "empty"
 *	string representation will be used.
 *	
 *	A third argument may be provided as the default return value.
 *	If no value exists for the given key, then this value is
 *	returned.
 * 
 * Results:
 *	A standard Tcl result. If the tag or index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *	$t array get row column key ?defValue?
 *
 *---------------------------------------------------------------------- 
 */
static int
ArrayGetOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    CONST char *key;
    Tcl_Obj *valueObjPtr;
    long row, col;

    table = cmdPtr->table;
    if (Blt_DataTableGetRow(interp, table, objv[3], &row) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetColumn(interp, table, objv[4], &col) != TCL_OK) {
	return TCL_ERROR;
    }
    key = Tcl_GetString(objv[5]);
    if (Blt_DataTableGetArrayValue(interp, table, row, col, key, &valueObjPtr) 
	!= TCL_OK) {
	return TCL_ERROR;
    }
    if ((valueObjPtr == NULL) || (objc == 6)) {
	valueObjPtr = objv[5];
    }
    if (valueObjPtr == NULL) {
	valueObjPtr = cmdPtr->emptyValueObjPtr;
    }
    Tcl_SetObjResult(interp, valueObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ArrayNamesOp --
 *
 *	Retrieves the value from a given table for a designated
 *	column.  It's an error if the column key or row table is
 *	invalid.  If no key argument is provided then, all key-value
 *	pairs in the table are returned.  If a value is empty (the
 *	Tcl_Obj value is NULL), then the table command's "empty"
 *	string representation will be used.
 *	
 *	A third argument may be provided as the default return value.
 *	If no value exists for the given key, then this value is
 *	returned.
 * 
 * Results:
 *	A standard Tcl result. If the tag or index is invalid, 

 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *	$t array names row column
 *
 *---------------------------------------------------------------------- 
 */
static int
ArrayNamesOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    Blt_HashTable *tablePtr;
    Tcl_Obj *arrayObjPtr, *listObjPtr;
    long row, col;

    table = cmdPtr->table;
    if (Blt_DataTableGetRow(interp, table, objv[3], &row) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetColumn(interp, table, objv[4], &col) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetValue(interp, table, row, col, &arrayObjPtr)!=TCL_OK) {
	return TCL_ERROR;
    }
    if (arrayObjPtr == NULL) {
	return TCL_OK;		/* Empty array. */
    }
    if (Blt_GetArrayFromObj(interp, arrayObjPtr, &tablePtr) != TCL_OK) {
	return TCL_ERROR;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (hPtr = Blt_FirstHashEntry(tablePtr, &cursor); hPtr != NULL;
	 hPtr = Blt_NextHashEntry(&cursor)) {
	CONST char *key;
	Tcl_Obj *objPtr;

	key = Blt_GetHashKey(tablePtr, hPtr);
	objPtr = Tcl_NewStringObj(key, -1);
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ArraySetOp --
 *
 *	Retrieves the value from a given table for a designated
 *	column.  It's an error if the column key or row table is
 *	invalid.  If no key argument is provided then, all key-value
 *	pairs in the table are returned.  If a value is empty (the
 *	Tcl_Obj value is NULL), then the table command's "empty"
 *	string representation will be used.
 *	
 *	A third argument may be provided as the default return value.
 *	If no value exists for the given key, then this value is
 *	returned.
 * 
 * Results:
 *	A standard Tcl result. If the tag or index is invalid, 

 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *	$t array set row column key value
 *
 *---------------------------------------------------------------------- 
 */
static int
ArraySetOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;

    if (((objc - 3) % 4) != 0) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", 
		Tcl_GetString(objv[0]), 
		" array unset ?row column key value?...\"", (char *)NULL);
	return TCL_ERROR;
    }
    table = cmdPtr->table;
    for (i = 3; i < objc; i += 4) {
	Blt_DataTableIterator rowIter, colIter;
	long row;
	CONST char *key;

	if (Blt_DataTableGetRows(interp, table, objv[i], &rowIter) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Blt_DataTableGetColumns(interp, table, objv[i+1], &colIter) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	key = Tcl_GetString(objv[i+2]);
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    long col;

	    for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
		 col = Blt_DataTableNextColumn(&colIter)) {
		if (Blt_DataTableSetArrayValue(interp, table, row, col, key,
			objv[i+3]) != TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	}	    
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ArrayUnsetOp --
 *
 *	Retrieves the value from a given table for a designated
 *	column.  It's an error if the column key or row table is
 *	invalid.  If no key argument is provided then, all key-value
 *	pairs in the table are returned.  If a value is empty (the
 *	Tcl_Obj value is NULL), then the table command's "empty"
 *	string representation will be used.
 *	
 *	A third argument may be provided as the default return value.
 *	If no value exists for the given key, then this value is
 *	returned.
 * 
 * Results:
 *	A standard Tcl result. If the tag or index is invalid, 

 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *	$t array unset row column key
 *
 *---------------------------------------------------------------------- 
 */
static int
ArrayUnsetOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;

    if (((objc - 3) % 3) != 0) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", 
		Tcl_GetString(objv[0]), 
		" array unset ?row column key?...\"", (char *)NULL);
	return TCL_ERROR;
    }
    table = cmdPtr->table;
    for (i = 3; i < objc; i += 3) {
	Blt_DataTableIterator rowIter, colIter;
	long row;
	CONST char *key;

	if (Blt_DataTableGetRows(interp, table, objv[i], &rowIter) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Blt_DataTableGetColumns(interp, table, objv[i+1], &colIter) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	key = Tcl_GetString(objv[i+2]);
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    long col;

	    for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
		 col = Blt_DataTableNextColumn(&colIter)) {
		if (Blt_DataTableUnsetArrayValue(interp, table, row, col, 
			key) != TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	}	    
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ArrayOp --
 *
 *	Parses the given command line and calls one of several
 *	array-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec arrayOps[] =
{
    {"exists", 1, (Blt_Op)ArrayExistsOp, 6, 6, "row column key",},
    {"get",    1, (Blt_Op)ArrayGetOp,    6, 7, "row column key ?defValue?",},
    {"names",  1, (Blt_Op)ArrayNamesOp,  5, 5, "row column",},
    {"set",    3, (Blt_Op)ArraySetOp,    3, 0, "?row column key value?...",},
    {"unset",  1, (Blt_Op)ArrayUnsetOp,  3, 0, "?row column key?...",},
};

static int nArrayOps = sizeof(arrayOps) / sizeof(Blt_OpSpec);

static int
ArrayOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nArrayOps, arrayOps, BLT_OP_ARG2, objc, 
	objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

static void
StartCsvRecord(ExportCsvData *exportPtr)
{
    if (exportPtr->channel != NULL) {
	Tcl_DStringSetLength(exportPtr->dsPtr, 0);
	exportPtr->length = 0;
    }
    exportPtr->count = 0;
}

static int
EndCsvRecord(ExportCsvData *exportPtr)
{
    int nWritten;
    char *line;

    Tcl_DStringAppend(exportPtr->dsPtr, "\n", 1);
    exportPtr->length++;
    line = Tcl_DStringValue(exportPtr->dsPtr);
    if (exportPtr->channel != NULL) {
#if HAVE_UTF
	nWritten = Tcl_WriteChars(exportPtr->channel, line, exportPtr->length);
#else
	nWritten = Tcl_Write(exportPtr->channel, line, exportPtr->length);
#endif
	if (nWritten != exportPtr->length) {
	    Tcl_AppendResult(exportPtr->interp, "can't write csv record: ",
			     Tcl_PosixError(exportPtr->interp), (char *)NULL);
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

static void
AppendCsvRecord(
    ExportCsvData *exportPtr, 
    CONST char *field, 
    int length, 
    int type)
{
    CONST char *fp;
    char *p;
    int extra, doQuote;

    doQuote = ((type == TABLE_COLUMN_TYPE_STRING) || 
	       (type == TABLE_COLUMN_TYPE_UNKNOWN));
    extra = 0;
    if (field == NULL) {
	length = 0;
    } else {
	for (fp = field; *fp != '\0'; fp++) {
	    if ((*fp == '\n') || (*fp == exportPtr->sep[0]) || 
		(*fp == ' ') || (*fp == '\t')) {
		doQuote = TRUE;
	    } else if (*fp == exportPtr->quote[0]) {
		doQuote = TRUE;
		extra++;
	    }
	}
	if (doQuote) {
	    extra += 2;
	}
	if (length < 0) {
	    length = fp - field;
	}
    }
    if (exportPtr->count > 0) {
	Tcl_DStringAppend(exportPtr->dsPtr, exportPtr->sep, 1);
	exportPtr->length++;
    }
    length = length + extra + exportPtr->length;
    Tcl_DStringSetLength(exportPtr->dsPtr, length);
    p = Tcl_DStringValue(exportPtr->dsPtr) + exportPtr->length;
    exportPtr->length = length;
    if (field != NULL) {
	if (doQuote) {
	    *p++ = exportPtr->quote[0];
	}
	for (fp = field; *fp != '\0'; fp++) {
	    if (*fp == exportPtr->quote[0]) {
		*p++ = exportPtr->quote[0];
	    }
	    *p++ = *fp;
	}
	if (doQuote) {
	    *p++ = exportPtr->quote[0];
	}
    }
    exportPtr->count++;
}

static int
ExportCsvRows(Blt_DataTable table, ExportCsvData *exportPtr)
{
    Blt_ChainLink *linkPtr;

    for (linkPtr = Blt_ChainFirstLink(exportPtr->rowChainPtr); 
	 linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	Blt_ChainLink *linkPtr2;
	long row;
	CONST char *field;
	    
	row = (long)Blt_ChainGetValue(linkPtr);
	StartCsvRecord(exportPtr);
	field = Blt_DataTableRowLabel(table, row);
	AppendCsvRecord(exportPtr, field, -1, 0);
	for (linkPtr2 = Blt_ChainFirstLink(exportPtr->colChainPtr); 
	     linkPtr2 != NULL; linkPtr2 = Blt_ChainNextLink(linkPtr2)) {
	    long col;
	    Tcl_Obj *objPtr;
	    
	    col = (long)Blt_ChainGetValue(linkPtr2);
	    if (Blt_DataTableGetValue(exportPtr->interp, table, row, col,
				      &objPtr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (objPtr == NULL) {
		AppendCsvRecord(exportPtr, NULL, -1, 0);
	    } else {
		int length;
		int type;
		
		type = Blt_DataTableColumnType(table, col);
		field = Tcl_GetStringFromObj(objPtr, &length);
		AppendCsvRecord(exportPtr, field, length, type);
	    }
	}
	if (EndCsvRecord(exportPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

static int
ExportCsvColumns(Blt_DataTable table, ExportCsvData *exportPtr)
{
    Blt_ChainLink *linkPtr;

    StartCsvRecord(exportPtr);
    AppendCsvRecord(exportPtr, "*BLT*", 5, 0);
    for (linkPtr = Blt_ChainFirstLink(exportPtr->colChainPtr); 
	 linkPtr != NULL; linkPtr = Blt_ChainNextLink(linkPtr)) {
	long col;
	CONST char *field;
	
	col = (long)Blt_ChainGetValue(linkPtr);
	field = Blt_DataTableColumnLabel(table, col);
	AppendCsvRecord(exportPtr, field, -1, 0);
    }
    return EndCsvRecord(exportPtr);
}

static int
ExportDataXmlOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    return TCL_OK;
}

static int
ExportFileXmlOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    return TCL_OK;
}

#include <bltTree.h>

static int
ExportTree(
    Tcl_Interp *interp,    
    DataTableCmd *cmdPtr, 
    Blt_Tree tree, 
    Blt_TreeNode root, 
    Blt_TreeNode node)
{
    Blt_DataTable table;

    table = cmdPtr->table;
    for (node = Blt_TreeFirstChild(node); node != NULL;
	 node = Blt_TreeNextSibling(node)) {
	Blt_TreeKey key;
	Blt_TreeKeySearch cursor;
	long row;
	Tcl_DString ds;
	char *label;
	int result;

	if (Blt_DataTableExtendRows(interp, table, 1, &row) != TCL_OK) {
	    return TCL_ERROR;
	}
	Tcl_DStringInit(&ds);
	label = Blt_TreeNodeRelativePath(root, node, "/", TREE_INCLUDE_ROOT, 
		&ds);
	result = Blt_DataTableSetRowLabel(interp, table, row, label);
	Tcl_DStringFree(&ds);
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
	for (key = Blt_TreeFirstKey(tree, node, &cursor); key != NULL;
	     key = Blt_TreeNextKey(tree, &cursor)) {
	    long col;
	    Tcl_Obj *objPtr;

	    col = Blt_DataTableFindColumn(table, key);
	    if (col < 0) {
		if (Blt_DataTableExtendColumns(interp, table, 1, &col) 
		    != TCL_OK) {
		    return TCL_ERROR;
		}
		if (Blt_DataTableSetColumnLabel(interp, table, col, key) 
		    != TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	    if (Blt_TreeGetValue(interp, tree, node, key, &objPtr) != TCL_OK) {
		return TCL_ERROR;
	    }
#ifdef notdef
	    fprintf(stderr, "key = %s, Tcl_Obj = %x, (%s)\n", key, objPtr, 
		    Tcl_GetString(objPtr));
#endif
	    if (Blt_DataTableSetValue(interp, table, row, col, objPtr) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}
	if (ExportTree(interp, cmdPtr, tree, root, node) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}

static int
ExportCsvOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    ExportCsvData switches;
    Tcl_Channel channel;
    Tcl_DString ds;
    int closeChannel;
    int result;

    table = cmdPtr->table;
    closeChannel = FALSE;
    channel = NULL;
    if (strncmp(Tcl_GetString(objv[1]), "exportf", 7) == 0) {
	char *fileName;

	closeChannel = TRUE;
	fileName = Tcl_GetString(objv[3]);
	if ((fileName[0] == '@') && (fileName[1] != '\0')) {
	    int mode;
	    
	    channel = Tcl_GetChannel(interp, fileName+1, &mode);
	    if (channel == NULL) {
		return TCL_ERROR;
	    }
	    if ((mode & TCL_WRITABLE) == 0) {
		Tcl_AppendResult(interp, "channel \"", fileName, 
				 "\" not opened for writing", (char *)NULL);
		return TCL_ERROR;
	    }
	    closeChannel = FALSE;
	} else {
	    channel = Tcl_OpenFileChannel(interp, fileName, "w", 0666);
	    if (channel == NULL) {
		return TCL_ERROR;	/* Can't open export file. */
	    }
	}
	objv++, objc--;
    }
    result = TCL_ERROR;
    Tcl_DStringInit(&ds);
    memset(&switches, 0, sizeof(ExportCsvData));
    switches.sep = ",";
    switches.quote = "\"";
    switches.interp = interp;
    switches.dsPtr = &ds;
    switches.channel = channel;
    if (Blt_ParseSwitches(interp, exportCsvSwitches, objc - 3, objv + 3, 
		&switches, BLT_SWITCH_DEFAULTS) < 0) {
	goto error;
    }
    if (switches.rows != NULL) {
	Blt_Chain *chainPtr;
	Tcl_Obj **elv;
	int elc;

	if (Tcl_ListObjGetElements(interp, switches.rows, &elc, &elv)
	    !=TCL_OK) {
	    goto error;
	}
	chainPtr = Blt_ChainCreate();
	if (GetRows(interp, table, elc, elv, chainPtr) != TCL_OK) {
	    Blt_ChainDestroy(chainPtr);
	    goto error;
	}
	if (Blt_ChainGetLength(chainPtr) == 0) {
	    Blt_ChainDestroy(chainPtr);
	    Tcl_AppendResult(interp, "no rows specified.", (char *)NULL);
	    goto error;
	}
	switches.rowChainPtr = chainPtr;
    } else {
	switches.rowChainPtr = AllRows(table);
    }
    if (switches.cols != NULL) {
	Blt_Chain *chainPtr;
	Tcl_Obj **elv;
	int elc;

	if (Tcl_ListObjGetElements(interp, switches.cols, &elc, &elv)!=TCL_OK) {
	    goto error;
	}
	chainPtr = Blt_ChainCreate();
	if (GetColumns(interp, table, elc, elv, chainPtr) != TCL_OK) {
	    Blt_ChainDestroy(chainPtr);
	    goto error;
	}
	if (Blt_ChainGetLength(chainPtr) == 0) {
	    Blt_ChainDestroy(chainPtr);
	    Tcl_AppendResult(interp, "no columns specified.", (char *)NULL);
	    goto error;
	}
	switches.colChainPtr = chainPtr;
    } else {
	switches.colChainPtr = AllColumns(table);
    }
    result = ExportCsvColumns(table, &switches); 
    if (result == TCL_OK) {
	result = ExportCsvRows(table, &switches);
    }
    if ((switches.channel == NULL) && (result == TCL_OK)) {
	Tcl_DStringResult(interp, &ds);
    } 
 error:
    if (switches.rowChainPtr != NULL) {
	Blt_ChainDestroy(switches.rowChainPtr);
    }
    if (switches.colChainPtr != NULL) {
	Blt_ChainDestroy(switches.colChainPtr);
    }
    Tcl_DStringFree(&ds);
    if (closeChannel) {
	Tcl_Close(interp, channel);
    }
    return result;
}


static int
ExportDataTreeOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_Tree tree;
    Blt_TreeNode node;

    if (Blt_TreeCmdGetToken(interp, Tcl_GetString(objv[3]), &tree) != TCL_OK) {
	return TCL_ERROR;
    }
    if (objc == 5) {
	long inode;

	if (Tcl_GetLongFromObj(interp, objv[4], &inode) != TCL_OK) {
	    return TCL_ERROR;
	}
	node = Blt_TreeGetNode(tree, inode);
	if (node == NULL) {
	    return TCL_ERROR;
	}
    } else {
	node = Blt_TreeRootNode(tree);
    }
    return ExportTree(interp, cmdPtr, tree, node, node);
}

/*
 *----------------------------------------------------------------------
 *
 * ExportDataVectorOp --
 *
 *	Parses the given command line and calls one of several
 *	export-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *	$table exportdata vector 10 v1 20 v2 
 *
 *---------------------------------------------------------------------- 
 */
static int
ExportDataVectorOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;
    int result;
    double *values;

    if (((objc - 3) % 2) != 0) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", 
		Tcl_GetString(objv[0]), 
		" exportdata vector ?column vector?...\"", (char *)NULL);
	return TCL_ERROR;
    }
    table = cmdPtr->table;
    values = NULL;
    for (i = 3; i < objc; i += 2) {
	Blt_Vector *vector;
	Tcl_Obj *objPtr;
	double *vp;
	long col;
	long nRows, iRow;

	if (Blt_DataTableGetColumn(interp, table, objv[i], &col) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Blt_GetVectorFromObj(interp, objv[i+1], &vector) != TCL_OK) {
	    return TCL_ERROR;
	}
	nRows = Blt_DataTableNumRows(table);
	values = Blt_Malloc(sizeof(double) * nRows);
	for (vp = values, iRow = 1; iRow <= nRows; iRow++) {
	    long row;

	    row = Blt_DataTableRowOffset(table, iRow);
	    result = Blt_DataTableGetValue(interp, table, row, col, &objPtr);
	    if (result != TCL_OK) {
		goto error;
	    }
	    if (objPtr == NULL) {
		continue;
	    }
	    result = Tcl_GetDoubleFromObj(interp, objPtr, vp);
	    if (result != TCL_OK) {
		goto error;
	    }
	    vp++;
	}
	result = Blt_ResetVector(vector, values, vp-values, nRows, TCL_DYNAMIC);
	if (result != TCL_OK) {
	    goto error;
	}
    }
    return TCL_OK;
 error:
    if (values != NULL) {
	Blt_Free(values);
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * ExportDataOp --
 *
 *	Parses the given command line and calls one of several
 *	export-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec exportDataOps[] =
{
    {"csv",    1, (Blt_Op)ExportCsvOp,        3, 0, "?switches?",},
    {"tree",   1, (Blt_Op)ExportDataTreeOp,   5, 0, "tree node ?switches?",},
    {"vector", 1, (Blt_Op)ExportDataVectorOp, 3, 0, "?column vector?...",},
    {"xml",    1, (Blt_Op)ExportDataXmlOp,    3, 0, "?switches?",},
};

static int nExportDataOps = sizeof(exportDataOps) / sizeof(Blt_OpSpec);

static int
ExportDataOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nExportDataOps, exportDataOps, BLT_OP_ARG2,
		objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ExportFileOp --
 *
 *	Parses the given command line and calls one of several
 *	export-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec exportFileOps[] =
{
    {"csv",  1, (Blt_Op)ExportCsvOp,      4, 0, "fileName ?switches?",},
    {"xml",  1, (Blt_Op)ExportFileXmlOp,  4, 0, "fileName ?switches?",},
};

static int nExportFileOps = sizeof(exportFileOps) / sizeof(Blt_OpSpec);

static int
ExportFileOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nExportFileOps, exportFileOps, BLT_OP_ARG2,
		objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

static int
ImportGetBuffer(
    Tcl_Interp *interp, 
    ImportCsvData *importPtr, 
    char **bufferPtr,
    int *nBytesPtr)
{
    if (importPtr->channel != NULL) {
	int nBytes;

	if (Tcl_Eof(importPtr->channel)) {
	    *nBytesPtr = -1;
	    return TCL_OK;
	}
#define BUFFSIZE	8191
	nBytes = Tcl_Read(importPtr->channel, importPtr->buffer, 
		sizeof(char) * BUFFSIZE);
	*nBytesPtr = nBytes;
	if (nBytes < 0) {
	    Tcl_AppendResult(interp, "error reading file: ", 
			     Tcl_PosixError(interp), (char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	*nBytesPtr =  importPtr->nBytes;
	importPtr->nBytes = -1;
    }
    *bufferPtr = importPtr->buffer;
    return TCL_OK;
}


static int
ImportCsv(Tcl_Interp *interp, DataTableCmd *cmdPtr, ImportCsvData *importPtr)
{
    Blt_DataTable table;
    Tcl_DString dString;
    char *fp, *field;
    int fieldSize;
    int inQuotes, isQuoted, isPath;
    int result;
    long iCol;
    long row, col;

    table = cmdPtr->table;
    result = TCL_ERROR;
    isPath = isQuoted = inQuotes = FALSE;
    row = -1;
    iCol = 1;
    Tcl_DStringInit(&dString);
    fieldSize = 128;
    Tcl_DStringSetLength(&dString, fieldSize + 1);
    fp = field = Tcl_DStringValue(&dString);
    for (;;) {
	char *bp, *bend;
	int nBytes;

	result = ImportGetBuffer(interp, importPtr, &bp, &nBytes);
	if (result != TCL_OK) {
	    goto error;		/* I/O Error. */
	}
	if (nBytes < 0) {
	    break;		/* Eof */
	}
	for (bend = bp + nBytes; bp < bend; bp++) {
	    switch (*bp) {
	    case '\t':
	    case ' ':
		/* 
		 * Add whitespace only if it's not leading or we're
		 * inside of quotes or a path. 
		 */
		if ((fp != field) || (inQuotes) || (isPath)) {
		    *fp++ = *bp; 
		}
		break;

	    case '\\':
		/* 
		 * Handle special case CSV files that allow unquoted paths.
		 * Example:  ...,\this\path " should\have been\quoted\,...
		 */
		if (fp == field) {
		    isPath = TRUE; 
		}
		*fp++ = *bp;
		break;

	    case '"':
		if (inQuotes) {
		    if (*(bp+1) == '"') {
			*fp++ = '"';
			bp++;
		    } else {
			inQuotes = FALSE;
		    }
		} else {
		    /* 
		     * If the quote doesn't start a field, then treat
		     * all quotes in the field as ordinary characters.
		     */
		    if (fp == field) {
			isQuoted = inQuotes = TRUE; 
		    } else {
			*fp++ = *bp;
		    }
		}
		break;

	    case ',':
	    case '\n':
		if (inQuotes) {
		    *fp++ = *bp; /* Copy the comma or newline. */
		} else {
		    long col;
		    char *last;
		    Tcl_Obj *objPtr;

		    if ((isPath) && (*bp == ',') && (fp != field) && 
			(*(fp - 1) != '\\')) {
			*fp++ = *bp; /* Copy the comma or newline. */
			break;
		    }    
		    /* "last" points to the character after the last
		     * character in the field. */
		    last = fp;	

		    /* Remove trailing spaces only if the field wasn't
		     * quoted. */
		    if ((!isQuoted) && (!isPath)) {
			while ((last > field) && (isspace(*(last - 1)))) {
			    last--;
			}
		    }
		    if (row == -1) {
			if (*bp == '\n') {
			    break; /* Ignore empty lines. */
			}
			if (Blt_DataTableExtendRows(interp, table, 1, &row) 
			    != TCL_OK) {
			    goto error;
			}
		    }
		    /* End of field. Append field to row. */
		    col = Blt_DataTableColumnOffset(table, iCol);
		    if (col < 0) {
			if (Blt_DataTableExtendColumns(interp, table, 1, 
						       &col) != TCL_OK) {
			    goto error;
			}
		    }			
		    if ((last > field) || (isQuoted)) {
			objPtr = Tcl_NewStringObj(field, last - field);
			if (Blt_DataTableSetValue(interp, table, row, col, 
						  objPtr) != TCL_OK) {
			    goto error;
			}
		    }
		    iCol++;
		    if (*bp == '\n') {
			row = -1;
			iCol = 1;
		    }
		    fp = field;
		    isPath = isQuoted = FALSE;
		}
		break;

	    default:
		*fp++ = *bp;	/* Copy the character. */
	    }
	    if ((fp - field) >= fieldSize) {
		int offset;

		/* 
		 * We've exceeded the current maximum size of the
		 * field. Double the size of the field, but make sure
		 * to reset the pointers to the (possibly) new memory
		 * location. 
		 */
		offset = fp - field;
		fieldSize += fieldSize;
		Tcl_DStringSetLength(&dString, fieldSize + 1);
		field = Tcl_DStringValue(&dString);
		fp = field + offset;
	    }
	}
	if (nBytes < 1) {
	    /* 
	     * We're reached the end of input. But there may not have
	     * been a final newline to trigger the final append. So
	     * check if a last field is still needs appending the the
	     * last row.
	     */
	    if (fp != field) {
		char *last;
		Tcl_Obj *objPtr;

		last = fp;
		/* Remove trailing spaces */
		while (isspace(*(last - 1))) {
		    last--;
		}
		if (row == -1) {
		    if (Blt_DataTableExtendRows(interp, table, 1, &row) 
			!= TCL_OK) {
			goto error;
		    }
		}
		col = Blt_DataTableColumnOffset(table, iCol);
		if (col < 0) {
		    if (Blt_DataTableExtendColumns(interp, table, 1, &col)
			!= TCL_OK) {
			goto error;
		    }
		}			
		if ((last > field) || (isQuoted)) {
		    objPtr = Tcl_NewStringObj(field, last - field);
		    if (Blt_DataTableSetValue(interp, table, row, col, objPtr) 
			!= TCL_OK) {
			goto error;
		    }
		}		
	    }    
	    break;
	}
    }
    result = TCL_OK;
 error:
    Tcl_DStringFree(&dString);
    return result;
}

static int
ImportDataXmlOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    return TCL_OK;
}

static int
ImportFileXmlOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    return TCL_OK;
}

#include <bltTree.h>

static int
ImportTree(
    Tcl_Interp *interp,    
    DataTableCmd *cmdPtr, 
    Blt_Tree tree, 
    Blt_TreeNode root, 
    Blt_TreeNode node)
{
    Blt_DataTable table;

    table = cmdPtr->table;
    for (node = Blt_TreeFirstChild(node); node != NULL;
	 node = Blt_TreeNextSibling(node)) {
	Blt_TreeKey key;
	Blt_TreeKeySearch cursor;
	long row;
	Tcl_DString ds;
	char *label;
	int result;

	if (Blt_DataTableExtendRows(interp, table, 1, &row) != TCL_OK) {
	    return TCL_ERROR;
	}
	Tcl_DStringInit(&ds);
	label = Blt_TreeNodeRelativePath(root, node, "/", TREE_INCLUDE_ROOT, 
		&ds);
	result = Blt_DataTableSetRowLabel(interp, table, row, label);
	Tcl_DStringFree(&ds);
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
	for (key = Blt_TreeFirstKey(tree, node, &cursor); key != NULL;
	     key = Blt_TreeNextKey(tree, &cursor)) {
	    long col;
	    Tcl_Obj *objPtr;
	    Tcl_DString ds;

	    if (Blt_DataTableExtendColumns(interp, table, 1, &col) != TCL_OK) {
		return TCL_ERROR;
	    }
	    Tcl_DStringInit(&ds);
	    if (Blt_DataTableFindColumn(table, key) >= 0) {
		key = CopyOfColumnLabel(table, key, &ds);
	    }
	    result = Blt_DataTableSetColumnLabel(interp, table, col, key);
	    Tcl_DStringFree(&ds);
	    if (result != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (Blt_TreeGetValue(interp, tree, node, key, &objPtr) != TCL_OK) {
		return TCL_ERROR;
	    }
#ifdef notdef
	    fprintf(stderr, "key = %s, Tcl_Obj = %x, (%s)\n", key, objPtr, 
		    Tcl_GetString(objPtr));
#endif
	    if (Blt_DataTableSetValue(interp, table, row, col, objPtr) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}
	if (ImportTree(interp, cmdPtr, tree, root, node) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}
static int
ImportDataCsvOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    int nBytes;
    ImportCsvData switches;

    switches.channel = NULL;
    switches.buffer = Tcl_GetStringFromObj(objv[3], &nBytes);
    switches.nBytes = nBytes;
    switches.fileName = NULL;
    return ImportCsv(interp, cmdPtr, &switches);
}

static int
ImportFileCsvOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    int result;
    int closeChannel;
    Tcl_Channel channel;
    char *fileName;
    ImportCsvData switches;
    char buffer[BUFFSIZE+1];

    closeChannel = TRUE;
    fileName = Tcl_GetString(objv[3]);
    if ((fileName[0] == '@') && (fileName[1] != '\0')) {
	int mode;
	
	channel = Tcl_GetChannel(interp, fileName+1, &mode);
	if (channel == NULL) {
	    return TCL_ERROR;
	}
	if ((mode & TCL_READABLE) == 0) {
	    Tcl_AppendResult(interp, "channel \"", fileName, 
		"\" not opened for reading", (char *)NULL);
	    return TCL_ERROR;
	}
	closeChannel = FALSE;
    } else {
	channel = Tcl_OpenFileChannel(interp, fileName, "r", 0);
	if (channel == NULL) {
	    return TCL_ERROR;	/* Can't open dump file. */
	}
    }
    switches.channel = channel;
    switches.buffer = buffer;
    switches.fileName = fileName;
    result = ImportCsv(interp, cmdPtr, &switches);
    if (closeChannel) {
	Tcl_Close(interp, channel);
    }
    return result;
}


static int
ImportDataTreeOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_Tree tree;
    Blt_TreeNode node;

    if (Blt_TreeCmdGetToken(interp, Tcl_GetString(objv[3]), &tree) != TCL_OK) {
	return TCL_ERROR;
    }
    if (objc == 5) {
	long inode;

	if (Tcl_GetLongFromObj(interp, objv[4], &inode) != TCL_OK) {
	    return TCL_ERROR;
	}
	node = Blt_TreeGetNode(tree, inode);
	if (node == NULL) {
	    return TCL_ERROR;
	}
    } else {
	node = Blt_TreeRootNode(tree);
    }
    return ImportTree(interp, cmdPtr, tree, node, node);
}


#include <bltVector.h>

/*
 *----------------------------------------------------------------------
 *
 * ImportDataVectorOp --
 *
 *	Parses the given command line and calls one of several
 *	export-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *	$table importdata vector v1 v2 v3 
 *
 *---------------------------------------------------------------------- 
 */
static int
ImportDataVectorOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_Vector *vector;
    int nVectors, maxLen;
    int result;
    long *cols;
    int i;
    long nRows, nCols;
    int isInteger;

    isInteger = 0;
    table = cmdPtr->table;
    maxLen = 0;
    for (i = 3; i < objc; i++) {
	if (Blt_GetVectorFromObj(interp, objv[i], &vector) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Blt_VecLength(vector) > maxLen) {
	    maxLen = Blt_VecLength(vector);
	}
    }
    nRows = Blt_DataTableNumRows(table);
    nCols = Blt_DataTableNumColumns(table);
    nVectors = objc - 3;
    cols = Blt_Malloc(sizeof(long) * nVectors);
    result = Blt_DataTableExtendColumns(interp, table, nVectors, cols);
    if (result != TCL_OK) {
	goto error;
    }
    if (nRows < maxLen) {
	result = Blt_DataTableExtendRows(interp, table, maxLen - nRows, NULL);
	if (result != TCL_OK) {
	    goto error;
	}
    }
    for (i = 3; i < objc; i++) {
	Tcl_DString ds;
	char *label;
	double *array;
	int nElems;
	long n;

	label = Tcl_GetString(objv[i]);
	result = Blt_GetVector(interp, label, &vector);
	if (result != TCL_OK) {
	    goto error;
	}
	Tcl_DStringInit(&ds);
	if (Blt_DataTableFindColumn(table, label) >= 0) {
	    label = CopyOfColumnLabel(table, label, &ds);
	}
	result = Blt_DataTableSetColumnLabel(interp, table, cols[i], label);
	Tcl_DStringFree(&ds);
	if (result != TCL_OK) {
	    goto error;
	}
	array = Blt_VecData(vector);
	nElems = Blt_VecLength(vector);
	for (n = 0; n < nElems; n++) {
	    Tcl_Obj *objPtr;
	    long row;

	    if (isInteger) {
		objPtr = Tcl_NewLongObj((long)array[n]);
	    } else {
		objPtr = Tcl_NewDoubleObj(array[n]);
	    }
	    row = Blt_DataTableRowOffset(table, n + 1);
	    result = Blt_DataTableSetValue(interp, table, row, cols[n], objPtr);
	    if (result != TCL_OK) {
		goto error;
	    }
	}
    }
 error:
    Blt_Free(cols);
    return result;
}



/*
 *----------------------------------------------------------------------
 *
 * ImportDataOp --
 *
 *	Parses the given command line and calls one of several
 *	import-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec importDataOps[] =
{
    {"csv",    1, (Blt_Op)ImportDataCsvOp,    3, 0, "dataString ?switches?",},
    {"tree",   1, (Blt_Op)ImportDataTreeOp,   4, 5, "treeName node",},
    {"vector", 1, (Blt_Op)ImportDataVectorOp, 3, 0, "vecName...",},
    {"xml",    1, (Blt_Op)ImportDataXmlOp,    3, 0, "dataString ?switches?",},
};

static int nImportDataOps = sizeof(importDataOps) / sizeof(Blt_OpSpec);

static int
ImportDataOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nImportDataOps, importDataOps, BLT_OP_ARG2,
	objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ImportFileOp --
 *
 *	Parses the given command line and calls one of several
 *	import-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec importFileOps[] =
{
    {"csv",  1, (Blt_Op)ImportFileCsvOp,  3, 0, "fileName ?switches?",},
    {"xml",  1, (Blt_Op)ImportFileXmlOp,  3, 0, "fileName ?switches?",},
};

static int nImportFileOps = sizeof(importFileOps) / sizeof(Blt_OpSpec);

static int
ImportFileOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nImportFileOps, importFileOps, BLT_OP_ARG2,
	objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

/**************** Notify Operations *******************/

/*
 *----------------------------------------------------------------------
 *
 * NotifyDeleteOp --
 *
 *	Deletes one or more notifiers.  
 *
 * Results:
 *	A standard Tcl result.  If a name given doesn't represent
 *	a notifier, then TCL_ERROR is returned and an error message
 *	is left in the interpreter result.
 *
 *---------------------------------------------------------------------- 
 */
static int
NotifyDeleteOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 3; i < objc; i++) {
	Blt_HashEntry *hPtr;
	NotifierInfo *niPtr;

	hPtr = Blt_FindHashEntry(&cmdPtr->notifyTable, Tcl_GetString(objv[i]));
	if (hPtr == NULL) {
	    Tcl_AppendResult(interp, "unknown notifier id \"", 
			     Tcl_GetString(objv[i]), "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	niPtr = Blt_GetHashValue(hPtr);
	Blt_DeleteHashEntry(&cmdPtr->notifyTable, hPtr);
	FreeNotifierInfo(niPtr);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NotifierInfoOp --
 *
 *	Returns the details for a given notifier.  The string id of
 *	the notifier is passed as an argument.  
 *
 * Results:
 *	A standard Tcl result.  If the name given doesn't represent
 *	a notifier, then TCL_ERROR is returned and an error message
 *	is left in the interpreter result.  Otherwise the details
 *	of the notifier handler are returned as a list of three
 *	elements: notifier id, flags, and command.
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
NotifyInfoOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_HashEntry *hPtr;
    NotifierInfo *niPtr;
    char *what;
    Tcl_Obj *listObjPtr, *subListObjPtr, *objPtr;
    int i;
    struct Blt_DataTableNotifierStruct *notifierPtr;

    table = cmdPtr->table;
    hPtr = Blt_FindHashEntry(&cmdPtr->notifyTable, Tcl_GetString(objv[3]));
    if (hPtr == NULL) {
	Tcl_AppendResult(interp, "unknown notifier id \"", 
		Tcl_GetString(objv[3]), "\"", (char *)NULL);
	return TCL_ERROR;
    }

    niPtr = Blt_GetHashValue(hPtr);
    notifierPtr = niPtr->notifier;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    Tcl_ListObjAppendElement(interp, listObjPtr, objv[3]); /* Copy notify Id */

    subListObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    if (notifierPtr->flags & TABLE_NOTIFY_CREATE) {
	objPtr = Tcl_NewStringObj("-create", -1);
	Tcl_ListObjAppendElement(interp, subListObjPtr, objPtr);
    }
    if (notifierPtr->flags & TABLE_NOTIFY_DELETE) {
	objPtr = Tcl_NewStringObj("-delete", -1);
	Tcl_ListObjAppendElement(interp, subListObjPtr, objPtr);
    }
    if (notifierPtr->flags & TABLE_NOTIFY_RELABEL) {
	objPtr = Tcl_NewStringObj("-relabel", -1);
	Tcl_ListObjAppendElement(interp, subListObjPtr, objPtr);
    }
    if (notifierPtr->flags & TABLE_NOTIFY_PACK) {
	objPtr = Tcl_NewStringObj("-pack", -1);
	Tcl_ListObjAppendElement(interp, subListObjPtr, objPtr);
    }
    if (notifierPtr->flags & TABLE_NOTIFY_WHENIDLE) {
	objPtr = Tcl_NewStringObj("-whenidle", -1);
	Tcl_ListObjAppendElement(interp, subListObjPtr, objPtr);
    }
    Tcl_ListObjAppendElement(interp, listObjPtr, subListObjPtr);

    what = (notifierPtr->flags & TABLE_NOTIFY_ROW) ? "row" : "column";
    Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewStringObj(what, -1));
    if (notifierPtr->tag != NULL) {
	Tcl_ListObjAppendElement(interp, listObjPtr, 
			Tcl_NewStringObj(notifierPtr->tag, -1));
    } else {
	Tcl_ListObjAppendElement(interp, listObjPtr, 
	    Tcl_NewLongObj(Blt_DataTableRowIndex(table, notifierPtr->offset)));
    }
    subListObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (i = 0; i < niPtr->cmdc; i++) {
	Tcl_ListObjAppendElement(interp, subListObjPtr, niPtr->cmdv[i]);
    }
    Tcl_ListObjAppendElement(interp, listObjPtr, subListObjPtr);

    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyNamesOp --
 *
 *	Returns the names of all the notifiers in use by this
 *	instance.  Notifiers issues by other instances or object
 *	clients are not reported.  
 *
 * Results:
 *	Always TCL_OK.  A list of notifier names is left in the
 *	interpreter result.
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
NotifyNamesOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    Tcl_Obj *listObjPtr;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (hPtr = Blt_FirstHashEntry(&cmdPtr->notifyTable, &cursor); hPtr != NULL;
	 hPtr = Blt_NextHashEntry(&cursor)) {
	Tcl_Obj *objPtr;
	CONST char *name;

	name = Blt_GetHashKey(&cmdPtr->notifyTable, hPtr);
	objPtr = Tcl_NewStringObj(name, -1);
	Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NotifyOp --
 *
 *	Parses the given command line and calls one of several
 *	notifier-specific operations.
 *	
 * Results:
 *	Returns a standard Tcl result.  It is the result of 
 *	operation called.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec notifyOps[] =
{
    {"delete", 1, (Blt_Op)NotifyDeleteOp, 3, 0, "notifyId...",},
    {"info",   1, (Blt_Op)NotifyInfoOp,   4, 4, "notifyId",},
    {"names",  1, (Blt_Op)NotifyNamesOp,  3, 3, "",},
};

static int nNotifyOps = sizeof(notifyOps) / sizeof(Blt_OpSpec);

static int
NotifyOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nNotifyOps, notifyOps, BLT_OP_ARG2, objc, 
	objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc)(cmdPtr, interp, objc, objv);
    return result;
}

/************* Trace Operations *****************/

/*
 *----------------------------------------------------------------------
 *
 * TraceCreateOp --
 *
 *	Creates a trace for this instance.  Traces represent
 *	list of keys, a bitmask of trace flags, and a command prefix 
 *	to be invoked when a matching trace event occurs.  
 *
 *	The command prefix is parsed and saved in an array of
 *	Tcl_Objs. The qualified name of the instance is saved
 *	also.
 *
 * Results:
 *	A standard Tcl result.  The name of the new trace is
 *	returned in the interpreter result.  Otherwise, if it failed
 *	to parse a switch, then TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 *	$table trace create row column how command
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TraceCreateOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Blt_DataTableIterator rowIter, colIter;
    Blt_DataTableTrace trace;
    TraceInfo *tiPtr;
    char *rowTag, *colTag;
    int flags;
    long row, col;
    
    table = cmdPtr->table;
    if (Blt_DataTableGetRows(interp, table, objv[3], &rowIter) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetColumns(interp, table, objv[4], &colIter) != TCL_OK) {
	return TCL_ERROR;
    }
    flags = GetTraceFlags(Tcl_GetString(objv[5]));
    if (flags < 0) {
	Tcl_AppendResult(interp, "unknown flag in \"", Tcl_GetString(objv[5]), 
		"\"", (char *)NULL);
	return TCL_ERROR;
    }
    row = col = -1;
    colTag = rowTag = NULL;
    if (rowIter.type == TABLE_ITER_RANGE) {
	Tcl_AppendResult(interp, "can't trace range of rows: use a tag", 
		(char *)NULL);
	return TCL_ERROR;
    }
    if (colIter.type == TABLE_ITER_RANGE) {
	Tcl_AppendResult(interp, "can't trace range of columns: use a tag", 
		(char *)NULL);
	return TCL_ERROR;
    }
    if (rowIter.type == TABLE_ITER_INDEX) {
	row = Blt_DataTableFirstRow(&rowIter);
    } else {
	rowTag = (char *)rowIter.tagName;
    }
    if (colIter.type == TABLE_ITER_INDEX) {
	col = Blt_DataTableFirstColumn(&colIter);
    } else {
	colTag = (char *)colIter.tagName;
    }
    tiPtr = Blt_Malloc(sizeof(TraceInfo));
    if (tiPtr == NULL) {
	Tcl_AppendResult(interp, "can't allocate trace: out of memory", 
		(char *)NULL);
	return TCL_ERROR;
    }
    trace = Blt_DataTableCreateTrace(table, row, col, rowTag, colTag, 
	flags, TraceProc, TraceDeleteProc, tiPtr);
    if (trace == NULL) {
	Tcl_AppendResult(interp, "can't create individual trace: out of memory",
		(char *)NULL);
	Blt_Free(tiPtr);
	return TCL_ERROR;
    }
    /* Initialize the trace information structure. */
    tiPtr->cmdPtr = cmdPtr;
    tiPtr->trace = trace;
    tiPtr->tablePtr = &cmdPtr->traceTable;
    {
	Tcl_Obj **elv, **cmdv;
	int elc, i;

	if (Tcl_ListObjGetElements(interp, objv[6], &elc, &elv) != TCL_OK) {
	    return TCL_ERROR;
	}
	/* 
	 * Command + tableName + row + column + flags + NULL
	 */
	cmdv = Blt_Calloc(elc + 1 + 3 + 1, sizeof(Tcl_Obj *));
	for(i = 0; i < elc; i++) {
	    cmdv[i] = elv[i];
	    Tcl_IncrRefCount(cmdv[i]);
	}
	cmdv[i] = Tcl_NewStringObj(cmdPtr->hPtr->key.string, -1);
	Tcl_IncrRefCount(cmdv[i]);
	tiPtr->cmdc = elc;
	tiPtr->cmdv = cmdv;
    }
    {
	int isNew;
	char traceId[200];
	Blt_HashEntry *hPtr;

	do {
	    sprintf(traceId, "trace%d", cmdPtr->nextTraceId++);
	    hPtr = Blt_CreateHashEntry(&cmdPtr->traceTable, traceId, &isNew);
	} while (!isNew);
	tiPtr->hPtr = hPtr;
	Blt_SetHashValue(hPtr, tiPtr);
	Tcl_SetStringObj(Tcl_GetObjResult(interp), traceId, -1);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceDeleteOp --
 *
 *	Deletes one or more traces.  Can be any type of trace.
 *
 * Results:
 *	A standard Tcl result.  If a name given doesn't represent
 *	a trace, then TCL_ERROR is returned and an error message
 *	is left in the interpreter result.
 *
 *	$table trace delete $id...
 *---------------------------------------------------------------------- 
 */
static int
TraceDeleteOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 3; i < objc; i++) {
	Blt_HashEntry *hPtr;
	TraceInfo *tiPtr;

	hPtr = Blt_FindHashEntry(&cmdPtr->traceTable, Tcl_GetString(objv[i]));
	if (hPtr == NULL) {
	    Tcl_AppendResult(interp, "unknown trace \"", 
			     Tcl_GetString(objv[i]), "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	tiPtr = Blt_GetHashValue(hPtr);
	Blt_DataTableDeleteTrace(tiPtr->trace);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceInfoOp --
 *
 *	Returns the details for a given trace.  The name of the trace
 *	is passed as an argument.  The details are returned as a list
 *	of key-value pairs: trace name, tag, row index, keys, flags,
 *	and the command prefix.
 *
 * Results:
 *	A standard Tcl result.  If the name given doesn't represent
 *	a trace, then TCL_ERROR is returned and an error message
 *	is left in the interpreter result.
 *
 *	$table trace info $trace
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TraceInfoOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_HashEntry *hPtr;
    TraceInfo *tiPtr;
    Tcl_Obj *listObjPtr;

    hPtr = Blt_FindHashEntry(&cmdPtr->traceTable, Tcl_GetString(objv[3]));
    if (hPtr == NULL) {
	Tcl_AppendResult(interp, "unknown trace \"", Tcl_GetString(objv[3]), 
		"\"", (char *)NULL);
	return TCL_ERROR;
    }
    tiPtr = Blt_GetHashValue(hPtr);
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    PrintTraceInfo(interp, tiPtr, listObjPtr);
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceNamesOp --
 *
 *	Returns the names of all the traces in use by this instance.
 *	Traces created by other instances or object clients are not
 *	reported.
 *
 * Results:
 *	Always TCL_OK.  A list of trace names is left in the
 *	interpreter result.
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
TraceNamesOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;

    for (hPtr = Blt_FirstHashEntry(&cmdPtr->traceTable, &cursor);
	 hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
	Tcl_AppendElement(interp, Blt_GetHashKey(&cmdPtr->traceTable, hPtr));
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TraceOp --
 *
 * 	This procedure is invoked to process trace operations.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec traceOps[] =
{
    {"create", 1, (Blt_Op)TraceCreateOp, 7, 7, "row column how command",},
    {"delete", 1, (Blt_Op)TraceDeleteOp, 3, 0, "traceId...",},
    {"info", 1, (Blt_Op)TraceInfoOp, 4, 4, "traceId",},
    {"names", 1, (Blt_Op)TraceNamesOp, 3, 3, "",},
};

static int nTraceOps = sizeof(traceOps) / sizeof(Blt_OpSpec);

static int
TraceOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;

    proc = Blt_GetOpFromObj(interp, nTraceOps, traceOps, BLT_OP_ARG2, objc, 
	objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (cmdPtr, interp, objc, objv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * SetOp --
 *
 *	$t set $row $column $value ?row column value?...
 *
 *	Sets one or more key-value pairs for tables.  One or more
 *	tables may be set.  If any of the columns (keys) given don't
 *	already exist, the columns will be automatically created.  The
 *	same holds true for rows.  If a row index is beyond the end of
 *	the table (tags are always in range), new rows are allocated.
 * 
 * Results:
 *	A standard Tcl result. If the tag or index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *---------------------------------------------------------------------- 
 */
static int
SetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;

    if (((objc - 2) % 3) != 0) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", 
		Tcl_GetString(objv[0]), 
		" set ?row column value?...\"", (char *)NULL);
	return TCL_ERROR;
    }
    table = cmdPtr->table;
    for (i = 2; i < objc; i += 3) {
	Blt_DataTableIterator rowIter, colIter;
	long row;

	if (Blt_DataTableGetRows(interp, table, objv[i], &rowIter) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Blt_DataTableGetColumns(interp, table, objv[i+1], &colIter) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    long col;

	    for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
		 col = Blt_DataTableNextColumn(&colIter)) {
		if (Blt_DataTableSetValue(interp, table, row, col, objv[i+2])
		    != TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	}	    
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * UnsetOp --
 *
 *	$t unset row column ?row column?
 *
 *	Unsets one or more values.  One or more tables may be unset
 *	(using tags).  It's not an error if one of the key names
 *	(columns) doesn't exist.  The same holds true for rows.  If a
 *	row index is beyond the end of the table (tags are always in
 *	range), it is also ignored.
 * 
 * Results:
 *	A standard Tcl result. If the tag or index is invalid, 
 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *---------------------------------------------------------------------- 
 */
static int
UnsetOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int i;

    if (((objc - 2) % 2) != 0) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", 
		Tcl_GetString(objv[0]), " unset ?row column?...", 
		(char *)NULL);
	return TCL_ERROR;
    }
    table = cmdPtr->table;
    for (i = 2; i < objc; i += 2) {
	Blt_DataTableIterator rowIter, colIter;
	long row;

	if (Blt_DataTableGetRows(interp, table, objv[i], &rowIter) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Blt_DataTableGetColumns(interp, table, objv[i+1], &colIter) 
	    != TCL_OK) {
	    return TCL_ERROR;
	}
	for (row = Blt_DataTableFirstRow(&rowIter); row >= 0; 
	     row = Blt_DataTableNextRow(&rowIter)) {
	    long col;

	    for (col = Blt_DataTableFirstColumn(&colIter); col >= 0; 
		 col = Blt_DataTableNextColumn(&colIter)) {
		if (Blt_DataTableUnsetValue(interp, table, row, col) 
		    != TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	}	    
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RestoreOp --
 *
 * $table restore $string -overwrite -notags
 * $table restorefile $fileName -overwrite -notags
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
RestoreOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    RestoreData switches;
    char *data;

    table = cmdPtr->table;
    memset((char *)&switches, 0, sizeof(switches));
    if (Blt_ParseSwitches(interp, restoreSwitches, objc - 3, objv + 3, 
		&switches, BLT_SWITCH_DEFAULTS) < 0) {
	return TCL_ERROR;
    }
    data = Tcl_GetString(objv[2]);
    if (strncmp(Tcl_GetString(objv[1]), "restored", 8) == 0) {
	return Blt_DataTableRestore(interp, table, data, switches.flags);
    } else {
	return Blt_DataTableFileRestore(interp, table, data, switches.flags);
    }
    /*NOTREACHED*/
}

static int
WriteRecord(Tcl_Channel channel, Tcl_DString *dsPtr)
{
    int length, nWritten;
    char *line;

    length = Tcl_DStringLength(dsPtr);
    line = Tcl_DStringValue(dsPtr);
#if HAVE_UTF
    nWritten = Tcl_WriteChars(channel, line, length);
#else
    nWritten = Tcl_Write(channel, line, length);
#endif
    if (nWritten != length) {
	return FALSE;
    }
    Tcl_DStringSetLength(dsPtr, 0);
    return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * DumpHeader --
 *
 *	Prints the info associated with a column into a dynamic
 *	string.
 *
 * Results:
 *	None.
 *
 *---------------------------------------------------------------------- 
 */
static int
DumpHeader(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    DumpData *dp,
    long nRows, long nCols)
{
    /* i rows columns ctime mtime \n */
    Tcl_DStringAppendElement(dp->dsPtr, "i");

    /* # of rows and columns may be a subset of the table. */
    Tcl_DStringAppendElement(dp->dsPtr, Blt_Ltoa(nRows));
    Tcl_DStringAppendElement(dp->dsPtr, Blt_Ltoa(nCols));

    Tcl_DStringAppendElement(dp->dsPtr, Blt_Ltoa(0));
    Tcl_DStringAppendElement(dp->dsPtr, Blt_Ltoa(0));
    Tcl_DStringAppend(dp->dsPtr, "\n", 1);
    if (dp->channel != NULL) {
	return WriteRecord(dp->channel, dp->dsPtr);
    }
    return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * DumpValue --
 *
 *	Retrieves all tags for a given row or column into a tcl list.  
 *
 * Results:
 *	None.
 *
 *---------------------------------------------------------------------- 
 */
static int
DumpValue(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    DumpData *dp,
    long row, long col)
{
    Tcl_Obj *objPtr;

    if (Blt_DataTableGetValue(interp, table, row, col, &objPtr) != TCL_OK) {
	return FALSE;
    }
    if (objPtr == NULL) {
	return TRUE;
    }
    /* d row column value \n */
    Tcl_DStringAppendElement(dp->dsPtr, "d");
    Tcl_DStringAppendElement(dp->dsPtr, 
		Blt_Ltoa(Blt_DataTableRowIndex(table, row)));
    Tcl_DStringAppendElement(dp->dsPtr, 
		Blt_Ltoa(Blt_DataTableColumnIndex(table, col)));
    Tcl_DStringAppendElement(dp->dsPtr, Tcl_GetString(objPtr));
    Tcl_DStringAppend(dp->dsPtr, "\n", 1);
    if (dp->channel != NULL) {
	return WriteRecord(dp->channel, dp->dsPtr);
    }
    return TRUE;
}

/*
 *----------------------------------------------------------------------
 *
 * DumpColumn --
 *
 *	Prints the info associated with a column into a dynamic
 *	string.
 *
 *---------------------------------------------------------------------- 
 */
static int
DumpColumn(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    DumpData *dp,
    long col)
{
    Blt_Chain *chainPtr;
    Blt_ChainLink *linkPtr;
    CONST char *name;
    int type;

    /* c index label type tags \n */
    Tcl_DStringAppendElement(dp->dsPtr, "c");
    Tcl_DStringAppendElement(dp->dsPtr, 
		Blt_Ltoa(Blt_DataTableColumnIndex(table, col)));
    Tcl_DStringAppendElement(dp->dsPtr, ColumnLabel(table, col));
    type = Blt_DataTableColumnType(table, col);
    name = Blt_DataTableNameOfColumnType(type);
    if (name == NULL) {
	name = "";
    }
    Tcl_DStringAppendElement(dp->dsPtr, name);

    chainPtr = Blt_DataTableColumnTags(table, col);
    Tcl_DStringStartSublist(dp->dsPtr);
    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	char *tagName;

	tagName = Blt_ChainGetValue(linkPtr);
	Tcl_DStringAppendElement(dp->dsPtr, tagName);
    }
    Blt_ChainDestroy(chainPtr);
    Tcl_DStringEndSublist(dp->dsPtr);
    Tcl_DStringAppend(dp->dsPtr, "\n", 1);
    if (dp->channel != NULL) {
	return WriteRecord(dp->channel, dp->dsPtr);
    }
    return TRUE;
}

/*
 *----------------------------------------------------------------------
 *
 * DumpRow --
 *
 *	Prints the info associated with a row into a dynamic
 *	string.
 *
 *---------------------------------------------------------------------- 
 */
static int
DumpRow(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    DumpData *dp,
    long row)
{
    Blt_ChainLink *linkPtr;
    Blt_Chain *chainPtr;

    /* r index label tags \n */
    Tcl_DStringAppendElement(dp->dsPtr, "r");
    Tcl_DStringAppendElement(dp->dsPtr, 
	Blt_Ltoa(Blt_DataTableRowIndex(table, row)));
    Tcl_DStringAppendElement(dp->dsPtr, RowLabel(table, row));
    Tcl_DStringStartSublist(dp->dsPtr);
    chainPtr = Blt_DataTableRowTags(table, row);
    for (linkPtr = Blt_ChainFirstLink(chainPtr); linkPtr != NULL;
	 linkPtr = Blt_ChainNextLink(linkPtr)) {
	char *tagName;

	tagName = Blt_ChainGetValue(linkPtr);
	Tcl_DStringAppendElement(dp->dsPtr, tagName);
    }
    Blt_ChainDestroy(chainPtr);
    Tcl_DStringEndSublist(dp->dsPtr);
    Tcl_DStringAppend(dp->dsPtr, "\n", 1);
    if (dp->channel != NULL) {
	return WriteRecord(dp->channel, dp->dsPtr);
    }
    return TRUE;
}


/*
 *----------------------------------------------------------------------
 *
 * DumpTable --
 *
 *	Dumps data from the given table based upon the row and column
 *	maps provided which describe what rows and columns are to be
 *	dumped. The dump information is written to the file named. If
 *	the file name starts with an '@', then it is the name of an
 *	already opened channel to be used.
 *	
 * Results:
 *	A standard Tcl result.  If the dump was successful, TCL_OK
 *	is returned.  Otherwise, TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 * Side Effects:
 *	Dump information is written to the named file.
 *
 *---------------------------------------------------------------------- 
 */
static int
DumpTable(
    Tcl_Interp *interp, 
    Blt_DataTable table, 
    DumpData *dp)
{
    int result;
    Blt_ChainLink *linkPtr;

    result = DumpHeader(interp, table, dp, 
		Blt_ChainGetLength(dp->rowChainPtr),
		Blt_ChainGetLength(dp->colChainPtr));
    for (linkPtr = Blt_ChainFirstLink(dp->colChainPtr); 
	 (result) && (linkPtr != NULL); linkPtr = Blt_ChainNextLink(linkPtr)) {
	long col;

	col = (long)Blt_ChainGetValue(linkPtr);
	result = DumpColumn(interp, table, dp, col);
    }
    for (linkPtr = Blt_ChainFirstLink(dp->rowChainPtr); 
	 (result) && (linkPtr != NULL); linkPtr = Blt_ChainNextLink(linkPtr)) {
	long row;

	row = (long)Blt_ChainGetValue(linkPtr);
	result = DumpRow(interp, table, dp, row);
    }
    for (linkPtr = Blt_ChainFirstLink(dp->rowChainPtr); 
	 (result) && (linkPtr != NULL); linkPtr = Blt_ChainNextLink(linkPtr)) {
	long row;
	Blt_ChainLink *linkPtr2;

	row = (long)Blt_ChainGetValue(linkPtr);
	for (linkPtr2 = Blt_ChainFirstLink(dp->colChainPtr); 
	     (result) && (linkPtr2 != NULL); 
	     linkPtr2 = Blt_ChainNextLink(linkPtr2)) {
	    long col;

	    col = (long)Blt_ChainGetValue(linkPtr2);
	    result = DumpValue(interp, table, dp, row, col);
	}
    }
    return (result) ? TCL_OK : TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DumpOp --
 *
 * $table dumpdata -rows {} -columns {} 
 * $table dumpfile fileName -rows {} -columns {} 
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
DumpOp(cmdPtr, interp, objc, objv)
    DataTableCmd *cmdPtr;
    Tcl_Interp *interp;
    int objc;			/* Not used. */
    Tcl_Obj *CONST *objv;
{
    Blt_DataTable table;
    DumpData switches;
    int result;
    Tcl_DString ds;
    int closeChannel;
    Tcl_Channel channel;

    closeChannel = FALSE;
    channel = NULL;
    if (strncmp(Tcl_GetString(objv[1]), "dumpf", 5) == 0) {
	char *fileName;

	fileName = Tcl_GetString(objv[2]);

	closeChannel = TRUE;
	if ((fileName[0] == '@') && (fileName[1] != '\0')) {
	    int mode;
	    
	    channel = Tcl_GetChannel(interp, fileName+1, &mode);
	    if (channel == NULL) {
		return TCL_ERROR;
	    }
	    if ((mode & TCL_WRITABLE) == 0) {
		Tcl_AppendResult(interp, "channel \"", fileName, 
				 "\" not opened for writing", (char *)NULL);
		return TCL_ERROR;
	    }
	    closeChannel = FALSE;
	} else {
	    channel = Tcl_OpenFileChannel(interp, fileName, "w", 0666);
	    if (channel == NULL) {
		return TCL_ERROR;
	    }
	}
	objv++, objc--;
    }
    table = cmdPtr->table;
    result = TCL_ERROR;
    memset(&switches, 0, sizeof(DumpData));
    switches.channel = channel;
    switches.dsPtr = &ds;

    if (Blt_ParseSwitches(interp, dumpSwitches, objc - 2, objv + 2, 
		&switches, BLT_SWITCH_DEFAULTS) < 0) {
	goto error;
    }
    if (switches.rows == NULL) {
	switches.rowChainPtr = AllRows(table);
    } else {
	Blt_Chain *chainPtr;
	Tcl_Obj **elv;
	int elc;

	if (Tcl_ListObjGetElements(interp, switches.rows, &elc, &elv)
	    !=TCL_OK) {
	    goto error;
	}
	chainPtr = Blt_ChainCreate();
	if (GetRows(interp, table, elc, elv, chainPtr) != TCL_OK) {
	    Blt_ChainDestroy(chainPtr);
	    goto error;
	}
	if (Blt_ChainGetLength(chainPtr) == 0) {
	    Blt_ChainDestroy(chainPtr);
	    Tcl_AppendResult(interp, "no rows specified.", (char *)NULL);
	    goto error;
	}
	switches.rowChainPtr = chainPtr;
    }
    if (switches.cols == NULL) {
	switches.colChainPtr = AllColumns(table);
    } else {
	Blt_Chain *chainPtr;
	Tcl_Obj **elv;
	int elc;

	if (Tcl_ListObjGetElements(interp, switches.cols, &elc, &elv)!=TCL_OK) {
	    goto error;
	}
	chainPtr = Blt_ChainCreate();
	if (GetColumns(interp, table, elc, elv, chainPtr) != TCL_OK) {
	    Blt_ChainDestroy(chainPtr);
	    goto error;
	}
	if (Blt_ChainGetLength(chainPtr) == 0) {
	    Blt_ChainDestroy(chainPtr);
	    Tcl_AppendResult(interp, "no columns specified.", (char *)NULL);
	    goto error;
	}
	switches.colChainPtr = chainPtr;
    }
    Tcl_DStringInit(&ds);
    result = DumpTable(interp, table, &switches);
    if ((switches.channel == NULL) && (result == TCL_OK)) {
	Tcl_DStringResult(interp, &ds);
    }
    Tcl_DStringFree(&ds);
 error:
    if (closeChannel) {
	Tcl_Close(interp, channel);
    }
    if (switches.rowChainPtr != NULL) {
	Blt_ChainDestroy(switches.rowChainPtr);
    }
    if (switches.colChainPtr != NULL) {
	Blt_ChainDestroy(switches.colChainPtr);
    }
    return result;
}


/*
 *----------------------------------------------------------------------
 *
 * EmptyValueOp --
 *
 *	$t emptyvalue ?$value?
 *
 *---------------------------------------------------------------------- 
 */
static int
EmptyValueOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Tcl_SetObjResult(interp, cmdPtr->emptyValueObjPtr);
    if (objc == 3) {
	Tcl_IncrRefCount(objv[3]);
	Tcl_DecrRefCount(cmdPtr->emptyValueObjPtr);
	cmdPtr->emptyValueObjPtr = objv[2];
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ExistsOp --
 *
 *	$t exists $row $column
 *
 *---------------------------------------------------------------------- 
 */
static int
ExistsOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    int bool;
    long row, col;

    table = cmdPtr->table;
    if (Blt_DataTableGetRow(interp, table, objv[3], &row) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetColumn(interp, table, objv[4], &col) != TCL_OK) {
	return TCL_ERROR;
    }
    bool = Blt_DataTableValueExists(table, row, col);
    Tcl_SetBooleanObj(Tcl_GetObjResult(interp), bool);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * GetOp --
 *
 *	Retrieves the value from a given table for a designated
 *	column.  It's an error if the column key or row table is
 *	invalid.  If no key argument is provided then, all key-value
 *	pairs in the table are returned.  If a value is empty (the
 *	Tcl_Obj value is NULL), then the table command's "empty"
 *	string representation will be used.
 *	
 *	A third argument may be provided as the default return value.
 *	If no value exists for the given key, then this value is
 *	returned.
 * 
 * Results:
 *	A standard Tcl result. If the tag or index is invalid, 

 *	TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *	
 *	$t get row column ?defValue?
 *
 *---------------------------------------------------------------------- 
 */
static int
GetOp(
    DataTableCmd *cmdPtr, 
    Tcl_Interp *interp, 
    int objc, 
    Tcl_Obj *CONST *objv)
{
    Blt_DataTable table;
    Tcl_Obj *valueObjPtr;
    long row, col;

    table = cmdPtr->table;
    if (Blt_DataTableGetRow(interp, table, objv[2], &row) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetColumn(interp, table, objv[3], &col) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Blt_DataTableGetValue(interp, table, row, col, &valueObjPtr)!=TCL_OK) {
	return TCL_ERROR;
    }
    if ((valueObjPtr == NULL) && (objc == 5)) {
	valueObjPtr = objv[4];
    }
    if (valueObjPtr == NULL) {
	valueObjPtr = cmdPtr->emptyValueObjPtr;
    }
    Tcl_SetObjResult(interp, valueObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AttachOp --
 *
 *---------------------------------------------------------------------- 
 */
static int
AttachOp(
    DataTableCmd *cmdPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    if (objc == 3) {
	CONST char *qualName;
	Blt_ObjectName objName;
	Blt_DataTable table;
	Tcl_DString ds;
	int result;

	if (!Blt_ParseObjectName(interp, Tcl_GetString(objv[2]), &objName, 0)) {
	    return TCL_ERROR;
	}
	qualName = Blt_MakeQualifiedName(&objName, &ds);
	result = Blt_DataTableGetToken(interp, qualName, &table);
	Tcl_DStringFree(&ds);
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}
	if (cmdPtr->table != NULL) {
	    Blt_HashEntry *hPtr;
	    Blt_HashSearch cursor;
	    
	    Blt_DataTableReleaseToken(cmdPtr->table);
	    /* Free the current set of tags, traces, and notifiers. */
	    for (hPtr = Blt_FirstHashEntry(&cmdPtr->traceTable, &cursor); 
		 hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
		TraceInfo *tiPtr;

		tiPtr = Blt_GetHashValue(hPtr);
		Blt_DataTableDeleteTrace(tiPtr->trace);
	    }
	    Blt_DeleteHashTable(&cmdPtr->traceTable);
	    Blt_InitHashTable(&cmdPtr->traceTable, TCL_STRING_KEYS);
	    for (hPtr = Blt_FirstHashEntry(&cmdPtr->notifyTable, &cursor); 
		hPtr != NULL; hPtr = Blt_NextHashEntry(&cursor)) {
		NotifierInfo *niPtr;

		niPtr = Blt_GetHashValue(hPtr);
		FreeNotifierInfo(niPtr);
	    }
	    Blt_DeleteHashTable(&cmdPtr->notifyTable);
	    Blt_InitHashTable(&cmdPtr->notifyTable, TCL_STRING_KEYS);
	}
	cmdPtr->table = table;
    }
    Tcl_SetStringObj(Tcl_GetObjResult(interp), 
		     Blt_DataTableName(cmdPtr->table), -1);
    return TCL_OK;
}

static Blt_OpSpec tableOps[] =
{
    {"array",      1, (Blt_Op)ArrayOp,      3, 0, "op args",},
    {"attach",     1, (Blt_Op)AttachOp,     3, 0, "args",},
    {"column",     3, (Blt_Op)ColumnOp,     3, 0, "op args",},
#ifdef notdef
    {"dup",        1, (Blt_Op)DupOp,        3, 0, "args",},
#endif
    {"dumpdata",   5, (Blt_Op)DumpOp,       2, 0, "?switches?",},
    {"dumpfile",   5, (Blt_Op)DumpOp,       3, 0, "fileName ?switches?",},
    {"emptyvalue", 2, (Blt_Op)EmptyValueOp, 2, 3, "?newValue?",},
    {"exists",     3, (Blt_Op)ExistsOp,     4, 4, "row column",},
    {"exportdata", 7, (Blt_Op)ExportDataOp, 3, 0, "format args",},
    {"exportfile", 7, (Blt_Op)ExportFileOp, 4, 0, "format fileName args",},
#ifdef notdef
    {"find", 4, (Blt_Op)FindOp, 3, 0, "node ?switches?",},
#endif
    {"get",        1, (Blt_Op)GetOp,        4, 5, "row column ?defValue?",},
    {"importdata", 1, (Blt_Op)ImportDataOp, 4, 0, "format dataString args",},
    {"importfile", 1, (Blt_Op)ImportFileOp, 4, 0, "format fileName args",},
    {"notify",     1, (Blt_Op)NotifyOp,     2, 0, "op args",},
#ifdef notdef
    {"pack",       1, (Blt_Op)PackOp,       3, 0, "args",},
#endif
    {"restoredata",2, (Blt_Op)RestoreOp,    4, 0, "dataString ?switches?",},
    {"restorefile",2, (Blt_Op)RestoreOp,    4, 0, "fileName ?switches?",},
    {"row",        2, (Blt_Op)RowOp,        3, 0, "op args",},
    {"set",        1, (Blt_Op)SetOp,        3, 0, "?row column value?...",},
    {"trace",      2, (Blt_Op)TraceOp,      2, 0, "op args",},
    {"unset",      1, (Blt_Op)UnsetOp,      4, 0, "row column ?row column?",},
#ifdef notplanned
    {"append", 2, (Blt_Op)AppendOp, 3, 0, "table... ?flags...?",},
    {"-ancestor", 2, (Blt_Op)AncestorOp, 4, 4, "node1 node2",},
    {"-apply", 1, (Blt_Op)ApplyOp, 3, 0, "first last ?switches?",},
    {"column", 2, (Blt_Op)ColumnOp, 2, 0, "args", },
    {"copy", 2, (Blt_Op)CopyOp, 4, 0, 
	"srcNode ?destTable? destNode ?switches?",},
    {"index", 3, (Blt_Op)IndexOp, 3, 3, "name",},
    {"move", 1, (Blt_Op)MoveOp, 4, 0, "node newParent ?switches?",},
    {"-next", 4, (Blt_Op)NextOp, 3, 3, "node",},
    {"-position", 2, (Blt_Op)PositionOp, 3, 0, "?switches? node...",},
    {"-previous", 5, (Blt_Op)PreviousOp, 3, 3, "node",},
    {"range", 2, (Blt_Op)ChildrenOp, 4, 4, "first last",},
    {"width", 2, (Blt_Op)SizeOp, 3, 3, "?newWidth?",},
    {"length", 2, (Blt_Op)SizeOp, 3, 3, "?newLength?",},
    {"type", 2, (Blt_Op)TypeOp, 4, 4, "key",},
#endif
};

static int nTableOps = sizeof(tableOps) / sizeof(Blt_OpSpec);

/*
 * --------------------------------------------------------------
 *
 * DataTableInstObjCmd --
 *
 * 	This procedure is invoked to process commands on behalf of
 *	the instance of the table-object.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 * --------------------------------------------------------------
 */
static int
DataTableInstObjCmd(
    ClientData clientData,	/* Pointer to the command structure. */
    Tcl_Interp *interp,		/* Interpreter to report errors back to. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST *objv)	/* Vector of argument strings. */
{
    Blt_Op proc;
    DataTableCmd *cmdPtr = clientData;
    int result;

    proc = Blt_GetOpFromObj(interp, nTableOps, tableOps, BLT_OP_ARG1, objc, 
	objv, BLT_OP_LINEAR_SEARCH);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    Tcl_Preserve(cmdPtr);
    result = (*proc) (cmdPtr, interp, objc, objv);
    Tcl_Release(cmdPtr);
    return result;
}

/*
 * ----------------------------------------------------------------------
 *
 * DataTableInstDeleteProc --
 *
 *	Deletes the command associated with the table.  This is
 *	called only when the command associated with the table is
 *	destroyed.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The table object is released and bookkeeping data for traces
 *	and notifiers are freed.
 *
 * ----------------------------------------------------------------------
 */
static void
DataTableInstDeleteProc(ClientData clientData)
{
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    DataTableCmd *cmdPtr = clientData;

    for (hPtr = Blt_FirstHashEntry(&cmdPtr->traceTable, &cursor); hPtr != NULL;
	 hPtr = Blt_NextHashEntry(&cursor)) {
	TraceInfo *tiPtr;

	tiPtr = Blt_GetHashValue(hPtr);
	Blt_DataTableDeleteTrace(tiPtr->trace);
    }
    Blt_DeleteHashTable(&cmdPtr->traceTable);
    for (hPtr = Blt_FirstHashEntry(&cmdPtr->notifyTable, &cursor); hPtr != NULL;
	 hPtr = Blt_NextHashEntry(&cursor)) {
	NotifierInfo *niPtr;

	niPtr = Blt_GetHashValue(hPtr);
	FreeNotifierInfo(niPtr);
    }
    Tcl_DecrRefCount(cmdPtr->emptyValueObjPtr);
    Blt_DeleteHashTable(&cmdPtr->notifyTable);
    if (cmdPtr->hPtr != NULL) {
	Blt_DeleteHashEntry(cmdPtr->tablePtr, cmdPtr->hPtr);
    }
    Blt_DataTableReleaseToken(cmdPtr->table);
    Blt_Free(cmdPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * DataTableCreateOp --
 *
 *	Creates a new instance of a table object.  This routine
 *	ensures that instance and object names are the same.  For
 *	example, you can't create an instance with the name of an
 *	object that already exists.  And because each instance has a
 *	Tcl command associated with it that is used to access the
 *	object, we also check more generally that is it also not the
 *	name of an existing Tcl command.  
 *
 *	Instance names as namespace-qualified.  If no namespace is
 *	designated, it is assumed that instance is to be created in
 *	the current namespace (not the global namespace).
 *	
 * Results:
 *	A standard Tcl result.  If the instance is successfully created,
 *	the namespace-qualified name of the instance is returned. If not,
 *	then TCL_ERROR is returned and an error message is left in the 
 *	interpreter result.
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
DataTableCreateOp(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    CONST char *instName;
    Tcl_DString ds;
    Blt_DataTable table;

    instName = NULL;
    if (objc == 3) {
	instName = Tcl_GetString(objv[2]);
    }
    Tcl_DStringInit(&ds);
    if (instName == NULL) {
	instName = GenerateName(interp, "", "", &ds);
    } else {
	char *p;

	p = strstr(instName, "#auto");
	if (p != NULL) {
	    *p = '\0';
	    instName = GenerateName(interp, instName, p + 5, &ds);
	    *p = '#';
	} else {
	    Blt_ObjectName objName;
	    Tcl_CmdInfo cmdInfo;

	    /* 
	     * Parse the command and put back so that it's in a consistent
	     * format.  
	     *
	     *	t1         <current namespace>::t1
	     *	n1::t1     <current namespace>::n1::t1
	     *	::t1	   ::t1
	     *  ::n1::t1   ::n1::t1
	     */
	    if (!Blt_ParseObjectName(interp, instName, &objName, 0)) {
		return TCL_ERROR;
	    }
	    instName = Blt_MakeQualifiedName(&objName, &ds);
	    /* 
	     * Check if the command already exists. 
	     */
	    if (Tcl_GetCommandInfo(interp, (char *)instName, &cmdInfo)) {
		Tcl_AppendResult(interp, "a command \"", instName,
				 "\" already exists", (char *)NULL);
		goto error;
	    }
	    if (Blt_DataTableExists(interp, instName)) {
		Tcl_AppendResult(interp, "a table \"", instName, 
			"\" already exists", (char *)NULL);
		goto error;
	    }
	} 
    } 
    if (instName == NULL) {
	goto error;
    }
    if (Blt_DataTableCreate(interp, instName, &table) == TCL_OK) {
	DataTableCmd *cmdPtr;

	cmdPtr = NewDataTableCmd(interp, table, instName);
	Tcl_SetStringObj(Tcl_GetObjResult(interp), instName, -1);
	Tcl_DStringFree(&ds);
	return TCL_OK;
    }
 error:
    Tcl_DStringFree(&ds);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DataTableDestroyOp --
 *
 *	Deletes one or more instances of table objects.  The deletion
 *	process is done by deleting the Tcl command associated with
 *	the object.  
 *	
 * Results:
 *	A standard Tcl result.  If one of the names given doesn't
 *	represent an instance, TCL_ERROR is returned and an error
 *	message is left in the interpreter result.
 *
 *---------------------------------------------------------------------- 
 */
static int
DataTableDestroyOp(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 2; i < objc; i++) {
	DataTableCmd *cmdPtr;

	cmdPtr = GetDataTableCmd(interp, Tcl_GetString(objv[i]));
	if (cmdPtr == NULL) {
	    Tcl_AppendResult(interp, "can't find table \"", 
		Tcl_GetString(objv[i]), "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	Tcl_DeleteCommandFromToken(interp, cmdPtr->cmdToken);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DataTableNamesOp --
 *
 *	Returns the names of all the table-object instances matching a
 *	given pattern.  If no pattern argument is provided, then all
 *	object names are returned.  The names returned are namespace
 *	qualified.
 *	
 * Results:
 *	Always returns TCL_OK.  The names of the matching objects is
 *	returned via the interpreter result.
 *
 *	$table names ?pattern?
 *
 *---------------------------------------------------------------------- 
 */
/*ARGSUSED*/
static int
DataTableNamesOp(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    DataTableCmdInterpData *dataPtr = clientData;
    Blt_HashEntry *hPtr;
    Blt_HashSearch cursor;
    Tcl_Obj *listObjPtr;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **) NULL);
    for (hPtr = Blt_FirstHashEntry(&dataPtr->instTable, &cursor); hPtr != NULL;
	 hPtr = Blt_NextHashEntry(&cursor)) {
	CONST char *name;
	int match;
	int i;

	name = Blt_GetHashKey(&dataPtr->instTable, hPtr);
	match = TRUE;
	for (i = 2; i < objc; i++) {
	    match = Tcl_StringMatch(name, Tcl_GetString(objv[i]));
	    if (match) {
		break;
	    }
	}
	if (!match) {
	    continue;
	}
	Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewStringObj(name, -1));
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DataTableObjCmd --
 *
 *---------------------------------------------------------------------- 
 */
static Blt_OpSpec tableCmdOps[] =
{
    {"create",  1, (Blt_Op)DataTableCreateOp, 2, 3, "?name?",},
    {"destroy", 1, (Blt_Op)DataTableDestroyOp, 3, 0, "name...",},
    {"names",   1, (Blt_Op)DataTableNamesOp, 2, 3, "?pattern?...",},
};

static int nCmdOps = sizeof(tableCmdOps) / sizeof(Blt_OpSpec);

/*ARGSUSED*/
static int
DataTableObjCmd(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;

    proc = Blt_GetOpFromObj(interp, nCmdOps, tableCmdOps, BLT_OP_ARG1, objc, 
	objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    return (*proc) (clientData, interp, objc, objv);
}

/*
 * -----------------------------------------------------------------------
 *
 * DataTableInterpDeleteProc --
 *
 *	This is called when the interpreter hosting the "dtable" command
 *	is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Removes the hash table managing all table names.
 *
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
static void
DataTableInterpDeleteProc(
    ClientData clientData,	/* Interpreter-specific data. */
    Tcl_Interp *interp)
{
    DataTableCmdInterpData *dataPtr = clientData;

    /* All table instances should already have been destroyed when
     * their respective Tcl commands were deleted. */
    Blt_DeleteHashTable(&dataPtr->instTable);
    Tcl_DeleteAssocData(interp, DATATABLE_THREAD_KEY);
    Blt_Free(dataPtr);
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_DataTableInit --
 *
 *	This procedure is invoked to initialize the "dtable" command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Creates the new command and adds a new entry into a global Tcl
 *	associative array.
 *
 * ------------------------------------------------------------------------
 */
int
Blt_DataTableInit(Tcl_Interp *interp)
{
    static Blt_InitCmdSpec cmdSpec = { "datatable", DataTableObjCmd, };

    cmdSpec.clientData = GetDataTableCmdInterpData(interp);
    return Blt_InitCmd(interp, "blt", &cmdSpec);
}

/* Dump table to dbm */
/* Convert node data to datablock */

#endif /* NO_DATATABLE */
