unit module Terminal::ANSI:ver<0.0.4>;

constant CSI = "\e[";

sub nums(*@n) {
 # trim trailing nils?
 @n.map({$_ // ''}).join(';')
}

sub csi($str) {
  print-or-buffer CSI ~ $str;
}

sub esc($str) {
  print-or-buffer "\e" ~ $str;
}

sub print-or-buffer($str) {
  if ($*buffer) {
    $*out ~= $str;
  } else {
    print $str;
  }
}


#| Clear the screen.
sub clear-screen is export {
  csi '2J';
}

#| Move the cursor to line, column.
sub move-to($l,$c = 1) is export {
  csi nums($l,$c) ~ "H";
}

#| Move to home (0,0).
sub home is export {
  csi 'H';
}

my $scroll-bottom;
my $scroll-top;

#| Set scroll region to top, bottom.
sub set-scroll-region($top,$bottom) is export {
  $scroll-bottom = $bottom;
  $scroll-top = $top;
  csi nums($top,$bottom) ~ "r";
}

#| Reset the scroll region.
sub reset-scroll-region is export {
  $scroll-bottom = Nil;
  $scroll-top = Nil;
  csi 'r';
}

#| Scroll down 1.
sub scroll-down is export {
  move-to $scroll-top, 1 if $scroll-top;
  esc "M";
}

#| Scroll up 1.
sub scroll-up is export {
  move-to $scroll-bottom, 1 if $scroll-bottom;
  esc "D";
}

#| Hide the cursor.
sub hide-cursor is export {
  csi '?25l';
}

#| Show the cursor.
sub show-cursor is export {
  csi "?25h";
}

#| Save screen state (smcup).
sub save-screen is export {
  csi '?1049h'
}

#| Restore screen (rmcup).
sub restore-screen is export {
  csi '?1049l'
}

#| Turn off cursor (civis).
sub cursor-off is export {
  csi '?25l'
}

#| Turn on cursor (cnorm).
sub cursor-on is export {
  csi '?25h';
}

#| Move cursor down.
sub cursor-down($amt = 1) is export {
  csi $amt ~ 'B';
}

#| Move cursor up.
sub cursor-up($amt = 1) is export {
  csi $amt ~ 'A';
}

#| Move cursor right.
sub cursor-right($amt = 1) is export {
  csi $amt ~ 'C';
}

#| Move cursor left.
sub cursor-left($amt = 1) is export {
  csi $amt ~ 'D';
}

#| Atomic move + print.
sub print-at($r,$c,$str) is export {
  csi nums($r,$c) ~ "H" ~ $str;
}

#| Set fg color to $n.
sub set-fg-color($n) is export {
  csi "38;5;" ~ $n ~ "m";
}

#| Set bg color to $n.
sub set-bg-color($n) is export {
  csi "48;5;" ~ $n ~ "m";
}

#| Save the cursor position.
sub save-cursor is export {
  csi ~ 's';
}

#| restore the cursor position.
sub restore-cursor is export {
  csi ~ 'u';
}

#| Move to start of line.
sub start-of-line is export {
  print-or-buffer "\r"
}

#| Erase to end of line.
sub erase-to-end-of-line is export {
  csi '2J';
}

#| Reverse video
sub reverse-video is export {
  csi '7m';
}

#| Normal video
sub normal-video is export {
  csi '0m';
}

#| Bold (increased intensity)
sub bold is export {
  csi '1m';
}

#| Faint (lower intensity)
sub faint is export {
  csi '2m';
}

#| Get strings instead of printing them.
sub tget(&callable) is export {
  my $*buffer = True;
  my $*out = "";
  callable;
  $*out;
}

#| Combine a series of outputs into one.
sub atomically(&callable) is export {
  die "cannot nest atomically" with CALLERS::{'$*out'};
  my $*out = '';
  my $*buffer = True;
  callable;
  print $*out;
}

