package Role::HasPayload::Merged;
{
  $Role::HasPayload::Merged::VERSION = '0.006';
}
use Moose::Role;
# ABSTRACT: merge autogenerated payload with constructor-specified payload


use Role::HasPayload::Meta::Attribute::Payload;

has payload => (
  reader  => '_payload_to_merge',
  isa     => 'HashRef',
  default => sub {  {}  },
);

sub payload {
  my ($self) = @_;

  my @attrs = grep { $_->does('Role::HasPayload::Meta::Attribute::Payload') }
              $self->meta->get_all_attributes;

  my %payload = map {;
    my $method = $_->get_read_method;
    ($_->name => $self->$method)
  } @attrs;

  my $manual_payload = $self->_payload_to_merge;
  KEY: for my $key (keys %$manual_payload) {
    if (exists $payload{ $key }) {
      Carp::carp("declining to override automatic payload entry $key");
      next KEY;
    }

    $payload{ $key } = $manual_payload->{ $key };
  }

  return \%payload;
}

no Moose::Role;
1;

__END__

=pod

=head1 NAME

Role::HasPayload::Merged - merge autogenerated payload with constructor-specified payload

=head1 VERSION

version 0.006

=head1 SYNOPSIS

  package Example;
  use Moose;

  with qw(Role::HasPayload::Merged);

  sub Payload { 'Role::HasPayload::Meta::Attribute::Payload' }

  has height => (
    is => 'ro',
    traits   => [ Payload ],
  );

  has width => (
    is => 'ro',
    traits   => [ Payload ],
  );

  has color => (
    is => 'ro',
  );

...then...

  my $example = Example->new({
    height  => 10,
    width   => 20,
    color   => 'blue',
    payload => { depth => 30 },
  });

  $example->payload; # { height => 10, width => 20, depth => 30 }

=head1 DESCRIPTION

Role::HasPayload::Merged provides a C<payload> method and a C<payload>
attribute.  It computes the result of the C<payload> method when it's called,
first by gathering the values of attributes marked with
Role::HasPayload::Meta::Attribute::Payload, then by merging in the contents of
the C<payload> attribute (provided at construction).

If an entry in the constructor-provided payload already exists in the
autogenerated payload, a warning is issued and the autogenerated value is used.

For a bit more on the autogenerated payload, see L<Role::HasPayload::Auto>.

This role is especially useful when combined with L<Role::HasMessage::Errf>.

=head1 AUTHOR

Ricardo Signes <rjbs@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2013 by Ricardo Signes.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
