
#include "pv.h"

#define	ESC	27	/* Escape key ASCII code */


void
handle_events(windows)
	windat_t * windows;
/* DESCRIPTION:  handle_events processes events that are related to the windows
described by the "windows" data structure.  It calls the appropriate window
action function when an event is received.  If the Escape key is pressed, a
pop-up menu is posted confirming the user's desire to terminate the program.
*/
{
	long dev_id;
	short data;
	int win_num;
	windat_t * window;


	/* Loop until the user indicates the desire to quit */
	while (!quit) {
		handle_event(windows);
	}
}


void
handle_event(windows)
	windat_t * windows;
{
	long dev_id;
	short data;

	windat_t * window;
	int animated;


	/* Clear the global animation flag so that we know whether we've */
	/* already processed at least one of TIMER0 interrupts. */
	animated = FALSE;

	/* Clear all of the window "redrawn" flags so that we know which */
	/* are really in need of a redraw */
	for (window = windows; window != NULL; window = window->next) {
		window->redrawn = TRUE;
	}

	/* While events are backlogged, keep processing */
	while (qtest()) {
		dev_id = qread(&data);
		switch(dev_id){
		case INPUTCHANGE : 
			if (get_windat(windows, (int) data) != NULL)
				current.menu_id = (int) data;
			break;
		case KEYBD :
			switch (data) {
			case '+' :
				forward1(0);
				break;
			case '-' :
				reverse1(0);
				break;
			case 'p' :
			case 'P' :
				printf("\nat abs %f %f %f\n",
				       at[X], at[Y], at[Z]);
				printf("from abs %f %f %f %f\n",
				       from[X], from[Y], from[Z], twist/10.0);
				break;
			case ESC :
				if (dopup(quit_m) == 100)
					quit = TRUE;
				break;
			default :
				break;
			}
			break;
		case REDRAW :
			window = get_windat(windows, (int) data);
			if (window == NULL) {
				printf("REDRAW: NULL window encountered\n");
				printf("	in handle_event.\n");
				printf("        windows = %ld\n", windows);
				printf("	window %d\n", (int) data);
			}
			else {
				window->redrawn = FALSE;
			}
			break;
		case TIMER0:
			/* Animate if necessary */
			if (animation.reverse || animation.forward) {
				if (!animated) {
					do_animation(windows);
					animated = TRUE;
				}
			}
			break;
		default:
			window = get_windat(windows, (int) current.menu_id);
			if (window != NULL) {
				(*window->event_fn)(windows, dev_id,
						    current.menu_id);
			}
			else {
				printf("DEFAULT:  NULL window encountered.\n");
			}
			break;
		}
	}

	/* Redraw all of the windows that requested it. */
	for (window = windows; window != NULL; window = window->next) {
		if (!(window->redrawn)) {
			winset((long) window->win_id);
			reshapeviewport();
			(*window->redraw_fn)(window);
			window->redrawn = TRUE;
		}
	}
}


void
do_animation(windows)
	windat_t * windows;
{
	image_t ** image;
	image_t * old_image;
	windat_t * window;


	/* Update all view windows with a new dataset */
	for (window = windows; window != NULL; window = window->next) {
		/* If this is not a view window, go to the next one */
		if (window->redraw_fn != image_draw) {
			continue;
		}

		/* Clear this window's redrawn flag so that the system */
		/* knows it needs updating.  Set the animated flag so that */
		/* it knows why. */
		window->redrawn = FALSE;
		window->animated = TRUE;

		/* Get a pointer to the image data */
		image = (image_t **) &(window->data);


		/* If animating forward, set up the next frame */
		if (animation.forward) {
			if (animation.forward > 0) {
				animation.forward--;
			}
			if ((*image)->next == NULL) {
				*image = images;
			}
			else {
				*image = (*image)->next;
			}
		}

		/* Likewise, if going backward get the previous frame */
		else if (animation.reverse) {
			if (animation.reverse > 0) {
				animation.reverse--;
			}

			old_image = *image;
			if (*image == images) {
				while ((*image)->next != NULL) {
					*image = (*image)->next;
				}
			}
			else {
				*image = images;
				while ((*image)->next != old_image) {
					*image = (*image)->next;
				}
			}
		}

		qenter(REDRAW, (short) window->win_id);
	}
}


int
do_event(windows, dev_id, data)
	windat_t * windows;
	long dev_id;
	short data;
/* DESCRIPTION:  Processes the event described by "dev_id" and "data"
using the "windows" data structure.  This routine is a catch-all for
non-REDRAW / ESCKEY events.  Returns TRUE if an error occurs, FALSE
otherwise.
*/
{
	return FALSE;
}


int
title_event(windows, dev_id, data)
	windat_t * windows;
	long dev_id;
	short data;
/* DESCRIPTION:  Processes the event described by "dev_id" and "data"
using the "windows" data structure.  Used for the title screen that fills
the view window at program startup.  Any mouse click in the window resets
the event handlers for the window so that it will display and manipulate
the active dataset.
*/
{
	windat_t * window;

	window = get_windat(windows, (int) data);
	
	switch (dev_id) {
	case LEFTMOUSE:
	case MIDDLEMOUSE:
	case RIGHTMOUSE:
		/* Change the event handlers and redraw the window */
		window->redraw_fn = image_draw;
		window->event_fn = image_event;
		image_draw(window);

		/* Unfreeze the window but keep the same aspect */
		keepaspect(NTSC_XSIZ, NTSC_YSIZ);
		winconstraints();

		break;
	default:
		break;
	}
	return FALSE;
}


int
image_event(windows, dev_id, data)
	windat_t * windows;
	long dev_id;
	short data;
/* DESCRIPTION:  Allows the user to "drag" the image around in the
image window by holding down the left or right mouse buttons and
moving the mouse.
*/
{
	int i;
	long x,y;
	static float norm[DIMS];
	float scale_v;
	Angle twist_v;
	float rho_v, theta_v, phi_v;

	float angleCI;

	float CI[DIMS], CP[DIMS], CT[DIMS], CM[DIMS], V[DIMS];
	float magCI, magCM;
	float radius;

	float half, old_half;

	int both_pressed;

	long left, right, bottom, top;	/* Window size parameters */
	float cx, cy;			/* Coordinates of window center */


	/* If both buttons are pressed (to control the twist parameter), */
	/* then we are apparently re-entering the image event handler. */
	/* Quit before anything gets confused and the calling function */
	/* and the calling function will sort things out. */
	both_pressed = getbutton(LEFTMOUSE) && getbutton(MIDDLEMOUSE);
	if (both_pressed) {
		return FALSE;
	}

	/* Process the appropriate button event */
	switch (dev_id) {
	case LEFTMOUSE:
	case MIDDLEMOUSE:
		/* Change active window */
		current.active_id = (int) data;
		current.window = get_windat(windows,
					    (int) current.active_id);
		winset(current.active_id);

		/* Get the window size for use later in the loop */
		getorigin(&left, &bottom);
		getsize(&right, &top);
		right += left;
		top += bottom;
		cx = (float) (left + right) / 2.0;
		cy = (float) (bottom + top) / 2.0;

                /* My own "virtual trackball code" */
                radius = MIN((right-left)/2, (top-bottom)/2);
                CI[X] = (float) getvaluator(MOUSEX) - cx;
                CI[Y] = (float) getvaluator(MOUSEY) - cy;
		magCI = fhypot(CI[X], CI[Y]);

		if (magCI > radius) {
			CI[X] = radius * CI[X] / magCI;
			CI[Y] = radius * CI[Y] / magCI;
			magCI = radius;
		}
		CI[Z] = fsqrt(radius*radius - magCI*magCI);

		V[X] = from[X] - at[X];
		V[Y] = from[Y] - at[Y];
		V[Z] = from[Z] - at[Z];

		scale_v = scaling;
		twist_v = twist;

		rho_v = rho;
		theta_v = theta;
		phi_v = phi;

		for (i = 0; i < DIMS; i++) {
			norm[i] = -(at[i]-from[i])/rho;
		}

		half = fsin(phi_v);

		/* Loop while button is down, adjusting view on fly */
		while (getbutton(dev_id)) {
			/* Just in case the window changes size */
			reshapeviewport();
			/* For each move ... */
			CM[X] = getvaluator(MOUSEX) - cx;
			CM[Y] = getvaluator(MOUSEY) - cy;
			magCM = fhypot((float)CM[X], (float)CM[Y]);

			/* If the cursor has moved outside the */
			/* limit of the "virtual trackball", */
			/* normalize its position to the boundary */
			/* of the circle. */
			if (magCM > radius) {
				CM[X] = radius * CM[X] / magCM;
				CM[Y] = radius * CM[Y] / magCM;
				magCM = radius;
			}
			CM[Z] = fsqrt(radius*radius - magCM*magCM);

			if (getbutton(LEFTMOUSE) && getbutton(MIDDLEMOUSE)) {
				if (!both_pressed) {
					CT[X] = CI[X]-CP[X];
					CT[Y] = CI[Y]-CP[Y];
					CT[Z] = CI[Z]-CP[Z];
					angleCI = 1800*fatan2(CM[Y], CM[X])/PI;
					both_pressed = TRUE;
				}
				/* Adjust twist using left-right movement */
				twist = twist_v - (Angle) ((1800 * 
					fatan2(CM[Y], CM[X])/PI) - angleCI );
			}
			else {
			    if (both_pressed) {
				CI[X] = CM[X]+CT[X];
				CI[Y] = CM[Y]+CT[Y];
				CI[Z] = CM[Z]+CT[Z];
				twist_v = twist;
				both_pressed = FALSE;
			    }

			    CP[X] = CM[X];
			    CP[Y] = CM[Y];
			    CP[Z] = CM[Z];

			    if (dev_id == LEFTMOUSE) {
				/* Calculate new from point */
				rho = rho_v + 2.0*(CM[Y]-CI[Y])/radius;
				if (rho < 0.1) {
					rho = 0.1;
				}

				/* Adjust orthographic scaling parameter */
				scaling = scale_v - 0.02*(CM[Y]-CI[Y])/radius;
				if (scaling < 0.02) {
					scaling = 0.02;
				}

				for (i = 0; i < DIMS; i++) {
					from[i] = at[i]+norm[i]*rho;
				}
			    }

			    else if (dev_id == MIDDLEMOUSE) {
				theta = theta_v + PI*(CM[X]-CI[X])/radius;
				phi = phi_v + PI*(CM[Y]-CI[Y])/radius;

				old_half = half;
				half = fsin(phi);

				/* If the y axis is crossed, invert the */
				/* twist value */
				if (((half >= 0.0) && (old_half < 0.0)) ||
				    ((half <= 0.0) && (old_half > 0.0))) {
					twist = twist-1800;
					if (twist < 0) {
						twist += 3600;
					}
					twist_v = twist;
				}

				sphere_to_cart(rho, theta, phi,
					       &V[X], &V[Y], &V[Z]);

				from[X] = at[X] + V[X];
				from[Y] = at[Y] + V[Y];
				from[Z] = at[Z] + V[Z];
			    }
			}

			qenter(REDRAW, (short) current.active_id);
			handle_event(windows);
		}
		break;
	case RIGHTMOUSE :
		current.active_id = (int) data;
		current.window = get_windat(windows,
					    (int) current.active_id);
		switch(dopup(view_m)) {
		case 7:	/* Fly */
			do_fly(7);
			break;
		case 8: /* Pop */
			do_pop(8);
			break;
		case 9:	/* Redraw */
			do_redraw(9);
			break;
		case 10: /* Reset */
			do_reset(10);
			break;
		case 11: /* Write HDF */
			write_hdf(11);
			break;
		default:
			break;
		}
		break;
	default :
		break;
	}

	return FALSE;
}


int
palette_event(windows, dev_id, data)
	windat_t * windows;
	long dev_id;
	short data;
/* DESCRIPTION:  Allows the user to change the palette through a pop
up menu.  Also can press ESC key to get an exit menu.
*/
{
	long x,y;
	float old_origin, old_slope;
	float slope_p;
	
	switch (dev_id) {
	case LEFTMOUSE:
	case MIDDLEMOUSE:
		x = getvaluator(MOUSEX);
		y = getvaluator(MOUSEY);
		old_origin = origin;
		old_slope = slope;

		while (getbutton(dev_id)) {
			/* Just in case the window changes size */
			reshapeviewport();

			/* Adjust slope of mapping function */
			slope_p = slope;
			slope = fexp(flog(old_slope) +
			    (getvaluator(MOUSEX)-x)/50.0);
			if (slope < 1.0/512.0) slope = 1.0/512.0;
			if (slope > 255.0) slope = 255.0;

			/* Adjust old origin to account for shift */
			/* in slope */
			old_origin -= (slope - slope_p) * 128;

			/* Adjust origin of mapping function */
			origin = old_origin +
			    (getvaluator(MOUSEY)-y);

			/* Update the color map */
			build_colormap(MAP_SPEC);
		}
		break;

	case RIGHTMOUSE :
		dopup(vpalette_m);
		break;

	default :
		break;
	}
	
	return FALSE;
}


void
init_events(windows)
	windat_t * windows;
/* DESCRIPTION:  init_events queues devices for input, and queues the initial
input events for all of the "windows".
*/
{
	windat_t * current;

	/* Queue only redraw events.  Also, if the user presses the escape */
	/* key, we want to know. */
	qdevice(ESCKEY);
	qdevice(KEYBD);
	qdevice(REDRAW);
	qdevice(LEFTMOUSE);
	qdevice(MIDDLEMOUSE);
	qdevice(RIGHTMOUSE);
	qdevice(TIMER0);
	noise(TIMER0, 15);

	/* Place events in the queue to redraw all of the windows. */
	for (current=windows; current != NULL; current=current->next) {
		qenter(REDRAW, (short) current->win_id);
	}
}
