/* 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain all copyright 
 *    notices, this list of conditions and the following disclaimer.
 * 2. The names of the authors may not be used to endorse or promote products
 *    derived from this software withough specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
/*
 * if_wl.c - original MACH, then BSDI ISA wavelan driver
 *	ported to mach by Anders Klemets
 *	to BSDI by Robert Morris
 *	to FreeBSD by Jim Binkley
 *	again to BSD/OS 2.1 and V.2 WaveMODEM by Vladimir Vorobyev
 *
 * 1996/10/01  bob@turbo.nsk.su:
 * Ported to BSD/OS V2.1;
 * Added support for V.2 WaveMODEM (new half-size ISA board);
 * Cosmetic changes.
 *
 * Work done:
 * Ported to FreeBSD, got promiscuous mode working with bpfs,
 * and rewired timer routine.  The i82586 will hang occasionally on output 
 * and the watchdog timer will kick it if so and log an entry.
 * 2 second timeout there.  Apparently the chip loses an interrupt.
 * Code borrowed from if_ie.c for watchdog timer.
 *
 * The wavelan card is a 2mbit radio modem that emulates ethernet;
 * i.e., it uses MAC addresses.  This should not be a surprise since
 * it uses an ethernet controller as a major hw item.
 * It can broadcast, unicast or apparently multicast in a base cell 
 * using a omni-directional antennae that is 
 * about 800 feet around the base cell barring walls and metal.  
 * With directional antennae, it can be used point to point over a mile
 * or so apparently (haven't tried that).
 *
 * There are ISA and pcmcia versions (if_wlp.c).
 * The ISA card has an Intel 82586 lan controller on it.  It consists
 * of 2 pieces of hw, the lan controller (intel) and a radio-modem.
 * The latter has an extra set of controller registers that has nothing
 * to do with the i82586 and allows setting and monitoring of radio
 * signal strength, etc.  There is a nvram area called the PSA that
 * contains a number of setup variables including the IRQ and so-called
 * NWID or Network ID.  The NWID must be set the same for all radio
 * cards to communicate (unless you are using the ATT/NCR roaming feature
 * with their access points.  There is no support for that here. Roaming
 * involves a link-layer beacon sent out from the access points.  End
 * stations monitor the signal strength and only use the strongest
 * access point).  This driver assumes that the base ISA port, IRQ, 
 * and NWID are first set in nvram via the dos-side "instconf.exe" utility 
 * supplied with the card. This driver takes the ISA port from 
 * the kernel configuration setup.  It makes no attempt to match the 
 * PSA and config-side settings for base port and IRQ.  They *must* match.  
 * The hw also magically just uses the IRQ set in the nvram.
 * The NWID is used magically as well by the radio-modem
 * to determine which packets to keep or throw out.
 * One further note: promiscuous mode is a curious thing.  In this driver,
 * promiscuous mode apparently will catch ALL packets and ignore the NWID
 * setting.  This is probably more useful in a sense (for snoopers) if
 * you are interested in all traffic as opposed to if you are interested
 * in just your own.  
 */

/*
 *
 * Ported to BSDI UNIX by Robert Morris, rtm@das.harvard.edu.
 * device wl0 at isa? port 0x390
 * 
 * (ancient mach) HISTORY
 * $Log:	wavelan.sh,v $
 * Revision 1.1.1.1  92/06/02  09:52:16  machsrc
 * Initial checkin
 * 
 * Revision 1.1  92/06/02  09:52:11  machsrc
 * Initial revision
 * 
 * Revision 1.4  91/11/22  14:48:15  klemets
 * Added support for setting the IRQ number parameter of the WaveLAN.
 * 
 * Revision 1.3  91/05/08  20:43:42  klemets
 * Added DLI support. When DLI is active, packets will be passed both
 * to dli_input and to usual input routines.
 * Corrected bug in multicast address handling.
 * Various enhancements to the counter mechanism. Now keeps count of
 * SNR values.
 * Added copyright statements, and did various minor fixes.
 * 
 * Revision 1.2  91/03/27  16:11:04  klemets
 * Added rel5 multicast code from Stanford.
 * Extended functionality of SIOCCIFCNTRS ioctl.
 * 
 * Revision 1.1  91/02/11  15:06:33  klemets
 * Initial revision
 * 
 */

/* 
 *	Olivetti PC586 Mach Ethernet driver v1.0
 *	Copyright Ing. C. Olivetti & C. S.p.A. 1988, 1989
 *	All rights reserved.
 *
 */ 

/*
  Copyright 1988, 1989 by Olivetti Advanced Technology Center, Inc.,
Cupertino, California.

		All Rights Reserved

  Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appears in all
copies and that both the copyright notice and this permission notice
appear in supporting documentation, and that the name of Olivetti
not be used in advertising or publicity pertaining to distribution
of the software without specific, written prior permission.

  OLIVETTI DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
IN NO EVENT SHALL OLIVETTI BE LIABLE FOR ANY SPECIAL, INDIRECT, OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
LOSS OF USE, DATA OR PROFITS, WHETHER IN ACTION OF CONTRACT,
NEGLIGENCE, OR OTHER TORTIOUS ACTION, ARISING OUR OF OR IN CONNECTION
WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

/*
  Copyright 1988, 1989 by Intel Corporation, Santa Clara, California.

		All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appears in all
copies and that both the copyright notice and this permission notice
appear in supporting documentation, and that the name of Intel
not be used in advertising or publicity pertaining to distribution
of the software without specific, written prior permission.

INTEL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
IN NO EVENT SHALL INTEL BE LIABLE FOR ANY SPECIAL, INDIRECT, OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
LOSS OF USE, DATA OR PROFITS, WHETHER IN ACTION OF CONTRACT,
NEGLIGENCE, OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

/*
 * NOTE:
 *		by rvb:
 *  1.	The best book on the 82586 is:
 *		LAN Components User's Manual by Intel
 *	The copy I found was dated 1984.  This really tells you
 *	what the state machines are doing
 *  2.	In the current design, we only do one write at a time,
 *	though the hardware is capable of chaining and possibly
 *	even batching.  The problem is that we only make one
 *	transmit buffer available in sram space.
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/syslog.h>
#include <sys/device.h>
#include <sys/proc.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/bpf.h>
#include <net/bpfdesc.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#include <machine/cpu.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/isa/icu.h>

#define DELAYCONST	1000	/* was 1000 in original, fed to DELAY(x) */

#include "if_wl_i82586.h"	/* Definitions for the Intel chip */
#include "if_wl.h"
#include "if_wl_wavelan.h"

#ifdef	WLDEBUG
static int wldebug = 0;
#define	dprintf(x) {if (wldebug) printf x;}
#else
#define	dprintf(x)
#endif

static char t_packet[ETHERMTU + sizeof(struct ether_header) + sizeof(long)];

/*
 * Ethernet software status per interface.
 *
 * Each interface is referenced by a network interface structure,
 * qe_if, which the routing code uses to locate the interface.
 * This structure contains the output queue for the interface, its address, ...
 */
struct wl_softc { 
	/*
	 * Common stuff first.
	 */
	struct  device wl_dev;		/* base device */
	struct  isadev wl_id;		/* ISA device */
	struct  intrhand wl_ih;		/* interrupt vectoring */
	struct	arpcom wl_ac;		/* Ethernet common part */
#define	wl_if	wl_ac.ac_if		/* network-visible interface */
#define	wl_addr	wl_ac.ac_enaddr		/* hardware Ethernet address */

	/*
	 * WaveLAN-specific stuff from here down.
	 */
	int	vers;		/* version of modem */
	u_char	nwid[2];	/* current radio modem nwid */
	u_short	base;		/* location of board in I/O space */
	int	flags;		/* from the kernel config file */
	int     tbusy;		/* flag to determine if xmit is busy */
        u_short begin_fd;
	u_short end_fd;
	u_short	end_rbd;
	u_short	hacr;		/* latest host adapter CR command */
	short	mode;
};

#define	WLSOFTC(unit)	((struct wl_softc *)(wlcd.cd_devs[unit]))

int	wlprobe __P((struct device *, struct cfdata *, void *));
void	wlattach __P((struct device *, struct device *, void *));
int	wlintr __P((struct wl_softc *));
void	wlwatchdog __P((void *));
int	wlinit __P((int));
int	wlstart __P((struct ifnet *));
int	wlioctl __P((struct ifnet *, int, caddr_t));

int	wlreadirq __P((int));
void	wlsetirq __P((int, int));
u_char	wlreadpsa __P((int, int));
int	wlhwrst __P((int));
void	wlinitmmc __P((int));
void	wlmmcstat __P((int));
void	wlbldcu __P((int));
int	wldiag __P((int));
int	wlcmd __P((int, char *));
int	wlconfig __P((int));
void	wlrustrt __P((int));
void	wlxmt __P((int, struct mbuf *));
int	wlread __P((int, int));
void	wlsetnwid __P((int, int, int));
int	wlack __P((int));
void	wlrcv __P((int));
void	wlrequeue __P((int, int));
u_short	wlbldru __P((int));
u_short	wlmmcread __P((int, int));

void	wlsftwsleaze __P((u_short *, u_char **, struct mbuf **, int));
void	wlhdwsleaze __P((u_short *, u_char **, struct mbuf **, int));

struct cfdriver wlcd =
	{ NULL, "wl", wlprobe, wlattach, DV_IFNET, sizeof(struct wl_softc) };

#define	WAVELAN_CARD	"WaveLAN/ISA"

/*
 * Valid IRQs -- 3, 4, 5, 7, 10, 11, 12, 15
 */
#define WL_IRQS		(IRQ3|IRQ4|IRQ5|IRQ7|IRQ10|IRQ11|IRQ12|IRQ15)
#define	WL_IRQVALID(i)	(WL_IRQS & (1 << (i)))

/* array for maping irq numbers to values for the irq parameter register */
static int irqvals[16] = { 0, 0, 0, 0x01, 0x02, 0x04, 0, 0x08, 0, 0, 0x10,
				0x20, 0x40, 0, 0, 0x80 };
/*
 * Valid I/O port base addresses
 */
#define	WL_BASEVALID(b)	((b) == 0x300 || (b) == 0x390 || \
			 (b) == 0x3c0 || (b) == 0x3e0)
#define	WL_NPORT	16

extern int autodebug;


/*
 * wlprobe:
 *
 *	This function "probes" or checks for the WaveLAN board on the bus to
 *	see if it is there.  As far as I can tell, the best break between this
 *	routine and the attach code is to simply determine whether the board
 *	is configured in properly.  Currently my approach to this is to write
 *	and read a word from the SRAM on the board being probed.  If the word
 *	comes back properly then we assume the board is there.  The config
 *	code expects to see a successful return from the probe routine before
 *	attach will be called.
 *
 * input	: address device is mapped to, and unit # being checked
 * output	: a '1' is returned if the board exists, and a 0 otherwise
 *
 */
int
wlprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	register short base = ia->ia_iobase;
	char buf[80], *str = "wl%d: illegal port %x for WaveLAN\n";
	int irq;

	/*
	 * First check if I/O port base is valid
	 */
	if (!WL_BASEVALID(base)) {
		printf(str, cf->cf_unit, base);
		return 0;
	}
	if (!isa_portcheck(base, WL_NPORT)) {
		if (autodebug)
			printf("WaveLAN port range overlaps existing, ");
		return 0;
	}

	irq = splimp();
	PCMD(base, HACR_RESET);			/* reset the board */
	DELAY(DELAYCONST);	                /* >> 4 clocks at 6MHz */
	PCMD(base, HACR_RESET);			/* reset the board */
	DELAY(DELAYCONST);	                /* >> 4 clocks at 6MHz */
	splx(irq);

	/* clear reset command and set PIO#1 in autoincrement mode */
	PCMD(base, HACR_DEFAULT);
	PCMD(base, HACR_DEFAULT);
	outw(PIOR1(base), 0);	/* go to beginning of RAM */
	outsw(PIOP1(base), str, strlen(str)/2+1);	/* write string */

	outw(PIOR1(base), 0);	/* rewind */
	insw(PIOP1(base), buf, strlen(str)/2+1);	/* read result */

	if (bcmp(str, buf, strlen(str))) return 0;	/* no board found */

	/* find out what irq the board is using */
	irq = wlreadirq(base);
	if (irq == -1) {
		printf("wl%d: request invalid irq\n", cf->cf_unit);
		return 0;
	}
	if (ia->ia_irq == IRQUNK) ia->ia_irq = (1 << irq);
	else if (irq != ffs(ia->ia_irq) - 1) {
		irq = ffs(ia->ia_irq) - 1;
		if (!WL_IRQVALID(irq)) {
			printf("wl%d: illegal irq %d for WaveLAN\n", cf->cf_unit, irq);
			return 0;
		}
		wlsetirq(base, irq);
	}
	ia->ia_iosize = WL_NPORT;
	return 1;
}

int
wlreadirq(base)
	int base;
{
	int irqval, irq;

	PCMD(base, HACR_DEFAULT & ~HACR_16BITS);
	PCMD(base, HACR_DEFAULT & ~HACR_16BITS);

	outw(PIOR2(base), IRQNO);  /* param memory */
	irqval = inb(PIOP2(base));

	PCMD(base, HACR_DEFAULT);
	PCMD(base, HACR_DEFAULT);

	for (irq = 0; irq < 16; irq++)
		if (irqvals[irq] == irqval) return irq;
	return -1;
}

/*
 * Set the IRQ. Do not call this when the board is operating,
 * as it doesn't preserve the hacr.
 */
void
wlsetirq(base, irq)
	int base, irq;
{
	int s = splimp();

	PCMD(base, HACR_DEFAULT & ~HACR_16BITS);
	PCMD(base, HACR_DEFAULT & ~HACR_16BITS);

	DELAY(DELAYCONST);
	outw(PIOR2(base), IRQNO);  /* write param memory */
	DELAY(DELAYCONST);
	outb(PIOP2(base), irqvals[irq]);
	DELAY(DELAYCONST);

	PCMD(base, HACR_RESET);			/* reset the board */
	DELAY(DELAYCONST);	                /* >> 4 clocks at 6MHz */
	PCMD(base, HACR_RESET);			/* reset the board */
	DELAY(DELAYCONST);	                /* >> 4 clocks at 6MHz */

	PCMD(base, HACR_DEFAULT);
	PCMD(base, HACR_DEFAULT);

	splx(s);
}

u_char
wlreadpsa(base, reg)
	int base, reg;
{
	outw(PIOR2(base), reg);
	return inb(PIOP2(base));
}

u_short
wlmmcread(base, reg)
	short base;
	u_short reg;
{
	while (inw(HASR(base)) & HASR_MMC_BUSY) ;
	outw(MMCR(base), reg << 1);
	while (inw(HASR(base)) & HASR_MMC_BUSY) ;
	return (u_short)inw(MMCR(base)) >> 8;
}

void
wlmmcwrite(base, reg, val)
	short base;
	u_short reg, val;
{
	while (inw(HASR(base)) & HASR_MMC_BUSY) ;
	outw(MMCR(base), (val << 8) | (reg << 1) | 1);
}

void
wlwaitforx24(base)
	register base;
{
	register i;

	for (i = 1000; i && (wlmmcread(base, MMC_X24STAT) &
			     (MMC_X24STAT_DLWD | MMC_X24STAT_EEBUSY)); i--);
}

/*
 * wlattach:
 *
 *	This function attaches a WaveLAN board to the "system".  The rest of
 *	runtime structures are initialized here (this routine is called after
 *	a successful probe of the board).  Once the ethernet address is read
 *	and stored, the board's ifnet structure is attached and readied.
 *
 * input	: isa_dev structure setup in autoconfig
 * output	: board structs and ifnet is setup
 *
 */
void
wlattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct wl_softc *sc = (struct wl_softc *) self;
	struct isa_attach_args *ia = (struct isa_attach_args *) aux;
	register struct ifnet *ifp = &sc->wl_if;
	short base = ia->ia_iobase;
	int i, unit = sc->wl_dev.dv_unit;
	char *vers;

	sc->base = base;
	sc->flags = sc->wl_dev.dv_flags;
	sc->mode = 0;
	sc->hacr = HACR_RESET;
	CMD(unit);				/* reset the board */
	DELAY(DELAYCONST);	                /* >> 4 clocks at 6MHz */

	/*
	 * Clear reset command and set PIO#2 in parameter access mode
	 */
	sc->hacr = (HACR_DEFAULT & ~HACR_16BITS);
	CMD(unit);

	/*
	 * Get version of modem.
	 */
	switch (wlmmcread(base, MMC_X24STAT) & MMC_X24STAT_CHIPID) {
		case MMC_X24STAT_CHIPID_X24:
			sc->vers = 1;	/* Xilinx modem chip */
			vers = " Xilinx";
			break;
		case MMC_X24STAT_CHIPID_ARIADNE:
			sc->vers = 2;	/* Ariadne modem chip */
			vers = " Ariadne";
			break;
		default:
			sc->vers = 0;	/* old (v.1) modem */
			vers = "";
	}

	/*
	 * Fetch network ID
	 */
	sc->nwid[0] = wlreadpsa(base, PSA_NETW_ID);
	sc->nwid[1] = wlreadpsa(base, PSA_NETW_ID + 1);

	/*
	 * Fetch MAC address
	 */
	for (i = 0; i < WAVELAN_ADDR_SIZE; i++)
		sc->wl_addr[i] = wlreadpsa(base, PSA_MAC_ADDR + i);

	/*
	 * Enter normal 16 bit mode operation
	 */
	sc->hacr = HACR_DEFAULT;
	CMD(unit);

	outw(PIOR1(base), OFFSET_SCB + 8);	/* address of scb_crcerrs */
	outw(PIOP1(base), 0);			/* clear scb_crcerrs */
	outw(PIOP1(base), 0);			/* clear scb_alnerrs */
	outw(PIOP1(base), 0);			/* clear scb_rscerrs */
	outw(PIOP1(base), 0);			/* clear scb_ovrnerrs */

	wlinitmmc(unit);

	/*
	 * Initialize ifnet structure
	 */
	ifp->if_unit = unit;
	ifp->if_name = wlcd.cd_name;
	ifp->if_mtu = WAVELAN_MTU;
#ifdef	MULTICAST
	ifp->if_flags = IFF_BROADCAST|IFF_MULTICAST|IFF_NOTRAILERS|IFF_SIMPLEX;
#else
	ifp->if_flags = IFF_BROADCAST|IFF_NOTRAILERS|IFF_SIMPLEX;
#endif
	ifp->if_init = wlinit;
	ifp->if_start = wlstart;
	ifp->if_ioctl = wlioctl;
	ifp->if_watchdog = 0;
	ether_attach(ifp);
	bpfattach(&ifp->if_bpf, ifp, DLT_EN10MB, sizeof(struct ether_header));

	/*
	 * Print banner
	 */
	aprint_naive(": %s Ethernet", WAVELAN_CARD);
	printf("\n");
	aprint_normal("wl%d: %s%s nwid %x:%x address %s\n",
		      unit, WAVELAN_CARD, vers,
		      sc->nwid[0], sc->nwid[1], ether_sprintf(sc->wl_addr));

	/*
	 * Register with lower levels of kernel
	 */
	isa_establish(&sc->wl_id, &sc->wl_dev);
	sc->wl_ih.ih_fun = wlintr;
	sc->wl_ih.ih_arg = (void *)sc;
	intr_establish(ia->ia_irq, &sc->wl_ih, DV_NET);
}

/*
 * Print out interesting information about the 82596.
 */
void
wldump(unit)
	int unit;
{
	register struct wl_softc *sp = WLSOFTC(unit);
	int base = sp->base;
	int i;

	printf("hasr %04x\n", inw(HASR(base)));

	printf("scb at %04x:\n ", OFFSET_SCB);
	outw(PIOR1(base), OFFSET_SCB);
	for (i = 0; i < 8; i++)
		printf("%04x ", inw(PIOP1(base)));
	printf("\n");

	printf("cu at %04x:\n ", OFFSET_CU);
	outw(PIOR1(base), OFFSET_CU);
	for (i = 0; i < 8; i++)
		printf("%04x ", inw(PIOP1(base)));
	printf("\n");

	printf("tbd at %04x:\n ", OFFSET_TBD);
	outw(PIOR1(base), OFFSET_TBD);
	for (i = 0; i < 4; i++)
		printf("%04x ", inw(PIOP1(base)));
	printf("\n");
}

/*
 * Initialize the Modem Management Controller
 */
void
wlinitmmc(unit)
	int unit;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;
	int configured;
	u_char value;
	
	/* enter 8 bit operation */
	sc->hacr = (HACR_DEFAULT & ~HACR_16BITS);
	CMD(unit);

	configured = wlreadpsa(base, PSA_CONF_STATUS) & 1;
	
	/*
	 * Set default modem control parameters.
	 * Taken from NCR document 407-0024326 Rev. A
	 */
	wlmmcwrite(base, MMC_JABBER_ENABLE, 1); /* disable jabber timer */
	wlmmcwrite(base, MMC_ANTEN_SEL, MMC_ANTEN_SEL_AUTO);
	wlmmcwrite(base, MMC_IFS, MMC_IFS_DEFAULT);
	wlmmcwrite(base, MMC_MOD_DELAY, MMC_MOD_DEFAULT);
	wlmmcwrite(base, MMC_JAM_TIME, MMC_JAM_DEFAULT);
	wlmmcwrite(base, MMC_FREEZE, 0); /* collect signal lvl and quality */

	wlmmcwrite(base, MMC_ENCR_ENABLE, 0); /* disable security */
	wlmmcwrite(base, MMC_ENCR_IO_INVERT, 0);

	if (!configured) {
		/*
		 * Configure modem with new settings
		 */
		/* modem looptest selection */
		if (sc->mode & MOD_PROM) /* disable network id check */
			value = MMC_LOOPT_SEL_IGNWID;
		else	value = 0;	/* enable network id check */
		wlmmcwrite(base, MMC_LOOPT_SEL, value);

		/* modem threshold preset; 0x04 for AT and 0x01 for MCA */
		if (wlreadpsa(base, PSA_COMP_NUMBER) & 1) value = 0x01;
		else value = MMC_THR_DEFAULT;
		wlmmcwrite(base, MMC_THR_PRE_SET, value);

		/* modem quality threshold */
		wlmmcwrite(base, MMC_QUALITY_THR, MMC_QUALITY_DEFAULT);

	} else {
		/*
		 * Use configuration defaults from parameter storage area
		 */
		if (wlreadpsa(base, PSA_NETW_ID_SELECT) & 1) value = 0;
		else value = MMC_LOOPT_SEL_IGNWID;
		wlmmcwrite(base, MMC_LOOPT_SEL, value);
		wlmmcwrite(base, MMC_THR_PRE_SET, wlreadpsa(base, PSA_THR_PRE_SET) & 0x3f);
		wlmmcwrite(base, MMC_QUALITY_THR, wlreadpsa(base, PSA_QUALITY_THR) & 0x0f);
	}
	if (sc->vers) {	/* initialize V.2 modem */
		wlwaitforx24(base);
		wlmmcwrite(base, MMC_EEADDR, 0x0F);
		wlmmcwrite(base, MMC_EECTRL, MMC_EECTRL_DWLD | MMC_EECTRL_PRREAD);
		wlwaitforx24(base);
		wlmmcwrite(base, MMC_EEADDR, 0x61);
		wlmmcwrite(base, MMC_EECTRL, MMC_EECTRL_DWLD | MMC_EECTRL_PRREAD);
		wlwaitforx24(base);
		wlmmcwrite(base, MMC_ANALCTRL, MMC_ANALCTRL_EXTANT);
	}

	wlmmcwrite(base, MMC_NETW_ID_L, sc->nwid[1]);	/* set NWID */
	wlmmcwrite(base, MMC_NETW_ID_H, sc->nwid[0]);

	/* enter normal 16 bit mode operation */
	sc->hacr = HACR_DEFAULT;
	CMD(unit);
	CMD(unit); /* virtualpc1 needs this! */
}

/*
 * wlinit:
 *
 *	Another routine that interfaces the "if" layer to this driver.  
 *	Simply resets the structures that are used by "upper layers".  
 *	As well as calling wlhwrst that does reset the WaveLAN board.
 *
 * input	: board number
 * output	: structures (if structs) and board are reset
 *
 */	
int
wlinit(unit)
	int unit;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	struct ifnet *ifp = &sc->wl_if;
	int s, stat;

	dprintf(("wl%d: wlinit()\n", unit));

	/* address not known */
	if (ifp->if_addrlist == (struct ifaddr *)0) return;

	s = splimp();
	stat = wlhwrst(unit);
	if (stat) {
		sc->wl_if.if_flags |= IFF_RUNNING; /* same as DSF_RUNNING */
		/*
		 * OACTIVE is used by upper-level routines and must be set
		 */
		sc->wl_if.if_flags &= ~IFF_OACTIVE; /* same as tbusy below */
		sc->flags |= DSF_RUNNING;
		sc->tbusy = 0;
		untimeout(wlwatchdog, sc);
		wlstart(ifp);
	} else dprintf(("wl%d: hwrst trouble\n", unit));
	splx(s);
	return stat;
}

/*
 * wlhwrst:
 *
 *	This routine resets the WaveLAN board that corresponds to the 
 *	board number passed in.
 *
 * input	: board number to do a hardware reset
 * output	: board is reset
 *
 */
int
wlhwrst(unit)
	int unit;
{
	int i;
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;

	dprintf(("wl%d: hwrst()\n", unit));
	sc->hacr = HACR_RESET;
	CMD(unit);				/* reset the board */
	
	/* clear reset command and set PIO#1 in autoincrement mode */
	sc->hacr = HACR_DEFAULT;
	CMD(unit);

	if (sc->wl_if.if_flags & IFF_DEBUG) wlmmcstat(unit);

	wlbldcu(unit);		/* set up command unit structures */

	if (!wldiag(unit)) return 0;

	if (!wlconfig(unit)) return 0;

	/* 
	 * insert code for loopback test here
	 */
	wlrustrt(unit);		/* start receive unit */

	/* enable interrupts */
	sc->hacr = (HACR_DEFAULT | HACR_INTRON);
	CMD(unit);

	return 1;
}

/*
 * wlbldcu:
 *
 *	This function builds up the command unit structures.  It inits
 *	the scp, iscp, scb, cb, tbd, and tbuf.
 *
 */
void
wlbldcu(unit)
	int unit;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	short		base = sc->base;
	scp_t		scp;
	iscp_t		iscp;
	scb_t		scb;
	ac_t		cb;
	tbd_t		tbd;
	int		i;

	bzero(&scp, sizeof(scp));
	scp.scp_sysbus = 0;
	scp.scp_iscp = OFFSET_ISCP;
	scp.scp_iscp_base = 0;
	outw(PIOR1(base), OFFSET_SCP);
	outsw(PIOP1(base), &scp, sizeof(scp_t)/2);

	bzero(&iscp, sizeof(iscp));
	iscp.iscp_busy = 1;
	iscp.iscp_scb_offset = OFFSET_SCB;
	iscp.iscp_scb = 0;
	iscp.iscp_scb_base = 0;
	outw(PIOR1(base), OFFSET_ISCP);
	outsw(PIOP1(base), &iscp, sizeof(iscp_t)/2);

	scb.scb_status = 0;
	scb.scb_command = SCB_RESET;
	scb.scb_cbl_offset = OFFSET_CU;
	scb.scb_rfa_offset = OFFSET_RU;
	scb.scb_crcerrs = 0;
	scb.scb_alnerrs = 0;
	scb.scb_rscerrs = 0;
	scb.scb_ovrnerrs = 0;
	outw(PIOR1(base), OFFSET_SCB);
	outsw(PIOP1(base), &scb, sizeof(scb_t)/2);

	SET_CHAN_ATTN(unit);

	outw(PIOR0(base), OFFSET_ISCP + 0);	/* address of iscp_busy */
	for (i = 1000000; inw(PIOP0(base)) && (i-- > 0); );
	if (i <= 0) dprintf(("wl%d: iscp_busy timeout\n", unit));
	outw(PIOR0(base), OFFSET_SCB + 0);	/* address of scb_status */
	for (i = STATUS_TRIES; i-- > 0; ) {
		if (inw(PIOP0(base)) == (SCB_SW_CX|SCB_SW_CNA)) 
			break;
	}
	if (i <= 0) dprintf(("wl%d: not ready after reset\n", unit));
	wlack(unit);

	cb.ac_status = 0;
	cb.ac_command = AC_CW_EL;		/* NOP */
	cb.ac_link_offset = OFFSET_CU;
	outw(PIOR1(base), OFFSET_CU);
	outsw(PIOP1(base), &cb, 6/2);

	tbd.act_count = 0;
	tbd.next_tbd_offset = I82586NULL;
	tbd.buffer_addr = 0;
	tbd.buffer_base = 0;
	outw(PIOR1(base), OFFSET_TBD);
	outsw(PIOP1(base), &tbd, sizeof(tbd_t)/2);
}

/*
 * wlstart:
 *
 *	send a packet
 *
 * input	: board number
 * output	: stuff sent to board if any there
 *
 */
int
wlstart(ifp)
	struct ifnet *ifp;
{
        int unit = ifp->if_unit;
	struct mbuf *m;
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;
	int scb_status, cu_status, scb_command;

	dprintf(("wl%d: wlstart()\n", unit));

	outw(PIOR1(base), OFFSET_CU);
	cu_status = inw(PIOP1(base));
	outw(PIOR0(base),OFFSET_SCB + 0);	/* scb_status */
	scb_status = inw(PIOP0(base));
	outw(PIOR0(base), OFFSET_SCB + 2);
	scb_command = inw(PIOP0(base));

	/* don't need OACTIVE check as tbusy here checks to see
	 * if we are already busy 
	 */
	if (sc->tbusy) {
		if ((scb_status & 0x0700) == SCB_CUS_IDLE &&
		    (cu_status & AC_SW_B) == 0) {
			sc->tbusy = 0;
			untimeout(wlwatchdog, sc);
			sc->wl_if.if_flags &= ~IFF_OACTIVE;
			/*
			 * This is probably just a race.  The xmt'r is just
			 * became idle but WE have masked interrupts so ...
			 */
			dprintf(("wl%d: CU idle, scb %04x %04x cu %04x\n",
			       unit, scb_status, scb_command, cu_status));
		} else return;
	} else if ((scb_status & 0x0700) == SCB_CUS_ACTV ||
		   (cu_status & AC_SW_B)) {
		dprintf(("wl%d: CU unexpectedly busy; scb %04x cu %04x\n",
			 unit, scb_status, cu_status));
		return;
	}

	ifp = &(sc->wl_if);
	IF_DEQUEUE(&ifp->if_snd, m);
	if (m != (struct mbuf *)0) {
		/*
		 * Let the bpf subsystem have a peek at our packet
		 */
		if (sc->wl_if.if_bpf) bpf_mtap(sc->wl_if.if_bpf, m);
		sc->tbusy++;
		timeout(wlwatchdog, sc, 50);	/* 0.5 sec */
		sc->wl_if.if_flags |= IFF_OACTIVE;
		sc->wl_if.if_opackets++;
		wlxmt(unit, m);
	} else	sc->wl_if.if_flags &= ~IFF_OACTIVE;

	return;
}

/*
 * wlread:
 *
 *	This routine does the actual copy of data (including ethernet header
 *	structure) from the WaveLAN to an mbuf chain that will be passed up
 *	to the "if" (network interface) layer.  NOTE:  we currently
 *	don't handle trailer protocols, so if that is needed, it will
 *	(at least in part) be added here.  For simplicities sake, this
 *	routine copies the receive buffers from the board into a local (stack)
 *	buffer until the frame has been copied from the board.  Once in
 *	the local buffer, the contents are copied to an mbuf chain that
 *	is then enqueued onto the appropriate "if" queue.
 *
 * input	: board number, and an frame descriptor address
 * output	: the packet is put into an mbuf chain, and passed up
 * assumes	: if any errors occur, packet is "dropped on the floor"
 *
 */
int
wlread(unit, fd_p)
	int unit;
	u_short	fd_p;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	register struct ifnet	*ifp = &sc->wl_if;
	short			base = sc->base;
	fd_t			fd;
	struct	ether_header	eh;
    	struct	mbuf		*m, *tm;
	rbd_t			rbd;
	u_char			*mb_p;
	u_short			mlen, len, clen;
	u_short			bytes_in_msg, bytes_in_mbuf, bytes;

	dprintf(("wl%d: wlread()\n", unit));

	if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING)) {
		dprintf(("wl%d read(): board is not running\n", ifp->if_unit));
		sc->hacr &= ~HACR_INTRON;
		CMD(unit);		/* turn off interrupts */
	}
	/* read ether_header info out of device memory. doesn't
	 * go into mbuf.  goes directly into eh structure
	 */
	len = sizeof(struct ether_header);	/* 14 bytes */
	outw(PIOR1(base), fd_p);
	insw(PIOP1(base), &fd, (sizeof(fd_t) - len)/2);
	insw(PIOP1(base), &eh, (len-2)/2);
	eh.ether_type = ntohs((u_short)inw(PIOP1(base)));
	dprintf(("wl%d: rcv packet, type %x, ", unit, eh.ether_type));

	if (fd.rbd_offset == I82586NULL) {
		dprintf(("Invalid buffer\n"));
		if (!wlhwrst(unit)) dprintf(("wl%d: hwrst trouble\n", unit));
		return 0;
	}

	outw(PIOR1(base), fd.rbd_offset);
	insw(PIOP1(base), &rbd, sizeof(rbd_t)/2);
	bytes_in_msg = rbd.status & RBD_SW_COUNT;
	MGETHDR(m, M_DONTWAIT, MT_DATA);
	tm = m;
	if (m == (struct mbuf *)0) {
		/*
		 * not only do we want to return, we need to drop the packet on
		 * the floor to clear the interrupt.
		 */
		dprintf(("wl%d: No mbuf 1st\n", unit));
		if (!wlhwrst(unit)) {
			sc->hacr &= ~HACR_INTRON;
			CMD(unit);		/* turn off interrupts */
			dprintf(("wl%d: hwrst trouble\n", unit));
		}
		return 0;
	}
	m->m_next = (struct mbuf *) 0;
	m->m_pkthdr.rcvif = ifp;
	m->m_pkthdr.len = 0; /* don't know this yet */
	m->m_len = MHLEN;
	if (bytes_in_msg >= MINCLSIZE) {
		MCLGET(m, M_DONTWAIT);
		if (m->m_flags & M_EXT) m->m_len = MCLBYTES;
	}
	mlen = 0;
	clen = mlen;
	bytes_in_mbuf = m->m_len;
	mb_p = mtod(tm, u_char *);
	bytes = min(bytes_in_mbuf, bytes_in_msg);
	do {
		if (bytes & 1) len = bytes + 1;
		else len = bytes;
		outw(PIOR1(base), rbd.buffer_addr);
		insw(PIOP1(base), mb_p, len/2);
		clen += bytes;
		mlen += bytes;

		if (!(bytes_in_mbuf -= bytes)) {
			MGET(tm->m_next, M_DONTWAIT, MT_DATA);
			tm = tm->m_next;
			if (tm == (struct mbuf *)0) {
				m_freem(m);
				dprintf(("wl%d: No mbuf nth\n", unit));
				if (!wlhwrst(unit)) {
					sc->hacr &= ~HACR_INTRON;
					CMD(unit);  /* turn off interrupts */
					dprintf(("wl%d: hwrst trouble\n", unit));
				}
				return 0;
			}
			mlen = 0;
			tm->m_len = MLEN;
			bytes_in_mbuf = MLEN;
			mb_p = mtod(tm, u_char *);
		} else	mb_p += bytes;

		if (!(bytes_in_msg  -= bytes)) {
			if (rbd.status & RBD_SW_EOF ||
			    rbd.next_rbd_offset == I82586NULL) {
				tm->m_len = mlen;
				break;
			} else {
				outw(PIOR1(base), rbd.next_rbd_offset);
				insw(PIOP1(base), &rbd, sizeof(rbd_t)/2);
				bytes_in_msg = rbd.status & RBD_SW_COUNT;
			}
		} else	rbd.buffer_addr += bytes;
		bytes = min(bytes_in_mbuf, bytes_in_msg);
	} while(1);

	m->m_pkthdr.len = clen;

        /*
         * Check if there's a BPF listener on this interface. If so, hand off
         * the raw packet to bpf.
         */
        if (sc->wl_if.if_bpf) {
		/* bpf assumes header is in mbufs.  It isn't.  We can
		 * fool it without allocating memory as follows.
		 * Trick borrowed from if_ie.c
		 */
		struct mbuf m0;		
		m0.m_len = sizeof eh;
		m0.m_data = (caddr_t) &eh;
		m0.m_next = m;

		eh.ether_type = htons((u_short)eh.ether_type);
                bpf_mtap(sc->wl_if.if_bpf, &m0);
		eh.ether_type = ntohs((u_short)eh.ether_type);

                /*
                 * Note that the interface cannot be in promiscuous mode if
                 * there are no BPF listeners.  And if we are in promiscuous
                 * mode, we have to check if this packet is really ours.
		 *
		 * logic: if promiscuous mode AND not multicast/bcast AND
		 *	not to us, throw away
                 */
                if ((sc->wl_if.if_flags & IFF_PROMISC) &&
		    bcmp(eh.ether_dhost, sc->wl_addr,
			 sizeof(eh.ether_dhost)) != 0 &&
#ifdef MULTICAST
		    /* non-multicast (also non-broadcast) */
		    !ETHER_IS_MULTICAST(eh.ether_dhost)
#else
		    bcmp(eh.ether_dhost, etherbroadcastaddr,
			 sizeof(eh.ether_dhost)) != 0
#endif
		    ) {
			m_freem(m);
			return 1;
		}
	}
	dprintf(("bytes %d\n", clen));
	/*
	 * received packet is now in a chain of mbuf's.  next step is
	 * to pass the packet upwards.
	 *
	 */
	ether_input(&sc->wl_if, &eh, m);
	return 1;
}

/*
 * wlioctl:
 *
 *	This routine processes an ioctl request from the "if" layer
 *	above.
 *
 * input	: pointer the appropriate "if" struct, command, and data
 * output	: based on command appropriate action is taken on the
 *	 	  WaveLAN board(s) or related structures
 * return	: error is returned containing exit conditions
 *
 */
int
wlioctl(ifp, cmd, data)
	struct ifnet *ifp;
	int cmd;
	caddr_t	data;
{
	register struct ifaddr	*ifa = (struct ifaddr *)data;
	register struct ifreq	*ifr = (struct ifreq *)data;
	int			unit = ifp->if_unit;
	register struct wl_softc *sc = WLSOFTC(unit);
	short			base = sc->base;
	short			mode = 0;
	int			s, error = 0;
	struct	 proc           *p = curproc;	/* XXX */

	dprintf(("wl%d: wlioctl()\n", unit));

	s = splimp();
	switch (cmd) {
	case SIOCSIFADDR:
		/* Set own IP address and enable interface */
		ifp->if_flags |= IFF_UP;

		switch (ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET:
			wlinit(unit);
			arp_ifinit((struct arpcom *)ifp, ifa);
			break;
#endif
#ifdef NS
		case AF_NS:
		{
			register struct ns_addr *ina = &(IA_SNS(ifa)->sns_addr);
			if (ns_nullhost(*ina))
				ina->x_host = *(union ns_host *)(sc->wl_addr);
			else {
				ifp->if_flags &= ~IFF_RUNNING;
				bcopy((caddr_t)ina->x_host.c_host,
				      (caddr_t)sc->wl_addr, sizeof(sc->wl_addr));
			}
			wlinit(unit);
			break;
		}
#endif
		default:
			wlinit(unit);
			break;
		}
		break;
	case SIOCSIFFLAGS:
#ifdef	WLDEBUG
		wldebug = ifp->if_flags & IFF_DEBUG;
#endif
		if (ifp->if_flags & IFF_ALLMULTI)	mode |= MOD_ENAL;
		if (ifp->if_flags & IFF_PROMISC)	mode |= MOD_PROM;
		if (ifp->if_flags & IFF_LINK0)		mode |= MOD_PROM;
		/*
		 * force a complete reset if the receive multicast/
		 * promiscuous mode changes so that these take 
		 * effect immediately.
		 */
		if (sc->mode != mode) {
			sc->mode = mode;
			if (sc->flags & DSF_RUNNING) {
		    		sc->flags &= ~DSF_RUNNING;
				wlinit(unit);
			}
		}
		/* if interface is marked DOWN and still running then
		 * stop it.
		 */
		if ((ifp->if_flags & IFF_UP) == 0 && sc->flags & DSF_RUNNING) {
			sc->flags &= ~DSF_RUNNING;
			sc->hacr &= ~HACR_INTRON;
			CMD(unit);		  /* turn off interrupts */
		} 
		/* else if interface is UP and RUNNING, start it
		*/
		else if (ifp->if_flags & IFF_UP && (sc->flags & DSF_RUNNING) == 0) {
			wlinit(unit);
		}
		/* if DEBUG set on interface, then printf rf-modem regs
		*/
		if (ifp->if_flags & IFF_DEBUG) wlmmcstat(unit);
		break;
#ifdef	MULTICAST
	case SIOCADDMULTI:
	case SIOCDELMULTI:
		error = (cmd == SIOCADDMULTI) ?
			ether_addmulti(ifr, &sc->wl_ac) :
				ether_delmulti(ifr, &sc->wl_ac);
		if (error == ENETRESET) {
			if(sc->flags & DSF_RUNNING) {
				sc->flags &= ~DSF_RUNNING;
				wlinit(unit);
			}
			error = 0;
		}
		break;
#endif
	/* get the NWID out of the sc since we stored it there
	*/
	case SIOCGWLNWID:
		ifr->ifr_data = (caddr_t) (sc->nwid[0] << 8 | sc->nwid[1]);
		break;

	/* set the nwid
	*/
	case SIOCSWLNWID:
		/* root only
		*/
		error = suser(p->p_ucred, &p->p_acflag); 
		if (error != 0) break;
		wlsetnwid(unit, base, (int)ifr->ifr_data);
		break;
	default:
		error = EINVAL;
	}
	splx(s);
	return error;
}

/*
 * change the nwid dynamically.  This function
 * ONLY changes the radio modem and does not
 * change the PSA.  In order to change the PSA,
 * we need to know (which we don't) the checksum algorithm so that
 * we can still use the NCR/ATT software.  It will
 * proclaim that the card is broken if any PSA
 * values are changed.
 *
 * 3 steps:
 *	1. save in softc "soft registers"
 *	2. save in radio modem (MMC)
 */
void
wlsetnwid(unit, base, data)
	int unit, base, data;
{
	register struct wl_softc *sc = WLSOFTC(unit);

	/* soft c nwid shadows radio modem setting
	*/
	sc->nwid[0] = data >> 8;
	sc->nwid[1] = data & 0xff;
	wlmmcwrite(base, MMC_NETW_ID_L, sc->nwid[1]);
	wlmmcwrite(base, MMC_NETW_ID_H, sc->nwid[0]);
#ifdef CHANGEPSA    
	/* problem here is that PSA needs to be checksummed
	 * probably should have a different ioctl for that even
	 * if we learn how
	 */
	/* enter 8 bit operation */
	oldhacr = sc->hacr;
	sc->hacr = (HACR_DEFAULT & ~HACR_16BITS);
	CMD(unit);
	outw(PIOR2(base),NETW_ID);  /* write param memory */
	outb(PIOP2(base),sc->nwid[0]);
	outw(PIOR2(base),NETW_ID+1);
	outb(PIOP2(base),sc->nwid[1]);
	/* back to normal operation
	*/
	sc->hacr = oldhacr;
	CMD(unit);
	CMD(unit);
#endif
}

/*
 * wlwatchdog():
 *
 * The watchdog routine gets called if the transmitter failed to interrupt
 * within ifp->if_timer XXXseconds.  The interrupt service routine must reset
 * ifp->if_timer to 0 after an transmitter interrupt occurs to stop the
 * watchdog from happening.
 *
 * input	: which board is timing out
 * output	: board reset 
 *
 */
void
wlwatchdog(vp)
	void *vp;
{
	struct wl_softc *sc = vp;
	int unit = sc->wl_if.if_unit;

	dprintf(("wl%d: wlwatchdog()\n", unit));

	log(LOG_WARNING, "wl%d: timed out, resetting\n", unit);
	sc->wl_if.if_oerrors++;
	wlinit(unit);
}

/*
 * wlintr:
 *
 *	This function is the interrupt handler for the WaveLAN
 *	board.  This routine will be called whenever either a packet
 *	is received, or a packet has successfully been transfered and
 *	the unit is ready to transmit another packet.
 *
 * input	: board number that interrupted
 * output	: either a packet is received, or a packet is transfered
 *
 */
int
wlintr(sc)
	struct wl_softc *sc;
{
	int	unit = sc->wl_if.if_unit;
	short	base = sc->base;
	int	ac_status;
	u_short	int_type, int_type1;

	dprintf(("wl%d: wlintr()\n", unit));

	if ((int_type = inw(HASR(base))) & HASR_MMC_INTR) {
		/* handle interrupt from the modem management controler */
		/* This will clear the interrupt condition */ 
		/* ignored for now */
		int_type1 = wlmmcread(base, MMC_DCE_STATUS);
		dprintf(("wl%d: unexpected mmc intr, status 0x%04x\n", unit, int_type1));
	}
	if (!(int_type & HASR_INTR)) {	/* return if no interrupt from 82586 */
		wldump(unit);		/* it happens when reinit occurs */
		return 0;
	}
	for (;;) {
		outw(PIOR0(base), OFFSET_SCB + 0);	/* scb_status */
		if ((int_type = (inw(PIOP0(base)) & SCB_SW_INT)) == 0) break;
		int_type1 = wlack(unit);
		if (int_type != int_type1)
			dprintf(("wl%d: ack %04x != int_type %04x\n", unit,
				 int_type1, int_type));
		int_type = int_type1;
		/* incoming packet */
		if (int_type & SCB_SW_FR) {
			sc->wl_if.if_ipackets++;
			wlrcv(unit);
		}
		if (int_type & SCB_SW_RNR) {
			sc->wl_if.if_ierrors++;
			dprintf(("wl%d: intr overrun begin_fd = %x\n",
				 unit, sc->begin_fd));
			wlrustrt(unit);
		}
		if (int_type & SCB_SW_CNA) {
			/*
			 * At present, we don't care about CNA's.  We
			 * believe they are a side effect of XMT.
			 */
		}
		if (int_type & SCB_SW_CX) {
			/*
			 * At present, we only request Interrupt for XMT.
			 */
			outw(PIOR1(base), OFFSET_CU);
			ac_status = inw(PIOP1(base));
			if (!sc->tbusy)
				dprintf(("wl%d: xmt intr but not busy, CU %04x\n",
					 unit, ac_status));
			if (!ac_status)
				dprintf(("wl%d: xmt intr but ac_status = 0\n",
					unit));
			if ((!(ac_status & AC_SW_OK)) || (ac_status & 0xfff)) {
				if (ac_status & TC_DEFER) {
					dprintf(("wl%d: channel jammed\n", unit));
				}
				if (ac_status & TC_COLLISION) {
						dprintf(("wl%d: channel congestion\n",
							unit));
					if (!(ac_status & 0xf)) {
						sc->wl_if.if_collisions += 0x10;
					}
					sc->wl_if.if_oerrors++;
				}
				if (ac_status & TC_CARRIER) {
					dprintf(("wl%d: no carrier\n", unit));
				}
				if (ac_status & TC_CLS) {
					dprintf(("wl%d: no CTS\n", unit));
					sc->wl_if.if_oerrors++;
				}
				if (ac_status & TC_SQE) {
					dprintf(("wl%d: heart beat\n", unit));
				}
				if (ac_status & TC_DMA) {
					dprintf(("wl%d: DMA underrun\n", unit));
					sc->wl_if.if_oerrors++;
				}
				sc->wl_if.if_collisions += (ac_status & 0xf);
			}
			sc->tbusy = 0;
			untimeout(wlwatchdog, sc);
			sc->wl_if.if_flags &= ~IFF_OACTIVE;
			wlstart(&(sc->wl_if));
		}
	}
	return 1;
}

/*
 * wlrcv:
 *
 *	This routine is called by the interrupt handler to initiate a
 *	packet transfer from the board to the "if" layer above this
 *	driver.  This routine checks if a buffer has been successfully
 *	received by the WaveLAN.  If so, the routine wlread is called
 *	to do the actual transfer of the board data (including the
 *	ethernet header) into a packet (consisting of an mbuf chain).
 *
 * input	: number of the board to check
 * output	: if a packet is available, it is "sent up"
 *
 */
void
wlrcv(unit)
	int unit;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;
	u_short	fd_p, status, offset, link_offset;

	dprintf(("wl%d: wlrcv()\n", unit));

	for (fd_p = sc->begin_fd; fd_p != I82586NULL; fd_p = sc->begin_fd) {
		outw(PIOR0(base), fd_p + 0);	/* address of status */
		status = inw(PIOP0(base));
		outw(PIOR1(base), fd_p + 4);	/* address of link_offset */
		link_offset = inw(PIOP1(base));
		offset = inw(PIOP1(base));	/* rbd_offset */
		if (status == 0xffff || offset == 0xffff /*I82586NULL*/) {
			if (!wlhwrst(unit))
				dprintf(("wl%d: rcv hwrst ffff trouble\n", unit));
			return;
		} else if (status & AC_SW_C) {
			if (status == (RFD_DONE|RFD_RSC)) {
					/* lost one */;
				dprintf(("wl%d: rcv RSC %x\n", unit, status));
				sc->wl_if.if_ierrors++;
			} else if (!(status & RFD_OK)) {
				dprintf(("wl%d: rcv !OK %x\n", unit, status));
				sc->wl_if.if_ierrors++;
			} else if (status & 0xfff) {	/* can't happen */
				dprintf(("wl%d: rcv ERRs %x\n", unit, status));
				sc->wl_if.if_ierrors++;
			} else if (!wlread(unit, fd_p)) return;
			wlrequeue(unit, fd_p);
			sc->begin_fd = link_offset;
		} else break;
	}
}

/*
 * wlrequeue:
 *
 *	This routine puts rbd's used in the last receive back onto the
 *	free list for the next receive.
 *
 */
void
wlrequeue(unit, fd_p)
	int unit;
	u_short	fd_p;
{
	fd_t fd;
	u_short	l_rbdp, f_rbdp, rbd_offset;
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;

	outw(PIOR0(base), fd_p + 6);
	rbd_offset = inw(PIOP0(base));
	if ((f_rbdp = rbd_offset) != I82586NULL) {
		l_rbdp = f_rbdp;
		for (;;) {
			outw(PIOR0(base), l_rbdp + 0);	/* address of status */
			if (inw(PIOP0(base)) & RBD_SW_EOF) break;
			outw(PIOP0(base), 0);
			outw(PIOR0(base), l_rbdp + 2);	/* next_rbd_offset */
			if ((l_rbdp = inw(PIOP0(base))) == I82586NULL) break;
		}
		outw(PIOP0(base), 0);
		outw(PIOR0(base), l_rbdp + 2);	/* next_rbd_offset */
		outw(PIOP0(base), I82586NULL);
		outw(PIOR0(base), l_rbdp + 8);	/* address of size */
		outw(PIOP0(base), inw(PIOP0(base)) | AC_CW_EL);
		outw(PIOR0(base), sc->end_rbd + 2);
		outw(PIOP0(base), f_rbdp);	/* end_rbd->next_rbd_offset */
		outw(PIOR0(base), sc->end_rbd + 8);	/* size */
		outw(PIOP0(base), inw(PIOP0(base)) & ~AC_CW_EL);
		sc->end_rbd = l_rbdp;
	}

	fd.status = 0;
	fd.command = AC_CW_EL;
	fd.link_offset = I82586NULL;
	fd.rbd_offset = I82586NULL;
	outw(PIOR1(base), fd_p);
	outsw(PIOP1(base), &fd, 8/2);

	outw(PIOR1(base), sc->end_fd + 2);	/* addr of command */
	outw(PIOP1(base), 0);		/* command = 0 */
	outw(PIOP1(base), fd_p);	/* end_fd->link_offset = fd_p */
	sc->end_fd = fd_p;
}

/*
 * wlxmt:
 *
 *	This routine fills in the appropriate registers and memory
 *	locations on the WaveLAN board and starts the board off on
 *	the transmit.
 *
 * input	: board number of interest, and a pointer to the mbuf
 * output	: board memory and registers are set for xfer and attention
 *
 */
void
wlxmt(unit, m)
	int unit;
	struct mbuf *m;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	register u_short		xmtdata_p = OFFSET_TBUF;
	register u_short		xmtshort_p;
	struct mbuf			*tm_p = m;
	register struct ether_header	*eh_p = mtod(m, struct ether_header *);
	u_char				*mb_p = mtod(m, u_char *) + sizeof(struct ether_header);
	u_short				count = m->m_len - sizeof(struct ether_header);
	ac_t				cb;
	u_short				tbd_p = OFFSET_TBD;
	u_short				len, clen = 0;
	short				base = sc->base;

	dprintf(("wl%d: wlxmt()\n", unit));

	cb.ac_status = 0;
	cb.ac_command = (AC_CW_EL|AC_TRANSMIT|AC_CW_I);
	cb.ac_link_offset = I82586NULL;
	outw(PIOR1(base), OFFSET_CU);
	outsw(PIOP1(base), &cb, 6/2);
	outw(PIOP1(base), OFFSET_TBD);	/* cb.cmd.transmit.tbd_offset */
	outsw(PIOP1(base), eh_p->ether_dhost, WAVELAN_ADDR_SIZE/2);
	outw(PIOP1(base), eh_p->ether_type);

	dprintf(("wl%d: xmit mbuf: L%d @%x ", unit, count, mb_p));

	outw(PIOR0(base), OFFSET_TBD);
	outw(PIOP0(base), 0);		/* act_count */
	outw(PIOR1(base), OFFSET_TBD + 4);
	outw(PIOP1(base), xmtdata_p);	/* buffer_addr */
	outw(PIOP1(base), 0);		/* buffer_base */
	do {
		if (count) {
			if (clen + count > WAVELAN_MTU) break;
			if (count & 1) len = count + 1;
			else len = count;
			outw(PIOR1(base), xmtdata_p);
			outsw(PIOP1(base), mb_p, len/2);
			clen += count;
			outw(PIOR0(base), tbd_p);  /* address of act_count */
			outw(PIOP0(base), inw(PIOP0(base)) + count);
			xmtdata_p += len;
			if ((tm_p = tm_p->m_next) == (struct mbuf *)0) break;
			if (count & 1) {
				/* go to the next descriptor */
				outw(PIOR0(base), tbd_p + 2);
				tbd_p += sizeof (tbd_t);
				outw(PIOP0(base), tbd_p); /* next_tbd_offset */
				outw(PIOR0(base), tbd_p);
				outw(PIOP0(base), 0);	/* act_count */
				outw(PIOR1(base), tbd_p + 4);
				outw(PIOP1(base), xmtdata_p); /* buffer_addr */
				outw(PIOP1(base), 0);	      /* buffer_base */
				/* at the end -> coallesce remaining mbufs */
				if (tbd_p == OFFSET_TBD + (N_TBD-1) * sizeof (tbd_t)) {
					wlsftwsleaze(&count, &mb_p, &tm_p, unit);
					continue;
				}
				/* next mbuf short -> coallesce as needed */
				if ( (tm_p->m_next == (struct mbuf *) 0) ||
#define HDW_THRESHOLD 55
				      tm_p->m_len > HDW_THRESHOLD)
				      	/* ok */;
				else {
					wlhdwsleaze(&count, &mb_p, &tm_p, unit);
					continue;
				}
			}
		} else if ((tm_p = tm_p->m_next) == (struct mbuf *)0)
			break;
		count = tm_p->m_len;
		mb_p = mtod(tm_p, u_char *);

		dprintf(("mbuf+ L%d @%x ", count, mb_p));

	} while (1);

	dprintf(("CLEN = %d\n", clen));

	outw(PIOR0(base), tbd_p);
	if (clen < ETHERMIN) {
		outw(PIOP0(base), inw(PIOP0(base)) + ETHERMIN - clen);
		outw(PIOR1(base), xmtdata_p);
		for (xmtshort_p = xmtdata_p; clen < ETHERMIN; clen += 2)
			outw(PIOP1(base), 0);
	}
	outw(PIOP0(base), inw(PIOP0(base)) | TBD_SW_EOF);
	outw(PIOR0(base), tbd_p + 2);
	outw(PIOP0(base), I82586NULL);

	outw(PIOR0(base), OFFSET_SCB + 2);	/* address of scb_command */
	while (inw(PIOP0(base))) ;
	outw(PIOP0(base), SCB_CU_STRT);
	SET_CHAN_ATTN(unit);

	m_freem(m);
	return;
}

/*
 * wlbldru:
 *
 *	This function builds the linear linked lists of fd's and
 *	rbd's.  Based on page 4-32 of 1986 Intel microcom handbook.
 *
 */
u_short
wlbldru(unit)
	int unit;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	short	base = sc->base;
	fd_t	fd;
	rbd_t	rbd;
	u_short	fd_p = OFFSET_RU;
	u_short	rbd_p = OFFSET_RBD;
	int 	i;

	sc->begin_fd = fd_p;
	for (i = 0; i < N_FD; i++) {
		fd.status = 0;
		fd.command = 0;
		fd.link_offset = fd_p + sizeof(fd_t);
		fd.rbd_offset = I82586NULL;
		outw(PIOR1(base), fd_p);
		outsw(PIOP1(base), &fd, 8/2);
		fd_p = fd.link_offset;
	}
	fd_p -= sizeof(fd_t);
	sc->end_fd = fd_p;
	outw(PIOR1(base), fd_p + 2);
	outw(PIOP1(base), AC_CW_EL);	/* command */
	outw(PIOP1(base), I82586NULL);	/* link_offset */
	fd_p = OFFSET_RU;

	outw(PIOR0(base), fd_p + 6);	/* address of rbd_offset */
	outw(PIOP0(base), rbd_p);
	outw(PIOR1(base), rbd_p);
	for (i = 0; i < N_RBD; i++) {
		rbd.status = 0;
		rbd.buffer_addr = rbd_p + sizeof(rbd_t) + 2;
		rbd.buffer_base = 0;
		rbd.size = RCVBUFSIZE;
		if (i != N_RBD-1) {
			rbd_p += sizeof(ru_t);
			rbd.next_rbd_offset = rbd_p;
		} else {
			rbd.next_rbd_offset = I82586NULL;
			rbd.size |= AC_CW_EL;
			sc->end_rbd = rbd_p;
		}
		outsw(PIOP1(base), &rbd, sizeof(rbd_t)/2);
		outw(PIOR1(base), rbd_p);
	}
	return sc->begin_fd;
}

/*
 * wlrustrt:
 *
 *	This routine starts the receive unit running.  First checks if the
 *	board is actually ready, then the board is instructed to receive
 *	packets again.
 *
 */
void
wlrustrt(unit)
	int unit;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;
	u_short rfa;

	dprintf(("wl%d: wlrustrt()\n", unit));

	outw(PIOR0(base), OFFSET_SCB);
	if (inw(PIOP0(base)) & SCB_RUS_READY) {
		dprintf(("wl%d: wlrustrt: RUS_READY\n", unit));
		return;
	}

	outw(PIOR0(base), OFFSET_SCB + 2);
	outw(PIOP0(base), SCB_RU_STRT);		/* command */
	rfa = wlbldru(unit);
	outw(PIOR0(base), OFFSET_SCB + 6);	/* address of scb_rfa_offset */
	outw(PIOP0(base), rfa);

	SET_CHAN_ATTN(unit);
	return;
}

/*
 * wldiag:
 *
 *	This routine does a 586 op-code number 7, and obtains the
 *	diagnose status for the WaveLAN.
 *
 */
int
wldiag(unit)
	int unit;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;

	dprintf(("wl%d: wldiag()\n", unit));

	outw(PIOR0(base), OFFSET_SCB);
	if (inw(PIOP0(base)) & SCB_SW_INT) {
		/*???*/
		wlack(unit);
	}
	outw(PIOR1(base), OFFSET_CU);
	outw(PIOP1(base), 0);			/* ac_status */
	outw(PIOP1(base), AC_DIAGNOSE|AC_CW_EL);/* ac_command */
	if (!wlcmd(unit, "wldiag()")) return 0;
	outw(PIOR0(base), OFFSET_CU);
	if (inw(PIOP0(base)) & 0x0800) {
		dprintf(("wl%d: i82586 self test failed\n", unit));
		return 0;
	}
	return 1;
}

/*
 * wlconfig:
 *
 *	This routine does a standard config of the WaveLAN board.
 *
 */
int
wlconfig(unit)
	int unit;
{
	configure_t configure;
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;
#ifdef	MULTICAST
	int cnt = 0;
	struct ether_multi *enm;
	struct ether_multistep step;
#endif

	dprintf(("wl%d: wlconfig()\n", unit));

	outw(PIOR0(base), OFFSET_SCB);
	if (inw(PIOP0(base)) & SCB_SW_INT) {
		/*???*/
	}
	wlack(unit);

	outw(PIOR1(base), OFFSET_CU);
	outw(PIOP1(base), 0);				/* ac_status */
	outw(PIOP1(base), AC_CONFIGURE|AC_CW_EL);	/* ac_command */

#if 1
	/* This is the configuration block suggested by Marc Meertens
	 * <mmeerten@obelix.utrecht.NCR.COM> in an e-mail message to John
	 * Ioannidis on 10 Nov 92.
	 */
/*???	configure.fifolim_bytecnt 	= 0x040c; */
	configure.fifolim_bytecnt 	= 0x080c;
 	configure.addrlen_mode  	= 0x0600;
	configure.linprio_interframe	= 0x2060;
/*???	configure.slot_time      	= 0xf000; */
	configure.slot_time      	= 0xf200;
	configure.hardware	     	= 0x0008;	/* tx even w/o CD */
	configure.min_frame_len   	= 0x0040;
#else
	/*
	 * below is the default board configuration from p2-28 from 586 book
	 */
	configure.fifolim_bytecnt 	= 0x080c;
	configure.addrlen_mode  	= 0x2600;
	configure.linprio_interframe	= 0x7820;	/* IFS=120, ACS=2 */
	configure.slot_time      	= 0xf00c;	/* slottime=12    */
	configure.hardware	     	= 0x0008;	/* tx even w/o CD */
	configure.min_frame_len   	= 0x0040;
#endif
	if (sc->mode & MOD_PROM) configure.hardware |= 1;
	outw(PIOR1(base), OFFSET_CU + 6);
	outsw(PIOP1(base), &configure, sizeof(configure_t)/2);

	if (!wlcmd(unit, "wlconfig()-configure")) return 0;

#ifdef	MULTICAST
	outw(PIOR1(base), OFFSET_CU);
	outw(PIOP1(base), 0);				/* ac_status */
	outw(PIOP1(base), AC_MCSETUP|AC_CW_EL);		/* ac_command */
	outw(PIOR1(base), OFFSET_CU + 8);
	ETHER_FIRST_MULTI(step, &sc->wl_ac, enm);
	while (enm != NULL) {
		unsigned int lo, hi;
		lo = (enm->enm_addrlo[3] << 16) + (enm->enm_addrlo[4] << 8)
			+ enm->enm_addrlo[5];
		hi = (enm->enm_addrhi[3] << 16) + (enm->enm_addrhi[4] << 8)
			+ enm->enm_addrhi[5];
		while (lo <= hi) {
			outw(PIOP1(base),enm->enm_addrlo[0] +
			     (enm->enm_addrlo[1] << 8));
			outw(PIOP1(base),enm->enm_addrlo[2] +
			     ((lo >> 8) & 0xff00));
			outw(PIOP1(base), ((lo >> 8) & 0xff) +
			     ((lo << 8) & 0xff00));
			++cnt;
			++lo;
		}
		ETHER_NEXT_MULTI(step, enm);
	}
	outw(PIOR1(base), OFFSET_CU + 6);		/* mc-cnt */
	outw(PIOP1(base), cnt * WAVELAN_ADDR_SIZE);
	if (!wlcmd(unit, "wlconfig()-mcaddress")) return 0;
#endif

	outw(PIOR1(base), OFFSET_CU);
	outw(PIOP1(base), 0);				/* ac_status */
	outw(PIOP1(base), AC_IASETUP|AC_CW_EL);		/* ac_command */
	outw(PIOR1(base), OFFSET_CU + 6);
	outsw(PIOP1(base), sc->wl_addr, WAVELAN_ADDR_SIZE/2);

	if (!wlcmd(unit, "wlconfig()-address")) return 0;
	wlinitmmc(unit);
	return 1;
}

/*
 * wlcmd:
 *
 * Set channel attention bit and busy wait until command has
 * completed. Then acknowledge the command completion.
 */
int
wlcmd(unit, str)
	int unit;
	char *str;		/* diagnostic string */
{
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;
	int i;
	
	outw(PIOR0(base), OFFSET_SCB + 2);	/* address of scb_command */
	outw(PIOP0(base), SCB_CU_STRT);

	SET_CHAN_ATTN(unit);

	outw(PIOR0(base), OFFSET_CU);
	for(i = 0; i < 0xffff; i++)
		if (inw(PIOP0(base)) & AC_SW_C) break;

	if (i == 0xffff || !(inw(PIOP0(base)) & AC_SW_OK)) {
		printf("wl%d: %s failed; status = %d, inw = %x, outw = %x\n",
		       unit, str, inw(PIOP0(base)) & AC_SW_OK, inw(PIOP0(base)), inw(PIOR0(base)));
		outw(PIOR0(base), OFFSET_SCB);
		printf("\tscb_status %x\n", inw(PIOP0(base)));
		outw(PIOR0(base), OFFSET_SCB+2);
		printf("\tscb_command %x\n", inw(PIOP0(base)));
		outw(PIOR0(base), OFFSET_SCB+4);
		printf("\tscb_cbl %x\n", inw(PIOP0(base)));
		outw(PIOR0(base), OFFSET_CU+2);
		printf("\tcu_cmd %x\n", inw(PIOP0(base)));
		return 0;
	}

	outw(PIOR0(base), OFFSET_SCB);
	if ((inw(PIOP0(base)) & SCB_SW_INT) && (inw(PIOP0(base)) != SCB_SW_CNA)) {
		/*???*/
	}
	wlack(unit);
	return 1;
}

/*
 * wlack: if the 82596 wants attention because it has finished
 * sending or receiving a packet, acknowledge its desire and
 * return bits indicating the kind of attention. wlack() returns
 * these bits so that the caller can service exactly the
 * conditions that wlack() acknowledged.
 */
int
wlack(unit)
	int unit;
{
	int i;
	register u_short cmd;
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;

	outw(PIOR1(base), OFFSET_SCB);
	if (!(cmd = (inw(PIOP1(base)) & SCB_SW_INT))) return 0;

	dprintf(("wl%d: wlack()\n", unit));

	outw(PIOP1(base), cmd);
	SET_CHAN_ATTN(unit);
	outw(PIOR0(base), OFFSET_SCB + 2);	/* address of scb_command */
	for (i = 1000000; inw(PIOP0(base)) && (i-- > 0); );
	if (i < 1) printf("wl%d: board not accepting command\n", unit);
	return cmd;
}

void
wlhdwsleaze(countp, mb_pp, tm_pp, unit)
	u_short *countp;
	u_char **mb_pp;
	struct mbuf **tm_pp;
	int unit;
{
	struct mbuf	*tm_p = *tm_pp;
	u_char		*mb_p = *mb_pp;
	u_short		count = 0;
	u_char		*cp;
	int		len;

	/*
	 * can we get a run that will be coallesced or
	 * that terminates before breaking
	 */
	do {
		count += tm_p->m_len;
		if (tm_p->m_len & 1) break;
	} while ((tm_p = tm_p->m_next) != (struct mbuf *)0);

	if ((tm_p == (struct mbuf *)0) || count > HDW_THRESHOLD) {
		*countp = (*tm_pp)->m_len;
		*mb_pp = mtod((*tm_pp), u_char *);
		printf("\n");
		return;
	}

	/* we need to copy */
	tm_p = *tm_pp;
	mb_p = *mb_pp;
	count = 0;
	cp = (u_char *)t_packet;
	while (1) {
		bcopy(mtod(tm_p, u_char *), cp, len = tm_p->m_len);
		count += len;
		if (count > HDW_THRESHOLD) break;
		cp += len;
		if (tm_p->m_next == (struct mbuf *)0) break;
		tm_p = tm_p->m_next;
	}
	*countp = count;
	*mb_pp = (u_char *) t_packet;
	*tm_pp = tm_p;
}

void
wlsftwsleaze(countp, mb_pp, tm_pp, unit)
	u_short *countp;
	u_char **mb_pp;
	struct mbuf **tm_pp;
	int unit;
{
	struct mbuf	*tm_p = *tm_pp;
	u_char		*mb_p = *mb_pp;
	u_short		count = 0;
	u_char		*cp = (u_char *)t_packet;
	int		len;

	/* we need to copy */
	while (1) {
		bcopy(mtod(tm_p, u_char *), cp, len = tm_p->m_len);
		count += len;
		cp += len;
		if (tm_p->m_next == (struct mbuf *)0) break;
		tm_p = tm_p->m_next;
	}
	*countp = count;
	*mb_pp = (u_char *) t_packet;
	*tm_pp = tm_p;
}

void
wlmmcstat(unit)
	int unit;
{
	register struct wl_softc *sc = WLSOFTC(unit);
	short base = sc->base;
	u_short tmp;

	tmp = wlmmcread(base, MMC_DCE_STATUS) & 0x0f;
	printf("wl%d:\tDCE_STATUS: 0x%x, ", unit, tmp);

	tmp = wlmmcread(base, MMC_NWID_CORRECT_H) << 8;
	tmp |= wlmmcread(base, MMC_NWID_CORRECT_L);
	printf("Correct NWID's: %d, ", tmp);

	tmp = wlmmcread(base, MMC_NWID_WRONG_H) << 8;
	tmp |= wlmmcread(base, MMC_NWID_WRONG_L);
	printf("Wrong NWID's: %d\n", tmp);

	printf("\tSIGNAL_THR=%d, SIGNAL_LVL=%d, SILENCE_LVL=%d, SIGNAL_QUAL=%d\n",
	       wlmmcread(base, MMC_SIGNAL_THR) & 0x3F,
	       wlmmcread(base, MMC_SIGNAL_LVL) & 0x3F,
	       wlmmcread(base, MMC_SILENCE_LVL) & 0x3F,
	       wlmmcread(base, MMC_SIGNAL_QUAL) & 0x0F);
}
