# testutil.tcl --
#
# Utility procedures for TCL test packages.  These procedures all expect a 
# global "ModuleName" to be define, which contains the name of the test package
# being run.  This may be a short string.


# Check that a value is equal to what was expected.  If not, generate
# an error

proc check {got expect idnum} {
    global ModuleName currentTestNumber
    set currentTestNumber $idnum
    if {"$got" != "$expect"} {
        echo [replicate "-" 70]
        echo [format {%s test error %s: wanted "%s", got "%s"} \
               $ModuleName $idnum $expect $got]
        echo [replicate "-" 70]
    }
}


# Check that a numeric value is equal to what was expected.  If not, generate
# an error

proc checknum {got expect idnum} {
    global ModuleName currentTestNumber
    set currentTestNumber $idnum
    if {$got != $expect} {
        echo [replicate "-" 70]
        echo [format {%s test error %s: wanted "%s", got "%s"} \
              $ModuleName $idnum $expect $got]
        echo [replicate "-" 70]
    }
}

# Run and verify a command that should return TCL_ERROR, pass back the returned
# error message

proc do1cmd {cmd msgvar idnum} {
    global ModuleName currentTestCommand currentTestNumber
    set currentTestCommand $cmd
    set currentTestNumber $idnum
    if {[catch {uplevel $cmd} msg] != 1} {
        echo [replicate "-" 70]
        echo [format {%s test error %s: No error returned: %s} \
              $ModuleName $idnum $cmd]
        echo [replicate "-" 70]
    }
    uplevel set $msgvar [list $msg]
}


# Increment a name.  This takes a name and "adds one" to it, that is advancing
# each digit lexically through "0"..."9" -> "A"-"Z" -> "a"..."z".  When one
# digit wraps, the next one is advanced.  Optional arg forces upper case only
# if true and start with all upper case or digits.

proc IncrName {Name args} {
    set Upper [expr {([llength $args] == 1) && [lindex $args 0]}]
    set Last  [expr [clength $Name]-1]
    set Begin [csubstr $Name 0 $Last]
    set Digit [cindex $Name $Last]
    set Recurse 0
    case $Digit in {
        {9}     {set Digit A}
        {Z}     {if {$Upper} {set Recurse 1} else {set Digit a}}
        {z}     {set Recurse 1}
        default {set Digit [ctype char [expr [ctype ord $Digit]+1]]}
    }
    if {$Recurse} {
        if {$Last == 0} then {
            return 0 ;# Wrap around
        } else {
            return "[IncrName $Begin]0"
        }
    }
    return "$Begin$Digit"
}


