#==============================================================================
# installTcl.tcl -- 
#
# Tcl program to install Tcl onto the system.  It is run in the following
# manner:
#
#     tcl installTcl.tcl configFile
#
# configFile is a Tcl file that is sourced and contains and sets the following
# variables:  See the makefile for the definition of each of the variables:
#
#   o TCL_UCB_DIR
#   o TCL_DEFAULT
#   o TCL_OWNER
#   o TCL_GROUP
#   o TCL_BINDIR
#   o TCL_LIBDIR
#   o TCL_INCLUDEDIR
#   o TCL_TCLDIR
#   o TCL_MAN_INSTALL
#   o TCL_MAN_BASEDIR
#   o TCL_MAN_SECTION
#   o TCL_MAN_STYLE
#   o TCL_MAN_INDEX
#   o TCL_MAN_INDEX_MERGE
#
# Notes:
#   Must be run in the Tcl top level directory.
#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

#------------------------------------------------------------------------------
# GiveAwayFile --
#   Give away a file to the Tcl owner and group and set its permissions.
#
# Globals:
#    TCL_OWNER - Owner name for Tcl files.
#    TCL_GROUP - Group nmae for Tcl file.
#------------------------------------------------------------------------------

proc GiveAwayFile {file} {
    global TCL_OWNER TCL_GROUP

    if {[file isdirectory $file]} {
        chmod a+rx,go-w $file
    } else {
        chmod a+r,go-w $file
    }    
    chown [list $TCL_OWNER $TCL_GROUP] $file

} ;# GiveAwayFile

#------------------------------------------------------------------------------
# MakePath --
#
# Make sure all directories in a directory path exists, if not, create them.
#------------------------------------------------------------------------------
proc MakePath {pathlist} {
    foreach path $pathlist {
        set exploded_path [split $path /]
        set thisdir {}
        foreach element $exploded_path {
            append thisdir $element
            if {![file isdirectory $thisdir]} {
                mkdir $thisdir
                GiveAwayFile $thisdir
            }
            append thisdir /
        }
    }
}

#------------------------------------------------------------------------------
# CopyFile -- 
#
# Copy the specified file and change the ownership.  If target is a directory,
# then the file is copied to it, other target is a new file name.
#------------------------------------------------------------------------------

proc CopyFile {sourceFile target} {

    if {[file isdirectory $target]} {
        set targetFile "$target/[file tail $sourceFile]"
    } else {
        set targetFile $target
    }

    set sourceFH [open $sourceFile r]
    set targetFH [open $targetFile w]
    copyfile $sourceFH $targetFH
    close $sourceFH
    close $targetFH
    GiveAwayFile $targetFile

} ;# CopyFile

#------------------------------------------------------------------------------
# CopySubDir --
#
# Recursively copy part of a directory tree, changing ownership and 
# permissions.  This is a utility routine that actually does the copying.
#------------------------------------------------------------------------------

proc CopySubDir {sourceDir destDir} {
    foreach sourceFile [glob -nocomplain $sourceDir/*] {

        if [file isdirectory $sourceFile] {
            set destFile $destDir/[file tail $sourceFile]
            if {![file exists $destFile]} {
                mkdir $destFile}
            GiveAwayFile $destFile
            CopySubDir $sourceFile $destFile
        } else {
            CopyFile $sourceFile $destDir
        }
    }
} ;# CopySubDir

#------------------------------------------------------------------------------
# CopyDir --
#
# Recurisvely copy a directory tree.
#------------------------------------------------------------------------------

proc CopyDir {sourceDir destDir} {

    set cwd [pwd]
    if ![file exists $sourceDir] {
        error "\"$sourceDir\" does not exist"
    }
    if ![file isdirectory $sourceDir] {
        error "\"$sourceDir\" isn't a directory"
    }
    if {![file exists $destDir]} {
        mkdir $destDir
        GiveAwayFile $destDir
    }
    if ![file isdirectory $destDir] {
        error "\"$destDir\" isn't a directory"
    }
    cd $sourceDir
    set status [catch {CopySubDir . $destDir} msg]
    cd $cwd
    if {$status != 0} {
        global errorInfo errorCode
        error $msg $errorInfo $errorCode
    }
}

#------------------------------------------------------------------------------
# GenDefaultFile -- 
#
# Generate the tcl defaults file.
#------------------------------------------------------------------------------

proc GenDefaultFile {defaultFileBase sourceDir} {

    set defaultFile "$defaultFileBase[infox version]"

    if ![file writable [file dirname $defaultFile]] {
        puts stderr "Can't create $defaultFile -- directory is not writable"
        puts stderr "Please reinstall with correct permissions or rebuild"
        puts stderr "Tcl to select a default file where the directory path"
        puts stderr "you specify is writable by you."
        puts stderr ""
        puts stderr "Tcl will still be runnable from the current directory,"
        puts stderr "but maybe not any others..."
        puts stderr ""
        exit 1
    }

    set fp [open $defaultFile w]

    puts $fp "# Extended Tcl [infox version] default file"
    puts $fp ""
    puts $fp "set TCLINIT $sourceDir/TclInit.tcl"
    puts $fp ""
    puts $fp "set TCLPATH $sourceDir"

    close $fp
    GiveAwayFile $defaultFile

} ;# GenDefaultFile

#------------------------------------------------------------------------------
# InstallShortMan --
#   Install a manual page on a system that does not have long file names,
#   optionally adding an entry to the man index.
#
# Parameters:
#   o sourceDir - Directory containing the file.
#   o manNames - Name entry created from the name line of the file by
#     buildhelp.  Has file name and the names it is to be known by.
#   o indexFileHdl - File handle of the current index file being created, or
#     empty if no index is to be created.
# Globals
#   o TCL_MAN_BASEDIR - Base manual directory where all of the man.* and cat.* 
#     directories live.
#   o TCL_MAN_SECTION - The section that the manual file is to go in.
#   o TCL_MAN_SEPARATOR - The name separator between the directory and the
#     section.
#------------------------------------------------------------------------------

proc InstallShortMan {sourceDir manNames indexFileHdl} {
    global TCL_MAN_BASEDIR TCL_MAN_SECTION TCL_MAN_SEPARATOR

    set srcManFilePath "$sourceDir/[lindex $manNames 0]"
    set manFileBase [file tail [file root $srcManFilePath]]

    set destManDir "$TCL_MAN_BASEDIR/man$TCL_MAN_SEPARATOR$TCL_MAN_SECTION"

    CopyFile $srcManFilePath "$destManDir/$manFileBase.$TCL_MAN_SECTION"

    if {![lempty $indexFileHdl]} {
        foreach name [lindex $manNames 1] {
            puts $indexFileHdl "$name\t$manFileBase\t$TCL_MAN_SECTION"
        }
    }    

} ;# InstallShortMan

#------------------------------------------------------------------------------
# InstallShortManPages --
#   Install the manual pages using the short file name scheme.
#------------------------------------------------------------------------------

proc InstallShortManPages {} {
    global TCL_UCB_DIR   TCL_MAN_BASEDIR TCL_MAN_SECTION TCL_MAN_SEPARATOR
    global TCL_MAN_INDEX TCL_MAN_INDEX_MERGE

    set targetDir "$TCL_MAN_BASEDIR/man$TCL_MAN_SEPARATOR$TCL_MAN_SECTION"

    MakePath  $TCL_MAN_BASEDIR 
    MakePath  $targetDir
    MakePath "$TCL_MAN_BASEDIR/cat$TCL_MAN_SEPARATOR$TCL_MAN_SECTION"

    if {$TCL_MAN_INDEX} {
        set tclIndexFile $TCL_MAN_BASEDIR/index.TCL
        set indexFileHdl [open $tclIndexFile w]
    } else {
        set indexFileHdl {}
    }

    # Install all of the actual files.

    echo "    Installing Tcl 6.1 man files to $targetDir"

    for_file manNames "ucbsrc/ucbman.names" {
        InstallShortMan $TCL_UCB_DIR/doc $manNames $indexFileHdl
    }

    echo "    Installing Extended Tcl man files to $targetDir"

    for_file manNames "man/extdman.names" {
        InstallShortMan man $manNames $indexFileHdl
    }

    if {$TCL_MAN_INDEX} {
        close $indexFileHdl
        GiveAwayFile $tclIndexFile
    }

    # Merge the manual index, if requested.

    if {$TCL_MAN_INDEX_MERGE} {
        set indexFile $TCL_MAN_BASEDIR/index
        if {![file exists $indexFile]} {
            echo ""
            echo [replicate "*" 60]
            echo "* `$indexFile' man index file found."
            echo "* you may not have manual indexs on this system."
            echo "* File `$tclIndexFile' built,"
            echo "* but indexes not merged."
            echo [replicate "*" 60]
            echo ""
        } else {
            echo "    Generating new manual index: $indexFile"
            exec cat $indexFile $tclIndexFile | sort -u > ${indexFile}.new
            exec mv $indexFile ${indexFile}.bak
            exec mv ${indexFile}.new $indexFile
            GiveAwayFile $indexFile
        }
    }    
} ;# InstallShortManPages

#------------------------------------------------------------------------------
# InstallLongMan --
#   Install a manual page on a system that does have long file names.
#
# Parameters:
#   o sourceDir - Directory containing the file.
#   o manNames - Name entry created from the name line of the file by
#     buildhelp.  Has file name and the names it is to be known by.
# Globals
#   o TCL_MAN_BASEDIR - Base manual directory where all of the man.* and cat.* 
#     directories live.
#   o TCL_MAN_SECTION - The section that the manual file is to go in.
#   o TCL_MAN_SEPARATOR - The name separator between the directory and the
#     section.
#------------------------------------------------------------------------------

proc InstallLongMan {sourceDir manNames} {
    global TCL_MAN_BASEDIR TCL_MAN_SECTION TCL_MAN_SEPARATOR

    set srcManFilePath "$sourceDir/[lindex $manNames 0]"
    set manFileBase [file tail [file root $srcManFilePath]]

    set manLongNames [lindex $manNames 1]

    set destManDir "$TCL_MAN_BASEDIR/man$TCL_MAN_SEPARATOR$TCL_MAN_SECTION"
    set destManFile "$destManDir/[lvarpop manLongNames].$TCL_MAN_SECTION"

    # Copy file to the first name in the list.

    CopyFile $srcManFilePath $destManFile

    # Link it to the rest of the names in the list.

    foreach manEntry $manLongNames {
        link $destManFile "$destManDir/$manEntry.$TCL_MAN_SECTION"
    }

} ;# InstallLongMan

#------------------------------------------------------------------------------
# InstallLongManPages --
#   Install the manual pages using the long file name scheme.
#------------------------------------------------------------------------------

proc InstallLongManPages {} {
    global TCL_UCB_DIR   TCL_MAN_BASEDIR TCL_MAN_SECTION TCL_MAN_SEPARATOR

    set targetDir "$TCL_MAN_BASEDIR/man$TCL_MAN_SEPARATOR$TCL_MAN_SECTION"

    MakePath  $TCL_MAN_BASEDIR 
    MakePath  $targetDir
    MakePath "$TCL_MAN_BASEDIR/cat$TCL_MAN_SEPARATOR$TCL_MAN_SECTION"

    # Install all of the actual files.

    echo "    Installing Tcl 6.1 man files to $targetDir"

    for_file manNames "ucbsrc/ucbman.names" {
        InstallLongMan $TCL_UCB_DIR/doc $manNames
    }

    echo "    Installing Extended Tcl man files to $targetDir"

    for_file manNames "man/extdman.names" {
        InstallLongMan man $manNames
    }

} ;# InstallLongManPages

#------------------------------------------------------------------------------
# Main program code.
#------------------------------------------------------------------------------

echo ""
echo ">>> Installing Extended Tcl [infox version] <<<"

set argc [llength $argv]
if {$argc != 1} {
    puts stderr "usage: tcl installTcl.tcl configFile"
    exit 1
}

global TCL_UCB_DIR     TCL_DEFAULT     TCL_OWNER         TCL_GROUP  TCL_BINDIR
global TCL_LIBDIR      TCL_INCLUDEDIR  TCL_TCLDIR        TCL_MAN_INSTALL
global TCL_MAN_BASEDIR TCL_MAN_SECTION TCL_MAN_SEPARATOR TCL_MAN_STYLE
global TCL_MAN_INDEX   TCL_MAN_INDEX_MERGE

source $argv

global G_longFileNames


#
# Determine if long file names are available
#
set status [catch {set tmpFH [open $libDir/AVeryVeryBigFileName w]}]
if {$status != 0} {
    set G_longFileNames 0
} else {
    close $tmpFH
    unlink $libDir/AVeryVeryBigFileName
    set G_longFileNames 1
}

#
# Make sure all directories exists that we will be installing in.
#

MakePath [list $TCL_TCLDIR [file dirname $TCL_DEFAULT] $TCL_BINDIR]
MakePath [list $TCL_LIBDIR $TCL_INCLUDEDIR $TCL_TCLDIR]

echo "    Creating default file: $TCL_DEFAULT[infox version]"
GenDefaultFile $TCL_DEFAULT $TCL_TCLDIR

echo "    Installing `tcl' program in: $TCL_BINDIR"
CopyFile tcl $TCL_BINDIR
chmod +rx $TCL_BINDIR/tcl

echo "    Installing `libtcl.a' library in: $TCL_LIBDIR"
CopyFile libtcl.a $TCL_LIBDIR

echo "    Installing Tcl .h files in: $TCL_INCLUDEDIR"
CopyFile $TCL_UCB_DIR/tcl.h $TCL_INCLUDEDIR
CopyFile src/tclExtend.h $TCL_INCLUDEDIR
CopyFile src/tcl++.h $TCL_INCLUDEDIR

echo "    Installing Tcl source files in: $TCL_TCLDIR"
foreach srcFile [glob tcllib/*] {
    if {![file isdirectory $srcFile]} {
        CopyFile $srcFile $TCL_TCLDIR
    }
}

echo "    Installing Tcl help files in: $TCL_TCLDIR/help"
CopyDir tcllib/help          $TCL_TCLDIR/help

foreach file [glob $TCL_TCLDIR/*.tlib] {
    buildpackageindex $file
}

if {$TCL_MAN_INSTALL} {
    case $TCL_MAN_STYLE in {
        {short} InstallShortManPages
        {long} InstallLongManPages
        default {error "invalid value for TCL_MAN_STYLE: `$TCL_MAN_STYLE'"}
    }
}

echo "     *** TCL IS NOW INSTALLED ***"

