# TclInit.tcl -- Extended Tcl initialization (see tclshell man page)

global env TCLENV
set TCLENV(inUnknown) 0
set TCLENV(packageIndexesLoaded) 0

#
# Define a package (called from .tndx file only)
#
proc TCLSH:defpkg {packageName libName offset length args} {
    global TCLENV
    set TCLENV(PKG:$packageName) [list $libName $offset $length]
    foreach i $args {
        set TCLENV(PROC:$i) [list P $packageName]
    }
}

#
# Load Ousterhout style index.
#
proc TCLSH:LoadOusterIndex {dir} {
    global TCLENV

    set fileHdl [open $dir/tclIndex]
    while {[gets $fileHdl line] >= 0} {
        if {([string index $line 0] == "#") || ([llength $line] != 2)} {
            continue}
        set filename [lindex $line 1]
        if {"$filename" == "init.tcl"} {
            continue}
        set TCLENV(PROC:[lindex $line 0]) [list F $filename]

    }
    close $fileHdl
}

#
# Load a package library index.
#
proc loadlibindex {libFile} {
    set indexFile [file root $libFile].tndx
    if {![file readable $indexFile] || ([file mtime $indexFile] <
             [file mtime $libFile])} {
        demand_load buildpackageindex
        buildpackageindex $libFile
    }
    source $indexFile
}

#
# Load library indexes along path.
#
proc TCLSH:LoadPackageIndexes {} {
    global TCLPATH
    foreach dir $TCLPATH {
        foreach libFile [glob -nocomplain $dir/*.tlib] {
            loadlibindex $libFile
        }
        if {[file readable $dir/tclIndex]} {
            TCLSH:LoadOusterIndex $dir
        }
    }
}

#
# Unknown command trap handler.
#
proc unknown {cmdName args} {
    global TCLENV
    if $TCLENV(inUnknown) {
        error "recursive unknown command trap: \"$cmdName\""}
    set TCLENV(inUnknown) 1

    if [demand_load $cmdName] {
        set TCLENV(inUnknown) 0
        return [uplevel 1 [list eval $cmdName $args]]
    }

    global env interactiveSession noAutoExec

    if {$interactiveSession && ([info level] == 1) && ([info script] == "") &&
            (!([info exists noAutoExec] && [set noAutoExec]))} {
        if {[file rootname $cmdName] == "$cmdName"} {
            if [info exists env(PATH)] {
                set binpath [searchpath [split $env(PATH) :] $cmdName]
            } else {
                set binpath [searchpath "." $cmdName]
            }
        } else {
            set binpath $cmdName
        }
        if {[file executable $binpath]} {
            uplevel 1 [list system [concat $cmdName $args]]
            set TCLENV(inUnknown) 0
            return
        }
    }
    set TCLENV(inUnknown) 0
    error "invalid command name: \"$cmdName\""
}

#
# Search a path list for a file.
#
proc searchpath {pathlist file} {
    foreach dir $pathlist {
        if {"$dir" == ""} {set dir .}
        if [file exists $dir/$file] {
            return $dir/$file
        }
    }
    return {}
}

#
# Define a proc to be available for demand_load.
#
proc autoload {filenam args} {
    global TCLENV
    foreach i $args {
        set TCLENV(PROC:$i) [list F $filenam]
    }
}

#
# Load a proc from library or autoload file.
#
proc demand_load {name} {
    global TCLENV
    if [info exists TCLENV(PROC:$name)] {
        set autoloadRecord $TCLENV(PROC:$name)
        if {[lindex $autoloadRecord 0] == "F"} {
            load [lindex $autoloadRecord 1]
        } else {
            set pkgInfo $TCLENV(PKG:[lindex $autoloadRecord 1])
            uplevel #0 sourcepart [lindex $pkgInfo 0] [lindex $pkgInfo 1] [lindex $pkgInfo 2]
            if {"[info procs $name]" == ""} {
                return 0}
        }
        return 1
    }

    # Slow path load index file and try again.

    if {!$TCLENV(packageIndexesLoaded)} {
        TCLSH:LoadPackageIndexes
        set TCLENV(packageIndexesLoaded) 1
        return [demand_load $name]
    }
    return 0
}

#
# Search TCLPATH for a file to source.
#
proc load {name} {
    global TCLPATH errorCode
    set where [searchpath $TCLPATH $name]
    if [lempty $where] {
        error "couldn't find $name in Tcl search path" "" "TCLSH FILE_NOT_FOUND"
    }
    uplevel #0 source $where
}

autoload buildidx.tcl buildpackageindex

# == Put any code you want all Tcl programs to include here. ==

if !$interactiveSession return

# == Interactive Tcl session initialization ==

if [file exists ~/.tclrc] {source ~/.tclrc}

set TCLENV(topLevelPromptHook) {global programName; concat "$programName>" }

set TCLENV(downLevelPromptHook) {concat "=>"}

