/*
 * unixcmds.c --
 *
 * Tcl commands to access unix library calls.
 *---------------------------------------------------------------------------
 * Copyright 1991 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#include "tclExtdInt.h"


/*
 *----------------------------------------------------------------------
 *
 * Tcl_ExecvpCmd --
 *     Implements the TCL execvp command:
 *     execvp prog ["arg1...argN"]
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_ExecvpCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (argc < 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " prog [arg..]",
                          (char *) NULL);
        return TCL_ERROR;
    }

    /*
     * Argv always ends in a null.
     */
    if (execvp (argv[1], &argv[1]) < 0) {
        Tcl_AppendResult (interp, argv [0], ": ", argv [1], ": ",
                          Tcl_UnixError (interp), (char *) NULL);
        return TCL_ERROR;
    }

    panic ("no execvp");
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_ForkCmd --
 *     Implements the TCL fork command:
 *     fork
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_ForkCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int pid;

    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv[0], (char *) NULL);
        return TCL_ERROR;
    }

    pid = Tcl_Fork ();
    if (pid < 0) {
        Tcl_AppendResult (interp, argv [0], ": ", Tcl_UnixError (interp),
                          (char *) NULL);
        return TCL_ERROR;
    }

    sprintf(interp->result, "%d", pid);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_KillCmd --
 *     Implements the TCL kill command:
 *        kill [signal] proclist
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_KillCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int     argc;
    char      **argv;
{
    int    signalNum, idx, procId, procArgc, result = TCL_ERROR;
    char **procArgv;

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " [signal] processlist", (char *) NULL);
        return TCL_ERROR;
    }

    if (argc == 2)
        signalNum = SIGTERM;
    else {
        if (!Tcl_StrToInt (argv[1], 0, &signalNum)) {
            signalNum = Tcl_SigNameToNum (argv[1]);
        }
        if ((signalNum < 0) || (signalNum > NSIG)) {
            Tcl_AppendResult (interp, argv [0], ": invalid signal",
                              (char *) NULL);
            return TCL_ERROR;
        }
    }

    if (Tcl_SplitList (interp, argv [argc - 1], &procArgc, 
                       &procArgv) != TCL_OK)
        return TCL_ERROR;

    for (idx = 0; idx < procArgc; idx++) {

        if (Tcl_GetInt (interp, procArgv [idx], &procId) != TCL_OK)
            goto exitPoint;

        if (kill (procId, signalNum) < 0) {
            Tcl_AppendResult (interp, argv [0], ": pid ", procArgv [idx],
                              ": ", Tcl_UnixError (interp), (char *) NULL);
            goto exitPoint;
        }
     }

    result = TCL_OK;
exitPoint:
    ckfree ((char *) procArgv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_AlarmCmd --
 *     Implements the TCL Alarm command:
 *         Alarm seconds
 *
 * Results:
 *      Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_AlarmCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    unsigned time;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " seconds", 
                          (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_GetInt (interp, argv[1], &time) != TCL_OK)
        return TCL_ERROR;

    sprintf (interp->result, "%d", alarm (time));
    return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_SleepCmd --
 *     Implements the TCL sleep command:
 *         sleep seconds
 *
 * Results:
 *      Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_SleepCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    unsigned time;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " seconds", 
                          (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_GetUnsigned (interp, argv[1], &time) != TCL_OK)
        return TCL_ERROR;

    sleep (time);
    return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_SystemCmd --
 *     Implements the TCL system command:
 *     system command
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_SystemCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int exitCode;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " command",
                          (char *) NULL);
        return TCL_ERROR;
    }

    exitCode = Tcl_System (interp, argv[1]);
    if (exitCode == -1)
        return TCL_ERROR;
    sprintf (interp->result, "%d", exitCode);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_TimesCmd --
 *     Implements the TCL times command:
 *     times
 *
 * Results:
 *  Standard TCL results.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_TimesCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    struct tms tm;

    /*
     * Precompute milliseconds-per-tick, the " + CLK_TCK / 2" bit gets it to
     * round off instead of truncate.
     */
#define MS_PER_TICK ((1000 + CLK_TCK/2) / CLK_TCK)

    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv[0], (char *) NULL);
        return TCL_ERROR;
    }

    times(&tm);

    sprintf(interp->result, "%ld %ld %ld %ld", 
            tm.tms_utime  * MS_PER_TICK, 
            tm.tms_stime  * MS_PER_TICK, 
            tm.tms_cutime * MS_PER_TICK, 
            tm.tms_cstime * MS_PER_TICK);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_UmaskCmd --
 *     Implements the TCL umask command:
 *     umask [octalmask]
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_UmaskCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int mask;

    if ((argc < 1) || (argc > 2)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " octalmask",
                          (char *) NULL);
        return TCL_ERROR;
    }

    if (argc == 1) {
        mask = umask (0);  /* Get current mask      */
        umask (mask);      /* Now set it back (yuk) */
        sprintf (interp->result, "%o", mask);
    } else {
        if (!Tcl_StrToInt (argv[1], 8, &mask)) {
            Tcl_AppendResult (interp, "Expected octal number got: ", argv[1],
                              (char *) NULL);
            return TCL_ERROR;
        }

        umask(mask);
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_LinkCmd --
 *     Implements the TCL unlink command:
 *         link srcpath destpath
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_LinkCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " srcpath destpath", (char *) NULL);
        return TCL_ERROR;
    }
    if (link (argv [1], argv [2]) != 0) {
       Tcl_AppendResult (interp, argv [0], ": ", Tcl_UnixError (interp),
                        (char *) NULL);
       return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_UnlinkCmd --
 *     Implements the TCL unlink command:
 *         unlink fileList
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_UnlinkCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int    idx, fileArgc, result = TCL_ERROR;
    char **fileArgv;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " filelist", (char *) NULL);
        return TCL_ERROR;
    }
    if (Tcl_SplitList (interp, argv [1], &fileArgc, &fileArgv) != TCL_OK)
        return TCL_ERROR;

    for (idx = 0; idx < fileArgc; idx++) {
        if (unlink (fileArgv [idx]) != 0) {
           Tcl_AppendResult (interp, argv [0], ": ", fileArgv [idx], ": ",
                             Tcl_UnixError (interp), (char *) NULL);
           goto exitPoint;
        }
    }

    result = TCL_OK;
exitPoint:
    ckfree ((char *) fileArgv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_MkdirCmd --
 *     Implements the TCL Mkdir command:
 *         mkdir [-path] dirList
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_MkdirCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int           idx, dirArgc, result;
    char        **dirArgv, *scanPtr;
    struct stat   statBuf;

    if ((argc < 2) || (argc > 3))
        goto usageError;
    if ((argc == 3) && !STREQU (argv [1], "-path"))
        goto usageError;

    if (Tcl_SplitList (interp, argv [argc - 1], &dirArgc, &dirArgv) != TCL_OK)
        return TCL_ERROR;
    /*
     * Make all the directories, optionally making directories along the path.
     */

    for (idx = 0; idx < dirArgc; idx++) {
        /*
         * Make leading directories, if requested.
         */
        if (argc == 3) {
            scanPtr = dirArgv [idx];
            result = 0;  /* Start out ok, for dirs that are skipped */

            while (*scanPtr != '\0') {
                scanPtr = strchr (scanPtr+1, '/');
                if ((scanPtr == NULL) || (*(scanPtr+1) == '\0'))
                    break;
                *scanPtr = '\0';
                if (stat (dirArgv [idx], &statBuf) < 0)
#ifdef TCL_386BSD
                    result = mkdir (dirArgv [idx], S_IFDIR | 0777);
#else
                    result = mkdir (dirArgv [idx], S_IFDIR | 0777, 0);
#endif
                *scanPtr = '/';
                if (result < 0)
                   goto mkdirError;
            }
        }
        /*
         * Make final directory in the path.
         */
#ifdef TCL_386BSD
        if (mkdir (dirArgv [idx], S_IFDIR | 0777) != 0)
#else
        if (mkdir (dirArgv [idx], S_IFDIR | 0777, 0) != 0)
#endif
           goto mkdirError;
    }

    ckfree ((char *) dirArgv);
    return TCL_OK;

mkdirError:
    Tcl_AppendResult (interp, argv [0], ": ", dirArgv [idx], ": ",
                      Tcl_UnixError (interp), (char *) NULL);
    ckfree ((char *) dirArgv);
    return TCL_ERROR;

usageError:
    Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                      " [-path] dirlist", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_RmdirCmd --
 *     Implements the TCL Rmdir command:
 *         rmdir dirList
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_RmdirCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int    idx, dirArgc, result = TCL_ERROR;
    char **dirArgv;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " dirlist", (char *) NULL);
        return TCL_ERROR;
    }
    if (Tcl_SplitList (interp, argv [1], &dirArgc, &dirArgv) != TCL_OK)
        return TCL_ERROR;

    for (idx = 0; idx < dirArgc; idx++) {
        if (rmdir (dirArgv [idx]) != 0) {
           Tcl_AppendResult (interp, argv [0], ": ", dirArgv [idx], ": ",
                             Tcl_UnixError (interp), (char *) NULL);
           goto exitPoint;
        }
    }

    result = TCL_OK;
exitPoint:
    ckfree ((char *) dirArgv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_WaitCmd --
 *     Implements the TCL wait command:
 *     wait proclist
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_WaitCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int    waitPid, status, idx, procArgc, result = TCL_ERROR;
    char **procArgv;
    int   *procIdList;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " proclist", 
                          (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_SplitList (interp, argv [1], &procArgc, &procArgv) != TCL_OK)
        return TCL_ERROR;

    procIdList = (int *) ckalloc (procArgc * (sizeof (int)));

    for (idx = 0; idx < procArgc; idx++) {
        if (Tcl_GetInt (interp, procArgv [idx], &procIdList [idx]) != TCL_OK)
            goto exitPoint;
    }

    waitPid = Tcl_WaitPids (procArgc, procIdList, &status);

    if (waitPid < 0) {
        Tcl_AppendResult (interp, argv [0], ": ", Tcl_UnixError (interp),
                          (char *) NULL);
        return TCL_ERROR;
    }
    
    if (WIFEXITED (status))
        sprintf (interp->result, "%d %s %d", waitPid, "EXIT", 
                 WEXITSTATUS (status));
    else if (WIFSIGNALED (status))
        sprintf (interp->result, "%d %s %s", waitPid, "SIG", 
                 Tcl_SignalId (WTERMSIG (status)));
    else if (WIFSTOPPED (status))
        sprintf (interp->result, "%d %s %s", waitPid, "STOP", 
                 Tcl_SignalId (WSTOPSIG (status)));

    result = TCL_OK;
exitPoint:
    ckfree ((char *) procArgv);
    ckfree ((char *) procIdList);
    return result;
}
