/*
 * signal.c --
 *
 * Tcl Unix signal support routines and the signal and trap commands.
 *---------------------------------------------------------------------------
 * Copyright 1991 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#include "tclExtdInt.h"


#ifndef SIGCLD
#   define SIGCLD SIGCHLD
#endif
#ifndef SIGCHLD
#   define SIGCHLD SIGCLD
#endif

#ifndef MAXSIG
#    define MAXSIG 32
#endif

/*
 * Signal name table maps name to number.
 */

#define SIG_NAME_MAX 7

static struct {char *name;
        short num;
       } sigNameTable [] = {
    "HUP",     SIGHUP,
    "INT",     SIGINT,
    "QUIT",    SIGQUIT,
    "ILL",     SIGILL,
    "TRAP",    SIGTRAP,
    "IOT",     SIGIOT,
#ifdef SIGABRT
    "ABRT",    SIGABRT,
#endif
    "EMT",     SIGEMT,
    "FPE",     SIGFPE,
    "KILL",    SIGKILL,
    "BUS",     SIGBUS,
    "SEGV",    SIGSEGV,
    "SYS",     SIGSYS,
    "PIPE",    SIGPIPE,
    "ALRM",    SIGALRM,
    "TERM",    SIGTERM,
    "USR1",    SIGUSR1,
    "USR2",    SIGUSR2,
    "CLD",     SIGCLD,
    "CHLD",    SIGCHLD,
#ifdef SIGPWR
    "PWR",     SIGPWR,
#endif
#ifdef SIGPOLL
    "POLL",    SIGPOLL,
#endif
#ifdef SIGSTOP
    "STOP",    SIGSTOP,
#endif
#ifdef SIGTSTP
    "TSTP",    SIGTSTP,
#endif
#ifdef SIGCONT
    "CONT",    SIGCONT,
#endif
#ifdef SIGTTIN
    "TTIN",    SIGTTIN,
#endif
#ifdef SIGTTOU
    "TTOU",    SIGTTOU,
#endif
    NULL,         -1};

#ifdef TCL_SIG_PROC_INT
#   define SIG_PROC_TYPE int
#else
#   define SIG_PROC_TYPE void
#endif

/*
 * Globals that indicate if we got a signal and which ones we got.
 */
int           receivedSignal = FALSE;
static unsigned char signalsRecieved [MAXSIG];

/*
 * Table of commands to evaluate when a signal occurs.  If the command is
 * NULL and the signal is received, an error is returned.
 */
static char *signalTrapCmds [MAXSIG];

/*
 * Prototypes of internal functions.
 */

SIG_PROC_TYPE (*
GetSignalState _ANSI_ARGS_((int signalNum)));

int
SetSignalAction _ANSI_ARGS_((int             signalNum,
                             SIG_PROC_TYPE (*sigFunc)()));

static SIG_PROC_TYPE
TclSignalTrap _ANSI_ARGS_((int signalNum));

static int
EvalTrapCode _ANSI_ARGS_((Tcl_Interp *interp,
                          int         signalNum,
                          char       *command));

static int
ParseSignalList _ANSI_ARGS_((Tcl_Interp *interp,
                             char       *signalListStr,
                             int         signalList []));

static void
SignalCmdCleanUp _ANSI_ARGS_((ClientData clientData));


/*
 *----------------------------------------------------------------------
 *
 * Tcl_SigNameToNum --
 *     Converts a UNIX signal name to its number, returns -1 if not found.
 *     the name may be upper or lower case and may optionally have the 
 *     leading "SIG" omitted.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_SigNameToNum (sigName)
    char *sigName;
{
    char  sigNameUp [SIG_NAME_MAX+1];  /* Upshifted signal name */
    char *sigNamePtr; 
    int   idx;

    /*
     * Copy and upshift requested name.
     */

    if (strlen (sigName) > SIG_NAME_MAX)
        return -1;   /* Name too long */

    Tcl_UpShift (sigNameUp, sigName);

    if (STRNEQU (sigNameUp, "SIG", 3))
        sigNamePtr = &sigNameUp [3];
    else
        sigNamePtr = sigNameUp;

    for (idx = 0; sigNameTable [idx].num != -1; idx++)
        if (STREQU (sigNamePtr, sigNameTable [idx].name))
            break;

    return sigNameTable [idx].num;
}

/*
 *----------------------------------------------------------------------
 *
 * GetSignalState --
 *     Get the current state of the specified signal.
 * Parameters:
 *   o signalNum (I) - Signal number to query.
 * Results
 *   The signal function or SIG_DFL or SIG_IGN.  If an error occures,
 *   SIG_ERR is returned (check errno);
 *----------------------------------------------------------------------
 */
static SIG_PROC_TYPE (*
GetSignalState (signalNum))
    int signalNum;
{
#ifdef TCL_POSIX_SIG
    struct sigaction currentState;

    if (sigaction (signalNum, NULL, &currentState) < 0)
        return SIG_ERR;
    return currentState.sa_handler;
#else
    SIG_PROC_TYPE  (*actionFunc)();

    if (signalNum == SIGKILL)
        return SIG_DFL;

    actionFunc = signal (signalNum, SIG_DFL);
    if (actionFunc == SIG_ERR)
        return SIG_ERR;
    if (actionFunc != SIG_DFL)
        signal (signalNum, actionFunc);  /* reset */
    return actionFunc;
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * SetSignalAction --
 *     Set the action to occur when a signal is received.
 * Parameters:
 *   o signalNum (I) - Signal number to query.
 *   o sigFunc (O) - The signal function or SIG_DFL or SIG_IGN.
 * Results
 *   TRUE if ok,  FALSE if an error (check errno).
 *----------------------------------------------------------------------
 */
static int
SetSignalAction (signalNum, sigFunc)
    int             signalNum;
    SIG_PROC_TYPE (*sigFunc)();
{
#ifdef TCL_POSIX_SIG
    struct sigaction newState;
    sigset_t         sigUnblockSet;
    
    newState.sa_handler = sigFunc;
    sigfillset (&newState.sa_mask);
    newState.sa_flags = 0;

    if (sigaction (signalNum, &newState, NULL) < 0)
        return FALSE;

    sigemptyset (&sigUnblockSet);
    sigaddset (&sigUnblockSet, signalNum);
    if (sigprocmask (SIG_UNBLOCK, &sigUnblockSet, NULL) < 0)
        return FALSE;
    return TRUE;
#else
    if (signal (signalNum, sigFunc) == SIG_ERR)
        return FALSE;
    else
        return TRUE;
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * TclSignalTrap --
 *     Trap handler for UNIX signals.  Sets a flag indicating that the
 *     trap has occured, saves the name and rearms the trap.  The flag
 *     will be seen by the interpreter when its safe to trap.
 * Globals:
 *   o receivedSignal (O) - Set to TRUE, to indicate a signal was received.
 *   o signalsRecieved (O) - The entry indicating which signal we received
 *     will be set to TRUE;
 *----------------------------------------------------------------------
 */
static SIG_PROC_TYPE
TclSignalTrap (signalNum)
    int signalNum;
{
    signalsRecieved [signalNum] = TRUE;
    receivedSignal = TRUE;
#ifdef TCL_POSIX_SIG
    if (signalNum != SIGCHLD) {
        sigset_t sigBlockSet;

        sigemptyset (&sigBlockSet);
        sigaddset (&sigBlockSet, SIGCHLD);
        if (sigprocmask (SIG_BLOCK, &sigBlockSet, NULL) < 0)
            panic ("TclSignalTrap bug");
    }
#else
    if (signalNum != SIGCHLD) {
        if (SetSignalAction (signalNum, TclSignalTrap) < 0)
            panic ("TclSignalTrap bug");
    }
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * EvalTrapCode --
 *     Run code as the result of a signal.  The code will be run in the
 *     global context, with the symbolic signal name in a global variable.
 *     signalReceived.  If an error occured, then the result will be
 *     left in the interp, if no error occured, the result will be reset.
 * Parameters:
 *   o interp (I/O) - The interpreter to run the signal in.
 *   o signalNum (I) - The signal number of the signal that occured.
 *   o command (I) - The command string to execute.
 * Return:
 *   TCL_OK or TCL_ERROR.
 *----------------------------------------------------------------------
 */
static int
EvalTrapCode (interp, signalNum, command)
    Tcl_Interp *interp;
    int         signalNum;
    char       *command;
{
    Interp        *iPtr = (Interp *) interp;
    char          *signalName;
    int            result;
    CallFrame     *savedVarFramePtr;

    Tcl_ResetResult (interp);

    /*
     * Modify the interpreter state to execute in the global frame.
     */
    savedVarFramePtr = iPtr->varFramePtr;
    iPtr->varFramePtr = NULL;

    /*
     * Force name to always be SIGCHLD, even if system defines only SIGCLD.
     */
    if (signalNum == SIGCHLD)
        signalName = "SIGCHLD";
    else
        signalName = Tcl_SignalId (signalNum);

    if (Tcl_SetVar (interp, "signalRecieved", signalName,
                    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
        result = TCL_ERROR;
    else
        result = TCL_OK;
    if (result == TCL_OK);
        result = Tcl_Eval (interp, signalTrapCmds [signalNum], 0, NULL);

    /*
     * Restore the frame pointer and return the result (only OK or ERROR).
     */
    iPtr->varFramePtr = savedVarFramePtr;

    if (result == TCL_ERROR) {
        char errorInfo [TCL_RESULT_SIZE];

        sprintf (errorInfo, "\n    while executing signal trap code for %s%s",
                 signalName, " signal");
        Tcl_AddErrorInfo (interp, errorInfo);

        return TCL_ERROR;
    } else {
        Tcl_ResetResult (interp);
        return TCL_OK;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_ResetSignals --
 *  
 *   Reset all of the signal flags to indicate that no signals have 
 * occured.  This is used by the shell at the beginning of each interactive
 * command
 *
 * Globals:
 *   o receivedSignal (O) - Will be cleared.
 *   o signalsRecieved (O) - The indicates which signal where received.
 *----------------------------------------------------------------------
 */
void
Tcl_ResetSignals ()
{
    int  signalNum;

    receivedSignal = FALSE;
    for (signalNum = 0; signalNum < MAXSIG; signalNum++) 
        signalsRecieved [signalNum] = FALSE;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_CheckForSignal --
 *  
 *   Called by Tcl_Eval to check if a signal was received when Tcl_Eval is in
 * a safe state.  If the signal was received, this handles processing the
 * signal prehaps recursively eval-ing some code.  This is called just after a
 * command completes.  The results of the command are passed to this procedure
 * and may be altered by it.  If trap code is specified for the signal that
 * was received, then the trap will be executed, otherwise an error result
 * will be returned indicating that the signal occured.
 * 
 * Parameters:
 *   o interp (I/O) - Interp->result should contain the result for
 *     the command that just executed.  This will either be restored or
 *     replaced with a new result.
 *   o cmdResultCode (I) - The integer result returned by the command that
 *     Tcl_Eval just completed.
 * Globals:
 *   o receivedSignal (O) - Will be cleared.
 *   o signalsRecieved (O) - The indicates which signal where received.
 * Returns:
 *   Either the original result core, an error result if one of the
 *   trap commands returned an error, or an error indicating the
 *   a signal occured.
 *----------------------------------------------------------------------
 */
int
Tcl_CheckForSignal (interp, cmdResultCode)
    Tcl_Interp *interp;
    int         cmdResultCode;
{
    char   *savedResult;
    int     signalNum, result, retErrorForSignal = -1;

    if (!receivedSignal)
        return cmdResultCode;  /* Not signal received */

    savedResult = ckalloc (strlen (interp->result) + 1);
    strcpy (savedResult, interp->result);
    Tcl_ResetResult (interp);

    for (signalNum = 1; signalNum < MAXSIG; signalNum++) {
        if (signalsRecieved [signalNum]) {
            signalsRecieved [signalNum] = FALSE;
            if (signalTrapCmds [signalNum] == NULL)
                retErrorForSignal = signalNum;
            else {
                result = EvalTrapCode (interp, signalNum,
                                       signalTrapCmds [signalNum]);
                if (result == TCL_ERROR)
                    goto exitPoint;
            }
        }
    }

    if (retErrorForSignal >= 0) {
        char *signalName;

        /*
         * Force name to always be SIGCHLD, even if system defines only SIGCLD.
         */
        if (retErrorForSignal == SIGCHLD)
            signalName = "SIGCHLD";
        else
            signalName = Tcl_SignalId (retErrorForSignal);

        Tcl_SetErrorCode (interp, "UNIX SIG ", signalName, (char*) NULL);
        Tcl_AppendResult (interp, signalName, " signal received", 
                          (char *)NULL);
        result = TCL_ERROR;
    } else {
        Tcl_SetResult (interp, savedResult, TCL_DYNAMIC);
        savedResult = NULL;
        result = cmdResultCode;
    }

exitPoint:
    if (savedResult != NULL)
        ckfree (savedResult);
    /*
     * An error might have caused clearing of some signal flags to be missed.
     */
    Tcl_ResetSignals ();
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ParseSignalList --
 *  
 *   Parse a list of signal names or numbers.
 * 
 * Parameters:
 *   o interp (O) - Interpreter for returning errors.
 *   o signalListStr (I) - The Tcl list of signals to convert.
 *   o signalList (O) - The list of converted signal numbers, must be
 *     big enough to hold MAXSIG signals.
 *     Tcl_Eval just completed.
 * Returns:
 *   The number of signals converted, or -1 if an error occures.
 *----------------------------------------------------------------------
 */
static int
ParseSignalList (interp, signalListStr, signalList)
    Tcl_Interp *interp;
    char       *signalListStr;
    int         signalList [];
{
    char         **signalListArgv;
    int            signalListSize, signalNum, idx;
    int            result = -1;
    char          *signalName;

    if (Tcl_SplitList (interp, signalListStr, &signalListSize, 
                       &signalListArgv) != TCL_OK)
        return -1;

    if (signalListSize > MAXSIG) {
        Tcl_AppendResult (interp, "too many signals supplied in list",
                          (char *) NULL);
        goto exitPoint;
    }

    if (signalListSize == 0) {
        Tcl_AppendResult (interp, "signal list may not be empty",
                          (char *) NULL);
        goto exitPoint;
    }

    for (idx = 0; idx < signalListSize; idx++) {
        signalName = signalListArgv [idx];

        if (Tcl_StrToInt (signalName, 0, &signalNum))
            signalName = Tcl_SignalId (signalNum);
        else
            signalNum = Tcl_SigNameToNum (signalName);

        if (signalName == NULL) {
            char numBuf [20];

            sprintf (numBuf, "%d", signalNum);
            Tcl_AppendResult (interp, "invalid signal number: ",
                              numBuf, (char *) NULL);
            goto exitPoint;
        }

        if ((signalNum < 1) || (signalNum > NSIG)) {
            Tcl_AppendResult (interp, "invalid signal name: ",
                              signalName, (char *) NULL);
            goto exitPoint;
        }
        signalList [idx] = signalNum;
    }

    result = signalListSize;
exitPoint:
    ckfree ((char *) signalListArgv);
    return result;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_SignalCmd --
 *     Implements the TCL signal command:
 *         signal action siglist [command]
 *
 * Results:
 *      Standard TCL results, may return the UNIX system error message.
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */
static int
Tcl_SignalCmd (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int            signalListSize, signalNum, idx;
    int            signalList [MAXSIG];
    char          *signalName;
    SIG_PROC_TYPE  (*actionFunc)();
    int            commandLen = -1;

    if ((argc < 3) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " action signalList [commands]", (char *) NULL);
        return TCL_ERROR;
    }

    signalListSize = ParseSignalList (interp, argv [2], signalList);
    if (signalListSize < 0)    
        return TCL_ERROR;

    /*
     * Determine the action to take on all of the signals.
     */
    if (STREQU (argv [1], "trap")) {
        actionFunc = TclSignalTrap;
        if (argc != 4) {
            Tcl_AppendResult (interp, argv[0], ": command required for ",
                             "trapping signals", (char *) NULL);
            return TCL_ERROR;
        }
        commandLen = strlen (argv [3]);
    } else {
        if (STREQU (argv [1], "default")) {
            actionFunc = SIG_DFL;
        } else if (STREQU (argv [1], "ignore")) {
            actionFunc = SIG_IGN;
        } else if (STREQU (argv [1], "error")) {
            actionFunc = TclSignalTrap;
        } else if (!STREQU (argv [1], "get")) {
            Tcl_AppendResult (interp, "invalid signal action specified: ", 
                              argv [1], ": expected one of \"default\", ",
                              "\"ignore\", \"error\", \"trap\", or \"get\"",
                              (char *) NULL);
            return TCL_ERROR;
        }
    }

    /*
     * Either get or set the signals.
     */
    if (argv [1][0] == 'g') {
        char *actionList [MAXSIG];
        
        for (idx = 0; idx < signalListSize; idx ++) {
            signalNum = signalList [idx];

            actionFunc = GetSignalState (signalNum);
            if (actionFunc == SIG_ERR)
                goto unixSigError;

            if (actionFunc == SIG_DFL) 
                actionList [idx] = "default";
            else if (actionFunc == SIG_IGN)
                actionList [idx] = "ignore";
            else if (actionFunc == TclSignalTrap) {
                if (signalTrapCmds [signalNum] == NULL)
                    actionList [idx] = "error";
                else
                    actionList [idx] = "trap";
            }
        }
        Tcl_SetResult (interp, Tcl_Merge (signalListSize, actionList),
                       TCL_DYNAMIC);
    } else {
        for (idx = 0; idx < signalListSize; idx ++) {
            signalNum = signalList [idx];

            if (signalTrapCmds [signalNum] != NULL) {
                ckfree (signalTrapCmds [signalNum]);
                signalTrapCmds [signalNum] = NULL;
            }
            if (!SetSignalAction (signalNum, actionFunc))
                goto unixSigError;

            if (commandLen > 0) {
                signalTrapCmds [signalNum] = ckalloc (commandLen + 1);
                strcpy (signalTrapCmds [signalNum], argv [3]);
            }
        }
    }
    return TCL_OK;

unixSigError:
    Tcl_AppendResult (interp, "error setting or getting signal: ",
                      Tcl_UnixError (interp), (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 *  SignalCmdCleanUp --
 *      Clean up the signal table when the interpreter is deleted.  This
 *      is actually when the signal command is deleted.  It releases the
 *      all signal commands that have been allocated.
 *
 *----------------------------------------------------------------------
 */
static void
SignalCmdCleanUp (clientData)
    ClientData clientData;
{
    int idx;

    for (idx = 0; idx < MAXSIG; idx++)
        if (signalTrapCmds [idx] != NULL) {
            ckfree (signalTrapCmds [idx]);
            signalTrapCmds [idx] = NULL;
        }

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_InitSignalHandling --
 *      Initializes the TCL unix commands.
 *
 * Side effects:
 *    A catch trap is armed for the SIGINT signal.
 *
 *----------------------------------------------------------------------
 */
void
Tcl_InitSignalHandling (interp)
    Tcl_Interp *interp;
{
    int idx;

    for (idx = 0; idx < MAXSIG; idx++) {
        signalsRecieved [idx] = FALSE;
        signalTrapCmds [idx] = NULL;
    }
    Tcl_CreateCommand (interp, "signal", Tcl_SignalCmd, (ClientData)NULL,
                      SignalCmdCleanUp);
    /*
     * If interrupt is currently  being trapped, enabled it.  Other wise
     * leave it off, or if this process is running as a background job it will
     * get its parent's (shell's) signals.
     */
    if (GetSignalState (SIGINT) != SIG_IGN)
        SetSignalAction (SIGINT, TclSignalTrap);
}

