/* 
 * chmod.c --
 *
 *    Chmod, chown and chgrp Tcl commands.
 *---------------------------------------------------------------------------
 * Copyright 1991 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#include "tclExtdInt.h"

/*
 * Prototypes of internal functions.
 */
int
Tcl_ConvSymMode _ANSI_ARGS_((char *sysPerms,
                             int   modeVal));

int
Tcl_ConvAbsMode _ANSI_ARGS_((char *modeStr,
                             int   assumeInt));


/*
 *----------------------------------------------------------------------
 *
 * Tcl_ConvSymMode --
 *      Parse and convert symbolic file permissions as specified by chmod(C).
 *
 * Parameters:
 *      symPerms - The symbolic permissions to parse.
 *      modeVal - The existing permissions value on a file.
 *
 * Results:
 *      The new permissions, or -1 if invalid permissions where supplied.
 *
 *----------------------------------------------------------------------
 */
static int
Tcl_ConvSymMode (sysPerms, modeVal)
    char *sysPerms;
    int   modeVal;

{
    int  user, group, other;
    char operator;
    int  rwxMask, setUID, sticky, locking;
    int  newMode;

    while (*sysPerms != 0) {
        user = group = other = FALSE;

        /* 
         * Scan who field.
         */
        while (! ((*sysPerms == '+') || 
                  (*sysPerms == '-') || 
                  (*sysPerms == '='))) {
            switch (*sysPerms) {
                case 'a':
                    user = group = other = TRUE;
                    break;
                case 'u':
                    user = TRUE;
                    break;
                case 'g':
                    group = TRUE;
                    break;
                case 'o':
                    other = TRUE;
                    break;
                default:
                    return -1;  /* Error, not a recognized character or EOL */
            }
            sysPerms++;
        }

        /*
         * If none where specified, that means all.
         */

        if (! (user || group || other))
            user = group = other = TRUE;

        operator = *sysPerms++;

        /* 
         * Decode the permissions
         */

        rwxMask = 0;
        setUID = sticky = locking = FALSE;

        /* 
         * Scan permissions field
         */
        while (! ((*sysPerms == ',') || (*sysPerms == 0))) {
            switch (*sysPerms) {
                case 'r':
                    rwxMask |= 4;
                    break;
                case 'w':
                    rwxMask |= 2;
                    break;
                case 'x':
                    rwxMask |= 1;
                    break;
                case 's':
                    setUID = TRUE;
                    break;
                case 't':
                    sticky = TRUE;
                    break;
                case 'l':
                    locking = TRUE;
                    break;
                default:
                    return -1;  /* Error, not a recognized character */
            }
            sysPerms++;
        }

        /*
         * Build mode map of specified values.
         */

        newMode = 0;
        if (user)
            newMode |= rwxMask << 6;
        if (group)
            newMode |= rwxMask << 3;
        if (other)
            newMode |= rwxMask;
        if (setUID && user)
            newMode |= 04000;
        if ((setUID || locking) && group)
            newMode |= 02000;
        if (sticky)
            newMode |= 01000;

        /* 
         * Add to cumulative mode based on operator.
         */

        if (operator == '+')
            modeVal |= newMode;
        else if (operator == '-')
            modeVal &= ~newMode;
        else if (operator == '=')
            modeVal = newMode;
        if (*sysPerms == ',')
            sysPerms++;
    }

    return modeVal;

} /* Tcl_ConvSymMode */

/*
 *----------------------------------------------------------------------
 *
 * Tcl_ConvAbsMode --
 *      Convert a absolute file mode.  If this fails, it might be a 
 *      symbolic mode.  Mode is assumed to be octal unless otherwise
 *      specified.
 *
 * Parameters:
 *      modeStr - The file mode string.
 *      assumeInt - Assume and integer (although a leading 0 will indicate
 *                  octal.
 * Results:
 *      The new permissions, or -1 if invalid permissions where supplied.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_ConvAbsMode (modeStr, assumeInt)
    char *modeStr; 
    int   assumeInt;
{
    int   mode;

    if (!Tcl_StrToInt (modeStr, (assumeInt) ? 0 : 8, &mode))
      return -1;  /* Not an octal number */ 
    else
      return mode;

} /* Tcl_ConvAbsMode */

/*
 *----------------------------------------------------------------------
 *
 * Tcl_ChmodCmd --
 *     Implements the TCL chmod command:
 *     chmod [-i] mode filelist
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_ChmodCmd (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int         idx, modeVal, assumeInt, fileArgc, result = TCL_ERROR;
    char      **fileArgv, *modePtr;
    struct stat fileStat;

    if (argc < 3)
        goto wrongArgs;
    if ((argv [1][0] == '-') && (argv [1][1] == 'i') && (argv [1][2] == '\0')){
        if (argc < 4)
            goto wrongArgs;
        assumeInt = TRUE;
        modePtr = argv [2];
    } else {
        assumeInt = FALSE;
        modePtr = argv [1];
    }

    if (Tcl_SplitList (interp, argv [argc - 1], &fileArgc,
                       &fileArgv) != TCL_OK)
        return TCL_ERROR;

    for (idx = 0; idx < fileArgc; idx++) {
        if ((modeVal = Tcl_ConvAbsMode (modePtr, assumeInt)) < 0) {
            if (stat (fileArgv [idx], &fileStat) != 0)
                goto fileError;
            modeVal = Tcl_ConvSymMode (modePtr, fileStat.st_mode & 07777);
            if (modeVal < 0) {
                interp->result = "invalid file mode";
                goto exitPoint;
            }
            if (assumeInt) {
                Tcl_AppendResult (interp, argv [0], ": -i flag may not be",
                                  " used with symbolic modes", (char *) NULL);
                goto exitPoint;
            }
        }
        if (chmod (fileArgv [idx], modeVal) < 0)
            goto fileError;
    }

    result = TCL_OK;
exitPoint:
    ckfree ((char *) fileArgv);
    return result;

fileError:
    /*
     * Error accessing file, assumes file name is fileArgv [idx].
     */
    Tcl_AppendResult (interp, argv [0], ": ", fileArgv [idx], ": ",
                      Tcl_UnixError (interp), (char *) NULL);
    ckfree ((char *) fileArgv);
    return TCL_ERROR;

wrongArgs:
    Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                      " [-i] mode filelist", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_ChownCmd --
 *     Implements the TCL chown command:
 *     chown owner filelist
 *     chown {owner group} filelist
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_ChownCmd (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int            idx, ownArgc, fileArgc;
    char         **ownArgv, **fileArgv = NULL;
    struct stat    fileStat;
    int            useOwnerGrp, chGroup, ownerId, groupId;
    struct passwd *passwdPtr;
    struct group  *groupPtr;
    int            result = TCL_ERROR;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " owner|{owner group} filelist", (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_SplitList (interp, argv[1], &ownArgc, &ownArgv) != TCL_OK)
        return TCL_ERROR;
    if ((ownArgc < 1) || (ownArgc > 2)) {
        interp->result = "owner arg should be: owner or {owner group}";
        goto exitPoint;
    }
    if (ownArgc == 2) {
        useOwnerGrp = (ownArgv [1][0] == '\0');
        chGroup = TRUE;
    } else
        chGroup = FALSE;

    /*
     * Get the owner id, either convert the name or use it as an integer.
     */
    passwdPtr = getpwnam (ownArgv [0]);
    if (passwdPtr != NULL)
        ownerId = passwdPtr->pw_uid;
    else {
        if (!Tcl_StrToInt (ownArgv [0], 10, &ownerId)) {
            Tcl_AppendResult (interp, "unknown user id: ", ownArgv [0],
                              (char *) NULL);
            goto exitPoint;
        }
    }
    /*
     * Get the group id, this is either the specified id or name, or the
     * if associated with the specified user.
     */
    if (chGroup) {
        if (useOwnerGrp) {
            if (passwdPtr == NULL) {
                passwdPtr = getpwuid (ownerId);
                if (passwdPtr != NULL) {
                    Tcl_AppendResult (interp, "unknown user id: ", 
                                      ownArgv [0], (char *) NULL);
                    goto exitPoint;
                }
            }
            groupId = passwdPtr->pw_gid;                        
        } else {
            groupPtr = getgrnam (ownArgv [1]);
            if (groupPtr != NULL)
                groupId = groupPtr->gr_gid;
            else {
                if (!Tcl_StrToInt (ownArgv [1], 10, &groupId)) {
                    Tcl_AppendResult (interp, "unknown group id: ", 
                                      ownArgv [1], (char *) NULL);
                    goto exitPoint;
                }
            }
        }
    }
    if (Tcl_SplitList (interp, argv [2], &fileArgc, &fileArgv) != TCL_OK)
        goto exitPoint;

    for (idx = 0; idx < fileArgc; idx++) {
        if (!chGroup) {
            if (stat (fileArgv [idx], &fileStat) != 0) {
                Tcl_AppendResult (interp, argv [0], ": ", fileArgv [idx], ": ",
                                  Tcl_UnixError (interp), (char *) NULL);
                goto exitPoint;
            }
            groupId = fileStat.st_gid;
        }

        if (chown (fileArgv[idx], ownerId, groupId) < 0) {
            Tcl_AppendResult (interp, argv [0], ": ", fileArgv [idx], ": ",
                              Tcl_UnixError (interp), (char *) NULL);
            goto exitPoint;
        }

    } /* Modify each file */

    result = TCL_OK;
exitPoint:
    ckfree ((char *) ownArgv);
    if (fileArgv != NULL)
        ckfree ((char *) fileArgv);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_ChgrpCmd --
 *     Implements the TCL chgrp command:
 *     chgrp group filelist
 *
 * Results:
 *  Standard TCL results, may return the UNIX system error message.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_ChgrpCmd (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int            idx, fileArgc, groupId, result = TCL_ERROR;
    char         **fileArgv;
    struct stat    fileStat;
    struct group  *groupPtr;

    if (argc < 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " group filelist", (char *) NULL);
        return TCL_ERROR;
    }

    groupPtr = getgrnam (argv [1]);
    if (groupPtr != NULL)
        groupId = groupPtr->gr_gid;
    else {
        if (!Tcl_StrToInt (argv [1], 10, &groupId)) {
            Tcl_AppendResult (interp, "unknown group id: ", argv [1],
                              (char *) NULL);
            return TCL_ERROR;
        }
    }
    if (Tcl_SplitList (interp, argv [2], &fileArgc, &fileArgv) != TCL_OK)
        return TCL_ERROR;

    for (idx = 0; idx < fileArgc; idx++) {
        if ((stat (fileArgv [idx], &fileStat) != 0) ||
                (chown (fileArgv[idx], fileStat.st_uid, groupId) < 0)) {
            Tcl_AppendResult (interp, argv [0], ": ", fileArgv [idx], ": ",
                              Tcl_UnixError (interp), (char *) NULL);
            goto exitPoint;
        }
    } /* Modify each file */

    result = TCL_OK;
exitPoint:
    ckfree ((char *) fileArgv);
    return result;
}
