
/*
     Program PM-Info: a program for viewing GNU-style hypertext info
     documentation files.
     Copyright (C) 1992,1993  Colin Jensen
     
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

     Contact addresses:
	Colin Jensen
	email: cjensen@netcom.com
	US mail: 4902 Esguerra Terrace, Fremont CA, 94555
*/

#define INCL_PM
#include <os2.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include "font.h"
#include "stddia.h"
#include "pminfo.h"
#ifndef NDEBUG
#define NDEBUG
#endif
//#define SHOW_RETURNS
//#define SHOW_CHECKPOINTS
#include "bugme.h"


class PointSize
{
    int size;
public:
    void set(int new_size) { size = new_size; }
    void set(int digits, int tenths) 
    { 
	set(digits * 10 + tenths % 10);
    }
    int digits() { return size / 10; }
    int tenths() { return size % 10; }
    int operator==(PointSize pt) { return pt.size == size; }
    PointSize() { set(0); }
    PointSize(int size) { set(size); }
    PointSize(int dig, int ten) { set(dig, ten); }
};

class PointSizeList
{
public:
    virtual int number_of_point_sizes() = 0;
    virtual int is_ok_size(PointSize size) = 0;
    virtual void add_point(PointSize size) = 0;
    virtual PointSize get_ith_point_size(int index) = 0;
    virtual ~PointSizeList();
};

PointSizeList::~PointSizeList()
{
}

class FixedFontPointSizeList: public PointSizeList
{
    int alloc_size;
    int cur_size;
    PointSize *points;
public:
    virtual int number_of_point_sizes();
    virtual int is_ok_size(PointSize size);
    virtual PointSize get_ith_point_size(int index);
    virtual void add_point(PointSize size);
    void delete_all_points();
    void set_from_list(const char *pt_list);
    FixedFontPointSizeList();
    FixedFontPointSizeList(const char *pt_list);
    virtual ~FixedFontPointSizeList();
};

FixedFontPointSizeList::FixedFontPointSizeList()
    : alloc_size(8), cur_size(0)
{
    BUGME(("FixedFontPointSizeList::FixedFontPointSizeList()"));
    points = (PointSize *) malloc(alloc_size * sizeof(PointSize));
}

FixedFontPointSizeList::FixedFontPointSizeList(const char *pt_list)
{
    BUGME(("FixedFontPointSizeList::FixedFontPointSizeList(``%s'')",
	   pt_list));
    set_from_list(pt_list);
}

FixedFontPointSizeList::~FixedFontPointSizeList()
{
    free(points);
}

void FixedFontPointSizeList::set_from_list(const char *pt_list)
{
    char *dup = strdup(pt_list);
    char *tok;
    tok = strtok(dup, " \t\n\r");
    while(tok != NULL) {
	int units, tenths;
	int match = sscanf(tok, "%d.%d", &units, &tenths);
	if (match == 1) {
	    FixedFontPointSizeList::add_point(PointSize(units, 0));
	} else if (match == 2) {
	    FixedFontPointSizeList::add_point(PointSize(units, tenths));
	}
	tok = strtok(NULL, " \t\n\r");
    }
    free(tok);
}

void FixedFontPointSizeList::add_point(PointSize point)
{
    if (FixedFontPointSizeList::is_ok_size(point)) return;
    while (cur_size >= alloc_size) {
	alloc_size += 16;
	points = (PointSize *) realloc(points, alloc_size * sizeof(PointSize));
    }
    points[cur_size++] = point;
}

void FixedFontPointSizeList::delete_all_points()
{
    cur_size = 0;
}

int FixedFontPointSizeList::number_of_point_sizes()
{
    return cur_size;
}

int FixedFontPointSizeList::is_ok_size(PointSize size)
{
    int i;
    for (i=0; i < cur_size; i++) {
	if (size == points[i]) return 1;
    }
    return 0;
}

PointSize FixedFontPointSizeList::get_ith_point_size(int index)
{
    return (index >= cur_size) ? PointSize(0) : points[index];
}

class VectorFontPointSizeList: public FixedFontPointSizeList
{
public:
    virtual int is_ok_size(int size);
    virtual void add_point(PointSize size);
    VectorFontPointSizeList();
    virtual ~VectorFontPointSizeList();
};

VectorFontPointSizeList::VectorFontPointSizeList()
    : FixedFontPointSizeList("8 10 12 14 18 24")
{
#ifdef DEBUG
    BUGME(("VectorFontPointSizeList::VectorFontPointSizeList"));
    BUGME(("VectorFontPointSizeList::VectorFontPointSizeList: "
	   "contains %d basic point sizes", number_of_point_sizes()));
    int i;
    for (i=0; i < number_of_point_sizes(); i++) {
	PointSize p = get_ith_point_size(i);
	BUGME(("VectorFontPointSizeList::VectorFontPointSizeList: "
	       "includes %d.%d", p.digits(), p.tenths()));
    }
#endif
}

VectorFontPointSizeList::~VectorFontPointSizeList()
{
}

int VectorFontPointSizeList::is_ok_size(int)
{
    return 1;
}

void VectorFontPointSizeList::add_point(PointSize)
{
}

class FaceItem
{
public:
    PointSizeList *point_list;
    void set_vector_font();
    void add_point_size(PointSize ptsize);
    char * const face_name;
    FaceItem(const char *aface_name);
    ~FaceItem();
};

FaceItem::FaceItem(const char *aface_name)
    : point_list(NULL), face_name(strdup(aface_name))
{
    BUGME(("FaceItem::FaceItem(%d)", aface_name));
}

void FaceItem::set_vector_font()
{
    BUGME(("FaceItem::set_vector_font()"));
    if (point_list != NULL) delete point_list;
    point_list = new VectorFontPointSizeList();
}

void FaceItem::add_point_size(PointSize ptsize)
{
    BUGME(("FaceItem::add_point_size(...)"));
    if (point_list == NULL) point_list = new FixedFontPointSizeList();
    point_list->add_point(ptsize);
}

FaceItem::~FaceItem()
{
    BUGME(("FaceItem::~FaceItem"));
    if (point_list != NULL) delete point_list;
}

class FaceList
{
    int alloc, current;
    FaceItem **items;
public:
    int count() { return current; }
    FaceItem *access(int index)
    {
	return (index < current && index >= 0) ? items[index] : NULL;
    }
    FaceItem *get_face(const char *face_name);
    void create_face(const char *face_name);
    FaceList();
    ~FaceList();
};

FaceList::FaceList()
    : alloc(8), current(0)
{
    BUGME(("FaceList::FaceList()"));
    items = (FaceItem **) malloc(alloc * sizeof(FaceItem *));
}

FaceItem *FaceList::get_face(const char *face_name)
{
    BUGME(("FaceList::get_face(%s)", face_name));
    int i;
    for (i = 0; i < current; i++) {
	if (!strcmp(items[i]->face_name, face_name)) {
	    return items[i];
	}
    }
    return NULL;
}

void FaceList::create_face(const char *face_name)
{
    BUGME(("FaceList::create_face(%s)", face_name));
    if (get_face(face_name) != NULL) return;
    if (current >= alloc) {
	alloc += 8;
	items = (FaceItem **) realloc(items, alloc * sizeof(FaceItem *));
    }
    items[current++] = new FaceItem(face_name);
}

FaceList::~FaceList()
{
    BUGME(("FaceList::~FaceList"));
    int i;
    for (i = 0; i < current; i++) delete items[i];
    free(items);
}

class FamilyItem
{
public:
    FaceList face_list;
    char * const family_name;
    FamilyItem(const char *afamily_name);
    ~FamilyItem();
};

FamilyItem::FamilyItem(const char *afamily_name)
    : family_name(strdup(afamily_name))
{
    BUGME(("FamilyItem::FamilyItem(%s)", afamily_name));
}

FamilyItem::~FamilyItem()
{
    BUGME(("FamilyItem::~FamilyItem()"));
    free(family_name);
}

class FamilyList
{
    int alloc, current;
    FamilyItem **families;
public:
    FamilyList();
    ~FamilyList();
    int count() { return current; }
    FamilyItem *access(int index) 
    { 
	return (index < current && index >= 0) ? families[index] : NULL;
    }
    FamilyItem *get_family(const char *family_name);
    void create_family(const char *family_name);
};

FamilyList::FamilyList()
    : alloc(16), current(0)
{
    BUGME(("FamilyList::FamilyList()"));
    families = (FamilyItem **) malloc(alloc * sizeof(FamilyItem *));
}

FamilyList::~FamilyList()
{
    BUGME(("FamilyList::~FamilyList()"));
    int i;
    for (i=0; i < current; i++) delete(families[i]);
    free(families);
}

FamilyItem *FamilyList::get_family(const char *family_name)
{
    BUGME(("FamilyList::get_family(%s)", family_name));
    int i;
    for (i=0; i < current; i++) {
	if (!strcmp(families[i]->family_name, family_name)) {
	    return families[i];
	}
    }
    return NULL;
}

void FamilyList::create_family(const char *family_name)
{
    BUGME(("FamilyList::create_family(%s)", family_name));
    if (get_family(family_name) != NULL) return;
    BUGME(("FamilyList::create_family: family is a new family"));
    if (current >= alloc) {
	BUGME(("FamilyList::create_family: expanding allocated size"));
	alloc += 8;
	families = (FamilyItem **) realloc(families, 
					   alloc * sizeof(FamilyItem *));
    }
    families[current++] = new FamilyItem(family_name);
}

static FamilyList *the_font_list = NULL;

static void determine_fonts(HPS hps)
{
    // Init the font list
    if (the_font_list == NULL) delete the_font_list;
    the_font_list = new FamilyList;

    // Determine how many fonts exist
    long requested_fonts = 0;
    int code = GpiQueryFonts(hps, QF_PUBLIC, NULL, &requested_fonts,
			     0, NULL);
    if (code == GPI_ALTERROR) {
	BUGME(("detemine_fonts: Error on initial query of fonts"));
	return;
    }
    requested_fonts = code;
    
    // Get the full font descriptions
    FONTMETRICS *afmMetrics;
    afmMetrics = (FONTMETRICS *) malloc(sizeof(FONTMETRICS) * requested_fonts);
    code = GpiQueryFonts(hps, QF_PUBLIC, NULL, &requested_fonts,
			 sizeof(FONTMETRICS), afmMetrics);
    if (code == GPI_ALTERROR || code != 0) {
	BUGME(("determine_fonts: I can't cope with PM.  Where's my 386BSD?"));
	return;
    }
    BUGME(("determine_fonts: success from GpiQueryFonts"));
    BUGME(("determine_fonts: read %d fonts", requested_fonts));

    // Write out all the fonts to the debug file
#ifdef DEBUG
    int i;
    FONTMETRICS *f;
    HATOMTBL sys_atom_table = WinQuerySystemAtomTable();
    char buf[1024];
    for (i=0; i < requested_fonts; i++) {
	BUGME((""));
	BUGME(("Font #%d", i));
	f = afmMetrics + i;
	BUGME(("Family name: %s", f->szFamilyname));
	BUGME(("Face name: %s", f->szFacename));

	// Manufacture and reference the new font
	the_font_list->create_family(f->szFamilyname);
	FamilyItem *this_family = the_font_list->get_family(f->szFamilyname);

	// Manufacture and reference the new face
	this_family->face_list.create_face(f->szFacename);
	FaceItem *this_face = this_family->face_list.get_face(f->szFacename);

	if (/* is a vector font */ f->fsDefn & FM_DEFN_OUTLINE) {
	    this_face->set_vector_font();
	} else {
	    this_face->add_point_size(PointSize(f->sNominalPointSize));
	}


	int pt_size = f->sNominalPointSize / 10;
	int pt_size_tenths = f->sNominalPointSize % 10;
	if (pt_size_tenths == 0) {
	    BUGME(("Point size: %d", pt_size));
	} else {
	    BUGME(("Point size: %d.%d", pt_size, pt_size_tenths));
	}

	char fbuf[strlen(f->szFamilyname) + strlen(f->szFacename) + 3 + 32];
	if (pt_size_tenths == 0) {
	    sprintf(fbuf, "%d ", pt_size);
	} else {
	    sprintf(fbuf, "%d.%d ", pt_size, pt_size_tenths);
	}
	if (strcmp(f->szFamilyname, f->szFacename)) {
	    sprintf(fbuf + strlen(fbuf), "%s: %s", f->szFamilyname, 
		    f->szFacename);
	} else {
	    sprintf(fbuf + strlen(fbuf), "%s", f->szFamilyname);
	}

	CHECKPOINT;
	BUGME(("lMatch: %ld", f->lMatch));
	CHECKPOINT;
	BUGME(("fsType: %X =%s%s%s%s", f->fsType,
	       (f->fsType & FM_TYPE_FIXED) ? " fixed-width" : "",
	       (f->fsType & FM_TYPE_FAMTRUNC) ? " family-name-truncated" : "", 
	       (f->fsType & FM_TYPE_FACETRUNC) ? " face-name-truncated" : "",
	       (f->fsType & FM_TYPE_ATOMS) ? " atoms-availible" : ""));
	CHECKPOINT;
	if (f->fsType & FM_TYPE_FACETRUNC) {
	    int code = WinQueryAtomName(sys_atom_table, f->FaceNameAtom,
					(PSZ) buf, sizeof(buf)-1);
	    if (code == 0) {
		BUGME(("Real face name: %s", buf));
	    } else {
		BUGME(("Unable to read real face name"));
	    }
	}
	BUGME(("fsSelection: %X =%s%s%s%s%s%s", f->fsSelection,
	       (f->fsSelection & FM_SEL_ITALIC) ? " italic" : "",
	       (f->fsSelection & FM_SEL_UNDERSCORE) ? " underscore" : "",
	       (f->fsSelection & FM_SEL_NEGATIVE) ? " negative" : "",
	       (f->fsSelection & FM_SEL_OUTLINE) ? " outline" : "",
	       (f->fsSelection & FM_SEL_STRIKEOUT) ? " strikeout" : "",
	       (f->fsSelection & FM_SEL_BOLD) ? " bold" : ""));
	BUGME(("fsDefn: %X: %s%s%s%s", f->fsDefn,
	       (f->fsDefn & FM_DEFN_OUTLINE) ? " outline(vector-font)" : "",
	       (f->fsDefn & FM_DEFN_IFI) ? " ifi" : "",
	       (f->fsDefn & FM_DEFN_WIN) ? " win" : "",
	       (f->fsDefn & FM_DEFN_GENERIC) ? " generic" : ""));
    }
#endif
}



class FamilyNameClass: public StdComboBoxItemClass
{
    DECLARE_METHOD(entry_field_changed, void, ());
    DECLARE_METHOD(select, void, ());
public:
    virtual void family_changed() = 0;
    FamilyNameClass(StdDialogBaseClass *adialog_class)
	: StdComboBoxItemClass(DIAFONT_FAMILY_NAMES, adialog_class)
	{
	    BUGME(("FamilyNameClass[%X]::FamilyNameClass(...)", this));
	    OVERRIDE_METHOD(entry_field_changed);
	    OVERRIDE_METHOD(select);
	}
};

DEFINE_VOID_METHOD(FamilyNameClass, entry_field_changed, (), ());
DEFINE_VOID_METHOD(FamilyNameClass, select, (), ());

void FamilyNameClass::entry_field_changed()
{
    BUGME(("FamilyNameClass::entry_field_changed()"));
    family_changed();
}

void FamilyNameClass::select()
{
    BUGME(("FamilyNameClass::select()"));
    family_changed();
}

class FaceNameClass: public StdComboBoxItemClass
{
    DECLARE_METHOD(entry_field_changed, void, ());
    DECLARE_METHOD(select, void, ());
protected:
    virtual void face_changed() = 0;
    void clear() { delete_all(); }
    void add_face(char *f) { insert_order_ascending(f); }
    void face_select(int index) { select_item(index); }
    char *get_current_face() { return query_selection_text(); }
public:
    FaceNameClass(StdDialogBaseClass *adialog_class)
	: StdComboBoxItemClass(DIAFONT_FACE_NAMES, adialog_class)
	{
	    BUGME(("FaceNameClass[%X]::FaceNameClass(...)", this));
	    OVERRIDE_METHOD(entry_field_changed);
	    OVERRIDE_METHOD(select);
	}
};

DEFINE_VOID_METHOD(FaceNameClass, entry_field_changed, (), ());
DEFINE_VOID_METHOD(FaceNameClass, select, (), ());

void FaceNameClass::entry_field_changed()
{
    BUGME(("FaceNameClass::entry_field_changed()"));
    face_changed();
}

void FaceNameClass::select()
{
    BUGME(("FaceNameClass::select()"));
    face_changed();
}

class PointClass: public StdComboBoxItemClass
{
public:
    void set_point_item(PointSizeList *point_list);
    PointClass(StdDialogBaseClass *adialog_class)
	: StdComboBoxItemClass(DIAFONT_POINTS, adialog_class)
	{
	    BUGME(("PointClass[%X]::PointClass(...)", this));
	}
};

void PointClass::set_point_item(PointSizeList *point_list)
{
    BUGME(("PointClass::set_point_item(...)"));
    char buf[16];
    int i;
    delete_all();
    for (i=0; i < point_list->number_of_point_sizes(); i++) {
	PointSize pt = point_list->get_ith_point_size(i);
	if (pt.tenths() == 0) {
	    sprintf(buf, "%d", pt.digits());
	} else {
	    sprintf(buf, "%d.%d", pt.digits(), pt.tenths());
	}
	insert_at_end(buf);
    }
    select_item(0);
}

class FontDialog: public StdDialogClass, StdOKMessageItemClass,
    StdCancelMessageItemClass, FamilyNameClass, FaceNameClass,
    PointClass
{
    virtual void ok_message();
    virtual void cancel_message();
    virtual void dialog_init();
    virtual void family_changed();
    virtual void face_changed();
    void recalc_face();
    void recalc_points();
public:
    FontDialog();
};

static FontDialog font_dialog;

void SelectFont(HWND owner)
{
    font_dialog.ExecDialog(owner);
}


void FontDialog::ok_message()
{
    BUGME(("FontDialog::ok_message()"));
    BUGME(("FontDialog::ok_message: Current family is ``%s''",
	   FamilyNameClass::query_selection_text()));
}

void FontDialog::cancel_message()
{
    BUGME(("FontDialog::cancel_message()"));
}

FontDialog *the_font_dialog;

#ifdef DEBUG
static void show_all_fonts()
{
    if (the_font_list == NULL) {
	BUGME(("show_all_fonts: No fonts availible"));
    }
    int fam;
    for (fam = 0; fam < the_font_list->count(); fam++) {
	FamilyItem *fam_item = the_font_list->access(fam);
	BUGME(("\nFamily name: %s", fam_item->family_name));
	int face;
	for (face = 0; face < fam_item->face_list.count(); face++) {
	    FaceItem *face_item = fam_item->face_list.access(face);
	    BUGME(("\tFace name: %s", face_item->face_name));
	    PointSizeList *pts = face_item->point_list;
	    int point;
	    for (point = 0; point < pts->number_of_point_sizes(); point++) {
		PointSize pt_size = pts->get_ith_point_size(point);
		if (pt_size.tenths() != 0) {
		    BUGME(("\t\tPoint Size: %d.%d", pt_size.digits(),
			   pt_size.tenths()));
		} else {
		    BUGME(("\t\tPoint Size: %d", pt_size.digits()));
		}
	    }
	}
    }
}
#endif


void FontDialog::dialog_init()
{
    the_font_dialog = this;
    HPS hps = WinGetScreenPS(HWND_DESKTOP);
    determine_fonts(hps);
    WinReleasePS(hps);
#ifdef DEBUG
    show_all_fonts();
#endif
    if (the_font_list == NULL) return;
    int fam;
    for (fam = 0; fam < the_font_list->count(); fam++) {
	FamilyItem *fam_item = the_font_list->access(fam);
	FamilyNameClass::insert_order_ascending(fam_item->family_name);
    }
    FamilyNameClass::select_item(0);
//    recalc_face();
}

void FontDialog::family_changed()
{
    BUGME(("FontDialog::family_changed()"));
    recalc_face();
}

void FontDialog::face_changed()
{
    BUGME(("FontDialog::face_changed()"));
    recalc_points();
}

void FontDialog::recalc_face()
{
    BUGME(("FontDialog::recalc_face()"));
    FaceNameClass::clear();
    char *current_family = FamilyNameClass::query_selection_text();
    if (current_family == NULL) {
	BUGME(("FontDialog::recalc_face: no current family!"));
	return;
    }
    BUGME(("FontDialog::recalc_face: family is ``%s''", current_family));
    FamilyItem *fam = the_font_list->get_family(current_family);
    if (fam == NULL) return;
    BUGME(("FontDialog::recalc_face: got a real family reference"));
    FaceList *face_list = &(fam->face_list);
    int face;
    BUGME(("FontDialog::recalc_face: family has %d faces", 
	   face_list->count()));
    for (face = 0; face < face_list->count(); face++) {
	FaceItem *face_item = face_list->access(face);
	if (face_item == NULL) continue;
	BUGME(("FontDialog::recalc_face: inserting face ``%s''", 
	       face_item->face_name));
	add_face(face_item->face_name);
    }
    face_select(0);
//    recalc_points();
}

void FontDialog::recalc_points()
{
    BUGME(("FontDialog::recalc_points()"));
    char *current_family = FamilyNameClass::query_selection_text();
    if (current_family == NULL) {
	BUGME(("FontDialog::recalc_points: no current family!"));
	return;
    }
    BUGME(("FontDialog::recalc_points: family is ``%s''", current_family));
    FamilyItem *fam = the_font_list->get_family(current_family);
    if (fam == NULL) return;
    BUGME(("FontDialog::recalc_points: got a real family reference"));

    FaceList *face_list = &(fam->face_list);

    const char *current_face = get_current_face();
    BUGME(("FontDialog::recalc_points: current face is ``%s''", current_face));
    if (current_face == NULL) return;
    FaceItem *face = face_list->get_face(current_face);
    if (face == NULL) {
	BUGME(("FontDialog::recalc_points: Unable to access current face"));
	return;
    }
    set_point_item(face->point_list);
}

FontDialog::FontDialog()
    : StdDialogClass(DIAFONT),
      StdOKMessageItemClass(this),
      StdCancelMessageItemClass(this),
      FamilyNameClass(this),
      FaceNameClass(this),
      PointClass(this)
{
    BUGME(("FontDialog[%X]::FontDialog()", this));
}
