/* Copyright (c) 1999, 2000 by Kevin Forchione.  All Rights Reserved. */
/*
 *  TADS ADV.T/STD.T LIBRARY EXTENSION
 *  CONSULT.T				
 *  version 1.0
 *
 *   	This file defines the consultItem class and required grammar. 
 *      A consultItem object is a non-Actor object in which the player 
 *      can CONSULT or LOOK UP topics of interest, such as a book or 
 *      encyclopedia.
 *
 *----------------------------------------------------------------------
 *  REQUIREMENTS
 *
 *      + HTML TADS 2.3.0 or later
 *      + Should be #included after ADV.T and STD.T     
 *
 *----------------------------------------------------------------------
 *  IMPORTANT LIBRARY INTERFACE AND MODIFICATION
 *
 *      This module makes use of the deepverb parseUnknownXobj() parser
 *      hooks. 
 *
 *----------------------------------------------------------------------
 *  COPYRIGHT NOTICE
 *
 *  	You may modify and use this file in any way you want, provided that
 *		if you redistribute modified copies of this file in source form, the
 *   	copies must include the original copyright notice (including this
 *   	paragraph), and must be clearly marked as modified from the original
 *   	version.
 *
 *------------------------------------------------------------------------------
 *  REVISION HISTORY
 *
 *		99-03-13:	Creation.
 */

#define __CONSULT_MODULE_

/*
 *	class consultItem: readable
 *	
 *	The consultItem is simply a readable with special handling for 
 *	CONSULT and LOOK UP verbs. Processing should be overridden in 
 *	the consultWord() and consultObj() methods using switch() logic.
 */
class consultItem: readable
	isconsultItem = true
	wordlist = []
	verDoConsult( actor ) = {}
	/*
	 *	The player has entered CONSULT BOOK but hasn't given a subject. Here
	 *	we present a generic message.
	 */
	doConsult( actor ) =
	{
		"%You% search%es% through <<self.thedesc>>, but find%s% nothing 
		of interest.";
	}
	verDoConsultAbout( actor ) = {}
	doConsultAbout( actor, iobj ) =
	{
		if ( iobj <> self )
		{
			/*
			 *	An object has been identified by the parser. We 
			 *	call consultObj() which should switch( obj ) through
			 *	various responses.
			 */
			self.consultObj( actor, iobj );
		}
		else if ( length( self.wordlist ) )
		{
			/*
			 *	The dobj = iobj and we have a wordlist. This means
			 *	that the command has gone through parseUnknownXobj().
			 *	We clear out the wordlist each time in order to avoid
			 *	instances of self-referencing.
			 */
			local w := [];
			w := self.wordlist;
			self.wordlist := [];
			self.consultWord( actor, w );
		}
		else
		{
			/*
			 *	We have a case of self-referencing. The player has 
			 *	asked the 'book' to consult itself.
			 *	In this case we produce the generic message.
		 	 */			
			"%You% search%es% through <<self.thedesc>>, but find%s%
			no entry on this. ";
		}
	}
	verIoConsultAbout( actor, dobj ) = {}
	ioConsultAbout( actor, dobj ) =
	{
		dobj.doConsultAbout( actor, self );
	}
	doSynonym( 'ConsultAbout' ) = 'ConsultOn'
	ioSynonym( 'ConsultAbout' ) = 'ConsultOn'
	verDoLookupIn( actor, dobj ) = {}
	verIoLookupIn( actor ) = {}
	ioLookupIn( actor, dobj ) =
	{
		self.doConsultAbout( actor, dobj );
	}
	/*
	 * 	This method is called when the dobj <> iobj, which means that we are 
	 *	referencing an object that exists in the game. The method is passed 
	 *	the actor and the object. Switch( obj ) can then be used in the method
	 *	override to handle each object separately. Here we present a 
	 *	generic message.
	 */
	consultObj( actor, obj ) =
	{
		"%You% consult%s% <<self.thedesc>> about <q><<obj.sdesc>>,</q> but 
		find%s% nothing of interest. ";	
	}
	/*
	 *	This method is called when the dobj = iobj, which means that we are
	 *	referencing the wordlist created from one of the parseUnknownXobj().
	 *	The method is passed the actor and the wordlist, which can be handled
	 *	in the method override with switch( w[ i ] ) processing for each required 
	 *	word in the list. Here we present a generic message.
	 */
	consultWord( actor, w ) =
	{
		local i;
		
		"%You% consult%s% <<self.thedesc>> about <q>";

		for (i := 1 ; i <= length( w ) ; ++i)
		{
			if (i <> 1)
   				" ";
			say( w[ i ] );
		}
		"<\q>, but find%s% nothing of interest. ";	
	}
;

/*
 *	Modifies the thing: object to allow for the new CONSULT and LOOK UP verbs
 *	for objects within the game.
 */
modify thing
	verIoConsultAbout( actor, dobj ) = {}
	ioConsultAbout( actor, dobj ) =
	{
		dobj.doConsultAbout( actor, self );
	}
	ioSynonym( 'ConsultAbout' ) = 'ConsultOn'
	verDoLookupIn( actor, dobj ) = {}
;

/*
 *	consultVerb: deepverb
 *	
 *	Allows for CONSULT dobj ABOUT iobj and CONSULT dobj ON iobj statements. 
 *	Because TADS normally disambiguates the indirect object first 
 *	parseUnknownIobj() wouldn't know what the direct object is and would 
 *	require something like the unknownAskIobj cited in TADS Version Release
 *	2.2.4. Disambiguating the dobj first allows us to make the direct object
 *	the same as the indirect object. 
 */
consultVerb: deepverb
	sdesc = "consult"
	verb = 'consult'
	doAction = 'Consult'
	ioAction( aboutPrep ) = [disambigDobjFirst] 'ConsultAbout'
	ioAction( onPrep ) = [disambigDobjFirst] 'ConsultOn'
	validIo( actor, obj, seqno ) = { return (seqno = 1); }
    validIoList( actor, prep, dobj ) = (nil)
	/*
 	 *   For "consult," use special handling for unknown words so
 	 *   that the book can respond directly to the unknown words. 
 	 */
	parseUnknownIobj(actor, prep, dobj, words) =
	{
		/* if we're consulting about something, have the consultItem respond */
		if ( prep = aboutPrep or prep = onPrep )
		{
			/* use our special CONSULT object for the unknown words */
			dobj.wordlist := words;
			return dobj;
		}
        else
        {
        	/* 
       		 *   it's not CONSULT ON, return nil to use the
         	 *   default system handling 
             */
       		return nil;
    	}
	}
;

/*
 *	lookupVerb: deepverb
 *	
 *	Allows for LOOK UP iobj IN dobj statements. Since we are disambiguating
 *	the direct object we know the indirect object and can use it as our direct
 *	object, eliminating the need for two objects to handle a look up command.
 */
lookupVerb: deepverb
	sdesc = "look up"
	verb = 'look up'
	defaultPrep = inPrep
	ioAction( inPrep ) = 'LookupIn'
	validDo( actor, obj, seqno ) = { return (seqno = 1); }
    validDoList( actor, prep, iobj ) = (nil)
	parseUnknownDobj( actor, prep, iobj, words ) =
	{
		/* if we're LOOKing UP something, have the consultItem respond */
		if (prep = inPrep)
		{
			/* use our special CONSULT object for the unknown words */
			iobj.wordlist := words;
			return iobj;
		}
        else
        {
        	/* 
       		 *   it's not LOOKUP IN, return nil to use the
         	 *   default system handling 
             */
       		return nil;
        }
	}
;
