#charset "us-ascii"

#include <adv3.h>
#include <en_us.h>

ConSpaceId: ModuleID
{
    name = 'TADS 3 ConSpace Library Extension'
    byline = 'by Eric Eve and Steve Breslin'
    htmlByline = 'by <a href="mailto:eric.eve@hmc.ox.ac.uk">Eric Eve</a>'
    version = '2.1'
    listingOrder = 73
}

/* Version 2.1 fixes a couple of minor bugs that came to light in version 2 */
     

//-------------------------------------------------------------------
//   ConSpace objects and classes
//-------------------------------------------------------------------

/* Make sure all connectionGroups are recorded as lists.
*
* (We only allow Rooms to have connectionGroups.)
*
* Also make sure that all SpaceConnectors' locations are recorded in
* their locations' connectionGroup. SpaceConnector.locationList and
* Room.connectionGroup should be kept in sync (sort of like 'contents'
* and 'location' are kept in sync by the main library).
*/
conSpacePreinit: PreinitObject
    execAfterMe = [adv3LibPreinit]
    execute() {
        for(local obj = firstObj(Room) ; obj ;
            obj = nextObj(obj, Room)) {
            if(obj.connectionGroup
               && dataType(obj.connectionGroup) == TypeObject) {

                /* The room points to a single object for its
                 * connectionGroup. Convert this to a single-element
                 * list.
                 */
                obj.connectionGroup = [obj.connectionGroup];
            }

            /* if obj has a connection group, add obj to the list of
             * rooms it connects.
             */
            if(obj.connectionGroup != nil)
                foreach(local spaceConnector in obj.connectionGroup)
                    spaceConnector.moveIntoAdd(obj);
        }
        for(local obj = firstObj(SpaceConnector) ; obj ; 
            obj = nextObj(obj, SpaceConnector)) {
            foreach(local room in obj.locationList) {
                local group = room.getOutermostRoom().connectionGroup;
                if(group == nil)
                    group = [];
                if(!group.indexOf(obj))
                    room.getOutermostRoom().connectionGroup =
                                                         group + obj;
            }
        }
    }
;

/* SpaceConnector is a normal DistanceConnector with some additional
 * behaviors.
 */
class SpaceConnector: DistanceConnector

    /* We allow dynamic creation of a SpaceConnector. A game event
     * may wish to result in the spacial connection of rooms.
     */
    construct(lst) {
        inherited();
        foreach(local room in lst) {
            moveIntoAdd(room);
            if(room.connectionGroup == nil) 
              room.connectionGroup = [self];
            else
              room.connectionGroup += self;
        }
    }

    /* We allow dynamic destruction of a SpaceConnector. A game event
     * may wish to result in the removal of a spacial connection
     * between rooms. This is useful when, say, night falls.
     *
     * Note that the caller must know which SpaceConnector should be
     * destroyed.
     */
    destruct() {
        foreach(local room in locationList) {
            locationList -= room;
            room.connectionGroup -= self;
        }
    }

    /* We allow dynamic destruction of the spacial connection between a
     * list of rooms.
     *
     * This works like destruct(), but the caller need not know which
     * SpaceConnector it is that's linking these rooms. We'll try to
     * deduce that from the provided list of connected rooms.
     *
     * Note that this is somewhat less reliable: it's better to use
     * destruct() where possible.
     */
    destructGroup(lst) {
        local connector = findConnector(lst);
        if(connector) {
            connector.destruct();

            /* return true for success */
            return true;
        }
        /* We were unable to find a SpaceConnector connecting all
         * the rooms in the list: return nil for failure.
         */
        return nil;
    }

    /* findConnector takes a list of rooms, and attempts to find the
     * unique connector that connects all these rooms.
     */
    findConnector(lst) {
        local groups = nilToList(lst[1].connectionGroup);  //CHANGED THIS

        /* find the SpaceConnectors that connect all these rooms. */
        foreach(local elem in lst)
            groups = groups.intersect(nilToList(elem.connectionGroup)); // CHANGED THIS

        /* If there's more than one SpaceConnector connecting all the
         * rooms, iterate through all the remaining SpaceConnectors,
         * and disqualify those which connect rooms outside of the
         * given room list.
         */
        if(groups.length() > 1) {
            foreach(local group in groups) {
                if(group.locationList.length > lst.length)
                    groups -= group;
            }
        }

        /* If we've pared the list down to a single connector, return
         * it. Otherwise, return nil for failure.
         */
        if(groups.length() == 1)
            return groups[1];

        return nil;
    }

    /* Add a list of rooms to our locationList, and add self to each of
     * the rooms' connectionGroup lists. If self is an already working
     * SpaceConnector, this simply adds rooms to the current spacial
     * association.
     */
    addToSpace(lst) {
        if(dataType(lst) == TypeObject)
            lst = [lst];
        foreach(local room in lst) {
            if(!locationList.indexOf(room)) {
                moveIntoAdd(room);
                if(room.connectionGroup == nil)  // ADD THIS
                   room.connectionGroup = [self]; 
                else
                   room.connectionGroup += self;
            }
        }
    }

    /* Remove a list of rooms from our locationList, and remove self
     * from each of the rooms' connectionGroup lists. We would use
     * this method when one or some of (but not all of) the rooms in
     * a connected space should no longer be spacially connected.
     */
    removeFromSpace(lst) {
        if(dataType(lst) == TypeObject)
            lst = [lst];
        foreach(local room in lst) {
            locationList -= room;
            room.connectionGroup -= self;
        }
    }

    /* We modify afterTravel() to call our new method,
     * travelerEnteringSpace(), when the traveler is entering our
     * spacial group from outside our group.
     */
    afterTravel(traveler, conn) {
        inherited(traveler, conn);
        if(conn
           && !traveler.locationBefore
           || !locationList.indexOf(traveler.locationBefore.CSGetOutermostRoom())
           && locationList.indexOf(traveler.location.CSGetOutermostRoom()))
            travelerEnteringSpace(traveler, conn);
    }

    /* travelerEnteringSpace does nothing by default. It is designed to
     * enable side-effects for entering a connection-space.
     */
    travelerEnteringSpace(traveler, conn) { /* do nothing by default */ }

    /* We modify beforeTravel() to call our new method,
     * travelerLeavingSpace(), when the traveler is leaving our spacial
     * group for a room outside our group.
     */
    beforeTravel(traveler, conn) {
        inherited(traveler, conn);
        if(conn
           && !locationList.indexOf(conn.getDestination(traveler.location, traveler).CSGetOutermostRoom)
           && locationList.indexOf(traveler.location.CSGetOutermostRoom()))
            travelerLeavingSpace(traveler, conn);
    }
    travelerLeavingSpace(traveler, conn) { /* do nothing by default */ }

    /* allowIntermediaryMoves: a flag to allow/disallow intermediary
     * moves when moving through this connection space. This is enabled
     * by default. You can disable this flag if you want actors to move
     * through this space without visiting each room along the path.
     *
     * Note that this will, in effect, disable all side-effects of
     * travel which would be associated with the intermediary moves.
     */
    allowIntermediaryMoves = true
    
    /* lookAroundOnMove: a flag which, when true, forces the player to
     * look around when he moves within our space. This is nil by
     * default.
     */
    lookAroundOnMove = nil
    /* lookAroundOnEntering: a flag which, when true, forces the player
     * to look around when he enters our space. This is true by default.
     */
    lookAroundOnEntering = true
;

/* A SpaceConnectorDoor is a normal door which dynamically creates a
 * SpaceConnector (when the door is opened) that spacially connects
 * each side of the door's immediate locations. When the door is
 * closed, this spacial connection is removed.
 *
 * Note that only the master object in the door-pair needs to be a
 * SpaceConnectorDoor. It's find if both are, though.
 */
SpaceConnectorDoor: Door
    makeOpen(stat) {

        /* If the door is closing, we need to remove our associated
         * SpaceConnector.
         */
        if(stat == nil) {

            /* If we don't yet have an associated SpaceConnector.
             * try to find one, so we can remove it.
             */
            if(!myConnector)
                myConnector = SpaceConnector.findConnector([location,
                                                 otherSide.location]);
            if(myConnector)
                myConnector.destruct();

            /* close the door as usual. */
            return inherited(stat);
        }

        /* The door is opening, so we need to add or create an
         * associated SpaceConnector.
         */
        if(!myConnector) {
            myConnector = SpaceConnector.findConnector([location,
                                                 otherSide.location]);

            /* If we failed to find one, create one, and make it link
             * the appropriate location.
             */
            if(!myConnector)
                myConnector = new SpaceConnector([location,
                                                 otherSide.location]);
                                                 
           myConnector.lookAroundOnMove = lookAroundOnTraverse;                                      
        }

        /* If we already had an associated SpaceConnector, we simply
         * tell it to spacially link the appropriate locations.
         */
        myConnector.addToSpace([location, otherSide.location]);
        return inherited(stat);
    }

    /* We keep track of this door-pair's SpaceConnector. The 'master
     * object' will keep track of this information, so we don't need
     * to worry about keeping the doors in sync.
     */
    myConnector = nil
    
    /*  By default we want to perform a lookAround when the PC moves
     *  through this door when the door is opened. If you don't
     *  want the lookAround, set lookAroundOnTraverse to nil
     */
     
    lookAroundOnTraverse = true  
    
;

//-------------------------------------------------------------------
//   Modifications to Room class
//-------------------------------------------------------------------

modify Room

  /*
   *   The connectionGroup property may be used to define a group of
   *   adjacent locations that will behave, to some extent, as a single
   *   large space, with the individual rooms within it acting as
   *   regions within that space.
   *
   *   On all the rooms you want linked in such a way, set
   *   connectionGroup to the same DistanceConnector object. (This will
   *   be the DistanceConnector that links the rooms.) There is no need
   *   to list the rooms linked in the DistanceConnector:
   *   conSpacePreinit does this for you automatically.
   *
   *   The main implications of a connectionGroup, beyond the common
   *   DistanceConnector, are:
   *
   *   (1)  that when the PC moves between rooms in the same
   *   connectionGroup the travel is described without triggering the
   *   display of a fresh room description; this is designed to foster
   *   the impression that the PC remains in the same basic space.
   *
   *   (2)  that if an action on an object enforces a touchObj
   *   precondition, and the object and actor are both in rooms
   *   belonging to the same connectionGroup, then if the actor is not
   *   already in the same room as the object the actor will be moved
   *   to the object's room to allow execution of the command. This
   *   allows actions to be carried out across rooms in the same
   *   connectionGroup. (The common DistanceConnector only allows
   *   actions to be carried out with objects in the same room as the
   *   actor.)
   *
   */
  connectionGroup = nil

  /*  allowRemoteConversation:
   *
   *  Set this property to nil if an actor must be in this room in
   *  order to converse with another actor in this room.
   */

  allowRemoteConversation = true

  /* autoApproachForConversation:
   *
   *  Set this property to true to force an implicit move into this
   *  location from another room in the same connectionGroup in order
   *  to converse with an actor in this location. This property has no
   *  effect unless allowRemoteConversation is nil and both the
   *  speaking actor and the responding actor are in rooms within the
   *  same location group.
   */

  autoApproachForConversation = nil

    checkMovingActorInto(allowImplicit)
    {
        /*
         *   If the actor isn't somewhere within us, try moving the
         *   actor into us implicitly if he's in a room with the same
         *   connection group as ours.
         */
        if (!gActor.isIn(self))
      {
        if(allowImplicit
           && inSameConnectionGroup(gActor))
        {
            if (tryImplicitAction(WalkOverTo, self))
                {
                if (!gActor.isIn(self))
                    exit;
                }
            }
        }
       return inherited(allowImplicit);
    }
    
    dobjFor(Enter) asDobjFor(TravelVia)
;

//-----------------------------------------------------------------
//   New and amended preconditions
//-----------------------------------------------------------------

  /*
   * The actorOutOfNested precondition take the current actor out
   * of any nested rooms s/he may be in
   */

actorOutOfNested : PreCondition
  checkPreCondition(obj, allowImplicit)
     {
         local loc = gActor.roomLocation;

         while(loc.ofKind(NestedRoom))
         {
            /* try removing the actor from the NestedRoom */
           if (allowImplicit)
           {
               gActor.roomLocation.tryRemovingFromNested;

             /*
              *  <ake sure that leaves the actor in the right location
              *  If not, we exit silently, since the reason for failure
              *  will have been reported by the "walk over to" action.
              */
              if (gActor.roomLocation == loc)
                  exit;

             /* indicate that we executed an implicit command */
             if (!gActor.roomLocation.ofKind(NestedRoom))
               return true;

             loc = gActor.roomLocation;
          }
          else
          {
            /* we can't walk over implicitly - report the problem and
             * exit
             */
            reportFailure(&tooDistantMsg, obj);
            exit;
          }
         }

        /* If we weren't in a NestedRoom to start with, there's nothing
         * to do.
         */
        return nil;
     }
;

/* The nearObj precondition can be used to enforce proximity but not
* touchability (e.g. with an OutOfReach). It could also be used as an
* additional precondition on certain objects to cause implicit travel
* between rooms not part of the same connectionGroup.
*/

nearObj: PreCondition
     checkPreCondition(obj, allowImplicit)
     {
         /* check to see if the actor is in the object's location.
          * If so, we're done.
          */
         if (gActor.isIn(obj.nearLoc))
             return nil;

         /* the actor isn't there - try a "walk over to" command */
         if (allowImplicit && tryImplicitAction(WalkOverTo, obj))
         {
             /*
              *  make sure that leaves the actor in the right location.
              *  If not, we exit silently, since the reason for failure
              *  will have been reported by the "walk over to" action.
              */
             if (!gActor.isIn(obj.nearLoc))
                 exit;

             /* indicate that we executed an implicit command */
             return true;
         }

         /* we can't walk over implicitly - report the problem and
          * exit.
          */
         reportFailure(&tooDistantMsg, obj);
         exit;
     }
;

 /* A version of the nearObj precondition given highest priority */

nearObjFirst: nearObj
   preCondOrder = 1
;

/*
* Modify touchObj so that we first move the actor to the object's
* CSGetOutermostRoom if possible, before trying out the inherited
* behaviour. The modification only takes effect if both the actor
* and the object are in rooms belonging to the same connectionGroup.
*/

modify TouchObjCondition
    checkPreCondition(obj, allowImplicit)
    {

        local actionAttempted = nil;

        /*
         * If the actor is in the same outermost room but in a nested
         * space that does not contain the object, remove the actor from
         * the nested room if being in the nested room prevents the actor
         * touching the object
         */

        if(allowImplicit)
         while(gActor.CSGetOutermostRoom == obj.CSGetOutermostRoom
            && gActor.roomLocation.ofKind(NestedRoom)
            && !gActor.canTouch(obj))
          {
             local oldLoc = gActor.location;
             gActor.roomLocation.tryRemovingFromNested;

             /* Check that the actor actually moved; if not report
              * an error and exit to prevent an infinite loop
              */
             if(gActor.location == oldLoc)
             {
               reportFailure(&tooDistantMsg, obj);
               exit;
             }

             actionAttempted = true;
          }

        /* If the actor is already in the object's location, there's
         * nothing more for the added behaviour to do.
         */
        if (gActor.CSGetOutermostRoom != obj.CSGetOutermostRoom)
        {
            /*
             * Attempt an implicit action to move near this object only
             * if an implicit action is allowed, the object defines a
             * nearLoc, and the object allows autoApproach.
             */
            if (allowImplicit && obj.nearLoc != nil && obj.autoApproach
                && gActor.inSameConnectionGroup(obj))
            {
                if (tryImplicitAction(WalkOverTo, obj))
                {
                    /*
                     * Make sure that leaves the actor in the right
                     * location. Failing this, exit silently, since
                     * the reason for failure will have been reported
                     * by the "walk over to" action
                     */
                    if (!gActor.isIn(obj.CSGetOutermostRoom))
                        exit;

                    actionAttempted = true;
                }
            }
        }
        return inherited(obj, allowImplicit) || actionAttempted;
    }
    verifyPreCondition(obj)
    {
         if(obj.nearLoc != nil && obj.autoApproach
                && gActor.inSameConnectionGroup(obj))

        return;

      inherited(obj);
    }
;

/*
* Change canTalkToObj so that we can optionally enforce the condition
* that both parties must be in the same room to converse, and, if this
* restriction applies, optionally allow an implicit move to the
* location of the actor being addressed.
*/

modify canTalkToObj
    checkPreCondition(obj, allowImplicit)
    {
        local actionAttempted = nil;

        if(!obj.CSGetOutermostRoom.allowRemoteConversation
           && obj.CSGetOutermostRoom != gActor.CSGetOutermostRoom)
        {
            if (allowImplicit
                && obj.CSGetOutermostRoom.autoApproachForConversation
                && gActor.inSameConnectionGroup(obj))
            {
                if (tryImplicitAction(WalkOverTo, obj))
                {
                    /*
                     * Make sure that leaves the actor in the right
                     * location. If not, we exit silently, since the
                     * reason for failure will have been reported by
                     * the "walk over to" action.
                     */
                    if (!gActor.isIn(obj.CSGetOutermostRoom))
                        exit;

                    actionAttempted = true;
                }
            }
            if(gActor.CSGetOutermostRoom != obj.CSGetOutermostRoom)
            {
                reportFailure(&objCannotHearActorMsg, obj);
                exit;
            }
        }
        return inherited(obj, allowImplicit) || actionAttempted;
    }
;

//-------------------------------------------------------------------
//  Modifications to various classes
//-------------------------------------------------------------------



modify ActorState
/*
  *  Handle a before-travel notification; we don't want to terminate
  *  a conversation if the traveler is moving to a location from which
  *  conversation is still possible.
  */
     beforeTravel(traveler, connector)
     {
         if(connector != nil
            && getActor.CSGetOutermostRoom.allowRemoteConversation
             && traveler.inSameConnectionGroup(
               connector.getDestination(traveler.CSGetOutermostRoom,
                                            traveler)))
             return;
         inherited(traveler, connector);
     }

;

/*
* Modify Traveler.describeArrival so that travel is described
* differently (i.e. without a lookAround) if the player char moves
* between two ConnectedLoc rooms that belong to the same
* connectionGroup.
*
* This effect is basically cancelled if any of the groups mutual to
* the origin and target location define lookAroundOnMove.
*
* This effect is basically over-ridden if any of the groups being
* newly entered define lookAroundOnEntry.
*/

modify Traveler
    describeArrival(origin, backConnector)
    {
        /* Get the groups which are being newly entered. (I.e., which
         * are in the target location's connectionGroup list, but which
         * are not in the origin's connectionGroup list.)
         */
        local newGroups = nilToList(getOutermostRoom.connectionGroup)
                        - nilToList(origin.connectionGroup);

        if(inSameConnectionGroup(origin)
          && isActorTraveling(gPlayerChar)
          && !commonConnectionGroup(origin).
                        indexWhich({x: x.lookAroundOnMove}) 
          && !newGroups.indexWhich({x: x.lookAroundOnEntering})) {
            if(!gAction.isImplicit)
                defaultReport(&walkIntoMsg, CSGetOutermostRoom);
        }
        else
            inherited(origin, backConnector);
    }
;

/* Modify MultiLoc so that CSGetOutermostRoom returns the room in its
 * locationList which is nearest to gActor's location.
 *
 * If for whatever reason no such path is found, return locationList[1]
 * (or nil if we have no locationList).
 */

modify MultiLoc
  CSGetOutermostRoom()
  {
    if(gActor) {
      local path = CSPathfinder.findPath(gActor, self);
      if(path && path.length())
        return path[path.length()];
    }
    return (locationList && locationList.length() ?
                                             locationList[1] : nil);
  }
;

/*
* This modification is an attempt to allow movement through a
* SenseConnector, e.g. to allow something to be thrown through a
* DistanceConnector linking rooms in the same connectionGroup.
*/

modify DistanceConnector
   checkMoveThrough(obj, dest)
   {
       if(obj.inSameConnectionGroup(dest))

           return checkStatusSuccess;

       /* return the library's default: cannot move through <self> */
       return inherited(obj, dest);
   }
;


//----------------------------------------------
// Modifications to Thing
//----------------------------------------------

modify Thing
  nearLoc = (roomLocation)
  underLoc = nil
  behindLoc = nil
  frontLoc = nil

  /* In most cases, CSGetOutermostRoom() should behave just like the
   * normal getOutermostRoom(). But if there's an intervening multiLoc,
   * in the containment tree, we will perform a search based on gActor's
   * location, and we will return the multiLoc's SCGetOutermostRoom,
   * which is the location in the multiLoc's locationList that is
   * nearest the gActor. (Normally, because multiLoc has no location
   * property, multiLoc returns 'self' for its outermost room.)
   */
  CSGetOutermostRoom() {
    return (location ? location.CSGetOutermostRoom() : self);
  }

  /* If autoApproach is true then the touchObj precondition will try to
   * move the actor into my nearLoc, if he's not already there. To
   * disable this behaviour for an individual object or classes of
   * objects, set autoApproach to nil.
   */

  autoApproach = true

  /* Normally the approach verbs (WalkOverTo, StandNear, Sit Near etc.)
   * are only allowed from locations within the same connectionGroup.
   * Exceptionally, however, you may have a landmark item such as a
   * large tree, visible through a SenseConnector, but not within the
   * same connectionGroup, that it would be convenient to allow
   * approach to using these verbs.
   *
   * The extraApproachRooms property allows you to set this up; it
   * should contain a list of the rooms (outside the connectionGroup, if
   * any) from which this object can be approached using the approach
   * verbs.
   *
   * Note that you would not not normally want to include a room in this
   * list that was not adjacent to the room containing the landmark
   * object, unless you were prepared to allow travel the bypassed the
   * intervening rooms.
   *
   * Note also that adding a Room to the list of extraApproachRooms will
   * not cause an implicit approach action from one of those rooms in
   * the case of an action that requires touching me.
   */

  extraApproachRooms = []

  /* inSameConnectionGroup(obj) returns true if obj is in the same
   * connectionGroup as self.
   */
  inSameConnectionGroup(obj) {
    return (commonConnectionGroup(obj) != nil);
  }

  /* Find a connectionGroup linking the outerMostRooms of self and obj.
   * If none exists, return nil. If one or more exist, return a list
   * of the common (i.e., shared) connection group(s).
   */
  commonConnectionGroup(obj) {

    local grp1 = nilToList(CSGetOutermostRoom.connectionGroup);
    local grp2 = nilToList(obj.CSGetOutermostRoom.connectionGroup);
    local grp = grp1.intersect(grp2);
    return (grp.length() ? grp : nil);
  }

//  dobjFor(StandInFront) asDobjFor(StandNear)


  dobjFor(WalkOverTo)
  {
    preCond = [objVisible, actorStanding] // is actorOutOfNested needed here?
    verify()
    {
      verifyStandClose(posNear, standing);
    }
    action()
    {
      sendActorTo(nearLoc, standing);
      if(!gAction.isImplicit)
         mainReport(&okayGoOverToMsg);
    }
  }

  verifyStandClose(pos, post)
  {
    local locProp = whichLocProp(pos);
    if(self.(locProp) == nil)
      illogical(&cannotStandCloseMsg, pos);


    if(gActor.isIn(self.(locProp))
       && !gActor.roomLocation.ofKind(NestedRoom)
       && gActor.posture == post)
      illogicalAlready(&actorPostureThereMsg, self, pos);

   /* Don't allow this kind of travel to occur unless
    * the object is within the same connectionGroup
    * as the actor, and autoApproach is allowed (true)
    * for this object, unless the actor is in one of the
    * locations specified in my extraApproachRooms list.
    */

   if((gActor.CSGetOutermostRoom.connectionGroup == nil
       || !inSameConnectionGroup(gActor)
          || !autoApproach)
       && extraApproachRooms.indexOf(gActor.CSGetOutermostRoom) == nil)
     illogical(&tooDistantMsg, self);
  }

    sendActorTo(loc, post) {
        local dest = loc.CSGetOutermostRoom;
        local path;

        /* If the actor is not contained by loc's outermost location,
         * we want to move him there.
         */
        if(!gActor.isIn(dest)) {

            /* If there's a SpaceConnector linking the actor and the
             * loc which has disabled allowIntermediaryMoves, we want to
             * move the actor to the destination "by fiat".
             */
            foreach(local conn in
                    nilToList(gActor.CSGetOutermostRoom.connectionGroup).
                    intersect(nilToList(loc.CSGetOutermostRoom.connectionGroup))) {
                if(!conn.allowIntermediaryMoves) {
                    gActor.moveIntoForTravel(loc);
                    gActor.makePosture(post);
                    return;
                }
            }

            /* Ok, we didn't need to move the actor by fiat.
             *
             * First we check if there's a direct connection between the
             * actor's current room and the destination. If so, we take
             * that connection.
             */
            if(gActor.CSGetOutermostRoom.getConnectorTo(gActor, dest))
                gActor.scriptedTravelTo(dest);
            else {
                path = CSPathfinder.findPath(gActor, dest);

                pathLoop:
                for(local i = 2 ; i <= path.length() ; i++) {

                    /* If the SpaceConnector linking the actor to the
                     * next step in the path which has disabled
                     * allowIntermediaryMoves, then we want to move the
                     * actor to the destination "by fiat".
                     */
                    foreach(local conn in
                      gActor.CSGetOutermostRoom.connectionGroup.
                      intersect(path[i].CSGetOutermostRoom.connectionGroup)) {
                        if(!conn.allowIntermediaryMoves) {
                            gActor.moveIntoForTravel(path[i]);
                            continue pathLoop;
                        }
                    }

                    if(gActor.CSGetOutermostRoom.getConnectorTo(gActor, path[i]))

// should be Travel or TravelVia, or a new "service verb" of our own design?
// Currently, this is circular, since walkOverTo calls sendActorTo.
                    {
                        if(path[i].ofKind(BasicLocation))
                           tryImplicitAction(Enter, path[i]);                        
                        else
                           tryImplicitAction(TravelVia, path[i]);
                    }
                    else
                        gActor.travelTo(path[i], path[i], gActor.CSGetOutermostRoom);
                }
            }
            if(!gActor.isIn(dest))
                gActor.travelTo(dest, dest, gActor.CSGetOutermostRoom);
        }

        /* At this point the actor should be contained by loc's
         * outermost location. If loc is a nested room, we move the
         * actor into the nested room.
         */
        if(loc.ofKind(NestedRoom))
            gActor.travelWithin(loc);

        /* Adjust the actor's posture as necessary. */
        if(gActor.posture != post)
            gActor.makePosture(post);
    }

  verifyPutClose(pos)
  {
   local locProp = whichLocProp(pos);
   if(self.(locProp) == nil)
        illogical(&cannotPutCloseMsg, pos);
   if (gDobj == nil)
     {
         /*
          *   check the tentative direct objects to see if all of
          *   them are directly inside me already.
          */
         if (gTentativeDobj.indexWhich(
             {x: !x.obj_.isIn(self.(locProp))}) == nil)
         {
             /*
              *   All of the potential direct objects are already
              *   directly inside me.  This makes this object
              *   illogical, since there's no need to move any of these
              *   objects into me.
              */
             illogicalAlready(&alreadyCloseMsg, pos);
         }
     }
    else
    {
       if(gDobj.isIn(self.(locProp)) && !gDobj.isIn(gActor))
            illogicalAlready(&alreadyCloseMsg, pos);
    }
  }

  actionPutClose(pos)
  {
    local locProp = whichLocProp(pos);
    gDobj.moveInto(self.(locProp));
    mainReport(&okayPutCloseMsg, pos);
  }

  dobjFor(PutNear)
  {
     preCond = [objHeld]
     verify() { }
  }

  iobjFor(PutNear)
  {
     preCond = [touchObj]
     verify() { verifyPutClose(posNear); }
     action() { actionPutClose(posNear); }
  }

  dobjFor(PutInFront) {  preCond = [objHeld]  }

  iobjFor(PutInFront)
  {
     preCond = [touchObj]
     verify() { verifyPutClose(posFront); }
     action() { actionPutClose(posFront); }
  }
  iobjFor(PutUnder)
  {
     preCond = [touchObj]
     verify() { verifyPutClose(posUnder); }
     action() { actionPutClose(posUnder); }
  }

  // dobjFor(PutBehind) is already defined in the library

  iobjFor(PutBehind)
  {
     preCond = [touchObj]
     verify() { verifyPutClose(posBehind); }
     action() { actionPutClose(posBehind); }
  }



  dobjFor(StandNear)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posNear, standing); }
    action()
    {
      sendActorTo(nearLoc, standing);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posNear, standing);
    }
  }

  dobjFor(SitNear)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posNear, sitting); }
    action()
    {
      sendActorTo(nearLoc, sitting);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posNear, sitting);
    }
  }

  dobjFor(LieNear)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posNear, lying); }
    action()
    {
      sendActorTo(nearLoc, lying);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posNear, lying);
    }
  }

  dobjFor(StandInFront)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posFront, standing); }
    action()
    {
      sendActorTo(frontLoc, standing);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posFront, standing);
    }
  }


  dobjFor(SitInFront)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posFront, sitting); }
    action()
    {
      sendActorTo(frontLoc, sitting);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posFront, sitting);
    }
  }

  dobjFor(LieInFront)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posFront, lying); }
    action()
    {
      sendActorTo(frontLoc, lying);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posFront, lying);
    }
  }


  dobjFor(StandUnder)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posUnder, standing); }
    action()
    {
      sendActorTo(underLoc, standing);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posUnder, standing);
    }
  }

  dobjFor(GetUnder) asDobjFor(StandUnder)

  dobjFor(SitUnder)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posUnder, sitting); }
    action()
    {
      sendActorTo(underLoc, sitting);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posUnder, sitting);
    }
  }

  dobjFor(LieUnder)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posUnder, lying); }
    action()
    {
      sendActorTo(underLoc, lying);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posUnder, lying);
    }
  }


  dobjFor(StandBehind)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posBehind, standing); }
    action()
    {
      sendActorTo(behindLoc, standing);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posBehind, standing);
    }
  }

  dobjFor(SitBehind)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posBehind, sitting); }
    action()
    {
      sendActorTo(behindLoc, sitting);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posBehind, sitting);
    }
  }

  dobjFor(LieBehind)
  {
    preCond = [objVisible, actorOutOfNested]
    verify() { verifyStandClose(posBehind, lying); }
    action()
    {
      sendActorTo(behindLoc, lying);
      if(!gAction.isImplicit)
        mainReport(&okayStandCloseMsg, posBehind, lying);
    }
  }

  dobjFor(GoAwayFrom)
  {
    preCond = [objVisible]
    verify()
    {
      if(!gActor.canTouch(self))
        illogicalAlready(&alreadyAwayFromMsg);
    }
    action()
    {
      foreach (local loc in [nearLoc, underLoc, behindLoc, frontLoc])
        if(loc != nil && loc.ofKind(NestedRoom) && gActor.isIn(loc))
        {
          loc.removeFromNested;
          break;
        }

      if(gActor.CSGetOutermostRoom.connectionGroup != nil )
         askForDobj(WalkOverTo);
      else
         mainReport(&GoWhereMsg);
    }
  }

  whichLocProp(pos)
  {
    switch(pos)
    {
      case posUnder: return &underLoc;
      case posBehind: return &behindLoc;
      case posFront: return &frontLoc;
      default: return &nearLoc;
    }
  }

  /* We alter some pre-conditions for some verbs. */
  iobjFor(ShowTo)  { preCond = [canTalkToObj] } 
  dobjFor(LookBehind) { preCond = [objVisible, nearObj] }
  dobjFor(LookUnder) { preCond = [objVisible, nearObj] }
  dobjFor(LookIn) { preCond = [objVisible, nearObj] }  
;
;




