/*
     spar - Show Process Accounting Records
     Copyright (C) 1993 Douglas Lee Schales, David K. Hess, David R. Safford

     Please see the file `COPYING' for the complete copyright notice.

spar.c - 10/28/93

*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netdb.h>
#include <errno.h>
#include <sys/acct.h>
#include <signal.h>

#include "config.h"

#include "spar.h"
#include "chario.h"
#include "parser.h"
#include "interp.h"
#include "timesub.h"

extern char *optarg;
extern int optind;

extern struct parsetree *parse(void);
static void extract(int, struct parsetree *, int);
static void flushbuf(void);
static void writebin(struct acct *);
static void writeascii(struct acct *);
extern void writebuf(struct acct *);
static void setlogmode(int);
static void setlogfile(FILE *);
extern char *gettimestr(struct timeval);
static void helpscreen(char *);

extern char *cgetnametty(dev_t);
extern char *cgetnamuid(uid_t);

#define LOGBINARY 0
#define LOGASCII 1

static int unbuffered = 0;
int quietflag;

#ifndef DEFAULT_PACCT
#define DEFAULT_PACCT "/usr/adm/pacct"
#endif

int
main(int argc, char **argv)
{
     struct parsetree *parsetree;
     int logmode = LOGASCII;
     FILE *outfile = stdout;
     FILE *dumpfile = (FILE *)0;
     int file = 0, expr = 0, errflag = 0;
     int reverse = 0;
     int status;
     char *membufname;
     int c;

     quietflag = 0;

     while((c=getopt(argc, argv, "d:e:f:E:F:o:abhUrq")) != -1){
	  switch(c){
	  case 'a':
	       logmode = LOGASCII;
	       break;
	  case 'b':
	       logmode = LOGBINARY;
	       break;
	  case 'q':
	       quietflag = 1;
	       break;
	  case 'r':
	       if(dumpfile==stdin){
		    (void)fprintf(stderr,
				  "Can not use reverse flag with stdin\n");
		    exit(1);
	       }
	       else
		    reverse = 1;
	       break;
	  case 'f':
	       if((status = addfile(optarg)) != NOERR){
		    errflag++;
		    if(status == FILEERR){
			 (void)perror(optarg);
			 exit(1);
		    }
		    else {
			 (void)fprintf(stderr,
				       "%s: unexpected error\n", optarg);
			 exit(1);
		    }
	       }
	       file++;
	       break;
	  case 'F':
	       if((status = includefile(optarg)) != NOERR){
		    errflag++;
		    if(status == FILEERR){
			 (void)perror(optarg);
			 exit(1);
		    }
		    else {
			 (void)fprintf(stderr,
				       "%s: unexpected error\n", optarg);
			 exit(1);
		    }
	       }
	       file++;
	       break;
	  case 'e':
	       membufname = (char *)malloc(20);
	       (void)sprintf(membufname, "argv[%d]", optind-1);
	       if((status = addmembuf(optarg, membufname)) != NOERR){
		    (void)fprintf(stderr, "-e: unexpected error\n");
		    errflag++;
	       }
	       expr++;
	       break;
	  case 'E':
	       membufname = (char *)malloc(20);
	       (void)sprintf(membufname, "argv[%d]", optind-1);
	       if((status = includemembuf(optarg, membufname)) != NOERR){
		    (void)fprintf(stderr, "-E: unexpected error\n");
		    errflag++;
	       }
	       expr++;
	       break;
	  case 'd':
	       if(strcmp(optarg, "-") == 0){
		    if(reverse){
			 (void)fprintf(stderr,
				       "Can not use reverse flag with stdin\n");
			 exit(1);
		    }
		    else
			 dumpfile = stdin;
	       }
	       else if(!(dumpfile = fopen(optarg, "r"))){
		    (void)perror(optarg);
		    errflag++;
	       }
	       break;
	  case 'o':
	       if(!errflag && !(outfile = fopen(optarg, "w"))){
		    (void)perror(optarg);
		    errflag++;
	       }
	       break;
	  case 'h':
	       helpscreen(argv[0]);
	       exit(0);
	       break;
	  case 'U':
	       unbuffered = 1;
	       break;
	  default:
	       errflag++;
	       break;
	  }
     }
     if(errflag)
	  exit(1);

     if(!file && !expr)
	  (void)addmembuf("{print}", "");

     if(!dumpfile)
	  if(!(dumpfile =fopen(DEFAULT_PACCT, "r"))){
	       (void)perror(DEFAULT_PACCT);
	       exit(1);
	  }

     setlogfile(outfile);
     setlogmode(logmode);
     setclockrate();
     if((parsetree = parse()))
	  extract(fileno(dumpfile), parsetree, reverse);
     else
	  exit(1);
     return 0;
}

void
helpscreen(char *name)
{
     (void)fprintf(stdout,"\
%s: usage: %s [options]\n\
	options: -a		ASCII output (the default)\n\
		 -b		Binary output\n\
                 -r             Print records in reverse order\n\
                 -U             Don't do buffered I/O\n\
		 -h		Print this help screen\n\
		 -e script	Specify script inline\n\
		 -E script	Specify script inline\n\
		 -f file	Specify script filename\n\
		 -F file	Specify script filename\n\
		 -d acctfile	Specify pacct file to process\n\
		 -o outfile	Specify output file (def: stdout)\n\
\n\
%s processes UNIX process accounting files, selecting records to\n\
output.  The syntax of the script file is similar to awk(1).\n\
The script can be specified on the command line or can be stored\n\
in a file.  All scripts specified by '-e', '-E', '-f', '-F' are\n\
in effect, processed as a single file.  Scripts specified by '-e' and\n\
'-f' are appended, whereas scripts specified by '-E' and '-F' are pre-\n\
pended.  The use of '-E' or '-F' along with a '#!%s -f' script\n\
executable allows insertion of additional script lines from the command\n\
line.\n",
	     name, name, name, name);
}

static int
nread(int fd, char *buf, int numbytes)
{
     int count = numbytes;
     int n;

     while(count){
	  if((n = read(fd, buf, count)) > 0){
	       buf += n;
	       count -= n;
	  }
	  else
	       break;
     }
     return numbytes-count;
}

static void
extract(int fd, struct parsetree *pt, int revflag)
{
     struct acct buf[1024];
     int i, n;
     int bsize = sizeof(buf);

     if(unbuffered)
	  bsize=sizeof(struct acct);

     if(!revflag){
	  while((n=nread(fd, (char *)&buf, bsize)) > 0){
	       for(i=0;i<(n/sizeof(struct acct));i++)
		    interp(&buf[i], pt);
	  }
     }
     else {
	  int prev;
	  prev = lseek(fd, -bsize , SEEK_END);
	  while((n=nread(fd, (char *)&buf, bsize)) > 0){
	       for(i=(n/sizeof(struct acct))-1;i>=0;i--)
		    interp(&buf[i], pt);
	       if(!prev)
		    break;
	       if(prev < bsize){
		    (void)lseek(fd, 0, SEEK_SET);
		    bsize = prev;
		    prev = 0;
	       }
	       else {
		    prev = lseek(fd, -(bsize<<1), SEEK_CUR);
	       }
	  }
     }
     (void)flushbuf();
}

static int outlogmode = LOGBINARY;
static FILE *logfile = stdout;

static void
setlogfile(FILE *f)
{
     logfile = f;
}

static void
setlogmode(int logmode)
{
     outlogmode = logmode;
}

#define OUTBUFSIZE 1000
struct acct outbuffer[OUTBUFSIZE];
static int outptr = 0;

void
writebuf(struct acct *outbuf)
{
     switch(outlogmode){
     case LOGBINARY:
	  writebin(outbuf);
	  break;
     case LOGASCII:
	  writeascii(outbuf);
	  break;
     default:
	  (void)fprintf(stderr, "Internal error... unknown output mode.\n");
	  exit(1);
	  break;
     }
}

static void
writebin(struct acct *outbuf)
{
     if(outptr == OUTBUFSIZE)
	  (void)flushbuf();
     (void)memcpy((void *)&outbuffer[outptr++],
		  (void *)outbuf,
		  sizeof(struct acct));
     if(unbuffered)
	  (void)flushbuf();
}

static void
flushbuf(void)
{
     if(outptr)
	  (void)write(fileno(logfile), (char *)outbuffer,
		sizeof(struct acct)*outptr);
     outptr = 0;
}

static char *sigtable[127];
static int sigtblinit = 0;

void
initsigtbl(void)
{
     sigtblinit = 1;

     memset(sigtable, 0, sizeof(sigtable));
     sigtable[SIGHUP] = "HUP";
     sigtable[SIGINT] = "INT";
     sigtable[SIGQUIT] = "QUIT";
     sigtable[SIGILL] = "ILL";
     sigtable[SIGABRT] = "ABORT";
     sigtable[SIGFPE] = "FPE";
     sigtable[SIGKILL] = "KILL";
     sigtable[SIGBUS] = "BUS";
     sigtable[SIGSEGV] = "SEGV";
     sigtable[SIGPIPE] = "PIPE";
     sigtable[SIGALRM] = "ALARM";
     sigtable[SIGTERM] = "TERM";
#ifdef SIGCHLD
     sigtable[SIGCHLD] = "CHILD";
#endif
#ifdef SIGCLD
     sigtable[SIGCLD] = "CHILD";
#endif
     sigtable[SIGSTOP] = "STOP";
#ifdef SIGXCPU
     sigtable[SIGXCPU] = "XCPU";
#endif
#ifdef SIGXFSZ
     sigtable[SIGXFSZ] = "XFSZ";
#endif
}

static void
writeascii(struct acct *outbuf)
{
     struct timeval tm;
     char cbuf[AC_COMMLEN+1];
     char *tty;
     char *signame;
     int signum;

     if(!sigtblinit)
	  initsigtbl();
     
     tm.tv_sec = outbuf->ac_btime;
     tm.tv_usec = 0;
     (void)memcpy((void *)cbuf, (void *)outbuf->ac_comm, AC_COMMLEN);
     cbuf[AC_COMMLEN] = 0;

     tty = (char *)cgetnametty(outbuf->ac_tty);

     if(strncmp(tty, "/dev/", 5) == 0)
	  tty += 5;
	  

     (void)fprintf(logfile, "%s %c%c %-8s %-8s %-10s %s",
	     gettimestr(tm),
	     outbuf->ac_flag & AFORK ? 'F' : ' ',
	     outbuf->ac_flag & ASU ? 'P' : ' ',
	     cbuf, 
	     cgetnamuid(outbuf->ac_uid),
	     tty,
	     getelapsed(outbuf->ac_etime));
     putc(' ', logfile);
     (void)fputs(getelapsed2(outbuf->ac_utime), logfile);
     putc(' ', logfile);
     (void)fputs(getelapsed2(outbuf->ac_stime), logfile);
     if((signum = ACSIGEXIT(outbuf->ac_stat))){
	  if((signame = sigtable[signum])){
	       putc(' ', logfile);
	       (void)fputs(signame, logfile);
	  }
	  else {
	       fprintf(logfile, " SIG#%02d", signum);
	  }
     }
     putc('\n', logfile);
}
