/*
     spar - Show Process Accounting Records
     Copyright (C) 1993 Douglas Lee Schales, David K. Hess, David R. Safford

     Please see the file `COPYING' for the complete copyright notice.

interp.c - 10/28/93

*/
#include <stdio.h>
#include <sys/types.h>
#include <time.h>
#include <sys/acct.h>

#include "config.h"
#include "spar.h"
#include "parser.h"
#include "y.tab.h"
#include "interp.h"
#include "timesub.h"

#define CONTINUE 0
#define RETURN 1

static unsigned long evaluate(struct acct *, struct parsenode *);
static unsigned long checkcond(unsigned long, unsigned long, int);
static unsigned long strcond(unsigned long, unsigned long, int);
static int perform(struct acct *, struct actionlist *);
extern void writebuf(struct acct *);

void
interp(struct acct *rec, struct parsetree *pt)
{
     struct parsetree *rove;
     int status;

     for(rove = pt;rove;rove=rove->next){
	  if(!rove->conditions || evaluate(rec, rove->conditions)){
	       status = perform(rec, rove->actions);
	       if(status == RETURN)
		    return;
	  }
     }
}

unsigned long
evaluate(struct acct *rec, struct parsenode *pn)
{
     unsigned long lhs, rhs;
     struct tm *tmb;
     time_t etime;

     if(pn->nodetype == KEY){
	  switch(pn->nodeval){
	  case USER:
	       return rec->ac_uid;
	  case GROUP:
	       return rec->ac_gid;
	  case COMMAND:
	       return (unsigned long)rec->ac_comm;
	  case PRIV:
	       return !!(rec->ac_flag & ASU);
	  case TTY:
	       return rec->ac_tty;
	  case STARTDATE:
	       return rec->ac_btime;
	  case ENDDATE:
	       return rec->ac_btime + fromticks((unsigned long)rec->ac_etime);
	  case CPUTIME:
	       return fromticks((unsigned long)rec->ac_utime) + 
		    fromticks((unsigned long)rec->ac_stime);
          case USERTIME:
	       return fromticks((unsigned long)rec->ac_utime);
	  case SYSTIME:
	       return fromticks((unsigned long)rec->ac_stime);
	  case STARTTIME:
	       tmb = localtime(&rec->ac_btime);
	       return tmb->tm_hour*3600 + tmb->tm_min*60 + tmb->tm_sec;
	  case ENDTIME:
	       etime = (time_t)rec->ac_btime + 
		    fromticks((unsigned long)rec->ac_etime);
	       tmb = localtime(&etime);
	       return tmb->tm_hour*3600 + tmb->tm_min*60 + tmb->tm_sec;
	  default:
	       /* hmmmm... */
	       break;
	  }
     }
     else if(pn->nodetype == VAL)
	  return pn->nodeval;
     else if(pn->nodetype == UOPER){
	  rhs = evaluate(rec, pn->rhs);
	  switch(pn->nodeval){
	  case '!':
	       return !rhs;
	  default:
	       break;
	  }
     }
     else {
	  switch(pn->nodeval){
	  case OR:
	       if((lhs = evaluate(rec, pn->lhs)))
		    return lhs;
	       else
		    return evaluate(rec, pn->rhs);
	  case AND:
	       if(!(lhs = evaluate(rec, pn->lhs)))
		    return lhs;
	       else
		    return evaluate(rec, pn->rhs);
	  case MASKOPER:
	  case '&':
	       lhs = evaluate(rec, pn->lhs);
	       rhs = evaluate(rec, pn->rhs);
	       return lhs & rhs;
	  default:
	       lhs = evaluate(rec, pn->lhs);
	       rhs = evaluate(rec, pn->rhs);
	       return checkcond(lhs, rhs, pn->nodeval);
	  }
     }
     return 0;
}
     
unsigned long
checkcond(unsigned long lhs, unsigned long rhs, int cond)
{
     switch(cond){
     case '=':
	  return lhs == rhs;
     case '>':
	  return lhs > rhs;
     case '<':
	  return lhs < rhs;
     case NEQ:
	  return lhs != rhs;
     case LEQ:
	  return lhs <= rhs;
     case GEQ:
	  return lhs >= rhs;
     default:
	  return strcond(lhs, rhs, cond);
     }
     return 0;
}

static unsigned long
strcond(unsigned long lhs, unsigned long rhs, int cond)
{
     char slhs[AC_COMMLEN+1], srhs[AC_COMMLEN+1];

     (void)strncpy(slhs, (char *)lhs, AC_COMMLEN);
     (void)strncpy(srhs, (char *)rhs, AC_COMMLEN);
     slhs[AC_COMMLEN] = 0;
     srhs[AC_COMMLEN] = 0;
     switch(cond){
     case STREQ:
	  return !strcmp(slhs, srhs);
     case STRNE:
	  return !!strcmp(slhs, srhs);
     }
     return 0;
}

int
perform(struct acct *rec, struct actionlist *al)
{
     int topaction = CONTINUE;
     struct actionlist *rove;

     for(rove=al;topaction != RETURN && rove;rove=rove->next)
	  switch(rove->action){
	  case PRINT:
	       writebuf(rec);
	       break;
	  case NEXT:
	       topaction = RETURN;
	       break;
	  default:
	       /* Hmmmm */
	       break;
	  }

     return topaction;
}

