/* BOX-RELATIONS.C -- functions implementing creation/access of relationships
   between boxes.

   $Header: box-relations.c,v 1.7 91/02/18 16:55:30 heydon Locked $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   IMPLEMENTATION NOTES

   At the cost of n^2 space, the current implementation is such that the type
   of containment relation can be determinded in O(1) time. If space becomes
   more restricted, it may be necessary to change the implementation to a more
   space-efficient one; of course, this space savings would be at the cost of
   some processing time.
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "error.h"
#include "box.h"
#include "box.g"
#include "box-relations.h"

/* GLOBAL VARIABLE DEFINITIONS ============================================ */

BoxRelationMatrix rel_matrix[NumRoles];

/* LOCAL VARIABLES ======================================================== */

/* current parent (this points to a row of BoxRelations in the matrix */
static BoxRelation *curr_parent = NULL;

/* role of 'curr_parent' */
static BoxRole curr_parent_role;

/* maximum number of boxes used by any BoxRole */
static int max_num_boxes = 0;

#ifdef DEBUG

/* array to convert from box indices to sysnames */
static SysName *index_to_sysname[NumRoles];

#endif DEBUG

/* LOCAL FUNCTIONS ======================================================== */

static BoxRelation **Build2DBoxRelArray(num_rows,num_cols)
  int num_rows,num_cols;
/* This routine builds a 'num_rows' by 'num_cols' 2D "array" of BoxRelation
   values. The structure is not a true array, but rather a column vector of
   pointers to rows of BoxRelation values (i.e., each pointer in the column
   vector points to an array of BoxRelation's).
*/
{
    register int row;
    BoxRelation **rows,**row_ptr;

    /* allocate column memory */
    rows = row_ptr = AllocPtrArray(BoxRelation,(unsigned)num_rows);

    /* make row pointers point to newly allocated rows */
    for (row=0; row < num_rows; row++) {
	*row_ptr++ = AllocArray(BoxRelation,(unsigned)num_cols);
    }
    return(rows);
}

static void Free2DBoxRelArray(col_ptr,num_rows)
  BoxRelation **col_ptr;
  int num_rows;
/* This routine frees the memory used by the structure 'col_ptr' which should
   have been created by the cousin routine 'Build2DBoxRelArray()'. The value
   'num_rows' is the number of rows in the array.
*/
{
    register int row;

    StepIndex(row,0,num_rows) {
	FreeBlock((Generic *)col_ptr[row]);
    }
    FreeBlock((Generic *)col_ptr);
}

#ifdef DEBUG
static SysName *BuildIndexArray(role,number_of_boxes)
  BoxRole role;
  int number_of_boxes;
/* RETURNS an array 'r' such that if I is the index of a box B of role 'role'
   and if r[I] = S, then S is the sysname of B.
*/
{
    SysName *result;
    Box *curr;

    result = AllocArray(SysName,number_of_boxes);
    StepLinkedList(curr,all_head[(int)role]) {
	result[IndexOf(curr)] = BSysNameOf(curr);
    }
    return(result);
}
#endif DEBUG

static void CloseBoxContainmentRelations(role,new)
  BoxRole role;
  BoxRelation **new;		/* represents data in "row" k */
/* Form transitive closure on ProperAncestor relations for boxes of type
   'role'. 'new' is a pointer to an array of pointers to rows of BoxRelations.
   Initially, 'new' can point to garbage. It is important, however, that 'new'
   point to a 2D matrix at least as big as the BoxRelation matrix for 'role'
   (it may be bigger, however).

   The algorithm implemented here is a slightly modified version of the
   reflexive-transitive closure algorithm on page 199 of "The Design and
   Analysis of Computer Algorithms" by Aho, Hopcroft, and Ullman. The var-
   iables 'i', 'j', and 'k' in the book correspond to my program variables
   'row', 'col', and 'level' respectively.

   The primary modification is that only 2*(n^2) space is used (implemented
   as two n^2 arrays) rather than (n^3) space as is suggested in the algor-
   ithm on page 198. The space savings is achievable because we only need to
   know the values from the *previous* step in the computation (i.e., the C^k
   value is computed only from C^(k-1) values). The two pointers 'old' and
   'new' point to the "C^k-1" and "C^k" values respectively; they are simply
   swapped each time 'k' is incremented.

   When the dynamic part of the algorithm finishes, depending on whether an
   even or odd number of swaps were made, we made need to copy the final
   results back into the original storage area.

   Another modification to the algorithm was necessary because it is necessary
   to distinguish between DirectContainment and StarredContainment. In
   general, the expression for new[row][col] is:

     new[row][col] = old[row][col] OR (old[row][level] AND old[level][col])

   There are thus 2 ways for new[row][col] to be set non-zero. If there is
   already a path from 'row' to 'col' going through no node higher than
   'level'-1 (i.e. the OR is satisfied because old[row][col] is non-zero),
   then we should *preserve* that information (i.e., if it was
   DirectContainment already, retain that value; if it was StarredContainment,
   retain that value) -- we do that (obviously) by setting:

     new[row][col] = old[row][col].

   However, if the first part of the OR fails, but the second part succeeds,
   then there is a path from 'row' to 'col', and what's more, it is of length
   at least 2. In this case the containment is always starred, so we do:

     new[row][col] = StarredContainment.

   As the comment below points out, it is not necessary to set new[row][col]
   to be NoRelation if both parts of the OR fail because no positive
   containment relation (Direct or Starred) will ever need to be undone.
   However, since new may contain garbage, this step is necessary on the first
   pass.
*/
{
    register int col,row,level;	/* in decreasing order of speed needed */
    int num_boxes;
    BoxRelation **old,**temp;

    old = Column(role);		/* represents data in "row" k-1 */
    num_boxes = NumBoxesOf(MatrixStructPtr(role));

    /* initialize old data to be reflexive */
    StepIndex(row,0,num_boxes) {
	old[row][row] = DirectContainment;
    }

    /* proceed w/ dynamic programming loop */
    StepIndex(level,0,num_boxes) {        /* |                            */
	StepIndex(row,0,num_boxes) {      /* | <-- O(n^3) dynamic program */
	    StepIndex(col,0,num_boxes) {  /* |                            */
		/* 
		 * In the following test, we rely on the fact that an entry
		 * will evaluate non-zero iff it represents either
		 * DirectContainment or StarredContainment. This is valid
		 * because NoRelation is equated with 0, and at this point,
		 * there are no CrissCross values in the matrix.
                */
		if (old[row][col]) {
		    new[row][col] = old[row][col];
		} else if (old[row][level] && old[level][col]) {
		    new[row][col] = StarredContainment;
		} else if (level == 0) {
		/* new[row][col] = NoRelation is unnecessary after the first
		   pass because once an entry is set non-zero, it will never
		   need to become = 0 later on */
		    new[row][col] = NoRelation;
		}
	    }
	}
	/* swap pointers so we work back in the other direction */
	Swap(old,new,temp);
    }

    /* if there are an odd number of boxes, then new == Column(role), but
     * old points to the most recent matrix, so copy the final results back
     * to the Column(role) array */
    if (num_boxes % 2) {
	StepIndex(row,0,num_boxes) {
	    StepIndex(col,0,num_boxes) {
		new[row][col] = old[row][col];
	    }
	}
    }
}

#define FastIndex col

void ComputeEqualsRelations(role,eq)
  BoxRole role;
  BoxEqual eq;
/* Form "equal" relations amongst pairs of boxes of type 'role' according to
   "equality" 'eq'. If 'eq' is Strict, we make identical boxes "equal". If
   'eq' is Bowtie, we do that, but in addition, make boxes that crisscross
   "equal". Two boxes A and B crisscross iff:

          1) "A does NOT contain B"
      AND 2) "B does NOT contain A"
      AND 3) "there exists some box C such that A and B both contain C".

   where by "contain" we mean either DirectContainment or StarredContainment.
   This is a naive O(n^2) time algorithm.
*/
{
    int row1,row2,num_boxes;
    register int col;
    BoxRelation *first,*second,**col_ptr; /* save on execution time by
					     requiring less dereferencing */

    /* initialize for faster execution */
    col_ptr = Column(role);
    num_boxes = NumBoxesOf(MatrixStructPtr(role));

    /* Set relations so every box is Equal to itself */
    StepIndex(FastIndex,0,num_boxes) {
	col_ptr[FastIndex][FastIndex] = Equal;
    }

    if (eq == Bowtie) {
	/* For every 2 boxes (row1,row2), see if they contain a 3rd box in
	   common (namely, col); if so, set the relation between them =
	   Equal. */
	StepIndex(row1,0,(num_boxes-1)) {
	    first = col_ptr[row1];
	    StepIndex(row2,(row1+1),num_boxes) {
		second = col_ptr[row2];
		if (!(DirectOrStarContains(first[row2]) ||
		      DirectOrStarContains(second[row1]))) {
		    /* see if the 2 boxes contain a common box */
		    StepIndex(col,0,num_boxes) {
			if (DirectOrStarContains(first[col]) &&
			    DirectOrStarContains(second[col])) {
			    first[row2] = Equal;
			    second[row1] = Equal;
			    /* we can immed. try the next pair of boxes */
			    break;
			}
		    }
		}
	    }
	}
    }
}

#ifdef DEBUG
#define IndexPrintWidth   6
#define NumIndecesPerLine 6

static void ShowContainmentSingleBox(role,parent_index)
  BoxRole role;
  BoxIndex parent_index;
/* Displays the containment information associated with the box with BoxRole
   'role' and BoxIndex 'parent_index'. The sysnames of the parent and those
   of the boxes it contains are printed.
*/
{
    register int child_index;
    short output_cnt = 0;
    BoxRelation entry_val,*curr_row;
    SysName *i2s;

    i2s = index_to_sysname[(int)role];
    printf("    %*d: ",IndexPrintWidth,i2s[parent_index]);
    curr_row = Row(role,parent_index);
    StepIndex(child_index,0,NumBoxesOf(MatrixStructPtr(role))) {
	entry_val = curr_row[child_index];
	if (entry_val==DirectContainment || entry_val==StarredContainment) {
	    printf("%*d",IndexPrintWidth,i2s[child_index]);
	    MyPuts(entry_val == StarredContainment ? "(*)" : "   ");
	    MyPuts((++output_cnt % NumIndecesPerLine) == 0 ?
		   "\n            " : " ");
	}
    }
    (void)putchar('\n');
}

#undef  NumIndecesPerLine
#define NumIndecesPerLine 8

static void ShowEqualsSingleBox(role,index)
  BoxRole role;
  BoxIndex index;
/* Displays the Equal information associated with the box with BoxRole
   'role' and BoxIndex 'index'.
*/
{
    register int other_index,output_cnt=0;
    BoxRelation *curr_row;

    printf("    %*d: ",IndexPrintWidth,index);
    curr_row = Row(role,index);
    StepIndex(other_index,0,NumBoxesOf(MatrixStructPtr(role))) {
	if (curr_row[other_index] == Equal) {
	    printf("%*d",IndexPrintWidth,other_index);
	    MyPuts((++output_cnt % NumIndecesPerLine) == 0 ?
		   "\n            " : " ");
	}
    }
    (void)putchar('\n');
}
#endif DEBUG

/* GLOBAL Functions ======================================================= */

void InitBoxRelations(role,number_of_boxes)
  BoxRole role;
  int number_of_boxes;
{
    /* Initialize number of boxes */
    NumBoxesOf(MatrixStructPtr(role)) = number_of_boxes;
    max_num_boxes = Max(max_num_boxes,number_of_boxes);

    /* Build the 2D array of BoxRelation's */
    Column(role) = Build2DBoxRelArray(number_of_boxes,number_of_boxes);

#ifdef DEBUG
    /* Build the index_to_sysname[] array for this role */
    index_to_sysname[(int)role] = BuildIndexArray(role,number_of_boxes);
#endif DEBUG
}

void AssertDirectContainmentRelation(parent_ptr,child_ptr,line_no)
  Box *parent_ptr,*child_ptr;
  int line_no;
{
    BoxRole parent_role;

    if ((parent_role=RoleOf(parent_ptr)) != RoleOf(child_ptr)) {
	ParseErrorI(line_no,
		    "child box %d has a different role from its parent",
		    BSysNameOf(child_ptr));
    } else {
	Ent(parent_role,IndexOf(parent_ptr),IndexOf(child_ptr)) =
	    DirectContainment;
    }
}

void AssertParentOfContainment(parent_ptr)
  Box *parent_ptr;
{
    curr_parent_role = RoleOf(parent_ptr);
    curr_parent = Row(curr_parent_role,IndexOf(parent_ptr));
    AssertBoxNonAtomic(parent_ptr);
}

void AssertChildOfContainment(child_ptr,line_no)
  Box *child_ptr;
  int line_no;
{
#ifdef DEBUG
    if (curr_parent == NULL) {    /* parent not initialized */
	ProgrammerErrorI("box-relations.c",
	      "current parent not initialized adding child index %d",
			 IndexOf(child_ptr));
    }
#endif DEBUG
    if (RoleOf(child_ptr) != curr_parent_role) {
	ParseErrorI(line_no,
		    "child box %d has a different role from its parent",
		    BSysNameOf(child_ptr));
    } else {
	curr_parent[IndexOf(child_ptr)] = DirectContainment;
    }
}

void CloseAllBoxContainmentRelations()
{
    int role;
    BoxRelation **tmp_closure_array; /* temp storage for trans-closure alg. */

    /* allocate temp storage for largest number of boxes necessary */
    tmp_closure_array = Build2DBoxRelArray(max_num_boxes,max_num_boxes);

    /* do closure on each of the BoxRole's */
    StepIndex(role,0,NumRoles) {
	CloseBoxContainmentRelations((BoxRole)role,tmp_closure_array);
    }

    /* free temp storage 'tmp_closure_array' */
    Free2DBoxRelArray(tmp_closure_array,max_num_boxes);
}

void ComputeAllEqualsRelations(eq)
  BoxEqual eq;
{
    int role;
    StepIndex(role,0,NumRoles) {
	ComputeEqualsRelations((BoxRole)role,eq);
    }
}

#ifdef DEBUG
void ShowContainmentAllBoxes()
{
    int role,parent_index;

    MyPuts("\nSHOW-CONTAINMENT-ALL-BOXES():");
    StepIndex(role,0,NumRoles) {
	printf("\n  Role: %s\n",StringBoxRole((BoxRole)role));
	StepIndex(parent_index,0,NumBoxesOf(MatrixStructPtr(role))) {
	    ShowContainmentSingleBox((BoxRole)role,parent_index);
	}
    }
}
#endif DEBUG

#ifdef DEBUG
void ShowEqualsAllBoxes()
/* Displays the Equal information of all roles of boxes.
*/
{
    int role,parent_index;

    MyPuts("\nSHOW-EQUALS-ALL-BOXES():");
    StepIndex(role,0,NumRoles) {
	printf("\n  Role: %s\n",StringBoxRole((BoxRole)role));
	StepIndex(parent_index,0,NumBoxesOf(MatrixStructPtr(role))) {
	    ShowEqualsSingleBox((BoxRole)role,parent_index);
	}
    }
}
#endif DEBUG
