/* PARSER.C -- top level routines to parse input and construct parse tree

   $Header: parser.c,v 1.8 91/08/13 14:44:35 heydon Exp $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <stdio.h>
#include <my-types.h>
#include "mem.h"
#include <my-defs.h>

#include "id-hash.h"
#include "parser.h"
#include "iff.y.h"

/* GLOBAL VARIABLE DEFINITIONS ============================================ */

int parse_error_cnt;
int curr_line_no;
char quoted_string[STRING_BUFF_SIZE];

/* LOCAL FUNCTIONS ======================================================== */

ListEntry *CopyListEntry(l)
  ListEntry *l;
{
    ListEntry *result = AllocOne(ListEntry);
    result->list_prop_val_ptr = CopyPropVal(l->list_prop_val_ptr);
    result->next = (l->next == NULL) ? NULL : CopyListEntry(l->next);
    return(result);
}

static void FreePropVal(p)
  PropVal *p;
/* Free the space allocated for strings or lists in PropVal 'p'. Then free the
   space allocated for 'p' itself. Note that space for identifiers is *not*
   freed, since they live in the ID hash table. Space for integers is
   implicitly freed when 'p' is freed, since integers are stored in the
   PropVal itself.
*/
{
    void FreePropValList();

    switch (p->prop_val_type) {
      case StringPValType:
	FreeBlock(p->val.string_val);
	break;
      case ListPValType:
	FreePropValList(p->val.list_head);
	break;
    }
    FreeBlock(p);
}

static void FreePropValList(l)
  ListEntry *l;
{
    ListEntry *temp;

    while (l) {
	temp = l;
	l = l->next;
	FreePropVal(temp->list_prop_val_ptr);
	FreeBlock(temp);
    }
}

static void FreeAttrEntry(e)
  AttrEntry *e;
{
    FreePropVal(e->prop_val_ptr);
    FreeBlock(e);
}

static void FreeAttrList(l)
  AttrEntry *l;
{
    AttrEntry *temp;
    while (l) {
	temp = l;
	l = l->next;
	FreeAttrEntry(temp);
    }
}

static void FreeEntry(e)
  Entry *e;
{
    FreeAttrList(e->attr_list_head);
    FreeBlock(e);
}

#ifdef DEBUG
static void ShowPropVal(pval_ptr)
  PropVal *pval_ptr;
{
    ListEntry *list_entry_ptr;

    switch (PropValTypeOf(pval_ptr)) {
      case IntPValType:
	printf("%d",IntValOf(pval_ptr));
	break;
      case IdPValType:
	printf("%s (0x%x)",IdValOf(pval_ptr),(int)IdValOf(pval_ptr));
	break;
      case StringPValType:
	printf("'%s'",StringValOf(pval_ptr),
	       (int)StringValOf(pval_ptr));
	break;
      case ListPValType:
	printf("{ ");
	StepLinkedList(list_entry_ptr,ListHeadOf(pval_ptr)) {
	    ShowPropVal(ListPropValPtrOf(list_entry_ptr));
	    (void)putchar(' ');
	}
	printf("}");
	break;
    }
}
#endif

#ifdef DEBUG
static void ShowAttr(attr_ptr)
  AttrEntry *attr_ptr;
{
    printf("  ** Line Number     = %d\n",AttrLineNumberOf(attr_ptr));
    printf("     Attribute Name  = %s\n",AttrNameOf(attr_ptr));
    printf("     Attribute Value = ");
    ShowPropVal(PropValPtrOf(attr_ptr));
    (void)putchar('\n');
}
#endif

#ifdef DEBUG
static void ShowEntry(entry_ptr)
  Entry *entry_ptr;
{
    AttrEntry *attr_ptr;
    printf("Entry Type: %s (0x%x); Line No. = %d\n",EntryNameOf(entry_ptr),
	   EntryTypeOf(entry_ptr),EntryLineNumberOf(entry_ptr));
    StepLinkedList(attr_ptr,AttrListHeadOf(entry_ptr)) {
	ShowAttr(attr_ptr);
    }
}
#endif

/* GLOBAL FUNCTIONS ======================================================= */

void InitParser()
{
    InitMem();
    InitIdHashTable();
}

Entry *ParseFile(fp)
  FILE *fp;
{
    extern FILE *yyin;		/* file pointer lex reads from */
    extern YYSTYPE yyval;	/* yacc's last formed value */
    Entry *parse_tree;		/* top-level result of the parse */
    int errors_found;		/* flag of errors found during parse */

#ifdef DEBUG
/* debug info before parse */
    if (InDebugRangeP(ShowIdHashTableBeforeParseLevel))
	ShowIdHashTable();
#endif DEBUG

    /* initialize globals for parse */
    yyin = fp;
    parse_error_cnt = 0;
    curr_line_no = 1;

    /* parse the file */
    errors_found = yyparse();
    parse_tree = yyval.entry_list.first;

#ifdef DEBUG
    /* debug info after parse */
    if (InDebugRangeP(ShowIdHashTableAfterParseLevel))
	ShowIdHashTable();
    if (InDebugRangeP(ShowParseTreeLevel))
	ShowParseTree(parse_tree);
#endif DEBUG

    /* return NULL iff errors have occurred */
    return((errors_found || parse_error_cnt>0) ? (Entry *)NULL : parse_tree);
}

PropVal *CopyPropVal(pv)
  PropVal *pv;
{
    PropVal *result = AllocOne(PropVal);
    switch (result->prop_val_type = pv->prop_val_type) {
      case IntPValType:
	result->val.int_val = pv->val.int_val; break;
      case IdPValType:
	/* string copy is not necessary since identifiers are never freed;
	   they live permanently in the ID hash table */
	result->val.id_val = pv->val.id_val;
      case StringPValType:
	CopyString(result->val.string_val,pv->val.string_val); break;
      case ListPValType:
	result->val.list_head = CopyListEntry(pv->val.list_head); break;
      default:
	fprintf(stderr,"CopyPropVal(): unknown property value type 0x%x.\n",
		pv->prop_val_type);
	exit(-1);
    }
    return(result);
}

void FreeParseTree(p_tree)
  Entry *p_tree;
{
    Entry *temp;
    while (p_tree) {
	temp = p_tree;
	p_tree = Next(p_tree);
	FreeEntry(temp);
    }
}

void ShutDownParser()
{
    ShutDownIdHashTable();
    ShutDownMem();
}

#ifdef DEBUG
void ShowParseTree(p_tree)
  Entry *p_tree;
{
    printf("\nSHOW-PARSE-TREE():\n");
    for (; p_tree != NULL; Next(p_tree))
	ShowEntry(p_tree);
}
#endif
