/* EXTRACT.C -- functions for decomposing (extracting data from) Entry's in
                the parse tree.

   $Header: extract.c,v 1.7 90/12/12 23:42:46 heydon Locked $

   Written by Allan Heydon for the Miro project at Carnegie Mellon
*/

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include <my-types.h>

#include "error.h"
#include "mem.h"
#include <my-defs.h>
#include "parser.h"
#include "parser.g"
#include "convert.h"
#include "extract.h"

/* LOCAL VARIABLES ======================================================== */

/* Head of linked list of PNameDesignator's */
static PNameDesignator *p_name_list;

/* Size of current PNameDesignator list */
static int curr_p_name_list_sz;

#ifdef DEBUG
/* Max size of the PNameDesignator list */
static int max_list_sz;
#endif

/* GLOBAL FUNCTIONS ======================================================= */

void InitExtract(max_size)
  int max_size;
{
    p_name_list = AllocArray(PNameDesignator,(unsigned)max_size);
#ifdef DEBUG
    max_list_sz = max_size;
#endif
}

void StartNewPNameList()
{
    curr_p_name_list_sz = 0;
}

void AddPNameDesignator(p_name,required,f,v_ptr,list_desc_ptr)
  String p_name;
  Boolean required;
  ConvertFunc f;
  Generic *v_ptr;
  ListDesc *list_desc_ptr;
{
    PNameDesignator *curr_desig_ptr;

#ifdef DEBUG
    if (curr_p_name_list_sz > max_list_sz) {
	ProgrammerErrorI("extract.c","list size of %d too small",max_list_sz);
    }
#endif
    curr_desig_ptr = p_name_list + curr_p_name_list_sz;
    PNameOf(curr_desig_ptr) = p_name;
    ConvertFuncOf(curr_desig_ptr) = f;
    ValPtrOf(curr_desig_ptr) = v_ptr;
    ListDescPtrOf(curr_desig_ptr) = list_desc_ptr;
    if (list_desc_ptr != NULL) DesigPtrOf(list_desc_ptr) = curr_desig_ptr;
    ValRequiredPOf(curr_desig_ptr) = required;
    curr_p_name_list_sz++;
}

Boolean ValFoundP(p_name)
  String p_name;
{
    register int desig_ix;
    register PNameDesignator *desig_ptr;

    for (desig_ix=0; desig_ix < curr_p_name_list_sz; desig_ix++) {
	desig_ptr = p_name_list + desig_ix;
	if (SameString(PNameOf(desig_ptr),p_name)) {
	    return(ValFoundPOf(desig_ptr));
	}
    }
    return(False);
}

Boolean MatchPNames(entry_ptr)
  Entry *entry_ptr;
/* IMPLENTATION NOTE: This function uses an inefficient O(n^2) algorithm to
   match property names to slots in the p_name_list (list of property name
   designators).
*/
{
    register AttrEntry *attr_ptr;
    register int desig_ix;
    register PNameDesignator *desig_ptr;
    ListDesc *desc_ptr;
    int start_error_cnt = parse_error_cnt;

/* initialize the list of designators */
    for (desig_ix=0; desig_ix < curr_p_name_list_sz; desig_ix++) {
	desig_ptr = p_name_list + desig_ix;
	ValFoundPOf(desig_ptr) = False;
	if (ListDescPtrOf(desig_ptr) != NULL) {
	    MatchedPOf(ListDescPtrOf(desig_ptr)) = False;
	}
    }

/* try to match each attribute name associated with Entry '*entry_ptr' */
    StepLinkedList(attr_ptr,AttrListHeadOf(entry_ptr)) {
	/* try to match against each PNameDesignator
	 */
	for (desig_ix=0; desig_ix < curr_p_name_list_sz; desig_ix++) {
	    desig_ptr = p_name_list + desig_ix;
	    if (!ValFoundPOf(desig_ptr) &&
		SameString(AttrNameOf(attr_ptr),PNameOf(desig_ptr))) {
		ValFoundPOf(desig_ptr) = True;
/* switch based on whether attribute value is a list or not */
if (PropValTypeOf(PropValPtrOf(attr_ptr)) != ListPValType) {
    if (ValPtrOf(desig_ptr) == NULL) {
	/* print an error message */
	ParseErrorS(AttrLineNumberOf(attr_ptr),
	    "value associated with property name '%s' should be a list",
		    AttrNameOf(attr_ptr));
    } else {
	/* call the conversion function corresponding to this
	   attribute value */
	(*ConvertFuncOf(desig_ptr))
	    (PropValPtrOf(attr_ptr),ValPtrOf(desig_ptr),
	     AttrLineNumberOf(attr_ptr));
    }
} else {
    if (ListDescPtrOf(desig_ptr) == NULL) {
	/* print an error message */
	ParseErrorS(AttrLineNumberOf(attr_ptr),
	    "value associated with property name '%s' should not be a list",
		    AttrNameOf(attr_ptr));
    } else {
	/* initialize the corresponding ListDesc */
	desc_ptr = ListDescPtrOf(desig_ptr);
	MatchedPOf(desc_ptr) = True;
	StartLineNumberOf(desc_ptr) = AttrLineNumberOf(attr_ptr);
	NextListEntryPtrOf(desc_ptr) = ListHeadOf(PropValPtrOf(attr_ptr));
    }
}
		break;		/* try the next attribute */
	    }
	}
    }

/* check to see if all attributes were found; report errors for those not
   found */
    StepIndex(desig_ix,0,curr_p_name_list_sz) {
	desig_ptr = p_name_list + desig_ix;
	if ((!ValFoundPOf(desig_ptr)) && ValRequiredPOf(desig_ptr)) {
	    /* required attribute was not found; report an error */
	    ParseErrorSS(EntryLineNumberOf(entry_ptr),
		"this '%s' entry is missing the required property named '%s'",
			 EntryNameOf(entry_ptr),PNameOf(desig_ptr));
	}
    }
    return(NotOf(start_error_cnt==parse_error_cnt));
}

Boolean MatchNextListElement(list_desc,v_ptr)
  ListDesc *list_desc;
  Generic *v_ptr;
{
    int start_error_cnt = parse_error_cnt;

    (*(ConvertFuncOf(DesigPtrOf(list_desc))))
	(ListPropValPtrOf(NextListEntryPtrOf(list_desc)),
	 v_ptr,StartLineNumberOf(list_desc));
    Next(NextListEntryPtrOf(list_desc));
    return(NotOf(start_error_cnt==parse_error_cnt));
}
