
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* SYSNAME-HASH.H -- Declarations for types/functions related to data-
                     structures dealing with hashing of box/arrow SysName's.

   $Header: sysname-hash.h,v 1.3 90/07/24 17:12:25 heydon Stable $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   This module implements a hash table for Box and Arrow Sysname's. The type
   SysName is defined by the "box" module. Associated with each SysName entry
   in the hash table is the type of the entry (i.e., either box or arrow) and
   a pointer to the corresponding Box or Arrow object. By associating a type
   with each entry, a Box and Arrow having the same SysName can both be stored
   in the hash table. However, no more than one box and one arrow of a
   particular SysName can be stored in the table at once.
   
   The hash table is implemented with bucket hashing. Each bucket is
   implemented as a linked list of SysNameDrops (get it? drops in buckets? ha
   ha). The type SysNameDropType has been defined to enumerate the type of
   each SysNameDrop.

   WHAT THIS MODULE PROVIDES

   This module defines the following TYPES:
     SysNameDropType - the type of each SysNameDrop
     SysNameDrop     - the structures in the hash table holding the Sysname
                       and pointer to corresponding Box or Arrow

   It provides MACRO DEFINITIONS for accessing the SysNameDrop structure; the
   below for details.

   It provides the following FUNCTIONS:
     InitSysNameHashTable() - initialize this module; must be called before
                              any other routines in this module
     AddSysName()           - add a new SysName to the hash table
     FindBoxSysName()       - find the Box with the given SysName
     FindArrowSysName()     - find the Arrow with the given SysName
*/

#ifndef SYSNAME_HASH_H
#define SYSNAME_HASH_H

#include "box.h"
#include "arrow.h"

/* TYPES ================================================================== */

typedef enum { BoxDrop=1, ArrowDrop=2
} SysNameDropType;

typedef struct sys_name_drop {
    struct sys_name_drop *next;
    SysName sys_name;
    SysNameDropType drop_type;
    union {
	Box *box_ptr;
	Arrow *arrow_ptr;
    } val;
} SysNameDrop;

/* MACRO FUNCTIONS ======================================================== */

/* Macros to access SysNameDrop's */
#define SysNameOf(_drop_ptr)         ((_drop_ptr)->sys_name)
#define SysNameDropTypeOf(_drop_ptr) ((_drop_ptr)->drop_type)
#define BoxPtrOf(_drop_ptr)          ((_drop_ptr)->val.box_ptr)
#define ArrowPtrOf(_drop_ptr)        ((_drop_ptr)->val.arrow_ptr)

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

void InitSysNameHashTable( /* void */ );
/* Initialize the sysname-hash module. This routine must be called before all
   other routines in this module.
*/

void AddSysName( /* SysName sys_name, SysNameDropType drop_type,
		    Generic *element_ptr */ );
/* Adds the Box or Arrow (depending on 'drop_type') pointed to by
   'element_ptr' to the hash table under SysName 'sys_name'.
*/

Box *FindBoxSysName( /* SysName sys_name */ );
/* RETURNS a pointer to the Box with SysName 'sys_name' or NULL if no such Box
   exists (according to the SysName hash table).
*/

Arrow *FindArrowSysName( /* SysName sys_name */ );
/* RETURNS a pointer to the Arrow with SysName 'sys_name' or NULL if no
   such Arrow exists (according to the SysName Hash Table).
*/

#ifdef DEBUG
void   ShowSysNameHashTable( /* void */ );
/* Display the current contents of the SysName hash table.
*/
#endif

#endif
