

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* BOX.H -- Declarations for types/functions related to data-structures
            dealing with boxes.

   $Header: box.h,v 1.4 90/07/24 17:11:57 heydon Locked $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   This module implements functions for managing the data structures
   associated with instance boxes. A box in this implementation has the
   following parts (types are shown in parentheses): 1) a sysname (SysName),
   2) an index (BoxIndex), 3) a role (BoxRole), 4) and indication as to
   whether the box is atomic or not (Boolean).

   A BoxIndex is a small integer assigned to a box indicating which row and
   column of a BoxRelationMatrix it represents. A BoxRole is an indication as
   to whether the box fills the role of a user box or a file box in the
   instance picture.

   Boxes are stored in simple linked list, with one linked list for each
   BoxRole. When a new box is added to the data structure, it is assumed to be
   atomic. The function AssertBoxNonAtomic() asserts that a box is non-atomic.
   The function FormAtomNonatomLists() is then used to order the boxes in the
   linked list such that all non-atomic boxes appear first in the list,
   followed by all atomic boxes. The global variables all_head[] and
   atom_head[] defined in box.g are set to point to the head of each list and
   into the middle of each list where the atomic boxes start, respectively.
   These variables are accessed with the macros AllBoxesListHead(role) and
   AtomicBoxesListHead(role), respectively. Before FormAtomNonatomLists() is
   called, AtomicBoxesListHead(role) == AllBoxesListHead(role). The global
   variables num_all_boxes[] and num_atomic_boxes[] (also defined in box.g)
   are set to the number of total and atomic boxes for each BoxRole. These
   variables are accessed with the macros NumberOfBoxes(role) and
   NumberOfAtomicBoxes(role), respectively.

   WHAT THIS MODULE PROVIDES

   This module defines the following TYPES:
     BoxIndex - index of each box (a small integer)
     SysName  - sysname of a box (or arrow)
     BoxRole  - whether the box is a user of file box
     Box      - box structure

   It provides MACRO DEFINITIONS for accessing the Box structure, and
   additionally provides the following macro definitions:
     NumRoles                  - number of BoxRole's
     FirstRole                 - number assigned to first BoxRole
     AllBoxesListHead(role)    - pointer to head of linked list of all boxes
                                 for BoxRole 'role' 
     AtomicBoxesListHead(role) - pointer to head of linked list of all atomic
                                 boxes for BoxRole 'role'
     NumberOfBoxes(role)       - total number of boxes of BoxRole 'role'
     NumberOfAtomicBoxes(role) - total number of atomic boxes of BoxRole 'role'

   It provides the following FUNCTIONS:
     InitBoxes()            - initialize this module; must be called before
                              any other routines in this module
     AddBox()               - add a new box to the data structure
     AssertBoxNonAtomic()   - assert that a specified box is non-atomic
     FormAtomNonatomLists() - reorder the linked lists of boxes so all
                              non-atomic boxes in each appears before all
		              atomic boxes

   It provides the following CONVERSION FUNCTIONS related to BOX attributes:
     ConvertBoxRole()       - convert "user" or "file" to type (BoxRole)
     ConvertBoxName()       - converts a box sysname to type (Box *)
*/

#ifndef BOX_H
#define BOX_H

#include <my-types.h>

/* TYPES ================================================================== */

typedef int BoxIndex;		/* type of the index of the box in the
				   relation array */ 
typedef int SysName;		/* type of the sysname of a box (or arrow) */

#define NumRoles 2
#define FirstRole 0
typedef enum { UserRole=0, FileRole=1, UnknownRole=2
} BoxRole;

typedef struct box_tag {
    struct box_tag *next;	/* points to next box in linked list */
    SysName b_sys_name;		/* sysname of this box */
    BoxIndex index;		/* index of this box */
    BoxRole role;		/* role filled by this box */
    Boolean atomic_p;		/* is this an atomic box? */
} Box;

/* MACRO FUNCTIONS ======================================================== */

/* Box Access Functions */
#define BSysNameOf(_box_ptr) ((_box_ptr)->b_sys_name)
#define IndexOf(_box_ptr)    ((_box_ptr)->index)
#define RoleOf(_box_ptr)     ((_box_ptr)->role)
#define AtomicPOf(_box_ptr)  ((_box_ptr)->atomic_p)

/* Macros to access heads of lists of any role */
#define AllBoxesListHead(_role)    (all_head[(int)(_role)])
#define AtomicBoxesListHead(_role) (atom_head[(int)(_role)])

/* Macros to access the number of total and atomic boxes of any role */
#define NumberOfBoxes(_role)       (num_all_boxes[(int)(_role)])
#define NumberOfAtomicBoxes(_role) (num_atomic_boxes[(int)(_role)])

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

void InitBoxes( /* void */ );
/* Initialize this module. This routine must be called before any other
   routines in this module.
*/

void AddBox( /* SysName sys_name, BoxRole role, int line_no */ );
/* Adds the box with SysName 'sys_name' and role 'role' to the sysname hash
   table AND to the head of the linked list of boxes AllBoxesListHead(role).
   Both the NumberOfBoxes(role) and NumberOfAtomicBoxes(role) are incremented.
   The parameter 'line_no' is used for error reporting, and should be the
   first line number (in the input) of the BOX entry.
*/

void AssertBoxNonAtomic( /* Box *box_ptr */ );
/* Assert that the Box *box_ptr is known to be non-atomic (i.e., it contains
   at least one other box), and decrement the number off atomic boxes having
   the same role as the Box '*box_ptr'. If the box is already atomic, no
   changes are made.
*/

void FormAtomNonatomLists( /* void */ );
/* For each BoxRole 'role', the linked list of boxes starting with
   AllBoxesListHeadRole(role) is reordered, so that all non-atomic boxes
   appear in the list before all atomic boxes. The pointer
   AtomicBoxesListHead(role) is set to point into the middle of this list at
   the first atomic box in the list.
*/

void ConvertBoxRole( /* PropVal *p_val_ptr, Generic *val_ptr, int line_no */ );
/* Assumes 'p_val_ptr' points to a PropVal with a PropValTypeOf IdPValType
   and sets *((BoxRole *)val_ptr) to UserRole or FileRole depending on whether
   IdValOf(p_val_ptr) is "user" or "file"; an error is reported, and
   *p_val_ptr is set to UnknownRole, otherwise.
*/

void ConvertBoxName( /* PropVal *p_val_ptr, Generic *val_ptr, int line_no */ );
/* Assumes 'p_val_ptr' points to a PropVal with a PropValTypeOf IntPValType
   which is to be interpretted as the SysName of a box; sets *((Box **)val_ptr)
   to be a pointer to the box with SysName IntValOf(p_val_ptr). An error is
   reported, and *p_val_ptr is set to NULL, if there is no such box in the box
   data-structure.
*/

#ifdef DEBUG
/* MACRO:
char *StringBoxRole(Boxrole role);
*/
/* Evaluates to "User", "File", or "Unknown" depending on the value of role.
*/
#endif

#ifdef DEBUG
void ShowBoxes( /* void */ );
/* Display the current state of the boxes data structure. For each role,
   information about the atomic and non-atomic boxes is displayed.
*/
#endif

#endif
