
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* BOX-RELATIONS.H -- Declarations for types/functions related to data-
                      structures dealing with relations amongst boxes.

   $Header: box-relations.h,v 1.7 90/11/01 11:18:21 heydon Locked $

   Written by Allan Heydon for the Miro project at Carnegie Mellon

   OVERVIEW OF THIS MODULE

   This module implements functions for managing the data structures that keep
   track of "geographic" relations between every pair of boxes of the same
   type (such as one box contains another, or whether they overlap).

   One of six different containment relations may exists between any pair of
   boxes A and B of the same BoxRole: 1) A is directly contained in B, 2)
   B is directly contained in A, 3) A is contained in B (but not directly),
   4) B is contained in A, 5) A and B are "equal" (as described in the next
   paragraph), and 6) no containment relation exists between A and B. It is
   assumed that relation type (6) always holds between boxes A and B if they
   have different BoxRoles.

   As of October, 1990, this module supports two different notions of box
   "equality". The first is strict equality; that is, A "equals" B iff A and B
   are the same box. The second is the criss-cross relation; that is, A
   "equals" B iff neither A contains B nor B contains A and there is some box
   which they both contain. The two different notions are realized by the
   BoxEqual type. Which form of equality you use depends on the value passed
   to the ComputeAllEqualsRelations() routine.

   The BoxRelation type is defined to represent the relation between any two
   boxes of the same type. This type represents possibilities (1), (3), (5),
   and (6) ((2) and (4) are represented by comparing B to A). Initially, the
   type-(6) relationship is assumed to exist between all pairs of boxes of the
   same type. Routines exist to assert type-(1) relations only; type-(3) and
   -(5) relations are computed by other routines.

   The relations between boxes are stored in 2D square "matrices". There is
   one matrix for the relation between every pair of user boxes, and one
   matrix for the relation between every pair of file boxes. The type
   BoxRelationMatrix is defined to represent each of these matrices. In
   addition to the 2D matrix of BoxRelations, a BoxRelationMatrix also stores
   the number of boxes whose relations are represented in the matrix.

   Each matrix entry takes on a value of type BoxRelation. A matrix is
   represented as a (column) vector of pointers to row vectors. Hence, the
   entry in row i and column j is given by matrix[i][j], where 'matrix' is
   either the user matrix or the file matrix.

   The global variable "rel_matrix" defined in box-relations.g is a size-2
   arrow of BoxRelationMatrix's. It stores the user BoxRelationMatrix in index
   0 and the file BoxRelationMatrix in index 1.

   WHAT THIS MODULE PROVIDES

   This module defines the following TYPES:
     BoxRelation       - the relation between two boxes of the same type
     BoxRelationMatrix - a structure for storing all possible n^2 relations
                         between n boxes of the same type
     BoxEqual          - determines whether equality is strict or criss-cross

   In addition to MACRO DEFINITIONS for accessing the BoxRelationMatrix
   structure (see the definitions below for details) it provides the following
   macro definitions:
      Ent(_role,_boxA,_boxB) - 
        Gives the BoxRelation between the two BoxIndexes of the given BoxRole;
	a result of DirectContainment means _boxA directly contains _boxB (and
	similarly for StarredContainment).
      DirectOrStarContains(_box_rel) -
        Evaluates non-zero iff the BoxRelation _box_rel is either
	DirectContainment or StarredContainment
      Equals(_box_rel) -
        Evaluates non-zero iff the BoxRelation _box_rel is Equal

   This module also provides the following FUNCTIONS:
     InitBoxRelations() -
       Initialize this module; must be called before any other routines in
       this module.
     AssertDirectContainmentRelation() -
       Assert that one box is directly contained in another.
     AssertParentOfContainment() -
       Assert that a particular box is the parent of all boxes in subsequent
       calls to SetChildOfContainment().
     AssertChildOfContainment() -
       Assert that a specified box is directly contained in the box most
       recently passed as an argument to SetParentOfContainment().
     CloseAllBoxContainmentRelations() -
       Compute the transitive closure for DirectContainment and
       StarredContainment of all BoxRelationMatrix's.
     ComputeAllCrissCrossesRelations() -
       Compute the CrissCross relations of all BoxRelationMatrix's.
*/

#ifndef BOX_RELATIONS_H
#define BOX_RELATIONS_H

/* TYPES ================================================================== */

typedef enum {
    NoRelation=0,
    Equal=1,
    DirectContainment=2,
    StarredContainment=3
} BoxRelation;

typedef struct {
    int num_boxes;		/* number of boxes of this role */
    BoxRelation **column_ptr;	/* pointer to column vector of matrix */
} BoxRelationMatrix;

typedef enum {
    Strict,			/* strict equality */
    Bowtie			/* criss-crosses equality */
} BoxEqual;

/* DEFINITIONS ============================================================ */

#define DirectOrStarContains(_x) ((short)(_x) >= (int)DirectContainment)
#define Equals(_x)               ((short)(_x) == (int)Equal)

/* MACRO FUNCTIONS ======================================================== */

/* BoxRelationMatrix access funtions */
#define NumBoxesOf(_matrix_ptr)  ((_matrix_ptr)->num_boxes)
#define ColumnPtrOf(_matrix_ptr) ((_matrix_ptr)->column_ptr)

/* 2D matrix access functions */
#define MatrixStructPtr(_role) (rel_matrix + ((int)(_role)))
#define Column(_role)          ColumnPtrOf(MatrixStructPtr(_role))
#define Row(_role,_row)        (Column(_role)[(_row)])
#define Ent(_role,_row,_col)   (Row(_role,_row)[(_col)])

/* GLOBAL FUNCTION DECLARATIONS =========================================== */

void InitBoxRelations( /* BoxRole role, int number_of_boxes */ );
/* Initialize data structures for the box-relations routines on boxes of type
   'role'; 'number_of_boxes' is the number of boxes having that BoxRole.
*/

void AssertDirectContainmentRelation(
       /* Box *parent_ptr, Box *child_ptr, int line_no */ );
/* This asserts that the box *parent_ptr directly contains the box *child_ptr.
   If the boxes are not of the same role, an error is reported. 'line_no' is
   the first line number (in the input) of the INSIDE entry. If you are
   asserting that several children in a row are contained in some single
   parent box, it is more efficient to use the routines two
   SetParentOfContainment() and SetChildOfContainment() below.
*/

void AssertParentOfContainment( /* Box *parent_ptr */ );
/* This sets the parent box that will be used for all subsequent calls to
   AssertChildOfContainment(). It also asserts that '*parent_ptr' is
   non-atomic.
*/

void AssertChildOfContainment( /* Box *child_ptr, int line_no */ );
/* This routine works hand-in-hand with AssertParentOfContainment(). That
   routine sets the box which will be considered the parent of the box
   referred to by '*child_ptr'; if '*child_ptr' does not have the same role as
   that box, an error is reported. This routine asserts that the current
   parent box directly contains the box of the same role with index
   'child_index'. The parameter 'line_no' is used for error reporting, and
   should be the first line number (in the input) of the INSIDE entry.
*/

void CloseAllBoxContainmentRelations( /* void */ );
/* Form reflexive-transitive closure on DirectContainment and
   StarredContainment relations for all pairs of boxes of the same BoxRole
   (for all BoxRole's).
*/

void ComputeAllEqualsRelations( /* BoxEqual eq */ );
/* Compute "equal" relations amongst all pairs of boxes of the same BoxRole
   (for all BoxRole's) according to the equality type 'eq'.
*/

#ifdef DEBUG
void      ShowContainmentAllBoxes( /* void */);
/* Display the containment information of all boxes. For each BoxRole, each
   box is considered in turn, and the indeces of those boxes either directly
   or indirectly contained in that box are listed (those indirectly contained
   are followed by "(*)").
*/
#endif

#ifdef DEBUG
void      ShowCrissCrossesAllBoxes();
/* Displays the criss-cross information of all roles of boxes. For each
   BoxRole, each box is considered in turn, and the indeces of the boxes it
   criss-crosses are listed.
*/
#endif

#endif
