

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

#include <stdio.h>
#include <string.h>		/* for strchr() */
#include <ctype.h>		/* for isspace() */
#include "vf.h"
#include "list_type.h"		/* for LoadUserLists */
#include "bin_search.h"

/* $Source: /usr/miro/src/bin/probe/RCS/vf_input.c,v $

   $Header: vf_input.c,v 1.7 91/01/11 14:07:42 ky Exp $

   $Log:	vf_input.c,v $
 * Revision 1.7  91/01/11  14:07:42  ky
 * Added empty string for linecomment_chars argument to fp2list().
 * 
 * Revision 1.6  90/12/12  07:47:43  mwm
 * fixed some comments, renamed group_line_max to line_buf_max because it
 * was being used in many more contexts
 * 
 * Revision 1.5  90/12/03  03:06:35  mwm
 * Changed handling of level number to scan for ':' (or whatever) from the
 * right, instead of the left.  This allows nested ':' characters, so long
 * as the level is also *explicitly* given.
 * 
 * Not worth documenting in the man page, as this is not a complete fix, 
 * just one to make some interfacing issues simpler.
 * 
 * Revision 1.4  90/12/01  23:32:34  mwm
 * Added support for verbose flags, changed LoadUserLists to take the UNION of
 * user lists rather than intersection, changed write_perms to capitalize the
 * permissions it writes.
 * 
 * Revision 1.3  90/11/28  08:19:01  mwm
 * Got rid of "extern int perm_len" to appease the silly RT compiler.
 * 
 * Revision 1.2  90/11/28  07:45:14  mwm
 * Added fuzzy permission handling, changed all error output to use routines
 * in vf_msg.c, fixed bug in sorting the permissions list.
 * 


/* TODO:

	Modify LoadPermissions to sort the permission names.
	Add a global variable with a vector of all permission names, and
	   have a routine that looks up membership based on common prefix,
	   rather than exact match.
*/


/* These routines read in the definitions of Group files generated by the
   prober.  There may be any number of nested tables, but each table may
   only use identifiers from the next lower table.  In UNIX, only one
   level of grouping is implemented (the file /etc/group), so what this
   means is that groups may have only userids as members; a group cannot
   be contained within a group.

	The format of the group files is as follows:

	The "#" character denotes a comment to end of line.

	Blank lines and all whitespace are ignored. (So it's okay to have
	   newlines nested into a group definition)

	<group_id> = .
	<group_id> = <lower_level_id> .
	<group_id> = <lower_level_id> , <lower_list> .

	Each id (group_id, lower_level_id) is either an alphanumeric
	   string, or an alphanumeric string followed by ":" and a number.
	   The number signifies the level of the identifier.  For example,
	   in UNIX userids would have level 0, group ids would have level
	   1.  Any ids that do not explicitly give this level will be
	   assumed to belong to the current level.  That is, all group ids
	   in one file will have the same level, and all lower level
	   identifiers will have the same level, one less than that given
	   to the group.

	Here's a sample group file:

	# Test file for the Miro' group
	# as of 9 September 1990
	#
	miro:1 = mwm:0, amy:0, heydon:0, wing:0, tygar:0 .
	csgrad87:1 = mwm:0, amy:0 .
	csgrad89:1 = .
	faculty:1 = wing:0, tygar:0,
			reynolds:0, mason:0.
	admin = root .

	Things to notice:
		- whitespace doesn't matter, it's ignored
		- numbers are optional
		- group lists may be empty
		- all lists must begin with "=" and terminate with a
		   period.

	This implementation allows the tokens "=", ",", and "." to be
   easily redefined at compile time.  However, they may only be changed to
   another single, nonalphanumeric character.  Any other change would
   require nontrivial changes.

	Some care is taken to recover from simple errors, like a missing
   comma, period, or equal sign.  But no allowances are made for any other
   numbering scheme than the one mentioned (all LHS numbers are the same,
   all RHS numbers are the same, with the former one greater than the
   latter).
*/

#define ONE_CHAR_TOKEN(x) (strchr ("=,.()", (x)))/* TRUE iff x is in the
						   string */
#define MAX_PERM_LEN 100	/* max length of permission string */
#define MAX_IDENT_LEN 100	/* max length of idents in ACLs */
#define BITS_PER_CHAR 8		/* SYSTEM DEPENDENT number of bits per
				   char, used to compute the max number of
				   permissions allowed */
#define POSSTR "POS"
#define NEGSTR "NEG"


static char *Null = NULL;
static char *tmp_ptr;
static int reuse = 0;	/* remember previous token? */
static int perm_len = 0;
static char *id_backup = NULL;
static char **perm_idents = NULL;

LoadGroupFiles (file_vector)
char *file_vector[];		/* NULL terminated list of filenames */
{
    int len, i;
    extern HashTableT **group_tables;

/* Compute number of group files */

    for (len = 0; file_vector[len]; len++)
	;

    group_tables = (HashTableT **) must_malloc ((len + 1) *
	    (int) sizeof (HashTableT *), "group hash table vector");
    group_tables[len] = NULL;

    for (i = 1; i <= len; i++) {
	FILE *fp, *fopen ();
	static char buf[20];

	sprintf (buf, "group level %d", i);
	SetFilePointer (&fp, file_vector[i - 1], buf, "r", 0);
	if (fp == NULL)
	    MustWarn ("%s: Skipping group level %d, '%s' is unreadable\n",
		    this_program, i, file_vector[i - 1]);
	else {
	    LoadGroupFile (fp, i, buf, &group_tables[i - 1]);
	    fclose (fp);
	} /* else */
    } /* for i = 1 */

    group_tables[i - 1] = NULL;
} /* LoadGroupFiles */


#define INIT_GROUP_TABLE(t) InitHashTable ((t), (char *) &Null, \
	GROUPS_IN_FILE, sizeof (char *), sizeof (GroupInfo *), strptrcmp, \
	default_str_hash, NULL)

LoadGroupFile (fp, level, desc, st)
FILE *fp;		/*   fp   must be valid */
int level;		/* expected group level */
char *desc;		/* text description of this file */
HashTableT **st;	/* symbol table for one group file */
{
    static char *store = NULL;

/* Allocate buffer storage for reading in the file */

    if (store == NULL)
	store = must_malloc (line_buf_max + 1, "group file buffer");
    store[line_buf_max] = '\0';

/* Initialize the internal hash table */

    if (!INIT_GROUP_TABLE (st))
	fprintf (stderr,
		"%s [LoadGroupFile]:  couldn't init '%s' group table\n",
		this_program, desc);
    else
	while (LoadOneGroup (fp, level, desc, *st, store, line_buf_max))
	    ;
} /* LoadGroupFile */



#define CHECK_RETVAL \
	if (retval == 0) return 0; \
	else if (store[store_size]) { \
	    fprintf (stderr, "%s [LoadOneGroup]: word too long in %s\n", \
		    this_program, desc); \
	    fprintf (stderr, "\tuse    %s -lb #   to increase buffer ", \
		    this_program); \
	    fprintf (stderr, "beyond %d\n", store_size); \
	    store[store_size] = '\0'; \
	    fprintf (stderr, "\toffending word is '%s'\n", store); \
	    exit (1); } /* if store[store_size] */

#define CHECK_LEVEL(l,store)\
	if (ptr = strrchr (store, idsep)) { \
	    if (atoi (ptr + 1) != l) \
		GroupError ("Bad level, %s should have level %d\n", \
			store, l); \
	    *ptr = '\0'; 	} /* if ptr = strchr */

#define INIT_USER_TABLE(ut) InitHashTable (&(ut), (char *) &Null, \
	USERS_IN_GROUP,	sizeof (char*), 0, strptrcmp, default_str_hash, NULL)
#define STORE_USER_TABLE(st,store,ut) (tmp_ptr = new_string ((store)), \
	Insert ((st), &tmp_ptr, &(ut)))

#define GROUP_DEFN '='
#define GROUP_SEP ','
#define GROUP_END '.'

/* LoadOneGroup -- read in one line from a group file.  Returns 0 on end
   of file, 1 otherwise.  Note that a return from this routine does *not*
   imply that any useful data were read.  On a syntax error this will skip
   the rest of the line, and still return 1.

	level -- expected level of the *group*, which is 1 greater than
	   that expected of it's constituent parts.
*/

int LoadOneGroup (fp, level, desc, st, store, store_size)
FILE *fp;
int level, store_size;
HashTableT *st;
char *store, *desc;
{
    int retval = 1;
    HashTableT *ut;		/* new hash table for this list of users */
    char *ptr, *user;

    retval =  LoadOneId (fp, store);

    CHECK_RETVAL;
    CHECK_LEVEL (level, store);

/* Create a new table to hold all of the users in this group */

    if (!INIT_USER_TABLE (ut)) {
	fprintf (stderr, "%s [LoadOneGroup]: failed to allocate user lookup ",
		this_program);
	fprintf (stderr, "tables for\n\t'%s' group in %s\n", store, desc);
	exit (1);
    } else if (!STORE_USER_TABLE (st, store, ut)) {
	fprintf (stderr, "%s [LoadOneGroup]: couldn't store user hash table");
	fprintf (stderr, " for group %s\n", store);
	exit (1);
    } else {
	user = store + strlen (store) + 1;
	if (retval = LoadOneId (fp, user))
	    if (*user != GROUP_DEFN || *(user + 1) != '\0') {
		GroupError ("%s: missing '%c' for %s in %s, skipping\n",
			this_program, GROUP_DEFN, store, desc);
		while (!feof (fp) && getc (fp) != '\n')
		    ;
	    } else {
		retval = LoadUserList (fp, level - 1, desc, ut, store,
			user, store_size);
	    } /* else */
    } /* else */

    CHECK_RETVAL;
    return retval;
} /* LoadOneGroup */


#define ADD_USER_TO_GROUP(ut,name) (tmp_ptr = new_string (name), \
	anyusers = 1, Insert ((ut), &tmp_ptr, &Null))
#define RESTORE_SEP() \
	GroupError ("Missing '%c' in %s:%d group inserted.\n", GROUP_SEP, \
		groupname, level + 1); \
	if (!ADD_USER_TO_GROUP (ut, backup)) \
	    InternalWarn ("%s: Failed to insert %s into %s!\n", \
		    this_program, backup, groupname);
#define RESTORE_DEFN() \
	GroupError ("Missing '%c' in %s:%d group inserted.\n", GROUP_END, \
		groupname, level + 1); \
	ungetc (GROUP_DEFN, fp); \
	strcpy (id_backup, backup)

#define ID 0
#define COMMA 1
#define MISSING 2

/* LoadUserList -- returns 0 on end of file, 1 on success or syntax error

	groupname -- points to start of storage buffer, which has length
	   buflen.
	buf -- points one past the end of the group name.  
*/

int LoadUserList (fp, level, desc, ut, groupname, buf, buflen)
FILE *fp;
int level, buflen;
char *desc, *groupname, *buf;
HashTableT *ut;
{
    char *ptr;		/* Used by CHECK_LEVEL macro */
    int retval = 1;
    int state = ID, anyusers = 0;
    static char *backup = NULL;

    if (backup == NULL)
	backup = must_malloc (buflen + 1, "backup group file buffer");

    while (!feof (fp) && (retval = LoadOneId (fp, buf))) {
	if (buf[1] == '\0')
	    switch (*buf) {
	        case GROUP_END:
		    if (state == MISSING) {
			RESTORE_SEP();		/* Missing comma */
		    } else if (state == ID && anyusers)
			GroupError ("Extra '%c' in %s:%d group ignored\n",
				GROUP_SEP, groupname, level + 1);
		    goto end;
		case GROUP_DEFN:
		    if (state == MISSING) {
			RESTORE_DEFN();		/* Missing period */
			reuse = 1;
		    } else {
			GroupError ("Too many '%c's in defn of %s\n", 
				GROUP_DEFN, groupname);
		    } /* else */
		    goto end;
		case GROUP_SEP:
		    if (state == MISSING) {
			RESTORE_SEP();		/* Missing comma */
		    } else if (state == ID) {
			GroupError ("Too many '%c's in defn of %s\n",
				GROUP_SEP, groupname);
			goto end;
		    }
		    state = ID;
		    continue;
	    } /* switch */
	switch (state) {
	    case ID:
		CHECK_LEVEL (level, buf);
		if (!ADD_USER_TO_GROUP (ut, buf))
		    fprintf (stderr, "%s: Failed to insert %s into %s!\n",
			    this_program, buf, groupname);
		state = COMMA;
	        break;
	    case COMMA:
	        strcpy (backup, buf);
		state = MISSING;
	        break;
	    default:
	        GroupError ("Too much punctuation missing in %s defn, %s\n",
			groupname, "skipping some");
		goto end;
	} /* switch */
    } /* while !feof */

/* If we get here, then the file ended without finishing the final group */

    switch (state) {
        case ID:    GroupError (
		    "Found '%c' instead of '%c' in %s:%d (EOF), ignored.\n",
		    GROUP_SEP, GROUP_END, groupname, level + 1);
	    break;
	case COMMA: GroupError ("Missing '%c' in %s:%d (EOF), inserted.\n",
		    GROUP_END, groupname, level + 1);
	    break;
	default:    GroupError (
		   "Missing punctuation in %s:%d (EOF), last entry ignored\n",
		    groupname, level + 1);
	    break;
    } /* switch */
end:
    if (feof (fp))
	retval = 0;
    return retval;
} /* LoadUserList */




/* This routine reads in the user ids from the prober and ambiguity
   checker, making sure to remember only the useful names.
*/

/* LoadUserLists -- read in the list of users generated by the prober and
   by the ambiguity checker.  Take the *union* of these lists, and
   save that set in the symbol table.  No point in comparing two users
   that aren't in both data sets.  Write out all userids found in exactly
   one (but not both) of these sets to the error log.

   PRE:  Input files are sorted
*/

LoadUserLists (prober, ambig)
char *prober, *ambig;
{
    FILE *pfp, *afp, *fopen ();
    int LoadOneId (), pempty = 0, aempty = 0;
    static char *pstore = NULL, *astore = NULL;
    char *new_string ();
    list_type users = nil, last = nil;

    if ((pfp = fopen (prober, "r")) == NULL) {
	fprintf (stderr,
		"%s:  Couldn't open prober users file '%s', quitting\n",
		this_program, prober);
	exit (1);
    } else if ((afp = fopen (ambig, "r")) == NULL) {
	fprintf (stderr,
		"%s:  Couldn't open ambig users file '%s', quitting\n",
		this_program, ambig);
	exit (2);
    } /* if afp = fopen */

    if (astore == NULL)
	astore = must_malloc (line_buf_max, "ambig ID line buffer");
    if (pstore == NULL)
	pstore = must_malloc (line_buf_max, "prober ID line buffer");

/* Prober EMPTY Ambig some
   Prober some  Ambig EMPTY
   Prober EMPTY Ambig EMPTY
   Prober some  Ambig some  ==> read in one each.  Skip lines beginning
	   with any character in COMMENT_CHARS.

	case p < a:  log "A missing p".  Advance p.
	case p = a:  add symbol to hash table.  Advance both.
	case p > a:  log "P missing a".  Advance a.

	If eof was reached on either side, log any remaining missing ids.
*/

#define MISSING_PID(name) (verbose ? Warn ( \
	"Prober ID %s missing in Miro' picture.\n", (name)) : 0)
#define MISSING_AID(name) (verbose ? Warn ( \
	"Miro' ID %s not found in filesystem.\n", (name)) : 0)

/* Insert this user at the *tail* of the user list */

#define ADD_USER(store) (null (users) ? \
	    (last = users = cons ((list_type) new_string (store), nil)) \
	: /* else */ \
	    (rplacd (last, cons ((list_type) new_string (store), nil)), \
	     last = cdr (last)))

    *astore = *pstore = '\0';	/* Lowest string in a lexicographic order */
    pempty = !LoadOneSortedId (pfp, pstore, prober);
    aempty = !LoadOneSortedId (afp, astore, ambig);
    if (pempty)
	Warn ("Empty user list from prober!!\n");
    if (aempty)
	Warn ("Empty user list from ambiguity checker!!\n");

    while (!pempty && !aempty) {
	int cmp = strcmp (pstore, astore);

	cmp = (cmp < 0) ? -1 : (cmp > 0) ? 1 : 0;
	switch (cmp) {
	    case -1:
		MISSING_PID (pstore);
		ADD_USER (pstore);
		pempty = !LoadOneSortedId (pfp, pstore, prober);
	        break;
	    case 0:
		ADD_USER (pstore);
		pempty = !LoadOneSortedId (pfp, pstore, prober);
		aempty = !LoadOneSortedId (afp, astore, ambig);
	        break;
	    case 1:
		MISSING_AID (astore);
		ADD_USER (astore);
		aempty = !LoadOneSortedId (afp, astore, ambig);
	        break;
	} /* switch */
    } /* while !pempty && !aempty */

    if (!pempty)
	do {
	    MISSING_PID (pstore);
	    ADD_USER (pstore);
	} while (LoadOneSortedId (pfp, pstore, prober));

    if (!aempty)
	do {
	    MISSING_AID (astore);
	    ADD_USER (astore);
	} while (LoadOneSortedId (afp, astore, prober));

    InitUidVector (users);
    free_cons (users);		/* frees up *ALL* list memory, since the
				   cell pointed to by   users   is lowest
				   on the stack. */
    fclose (pfp);
    fclose (afp);
} /* LoadUserLists */




/* InitUidVector -- To perform the file by file verification, we'll need
   to check each userid.  In this routine we place the login ids in a
   string vector, and place their indicies in a hash table.  Input is a
   linked list of userids in lexicographical order.  This routine is
   called by   LoadUserLists.   */

InitUidVector (users)
list_type users;
{
    char *must_malloc ();
    int len = length (users), i;

    uid_vector = (char **) must_malloc ((len + 1) * (int) sizeof (char *), 
	    "Vector of UIDs");
    uid_vector[len] = NULL;
    vector_len = len;
    pv = (PermVectorT *) must_malloc ((len + 1) * (int) sizeof (PermVectorT),
	    "Prober Permission vector");
    av = (PermVectorT *) must_malloc ((len + 1) * (int) sizeof (PermVectorT),
	    "Ambiguity Permission vector");
    if (!InitHashTable (&uid2index, (char *) &Null, 2 * len, sizeof (char *),
	    sizeof (int), strptrcmp, default_str_hash, NULL)) {
	fprintf (stderr,
		"%s [InitUidVector]:  Couldn't allocate hash table!\n",
		this_program);
	exit (1);
    } /* if !InitHashTable */

    for (i = 0; !null (users); i++, users = cdr (users)) {
	uid_vector[i] = (char *) car (users);

/* Insert the vector index into a hash table */
	if (!Insert (uid2index, &uid_vector[i], &i))
	    fprintf (stderr,
		    "%s: can't insert '%s' into uid hash table, no memory?\n",
		    this_program, uid_vector[i]);

    } /* for i = 0 */

    if (i != len)
	fprintf (stderr, "InitUidVector:  bad length count, %d leftovers!\n",
		len - i);
} /* InitUidVector */



/* Routines for reading in the long list of files from the prober and the
   ambiguity checker. */

LoadPermissions (perm_file)
char *perm_file;
{
    int i, pcmp ();
    list_type perms = nil, mark = cons (nil, nil);
    FILE *fp, *fopen ();
    char *ptr;
    char *buf = must_malloc (MAX_PERM_LEN + 1,
	    "Permission identifier buffer");

    buf[MAX_PERM_LEN] = '\0';
    if ((fp = fopen (perm_file, "r")) == NULL) {
        fprintf (stderr,
		"%s:  couldn't open permissions file '%s', quitting\n",
		this_program, perm_file);
	exit (1);
    } /* if fp == NULL */

    while (LoadOneId (fp, buf)) {
	if (buf[MAX_PERM_LEN]) {
	    buf[MAX_PERM_LEN] = '\0';
	    fprintf (stderr,
		    "ERROR permission string too long, truncating to '%s'\n",
		    buf);
	} /* if */
	perms = cons (new_string (buf), perms);
    } /* while */
    perm_len = length (perms);
    if (perm_len > BITS_PER_CHAR * sizeof (PermVectorT)) {
	fprintf (stderr,
	    "%s: FATAL ERROR tried to handle %d permissions, max is %d\n",
		this_program, perm_len, BITS_PER_CHAR * sizeof (PermVectorT));
	exit (1);
    } /* if len > BITS_PER_CHAR * sizeof */

/* Now copy the permissions into a vector, so we can sort them later on */

    perm_idents = (char **) must_malloc ((perm_len + 1) *
	    (int) sizeof (char *), "Vector of permission names");

    for (i = 0; i < perm_len; i++, perms = cdr (perms)) {
	ptr = (char *) car (perms);
	lower_string (ptr);
	perm_idents[i] = ptr;
    } /* for i = 0 */

    qsort (perm_idents, perm_len, sizeof (char *), strptrcmp);

/* Assign each permission a mask, according to its position.  This is done
   implicitly, by using 2^{index} as the mask value */

    free_cons (mark);
} /* LoadPermissions */




/* LoadOneFile -- reads all the permissions for one file, storing the
   filename in   file   and permissions in   vec.   Returns 0 on end of
   file, 1 otherwise. */

int LoadOneFile (fp, file, vec, desc)
FILE *fp;
char *file, *desc;
PermVectorT *vec;
{
    int c, len;
    list_type mark = cons (nil, nil);	/* Used to free up storage */
    static StackMemoryT *store = NULL;	/* memory for identifiers in ACLs */
    char *mark2;

    if (store == NULL)
	store = st_init (line_buf_max);

    mark2 = st_alloc (&store, 1);
    while (!feof (fp) && ((c = getc (fp)) == '\n' || isspace(c) || strchr
	    (COMMENT_CHARS, (char) c)))
	if (strchr (COMMENT_CHARS, (char) c))
	    while (!feof (fp) && getc (fp) != '\n')
		;

    if (feof (fp))
	return 0;

    *file = c;
    fgets (file + 1, MAXPATHLEN + 2, fp);
    if (file[MAXPATHLEN + 2]) {
	file[MAXPATHLEN + 2] = '\0';
	MustWarn ("Input filename too long!  Truncated to '%s'\n",
		file - 1);
    } /* if file[MAXPATHLEN] */
    len = strlen (file);

/* Get rid of trailing newline from fgets, plus the colon inserted by the
   prober */

    if (len && file[len - 1] == '\n')
	file[--len] = '\0';
    if (len && file[len - 1] == ':')
	file[--len] = '\0';

/* Initialize the permission vector */
    ClearPermVector (vec);

/* Now read in the access lists for each permission.  All lists are
   enclosed in parentheses, and the final list has a period following the
   close paren. */

    while (LoadACL (fp, desc, vec, &store))
	;

/* By now the permission vector has been wholly initialized, so free up
   the memory used to store the access control lists; list pointers and
   string identifiers alike. */

    free_cons (mark);
    if (!st_free (&store, mark2))
	InternalWarn ("LoadOneFile:  Couldn't free ACL list memory!\n");
    return !feof (fp);
} /* LoadOneFile */



PermVectorT find_mask (perm)
char *perm;
{
bin_search_function_declaration (bsearch);
    register int len, i;

    lower_string (perm);
    i = bsearch (perm, perm_idents, perm_len);

/* We want to allow "fuzzy" matches, so that "R" matches "read", and "r"
   matches "Read", etc.  All permissions are forced into lower case to
   facilitate this.  Assuming a fuzzy match exists, the index returned by
   bsearch might point to the right entry, or just below it, because
   searching for "r" when "read" is in the table will return the index
   below "read"'s. */

    if (i == -1 || (strncmp (perm_idents[i], perm, (len = strlen (perm))) != 0
	    && strncmp (perm_idents[i], perm, (len = strlen
	    (perm_idents[i]))) != 0)) {
	if (i < perm_len - 1 && strncmp (perm_idents[i+1], perm,
		(len = strlen (perm))) != 0 &&
		strncmp (perm_idents[i+1], perm, (len = strlen
		(perm_idents[i+1]))) != 0)
	    return 0;
	else
	    i++;
    } /* if i == -1 */

    if (i < perm_len - 1 && strncmp (perm_idents[i + 1], perm, len) == 0)
	WarnAmbigPerm (perm, perm_idents[i], perm_idents[i + 1]);
    else if (i && strncmp (perm_idents[i - 1], perm, len) == 0)
	WarnAmbigPerm (perm, perm_idents[i - 1], perm_idents[i]);
    else
	return (PermVectorT) 1 << i;

    return 0;
} /* find_mask */

/* LoadACL -- read in an access list associated with a file.  Returns 1
   after reading one, returns zero on end of file, or when all ACLs for
   one file have been read (as indicated by a '.' in the input) */

int LoadACL (fp, desc, vec, store)
FILE *fp;
char *desc;
PermVectorT *vec;
StackMemoryT **store;
{
    PermVectorT mask = 0, new = 0;
    static char *buf = NULL;
    int retval = 1;
    list_type lis = nil;

    if (buf == NULL) {
	buf = must_malloc (MAX_IDENT_LEN + 1, "ACL identifier buffer");
	buf[MAX_IDENT_LEN] = '\0';
    } /* if buf == NULL */

/* Let's assume we've got the proper format:

	Permission names separated by whitespace, terminated by a colon
	After the colon comes a list of sublists, delimited by parentheses.
	Except, that the last time through *no* permissions will be read,
	   only a period will be read, denoting the end of these
	   permissions.
*/

    while (LoadOneId (fp, buf) && *buf != '.' && *buf != ':') {
	int quit = 0;

	if (buf[MAX_IDENT_LEN]) {
	    buf[MAX_IDENT_LEN] = '\0';
	    fprintf (stderr,
		    "%s: Identifier too long in ACL, truncated to '%s'!!\n",
		    desc, buf);
	} /* if buf[MAX_IDENT_LEN] */
	if (buf[strlen (buf) - 1] == ':') {
	    buf[strlen (buf) - 1] = '\0';
	    quit = 1;
	} /* if buf[strlen (bug - 1] */

	new = find_mask (buf);
	if (new)
	    mask |= new;
	else
	    WarnNoPerm (buf, desc);

	if (quit)
	    break;
    } /* while */
    if (*buf == '.' || feof (fp))
	return 0;

/* Now we know which permissions to associate with this access list, so
   read in the list, then update the permission vector according to that
   list. */

    if (retval = LoadACLidents (fp, &lis, store))
	AddACLtoVector (lis, vec, mask);

    return retval;
} /* LoadACL */


/* LoadACLidents -- Read in an access control list from   fp.   The parsed
   list will be stored in   lisp.   All storage associated with this
   access list will be reclaimed.  */

int LoadACLidents (fp, lisp, stack)
FILE *fp;
list_type *lisp;
StackMemoryT **stack;
{
    static char *store = NULL;

    if (store == NULL)
	store = must_malloc (line_buf_max, "LoadACLidents ID buffer");

    if (fp2list (fp, lisp, store, stack, " \n\t\f\r", "(", ")", "", vf_log) > 0)
	return ValidateACLidents (*lisp);

    return 0;
} /* LoadACLidents */



/* ValidateACLidents -- This routine ensures that the access control list
   has the expected format.  We need this routine because the list parsing
   code doesn't know the exact format that we want.  Ideally, this would
   check each identifier against the appropriate hash table to make sure
   they're all available.  However, we might intentionally omit some user
   IDs from the hash table (if they're not common to both the prober and
   ambig lists), so don't get crazy with the error messages.  */

int ValidateACLidents (lis)
list_type lis;
{
    if (atom (lis)) {
	MatrixError ("Syntax error in ACL!  Atomic ACL '%s' not allowed.\n",
		lis == NULL ? "NULL" : (char *) lis);
	return 0;
    } /* if atom (lis) */

    for (; !atom (lis); lis = cdr (lis)) {
	if (!consp (car (lis))) {
	    if (strcmp (POSSTR, (char *) car (lis)) &&
		    strcmp (NEGSTR, (char *) car (lis))) {
		MatrixError (
			"Syntax error in ACL!  Expected %s or %s, got %s\n",
			POSSTR, NEGSTR, (char *) car (lis));
		return 0;
	    } /* strcmp (POSSTR */
	    if (!null (cdr (lis))) {
		MatrixError ("Syntax error in ACL!  %s not at end of list!\n",
			(char *) car (lis));
		return 0;
	    } /* if !null (cdr (lis)) */
	} else if (ValidateShortACL (car (lis), 1) == 0)
	    return 0;
    } /* for */
    return 1;
} /* ValidateACLidents */



/* ValidateShortACL -- Verify a sublist from an ACL.  Must be a list of
   identifiers, beginning with one of POSSTR or NEGSTR, containing at most
   one sublist. ASSUMPTION -- car (lis) is a cons cell.  */

int ValidateShortACL (lis, allow_sublists)
list_type lis;
int allow_sublists;
{
    if (allow_sublists && strcmp (POSSTR, car (lis)) &&
	    strcmp (NEGSTR, car (lis))) {
	MatrixError (
		"Syntax Error in ACL!  Sublist must start with %s or %s\n",
		POSSTR, NEGSTR);
	MatrixError ("\t'%s' is not allowed to start a sublist.\n", lis);
	return 0;
    } /* if strcmp (POSSTR */
    if (allow_sublists)
	lis = cdr (lis);

    for (; !atom (lis); lis = cdr (lis)) {
	if (consp (car (lis)))
	    if (allow_sublists) {
		if (ValidateShortACL (car (lis), 0) == 0)
		    return 0;
	    } else {
		MatrixError (
	    "Syntax error in ACL!  Only one level of sublists allowed!\n");
		return 0;
	    } /* else */
	else if (CheckID ((char *) car (lis)) == 0)
	    return 0;
    } /* for */

    return 1;
} /* ValidateShortACL */



/* CheckID -- Verify that   str   is a legal identifier in an access list.
   Ideally this should check it against the proper hash tables to ensure
   presence in the lists */

int CheckID (str)
char *str;
{
    return 1;
} /* CheckID */

/* AddACLtoVector -- This expands the permissions encoded in an access
   list into the full vector of user ids.  It's a shame to lose the
   concise representation, but this is really the only way to handle all
   possible cases.  Unless you want to deal with self modifying data
   structures, but the comparison function for something like that is
   likely to be horrendous. */

AddACLtoVector (lis, vec, mask)
list_type lis;
PermVectorT *vec;
PermVectorT mask;
{
    int i;
/* Iterate over all users in the permission vector */

    for (i = 0; i < vector_len; i++) {
	if (ACLperm (uid_vector[i], lis))
	    vec[i] |= mask;
    } /* for i = 0 */
} /* AddACLtoVector */



/* ACLperm -- returns TRUE iff the user named by   str   has positive
   access according to the ACL represented by   lis   */

int ACLperm (str, lis)
char *str;
list_type lis;
{
    int retval = 0;

    if (atom (lis)) {
	fprintf (stderr, "ACLperm:  FATAL error, atomic list! %x='%s'\n",
		lis, (null (lis) ? "nil" : ((lis == NULL) ? "NULL" :
		(char *) lis)));
	exit (2);
    } /* if atom (lis) */

    for (; !atom (lis); lis = cdr (lis)) {
	if (atom (car (lis))) {
	    retval = strcmp ((char *) car (lis), POSSTR) == 0;
	    goto end;
	} else {
	    int access = strcmp ((char *) caar (lis), POSSTR) == 0;

	    if (UserInACL (str, cdar (lis), 1)) {
		retval = access;
		goto end;
	    } /* if UserInACL */
	} /* else */
    } /* for */
    MatrixError ("Syntax Error in ACL!  No default case for '%s'\n",
	    str);
end:
    return retval;
} /* ACLperm */



/* UserInACL -- returns TRUE iff the user represented by   str   is found
   to match the pattern of   lis.   The parameter   or_mode   is used to
   implement the sublist checking; if the user appears in ANY of the top
   level identifiers, there is a match, but the user must appear in ALL of
   the sublist identifiers to match. */

int UserInACL (str, lis, or_mode)
char *str;
list_type lis;
int or_mode;
{
    for (; !atom (lis); lis = cdr (lis)) {
	if (consp (car (lis))) {
	    if (UserInACL (str, car (lis), or_mode ? 0 : 1)) {
		if (or_mode)
		    return 1;
	    } else if (!or_mode)
		return 0;
	} else if (UserIdentMatch (str, (char *) car (lis))) {
	    if (or_mode)
		return 1;
	} else if (!or_mode)
	    return 0;
    } /* for */
    return !or_mode;
} /* UserInACL */



/* UserIdentMatch -- compares a user id against an identifier from an
   access list.  The identifier may be a user id, or a group id, or a
   higher-level group id.  Have to compensate for, or take advantage of,
   the numeric level indicator on the identifier. */

int UserIdentMatch (str, ident)
char *str, *ident;
{
    char *ptr;
    int level = 0, retval = 0;

    if (strrchr (str, idsep)) {
	fprintf (stderr, "UserIdentMatch -- Error: '%s' shouldn't have %c0!\n",
		str, idsep);
	exit (2);
    } /* if strchr */

    if (ptr = strrchr (ident, idsep)) {
	level = atoi (ptr + 1);
	*ptr = '\0';
    } else
	level = FindBestLevel (ident);

    retval = CheckMembership (str, ident, level);
    if (ptr)
	*ptr = idsep;
    return retval;
} /* UserIdentMatch */



/* CheckMembership -- find the user id in the named group, or higher level
   group.  If it's there, return TRUE else FALSE. */

int CheckMembership (user, group, level)
char *user, *group;
int level;
{
    if (level == 0)
	return strcmp (user, group) == 0;
    else if (level == 1) {
	char *addr = Lookup (group_tables[0], (char *) &group);
	HashTableT *h;

	if (addr == NULL || Empty (addr, (char *) &Null, sizeof (char *)))

/* Group not found! */

	    return 0;
	h = *(HashTableT **) (addr + sizeof (char *));
	addr = Lookup (h, (char *) &user);
	return ! (addr == NULL || Empty (addr, (char *) &Null,
		sizeof (char *)));
    } else {
	fprintf (stderr,
	 "CheckMembership -- level %d membership not implemented, sorry!\n",
		level);
    } /* else */

    return 0;
} /* CheckMembership */


/* FindBestLevel -- ident   is known not to have an explicit level, so
   look for it in all the hash tables and return the best guess at its
   level.  For now, just assume it's a user id. */

int FindBestLevel (ident)
char *ident;
{
    return 0;
} /* FindBestLevel */

/* Common Input Utility Routines */

/* SetFilePointer -- open a file for output */

SetFilePointer (fpp, filename, type, mode, use_std_default)
FILE **fpp;
char *filename, *type, *mode;
int use_std_default;
{
    FILE *fopen ();
    char *whats_today ();

    if (fpp == NULL) {
	fprintf (stderr,
		"SetFilePointer:  null file pointer, %s file not set.\n",
		type);
	return;
    } /* if fpp == NULL */

    if (filename == NULL || *filename == '\0')
	*fpp = (*mode == 'w') ? stderr : stdin;
    else if (strcmp (filename, "-") == 0)
	*fpp = (*mode == 'w') ? stdout : stdin;
    else if ((*fpp = fopen (filename, mode)) == NULL) {
	fprintf (stderr, "%s: Couldn't open %s file '%s', using std%s\n",
		this_program, type, filename, mode == "w" ? "err" : "in");
	*fpp = (*mode == 'w') ? stderr : stdin;
    } /* if *fpp = fopen */
    if (!use_std_default && (*fpp == stderr || *fpp == stdin ||
	    *fpp == stdout)) {
	fprintf (stderr, "%s: Sorry, std%s not allowed!\n", this_program,
		((*fpp == stdin) ? "in" : (*fpp == stdout ? "out" : "err")));
	*fpp = NULL;
    } /* if !use_std_default */

} /* SetFilePointer */



/* LoadOneSortedId -- Same as LoadOneId, except that the current and
   previous strings are compared.  If they are not in lexicographic order,
   a warning message is output. */

LoadOneSortedId (fp, word, desc)
FILE *fp;
char *word, *desc;
{
    static char *store = NULL;

/* To implement the sorted order check, we'll keep the next ID in a local
   buffer */

    if (store == NULL) {
	store = must_malloc (line_buf_max + 1,
		"LoadOneSortedId line buffer");
	store[line_buf_max] = '\0';
    } /* if store == NULL */

    if (LoadOneId (fp, store) == 0)
	return 0;

    if (store[line_buf_max]) {
	store[line_buf_max] = '\0';
	fprintf (stderr, "ERROR!!  ID too long in '%s', truncated to '%s'\n",
		desc, store);
    } /* if store[line_buf_max] */

/* Compare the new ID with the old one still pointed at by the storage
   buffer parameter */

    if (strcmp (store, word) < 0)
	MustWarn (
		"WARNING!!  Input file '%s' is not sorted at word '%s'!\n",
		desc, store);
    strcpy (word, store);
    return 1;
} /* LoadOneSortedId */



/* LoadOneId -- Reads in an identifier from the the file pointer, skipping
   whitespace, blank lines, and comment-to-end-line.  Returns 0 iff no
   more tokens are available (e.g. at end of file), 1 otherwise */

int LoadOneId (fp, word)
FILE *fp;
char *word;
{
    int retval = 1;
    char *strchr ();

    if (id_backup == NULL)
        id_backup = must_malloc (line_buf_max + 1,
		"LoadOneId token buffer");

    if (word == NULL) {
	fprintf (stderr, "LoadOneId:  NULL storage argument!\n");
	retval = 0;
    } else {

/* See if we've got one in the queue */

	if (reuse) {
	    strcpy (word, id_backup);
	    reuse = 0;
	    return 1;
	} /* if reuse */

/* Skip whitespace */

	*word = '\0';
	while (!feof (fp) && (isspace (*word = getc (fp)) ||
		strchr (COMMENT_CHARS, *word)))
	    if (!isspace (*word))
		while (!feof (fp) && getc (fp) != '\n')
		    ;

	if (feof (fp))
	    retval = 0;
	else {

/* Read in the rest of the token */

	    ++word;
	    if (!ONE_CHAR_TOKEN (*(word - 1))) {
		while (!(isspace (*word = getc (fp)) || feof (fp)
			|| strchr (COMMENT_CHARS, *word) ||
			ONE_CHAR_TOKEN (*word)))
		    word++;
		if (!feof (fp))
		    if (strchr (COMMENT_CHARS, *word))
			while (!feof (fp) && getc (fp) != '\n')
			    ;
		    else if (ONE_CHAR_TOKEN (*word))
			ungetc ((int) *word, fp);
	    } /* if !ONE_CHAR_TOKEN */

	    *word = '\0';

	} /* else */
    } /* else */

    return retval;
} /* LoadOneId */


char *str_print (str, ishead, list_depth, item_num, fp)
list_type str;
int ishead, list_depth, item_num;
FILE *fp;
{
    return (char *) str;
} /* str_print */


static lower_string (str)
char *str;
{
    for (; *str; str++)
	if (isupper (*str))
	    *str = tolower (*str);
} /* lower_string */



write_perms (fp, mask)
FILE *fp;
PermVectorT mask;
{
    int total, count = 0, i;
    PermVectorT temp = mask;

/* Count the number of bits in the mask */

    for (total = 0; temp; temp &= temp - 1)
	total++;

    for (i = 0, temp = 1; i < perm_len; i++, temp <<= 1)
	if (mask & temp) {
	    putc (toupper (*perm_idents[i]), fp);
	    fprintf (fp, "%s", perm_idents[i] + 1);
	    if (count++ == total)
		break;
	    if (total - count > 1)
		fprintf (fp, ", ");
	    else if (total == count + 1)
		fprintf (fp, " and ");
	} /* if mask & temp */
} /* write_perms */



#define key_of(x) (x)
#define eq(x,y) (strcmp ((x), (y)) == 0)
#define lt(x,y) (strcmp ((x), (y)) < 0)

bin_search_function_definition (bsearch, char *, char *, key_of,lt,eq)
