
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

/* ptest.c -- 

$Header: ptest.c,v 1.4 90/12/12 07:28:06 mwm Exp $

	Program by:  Mark Maimone   24/03/90   CMU Computer Science
	Last modified:  24/03/90

$Log:	ptest.c,v $
 * Revision 1.4  90/12/12  07:28:06  mwm
 * renamed group_line_max to line_buf_max, changed interpretation of 
 * users_per_group to be the estimated number of users in a group, rather
 * than twice that number.
 * 
 * Revision 1.3  90/12/03  03:01:49  mwm
 * Changed default max_subdirs to not exceed the max_perm_stack_depth, as
 * that would have caused confusion to the user.  Now a warning (with
 * explanation) is issued when max_subdirs is too large.
 * 
 * Revision 1.2  90/12/01  23:20:55  mwm
 * Changed output files to NULL, added -dir switch to generate ACLs for
 * directories, -d switch for pruning the search at a given depth, changed
 * -nc to -c.  Now sorting the pathnames on the command line before calling
 * ProbeUNIX, to preserve global sorting characteristic, added general
 * info about probe to the help message
 * NOTE I did not attempt to remove duplicates from the command line, so if
 * the same directory is listed twice (or some subdirectory is implicitly
 * list twice) output will *not* be sorted.
 * 
 * Revision 1.1  90/11/27  23:26:31  mwm
 * Initial revision
 * 
 * Revision 1.1  90/09/11  13:35:03  mwm
 * Initial revision
 * 

*/

#include <stdio.h>
#include <mwm.h>
#include "prober.h"
#include "util.h"

#define my_entry(swit,count,type,store,size) p_entry \
	("-", (swit), P_CASE_INSENSITIVE, (count), (type), (store), (size))
#define MAX_DIRS 10
#define USAGE_MAX 1000
#define DEFAULT_USER_FILE NULL
#define DEFAULT_GROUP_FILE NULL
#define DEFAULT_SYS_USER_FILE "/etc/passwd"
#define DEFAULT_SYS_GROUP_FILE "/etc/group"
#define DEFAULT_USER_OUTPUT_FILE NULL
#define DEFAULT_GROUP_OUTPUT_FILE NULL
#define DEFAULT_MISSING_USER_FILE NULL
#define DEFAULT_STACK_DEPTH 20
#define DEFAULT_SYMBOL_TABLE_SIZE 1000
#define DEFAULT_LINE_BUF_MAX 2000
#define DEFAULT_USERS_PER_GROUP 8
#define DEFAULT_OUTPUT_FILE NULL
#define DEFAULT_DO_DIRS 0
#define DEFAULT_MAX_SUBDIRS DEFAULT_STACK_DEPTH
#define DEFAULT_COMPRESS 0
int show_help = 0;


/* INCLUDE PARSE STATE VARIABLES HERE */

char *instance_users = DEFAULT_USER_FILE;
char *instance_groups = DEFAULT_GROUP_FILE;
char *sys_users = DEFAULT_SYS_USER_FILE;
char *sys_groups = DEFAULT_SYS_GROUP_FILE;
char *user_output_filename = DEFAULT_USER_OUTPUT_FILE;
char *group_output_filename = DEFAULT_GROUP_OUTPUT_FILE;
char *missing_user_filename = DEFAULT_MISSING_USER_FILE;
int max_perm_stack_depth = DEFAULT_STACK_DEPTH;
int symbol_table_size = DEFAULT_SYMBOL_TABLE_SIZE;
int line_buf_max = DEFAULT_LINE_BUF_MAX;
int users_per_group = DEFAULT_USERS_PER_GROUP;
char *output_file = DEFAULT_OUTPUT_FILE;
int compress = DEFAULT_COMPRESS;
int max_subdirs = DEFAULT_MAX_SUBDIRS;
int do_dirs = DEFAULT_DO_DIRS;	/* Write ACLs for directories too? */

char *this_program;

arg_info table[] = {
    my_entry ("ui", P_ONE_ARG, P_OLD_FILE, &instance_users, 0),
    my_entry ("gi", P_ONE_ARG, P_OLD_FILE, &instance_groups, 0),
    my_entry ("su", P_ONE_ARG, P_OLD_FILE, &sys_users, 0),
    my_entry ("sg", P_ONE_ARG, P_OLD_FILE, &sys_groups, 0),
    my_entry ("uo", P_ONE_ARG, P_FILE, &user_output_filename, 0),
    my_entry ("go", P_ONE_ARG, P_FILE, &group_output_filename, 0),
    my_entry ("mu", P_ONE_ARG, P_FILE, &missing_user_filename, 0),
    my_entry ("sd", P_ONE_ARG, P_INT, &max_perm_stack_depth, 0),
    my_entry ("st", P_ONE_ARG, P_INT, &symbol_table_size, 0),
    my_entry ("lb", P_ONE_ARG, P_INT, &line_buf_max, 0),
    my_entry ("c", P_NO_ARGS, P_INT, &compress, !DEFAULT_COMPRESS),
    my_entry ("o", P_ONE_ARG, P_FILE, &output_file, 0),
    my_entry ("dir", P_NO_ARGS, P_INT, &do_dirs, !DEFAULT_DO_DIRS),
    my_entry ("d", P_ONE_ARG, P_INT, &max_subdirs, 0),
    my_entry ("h", P_NO_ARGS, P_INT, &show_help, 1),
    my_entry ("?", P_NO_ARGS, P_INT, &show_help, 1)

/* INCLUDE ADDITIONAL COMMAND LINE ARGUMENTS HERE */

}; /* table */
#define arg_table_size (sizeof (table) / sizeof (arg_info))


main (argc, argv)
int argc;
char *argv[];
{
    char *dirs[MAX_DIRS + 1];

    if (argv[0])
	this_program = argv[0];
    else
	this_program = "";

    if (!parse_args (argc, argv, table, arg_table_size, dirs, MAX_DIRS)) {
	fprintf (stderr, "%s:  Illegal arguments, quitting\n", this_program);
	exit (1);
    } /* if (!parse_args) */

    if (show_help) {
	do_show_help ();
	exit (0);
    } /* if (show_help) */

    if (max_subdirs > max_perm_stack_depth) {
	fprintf (stderr, "%s:  Can't have more subdirs than stack depth!\n",
		this_program);
	max_subdirs = max_perm_stack_depth;
	fprintf (stderr, "\t==> Searching at most %s subdirectories deep\n",
		max_subdirs);
	fprintf (stderr, "\t    (Use -sd to increase stack depth)\n");
    } /* if max_subdirs > max_perm_stack_depth */

    if (dirs[0] == NULL)
	fprintf (stderr, "%s: No directories given, nothing to probe.\n",
	    this_program);
    else {
	int i;

	init_cwd ();
	if (output_file)
	    if (freopen (output_file, "w", stdout) == NULL) {
		fprintf (stderr,
			"%s:  Couldn't open '%s', quitting.\n",
			this_program, output_file);
	    } /* if freopen == NULL */

/* Count the number of files/directories to probe, then sort them */

	for (i = 0; i <= MAX_DIRS && dirs[i]; i++)
	    ;
	qsort (dirs, i, sizeof (char*), strptrcmp);

/* Call the prober */

	for (i = 0; i < MAX_DIRS + 1 && dirs[i]; i++)
	    ProbeUNIX (dirs[i], instance_users, instance_groups, sys_users,
		    sys_groups);


	write_users (user_output_filename);
	write_groups (group_output_filename);
	write_missing_users (missing_user_filename);
    } /* else */
} /* main */

typedef struct {
    int *num;
    StringInfoT s;
} StrSymbolT;

typedef struct {
    int *num;
    GroupInfoT g;
} GroupSymbolT;
extern int bad_id, strptrcmp ();
extern char *int_print (), *str_print ();

write_users (filename)
char *filename;
{
    FILE *fp, *fopen ();
    StringInfoT *value;
    char *store, **ptr, *malloc ();
    int count, mask;

/* Make sure we've got enough storage, and a reasonable output file. */

    if (filename == NULL)
	return;
    if ((fp = fopen (filename, "w")) == NULL) {
	fprintf (stderr, "%s:  Couldn't open '%s' to write out users!\n",
		this_program, filename);
	return;
    } /* if fp = fopen */
    if ((store = malloc ((unsigned) (HashTableSize (un2s) + 1) *
	    sizeof (char *))) == NULL) {
	fprintf (stderr, "%s:  Couldn't malloc storage for %d users!\n",
		this_program, HashTableSize (un2s) + 1);
	return;
    } /* if store */

/* Now let's define the mask that will be used to filter out appropriate
   user ids.  If a list of users was given, then just use ID_WANTED (those
   users given).  Otherwise, write out all needed users */

    mask = instance_users ? ID_WANTED : (ID_WANTED | ID_NEEDED);

    ptr = (char **) store;

    for (HashEnumInit (un2s); HashEnumTest (un2s, NULL, &value);
	    HashEnumInc (un2s))
	if (value -> flags & mask)
	    *ptr++ = value -> name;

    *ptr = NULL;
    qsort (store, (int) (((char *) ptr - store) / sizeof (char *)),
	    sizeof (char *), strptrcmp);

    fprintf (fp, "# List of USERS found while probing\n");
    fprintf (fp, "#    on the %s\t%s\n", what_host (), whats_today ());
    fprintf (fp, "\n");
    for (ptr = (char **) store; *ptr; ptr++)
        fprintf (fp, "%s\n", *ptr);
    free (store);
    fclose (fp);
} /* write_users */



write_groups (filename)
char *filename;
{
    FILE *fp, *fopen ();
    GroupInfoT *ginfo, **ptr;
    char *store;
    int count, mask;

/* Make sure we've got enough storage, and a reasonable output file */

    if (filename == NULL)
	return;
    if ((fp = fopen (filename, "w")) == NULL) {
	fprintf (stderr, "%s:  Couldn't open '%s' to write out groups!\n",
		this_program, filename);
	return;
    } /* if fp = fopen */
    if ((store = malloc ((unsigned) (HashTableSize (gn2i) + 1) *
	    sizeof (char *))) == NULL) {
	fprintf (stderr, "%s:  Couldn't malloc storage for %d users!\n",
		this_program, HashTableSize (gn2i) + 1);
	return;
    } /* if store */

/* Now let's define the mask that will be used to filter out appropriate
   group ids.  If a list of groups was given, then just use ID_WANTED (those
   users given).  Otherwise, write out all needed users */

    mask = instance_groups ? ID_WANTED : (ID_WANTED | ID_NEEDED);

    ptr = (GroupInfoT **) store;
    for (HashEnumInit (gn2i); HashEnumTest (gn2i, NULL, (char **) &ginfo);
	    HashEnumInc (gn2i))
	if (ginfo -> flags & mask)
	    *ptr++ = ginfo;

/* HACK HACK HACK HACK HACK HACK  Here I'm relying on the fact that the
   name field is the first one in the GroupInfoT structure.  Sorry!!! */

    *ptr = NULL;
    qsort (store, (int) (((char *) ptr - store) / sizeof (GroupInfoT *)),
	    sizeof (GroupInfoT *), strptrptrcmp);

    fprintf (fp, "# List of GROUPS found while probing\n");
    fprintf (fp, "#    on the %s\t%s\n", what_host (), whats_today ());
    fprintf (fp, "\n");
    for (ptr = (GroupInfoT **) store; *ptr; ptr++) {
	HashTableT *h;
	char *unum;
	int did_one = 0;

        fprintf (fp, "   %s = ", (*ptr) -> name);

	h = (*ptr) -> users;
	if (h)
	    for (HashEnumInit (h); HashEnumTest (h, &unum, NULL);
		    HashEnumInc (h)) {
		char *s = unum2name (unum);

		if (s == NULL)
		    fprintf (stderr,
			    "write_groups:  Couldn't find usernum '%x'=%d\n",
			    unum, *(int *) unum);
		else {
		    if (did_one)
			fprintf (fp, ", ");
		    did_one = 1;
		    fprintf (fp, "%s", s);
		} /* else */
	    } /* for HashEnumInit */

	fprintf (fp, ".\n");
    } /* for ptr = (GroupInfoT **) store */
    free (store);
    fclose (fp);
} /* write_groups */



write_missing_users (filename)
char *filename;
{
    FILE *fp, *fopen ();
    StringInfoT *value;
    char *store, **ptr;
    int count, mask, oldmask;

    if (filename == NULL)
	return;
    if ((fp = fopen (filename, "w")) == NULL) {
	fprintf (stderr,
		"%s:  Couldn't open '%s' to write out missing users!\n",
		this_program, filename);
	return;
    } /* if fp = fopen */

    if ((store = malloc ((unsigned) (HashTableSize (un2s) + 1) *
	    sizeof (char *))) == NULL) {
	fprintf (stderr, "%s:  Couldn't malloc storage for %d users!\n",
		this_program, HashTableSize (un2s) + 1);
	return;
    } /* if store */

/* Now let's define the mask that will be used to filter out appropriate
   user ids.  If a list of users was given, then anything that was needed is
   missing.  Otherwise, just list those generated for groups */

    oldmask = instance_users ? ID_WANTED : (ID_WANTED | ID_NEEDED);
    mask = instance_users ? (ID_NEEDED | ID_CREATED) : ID_CREATED;

    ptr = (char **) store;
    for (HashEnumInit (un2s); HashEnumTest (un2s, NULL, &value);
	    HashEnumInc (un2s))
	if ((value -> flags & oldmask) == 0 && (value -> flags & mask))
	    *ptr++ = value -> name;

    *ptr = NULL;
    qsort (store, (int) (((char *) ptr - store) / sizeof (char *)),
	    sizeof (char *), strptrcmp);

    fprintf (fp, "# List of MISSING USERS found while probing\n");
    fprintf (fp, "#    on the %s\t%s\n", what_host (), whats_today ());
    fprintf (fp, "\n");
    for (ptr = (char **) store; *ptr; ptr++)
        fprintf (fp, "%s\n", *ptr);
    free (store);
    fclose (fp);
} /* write_missing_users */

do_show_help ()
{
    char usage[USAGE_MAX];

    fprintf (stderr, "%s:  Filesystem prober\n\n",
	    this_program);
    fprintf (stderr, "   %s generates access lists for each of the director",
	    this_program);
    fprintf (stderr, "ies or files\nnamed on the command line [up to %d], ",
	    MAX_DIRS);
    fprintf (stderr, "with subdirectories probed\nrecursively.  Output is a ");
    fprintf (stderr, "list of file permissions; an access\nlist is given ");
    fprintf (stderr, "for each permission (read, write and execute).  The\n");
    fprintf (stderr, "pathnames in the output will be lexicographically ");
    fprintf (stderr, "sorted, unless\nthe -c flag is set.\n");
    fprintf (stderr, "\n\nSWITCHES:\n");
    fprintf (stderr, "\t-c\tCOMPRESS the output%s\n",
	    DEFAULT_COMPRESS ? " [default]" : "");
    fprintf (stderr, "\t-d\tMax subdirectory DEPTH [default %d]\n",
	    DEFAULT_MAX_SUBDIRS);
    fprintf (stderr, "\t-dir\tWrite ACLs for DIRECTORIES also%s\n",
	    DEFAULT_DO_DIRS ? " [default]" : "");
    fprintf (stderr, "\t-o\tOUTPUT filename [default %s]\n",
	    DEFAULT_OUTPUT_FILE ? DEFAULT_OUTPUT_FILE : "stdout");
    fprintf (stderr, "\t-ui\tName of USER INPUT file [default \"%s\"]\n",
	    DEFAULT_USER_FILE ? DEFAULT_USER_FILE : "<none>");
    fprintf (stderr, "\t-gi\tName of GROUP INPUT file [default \"%s\"]\n",
	    DEFAULT_GROUP_FILE ? DEFAULT_GROUP_FILE : "<none>");
    fprintf (stderr, "\t-su\tName of SYSTEM's PASSWD file [default \"%s\"]\n",
	    DEFAULT_SYS_USER_FILE);
    fprintf (stderr, "\t-sg\tName of SYSTEM's GROUP file [default \"%s\"]\n",
	    DEFAULT_SYS_GROUP_FILE);
    fprintf (stderr, "\t-uo\tName of USER OUTPUT file [default \"%s\"]\n",
	    DEFAULT_USER_OUTPUT_FILE ? DEFAULT_USER_OUTPUT_FILE : "<none>");
    fprintf (stderr, "\t-go\tName of GROUP OUTPUT file [default \"%s\"]\n",
	    DEFAULT_GROUP_OUTPUT_FILE ? DEFAULT_GROUP_OUTPUT_FILE : "<none>");
    fprintf (stderr,
	    "\t-mu\tName of MISSING USERS output file [default \"%s\"]\n",
	    DEFAULT_MISSING_USER_FILE ? DEFAULT_MISSING_USER_FILE : "<none>");
    fprintf (stderr, "\n");
    fprintf (stderr,
	    "\t-sd\tMax number of DIRECTORIES in path [default %d]\n",
	    DEFAULT_STACK_DEPTH);
    fprintf (stderr,
	    "\t-st\tInitial size of SYMBOL TABLE [default %d]\n",
	    DEFAULT_SYMBOL_TABLE_SIZE);
    fprintf (stderr,
	    "\t-lb\tSize of LINE BUFFER for input files [default %d]\n",
	    DEFAULT_LINE_BUF_MAX);

    fprintf (stderr, "\t-h,?\tShow this HELP message\n");
    (void) build_usage (table, arg_table_size, usage, USAGE_MAX);
    fprintf (stderr, "\n%s\n", usage);
} /* show_help */


