/*

  iff2vf.c -- a replacement for vf.awk, which produces input for vf

  Created by Karen Kietzke (ky+@cs.cmu.edu) 1/8/91

 */

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include "iff2vf.h"

StackMemoryT *global_stack;

/* Extract the relevant information from an iff parse tree.  Returns 1 */
/* if extraction was successful, 0 otherwise.  box_list will be an */
/* array of pointers to extracted_value_t's.  The index of a sysname */
/* in this array is (sysname - (*min_sysname)), i.e. the */
/* sysnames are offset by the smallest sysname. */
int extract_iff_info (parse_tree, box_list, min_sysname, max_sysname)
Entry *parse_tree;
extracted_value_t ***box_list;
int *min_sysname;
int *max_sysname;
{
  extracted_value_t **bp, *bl;
  extracted_relation_t *inside_list, **ip, *in;
  Entry *p;
  int min_sys = -1, max_sys = 0;
  String extracted_name, extracted_type;
  int extracted_sysname, extracted_parent, extracted_child;
  ListDesc extracted_children;
  int i;
  
  /* Initialize the extract package.  We won't be trying to extract */
  /* more than 3 values at once. */
  InitExtract(3);

  /* Prepare to extract box information. */
  StartNewPNameList();
  AddPNameDesignator("sysname",True,ConvertInt,(Generic*)(&extracted_sysname),
		     NULL_LIST);
  AddPNameDesignator("name",False,ConvertString,(Generic*)(&extracted_name),
		     NULL_LIST);
  AddPNameDesignator("type",False,ConvertId,(Generic*)(&extracted_type),
		     NULL_LIST);

  /* Separate the entries into box and inside lists. */
  for (p=parse_tree,bp=(&bl),ip=(&inside_list); p!= NULL; p=p->next){
    switch (EntryTypeOf(p)){
    case BoxEntry: {
      extracted_value_t *ev;
      /* extract box information for this box */
      if (MatchPNames(p)){
	fprintf (stderr, "Warning: error extracting box information\n");
      } /* if */

      /* update min/max sysnames */
      if ((min_sys < 0) || (min_sys > extracted_sysname)){
	min_sys = extracted_sysname;
      } /* if */
      max_sys = MAX(max_sys,extracted_sysname);

      /* try to figure out what kind of box this is */
      if (extracted_type != NULL){
	int box_type = 0;
	if (!strcasecmp(extracted_type,"user")){
	  box_type = USER_TYPE;
	}else if (!strcasecmp(extracted_type,"group") ||
		  !strcasecmp(extracted_type,"world")){
	  box_type = GROUP_TYPE;
	}else if (!strcasecmp(extracted_type,"file") ||
		  !strcasecmp(extracted_type,"directory") ||
		  !strcasecmp(extracted_type,"dir")){
	  box_type = FILE_TYPE;
	}else{
	  fprintf (stderr, "Warning: unexpected type: %s\n",
		   extracted_type);
	} /* if/else */
	if (box_type){
	  /* we need to hold on to this one -- malloc some space for */
	  /* it */
	  if (!MALLOC(sizeof(extracted_value_t),(Generic**)(&ev),
		      "malloc failed in function extract_iff_info\n")){
	    *box_list = NULL;
	    return (0);
	  } /* if */

	  ev->box_type = box_type;
	  ev->sysname = extracted_sysname;
	  ev->name = extracted_name;
	  ev->parent_sysname = -1;
	  ev->fullname = NULL;
	  APPEND(ev,bp);
	} /* if (box_type) */
      } /* if (extracted_type != NULL) */
      break;
    } /* case BoxEntry */
    case InsideEntry:{
      extracted_relation_t *er;
      /* we need to save all of these */
      if (!MALLOC(sizeof(extracted_relation_t),(Generic**)(&er),
		  "malloc failed in function extract_iff_info\n")){
	*box_list = NULL;
	return (0);
      } /* if */
      er->entry = p;
      APPEND(er,ip);
      break;
    } /* case InsideEntry */
    } /* switch */
  } /* for */
  *ip = NULL;
  *bp = NULL;
  *max_sysname = max_sys;
  *min_sysname = min_sys;

  /* Now that we know the range of sysnames, make the list of boxes */
  /* into an array. */
  if (!MALLOC((((max_sys - min_sys) + 1) * sizeof(extracted_value_t*)),
	      (Generic**)(box_list),
	      "malloc failed in function extract_iff_info\n")){
    *box_list = NULL;
    return(0);
  } /* if */
  for (i=0; i <= (max_sys - min_sys); ((*box_list)[i]=NULL),i++);
  for (; bl!=NULL; bl=bl->next){
    if ((bl->sysname < min_sys) || (bl->sysname > max_sys)){
      fprintf (stderr,
	       "THIS SHOULDN'T HAPPEN!!! sysname=%d max_sys=%d min_sys=%d\n",
	       bl->sysname, max_sys, min_sys);
    }else{
      (*box_list)[bl->sysname - min_sys] = bl;
    } /* if/else */
  } /* for */

  /* process the inside entries */
  StartNewPNameList();
  AddPNameDesignator("parent",True,ConvertInt,
		     (Generic*)(&extracted_parent),NULL_LIST);
  AddPNameDesignator("children",True,ConvertInt,NULL,
		     &extracted_children);
  for (; inside_list!=NULL; inside_list=in){
    in = inside_list->next;
    if (MatchPNames(inside_list->entry)){
      fprintf (stderr, "Warning: error extracting box information\n");
    } /* if */
    while (NextListEntryPtrOf(&extracted_children) != NULL){
      if (MatchNextListElement(&extracted_children,
			       (Generic*)(&extracted_child))){
	fprintf (stderr, "Warning: error extracting box information\n");
      } /* if */
      if ((extracted_child < min_sys) || (extracted_child > max_sys)){
	fprintf (stderr, "THIS SHOULDN'T HAPPEN!!! sn=%d min=%d max=%d\n",
		 extracted_child, min_sys, max_sys);
      }else{
	/* If both parent and child are in the list, set child's */
	/* parent pointer. */
	if (((*box_list)[extracted_child - min_sys] != NULL) &&
	    ((*box_list)[extracted_parent - min_sys] != NULL)){
	  if ((*box_list)[extracted_child - min_sys]->parent_sysname != -1){
	    fprintf
	      (stderr,
	       "Warning: overriding previous parent assignment for sysname %d\n",
	       extracted_child);
	  } /* if */
	  (*box_list)[extracted_child - min_sys]->parent_sysname =
	    extracted_parent;
	} /* if */
	free (inside_list);
      } /* if/else */
    } /* while */
  } /* for */
  return (1);
} /* extract_iff_info() */


/* read_ambig_relations -- get a list of all the semantic arrows from */
/* a file containing output from the ambiguity checker.  Returns 1 if */
/* successful, 0 otherwise. */
int extract_ambig_info (ambig_fp, sem_arrow_list, min_sysname,
			max_sysname, box_list)
FILE *ambig_fp;
extracted_sem_arrow_t **sem_arrow_list;
int min_sysname, max_sysname; /* for sanity checking */
extracted_value_t **box_list; /* for sanity checking */
{
  list_type an_arrow, rest;
  char buffer[FP2LIST_BUFSIZE];
  extracted_sem_arrow_t **ap, *sv;
  String tmp_access_type, tmp_access;

  for (ap = sem_arrow_list;
       /* read an arrow "list" */
       fp2list (ambig_fp, &an_arrow, buffer, &global_stack, " \n\t\f\r", "(",
		")", ";", stderr);
       /* we shouldn't need the "list" any more */
       free_list (an_arrow)){

    /* Is this a proper list? */
    if (!consp(an_arrow)){
      fprintf (stderr, "extract_ambig_info: improper list\n");
      free_list (an_arrow);
      return (0);
    } /* if */

    /* We are only interested in semantic arrows. */
    if (strcasecmp (((String)car(an_arrow)), "sem")) continue;
    rest = cdr(an_arrow);

    /* We found one! */
    if (!MALLOC(sizeof(extracted_sem_arrow_t), (Generic**)(&sv),
		"malloc failed in function extract_ambig_info\n")){
      *sem_arrow_list = NULL;
      return (0);
    } /* if */

    sv->user_sysname = atoi((String)(car(rest)));
    rest = cdr(rest);
    if ((sv->user_sysname < min_sysname) ||
	(sv->user_sysname > max_sysname) ||
	(box_list[sv->user_sysname-min_sysname] == NULL) ||
	((box_list[sv->user_sysname-min_sysname]->box_type !=
	  USER_TYPE) &&
	 (box_list[sv->user_sysname-min_sysname]->box_type !=
	  GROUP_TYPE))){
      fprintf (stderr, "Warning: invalid user sysname %d\n", sv->user_sysname);
    } /* if */

    sv->file_sysname = atoi((String)(car(rest)));
    rest = cdr(rest);
    if ((sv->file_sysname < min_sysname) ||
	(sv->file_sysname > max_sysname) ||
	(box_list[sv->file_sysname-min_sysname] == NULL) ||
	(box_list[sv->file_sysname-min_sysname]->box_type != FILE_TYPE)){
      fprintf (stderr, "Warning: invalid file sysname %d\n", sv->file_sysname);
    } /* if */

    tmp_access_type = (String)(car(rest));
    rest = cdr(rest);
    if (*tmp_access_type != '\''){
      fprintf (stderr, "Warning: unexpected format for access type %s\n",
	       tmp_access_type);
    } /* if */
    if (!MALLOC(strlen(tmp_access_type), &(sv->access_type),
		"malloc failed in function extract_ambig_info\n")){
      *sem_arrow_list = NULL;
      return (0);
    } /* if */
    strcpy (sv->access_type, tmp_access_type+1);

    tmp_access = (String)(car(rest));
    if (!strcasecmp(tmp_access, "'pos")){
      sv->access = POSITIVE_ACCESS;
    }else if (!strcasecmp(tmp_access, "'neg")){
      sv->access = NEGATIVE_ACCESS;
    }else if (!strcasecmp(tmp_access, "'none") ||
	      !strcasecmp(tmp_access, "'ambig")){
      sv->access = NONE_ACCESS;
    }else{
      /* don't understand this one */
      fprintf (stderr, "warning: don't understand \"%s\" access\n",
	       tmp_access+1);
      free(sv);
      continue;
    }
    
    /* add this one to our list */
    APPEND(sv,ap);
  } /* for */

  /* terminate the list */
  *ap = NULL;

  /* we must have succeeded if we made it this far... */
  return (1);
} /* extract_ambig_info() */


void print_access_list (l, this_access, default_access, last_one)
list_type l;
String this_access, default_access;
int last_one;
{
  extracted_sem_arrow_t *sa;

  if (null(l)){
    if (last_one){
      printf ("(%s).\n\n", default_access);
    }else printf ("(%s)\n", default_access);
  }else{
    printf ("((%s", this_access);
    for (; !null(l); l=cdr(l)){
      printf (" %s", (String)car(l));
    }
    if (last_one){
      printf (") %s).\n\n", default_access);
    }else printf (") %s)\n", default_access);
  } /* if/else */
} /* print_access_list() */



char *this_program;

int main(argc,argv)
int argc;
char *argv[];
{
  Entry *parse_tree;
  extracted_value_t **box_list;
  int min_sysname, max_sysname, i;
  String iff_file, ambig_file, user_file;
  FILE *iff_fp, *ambig_fp, *user_fp;
  extracted_sem_arrow_t *sem_arrow_list, *sal;
  list_type pos_access = nil, neg_access = nil, marker;
  int last_file = -1;
  String last_access_type = "\0";

  /* get files from the argument list. */
  this_program = argv[0];
  if (argc != 4){
    fprintf (stderr, "usage: vfawk file.iff file.rels users_file\n");
    fprintf (stderr, "  see vf.awk(5) for more information.\n");
    exit (-1);
  } /* if */
  iff_file = argv[1];
  ambig_file = argv[2];
  user_file = argv[3];

  /* try to open the files. */
  if ((iff_fp = fopen(iff_file,"r")) == NULL){
    fprintf (stderr, "iff file %s not found\n", iff_file);
    exit (-1);
  } /* if */
  if ((ambig_fp = fopen(ambig_file,"r")) == NULL){
    fprintf (stderr, "ambig file %s not found\n", ambig_file);
    exit (-1);
  } /* if */
  if ((user_fp = fopen(user_file,"w")) == NULL){
    fprintf (stderr, "users file %s could not be opened for writing\n", user_file);
    exit (-1);
  } /* if */

  /* initialize the stack. */
  if ((global_stack = st_init(FP2LIST_BUFSIZE)) == NULL){
    fprintf (stderr, "couldn't allocate stack\n");
    exit (-1);
  }

  /* parse the iff file. */
  InitParser();
  AddEntryName("BOX", BoxEntry);
  AddEntryName("INSIDE", InsideEntry);
  AddPropName("children", InsidePName, IntListPValType);
  AddPropName("parent", InsidePName, IntPValType);
  AddPropName("sysname", BoxPName,  IntPValType);
  AddPropName("name", BoxPName, StringPValType);
  AddPropName("type", BoxPName, IdPValType);
  if ((parse_tree=ParseFile(iff_fp)) == NULL){
    fprintf (stderr, "\nparse failed!\n");
#ifdef DEBUG
  }else{
    fprintf (stderr, "\nparse succeeded\n");
#endif DEBUG
  } /* if/else */

  /* extract the relevant information from the iff file. */
  if (!extract_iff_info(parse_tree,&box_list,&min_sysname,&max_sysname)){
    fprintf (stderr, "\nextract failed!\n");
#ifdef DEBUG
  }else{
    fprintf (stderr, "\nextract succeeded\n");
#endif DEBUG
  } /* if */


  /* write the users file. */
  for (i=0; i<=(max_sysname-min_sysname); i++){
    if (box_list[i] != NULL){
      if (box_list[i]->box_type == USER_TYPE){
	fprintf (user_fp, "%s\n", box_list[i]->name);
      } /* if */
    } /* if */
  } /* for */

#ifdef DEBUG
  /* print out what we have so far, for debugging purposes. */
  for (i=0; i<=(max_sysname-min_sysname); i++){
    if (box_list[i] != NULL){
      fprintf (stderr, "s=%d ps=%d n=%s t=%d\n",
	       box_list[i]->sysname, box_list[i]->parent_sysname,
	       box_list[i]->name, box_list[i]->box_type);
    } /* if */
  } /* for */
#endif DEBUG


  /* extract the relevant information from the ambig results. */
  if (!extract_ambig_info (ambig_fp, &sem_arrow_list, min_sysname,
			   max_sysname, box_list)){
    fprintf (stderr, "attempt to read ambig results failed!\n");
#ifdef DEBUG
  }else{
    fprintf (stderr, "attempt to read ambig results succeeded!\n");
#endif DEBUG
  } /* if/else */

  /* print out what we got. */
  for (sal=sem_arrow_list; sal != NULL; sal = sal->next){
    if (box_list[sal->file_sysname-min_sysname]->fullname == NULL){
      list_type names = nil;
      list_type n;
      int b;
      String fn;
      if (!MALLOC(MAXPATHLEN, &fn, "malloc failed\n")){
	exit (-1);
      } /* if */
      *fn = NULL;
      for (b = sal->file_sysname; b != -1;
	   names = cons(box_list[b-min_sysname]->name, names),
	   b = box_list[b-min_sysname]->parent_sysname);
      for (n = names; !null(n); n = cdr(n)){
	if ((strlen(fn) + strlen ((String)car(n)) + 1) > MAXPATHLEN){
	  fprintf (stderr, "path too long\n");
	  exit (-1);
	} /* if */
	strcat(fn, (String)car(n));
      } /* for */
      free_list(names);
      box_list[sal->file_sysname-min_sysname]->fullname = fn;
    } /* if */

    /* print the filename if necessary */
    if (last_file != sal->file_sysname){
      if (last_file != -1){
	if (length(pos_access) <= length(neg_access)){
	  print_access_list(pos_access,"POS","NEG",TRUE);
	}else print_access_list(neg_access,"NEG","POS",TRUE);
	free_cons (marker);
	marker = cons(NULL,nil);
      } /* if */
      pos_access = nil;
      neg_access = nil;
      last_access_type = "\0";
      printf ("%s:\n",
	      box_list[sal->file_sysname-min_sysname]->fullname);
      last_file = sal->file_sysname;
    } /* if */

    if (strcasecmp(last_access_type, sal->access_type)){
      if (*last_access_type != NULL){
	if (length(pos_access) <= length(neg_access)){
	  print_access_list(pos_access,"POS","NEG",FALSE);
	}else print_access_list(neg_access,"NEG","POS",FALSE);
	if (last_file != -1){
	  free_cons (marker);
	  marker = cons(NULL,nil);
	} /* if */
	pos_access = nil;
	neg_access = nil;
      } /* if */
      last_access_type = sal->access_type;
      printf ("%s: ", last_access_type);
    } /* if */

    /* gather access information for this file. */
    if (sal->access == POSITIVE_ACCESS){
      pos_access = cons(box_list[sal->user_sysname-min_sysname]->name,pos_access);
    }else{
      neg_access = cons(box_list[sal->user_sysname-min_sysname]->name,neg_access);
    } /* if/else */
    
#ifdef DEBUG
    fprintf (stderr, "(sem %s(%d) %s(%d) %s %d)\n",
	     box_list[sal->user_sysname-min_sysname]->name,
	     sal->user_sysname,
	     box_list[sal->file_sysname-min_sysname]->fullname,
	     sal->file_sysname, sal->access_type, sal->access);
#endif DEBUG
  } /* for */

  /* finish the printout if needed */
  if (last_file != -1){
    if (length(pos_access) <= length(neg_access)){
      print_access_list(pos_access,"POS","NEG",TRUE);
    }else print_access_list(neg_access,"NEG","POS",TRUE);
  } /* if */
  
  /* Clean up. */
  ShutDownParser();
  exit(0);
} /* main() */
