
/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/

#include <stdio.h>
#include <sys/param.h>		/* for MAXPATHLEN */
#include <pwd.h>		/* for LoadPasswordFile */
#include <strings.h>		/* for strcpy */
#include <ctype.h>		/* for isspace, isalpha */
#include "Tables.h"
#include "util.h"		/* for must_malloc */


/* Tables.c -- UNIX System-dependent routines for initializing the four
   hash tables 

$Header: Tables.c,v 1.3 91/01/10 23:29:55 mwm Exp $

$Log:	Tables.c,v $
 * Revision 1.3  91/01/10  23:29:55  mwm
 * added comments to some variable declarations
 * 
 * Revision 1.2  90/12/12  07:23:18  mwm
 * renamed global group_line_max to line_buf_max, since it was being used in
 * many more contexts than just the group file.  Replaced a couple malloc
 * calls with must_malloc.  Changed interpretation of users_per_group to
 * be the expected number of users, rather than 2 * that number.
 * 
 * Revision 1.1  90/11/27  23:11:56  mwm
 * Initial revision
 * 
 * Revision 1.2  90/09/11  13:39:42  mwm
 * *** empty log message ***
 * 

*/

/* All characters in COMMENT_CHARS comment to end of line */

#define COMMENT_CHARS "#"

extern int symbol_table_size;	/* Initial size of user & group symbol
				   tables; there are 4 tables total (us2n,
				   un2s, gs2n, gn2i */
extern int users_per_group;	/* Initial guess at number of users per
				   group */
extern int line_buf_max;	/* size of line buffer to use for all
				   input files */

HashTableT *us2n, *un2s, *gs2n, *gn2i;
int bad_id = -1;
char *zerop = NULL;
static StringInfoT str_dummy = {NULL, 0};	/* Keep this around to
						   simplify inserts */
static GroupInfoT group_dummy = {NULL, 0, NULL};


char *new_string ();
int strptrcmp ();

/* ASSUMPTION that statics get initialized to zero */

static struct {
    char filename[MAXPATHLEN];
    HashTableT *s2n, *n2s;
} old_passwds[PASSWD_TABLES_TO_SAVE];

int intcmp (i1, i2)
int *i1, *i2;
{
    if (*i1 < *i2)
	return -1;
    else
	return (*i1 == *i2) ? 0 : 1;
} /* intcmp */

char *int_print (i, size)
int *i, size;
{
    static char store1[1000], store2[1000];
    static int which = 1;
    char *ptr;

    ptr = which ? store1 : store2;
    sprintf (ptr, "%d", *i);
    which = !which;
    return which ? store2 : store1;
} /* int_print */



char *str_print (s, size)
StringInfoT *s;
int size;
{
    static char store[1000];

    sprintf (store, "%s%s%s%s", s -> name, (s -> flags & ID_NEEDED) ? " NEED"
	    : "", (s -> flags & ID_WANTED) ? " WANT" : "",
	    (s -> flags & ID_CREATED) ? " CREATE" : "");
    return store;
} /* str_print */


char *info_print (i, size)
GroupInfoT *i;
int size;
{
    static char store[10000];
    int did_one = 0;
    int *table, a;

    strcpy (store, i -> name);
    strcat (store, ":  (");
    table = (int *) i -> users -> table;
    for (a = i -> users -> table_max; a; a--) {
	if (*table != bad_id) {
	    sprintf (store, "%s%s%d", store, (did_one ? " " : ""), *table);
	    did_one = 1;
	}
	table += 2;
    }
    strcat (store, ")");
    return store;
} /* info_print */


LoadPasswordFile (filename, s2n, n2s)
char *filename;
HashTableT **s2n, **n2s;
{
    FILE *fp, *fopen ();
    int count;
    struct passwd *e;

/* See if we've already built this table */

    for (count = 0; count < PASSWD_TABLES_TO_SAVE && old_passwds[count].s2n;
	    count++)
	if (strcmp (old_passwds[count].filename, filename) == 0 &&
		old_passwds[count].s2n == *s2n &&
		old_passwds[count].n2s == *n2s) {
	    fprintf (stderr, "%s: Reusing internal tables for %s\n",
		    this_program, filename);
	    return;
	} /* if strcmp  == 0 */

/* No, start from scratch */

    if ((fp = fopen (filename, "r")) == (FILE *) NULL) {
	fprintf (stderr, "%s:  Couldn't open '%s' for reading!\n",
		this_program, filename);
	return;
    } /* if fp = fopen (filename, "r") */
    fclose (fp);

    if (InitHashTable (s2n, (char *) &zerop, symbol_table_size, sizeof (char *),
	    sizeof (int), strptrcmp, default_str_hash, NULL) == 0 ||
	    InitHashTable (n2s, (char *) &bad_id, symbol_table_size, sizeof
	    (int), sizeof (StringInfoT), intcmp, NULL, NULL) == 0) {
	fprintf (stderr,
		"%s [LoadPasswordFile]:  InitHashTable failed for %s\n",
		this_program, filename);
	return;
    } /* if InitHashTable == 0 */

/* Save it for the next time */

    if (count < PASSWD_TABLES_TO_SAVE) {
	strcpy (old_passwds[count].filename, filename);
	old_passwds[count].s2n = *s2n;
	old_passwds[count].n2s = *n2s;
    } /* if count < PASSWD_TABLES_TO_SAVE */
    setpwfile (filename);
    setpwent ();
    count = 0;

    while (e = getpwent ()) {
	str_dummy.name = new_string (e -> pw_name);

	if (!Insert (*n2s, &(e -> pw_uid), &str_dummy)) {
	    fprintf (stderr, "%s [LoadPasswordFile]:  Insert (%d,%s) ",
		    this_program, e -> pw_uid, str_dummy.name);
	    fprintf (stderr, "failed\n");
	    break;
	} /* if (!Insert) */
	if (!Insert (*s2n, &str_dummy.name, &(e -> pw_uid))) {
	    fprintf (stderr, "%s [LoadPasswordFile]:  Insert (%s,%d) ",
		    this_program, str_dummy.name, e -> pw_uid);
	    fprintf (stderr, "failed\n");
	    break;
	} /* if (!Insert) */

	count++;
    } /* while e = getpwent () */
    endpwent ();
    fprintf (stderr, "Read %d entries from '%s'\n", count, filename);
} /* LoadPasswordFile */


LoadGroupFile (filename, s2n, n2i)
char *filename;
HashTableT **n2i, **s2n;
{
    FILE *fp, *fopen ();
    char *fgets (), *strtok ();
    static char *store = NULL;
/*ShowTable (us2n, stderr, str_print, int_print);*/
    if ((fp = fopen (filename, "r")) == (FILE *) NULL) {
	fprintf (stderr, "%s:  Couldn't open '%s' for reading!\n",
		this_program, filename);
	return;
    } /* if fp = fopen (filename, "r") */

    if (store == NULL)
	store = must_malloc (line_buf_max, "group line buffer");

    if (InitHashTable (s2n, (char *) &zerop, symbol_table_size, sizeof (char *),
	    sizeof (int), strptrcmp, default_str_hash, NULL) == 0 ||
	    InitHashTable (n2i, (char *) &bad_id, symbol_table_size,
	    sizeof (int), sizeof (GroupInfoT), intcmp, NULL, NULL) == 0) {
	fprintf (stderr, "%s [LoadGroupFile]:  InitHashTable failed for %s\n",
		this_program, filename);
	return;
    } /* if InitHashTable == 0 */

    do {
	HashTableT **user_table;
	char group[MAX_GROUP_IDENT_LEN + 1], *ptr, *this_group_ptr;
	int gid, i, count;

/* fgets behaves weirdly at the end of file.  If the end of file has a
   newline, then feof() is not true until you try to fgets() again.
   However, if there is *no* newline at the end of file, you will get an
   feof() *with* a valid string.  The way to distinguish between these two
   is to quit immediately when you get a NULL value, and only quit when
   you've got an EOF *and* you've finished parsing the final string */

	store[line_buf_max - 1] = '\0';
	if (fgets (store, line_buf_max, fp) == NULL)
	    break;
	if (store[line_buf_max - 1]) {
	    fprintf (stderr, "%s [LoadGroupFile]:  Input line too long, ",
		    this_program);
	    fprintf (stderr, "rerun with at least\n\t%s -lb %d\n",
		    this_program, line_buf_max * 2);
	    exit (12);
	} /* if store[line_buf_max - 1] */

	if (store[strlen (store) - 1] == '\n')
	    store[strlen (store) - 1] = '\0';
	group[MAX_GROUP_IDENT_LEN] = '\0';

/* The first field holds the text name of the group */

	if ((ptr = strtok (store, ":")) == NULL) {
	    fprintf (stderr, "%s:  ERROR in group file %s, missing ':'\n",
		    this_program, filename);
	    break;
	} /* if ptr = strtok (store, ":") */

	strncpy (group, ptr, MAX_GROUP_IDENT_LEN);
	if (group[MAX_GROUP_IDENT_LEN]) {
	    fprintf (stderr, "%s:  Group name '%s' too long, truncated to ",
		    this_program, store);
	    group[MAX_GROUP_IDENT_LEN] = '\0';
	    fprintf (stderr, "'%s'\n", group);
	} /* if group[MAX_GROUP_IDENT_LEN] */

/* Skip the second field, then get the gid from the third field */

	if (strtok ((char *) NULL, ":") == NULL) {
	    fprintf (stderr, "%s:  ERROR in group file %s, missing ':' #2\n",
		    this_program, filename);
	    break;
	} /* if strtok (NULL, ":") == NULL */
	if ((ptr = strtok ((char *) NULL, ":")) == NULL) {
	    fprintf (stderr, "%s:  ERROR in group file %s, missing ':' #2\n",
		    this_program, filename);
	    break;
	} /* if strtok (NULL, ":") == NULL */

	if ((gid = atoi (ptr)) == 0)
	    fprintf (stderr, "%s:  WARNING Group %s has ID '%s'=0?\n",
		    this_program, group, ptr);
	group_dummy.name = new_string (group);
	if (!Insert (*s2n, &group_dummy.name, &gid)) {
	    fprintf (stderr, "%s [LoadGroupFile]:  Insert(%s,%d) failed\n",
		    this_program, group, gid);
	    break;
	} /* if !Insert */
	if (!Insert (*n2i, &gid, &group_dummy)) {
	    fprintf (stderr, "%s [LoadGroupFile]:  Insert(%d,%s) failed\n",
		    this_program, gid, group);
	    break;
	} /* if !Insert */
	this_group_ptr = Lookup (*n2i, (char *) &gid);
	if (this_group_ptr == NULL) {
	    fprintf (stderr,
		    "%s [LoadGroupFile]:  Couldn't find user table!\n",
		    this_program);
	    break;
	} /* if this_group_ptr == NULL */
	user_table = &hash2userhash (this_group_ptr);
	    
	if (InitHashTable (user_table, (char *) &bad_id, 2 * users_per_group,
		sizeof (int), sizeof (int), intcmp, NULL, NULL) == 0) {
	    fprintf (stderr, "%s [LoadGroupFile]:  Couldn't init user table\n",
		    this_program);
	    break;
	} /* if InitHashTable == 0 */

/* From here on are loginids, separated by commas. */

	for (count = 0, i = strlen (ptr) + 1; ptr[i]; i++)
	    if (ptr[i] == ',')
		count++;

	if (count || ptr[strlen (ptr) + 1]) {
	    while (ptr = strtok ((char *) NULL, ",")) {
		int *numptr = uname2num (&ptr);

/*fprintf (stderr, "INSERTING '%s'=%d into '%s'\n", ptr, numptr, group);*/

/* If   numptr   is null, then the userid listed in /etc/group was not
   found in   /etc/passwd,   so don't include it in our group
   representation */

		if (numptr && Insert (*user_table, numptr, &count) == 0)
		    fprintf (stderr, "Couldn't insert %s into group %s\n",
			    ptr, group);
		count--;
	    } /* while ptr = strtok ((char *) NULL, ",") */
	    if (count != -1) {
		fprintf (stderr, "%s [LoadGroupFile]:  Bad count!  %d ",
			this_program, count);
		fprintf (stderr, "should be -1\n");
	    } /* if count != -1 */
/*fprintf (stderr, "GROUP '%s' MEMBERS:  ", group);
ShowTable (*user_table, stderr, int_print, int_print);*/
	} /* if count || ptr[strlen (ptr) + 1] */

    } while (!feof (fp));

    fclose (fp);
} /* LoadGroupFile */

LoadIDFile (is_group, filename)
int is_group;
char *filename;
{
    FILE *fp, *fopen ();
    char *type = is_group ? "group" : "user";
    static char *store = NULL;
    int lineno = 0;

    if (filename == NULL)
	return;
    if ((fp = fopen (filename, "r")) == NULL) {
	fprintf (stderr, "%s:  Can't open ID file '%s'\n", this_program,
		filename);
	return;
    } /* if fp = fopen */

    if (store == NULL)
	store = must_malloc (line_buf_max, "ID line buffer");

    do {
	char *ptr, *end;

	store[line_buf_max - 1] = '\0';
	if (fgets (store, line_buf_max, fp) == NULL)
	    break;
	lineno++;
	if (store[line_buf_max - 1]) {
	    fprintf (stderr, "%s [LoadIDFile]:  Input line too long, ",
		    this_program);
	    fprintf (stderr, "rerun with at least\n\t%s -lb %d\n",
		    this_program, line_buf_max * 2);
	    exit (12);
	} /* if store[line_buf_max - 1] */

	if (store[strlen (store) - 1] == '\n')
	    store[strlen (store) - 1] = '\0';

/* Get rid of comment lines and leading whitespace */

	if (ptr = strtok (store, COMMENT_CHARS))
	    *(ptr - 1) = '\0';

/* Process all ID's on this line */

	for (ptr = store; *ptr;) {
	    char *num;
	    union { GroupInfoT *g; StringInfoT *s; } p;

	    while (*ptr && isspace (*ptr))
		ptr++;
	    if (*ptr == '\0')
		break;
	    for (end = ptr + 1; *end && !isspace (*end);)
		end++;
	    if (*end)
		*end = '\0';
	    else
		end--;

/* Mark this identifier, the string pointed to by   ptr   */
/*fprintf (stderr, "ID:%s ", ptr);*/
	    num = Lookup ((is_group ? gs2n : us2n), (char *) &ptr);
/*fprintf (stderr, "num=%x *num=%x ", num, *(char**)num);*/
	    p.s = NULL;
	    if (!Empty (num, (char *) &zerop, sizeof (char *)))
		if (is_group)
		    p.g = gnum2info (hash2intinfo (num));
		else
		    p.s = unum2info (hash2intinfo (num));

	    if (p.s == NULL)
		fprintf (stderr,
			"%s: %s (line %d of \"%s\") is not a known %s id!\n",
			this_program, ptr, lineno, filename, type);
	    else if (is_group)
		MarkWant (p.g);
	    else
		MarkWant (p.s);
	    ptr = end + 1;
	} /* for ptr = store */
    } while (!feof (fp));
    fclose (fp);
} /* LoadIDFile */


