;;; OBJS.CL -- Picture object types and instantiation functions
;;;
;;; $Header: objs.cl,v 1.6 91/09/26 01:39:10 heydon Exp $
;;;
;;; Written by Allan Heydon for the Miro project at Carnegie Mellon
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

(provide 'objs)
(in-package 'objs)

(export
 '(boolean))				  ;boolean type
(export
 '(*block-array*			  ;array of object blocks
   *col-array*				  ;array of column indices/parity/subj
   *subj-array*				  ;array of subjects/parity/column
   *subj-sysname*			  ;array of subject sysnames/index
   *obj-array*))			  ;array of object HDSs/column
(export
 '(range-low range-high			  ;constraint range type accessor funcs
   syn-from syn-to syn-perm syn-parity	  ;syn structure accessor funcs
   sem-from sem-to sem-perm sem-parity	  ;sem structure accessor funcs
   subj-from subj-index			  ;subj structure accessor funcs
   obj-to obj-index obj-perm		  ;obj structure accessor funcs
   con-from con-to con-direct		  ;con structure accessor funcs
   new-box new-syn new-subj new-obj new-con)) ;instantiation funcs

(deftype boolean () '(member t nil))

(defvar *block-array*  nil "Array of semantic arrow blocks.")
(defvar *col-array*    nil "Array of column indices per subject per parity.")
(defvar *subj-array*   nil "Array of subjects per parity per column.")
(defvar *subj-sysname* nil "Array of subject sysnames per subject index.")
(defvar *obj-array*    nil "Array of objects by permission per column.")

;;; (NEW-BOX sysname "name" 'type &optional (args NIL))
;;;
;;; Create a new box having sysname SYSNAME, name NAME, and type TYPE. The
;;; ARGS is a optional list of attribute-value pairs for this particular
;;; TYPE. The attributes should be keyword attributes, and the values are
;;; evaluated, so they should be quoted as necessary. For example:
;;;   (new-box 13 "bar" 'file t :owner 'heydon :group 'miro)
;;; will create a new 'file' box with additional attributes 'owner' and
;;; 'group'.

;;; TYPES =====================================================================

;;; Constraint Ranges ---------------------------------------------------------

(deftype RangeVal () '(or fixnum nil))

(defstruct range
  (low  NIL :type RangeVal :read-only t)  ;low value (or nil)
  (high NIL :type RangeVal :read-only t)) ;high value (or nil)

;;; Arrows --------------------------------------------------------------------

(defstruct syn
  from					;sysname of tail box
  to					;sysname of head box
  perm					;permission
  parity)				;T or NIL; T iff positive parity

(defstruct sem
  from					;sysname of tail box
  to					;sysname of head box
  perm					;permission
  parity)				;T or NIL; T iff positive parity

(defstruct con
  from					;sysname of contained box
  to					;sysname of containing box
  direct)				;T or NIL; T iff containment is direct

(defstruct subj
  from					;sysname of subject
  index)				;its index into a block array

(defstruct obj
  to					;sysname of object
  index					;its index into the block table
  perm)					;the permission for that block

;;; CREATION FUNCTIONS ========================================================

;(defmacro new-box (sysname name type atomic &rest args)
;  `(,(intern (concatenate 'string "MAKE-" (string type)))
;    :sysname ,sysname :name ,name :atomic ,atomic ,@args))

(defun find-instantiator (type)
  ;; Assuming here that type is a symbol. Used to be coerced to
  ;; a string using #'string.
  (intern (concatenate 'string "MAKE-" (symbol-name type))))

(defun new-box (sysname name type atomic args)
  (declare (type fixnum sysname)
	   (type string name)
	   (type symbol type)
	   (type boolean atomic)
	   (type list args))
  (apply (find-instantiator type)
	 :sysname sysname :name name :atomic atomic args))

(defmacro new-syn (from to perm positive?)
  (declare (type fixnum from to)
	   (type symbol perm)
	   (type boolean positive?))
  `(make-syn :from ,from :to ,to :perm ,perm :parity ,positive?))

(defmacro new-con (from to direct?)
  (declare (type fixnum from to)
	   (type boolean direct?))
  `(make-con :from ,from :to ,to :direct ,direct?))

(defmacro new-subj (from index)
  (declare (type fixnum from index))
  `(make-subj :from ,from :index ,index))

(defmacro new-obj (to index perm)
  (declare (type fixnum to index)
	   (type symbol perm))
  `(make-obj :to ,to :index ,index :perm ,perm))
