;;; IPQL.CL -- Instance Picture Query Language
;;;
;;; $Header: ipql.cl,v 1.14 91/10/30 23:45:51 heydon Locked $
;;;
;;; Written by Allan Heydon for the Miro project at Carnegie Mellon
;
;/*****************************************************************************
;                Copyright Carnegie Mellon University 1992
;
;                      All Rights Reserved
;
; Permission to use, copy, modify, and distribute this software and its
; documentation for any purpose and without fee is hereby granted,
; provided that the above copyright notice appear in all copies and that
; both that copyright notice and this permission notice appear in
; supporting documentation, and that the name of CMU not be
; used in advertising or publicity pertaining to distribution of the
; software without specific, written prior permission.
;
; CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
; ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
; CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
; ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
; WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
; ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
; SOFTWARE.
;*****************************************************************************/
;

(provide 'ipql)
(in-package 'ipql)
(export
 '(*timer-on?*				;flag to turn on timer
   *out-stream*				;output stream
   new-constraint			;start a new constraint
   box syn subj obj con			;create and register each object type
   init-sem blk				;initialize and create sem arrow blks
   load-instance			;load a user-specified instance pict
   constraint-begin constraint-end	;start and end a constraint query
   thick thick* thin-range thin thin*))	;constraint query commands

(require 'flavors)
(require 'constraints)
(require 'objs)
(require 'boxtype)
(require 'hds)
(require 'iter)
(use-package 'flavors)
(use-package 'objs)
(use-package 'boxtype)
(use-package 'hds)
(use-package 'iter)

;;; IPQL OVERVIEW =============================================================
;;;
;;; This module defines functions and macros for installing new HDS
;;; structures, loading instance objects, and performing instance queries as
;;; part of an IPQL program. The functionality of the module can be roughly
;;; divided into three catagories:
;;;   1) Global variables for HDS structures
;;;   2) Registering new instance object structures into registered HDS's
;;;   3) Performing queries on registered instance objects
;;;
;;; (1) Registering New HDS Structures ----------------------------------------
;;;
;;; Four kinds of instance objects are stored in HDS structures: boxes,
;;; syntactic arrows, semantics arrows, and containment arrows. Functions for
;;; creating individual HDS structures are provided by the "hds" module.
;;; However, multiple HDS's may be required for each object type. This module
;;; provides global variables for an HDS storing each kind of constraint
;;; element; when objects are created, they are stored in the HDS bound to the
;;; appropriate global variable.
;;;
;;; (NEW-CONSTRAINT)
;;;    Clear registrations of all HDS structures. This function must be called
;;;    (usually at the start of a constraint) before any new HDS structures
;;;    for a constraint are created.

;;; (2) Registering New Instance Object Structures ----------------------------
;;;
;;; Functions are provided in the "objs" module for creating instances of each
;;; object type. This package provides functions for creating and registering
;;; those objects with the HDs bound to the appropriate global variable. The
;;; exported functions in this catagory both create the object (by calling the
;;; appropriate object-creation function in the "objs" module) *and* register
;;; that object. These operations were combined in a single function (as
;;; opposed to composing them explictly) out of space considerations. The
;;; exported functions are:
;;;
;;; (BOX sysname name type atomic? &rest args)
;;;   Create and register a new box structure; the parameters are passed to
;;;   the "objs:new-box" function.
;;;
;;; (SYN from to type positive?)
;;;   Create and register a new syn structure; the parameters are passed to
;;;   the "objs:new-syn" function.
;;;
;;; (CON from to direct?)
;;;   Create and register a new con structure; the parameters are passed to
;;;   the "objs:new-con" function.
;;;
;;; (INIT-SEM subj-cnt blk-cnt)
;;;   Initialize semantic arrow processing. SUBJ-CNT is the number of subjects
;;;   in the access matrix. BLK-CNT is the number of blocks (columns) in the
;;;   compressed access matrix. This command *must* come before any SUBJ, OBJ,
;;;   or BLK command.
;;;
;;; (SUBJ sysname index)
;;;   Declare that the subject box with sysname SYSNAME has index INDEX. Each
;;;   subject should have a different index in the range [0,SUBJ-CNT), where
;;;   SUBJ-CNT is the first value passed to INIT-SEM.
;;;
;;; (OBJ sysname index perm)
;;;   Declare that the object box with sysname SYSNAME uses the permission bit
;;;   vector with index INDEX in the compressed access matrix for permission
;;;   PERM. INDEX should be in the range [0,BLK-CNT), where BLK-CNT is the
;;;   second value passed to INIT-SEM.
;;;
;;; (BLK index bit-vector)
;;;   Declare that the INDEX'th column in the compressed access matrix is
;;;   BIT-VECTOR. INDEX should be in the range [0,BLK-CNT), where BLK-CNT is
;;;   the second value passed to INIT-SEM.
;;;
;;; (LOAD-INSTANCE [filename])
;;;   If 'filename' is not supplied, the user is prompted for a filename. The
;;;   corresponding instance file is loaded. It should contain BOX, SYN, SEM,
;;;   and CON function-calls to create and register the instance objects. When
;;;   the loading is finished, a status line is sent to the standard output
;;;   indicating how many objects of each type were loaded.

;;; (3) Performing Queries on Registered Instance Objects ---------------------
;;;
;;; The heart of the IPQL provides macros for performing queries on HDS
;;; structures. These macros must be composed in the following order:
;;;   CONSTRAINT-BEGIN THICK* THIN-RANGE THIN* CONSTRAINT-END
;;; where the (*) denotes 0 or more occurrences of that macro. The exported
;;; macros are:
;;;
;;; (CONSTRAINT-BEGIN form)
;;;   This macro should start each constraint. It executes 'form' as a
;;;   constraint, returning the value it returns.
;;;
;;; (THICK var iter form*)
;;;   Bind the variable-name 'var' to each object in the iterator 'iter', and
;;;   execture 'form*' with that binding in an implicit PROGN.
;;;
;;; (THICK* var hds path form*)
;;;   Bind the variable-name 'var' to each object in the subtree of the HDS
;;;   'hds' at the end of 'path' in that HDS, and execute 'form*' with that
;;;   binding in an implicit PROGN.
;;;
;;; (THIN-RANGE (low high) thick-obj-list form*)
;;;   Sets the thin range to ['low','high'] (where high may be specified as
;;;   NIL to indicate an infinite upper-bound). 'Thick-obj-list' should be a
;;;   list of objects comprising the "thick" part of the picture. This will
;;;   typically be a backquoted list of the evaluated variable names from each
;;;   of the preceeding THICK constructs. For example, if the variables in the
;;;   thick constructs were "box1", "arrow", and "box2", then the
;;;   'thick-obj-list' would be: `(,box1 ,arrow ,box2).
;;;
;;;   This macro initializes a counter to 0. The 'form*' are then evaluated in
;;;   a PROGN. The counter will be incremented some number of times by the
;;;   forms. The counter is then tested, to see if it is in the range. If not,
;;;   a description of the objects in 'thick-obj-list' are printed to the
;;;   standard error output, along with the erroneous count.
;;;
;;; (THIN var iter form*)
;;;   Like THICK, only returns T immediately if (PROGN 'form*') returns T for
;;;   any binding; otherwise, returns NIL.
;;;
;;; (THIN* var hds path form*)
;;;   Like THICK*, only returns T immediately if (PROGN 'form*') returns T for
;;;   any binding; otherwise, returns NIL.
;;;
;;; (CONSTRAINT-END)
;;;   This macro should appear as the inner-most form of the constaint.

;;; GLOBAL VARIABLES ==========================================================

(defvar *box-cnt*   0 "Number of Boxes.")
(defvar *syn-cnt*   0 "Number of Syntax Arrows.")
(defvar *con-cnt*   0 "Number of Containment Arrows.")
(defvar *subj-cnt*  0 "Number of Subjects.")
(defvar *obj-cnt*   0 "Number of Objects.")
(defvar *block-cnt* 0 "Number of Semantic Arrow Blocks.")

(defvar *timer-on?* nil "Flag to indicate if checking should be timed or not")
(defvar *out-stream* *standard-output* "Constraint chcker output stream")

;;; START NEW CONSTRAINT ======================================================

;;; (NEW-CONSTRAINT)
;;;
;;; Reset all 4 of the above global variables. This function should be called
;;; at the start of each constraint, before any of the HDS structures are
;;; created.
;;;
(defun new-constraint ()
  (declare (type fixnum *box-cnt* *syn-cnt* *subj-cnt* *obj-cnt* *con-cnt*))
  (setq *box-dt* nil *syn-dt* nil *subj-dt* nil *obj-dt* nil *con-dt* nil)
  (setq *box-cnt* 0 *syn-cnt* 0 *subj-cnt* 0 *obj-cnt* 0 *con-cnt* 0
	*block-cnt* 0))

;;; PICTURE-OBJECT CREATION/INSERTION FUNCTIONS ===============================

(defun box (sysname name type atomic &rest args)
  (declare (type fixnum sysname *box-cnt*)
	   (type string name)
	   (type symbol type)
	   (type boolean atomic)
	   (type list args))
  (let ((b (new-box sysname name type atomic args)))
    (incf *box-cnt*)
    (add-item *box-dt* b)))

(defun syn (from to perm positive?)
  (declare (type fixnum from to *syn-cnt*)
	   (type symbol perm)
	   (type boolean positive?))
  (let ((a (new-syn from to perm positive?)))
    (incf *syn-cnt*)
    (add-item *syn-dt* a)))

(defun con (from to direct?)
  (declare (type fixnum from to *con-cnt*)
	   (type boolean direct?))
  (let ((a (new-con from to direct?)))
    (incf *con-cnt*)
    (add-item *con-dt* a)))

(defun subj (from index)
  (declare (type fixnum from index))
  (let ((subj (new-subj from index)))
    (setf (aref *subj-sysname* index) from)
    (add-item *subj-dt* subj)))

(defun obj (to index perm)
  (declare (type fixnum to index) (type symbol perm))
  (let ((obj (new-obj to index perm)))
    (incf *obj-cnt*)
    (add-item *obj-dt* obj)
    (add-item (svref *obj-array* index) obj)))

(defun init-sem (subj-cnt blk-cnt)
  (declare (type fixnum subj-cnt blk-cnt))
  (setq *subj-cnt* subj-cnt)
  (setq *block-cnt* blk-cnt)
  (setq *block-array* (make-array blk-cnt))
  (setq *col-array*
	(make-array `(,subj-cnt 2) :element-type 'list :initial-element nil))
  (setq *subj-array*
	(make-array `(,blk-cnt 2) :element-type 'list :initial-element nil))
  (setq *subj-sysname* (make-array subj-cnt :element-type 'fixnum))
  (setq *obj-array* (make-array blk-cnt))
  (dotimes (i blk-cnt)
    (declare (type fixnum i))
    (setf (svref *obj-array* i) (make-xht #'obj-perm :size 10 :threshold 1.0))
    (extend (svref *obj-array* i) '(:NEW-SLOT) (make-bag))))

(defun blk (index data)
  (declare (type fixnum index) (type bit-vector data))
  (setf (svref *block-array* index) data)
  (dotimes (i *subj-cnt*)
    (push index (aref *col-array* i (aref data i)))
    (push (aref *subj-sysname* i) (aref *subj-array* index (aref data i)))))

;;; (LOAD-INSTANCE [filename])
;;;
(defun load-instance (&optional file-name)
  (declare (type (or string null) file-name)
	   (type fixnum *box-cnt* *syn-cnt* *subj-cnt* *obj-cnt*)
	   (type fixnum *block-cnt* *con-cnt*))
  (when (null file-name)
    (write-string "Instance picture filename? " *query-io*)
    (setq file-name (read-line *query-io*)))
  (if *timer-on?* (time (load file-name)) (load file-name))
  (format *out-stream*
    "~%Loaded ~D boxes, ~D syn-arrows, ~D con-arrows.~%"
    *box-cnt* *syn-cnt* *con-cnt*)
  (format *out-stream*
    "Loaded ~D subjs, ~D objs, ~D access blocks.~%"
    *subj-cnt* *obj-cnt* *block-cnt*))

;;; INSTANCE PICTURE QUERY CONSTRUCTS ========================================

;;; Top-Level Constructs  ----------------------------------------------------

;;; (CONSTRAINT-BEGIN form)
;;;
(defmacro constraint-begin (form)
  `(let ((.violations. 0))
    (declare (type fixnum .violations.))
    (if *timer-on?* (time (,@form)) (,@form))
    (if (> .violations. 0)
      (format *out-stream*
        "~%Instance is *not* consistent with constraint; ~
         ~D violation~:P total.~%" .violations.)
      (format *out-stream*
	"Instance *is* consistent with constraint.~%"))))

;;; (CONSTRAINT-END)
;;;
;;; Increments .count. and returns T iff count is known to be in or
;;; outside the range [.low.,.high.].
;;;
(defmacro constraint-end (high-val)
  (if high-val
    `(when (> (incf .count.) .high.) (return-from thin-block NIL))
    `(when (>= (incf .count.) .low.) (return-from thin-block T))))

;;; Query Constructs  ---------------------------------------------------------

;;; (THICK var iter form*)
;;;
(defmacro thick (var iter &rest forms)
  `(do ((,var (next-item ,iter) (next-item ,iter)))
       ((null ,var))
     (progn ,@forms)))

;;; (THICK* var hds path form*)
;;;
(defmacro thick* (var hds path &rest forms)
  (declare (type list path forms))
  `(let ((it (make-iter-on ,hds ,path)))
     (thick ,var it ,@forms)))

;;; (THIN var iter form*)
;;;
(defmacro thin (var iter &rest forms)
  `(do ((,var (next-item ,iter) (next-item ,iter)))
       ((null ,var) NIL)
     (progn ,@forms)))

;;; (THIN* var hds path form*)
;;;
(defmacro thin* (var hds path &rest forms)
  (declare (type list path forms))
  `(let ((it (make-iter-on ,hds ,path)))
     (thin ,var it ,@forms)))

;;; Counting Constructs ------------------------------------------------------

;;; (PRINT-OBJ obj)
;;;
;;; Print a description of object 'obj' to the standard error output. What
;;; gets printed depends on the type of 'obj'.Recognized types are
;;; 'objs::box', 'objs::syn', 'objs::sem', and 'objs::con'.
;;;
;;; IMPLEMENTATION NOTE: It would be cleaner if these objects were true
;;; objects instead of structures, and a method could be defined for each to
;;; print an abbreviated version to the error output (or some parameterized
;;; stream).
;;;
(defun print-obj (obj)
  (format *out-stream* "  ")
  (typecase obj
    (objs::syn (format *out-stream*
		 "SYN: parity = ~S, permission = ~S, from = ~D, to = ~D~%"
		 (syn-parity obj) (syn-perm obj) (syn-from obj) (syn-to obj)))
    (objs::sem (format *out-stream*
		 "SEM: parity = ~S, permission = ~S, from = ~D, to = ~D~%"
		 (sem-parity obj) (sem-perm obj) (sem-from obj) (sem-to obj)))
    (objs::con (format *out-stream*
		 "CON: from = ~D, to = ~D, direct = ~S~%"
		 (con-from obj) (con-to obj) (con-direct obj)))
    (otherwise (format *out-stream*
		 "BOX: sysname = ~D, name = ~S, type = ~S~%"
		 (constraints::box-sysname obj) (constraints::box-name obj)
		 (box-type obj)))))

;;; (IN-RANGE? value)
;;;
;;; Predicate that returns T iff <value> is in the range [.low.,.high.].
;;; This macro must be executed within the context of the THIN-RANGE macro.
;;;
(defmacro in-range? (val)
  (declare (type fixnum val))
  `(let ((.val. ,val))
     (and (>= .val. .low.) (or (not .high.) (<= .val. .high.)))))

;;; (THIN-RANGE (low high) 'thick-obj-list form*)
;;;
(defmacro thin-range ((low high) thick-obj-list &rest forms)
  `(let ((.low. ,low)
	 (.high. ,high)
	 (.count. 0))
    (declare (type fixnum .low. .high. .count.))
    (unless (block thin-block ,@forms (in-range? .count.))
      (format *out-stream* "~%Count = ~D~%" .count.)
      (dolist (.obj. ,thick-obj-list) (print-obj .obj.))
      (incf .violations.))))
