/*

  iff2lisp.c -- converts an iff file to a lisp file that can be loaded
                directly by the editor

  Created by Karen Kietzke (ky+@cs.cmu.edu) 1/8/91

 */

/*****************************************************************************
                Copyright Carnegie Mellon University 1992

                      All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of CMU not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 SOFTWARE.
*****************************************************************************/


#include "iff2lisp.h"
char *this_program;
int output_lisp = 1;


/* Extract the relevant information from an iff parse tree.  Returns 1 */
/* if extraction was successful, 0 otherwise.  box_list will be an */
/* array of pointers to extracted_box_value_t's.  The index of a sysname */
/* in this array is (sysname - (*min_sysname)), i.e. the */
/* sysnames are offset by the smallest sysname. */
int parse_iff_file (fp, box_list, arrow_list, inside_list, editor_info)
FILE *fp;
extracted_box_value_t **box_list;
extracted_arrow_value_t **arrow_list;
extracted_relation_t **inside_list;
extracted_editor_info_t *editor_info;
{
  extracted_box_value_t **bp;
  extracted_arrow_value_t **ap;
  extracted_relation_t **ip;
  Entry *p, *parse_tree;

  /* initialize the parse tree */
  InitParser();
  AddEntryName("BOX", BoxEntry);
  AddEntryName("ARROW", ArrowEntry);
  AddEntryName("INSIDE", InsideEntry);
  AddEntryName("EDITOR", EditorEntry);

  AddPropName("parity", AorEPNames, IdPValType);

  AddPropName("from", ArrowPName, IntPValType);
  AddPropName("kind", ArrowPName, IdPValType);
  AddPropName("permissions", ArrowPName, IdListPValType);
  AddPropName("to", ArrowPName, IntPValType);

  AddPropName("loc", BoxPName, IntListPValType);
  AddPropName("name", BoxPName, StringPValType);
  AddPropName("role", BoxPName, IdPValType);
  AddPropName("size", BoxPName, IntListPValType);
  AddPropName("type", BoxPName, IdPValType);

  AddPropName("pictype", EditorPName, IdPValType);
  AddPropName("range", EditorPName, IdPValType);

  AddPropName("children", InsidePName, IntListPValType);
  AddPropName("parent", InsidePName, IntPValType);

  AddPropName("sysname", ObjectPNames,  IntPValType);
  AddPropName("starred?", ObjectPNames, IdPValType);
  AddPropName("thickness", ObjectPNames, IdPValType);

  if ((parse_tree=ParseFile(fp)) == NULL){
    return (0);
  }				/* if */
  
  /* Separate the entries into box and inside lists. */
  for (p=parse_tree,bp=box_list,ap=arrow_list,ip=inside_list; p!= NULL;
       p=p->next){
    switch (EntryTypeOf(p)){
    case BoxEntry: {
      extracted_box_value_t *ev;
      if (!MALLOC(sizeof(extracted_box_value_t),(void**)(&ev),
		  "malloc failed in function extract_iff_info\n")){
	*box_list = NULL;
	return (0);
      }				/* if */
      ev->entry = p;
      ev->arrow = 0;
      ev->sysname = ev->x = ev->y = ev->width = ev->height = ev->thick
	= ev->starred = -1;
      ev->name = ev->box_role = ev->box_type = (char*)NULL;
      ev->parent_sysnames = ev->child_sysnames = nil;
      APPEND(ev,bp);
      break;
    }				/* case BoxEntry */
    case ArrowEntry:{
      extracted_arrow_value_t *ev;
      if (!MALLOC(sizeof(extracted_arrow_value_t),(void**)(&ev),
		  "malloc failed in function extract_iff_info\n")){
	*arrow_list = NULL;
	return (0);
      }				/* if */
      ev->entry = p;
      ev->sysname = ev->from = ev->to = ev->negative_parity =
	ev->thick = ev->starred = -1;
      ev->label = nil;
      ev->kind = (char*)NULL;
      ev->visible = 1;
      APPEND(ev,ap);
      break;
    }				/* case ArrowEntry */
    case EditorEntry:{
      if (editor_info->entry != NULL){
	fprintf (stderr,
		 "iff2lisp: warning: ignoring previous editor entry\n");
      }				/* if */
      editor_info->entry = p;
      editor_info->pictype = editor_info->range = editor_info->parity
	= (char*)NULL;
      break;
    }				/* case EditorEntry */
    case InsideEntry:{
      extracted_relation_t *er;
      if (!MALLOC(sizeof(extracted_relation_t),(void**)(&er),
		  "malloc failed in function extract_iff_info\n")){
	*box_list = NULL;
	return (0);
      }				/* if */
      er->entry = p;
      APPEND(er,ip);
      break;
    }				/* case InsideEntry */
    }				/* switch */
  } /* for */
  *ip = NULL;
  *bp = NULL;
  *ap = NULL;
  return (1);
} /* parse_iff_info() */

int print_editor_entry (fp, editor_info)
FILE *fp;
extracted_editor_info_t *editor_info;
{
  char *pictype=NULL, *range=NULL, *parity=NULL;

  /* set up the PName list */
  StartNewPNameList();
  AddPNameDesignator("pictype",False,ConvertId,(void*)&pictype,NULL_LIST);
  AddPNameDesignator("range",False,ConvertId,(void*)&range,NULL_LIST);
  AddPNameDesignator("parity",False,ConvertId,(void*)&parity,NULL_LIST);

  /* try to match the editor entry against the PName list */
  if (editor_info->entry != NULL){
    if (MatchPNames(editor_info->entry)){
      fprintf (stderr, "Invalid editor entry\n");
      return (0);
    }				/* if */
  }				/* if */

  /* print the results */
  fprintf (fp, "\n %c ===== Editor Entry =====\n",
	   (output_lisp ? ';' : '#'));
  if (!output_lisp) fputs (">EDITOR\n", fp);
  if (pictype != NULL){
    if (!strcasecmp(pictype,"instance") &&
	!strcasecmp(pictype,"constraint")){
      fprintf (stderr, "unrecognized value for pictype: %s", pictype);
      fprintf (stderr, " using default value: instance\n");
      pictype = "instance";
    }				/* if */
  }else pictype = "instance";
  editor_info->pictype = pictype;
  if (output_lisp){
    fprintf (fp, " (s-value *a* :pictype :%s)\n", pictype);
  }else fprintf (fp, " pictype = %s;\n", pictype);
  
  if (range != NULL){
    editor_info->range = range;
    if (output_lisp){
      fprintf (fp, " (s-value *a* :range :%s)\n", range);
    }else fprintf (fp, " range = %s;\n", range);
  }				/* if */
  
  if (parity != NULL){
    if (!strcasecmp(pictype,"neg") && !strcasecmp(pictype,"pos")){
      fprintf (stderr, "unrecognized value for parity: %s", parity);
      fprintf (stderr, " Not specifying parity\n");
      parity = NULL;
    }else{
      if (output_lisp){
	fprintf (fp, " (s-value *a* :parity :%s)\n", parity);
      }else fprintf (fp, " parity = %s;\n", parity);
    }				/* if/else */
  }				/* if */
  editor_info->parity = parity;

  return (1);
}				/* print_editor_entry */

int extract_boxes (box_list, box_array, min_box_sysname,
		   max_box_sysname, editor_info)
extracted_box_value_t *box_list;
extracted_box_value_t ***box_array;
int *min_box_sysname, *max_box_sysname;
extracted_editor_info_t *editor_info;
{
  int sysname, min_sysname, max_sysname, inst_pic, i;
  char *name, *role, *type, *thickness, *starred;
  ListDesc size, loc;
  extracted_box_value_t *bl;

  inst_pic = strcasecmp(editor_info->pictype,"constraint");

  /* set up the PName list */
  StartNewPNameList();
  AddPNameDesignator("sysname",True,ConvertInt,(void*)&sysname,NULL_LIST);
  AddPNameDesignator("name",False,ConvertString,(void*)&name,NULL_LIST);
  AddPNameDesignator("loc",False,ConvertInt,NULL,&loc);
  AddPNameDesignator("size",False,ConvertInt,NULL,&size);
  AddPNameDesignator("role",False,ConvertId,(void*)&role,NULL_LIST);
  AddPNameDesignator("type",False,ConvertId,(void*)&type,NULL_LIST);
  AddPNameDesignator("thickness",False,ConvertId,(void*)&thickness,NULL_LIST);
  AddPNameDesignator("starred?",False,ConvertId,(void*)&starred,NULL_LIST);
  
  /* walk through the list and extract information */
  for (bl=box_list; bl; bl=bl->next){
    if (MatchPNames(bl->entry)){
      fprintf (stderr, "invalid box entry\n");
      return (0);
    }				/* if */

    bl->sysname = sysname;

    if (ValFoundP("name")) bl->name = name; else bl->name = "";

    if (ValFoundP("loc") && ValFoundP("size")){
      if (NextListEntryPtrOf(&loc) != NULL)
	if (MatchNextListElement(&loc,(int*)&(bl->x))) return(0);
      if (NextListEntryPtrOf(&loc) != NULL)
	if (MatchNextListElement(&loc,(int*)&(bl->y))) return(0);
      if (NextListEntryPtrOf(&size) != NULL)
	if (MatchNextListElement(&size,(int*)&(bl->width))) return(0);
      if (NextListEntryPtrOf(&size) != NULL)
	if (MatchNextListElement(&size,(int*)&(bl->height))) return (0);
      if ((bl->y < 0) ||	/* x<0 -> y<0 */
	  (bl->height < 0) ||
	  (NextListEntryPtrOf(&loc) != NULL) ||
	  (NextListEntryPtrOf(&size) != NULL)){
	fprintf (stderr, "Warning: problem with size and/or loc for\n");
	fprintf (stderr, "         sysname %d -- ignoring size and loc\n",
		 sysname);
	bl->x = bl->y = bl->width = bl->height = -1;
      }				/* if */
    }				/* if */

    if (ValFoundP("role")){
      if (strcasecmp(role,"user") && strcasecmp(role,"file") &&
	  strcasecmp(role,"unknown")){
	fprintf (stderr, "unrecognized value for role in sysname %d: %s\n",
		 sysname, role);
	fprintf (stderr, "using default value: unknown\n");
	role = "unknown";
      }
    }else role = "unknown";
    bl->box_role = role;
    
    if (ValFoundP("type")) bl->box_type = type; else bl->box_type = "unknown";
    
    if (ValFoundP("thickness")){
      if (inst_pic){
	fprintf (stderr,
		 "ignoring thickness for sysname %d in instance picture\n",
		 sysname);
      }else if (!strcasecmp(thickness, "thick")){
	bl->thick = 1;
      }else if (!strcasecmp(thickness, "thin")){
	bl->thick = 0;
      }else{
	fprintf (stderr, "unexpected value for thickness in sysname %d: %s\n",
		 sysname, thickness);
	fprintf (stderr, "using default value: thin\n");
	bl->thick = 0;
      }				/* if/else */
    }else if (!inst_pic) bl->thick = 0;

    if (ValFoundP("starred?")){
      if (inst_pic){
	fprintf (stderr,
		 "ignoring starred? for sysname %d in instance picture\n",
		 sysname);
      }else if (!strcasecmp(starred,"true")){
	bl->starred = 1;
      }else if (!strcasecmp(starred,"false")){
	bl->starred = 0;
      }else{
	fprintf (stderr, "unexpected value for starred? in sysname %d: %s\n",
		 sysname, starred);
	fprintf (stderr, "using default value: false\n");
	bl->starred = 0;
      }				/* if/else */
    }else if (!inst_pic) bl->starred = 0;
  }				/* for */

  /* find the range of sysnames */
  for (bl=box_list, max_sysname = min_sysname = (bl ? bl->sysname : 0);
       bl != NULL;
       max_sysname = MAX(max_sysname,bl->sysname),
       min_sysname = MIN(min_sysname,bl->sysname),
       bl=bl->next);

  /* put the boxes into an array for easy access */

  /* malloc enough space to hold pointers to all the boxes */
  if (!MALLOC((((max_sysname - min_sysname) + 1) *
	       sizeof(extracted_box_value_t*)), (void*)box_array,
	      "malloc failed in extract_boxes()")) return(0);

  /* initialize the array */
  for (i=0; i<=(max_sysname-min_sysname); (*box_array)[i++]=NULL);
  for (bl=box_list; bl!=NULL;
       (*box_array)[bl->sysname-min_sysname] = bl,
       bl=bl->next);

  /* pass back the minimum and maximum sysnames */
  *min_box_sysname = min_sysname;
  *max_box_sysname = max_sysname;

  return (1);
}				/* extract_boxes */

int process_inside_entries (inside_list, box_array, min_sysname,
			    max_sysname)
extracted_relation_t *inside_list;
extracted_box_value_t **box_array;
int min_sysname, max_sysname;
{
  int parent, child;
  ListDesc child_list;

  /* set up the PName list */
  StartNewPNameList();
  AddPNameDesignator("parent",True,ConvertInt,(void*)&parent,NULL_LIST);
  AddPNameDesignator("children",True,ConvertInt,NULL,&child_list);

  /* read all the inside entries */
  for (; inside_list!=NULL; inside_list=inside_list->next){
    if (MatchPNames(inside_list->entry)) return(0);
    if ((parent < min_sysname) || (parent > max_sysname)){
      fprintf (stderr, "invalid parent %d in inside entry\n", parent);
      return(0);
    }else while (NextListEntryPtrOf(&child_list) != NULL){
      if (MatchNextListElement(&child_list,(void*)&child)) return(0);
      if ((child < min_sysname) || (child > max_sysname)){
	fprintf (stderr, "invalid child %d in inside entry\n", child);
	return(0);
      }				/* if */
      if ((box_array[parent-min_sysname] == NULL) ||
	  (box_array[child-min_sysname] == NULL)){
	fprintf (stderr, "problem with inside entry: box does not exist\n");
	return (0);
      }else{
	box_array[parent-min_sysname]->child_sysnames =
	  cons(child,box_array[parent-min_sysname]->child_sysnames);
	box_array[child-min_sysname]->parent_sysnames =
	  cons(parent,box_array[child-min_sysname]->parent_sysnames);
      }				/* if/else */
    }				/* while */
  }				/* for */
  return(1);
}				/* process_inside_entries */

int find_or_add_to_string_table(table, size, the_string)
char ***table, *the_string;
int *size;
{
  int i;

  /* find the label in the string table */
  for (i=0; (*table != NULL) && ((*table)[i] != NULL) &&
       strcasecmp(the_string,(*table)[i]); i++);

  /* malloc more space if we need it */
  if ((*table == NULL) || (i == *size - 1)){
    int j;
    j = (*table == NULL) ? 0 : i+1;
    *size *= 2;
    if (*size == 0) *size = 4;
    if (!REALLOC(*size,table,
                 "realloc failed in find_or_add_to_string_table()")){
      return (-1);
    } /* if */
    for (; j < *size; (*table)[j++] = NULL);
  } /* if */
  
  /* put the string in the table if we didn't find it */
  if ((*table)[i] == NULL) (*table)[i] = the_string;
  return (i);
} /* find_or_add_to_string_table() */

int extract_arrows (arrow_list, editor_info, box_array, min_sysname,
		    max_sysname, arrow_labels, arrow_labels_size)
extracted_arrow_value_t *arrow_list;
extracted_editor_info_t *editor_info;
extracted_box_value_t **box_array;
int min_sysname, max_sysname;
char ***arrow_labels;
int *arrow_labels_size;
{
  int sysname, inst_pic, from, to;
  char *type, *parity, *kind, *thickness, *starred;
  ListDesc type_list;
  extracted_arrow_value_t *al;

  *arrow_labels_size = 0;
  *arrow_labels = NULL;
  inst_pic = strcasecmp(editor_info->pictype,"constraint");

  /* Set up the PName list. */
  StartNewPNameList();
  AddPNameDesignator("sysname",True,ConvertInt,(void*)&sysname,NULL_LIST);
  AddPNameDesignator("from",True,ConvertInt,(void*)&from,NULL_LIST);
  AddPNameDesignator("to",True,ConvertInt,(void*)&to,NULL_LIST);
  AddPNameDesignator("permissions",False,ConvertId,(void*)&type,&type_list);
  AddPNameDesignator("parity",False,ConvertId,(void*)&parity,NULL_LIST);
  AddPNameDesignator("kind",False,ConvertId,(void*)&kind,NULL_LIST);
  AddPNameDesignator("thickness",False,ConvertId,(void*)&thickness,NULL_LIST);
  AddPNameDesignator("starred?",False,ConvertId,(void*)&starred,NULL_LIST);

  /* Walk through the list and extract information... */
  for (al=arrow_list; al; al=al->next){
    if (MatchPNames(al->entry)){
      fprintf (stderr, "invalid arrow entry\n");
      return (0);
    }				/* if */

    al->sysname = sysname;
    al->from = from;
    al->to = to;
    box_array[from-min_sysname]->arrow = 1;
    box_array[to-min_sysname]->arrow = 1;

    if (ValFoundP("permissions")){
      if (MatchedPOf(&type_list)){
	while (NextListEntryPtrOf(&type_list)){
	  MatchNextListElement(&type_list,(void*)&type);
	  al->label =
	    cons (find_or_add_to_string_table(arrow_labels,
		                              arrow_labels_size,
					      type), al->label);
          if ((int)car(al->label) == -1) return(0);
	}			/* while */
	al->label = reverse (al->label);
      }else al->label =
	cons(find_or_add_to_string_table(arrow_labels,
					 arrow_labels_size,
					 type), nil);
    }				/* if */

    if (ValFoundP("parity")){
      if (!strcasecmp(parity,"neg")){
	al->negative_parity = 1;
      }else if (!strcasecmp(parity,"pos")){
	al->negative_parity = 0;
      }else{
	fprintf (stderr, "unrecognized parity in sysname %d: %s\n",
		 sysname, parity);
	fprintf (stderr, "using default parity: pos\n");
	al->negative_parity = 0;
      }				/* if/else */
    }else al->negative_parity = 0;

    if (ValFoundP("kind")){
      if (inst_pic){
	fprintf (stderr, "ignoring kind for sysname %d in instance picture\n",
		 sysname);
      }else if (strcasecmp(kind,"syn") && strcasecmp(kind,"sem") &&
		strcasecmp(kind,"con")){
	fprintf (stderr, "unrecognized kind in sysname %d: %s\n",
		 sysname, kind);
	fprintf (stderr, "using default kind: syn");
	al->kind = "syn";
      }else al->kind = kind;
    }else if (!inst_pic) al->kind = "syn";

    if (ValFoundP("thickness")){
      if (inst_pic){
	fprintf (stderr,
		 "ignoring thickness for sysname %d in instance picture\n",
		 sysname);
      }else if (!strcasecmp(thickness, "thick")){
	al->thick = 1;
      }else if (!strcasecmp(thickness, "thin")){
	al->thick = 0;
      }else{
	fprintf (stderr, "unexpected value for thickness in sysname %d: %s\n",
		 sysname, thickness);
	fprintf (stderr, "using default value: thin\n");
	al->thick = 0;
      }				/* if/else */
    }else if (!inst_pic) al->thick = 0;

    if (ValFoundP("starred?")){
      if (inst_pic){
	fprintf (stderr,
		 "ignoring starred? for sysname %d in instance picture\n",
		 sysname);
      }else if (!strcasecmp(starred,"true")){
	al->starred = 1;
      }else if (!strcasecmp(starred,"false")){
	al->starred = 0;
      }else{
	fprintf (stderr, "unexpected value for starred? in sysname %d: %s\n",
		 sysname, starred);
	fprintf (stderr, "using default value: false\n");
	al->starred = 0;
      }				/* if/else */
    }else if (!inst_pic) al->starred = 0;
  }				/* for */

  /* later: Does the */
  /* editor "parity" entry have anything to do with us?  Can we safely */
  /* ignore tail-loc and head-loc? */

  return (1);
}				/* extract_arrows */

void print_boxes (fp, box_list)
FILE *fp;
extracted_box_value_t *box_list;
{
  int min_x = -1, min_y = -1, max_x = 0, max_y = 0;
  extracted_box_value_t *l;
  list_type child_list;

  fprintf (fp, "\n %c ===== Boxes =====\n",
	   (output_lisp ? ';' : '#'));
  if (output_lisp) fputs (" (s-value *a* :boxes\n  (list\n", fp);

  for (l=box_list;l; l = l->next){
    if ((l->x == -1) && !l->arrow && output_lisp &&
	!strcasecmp(l->box_role,"user")) continue;

    if (output_lisp){
      char *np;
      fputs ("   (create-instance nil miro-box\n", fp);
      fprintf (fp, "    (:sysname %d)\n", l->sysname);
      fprintf (fp, "    (:string \"", l->name);
      if ((!index (l->name, '"')) &&
	  (!index (l->name, '\n')) &&
	  (!index (l->name, '\\'))){
	fputs (l->name, fp);
      }else for (np=l->name; *np; np++){
	if (index ("\"\n\\", *np)) fputc ('\\', fp);
	fputc (*np, fp);
      }
      fputs ("\")\n", fp);
    }else{
      char *np;
      fputs (">BOX\n", fp);
      fprintf (fp, " sysname = %d;\n", l->sysname);
      fputs (" name = \"", fp);
      if ((!index (l->name, '"')) &&
	  (!index (l->name, '\n')) &&
	  (!index (l->name, '\\'))){
	fputs (l->name, fp);
      }else for (np=l->name; *np; np++){
	if (index ("\"\n\\", *np)) fputc ('\\', fp);
	fputc (*np, fp);
      }
      fputs ("\";\n", fp);
    }				/* if/else */

    if (l->x > -1){
      if (output_lisp){
	fprintf (fp, "    (:box '(%d %d %d %d))\n", l->x,
		 l->y, l->width, l->height);
      }else fprintf (fp, " loc = {%d, %d};\n size = {%d, %d};\n",
		     l->x, l->y, l->width,
		     l->height);
      if (min_x == -1){
	min_x = l->x;
	min_y = l->y;
      }				/* if */
      min_x = MIN(l->x,min_x);
      min_y = MIN(l->y,min_y);
      max_x = MAX(l->x+l->width,max_x);
      max_y = MAX(l->y+l->height,max_y);
    }				/* if */

    if (output_lisp){
      fprintf (fp, "    (:box-role \"%s\")\n", l->box_role);
      fprintf (fp, "    (:box-type \"%s\")\n", l->box_type);
    }else fprintf (fp, " role = %s;\n type = %s;\n",
		   l->box_role, l->box_type);

    if (l->thick > -1){
      if (output_lisp){
	fprintf (fp, "    (:thick %s)\n", (l->thick ? "T" : "nil"));
      }else fprintf (fp, " thickness = %s;\n",
		     (l->thick ? "thick" : "thin"));
    }				/* if */
    if (l->starred > -1){
      if (output_lisp){
	fprintf (fp, "    (:starred %s)\n", (l->starred ? "T" : "nil"));
      }else fprintf (fp, " starred? = %s;\n",
		     (l->starred ? "true" : "false"));
    }				/* if */
    if (output_lisp){
      fputs ("    )\n", fp);
    }				/* if */
  }				/* for */

  if (!output_lisp) for (l=box_list; l; l=l->next){
    if (!null(l->child_sysnames)){
      fprintf (fp, ">INSIDE parent = %d; children = {",
	       l->sysname);
      for (child_list=l->child_sysnames; !null(child_list);
	   child_list=cdr(child_list)){
	fprintf (fp, "%d%s", (int)car(child_list),
		 (null(cdr(child_list)) ? "};\n" : ","));
      }				/* for */
    }else if (!strcasecmp(l->box_type, "group") ||
	      !strcasecmp(l->box_type, "world")){
      fprintf (fp, ">INSIDE parent = %d; children = {};\n",
	       l->sysname);
    }				/* if */
  }				/* for */

  if (output_lisp) fputs ("   ))\n", fp);

  /* print the bounding box, if we found one */
  if (output_lisp && (min_x > -1)){
    fprintf (fp, " (s-value *a* :bounding-box '(%d %d %d %d))\n",
	     min_x, min_y, max_x, max_y);
  }				/* if */
}				/* print_boxes */

void print_arrows (fp, arrow_list, arrow_labels)
FILE *fp;
extracted_arrow_value_t *arrow_list;
char **arrow_labels;
{
  fprintf (fp, "\n %c ===== Arrows =====\n",
	   (output_lisp ? ';' : '#'));
  if (output_lisp) fputs (" (s-value *a* :arrows\n  (list\n", fp);

  for (; arrow_list; arrow_list=arrow_list->next){
    if (!arrow_list->visible) continue;
    
    if (output_lisp){
      fputs ("   (create-instance nil miro-arrow\n", fp);
    }else fputs (">ARROW\n", fp);

    if (output_lisp){
      fprintf (fp, "    (:sysname %d)\n", arrow_list->sysname);
      fprintf (fp, "    (:from-sysname %d)\n" , arrow_list->from);
      fprintf (fp, "    (:to-sysname %d)\n", arrow_list->to);
      fprintf (fp, "    (:neg %s)\n", (arrow_list->negative_parity?"T":"nil"));
    }else{
      fprintf (fp, " sysname = %d;\n", arrow_list->sysname);
      fprintf (fp, " from = %d;\n", arrow_list->from);
      fprintf (fp, " to = %d;\n", arrow_list->to);
      fprintf (fp, " parity = %s;\n",
	       (arrow_list->negative_parity ? "neg" : "pos"));
    }				/* if/else */

    if (!null(arrow_list->label)){
      list_type l;
      if (output_lisp){
	fprintf (fp, "    (:string \"");
      }else fprintf (fp, " type = {");
      for (l=arrow_list->label; !null(l); l=cdr(l)){
	fprintf (fp, "%s%s", arrow_labels[(int)car(l)],
		 (null(cdr(l)) ? (output_lisp ? "\")\n" : "};\n") : ","));
      }				/* for */
    }				/* if */

    if (arrow_list->kind != NULL){
      if (output_lisp){
	fprintf (fp, "    (:arrow-type :%s)\n", arrow_list->kind);
      }else fprintf (fp, " kind = %s;\n", arrow_list->kind);
    }				/* if */

    if (arrow_list->thick > -1){
      if (output_lisp){
	fprintf (fp, "    (:thick %s)\n", (arrow_list->thick ? "T" : "nil"));
      }else fprintf (fp, " thickness = %s;\n",
		     (arrow_list->thick ? "thick" : "thin"));
    }				/* if */

    if (arrow_list-> starred > -1){
      if (output_lisp){
	fprintf (fp, "    (:starred %s)\n", (arrow_list->starred ? "T":"nil"));
      }else fprintf (fp, " starred? = %s;\n",
		     (arrow_list->starred ? "true" : "false"));
    }				/* if */

    if (output_lisp) fputs ("    )\n", fp);
  }				/* for */

  if (output_lisp) fputs ("   ))\n", fp);
}				/* print_arrows */

int arrow_compare (a1, a2)
extracted_arrow_value_t **a1, **a2;
{
  if ((*a1)->from < (*a2)->from) return(-1);
  if ((*a1)->from > (*a2)->from) return(1);
  if ((*a1)->to < (*a2)->to) return(-1);
  if ((*a1)->to > (*a2)->to) return(1);
  if ((*a1)->negative_parity < (*a2)->negative_parity) return(-1);
  if ((*a1)->negative_parity > (*a2)->negative_parity) return(1);
  return(0);
}				/* arrow_compare() */

int int_member (the_int, int_list)
int the_int;
list_type int_list;
{
  for (; !null(int_list) && (the_int != (int)car(int_list));
       int_list = cdr(int_list));
  if (null(int_list)) return (0); else return (1);
} /* int_member() */

int compare_int_lists (list1, list2)
list_type list1, list2;
{
  list_type l1, l2;
  
  /* take care of the easy cases first */
  if (null(list1) && null(list2)) return (1);
  if (null(list1) || null(list2)) return (0);
  if (length(list1) != length(list2)) return(0);

  /* We assume that the lists are fairly short.  If not, it might be
     better to copy them into arrays and sort them. */
  for (l1=list1, l2=list2;
       !null(l1) && int_member((int)car(l1),l2);
       l1=cdr(l1));
  if (!null(l1)) return(0);
  for (l1=list2, l2=list1;
       !null(l1) && int_member((int)car(l1),l2);
       l1=cdr(l1));
  if (!null(l1)) return(0);
  return(1);
} /* compare_int_lists() */
  
int combine_arrows (arrow_list, box_array, box_array_size)
extracted_arrow_value_t *arrow_list;
extracted_box_value_t **box_array;
int box_array_size;
{
  int num_arrows = 0, i, j, to, from, changed_something;
  int arrow_sp, arrow_base, min_box;
  list_type child_list, parent_list;
  extracted_arrow_value_t *a, **arrow_array;

  if (arrow_list == NULL) return (1);
  min_box = box_array[0]->sysname;
  
  /* figure out how many arrows there are */
  for (a=arrow_list; a!=NULL; a=a->next, num_arrows++);

  /* malloc an array to hold the arrows */
  if (!MALLOC(num_arrows*sizeof(extracted_arrow_value_t*),
	      (void*)&arrow_array,
	      "malloc failed in combine_arrows()")) return(0);
  
  /* put the arrows into the array and sort them */
  for (a=arrow_list, i=0; a!=NULL; arrow_array[i++]=a, a=a->next);
  qsort (arrow_array, num_arrows, sizeof(extracted_arrow_value_t*),
	 arrow_compare);

  do{
    changed_something = 0;
    /* look at all the arrows */
    arrow_base = 0;
    while (arrow_base<num_arrows){
      /* move to the next usable set of arrows */
      for (;(arrow_base<num_arrows) &&
	   ((arrow_array[arrow_base] == NULL) ||
	    null(box_array[arrow_array[arrow_base]->to-min_box]
		 ->parent_sysnames));
	   arrow_base++);
      if (arrow_base >= num_arrows) continue;
      parent_list = box_array[arrow_array[arrow_base]->to-min_box]
	->parent_sysnames;

      /* put equivalently-labeled arrows together */
      for (arrow_sp=j=arrow_base+1, i=arrow_base;
	   (j<num_arrows) &&
	   (arrow_array[j] != NULL) &&
	   (arrow_array[i]->from == arrow_array[j]->from) &&
	   (arrow_array[i]->negative_parity ==
	    arrow_array[j]->negative_parity);
	   j++){
	/* are these arrows equivalent? check label, parent */
	if (compare_int_lists(arrow_array[i]->label,
			      arrow_array[j]->label) &&
	    compare_int_lists(parent_list,
			      box_array[arrow_array[j]->to-min_box]
			      ->parent_sysnames)){
	  /* swap arrows if necessary */
	  if (arrow_sp < j){
	    extracted_arrow_value_t *tmp = arrow_array[arrow_sp];
	    arrow_array[arrow_sp] = arrow_array[j];
	    arrow_array[j] = tmp;
	  }			/* if */
	  arrow_sp++;
	}			/* if */
      }				/* for */

      /* did we find more than one arrow?  are we looking at only one */
      /* parent? if not, move arrow_base forward and try again */
      if ((arrow_sp == arrow_base+1) || (length(parent_list) > 1)){
	arrow_base = arrow_sp;
	continue;
      }				/* if */

      /* make a list of all the children pointed to by this set of */
      /* arrows */
      for (child_list=nil, i=arrow_sp-1; i >= arrow_base; i--){
	child_list = cons (arrow_array[i]->to, child_list);
      }				/* for */

      /* do the arrows in question point to all the children?  If so, */
      /* replace them by a single arrow to the parent */
      if (compare_int_lists(box_array[((int)car(parent_list))-min_box]
			    ->child_sysnames, child_list)){
	arrow_array[arrow_base]->to = (int)car(parent_list);
	for (; --arrow_sp > arrow_base;
	     arrow_array[arrow_sp]->visible=0, arrow_array[arrow_sp]=NULL);
	changed_something = 1;
      }				/* if */

      arrow_base++;
    }				/* while */

    /* remove NULLs from the array */
    for (i=0,j=0; i<num_arrows; i++, j++){
      if ((i==j) && (arrow_array[i] != NULL)) continue;
      while ((arrow_array[i]==NULL) && (i<num_arrows)) i++;
      if (i >= num_arrows) continue;
      arrow_array[j] = arrow_array[i];
    }				/* for */
    num_arrows = j;

    /* sort the arrows again */
    qsort (arrow_array, num_arrows, sizeof(extracted_arrow_value_t*),
	   arrow_compare);
  }while(changed_something);

  /* look for equivalent arrows with different labels */
  for (i=0, j=1; i<num_arrows; j++){
    if ((j >= num_arrows) ||
	arrow_compare(&arrow_array[i],&arrow_array[j])){
      /* these arrows don't match */
      if (i != j-1){
	int k;
	list_type new_list, l;
	/* we found some matching arrows -- combine them */
	for (k=i, new_list = nil; k<j; k++){
	  arrow_array[k]->visible = 0;
	  for (l=arrow_array[k]->label; !null(l); l=cdr(l)){
	    if (!int_member((int)car(l),new_list)){
	      new_list=cons(car(l),new_list);
            }			/* if */
	  }			/* for */
	}			/* for */
	arrow_array[i]->label = new_list;
	arrow_array[i]->visible = 1;
	for (k=i+1; k<j; arrow_array[k++] = NULL);
      }				/* if */
      i = j;
    }				/* if */
  }				/* for */

  return (1);
}				/* combine_arrows() */

int assign_box_locations(box_array, min_sysname, max_sysname)
extracted_box_value_t **box_array;
int min_sysname, max_sysname;
{
  int num_boxes = 0, i, this_box;
  int *box_stack = NULL;
  box_location_t *loc_stack = NULL;
  int todo_sp, undo_sp, loc_sp;
  int array_size;
  list_type tmp_list;
  int children_on_stack;
  int too_big = 0;
  int max_child_x, max_child_y;
  int preferred_x, preferred_y, alternate_x, alternate_y;
  int max_x, max_y;
  box_location_t user_loc, file_loc;
  int last_box_was_user = -1;

  /* find out if we need to be here... */
  if (box_array[0] == NULL){
    fprintf (stderr, "first box does not exist!!!\n");
    return(0);
  }else if (box_array[0]->x > -1) return(-1);

  array_size = max_sysname - min_sysname + 1;

  /* figure out how many boxes there are */
  for (i=0; i < array_size;
       i++, num_boxes += (box_array[i]==NULL)?0:1);

  /* allocate the stack */
  if (!MALLOC(((2*num_boxes)+1)*sizeof(int), (void*)&box_stack,
	      "malloc failed in assign_box_locations()")) return(0);
  if (!MALLOC(num_boxes*sizeof(box_location_t), (void*)&loc_stack,
	      "malloc failed in assign_box_locations()")) return(0);
  todo_sp = loc_sp = 1;
  undo_sp = num_boxes;
  user_loc.max_x = 770;
  file_loc.max_x = user_loc.max_y = file_loc.max_y = 1600;
  user_loc.preferred_x = user_loc.preferred_y = file_loc.preferred_y =
    user_loc.alternate_x = user_loc.alternate_y = file_loc.alternate_y
      = 0;
  file_loc.preferred_x = file_loc.alternate_x = 830;
  loc_stack[0] = user_loc;
  box_stack[0] = -1;

  /* find all the top-level boxes and put them on the stack */
  last_box_was_user = strcasecmp(box_array[0]->box_role,"user");
  for (i=0; i < array_size; i++){
    if ((box_array[i] != NULL) &&
	null(box_array[i]->parent_sysnames)){
      if (last_box_was_user &&
	  strcasecmp(box_array[i]->box_role,"user")){
	box_stack[todo_sp++] = -1;
	loc_stack[loc_sp++] = user_loc;
	last_box_was_user = 0;
      }else if (!last_box_was_user &&
		strcasecmp(box_array[i]->box_role,"file")){
	box_stack[todo_sp++] = -1;
	loc_stack[loc_sp++] = file_loc;
	last_box_was_user = 1;
      }				/* if/else */
      box_stack[todo_sp++] = i;
    }				/* if */
  }				/* for */
  box_stack[todo_sp++] = -1;
  loc_stack[loc_sp++] = last_box_was_user ? user_loc : file_loc;

  /* loop until there are no more boxes left to do... */
  while (todo_sp > 0){
    /* Pop the first box off the stack. */
    this_box = box_stack[--todo_sp];

    /* Time to pop a new location? */
    if (this_box == -1){
      max_x = loc_stack[--loc_sp].max_x;
      max_y = loc_stack[loc_sp].max_y;
      preferred_x = loc_stack[loc_sp].preferred_x;
      preferred_y = loc_stack[loc_sp].preferred_y;
      alternate_x = loc_stack[loc_sp].alternate_x;
      alternate_y = loc_stack[loc_sp].alternate_y;
      continue;
    }				/* if */

    /* skip the children if we got here because something didn't fit. */
    if (!too_big){
      /* If there are un-placed children, put them on the stack.  We */
      /* will get back to the parent later. */
      max_child_x = preferred_x;
      if (box_array[this_box]->name != NULL){
	max_child_x += (int)(0.5 + (CHARACTER_WIDTH *
				    strlen(box_array[this_box]->name)));
      }else max_child_x += (int)(0.5 + CHARACTER_WIDTH);
      max_child_y = preferred_y;
      tmp_list=box_array[this_box]->child_sysnames;
      for (children_on_stack=0; !null(tmp_list); tmp_list = cdr(tmp_list)){
	i = (int)car(tmp_list) - min_sysname;
	if ((i < 0) || (i >= array_size)){
	  fprintf (stderr, "invalid sysname\n");
	  return(0);
	}			/* if */
	/* do we want to look at this box? */
	if (!strcasecmp(box_array[i]->box_role,"user") &&
	    !box_array[i]->arrow &&
	    !null(box_array[i]->child_sysnames)){
	  list_type t;
	  tmp_list = cdr(tmp_list);
	  for (t = box_array[i]->child_sysnames; !null(t); t=cdr(t))
	       tmp_list = cons(car(t),tmp_list);
	  tmp_list = cons(i+min_sysname,tmp_list);
	  continue;
	}			/* if */
	if (box_array[i]->x > -1){
	  /* this child already has a location */
	  max_child_x = MAX(max_child_x,
			    box_array[i]->x+box_array[i]->width);
	  max_child_y = MAX(max_child_y,
			    box_array[i]->y+box_array[i]->height);
	}else{
	  /* this child doesn't have a location yet */
	  if (!children_on_stack){
	    /* box is still on the stack, just need to increment the */
	    /* pointer */
	    todo_sp++;
	    /* save our current location info */
	    box_stack[todo_sp++] = -1;
	    loc_stack[loc_sp].max_x = max_x;
	    loc_stack[loc_sp].max_y = max_y;
	    loc_stack[loc_sp].preferred_x = preferred_x;
	    loc_stack[loc_sp].preferred_y = preferred_y;
	    loc_stack[loc_sp].alternate_x = alternate_x;
	    loc_stack[loc_sp].alternate_y = alternate_y;
	    loc_sp++;
	    preferred_x += INTER_BOX_SPACING;
	    preferred_y += INTER_BOX_SPACING;
	    alternate_x = preferred_x;
	    alternate_y = preferred_y;
	    max_x -= INTER_BOX_SPACING;
	    max_y -= INTER_BOX_SPACING + CHARACTER_HEIGHT;
	    children_on_stack = 1;
	  }			/* if */
	  box_stack[todo_sp++] = i;
	}			/* if/else */
      }				/* for */
      if (children_on_stack) continue; /* take care of the children first */

      /* If we got this far, we are ready to take care of this box.  Set */
      /* its location and put it on the undo stack. */
      box_array[this_box]->x = preferred_x;
      box_array[this_box]->y = preferred_y;
      box_array[this_box]->width = max_child_x + INTER_BOX_SPACING -
	preferred_x;
      box_array[this_box]->height = max_child_y + CHARACTER_HEIGHT +
	INTER_BOX_SPACING - preferred_y;
    }				/* if */

    /* If the location we have assigned this box puts it outside the */
    /* "drawing area", unwind and try again. */
    if (((box_array[this_box]->x + box_array[this_box]->width) > max_x) ||
	((box_array[this_box]->y + box_array[this_box]->height) >
	 max_y) || too_big){
      /* If we can do something about it here, unwind and try again, */
      /* otherwise handle it when we try to compute the parent's */
      /* size... */
      if ((preferred_x != alternate_x) || (preferred_y != alternate_y)){
	/* box is still on the stack, just need to increment the */
	/* pointer */
	todo_sp++;
	/* mark all the affected boxes as "un-placed" */
	box_stack[undo_sp--] = this_box;
	while (undo_sp < num_boxes){
	  i = box_stack[++undo_sp];
	  box_array[i]->x = -1;
	  for (tmp_list=box_array[i]->child_sysnames; !null(tmp_list);
	       tmp_list=cdr(tmp_list)){
	    box_stack[undo_sp--] = (int)(car(tmp_list)) - min_sysname;
	  }			/* for */
	}			/* while */
	/* reset x, y */
	preferred_x = alternate_x;
	preferred_y = alternate_y;
	too_big = 0;
      }else{
	/* back up to the parent and try again */
	while (box_stack[--todo_sp] > -1);
	too_big = 1;
	todo_sp++;
      }				/* if/else */
      continue;
    }				/* if */

    /* reset alternate_y to be below the other boxes, preferred_x to */
    /* be beside the other boxes on this row. */
    preferred_x += box_array[this_box]->width + INTER_BOX_SPACING;
    alternate_y = MAX(alternate_y, box_array[this_box]->y +
		      box_array[this_box]->height + INTER_BOX_SPACING);
    too_big = 0;
  }				/* while */

  if (too_big){
    fprintf (stderr, "picture too big\n");
    return(0);
  }else return (1);
}				/* assign_box_locations() */

int generate_lisp_output (fp, box_list, arrow_list, inside_list,
			  editor_info, need_locations)
FILE *fp;			/* where to send the output */
extracted_box_value_t *box_list; /* a list of boxes */
extracted_arrow_value_t *arrow_list; /* a list of arrows */
extracted_relation_t *inside_list; /* a list of inside entries */
extracted_editor_info_t *editor_info; /* the editor entry */
int need_locations;
{
  extracted_box_value_t **box_array;
  int min_box_sysname, max_box_sysname;
  extracted_arrow_value_t **arrow_array;
  int min_arrow_sysname, max_arrow_sysname;
  int retval;
  char **arrow_labels;
  int arrow_labels_size;

  /* initialize the extraction functions */
  InitExtract(20);

  /* print the first part of our lambda expression */
  if (output_lisp){
    fputs (";; eval this and call it on a newly-created aggregate\n", fp);
    fputs ("#'(lambda (*a*)\n", fp);
  }				/* if */

  /* print the editor entry */
  if (print_editor_entry(fp,editor_info) == 0) return (0);

  /* put the boxes into an array */
  if (extract_boxes(box_list,&box_array,&min_box_sysname,&max_box_sysname,
		    editor_info) == 0) return (0);

  /* extract arrow information */
  if (extract_arrows(arrow_list, editor_info, box_array,
		     min_box_sysname, max_box_sysname, &arrow_labels,
		     &arrow_labels_size) == 0) return (0);

  /* process the inside entries */
  if (process_inside_entries(inside_list,box_array,min_box_sysname,
			     max_box_sysname) == 0) return (0);

  /* assign box sizes/locations where necessary */
  if (need_locations)
    if ((retval =
	 assign_box_locations(box_array,min_box_sysname,
			      max_box_sysname)) == 0) return(0); 

  /* combine arrows where possible -- never finished this
  if (retval > 0) if (combine_arrows(arrow_list, box_array,
				     max_box_sysname-min_box_sysname+1)
		      == 0) return(0);
  */

  /* print lisp representation of the boxes and arrows */
  print_boxes (fp, box_list);
  print_arrows (fp, arrow_list, arrow_labels);

  /* print the end of the lambda expression */
  if (output_lisp){
    if (retval > 0) fputs ("\n (s-value *a* :pic_modified T)\n",fp);
    fputs ("\n ;; Make a hash table for the boxes and fix the labels.\n",
	   fp);
    fputs
      (" (let ((h (s-value *a* :hashtbl (make-hash-table :test #'equal)))\n",
	  fp);
    fputs("       l)\n",fp);
    fputs ("  (dolist (b (g-value *a* :boxes))\n", fp);
    fputs ("   (setf (gethash (g-value b :sysname) h) b)\n", fp);
    fputs ("   (when (setq l (g-value b :string))\n", fp);
    fputs ("    (s-value (g-value b :label) :string l)\n", fp);
    fputs ("    (setf (gethash l h) b)))\n",fp);
    fputs ("  (dolist (a (g-value *a* :arrows))\n", fp);
    fputs ("   (when (setq l (g-value a :string))\n", fp);
    fputs ("    (s-value (g-value a :label) :string l))))\n", fp);
    fputs ("\n ;; Return the aggregate\n *a*)\n", fp);
  }				/* if */

  return (1);
}				/* generate_lisp_output */

int main(argc,argv)
int argc;
char *argv[];
{
  extracted_box_value_t *box_list;
  extracted_arrow_value_t *arrow_list;
  extracted_relation_t *inside_list;
  extracted_editor_info_t editor_info;
  int min_sysname, max_sysname;
  char *input_file = NULL, *output_file = NULL;
  int need_locations = 1;
  FILE *in_fp, *out_fp;

  /* get files from the argument list. */
  this_program = argv[0];

  for (argv++, argc--; argc > 0; argv++, argc--){
    if (**argv == '-'){
      if (!strcasecmp (*argv, "-iff")){
	output_lisp = 0;
      }else if (!strcasecmp (*argv, "-lisp")){
	output_lisp = 1;
      }else if (!strcasecmp (*argv, "-input")){
	input_file = *++argv;
	argc--;
      }else if (!strcasecmp (*argv, "-output")){
	output_file = *++argv;
	argc--;
      }else if (!strcasecmp (*argv, "-noloc")){
	need_locations = 0;
      }else if (!strcasecmp (*argv, "-loc")){
	need_locations = 1;
      }else goto usage;
    }else{
    usage:
      fprintf(stderr,
	      "usage: iff2lisp [-iff] [-lisp] [-loc] [-noloc] [-output file] [-input file]\n");
      exit (-1);
    }				/* if/else */
  }				/* for */

  /* try to open the files. */
  if (input_file == NULL){
    in_fp = stdin;
  }else if ((in_fp = fopen(input_file,"r")) == NULL){
    fprintf (stderr, "file %s not found\n", input_file);
    exit (-1);
  }				/* if/else */
  if (output_file == NULL){
    out_fp = stdout;
  }else if ((out_fp = fopen(output_file,"w")) == NULL){
    fprintf (stderr, "couldn't open file %s for writing\n", output_file);
    exit (-1);
  }				/* if/else */

  /* parse the iff file. */
  if (parse_iff_file(in_fp,&box_list,&arrow_list,&inside_list,&editor_info)
      == 0){
    fprintf (stderr, "iff2lisp: parse failed!\n");
    exit (-1);
  }				/* if/else */

  /* print the results */
  if (generate_lisp_output(out_fp,box_list,arrow_list,inside_list,
			   &editor_info,need_locations)
      == 0){
    fprintf (stderr, "iff2lisp: couldn't generate lisp output\n");
    exit (-1);
  }				/* if */

  /* Clean up. */
  ShutDownParser();
}				/* main() */
